﻿using System;
using Microsoft.Win32;

namespace SdkEnvironmentCheckerLibrary.Checkers
{
    public static class DotNetFrameworkVersionChecker
    {
        private const string V4FullSubKeyName = @"Software\Microsoft\NET Framework Setup\NDP\v4\Full";
        private const string ReleaseValueName = @"Release";

        public static CheckerResult Check(DotNetFrameworkVersionInfo supportedVersion)
        {
            if (supportedVersion == null) throw new ArgumentNullException(nameof(supportedVersion));

            var name = supportedVersion.DisplayName;

            if (string.IsNullOrWhiteSpace(name))
            {
                return CheckerResult.ConfirmFailed("Unknown .NET Framework Version", "'DisplayName' is required");
            }

            try
            {
                var installedReleaseNumber = (int)Registry.LocalMachine.OpenSubKey(V4FullSubKeyName).GetValue(ReleaseValueName);

                if (supportedVersion.ReleaseNumber < installedReleaseNumber) return CheckerResult.NewVersionInstalled(name);
                if (supportedVersion.ReleaseNumber == installedReleaseNumber) return CheckerResult.Installed(name);
            }
            catch { /* ignored: 読み取り失敗は NotInstalled 扱い */ }

            return CheckerResult.NotInstalled(name);
        }
    }

    /// <summary>
    /// .NET Framework リリース番号については以下を参照してください。
    /// https://msdn.microsoft.com/ja-jp/library/hh925568(v=vs.110).aspx#net_d
    /// https://docs.microsoft.com/en-us/dotnet/framework/migration-guide/how-to-determine-which-versions-are-installed
    /// </summary>
    public class DotNetFrameworkVersionInfo
    {
        public DotNetFrameworkVersionInfo(string displayName, int releaseNumber)
        {
            DisplayName = displayName ?? throw new ArgumentNullException(nameof(displayName));
            ReleaseNumber = releaseNumber;
        }

        public string DisplayName { get; }

        public int ReleaseNumber { get; }
    }
}
