﻿using System;

namespace SdkEnvironmentCheckerLibrary
{
    public class CheckerResult
    {
        public static CheckerResult Installed(string name, string label = null)
            => new CheckerResult(name, InstallStatus.Installed, label, null);

        public static CheckerResult NotInstalled(string name, string label = null)
            => new CheckerResult(name, InstallStatus.NotInstalled, label, null);

        public static CheckerResult NewVersionInstalled(string name, string label = null)
            => new CheckerResult(name, InstallStatus.NewVersionInstalled, label, null);

        public static CheckerResult ConfirmFailed(string name, string error, string label = null)
            => new CheckerResult(name, InstallStatus.ConfirmFailed, label, error);

        private CheckerResult(string displayName, InstallStatus status, string statusLabel, string error)
        {
            if (string.IsNullOrWhiteSpace(displayName)) throw new ArgumentException(nameof(displayName));

            DisplayName = displayName;
            Status = status;
            StatusLabel = string.IsNullOrWhiteSpace(statusLabel) ? GetDefaultLabel(status) : statusLabel;
            ErrorMessage = string.IsNullOrWhiteSpace(error) ? null : error;
        }

        public string DisplayName { get; }

        public InstallStatus Status { get; }

        public string StatusLabel { get; }

        public bool IsAvailable => Status == InstallStatus.Installed || Status == InstallStatus.NewVersionInstalled;

        public string ErrorMessage { get; }

        public override string ToString()
        {
            var displayname = DisplayName.Replace(Environment.NewLine, " ");

            if (Status == InstallStatus.ConfirmFailed)
            {
                return $"[Error] ({StatusLabel})\t{displayname} : {ErrorMessage}";
            }

            return $"[{(IsAvailable ? "OK" : "NG")}] ({StatusLabel})\t{displayname}";
        }

        private static string GetDefaultLabel(InstallStatus status)
        {
            switch (status)
            {
                case InstallStatus.ConfirmFailed: return "Confirm failed";
                case InstallStatus.NotInstalled: return "Not installed";
                case InstallStatus.Installed: return "Installed";
                case InstallStatus.NewVersionInstalled: return "New version installed";
                default: throw new NotSupportedException();
            }
        }
    }

    /// <summary>
    /// インストール状況を示します。
    /// </summary>
    public enum InstallStatus
    {
        /// <summary>
        /// インストール状態の確認に失敗した
        /// </summary>
        ConfirmFailed = 0,

        /// <summary>
        /// インストールされていない
        /// </summary>
        NotInstalled = 1,

        /// <summary>
        /// インストールされている
        /// </summary>
        Installed = 2,

        /// <summary>
        /// より新しいバージョンがインストールされている
        /// </summary>
        NewVersionInstalled = 3,
    }
}
