﻿using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using Nintendo.ToolFoundation.CommandLine;
using SdkEnvironmentCheckerConsole.Properties;
using SdkEnvironmentCheckerLibrary;
using SdkEnvironmentCheckerLibrary.Serializers;

namespace SdkEnvironmentCheckerConsole
{
    class Program
    {
        private const string DefaultConfigPath = @"SdkEnvironmentChecker.config";

        static void Main(string[] args)
        {
            var assemblyDir = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
            Debug.Assert(assemblyDir != null);

            var configPath = Path.Combine(assemblyDir, DefaultConfigPath);
            var excludeNvn = false;
            var verbose = false;

            // parse arguments
            {
                var command = ActionCommand.CreateRootCommand();
                command.AddValueOption("config", path => configPath = path).GetBuilder().Hide();
                command.AddFlagOption("verbose", () => verbose = true).GetBuilder().SetDescription(Resources.CommandLineVerboseDescription);
                command.AddValueOption("exclude", target =>
                {
                    if (target == "nvn") excludeNvn = true;
                    else throw new ArgumentException($"[--exclude {target}] is invalid.");
                }).GetBuilder().SetValueName("target").SetDescription(Resources.CommandLineExcludeDescription);

                switch (CommandLine.ParseArgs(args, command))
                {
                    case ParseResultType.Success: break;
                    case ParseResultType.Failure: Environment.ExitCode = -1; return;
                    default: return;
                }
            }

            try
            {
                var config = SdkEnvironmentCheckerConfig.Load(configPath);
                if (excludeNvn) config.NvnGraphicsDriver = false;

                var results = SdkEnvironmentChecker.Check(config.ToCheckerInfo()).ToArray();

                foreach (var result in results)
                {
                    if (!verbose && result.Status == InstallStatus.Installed) continue;

                    if (result.Status == InstallStatus.Installed || result.Status == InstallStatus.NewVersionInstalled)
                    {
                        Console.WriteLine(result);
                    }
                    else
                    {
                        Console.Error.WriteLine(result);
                    }
                }

                if (results.Any(r => r.Status == InstallStatus.ConfirmFailed)) Environment.ExitCode = -1;
                else if (results.Any(r => r.Status == InstallStatus.NotInstalled)) Environment.ExitCode = 1;
            }
            catch (Exception e)
            {
                Console.Error.WriteLine($"[Error] (fatal error)\tSdkEnvironmentChecker : {e.Message}");
                Environment.ExitCode = -1;
            }
        }
    }
}
