﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Diagnostics;

namespace QCIT_Maker
{
    internal class Service
    {
        private MetaEditor m_MetaEditor;
        private CommandLineParams m_CLParams;
        private string tempDirName = CommonParams.WorkDirName;
        private bool isDeleteTempDir;
        private string outDirPath = null;
        private bool isVerbose = false;

        public void SetVerboseMode()
        {
            isVerbose = true;
        }

        // 環境変数を含んだパスを置換するラムダ式
        private Func<string, string> replaceEnvVar = (path) =>
        {
            // %で囲んだ環境変数名をその値に置換します。
            // 置換対象の文字列が含まれていない場合は入力文字列をそのまま返します
            return Environment.ExpandEnvironmentVariables(path);
        };

        private Executor m_ToolExec;
        private SyncListView m_SyncListView;

        public Service(CommandLineParams inParams)
        {
            m_CLParams = inParams;
            m_MetaEditor = new MetaEditor();
            m_ToolExec = new Executor();
            m_SyncListView = new SyncListView();

            if (inParams.VerboseFlag == true)
            {
                this.SetVerboseMode();
                m_ToolExec.SetVerboseMode();
            }
        }

        private string GetExecutingAppDirPath()
        {
            // 実行している自アプリ(.exe)のパスに合わせたディレクトリパスを返す
            return Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location);
        }

        private string GetExecutingAppCodePath()
        {
            // 実行している自アプリ(.exe)のパスに合わせたcodeディレクトリパスを返す
            var appDirPath = GetExecutingAppDirPath();
            return Path.Combine(appDirPath, @"code");
        }

        private void CheckInputFilePath()
        {
            if (m_CLParams.ContentDirectoryPath != null)
            {
                var targetPath = replaceEnvVar(m_CLParams.ContentDirectoryPath);

                if (Directory.Exists(targetPath) == false)
                {
                    // 指定ディレクトリパスが存在しない場合は例外を投げる
                    throw new Exception(" Not Found : --content-directory = " + m_CLParams.ContentDirectoryPath);
                }

                // また nsp ファイルが存在するかどうかも確認
                var nspList = FsUtil.GetNspFileList(targetPath);
                if (nspList.Count == 0)
                {
                    // 指定ディレクトリに nsp ファイルが存在しない場合は例外を投げる
                    throw new Exception(" No Nsp Files : --content-directory = " + m_CLParams.ContentDirectoryPath);
                }

                m_CLParams.ContentDirectoryPath = targetPath;
            }

            // ラムダ式で ファイルチェックの定形処理を関数化しておく
            Action<string, string> CheckFilePathAction = (string inoutFilePath, string inOptionName) =>
            {
                if (inoutFilePath != null)
                {
                    // 環境変数で指定された場合を考慮
                    var targetPath = replaceEnvVar(inoutFilePath);
                    if (File.Exists(targetPath) == false)
                    {
                        // 指定ファイルパスが存在しない場合は例外を投げる
                        throw new Exception(string.Format(" Not Found File : --{0} = {1}", inOptionName, inoutFilePath));
                    }
                    // 環境変数から変換された場合を考慮
                    inoutFilePath = targetPath;
                }
            };

            CheckFilePathAction(m_CLParams.SyncListPath, "sync-list");

            CheckFilePathAction(m_CLParams.OriginalNspPath, "original-nsp-path");
            CheckFilePathAction(m_CLParams.PreviousPatchPath, "previous-patch-path");

            if (m_CLParams.OriginalNspPath == null && m_CLParams.PreviousPatchPath != null)
            {
                // Previous パッチファイルのみ指定がある場合は例外を投げる
                throw new Exception(" Required \"--original-nsp-path\" Option");
            }
        }

        //(SIGLO-76759)空き容量指定オプションのチェック
        private void CheckFreeNandSize()
        {
            if (string.IsNullOrEmpty(m_CLParams.FreeNandSize) == false)
            {
                var freeNandSize = 0;
                var isSetFreeSize = int.TryParse(m_CLParams.FreeNandSize, out freeNandSize);

                if (isSetFreeSize == false || freeNandSize <= 0 || freeNandSize > CommonParams.MaxFreeNandSize)
                {
                    throw new Exception("Service.CheckFreeNandSize() : \"free-nand-size\" Option Invalid value.");
                }
            }
        }

        private void InitWorkDir()
        {
            this.isDeleteTempDir = true;
            if (m_CLParams.NoCleanupFlag == true)
            {
                // 処理終了時に作業ディレクトリを残す設定にする
                this.isDeleteTempDir = false;
            }

            try
            {
                if (Directory.Exists(tempDirName) == false)
                {
                    // 作業ディレクトリが存在しない場合は作成する
                    Directory.CreateDirectory(tempDirName);
                    if (isVerbose == true)
                    {
                        Console.WriteLine("Create WorkDirectory : Path = {0}", tempDirName);
                    }
                }

                // もし作業フォルダ内にファイルやフォルダが存在する場合は全て削除しておく
                var workDirInfo = new DirectoryInfo(tempDirName);
                foreach (var file in workDirInfo.GetFiles())
                {
                    file.Delete();
                }
                foreach (var dir in workDirInfo.GetDirectories())
                {
                    dir.Delete(true);
                }
            }
            catch (IOException err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        private string InitOutputDir()
        {
            if (string.IsNullOrEmpty(m_CLParams.OutputFilePath) == false)
            {
                m_CLParams.OutputFilePath = replaceEnvVar(m_CLParams.OutputFilePath);
                // 出力ファイルパスが設定されている場合は以下の出力ディレクトリは作成不要
                return string.Empty;
            }

            var dt = DateTime.Now;
            var timeString = dt.ToString("yyyyMMdd_HHmmss");
            var outputDir = timeString + @"_QCIT";

            try
            {
                if (Directory.Exists(outputDir) == false)
                {
                    // フォルダが存在しない場合は作成する
                    Directory.CreateDirectory(outputDir);
                    if (isVerbose == true)
                    {
                        Console.WriteLine("Create OutputDirectory : Path = {0}", outputDir);
                    }
                }
            }
            catch (IOException err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }

            return outputDir;
        }

        /// <summary>
        /// 入力パラメータのチェックとファイル関連の初期化処理
        /// </summary>
        public void InitInputFileParams()
        {
            // 最低限の入力(ファイル)チェック
            CheckInputFilePath();
            //(SIGLO-76759)空き容量指定オプションのチェック
            CheckFreeNandSize();
            // 作業ディレクトリのチェック・作成
            InitWorkDir();
            // 出力先ディレクトリのチェック・作成
            outDirPath = InitOutputDir();
        }

        /// <summary>
        /// 作業フォルダなどの後始末処理
        /// </summary>
        public void ClearWorkSpace()
        {
            try
            {
                if (Directory.Exists(tempDirName) == true && this.isDeleteTempDir == true)
                {
                    // 作業フォルダが存在する場合はすべて削除する
                    Directory.Delete(tempDirName, true);
                }
            }
            catch (IOException err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        private string MakeChildWorkDirectory(string inParam)
        {
            var childDirPath = Path.Combine(FsUtil.GetDirFullPath(tempDirName), string.Format("Item_{0}", inParam));
            if (Directory.Exists(childDirPath) == false)
            {
                Directory.CreateDirectory(childDirPath);
            }
            return childDirPath;
        }

        /// <summary>
        /// QCIT を作成するメイン処理
        /// </summary>
        public void CreateQCIT()
        {
            var setting = new SettingParams(m_CLParams);

            if (isVerbose)
            {
                if (setting.RatingList.Count > 0)
                {
                    Console.Write("Set Rating : ");
                    foreach (var rating in setting.RatingList)
                    {
                        Console.Write("{0} ", rating);
                    }
                    Console.Write("\n");
                }
            }

            setting.ChildWorkDirPath = this.MakeChildWorkDirectory(setting.ApplicationId);
            setting.NmetaFilePath = Path.Combine(setting.ChildWorkDirPath, CommonParams.TempMetaFileName);

            // (SIGLO-68974) ソフトリーガル情報を設定するかどうかの確認
            this.CheckLegalInfoFile(setting);

            // コマンドラインの入力情報から nmeta ファイルを作成
            m_MetaEditor.CreateNmetaFile(setting);

            // descファイルのパスを設定
            var descFilePath = Path.Combine(GetExecutingAppDirPath(), @"RidContentsInstallTool.autogen.desc");

            // スレッドセーフ化のためのDescファイルのコピー
            var copyDstDescFilePath = Path.Combine(setting.ChildWorkDirPath, "tempDesc.desc");
            File.Copy(descFilePath, copyDstDescFilePath, true);

            var npdmOutPath = Path.Combine(setting.ChildWorkDirPath, "tempNpdm.npdm");
            try
            {
                // MakeMeta を叩いて npdm ファイルを作成
                var result = m_ToolExec.ExecuteMakeMeta(npdmOutPath, copyDstDescFilePath, setting.NmetaFilePath);
                if (result.IsFailure())
                {
                    Console.Error.WriteLine("ExecuteMakeMeta() Failed : id={0}", setting.ApplicationId);
                    Console.Error.WriteLine("ErrorMessage：{0}", result.ErrorMessage);
                    throw new Exception("ExecuteMakeMeta() Failed");
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }

            var srcCodeDir = this.GetExecutingAppCodePath();
            var codeDir = Path.Combine(setting.ChildWorkDirPath, "code");
            FsUtil.CopyDirectory(srcCodeDir, codeDir);

            var moveNpdmFilePath = Path.Combine(codeDir, @"main.npdm");
            // ツール側で作成した npdm ファイルをdataディレクトリに移動
            FsUtil.MoveFile(npdmOutPath, moveNpdmFilePath);

            // アイコン画像を生成しリストにパスを設定しておく
            {
                var fileName = string.Format("_QCITIcon_{0}_.bmp", setting.IconSetting.Language);
                setting.IconSetting.IconFilePath = Path.Combine(setting.ChildWorkDirPath, fileName);
                PictureCreator.CreateBitmapIconFile(setting.IconSetting.IconFilePath, setting);
            }

            // Meta ファイルにファイル関連の設定情報を追記する
            m_MetaEditor.AddFilePathMetaParameter(setting);

            var romDirPath = MakeRomDirectoryData(setting);

            this.CreateApplicationInfoFile(romDirPath, setting);

            //(SIGLO-76759)QCITに渡す空き容量指定ファイルを生成する
            if (string.IsNullOrEmpty(setting.FreeNandSize) == false)
            {
                this.CreateFreeNandSizeFile(romDirPath, setting);
            }

            if (setting.IsDebugMode)
            {
                // debug-mode が有効であれば、それを示すファイルを ROM 内に作成する
                // そのファイルの有無で QCIT は挙動を変化させる
                this.CreateDebugModeFile(romDirPath);
            }

            var nspOutPath = string.Format(@"{0}\QCIT.nsp", outDirPath);
            if (string.IsNullOrEmpty(setting.OutputFilePath) == false)
            {
                nspOutPath = setting.OutputFilePath;
            }

            try
            {
                // (SIGLO-70748) OriginalNspFilePath の設定有無でパッチとして作成するかを判断する
                if (string.IsNullOrEmpty(setting.OriginalNspFilePath))
                {
                    // こちらはパッチなしの通常処理
                    // AuthoringToolを叩いて nsp ファイルを作成
                    var result = m_ToolExec.ExecuteAuthoringToolCreateNsp(nspOutPath, copyDstDescFilePath, codeDir, romDirPath, setting);
                    if (result.IsFailure())
                    {
                        Console.Error.WriteLine("ExecuteAuthoringToolCreateNsp() Failed : id={0}", setting.ApplicationId);
                        Console.Error.WriteLine("ErrorMessage：{0}", result.ErrorMessage);
                        throw new Exception("ExecuteAuthoringToolCreateNsp() Failed");
                    }
                }
                else
                {
                    // パッチ作成用の処理
                    var tempCurrentFilePath = Path.Combine(setting.ChildWorkDirPath, "__current_qcit__.nsp");
                    // AuthoringToolを叩いて nsp ファイルを(一時的に)作成
                    var result = m_ToolExec.ExecuteAuthoringToolCreateNsp(tempCurrentFilePath, copyDstDescFilePath, codeDir, romDirPath, setting);
                    if (result.IsFailure())
                    {
                        Console.Error.WriteLine("ExecuteAuthoringToolCreateNsp() Failed : id={0}", setting.ApplicationId);
                        Console.Error.WriteLine("ErrorMessage：{0}", result.ErrorMessage);
                        throw new Exception("ExecuteAuthoringToolCreateNsp() Failed");
                    }
                    // パッチの nsp ファイルを作成
                    result = m_ToolExec.ExecuteAuthoringToolMakePatch(nspOutPath, copyDstDescFilePath, tempCurrentFilePath, setting);
                    if (result.IsFailure())
                    {
                        Console.Error.WriteLine("ExecuteAuthoringToolMakePatch() Failed : id={0}", setting.ApplicationId);
                        Console.Error.WriteLine("ErrorMessage：{0}", result.ErrorMessage);
                        throw new Exception("ExecuteAuthoringToolMakePatch() Failed");
                    }
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }

            // (SIGLO-71102) 確認用のコンテンツ一覧情報ファイルを作成 (元ファイルは RomDir にある InstallContentInfoList.txt)
            {
                var srcFilePath = Path.Combine(romDirPath, CommonParams.InstallContentInfoFileName);
                var dstFilePath = this.MakeOutputContentInfoListFilePath(nspOutPath);
                File.Copy(srcFilePath, dstFilePath, true);
            }
        }

        private string MakeRomDirectoryData(SettingParams inParam)
        {
            var tempRomDirPath = Path.Combine(inParam.ChildWorkDirPath, "RomDir");
            if (Directory.Exists(tempRomDirPath) == false)
            {
                Directory.CreateDirectory(tempRomDirPath);
            }

            if (string.IsNullOrEmpty(inParam.SyncListPath) == false)
            {
                // Sync リストが設定されていれば読み込んでおく
                m_SyncListView.Deserialize(inParam.SyncListPath);
            }

            var cmInfoList = new List<ContentMetaInfoParam>();
            var nspPathList = FsUtil.GetNspFileList(inParam.ContentDirectoryPath);

            // 時間の掛かりそうなコピー処理部分は並列処理にしてみる (効果は薄いかもしれないが・・)
            var opt = new ParallelOptions();
            opt.MaxDegreeOfParallelism = 4;
            Parallel.For(0, nspPathList.Count, opt, i =>
            {
                // nsp ファイルを Extract してメタ情報等を抽出
                var infoList = ExtractNspInfo(inParam.ChildWorkDirPath, nspPathList[i]);
                if (this.IsRomImportContent(infoList))
                {
                    var romNspFilePath = Path.Combine(tempRomDirPath, Path.GetFileName(nspPathList[i]));
                    // Rom ディレクトリ内にコピーする
                    File.Copy(nspPathList[i], romNspFilePath, true);
                    cmInfoList.AddRange(infoList);
                }
            });

            this.CreateNspPropertyFile(tempRomDirPath, cmInfoList);

            if (m_SyncListView.IsDeserialized)
            {
                this.CreateSyncPropertyFile(tempRomDirPath);
            }

            return tempRomDirPath;
        }

        private void CreateNspPropertyFile(string inRomDirPath, List<ContentMetaInfoParam> inInfoList)
        {
            // メタ情報一覧のファイル書き込み用の文字列(csv形式)を作成する
            var sb = new StringBuilder();
            var titleLine = "Id,NspFileName,Type,Version,Digest,DataSize,ApplicationId";
            // まずはタイトル行を作成
            sb.Append(titleLine + "\r\n");
            foreach (var nspInfo in inInfoList)
            {
                var lineStr = string.Format(
                    "{0},{1},{2},{3},{4},{5},{6}",
                    nspInfo.Id,
                    nspInfo.NspFileName,
                    nspInfo.Type,
                    nspInfo.Version,
                    nspInfo.Digest,
                    nspInfo.DataSize,
                    nspInfo.ApplicationId);
                sb.Append(lineStr + CommonParams.LineFeedStr);
            }

            var nspInfoFilePath = Path.Combine(inRomDirPath, CommonParams.InstallContentInfoFileName);
            FsUtil.CreateRecordFile(nspInfoFilePath, sb.ToString());
        }

        private void CreateSyncPropertyFile(string inRomDirPath)
        {
            // メタ情報一覧のファイル書き込み用の文字列(csv形式)を作成する
            var sb = new StringBuilder();
            var titleLine = "Type,ApplicationId,Index";
            // まずはタイトル行を作成
            sb.Append(titleLine + CommonParams.LineFeedStr);
            foreach (var demo in m_SyncListView.SyncList.DemoList)
            {
                var lineStr = string.Format("Demo,{0},", demo.Id);
                sb.Append(lineStr + CommonParams.LineFeedStr);
            }
            foreach (var asset in m_SyncListView.SyncList.AssetList)
            {
                var lineStr = string.Format("Asset,{0},{1}", asset.Id, asset.Index);
                sb.Append(lineStr + CommonParams.LineFeedStr);
            }

            var nspInfoFilePath = Path.Combine(inRomDirPath, "SyncListInfo.txt");
            FsUtil.CreateRecordFile(nspInfoFilePath, sb.ToString());
        }

        private List<ContentMetaInfoParam> ExtractNspInfo(string inWorkDirPath, string inNspPath)
        {
            var nspFileName = Path.GetFileNameWithoutExtension(inNspPath);
            var tempExtractedDirPath = Path.Combine(inWorkDirPath, "Extracted_" + nspFileName);
            if (Directory.Exists(tempExtractedDirPath))
            {
                Directory.Delete(tempExtractedDirPath, true);
            }

            Directory.CreateDirectory(tempExtractedDirPath);

            // nsp ファイルを extract して、展開したファイルは tempExtractedDirPath へ出力
            var result = m_ToolExec.ExecuteAuthoringToolExtractNsp(tempExtractedDirPath, inNspPath);
            if (result.IsFailure())
            {
                Console.Error.WriteLine("ExtractNsp() Failed : nspPath={0}", inNspPath);
                Console.Error.WriteLine("ErrorMessage：{0}", result.ErrorMessage);
                throw new Exception("ExecuteAuthoringToolExtractNsp() Failed");
            }
            // 展開したディレクトリの中から *.cnmt.xml のファイルパスを抽出
            var cnmtXmlFilePathList = FindContentMetaXmlList(tempExtractedDirPath);

            var infoParamList = new List<ContentMetaInfoParam>();
            foreach (var cnmtXml in cnmtXmlFilePathList)
            {
                // cnmt.xml ファイルから QCIT(_Maker) で使用するメタ情報を取得
                var infoParam = m_MetaEditor.GetContentMetaInfo(cnmtXml);
                infoParam.NspFileName = Path.GetFileName(inNspPath);
                infoParamList.Add(infoParam);
            }

            return infoParamList;
        }

        private List<string> FindContentMetaXmlList(string extractedDirectoryPath)
        {
            var extractedDirectory = new DirectoryInfo(extractedDirectoryPath);
            return (from fileInfo in extractedDirectory.GetFiles() where fileInfo.FullName.EndsWith(".cnmt.xml") select fileInfo.FullName).ToList();
        }

        private void CreateApplicationInfoFile(string inRomDirPath, SettingParams inParam)
        {
            var appIdFilePath = Path.Combine(inRomDirPath, "QCIT_ID.txt");

            string appInfoStr = inParam.ApplicationId;

            {
                // 32ビットのバージョン値に変換
                var rvNum = Convert.ToUInt32(inParam.ReleaseVersion);
                // PrivateVersion はひとまず 0 固定
                uint pvNum = 0;
                var verNum = (rvNum << 16) | pvNum;

                appInfoStr += CommonParams.LineFeedStr;
                // バージョン情報も追記する
                appInfoStr += (verNum.ToString());
            }

            FsUtil.CreateRecordFile(appIdFilePath, appInfoStr);
        }

        //(SIGLO-76759)QCITに渡す空き容量指定ファイルを生成する
        private void CreateFreeNandSizeFile(string inRomDirPath, SettingParams inParam)
        {
            var freeNandSizeFilePath = Path.Combine(inRomDirPath, "RequiredFreeNandSize.txt");
            FsUtil.CreateRecordFile(freeNandSizeFilePath, inParam.FreeNandSize);
        }

        private void CreateDebugModeFile(string inRomDirPath)
        {
            var debugFilePath = Path.Combine(inRomDirPath, "DebugMode.txt");
            // ひとまずファイルの中身は見ないので、適当な値を入れておく
            string value = "abc";
            FsUtil.CreateRecordFile(debugFilePath, value);
        }

        private bool IsRomImportContent(List<ContentMetaInfoParam> inInfoList)
        {
            if (m_SyncListView.IsDeserialized == false)
            {
                // Sync リストが設定されていない場合は常に取り込む
                return true;
            }

            foreach (var info in inInfoList)
            {
                if (m_SyncListView.IsMatchContent(info))
                {
                    // Sync リストが設定されていて、リストに記載されているコンテンツに該当すると取り込む
                    return true;
                }
            }

            // 上記の条件に引っかからないコンテンツは取り込まない
            return false;
        }

        private void CheckLegalInfoFile(SettingParams inParam)
        {
            var targetDirPath = Path.Combine(this.GetExecutingAppDirPath(), CommonParams.LegalInfoDirectoryName);
            var targetLegalInfoPath = Path.Combine(targetDirPath, (inParam.ApplicationId + ".zip"));
            // "[ApplicationId値].zip"のファイルが存在するかどうかを確認する
            if (File.Exists(targetLegalInfoPath))
            {
                if (isVerbose)
                {
                    Console.WriteLine("Set LegalInformation : Path = {0}", targetLegalInfoPath);
                }
                inParam.LegalInfoFilePath = targetLegalInfoPath;
            }
        }

        // (SIGLO-71102) 確認用のコンテンツ一覧情報ファイルのパス文字列を生成する
        private string MakeOutputContentInfoListFilePath(string nspOutputPath)
        {
            var nspFileName = Path.GetFileNameWithoutExtension(nspOutputPath);
            var outputFileName = string.Format("InstallContentInfoList_{0}.csv", nspFileName);
            var outputDirPath = Path.GetDirectoryName(nspOutputPath);
            return Path.Combine(outputDirPath, outputFileName);
        }
    }
}
