﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Xml;

namespace QCIT_Maker
{
    internal class MetaEditor
    {
        /// <summary>
        /// オーサリングツールに渡すための nmeta ファイルを作成します
        /// </summary>
        /// <param name="settingData">メタ情報のパラメータ</param>
        /// <returns>作成したファイルパス</returns>
        public void CreateNmetaFile(SettingParams settingData)
        {
            // SettingParams のデータ内容に沿った XML ファイルを作成する
            try
            {
                // XMLファイルの中身は nmeta ファイルの書式に依存する
                var nmetaXmlDoc = new XmlDocument();
                var head = nmetaXmlDoc.CreateXmlDeclaration("1.0", "utf-8", null);
                nmetaXmlDoc.AppendChild(head);

                var root = nmetaXmlDoc.CreateElement(CommonParams.NmetaRootName);
                nmetaXmlDoc.AppendChild(root);

                {
                    var core = nmetaXmlDoc.CreateElement("Core");
                    root.AppendChild(core);

                    {
                        var idElement = nmetaXmlDoc.CreateElement("ApplicationId");
                        idElement.InnerText = settingData.ApplicationId;
                        core.AppendChild(idElement);
                    }

                    var app = nmetaXmlDoc.CreateElement("Application");
                    root.AppendChild(app);

                    {
                        var titleNode = nmetaXmlDoc.CreateElement("Title");
                        app.AppendChild(titleNode);

                        var langNode = nmetaXmlDoc.CreateElement("Language");
                        langNode.InnerText = settingData.SupportedLanguage;
                        titleNode.AppendChild(langNode);

                        var titleNameText = string.Format("QCIT_{0}_{1}", settingData.ApplicationId, settingData.ReleaseVersion);
                        var nameNode = nmetaXmlDoc.CreateElement("Name");
                        nameNode.InnerText = titleNameText;
                        titleNode.AppendChild(nameNode);

                        // ひとまず Publisher は Nintendo 固定
                        var pubNode = nmetaXmlDoc.CreateElement("Publisher");
                        pubNode.InnerText = "Nintendo";
                        titleNode.AppendChild(pubNode);
                    }

                    // ラムダ式で Application へのエレメント追加処理を関数化しておく
                    Action<string, string> addAppElementAction = (string inElementName, string inSetValue) =>
                    {
                        if (inSetValue != null)
                        {
                            var element = nmetaXmlDoc.CreateElement(inElementName);
                            element.InnerText = inSetValue;
                            app.AppendChild(element);
                        }
                    };

                    // ReleaseVersion を追加する
                    addAppElementAction("ReleaseVersion", settingData.ReleaseVersion);

                    // SupportedLanguage を追加する
                    addAppElementAction("SupportedLanguage", settingData.SupportedLanguage);

                    // DisplayVersion を追加する
                    var dvText = string.Format("{0}.{1}", settingData.ReleaseVersion, settingData.PrivateVersion);
                    addAppElementAction("DisplayVersion", dvText);

                    // LogoType を追加する
                    addAppElementAction("LogoType", "Nintendo");

                    // (SIGLO-68974) ソフトリーガル情報が設定されていれば LegalInformationFilePath を追加する
                    if (string.IsNullOrEmpty(settingData.LegalInfoFilePath) == false)
                    {
                        addAppElementAction("LegalInformationFilePath", settingData.LegalInfoFilePath);
                    }

                    if (settingData.IsDebugMode == false)
                    {
                        addAppElementAction("Screenshot", "Deny");
                        addAppElementAction("Attribute", "RetailInteractiveDisplay");
                    }

                    // (SIGLO-68973) レーティング設定を追加する
                    foreach (var rating in settingData.RatingList)
                    {
                        var rtg = nmetaXmlDoc.CreateElement("Rating");
                        app.AppendChild(rtg);

                        var org = nmetaXmlDoc.CreateElement("Organization");
                        org.InnerText = rating;
                        rtg.AppendChild(org);

                        var age = nmetaXmlDoc.CreateElement("Age");
                        // Age の値は 0 固定にする
                        age.InnerText = "0";
                        rtg.AppendChild(age);
                    }

                    // (SIGLO-71368) KeyGeneration 関連設定の追加
                    {
                        // ひとまず固定値を設定する
                        var keyGen = nmetaXmlDoc.CreateElement("KeyGeneration");
                        keyGen.InnerText = "2";
                        root.AppendChild(keyGen);

                        var keepGen = nmetaXmlDoc.CreateElement("KeepGeneration");
                        keepGen.InnerText = "true";
                        root.AppendChild(keepGen);
                    }

                    // (SIGLO-71368) CardSpec 設定の追加
                    {
                        var cardSpec = nmetaXmlDoc.CreateElement("CardSpec");
                        root.AppendChild(cardSpec);

                        var flags = nmetaXmlDoc.CreateElement("LaunchFlags");
                        // 固定値 3 (auto-boot かつ history-erase) を設定
                        flags.InnerText = "3";
                        cardSpec.AppendChild(flags);
                    }
                }

                nmetaXmlDoc.Save(settingData.NmetaFilePath);
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        /// <summary>
        /// ファイルパス関連の設定をメタファイルに追記します
        /// スレッドセーフ化のため、作業ディレクトリにファイルまたはディレクトリをコピーする
        /// </summary>
        /// <param name="inSettingParam">設定パラメータ</param>
        public void AddFilePathMetaParameter(SettingParams inSettingParam)
        {
            try
            {
                var xmlDoc = new XmlDocument();
                // metaファイルの読み込み
                xmlDoc.Load(inSettingParam.NmetaFilePath);

                var metaNode = xmlDoc.SelectSingleNode("/" + CommonParams.NmetaRootName);
                if (metaNode == null)
                {
                    // ありえないと思うが Meta ノードが存在しない場合は以降の処理は無意味なのですぐに抜ける
                    return;
                }

                var appNode = metaNode.SelectSingleNode("Application");
                if (appNode == null)
                {
                    // Application ノードが存在しない場合は作成して、以降の処理を行う
                    var appElement = xmlDoc.CreateElement("Application");
                    metaNode.AppendChild(appElement);
                    appNode = appElement;
                }

                // アイコン情報を指定
                var iconNodeList = appNode.SelectNodes("Icon");
                if (iconNodeList == null || iconNodeList.Count == 0)
                {
                    // デフォルト設定のアイコン画像用
                    {
                        var iconNode = xmlDoc.CreateElement("Icon");
                        appNode.AppendChild(iconNode);

                        var langNode = xmlDoc.CreateElement("Language");
                        langNode.InnerText = inSettingParam.IconSetting.Language;
                        iconNode.AppendChild(langNode);

                        var pathNode = xmlDoc.CreateElement("IconPath");
                        pathNode.InnerText = inSettingParam.IconSetting.IconFilePath;
                        iconNode.AppendChild(pathNode);
                    }
                }

                // metaファイルの上書き保存
                xmlDoc.Save(inSettingParam.NmetaFilePath);
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }
        }

        public ContentMetaInfoParam GetContentMetaInfo(string inCnmtXmlPath)
        {
            var infoParam = new ContentMetaInfoParam();
            try
            {
                var xmlDoc = new XmlDocument();
                xmlDoc.Load(inCnmtXmlPath);

                var cmNode = xmlDoc.SelectSingleNode("ContentMeta");
                if (cmNode == null)
                {
                    // あり得ないと思うが。。
                    return null;
                }

                // ラムダ式で Application へのエレメント追加処理を関数化しておく
                Func<string, string> ContentMetaGetFunc = (string inElementName) =>
                {
                    var node = cmNode.SelectSingleNode(inElementName);
                    return (node != null) ? node.InnerText : string.Empty;
                };

                infoParam.Type = ContentMetaGetFunc("Type");
                infoParam.Id = ContentMetaGetFunc("Id");
                // ひとまず ID と ApplicationID は同じ値に設定しておく (後からタイプ別に変更する)
                infoParam.ApplicationId = infoParam.Id;
                infoParam.Version = ContentMetaGetFunc("Version");
                infoParam.Digest = ContentMetaGetFunc("Digest");

                if (infoParam.Type == CommonParams.TypeApplication || infoParam.Type == CommonParams.TypePatch)
                {
                    ulong size = 0;
                    {
                        var programSizeNode = cmNode.SelectSingleNode("Content[Type=\"Program\"]/Size");
                        if (programSizeNode != null)
                        {
                            ulong num = 0;
                            var result = ulong.TryParse(programSizeNode.InnerText, out num);
                            if (result)
                            {
                                size += num;
                            }
                        }
                    }
                    {
                        var controlSizeNode = cmNode.SelectSingleNode("Content[Type=\"Control\"]/Size");
                        if (controlSizeNode != null)
                        {
                            ulong num = 0;
                            var result = ulong.TryParse(controlSizeNode.InnerText, out num);
                            if (result)
                            {
                                size += num;
                            }
                        }
                    }
                    infoParam.DataSize = size.ToString();
                }

                if (infoParam.Type == CommonParams.TypeAddOnContent)
                {
                    infoParam.Tag = ContentMetaGetFunc("Tag");
                    infoParam.Index = ContentMetaGetFunc("Index");
                    infoParam.ApplicationId = ContentMetaGetFunc("ApplicationId");

                    var contentDataSizeNode = cmNode.SelectSingleNode("Content[Type=\"Data\"]/Size");
                    if (contentDataSizeNode != null)
                    {
                        infoParam.DataSize = contentDataSizeNode.InnerText;
                    }
                }
                else if (infoParam.Type == CommonParams.TypePatch)
                {
                    infoParam.ApplicationId = ContentMetaGetFunc("ApplicationId");
                }
            }
            catch (Exception err)
            {
                Console.Error.WriteLine(err.StackTrace);
                Console.Error.WriteLine(err.Message);
                throw err;
            }

            return infoParam;
        }
    }
}
