﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.Contracts;
using Nintendo.Foundation.IO;
using Nintendo.Foundation.Audio;

namespace VibrationConverterConsole
{
    public class CommandLineParameters
    {
        public enum FileFormat
        {
            None,
            Wav,
            Bnvib,
            Amfmc,
            Nvib,
        }

        [CommandLineValue(0, ValueName = "Input", Description = "InputFile", DescriptionConverterName = "LocalizeDescription")]
        public string InputFilePath { get; set; }

        [CommandLineOption("input-format", ValueName = "Format", Description = "InputFileFormat", DescriptionConverterName = "LocalizeDescription", DefaultValue = FileFormat.None, IsHidden = true)]
        public FileFormat InputFileFormat { get; set; }

        [CommandLineOption('o', "output", ValueName = "Output", Description = "OutputFile", DescriptionConverterName = "LocalizeDescription", DefaultValue = "")]
        public string OutputFilePath { get; set; }

        [CommandLineOption('v', "verbose", Description = "Verbose", DescriptionConverterName = "LocalizeDescription", DefaultValue = false)]
        public bool IsVerbose { get; set; }

        [CommandLineOption("in-param", Description = "InputParamFile", DescriptionConverterName = "LocalizeDescription", DefaultValue = "")]
        public string InputParameterFilePath { get; set; }

        [CommandLineOption("out-param", Description = "OutputParamFile", DescriptionConverterName = "LocalizeDescription", DefaultValue = "")]
        public string OutputParameterFilePath { get; set; }

        [CommandLineOption("out-nvib", Description = "OutputNvibFile", DescriptionConverterName = "LocalizeDescription", DefaultValue = "", IsHidden = true)]
        public string OutputNvibFilePath { get; set; }

        [CommandLineOption("out-wav", Description = "OutputWaveFile", DescriptionConverterName = "LocalizeDescription", DefaultValue = "")]
        public string OutputWaveFilePath { get; set; }

        [CommandLineOption('n', "no-out-param", Description = "SkipParamFileOutput", DescriptionConverterName = "LocalizeDescription", DefaultValue = false)]
        public bool IsNvibcpOutputSkipped { get; set; }

        [CommandLineOption('e', "enable-auto-amp", Description = "EnableAutoAmp", DescriptionConverterName = "LocalizeDescription", DefaultValue = false)]
        public bool IsAutoAmpForceEnabled
        {
            get { return m_IsAutoAmpForceEnabled; }
            set
            {
                m_IsAutoAmpForceEnabled = value;
                if (m_IsAutoAmpForceEnabled)
                {
                    m_IsAutoAmpForceDisabled = false;
                }
            }
        }

        [CommandLineOption('d', "disable-auto-amp", Description = "DisableAutoAmp", DescriptionConverterName = "LocalizeDescription", DefaultValue = false)]
        public bool IsAutoAmpForceDisabled
        {
            get { return m_IsAutoAmpForceDisabled; }
            set
            {
                m_IsAutoAmpForceDisabled = value;
                if (m_IsAutoAmpForceDisabled)
                {
                    m_IsAutoAmpForceEnabled = false;
                }
            }
        }

        private bool m_IsAutoAmpForceEnabled = false;
        private bool m_IsAutoAmpForceDisabled = false;

        public static string LocalizeDescription(string description, string argName)
        {
            return Resources.CommandLineHelpDescriptions.ResourceManager.GetString(description);
        }
    }

    class Program
    {
        static CommandLineParameters m_Parameters;

        static void Main(string[] args)
        {
            CommandLineParser commandLineParser = CommandLineParser.Default;
            bool parseResult;
            try
            {
                parseResult = commandLineParser.ParseArgs(args, out m_Parameters);
            }
            catch
            {
                return;
            }

            if (parseResult == true)
            {
                AmpFreqExtractor.MessageQueue messageQueue = new AmpFreqExtractor.MessageQueue();
                AmpFreqExtractor.Executor executor = new AmpFreqExtractor.Executor(messageQueue);

                System.Threading.Thread executeThread = new System.Threading.Thread(Execute);
                executeThread.CurrentCulture = System.Globalization.CultureInfo.InvariantCulture;
                executeThread.Priority = System.Threading.ThreadPriority.BelowNormal;
                executeThread.Start(executor);

                while (!messageQueue.IsFinished())
                {
                    messageQueue.WaitMessage();
                    string[] msgArray = messageQueue.Read();
                    if (m_Parameters.IsVerbose)
                    {
                        foreach (string msg in msgArray)
                        {
                            Console.WriteLine("[Log] " + msg);
                        }
                    }
                    msgArray = messageQueue.ReadError();
                    foreach (string msg in msgArray)
                    {
                        Console.WriteLine("[Error] " + msg);
                    }
                }
                executeThread.Join();
            }
            else
            {
                return;
            }
        }

        static void Execute(Object obj)
        {
            AmpFreqExtractor.Executor executor = (AmpFreqExtractor.Executor)obj;
            executor.Execute(m_Parameters);
            executor.m_MessageQueue.Finish();
        }
    }
}
