﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace VibrationConverterConsole.Filter
{
    public interface DegreeFunction
    {
        double GetLogPower(double degree);
    }
    public abstract class TransferFunction : DegreeFunction
    {
        public abstract Complex GetValue(Complex z);
        public double GetLogPower(double degree)
        {
            Complex z = Complex.PolarCoords(1.0, degree);
            Complex h = this.GetValue(z);
            h = h * h;
            double y = Math.Log(h.Abs + 0.000000000001, 10.0);
            return y;
        }
    }

    public enum FrequencyResponse
    {
        HighFlag = 1,
        LowFlag = 2,
        ShelfFlag = 4,
        PassFlag = 8,
        HighShelf = HighFlag | ShelfFlag,
        LowShelf = LowFlag | ShelfFlag,
        HighPass = HighFlag | PassFlag,
        LowPass = LowFlag | PassFlag
    }
    public enum FilterFamily
    {
        Butterworth,
        Chebyshev,
    }

    public abstract class ShelfFilterFunctionBase : TransferFunction
    {
        private FrequencyResponse m_FrequencyResponse = FrequencyResponse.LowPass;
        public FrequencyResponse FrequencyResponse
        {
            get { return m_FrequencyResponse; }
            set
            {
                m_FrequencyResponse = value;
                UpdateCoeffs();
            }
        }
        private int m_Order = 8;
        public int Order
        {
            get { return m_Order; }
            set
            {
                if (value > 0 && value % 2 == 0)
                {
                    m_Order = value;
                    UpdateCoeffs();
                }
            }
        }
        private double m_CutoffLength = 1024.0;
        public double CutoffLength
        {
            get { return m_CutoffLength; }
            set
            {
                if (value > 0.0)
                {
                    m_CutoffLength = value;
                    UpdateCoeffs();
                }
            }
        }
        private double m_Attenuation = -20.0;
        public double Attenuation
        {
            get { return m_Attenuation; }
            set
            {
                if (value <= 0.0)
                {
                    m_Attenuation = value;
                    UpdateCoeffs();
                }
            }
        }

        private double[,] m_CoeffC;
        private double[,] m_CoeffD;
        private double[, ,] m_Coeff;
        public double GetCoeff(int position, int z, int n)
        {
            return m_Coeff[position, z, n];
        }

        protected delegate void FuncUpdateCD(int n, double alpha, double[,] c, double[,] d);
        protected FuncUpdateCD m_FuncUpdateCD;
        protected void UpdateCoeffs()
        {
            m_CoeffC = new double[2, m_Order / 2];
            m_CoeffD = new double[2, m_Order / 2];
            m_Coeff = new double[2, 3, m_Order / 2];

            double beta = 0.0;
            if (m_FrequencyResponse.HasFlag(FrequencyResponse.ShelfFlag))
            {
                beta = Math.Pow(10.0, m_Attenuation / 10.0);
            }
            m_FuncUpdateCD(m_Order, beta, m_CoeffC, m_CoeffD);

            double degree = 2.0 * Math.PI / m_CutoffLength;
            double t = Math.Tan(degree / 2.0);
            if (m_FrequencyResponse.HasFlag(FrequencyResponse.LowFlag)) t = 1.0 / t;
            double tt = t * t;
            double sgn = m_FrequencyResponse.HasFlag(FrequencyResponse.HighFlag) ? 1 : -1;

            for (int k = 0; k < m_Order / 2; k++)
            {
                double theta = Math.PI * (m_Order - 2.0 * k - 1.0) / (2.0 * m_Order);
                double cos = Math.Cos(theta);
                for (int i = 0; i < 2; i++)
                {
                    double a = m_CoeffC[i, k] * tt;
                    double b = 2.0 * m_CoeffD[i, k] * cos * t;
                    m_Coeff[i, 0, k] = a + b + 1.0;
                    m_Coeff[i, 1, k] = sgn * 2.0 * (a - 1.0);
                    m_Coeff[i, 2, k] = a - b + 1.0;
                }
            }
        }

        public override Complex GetValue(Complex z)
        {
            Complex h = 1.0;
            Complex zInverse = 1.0 / z;
            Complex zInverse2 = 1.0 / (z * z);

            for (int k = 0; k < m_Order / 2; k++)
            {
                Complex numerator = m_Coeff[0, 0, k] + m_Coeff[0, 1, k] * zInverse + m_Coeff[0, 2, k] * zInverse2;
                Complex denominator = m_Coeff[1, 0, k] + m_Coeff[1, 1, k] * zInverse + m_Coeff[1, 2, k] * zInverse2;
                h *= numerator;
                h /= denominator;
            }

            return h;
        }

        public BiQuadFilter GetBiquadFilterElement(int k, double amp = 1.0)
        {
            float[,] c = new float[2, 3];
            for (int i = 0; i < 3; i++)
            {
                c[0, i] = (float)(amp * m_Coeff[0, i, k] / m_Coeff[1, 0, k]);
                c[1, i] = (float)(-m_Coeff[1, i, k] / m_Coeff[1, 0, k]);
            }
            BiQuadFilter f = new BiQuadFilter(c[0, 0], c[0, 1], c[0, 2], c[1, 1], c[1, 2]);
            return f;
        }
        public SerialFilter<BiQuadFilter> GetSerialBiquadFilter(double amp = 1.0)
        {
            double subAmp = Math.Pow(amp, 2.0 / m_Order);

            SerialFilter<BiQuadFilter> s = new SerialFilter<BiQuadFilter>();
            for (int k = 0; k < m_Order / 2; k++)
            {
                s.AddFilter(GetBiquadFilterElement(k, subAmp));
            }
            return s;
        }
    }

    public class ShelfFilterFunction : ShelfFilterFunctionBase
    {
        private FilterFamily m_FilterFamily = FilterFamily.Butterworth;
        public FilterFamily FilterFamily
        {
            get { return m_FilterFamily; }
            set
            {
                m_FilterFamily = value;
                switch (m_FilterFamily)
                {
                    case FilterFamily.Butterworth:
                        m_FuncUpdateCD = UpdateCD_Butterworth;
                        break;
                    case FilterFamily.Chebyshev:
                        m_FuncUpdateCD = this.UpdateCD_Chebyshev;
                        break;
                }
                UpdateCoeffs();
            }
        }

        private double m_Epsilon = 1.0;
        public double Epsilon
        {
            get { return m_Epsilon; }
            set
            {
                if (value > 0.0)
                {
                    m_Epsilon = value;
                    UpdateCoeffs();
                }
            }
        }

        private static void UpdateCD_Butterworth(int n, double beta, double[,] c, double[,] d)
        {
            //double r = alpha / (1.0 + alpha);
            double r = beta;
            double valueC = Math.Pow(r, 1.0 / n);
            double valueD = Math.Pow(r, 0.5 / n);
            for (int k = 0; k < n / 2; k++)
            {
                c[0, k] = valueC;
                d[0, k] = valueD;
                c[1, k] = 1.0;
                d[1, k] = 1.0;
            }
        }
        private static double GetR(int n, double x)
        {
            double y = 1.0 + Math.Sqrt(1.0 + x * x);
            double p = 2.0 * Math.Pow(x * y, 1.0 / n);
            double q = Math.Pow(y, 2.0 / n) - Math.Pow(x, 2.0 / n);
            double r = p / q;
            return r;
        }
        private static void SetCoeff(double r, double sin, out double c, out double d)
        {
            double denominator = 1.0 + r * r * sin * sin;
            c = r * r / denominator;
            d = r / denominator;
        }
        private void UpdateCD_Chebyshev(int n, double beta, double[,] c, double[,] d)
        {
            double x = m_Epsilon;
            //double xx = m_Epsilon * Math.Sqrt(alpha / (1.0 + m_Epsilon * m_Epsilon + alpha));
            double xx = m_Epsilon * Math.Sqrt(beta / (1.0 + (1.0 - beta) * m_Epsilon * m_Epsilon));
            double r = GetR(n, x);
            double rr = GetR(n, xx);

            for (int k = 0; k < n / 2; k++)
            {
                double theta = Math.PI * (n - 2.0 * k - 1.0) / (2.0 * n);
                double sin = Math.Sin(theta);
                SetCoeff(rr, sin, out c[0, k], out d[0, k]);
                SetCoeff(r, sin, out c[1, k], out d[1, k]);
            }
        }

        public ShelfFilterFunction(FilterFamily f, FrequencyResponse r)
        {
            this.FilterFamily = f;
            this.FrequencyResponse = r;
            UpdateCoeffs();
        }
    }
}
