﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace VibrationConverterConsole.Filter
{
    public interface IFilter
    {
        float GetOutput(float input);
    }

    public interface IFilterElement<FilterType>
        where FilterType : IFilter
    {
        FilterType Filter { get; set; }
    }
    public interface IFilterElement : IFilterElement<IFilter> { }
    public abstract class CompoundFilterBase<ElementType, FilterType> : IFilter
        where ElementType : IFilterElement<FilterType>
        where FilterType : IFilter
    {
        protected System.Threading.Mutex m_Mutex = null;
        protected List<ElementType> m_Elements = null;

        protected CompoundFilterBase()
        {
            m_Mutex = new System.Threading.Mutex();
            m_Elements = new List<ElementType>();
        }
        public virtual void AddElement(params ElementType[] elements)
        {
            m_Mutex.WaitOne();
            foreach (ElementType element in elements)
            {
                m_Elements.Add(element);
            }
            m_Mutex.ReleaseMutex();
        }

        public int NumElements { get { return m_Elements.Count; } }
        public ElementType ElementAt(int index) { return m_Elements[index]; }
        public ElementType ElementCorrespoindingTo(IFilter filter) { return m_Elements.Find(e => (IFilter)(e.Filter) == filter); }
        public IEnumerator<ElementType> GetEnumerator() { return m_Elements.GetEnumerator(); }

        public abstract float GetOutput(float input);
    }
    public abstract class CompoundFilterBase<ElementType> : CompoundFilterBase<ElementType, IFilter>
        where ElementType : IFilterElement<IFilter> { }

    public class SerialFilterElement<FilterType> : IFilterElement<FilterType>
        where FilterType : IFilter
    {
        public FilterType Filter { get; set; }
        public bool Enabled;
        public SerialFilterElement(FilterType filter, bool enabled = true)
        {
            Filter = filter;
            Enabled = enabled;
        }
    }
    public class SerialFilterElement : SerialFilterElement<IFilter>
    {
        public SerialFilterElement(IFilter filter, bool enabled = true) :
            base(filter, enabled) { }
    }
    public class SerialFilter<FilterType> : CompoundFilterBase<SerialFilterElement<FilterType>, FilterType>
        where FilterType : IFilter
    {
        public SerialFilter(params FilterType[] filters)
        {
            AddFilter(filters);
        }
        public void AddFilter(params FilterType[] filters)
        {
            m_Mutex.WaitOne();
            foreach (FilterType filter in filters)
            {
                m_Elements.Add(new SerialFilterElement<FilterType>(filter, true));
            }
            m_Mutex.ReleaseMutex();
        }

        public override float GetOutput(float input)
        {
            m_Mutex.WaitOne();
            float output = input;
            foreach (SerialFilterElement<FilterType> e in m_Elements)
            {
                if (e.Enabled)
                {
                    output = e.Filter.GetOutput(output);
                }
            }
            m_Mutex.ReleaseMutex();
            return output;
        }
    }
    public class SerialFilter : SerialFilter<IFilter>
    {
        public SerialFilter(params IFilter[] filters) : base(filters) { }
    }

    public class ParallelFilterElement<FilterType> : IFilterElement<FilterType>
        where FilterType : IFilter
    {
        public FilterType Filter { get; set; }
        public float Amplitude;
        public bool Enabled;
        public ParallelFilterElement(FilterType filter, float amplitude = 1.0f, bool enabled = true)
        {
            Filter = filter;
            Amplitude = amplitude;
            Enabled = enabled;
        }
    }
    public class ParallelFilterElement : ParallelFilterElement<IFilter>
    {
        public ParallelFilterElement(IFilter filter, float amplitude = 1.0f, bool enabled = true)
            : base(filter, amplitude, enabled)
        { }
    }
    public class ParallelFilter<FilterType> : CompoundFilterBase<ParallelFilterElement<FilterType>, FilterType>
        where FilterType : IFilter
    {
        public ParallelFilter(params FilterType[] filters)
        {
            AddFilter(filters);
        }
        public void AddFilter(FilterType filter, float amplitude)
        {
            m_Mutex.WaitOne();
            m_Elements.Add(new ParallelFilterElement<FilterType>(filter, amplitude, true));
            m_Mutex.ReleaseMutex();
        }
        public void AddFilter(params FilterType[] filters)
        {
            m_Mutex.WaitOne();
            foreach (FilterType filter in filters)
            {
                m_Elements.Add(new ParallelFilterElement<FilterType>(filter, 1.0f, true));
            }
            m_Mutex.ReleaseMutex();
        }

        public void FlattenAmplitudes()
        {
            m_Mutex.WaitOne();
            int numEnabledFilters = m_Elements.Count(e => e.Enabled);
            if (numEnabledFilters > 0)
            {
                float flattenedAmplitude = 1.0f / numEnabledFilters;
                foreach (ParallelFilterElement<FilterType> e in m_Elements)
                {
                    if (e.Enabled)
                    {
                        e.Amplitude = flattenedAmplitude;
                    }
                }
            }
            m_Mutex.ReleaseMutex();
        }
        public override float GetOutput(float input)
        {
            float output = 0.0f;
            m_Mutex.WaitOne();
            foreach (ParallelFilterElement<FilterType> e in m_Elements)
            {
                if (e.Enabled)
                {
                    output += e.Amplitude * e.Filter.GetOutput(input);
                }
            }
            m_Mutex.ReleaseMutex();
            return output;
        }
    }
    public class ParallelFilter : ParallelFilter<IFilter>
    {
        public ParallelFilter(params IFilter[] filters) : base(filters) { }
    }

    // FilterWithBuffer の内部で使うリングバッファ
    public class RingBuffer<T> : ICloneable where T : struct
    {
        private T[] m_Buf = null;
        private int m_Offset = 0;
        public RingBuffer(int bufLength)
        {
            m_Buf = new T[bufLength];
            for (int i = 0; i < m_Buf.Length; i++)
            {
                m_Buf[i] = default(T);
            }
        }
        private int ConvertIndex(int index)
        {
            return (index + m_Offset + m_Buf.Length) % m_Buf.Length;
        }
        public T this[int index]
        {
            get { return m_Buf[ConvertIndex(index)]; }
            set { m_Buf[ConvertIndex(index)] = value; }
        }
        public void MovePositionForward()
        {
            m_Offset = (m_Offset + 1) % m_Buf.Length;
        }
        public int Length { get { return m_Buf.Length; } }
        public object Clone()
        {
            RingBuffer<T> clone = new RingBuffer<T>(m_Buf.Length);
            for (int i = 0; i < m_Buf.Length; i++)
            {
                clone.m_Buf[i] = this.m_Buf[i];
            }
            clone.m_Offset = this.m_Offset;
            return clone;
        }
    }

    public abstract class FilterWithBuffer : IFilter, ICloneable
    {
        protected RingBuffer<float> m_InBuf = null;
        protected RingBuffer<float> m_OutBuf = null;

        protected FilterWithBuffer(int inputBufLength, int outputBufLength)
        {
            m_InBuf = new RingBuffer<float>(inputBufLength + 1);
            if (outputBufLength == 0) outputBufLength = 1;
            m_OutBuf = new RingBuffer<float>(outputBufLength);
        }
        public virtual float GetOutput(float input)
        {
            m_InBuf.MovePositionForward();
            m_OutBuf.MovePositionForward();
            m_InBuf[0] = input;
            float output = CalcOutput();
            m_OutBuf[0] = output;
            return output;
        }
        protected abstract float CalcOutput();

        public abstract object Clone();
        protected void CopyBuffersToClone(FilterWithBuffer clone)
        {
            clone.m_InBuf = (RingBuffer<float>)m_InBuf.Clone();
            clone.m_OutBuf = (RingBuffer<float>)m_OutBuf.Clone();
        }
    }
    public class BiQuadFilter : FilterWithBuffer
    {
        public virtual float CoeffIn0 { get; set; }
        public virtual float CoeffIn1 { get; set; }
        public virtual float CoeffIn2 { get; set; }
        public virtual float CoeffOut1 { get; set; }
        public virtual float CoeffOut2 { get; set; }
        public BiQuadFilter()
            : base(2, 2)
        {
            CoeffIn0 = 1.0f;
            CoeffIn1 = 0.0f;
            CoeffIn2 = 0.0f;
            CoeffOut1 = 0.0f;
            CoeffOut1 = 0.0f;
        }
        public BiQuadFilter(float coeffIn0, float coeffIn1, float coeffIn2, float coeffOut1, float coeffOut2)
            : base(2, 2)
        {
            CoeffIn0 = coeffIn0;
            CoeffIn1 = coeffIn1;
            CoeffIn2 = coeffIn2;
            CoeffOut1 = coeffOut1;
            CoeffOut2 = coeffOut2;
        }
        protected override float CalcOutput()
        {
            float output = 0.0f;
            output += CoeffIn0 * m_InBuf[0];
            output += CoeffIn1 * m_InBuf[-1];
            output += CoeffIn2 * m_InBuf[-2];
            output += CoeffOut1 * m_OutBuf[-1];
            output += CoeffOut2 * m_OutBuf[-2];
            return output;
        }

        public override object Clone()
        {
            BiQuadFilter clone = new BiQuadFilter(CoeffIn0, CoeffIn1, CoeffIn2, CoeffOut1, CoeffOut2);
            CopyBuffersToClone(clone);
            return clone;
        }
    }
}
