﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace VibrationConverterConsole.Filter
{
    public struct Complex
    {
        public double Real;
        public double Imaginary;
        public Complex(double real, double imaginary)
        {
            Real = real;
            Imaginary = imaginary;
        }
        public double Abs
        {
            get
            {
                return Math.Sqrt(Real * Real + Imaginary * Imaginary);
            }
            set
            {
                double currentAbs = Abs;
                if (Abs > 0.0)
                {
                    double ratio = value / currentAbs;
                    Real *= ratio;
                    Imaginary *= ratio;
                }
            }
        }
        public double Arg
        {
            get
            {
                return Math.Atan2(Imaginary, Real);
            }
            set
            {
                double currentAbs = Abs;
                Real = Abs * Math.Cos(value);
                Imaginary = Abs * Math.Sin(value);
            }
        }
        public Complex Conjugate
        {
            get
            {
                return new Complex(Real, -Imaginary);
            }
        }
        public static Complex RectangularCoords(double x, double y)
        {
            return new Complex(x, y);
        }
        public static Complex PolarCoords(double abs, double arg)
        {
            return new Complex(abs * Math.Cos(arg), abs * Math.Sin(arg));
        }
        public static implicit operator Complex(double a)
        {
            return new Complex(a, 0.0);
        }
        public static Complex operator +(Complex a)
        {
            return a;
        }
        public static Complex operator -(Complex a)
        {
            return new Complex(-a.Real, -a.Imaginary);
        }
        public static Complex operator +(Complex a, Complex b)
        {
            return new Complex(a.Real + b.Real, a.Imaginary + b.Imaginary);
        }
        public static Complex operator -(Complex a, Complex b)
        {
            return new Complex(a.Real - b.Real, a.Imaginary - b.Imaginary);
        }
        public static Complex operator *(Complex a, Complex b)
        {
            return new Complex(
                a.Real * b.Real - a.Imaginary * b.Imaginary,
                a.Real * b.Imaginary + a.Imaginary * b.Real);
        }
        public static Complex operator /(Complex a, Complex b)
        {
            double denominator = b.Real * b.Real + b.Imaginary * b.Imaginary;
            return new Complex(
                (a.Real * b.Real + a.Imaginary * b.Imaginary) / denominator,
                (a.Imaginary * b.Real - a.Real * b.Imaginary) / denominator);
        }
        public static bool operator ==(Complex a, Complex b)
        {
            return (a.Real == b.Real) && (a.Imaginary == b.Imaginary);
        }
        public static bool operator !=(Complex a, Complex b)
        {
            return (a.Real != b.Real) || (a.Imaginary != b.Imaginary);
        }
        public override int GetHashCode()
        {
            return Real.GetHashCode() ^ Imaginary.GetHashCode();
        }
        public override bool Equals(object obj)
        {
            if (obj == null || this.GetType() != obj.GetType())
            {
                return false;
            }
            else
            {
                return (this == (Complex)obj);
            }
        }
        public override string ToString()
        {
            if (Real == 0.0)
            {
                if (Imaginary == 0.0)
                {
                    return "0";
                }
                else if (Imaginary == 1.0)
                {
                    return "i";
                }
                else if (Imaginary == -1.0)
                {
                    return "-i";
                }
                else
                {
                    return Imaginary.ToString() + "i";
                }
            }
            else
            {
                if (Imaginary == 0.0)
                {
                    return Real.ToString();
                }
                else if (Imaginary == 1.0)
                {
                    return Real.ToString() + "+i";
                }
                else if (Imaginary == -1.0)
                {
                    return Real.ToString() + "-i";
                }
                else if (Imaginary < 0.0)
                {
                    return Real.ToString() + Imaginary.ToString() + "i";
                }
                else
                {
                    return Real.ToString() + "+" + Imaginary.ToString() + "i";
                }
            }
        }
        public string ToString(string format)
        {
            if (Imaginary >= 0.0)
            {
                return Real.ToString(format) + "+" + Imaginary.ToString(format) + "i";
            }
            else
            {
                return Real.ToString(format) + "-" + (-Imaginary).ToString(format) + "i";
            }
        }
        public static Complex I = new Complex(0.0, 1.0);
    }
}
