﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.Contracts;
using VibrationConverterConsole.BnvibIO;

namespace VibrationConverterConsole.AmpFreqExtractor
{
    public class GradualSineWaveGenerator
    {
        public readonly int SamplingRate;
        private double m_Amplitude = 0.0;
        private double m_Frequency = 0.0;
        private double m_Angle = 0.0;

        public GradualSineWaveGenerator(int samplingRate, double baseFrequency)
        {
            Ensure.Argument.True(0 < samplingRate);
            Ensure.Argument.True(0.0 < baseFrequency);

            SamplingRate = samplingRate;
            m_Frequency = baseFrequency;
        }

        public float[] GenerateWave(int sampleCount, double amplitude, double frequency)
        {
            Ensure.Argument.True(0 < sampleCount);
            Ensure.Argument.True(0.0 <= amplitude && amplitude <= 1.0);
            Ensure.Argument.True(0.0 < frequency);

            double ampStep = (amplitude - m_Amplitude) / sampleCount;
            double freqStep = (frequency - m_Frequency) / sampleCount;

            float[] wave = new float[sampleCount];
            for (int i = 0; i < sampleCount; i++)
            {
                m_Amplitude += ampStep;
                m_Frequency += freqStep;

                double angleStep = m_Frequency * 2.0 * Math.PI / SamplingRate;
                m_Angle += angleStep;

                wave[i] = (float)(m_Amplitude * Math.Sin(m_Angle));
            }

            m_Amplitude = amplitude;
            m_Frequency = frequency;

            return wave;
        }
    }

    public class WaveReconstructor
    {
        public const int DefaultSamplingRate = 8000;

        static public float[] ReconstructWave(BnvibFile bnvibFile, int samplingRate = DefaultSamplingRate)
        {
            Ensure.Argument.NotNull(bnvibFile);
            Ensure.Argument.True(0 < samplingRate);
            Ensure.Argument.Equals(bnvibFile.Info.SampleLength, bnvibFile.Data.SampleCount);

            GradualSineWaveGenerator generatorLow = new GradualSineWaveGenerator(samplingRate, VibrationValue.DefaultFrequencyLow);
            GradualSineWaveGenerator generatorHigh = new GradualSineWaveGenerator(samplingRate, VibrationValue.DefaultFrequencyHigh);

            int waveSampleCount = bnvibFile.Info.SampleLength * samplingRate / bnvibFile.Info.SamplingRate;
            float[] wave = new float[waveSampleCount];
            int saveSampleIdx = 0;

            for (int i = 0; i < bnvibFile.Info.SampleLength; i++)
            {
                VibrationValue v = bnvibFile.Data.GetVibrationValueAt(i);
                int nextIdx = (i + 1) * samplingRate / bnvibFile.Info.SamplingRate;
                int sampleCountToGenerate = nextIdx - saveSampleIdx;

                float[] waveLow = generatorLow.GenerateWave(sampleCountToGenerate, v.amplitudeLow, v.frequencyLow);
                float[] waveHigh = generatorHigh.GenerateWave(sampleCountToGenerate, v.amplitudeHigh, v.frequencyHigh);

                for (int j = 0; j < sampleCountToGenerate; j++)
                {
                    float sample = waveLow[j] + waveHigh[j];
                    if (sample > 1.0f) sample = 1.0f;
                    if (sample < -1.0f) sample = 1.0f;
                    wave[saveSampleIdx + j] = sample;
                }

                saveSampleIdx = nextIdx;
            }

            return wave;
        }
    }
}
