﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.Foundation.Audio;
using VibrationConverterConsole.BnvibIO;

namespace VibrationConverterConsole.AmpFreqExtractor
{
    public class MessageQueue
    {
        private bool m_IsFinished;
        private Queue<string> m_Queue;
        private Queue<string> m_ErrorQueue;
        private Object m_LockObj;
        private System.Threading.AutoResetEvent m_Event;
        public MessageQueue()
        {
            m_IsFinished = false;
            m_Queue = new Queue<string>();
            m_ErrorQueue = new Queue<string>();
            m_LockObj = new Object();
            m_Event = new System.Threading.AutoResetEvent(false);
        }
        public void Write(string s)
        {
            lock (m_LockObj)
            {
                m_Queue.Enqueue(s);
            }
            m_Event.Set();
        }
        public void WriteError(string s)
        {
            lock (m_LockObj)
            {
                m_ErrorQueue.Enqueue(s);
            }
            m_Event.Set();
        }
        public string[] Read()
        {
            lock (m_LockObj)
            {
                string[] sArray = m_Queue.ToArray();
                m_Queue.Clear();
                return sArray;
            }
        }
        public string[] ReadError()
        {
            lock (m_LockObj)
            {
                string[] sArray = m_ErrorQueue.ToArray();
                m_ErrorQueue.Clear();
                return sArray;
            }
        }
        public void Finish()
        {
            lock (m_LockObj)
            {
                m_IsFinished = true;
            }
            m_Event.Set();
        }
        public bool IsFinished()
        {
            lock (m_LockObj)
            {
                return m_IsFinished;
            }
        }
        public void Reset()
        {
            lock (m_LockObj)
            {
                m_IsFinished = false;
                m_Queue.Clear();
                m_Event.Reset();
            }
        }
        public void WaitMessage()
        {
            m_Event.WaitOne();
        }
    }

    public class Executor
    {
        public readonly MessageQueue m_MessageQueue;

        private string m_InputWavFilename;
        private WaveFileInfo m_InputWavFileInfo;
        private float[] m_MonoSource;
        private bool m_IsMonoSourceLoaded = false;
        public string inputWavFilename { get { return m_InputWavFilename; } }
        public bool isMonoSourceLoaded { get { return m_IsMonoSourceLoaded; } }

        private string m_InputConfigFilename;
        private CombinedAmpFreqExtractor m_Extractor;
        private bool m_IsExtractorLoaded = false;
        public string inputConfigFilename { get { return m_InputConfigFilename; } }
        public bool isExtractorLoaded { get { return m_IsExtractorLoaded; } }
        public CombinedAmpFreqExtractorConfig GetConfig()
        {
            return m_Extractor.GetConfig();
        }
        public void SetConfig(CombinedAmpFreqExtractorConfig c)
        {
            m_Extractor.SetConfig(c);
            if (!m_Extractor.canUseSavedOutput)
            {
                m_IsAmpFreqExtracted = false;
            }
        }

        public int inputSamplingRate { get { return m_InputWavFileInfo.WaveFormat.SamplingRate; } }
        public float[] GetMonoSource() { return m_MonoSource; }

        private BnvibFile m_BnvibFile;
        private bool m_IsAmpFreqExtracted = false;
        public bool isAmpFreqExtracted { get { return m_IsAmpFreqExtracted; } }

        public Executor(MessageQueue commonString)
        {
            m_MessageQueue = commonString;
        }
        public void Execute(CommandLineParameters parameters)
        {
            if (parameters.InputFileFormat == CommandLineParameters.FileFormat.None)
            {
                parameters.InputFileFormat = GetFileFormat(parameters.InputFilePath);
            }

            if (parameters.InputFileFormat == CommandLineParameters.FileFormat.Wav)
            {
                ReadWavFile(parameters.InputFilePath);
                if (m_MessageQueue.IsFinished()) return;

                ReadConfigFile(parameters.InputParameterFilePath);
                if (m_MessageQueue.IsFinished()) return;

                if (parameters.IsAutoAmpForceEnabled)
                {
                    m_Extractor.isAmpCoeffAutoAdjusted = true;
                }
                if (parameters.IsAutoAmpForceDisabled)
                {
                    m_Extractor.isAmpCoeffAutoAdjusted = false;
                }

                ExtractAmpFreq();
                if (m_MessageQueue.IsFinished()) return;

                ExtractLoopInfo();
                if (m_MessageQueue.IsFinished()) return;
            }
            else if (
                parameters.InputFileFormat == CommandLineParameters.FileFormat.Bnvib ||
                parameters.InputFileFormat == CommandLineParameters.FileFormat.Amfmc)
            {
                ReadBnvibFile(parameters.InputFilePath);
                if (m_MessageQueue.IsFinished()) return;
                parameters.IsNvibcpOutputSkipped = true;
            }
            else if (parameters.InputFileFormat == CommandLineParameters.FileFormat.Nvib)
            {
                ReadNvibFile(parameters.InputFilePath);
                if (m_MessageQueue.IsFinished()) return;
                parameters.IsNvibcpOutputSkipped = true;
            }

            if (parameters.OutputFilePath == null || parameters.OutputFilePath == "")
            {
                parameters.OutputFilePath = MakeOutputFilename(".bnvib");
            }

            WriteBnvibFile(parameters.OutputFilePath);
            if (m_MessageQueue.IsFinished()) return;

            if (parameters.IsNvibcpOutputSkipped == false)
            {
                if (parameters.OutputParameterFilePath == null || parameters.OutputParameterFilePath == "")
                {
                    parameters.OutputParameterFilePath = MakeOutputFilename(".nvibcp");
                }
                WriteConfigFile(parameters.OutputParameterFilePath);
                if (m_MessageQueue.IsFinished()) return;
            }

            if (parameters.OutputNvibFilePath != "")
            {
                WriteNvibFile(parameters.OutputNvibFilePath);
            }

            if (parameters.OutputWaveFilePath != "")
            {
                WriteWaveFile(parameters.OutputWaveFilePath);
            }

            m_MessageQueue.Write("Execution Finished.");
            m_MessageQueue.Finish();
        }

        public CommandLineParameters.FileFormat GetFileFormat(string filename)
        {
            string suffix = System.IO.Path.GetExtension(filename).ToLower();
            switch (suffix)
            {
                case ".wav":
                    return CommandLineParameters.FileFormat.Wav;
                case ".bnvib":
                    return CommandLineParameters.FileFormat.Bnvib;
                case ".amfmc":
                    return CommandLineParameters.FileFormat.Amfmc;
                case ".nvib":
                    return CommandLineParameters.FileFormat.Nvib;
                default:
                    return CommandLineParameters.FileFormat.Wav;
            }
        }

        public void ReadBnvibFile(string bnvibFilename)
        {
            m_MessageQueue.Write("Reading BNVIB File... (" + bnvibFilename + ")");

            m_BnvibFile = null;
            try
            {
                m_BnvibFile = BnvibFileReader.ReadBnvibFile(bnvibFilename);
            }
            catch (System.Exception ex)
            {
                m_MessageQueue.WriteError(ex.Message);
                m_MessageQueue.Finish();
                return;
            }

            if (m_BnvibFile == null)
            {
                m_MessageQueue.WriteError("BnvibFile Null");
                m_MessageQueue.Finish();
                return;
            }

            m_InputWavFilename = bnvibFilename;
        }

        public void ReadNvibFile(string nvibFilename)
        {
            m_MessageQueue.Write("Reading NVIB File... (" + nvibFilename + ")");

            m_BnvibFile = null;
            try
            {
                m_BnvibFile = BnvibFileReader.ReadNvibFile(nvibFilename);
            }
            catch (System.Exception ex)
            {
                m_MessageQueue.WriteError(ex.Message);
                m_MessageQueue.Finish();
                return;
            }

            if (m_BnvibFile == null)
            {
                m_MessageQueue.WriteError("BnvibFile Null");
                m_MessageQueue.Finish();
                return;
            }

            m_InputWavFilename = nvibFilename;
        }

        public void ReadWavFile(string wavFilename)
        {
            m_MessageQueue.Write("Reading WAV File... (" + wavFilename + ")");

            WaveFile waveFile;
            try
            {
                waveFile = WaveFileReader.ReadWaveFile(wavFilename);
            }
            catch (System.Exception ex)
            {
                m_MessageQueue.WriteError(ex.Message);
                m_MessageQueue.Finish();
                return;
            }

            if (waveFile == null)
            {
                m_MessageQueue.WriteError("WaveFile Null");
                m_MessageQueue.Finish();
                return;
            }

            m_MessageQueue.Write("Mixing to Monaural...");
            m_MonoSource = new float[waveFile.Info.SampleCount];
            if (waveFile.Data.ChannelCount > 0)
            {
                float scale = (float)Math.Pow(2.0, waveFile.Info.WaveFormat.BitPerSample - 1);
                for (int i = 0; i < waveFile.Data.SampleCount; i++)
                {
                    m_MonoSource[i] = 0.0f;
                    for (int ch = 0; ch < waveFile.Data.ChannelCount; ch++)
                    {
                        m_MonoSource[i] += waveFile.Data.Samples[ch][i] / scale;
                    }
                    m_MonoSource[i] /= waveFile.Data.ChannelCount;
                }
            }

            if (m_IsExtractorLoaded)
            {
                CombinedAmpFreqExtractorConfig c = m_Extractor.GetConfig();
                m_Extractor = new CombinedAmpFreqExtractor();
                m_Extractor.SetConfig(c);
            }

            m_InputWavFilename = wavFilename;
            m_InputWavFileInfo = waveFile.Info;
            m_IsMonoSourceLoaded = true;
            m_IsAmpFreqExtracted = false;
            m_MessageQueue.Write("Monaural Source Loaded.");
        }
        public void ReadConfigFile(string configFilename)
        {
            CombinedAmpFreqExtractor extractor = new CombinedAmpFreqExtractor();
            if (configFilename != "")
            {
                m_MessageQueue.Write("Reading Config File... (" + configFilename + ")");
                System.IO.FileStream stream;
                try
                {
                    stream = new System.IO.FileStream(configFilename, System.IO.FileMode.Open, System.IO.FileAccess.Read);
                }
                catch (System.Exception ex)
                {
                    m_MessageQueue.WriteError(ex.Message);
                    m_MessageQueue.Finish();
                    return;
                }

                m_MessageQueue.Write("Setting Config to AmpFreqExtractor...");
                try
                {
                    AmpFreqExtractor.ConfigReader.Read(stream, ref extractor);
                }
                catch (System.Exception ex)
                {
                    m_MessageQueue.WriteError(ex.Message);
                    m_MessageQueue.Finish();
                    return;
                }
                stream.Close();
            }

            m_InputConfigFilename = configFilename;
            m_Extractor = extractor;
            m_IsExtractorLoaded = true;
            m_IsAmpFreqExtracted = false;
            m_MessageQueue.Write("Extractor Loaded.");
        }
        public void ExtractAmpFreq()
        {
            if (!m_IsMonoSourceLoaded)
            {
                m_MessageQueue.WriteError("Read Wav File.");
                m_MessageQueue.Finish();
                return;
            }
            if (!m_IsExtractorLoaded)
            {
                m_MessageQueue.WriteError("Read Config File.");
                m_MessageQueue.Finish();
                return;
            }

            m_MessageQueue.Write("Extracting Amplitude and Frequency...");
            m_BnvibFile = m_Extractor.Extract(m_MonoSource, inputSamplingRate, m_MessageQueue);
            m_IsAmpFreqExtracted = true;
            m_MessageQueue.Write("Amplitude and Frequency Extracted.");
        }

        private int ConvertWavSampleCountToVibSampleCount(long wavSampleCount)
        {
            long half = inputSamplingRate / 2;
            return (int)((wavSampleCount * m_BnvibFile.Info.SamplingRate + half) / inputSamplingRate);
        }
        public void ExtractLoopInfo()
        {
            if (m_InputWavFileInfo == null)
            {
                m_MessageQueue.WriteError("Read Wav File.");
                m_MessageQueue.Finish();
                return;
            }
            if (m_BnvibFile == null)
            {
                m_MessageQueue.WriteError("Make Bnvib File.");
                m_MessageQueue.Finish();
                return;
            }

            if (m_InputWavFileInfo.HasLoop)
            {
                m_MessageQueue.Write("Extracting LoopInfo from WAV File...");

                int loopStart = ConvertWavSampleCountToVibSampleCount(m_InputWavFileInfo.LoopInfo.Position);
                int loopLength = ConvertWavSampleCountToVibSampleCount(m_InputWavFileInfo.LoopInfo.Length);
                int loopEnd = loopStart + loopLength;

                // loopEnd がサンプル末尾を超えないように調整
                if (loopEnd > m_BnvibFile.Info.SampleLength)
                {
                    if (loopStart > 0)
                    {
                        loopStart -= 1;
                        loopEnd -= 1;
                    }

                    if (loopEnd > m_BnvibFile.Info.SampleLength)
                    {
                        loopEnd = m_BnvibFile.Info.SampleLength;
                    }
                }

                m_BnvibFile.Info.HasLoop = true;
                m_BnvibFile.Info.LoopStart = loopStart;
                m_BnvibFile.Info.LoopEnd = loopEnd;
            }
            else
            {
                m_MessageQueue.Write("Input WAV File doesn't have LoopInfo.");
            }
        }

        public string MakeOutputFilename(string suffix)
        {
            string outputFilename = System.IO.Path.GetDirectoryName(m_InputWavFilename);
            if (outputFilename != "") outputFilename += System.IO.Path.DirectorySeparatorChar;
            outputFilename += System.IO.Path.GetFileNameWithoutExtension(m_InputWavFilename);
            outputFilename += suffix;
            return outputFilename;
        }
        public void WriteBnvibFile(string filename)
        {
            // メタデータを通常フォーマット用に変更する
            m_BnvibFile.Info.FormatId = BnvibFileFormatId.PcmVer1;
            m_BnvibFile.Info.DataSize = m_BnvibFile.Data.SampleCount * VibrationValue.RawDataSize;

            m_MessageQueue.Write("Writing BNVIB File... (" + filename + ")");
            BnvibFileWriter.WriteBnvibFile(filename, m_BnvibFile);
            m_MessageQueue.Write("BNVIB File Written.");
        }
        public void WriteNvibFile(string filename)
        {
            m_MessageQueue.Write("Writing NVIB File... (" + filename + ")");
            BnvibFileWriter.WriteNvibFile(filename, m_BnvibFile);
            m_MessageQueue.Write("NVIB File Written.");
        }
        public void WriteConfigFile(string filename)
        {
            m_MessageQueue.Write("Writing Config File... (" + filename + ")");
            System.IO.FileStream stream;
            try
            {
                stream = new System.IO.FileStream(filename, System.IO.FileMode.Create);
            }
            catch (System.Exception ex)
            {
                m_MessageQueue.WriteError(ex.Message);
                m_MessageQueue.Finish();
                return;
            }

            AmpFreqExtractor.ConfigWriter.Write(stream, m_Extractor);
            stream.Close();

            m_MessageQueue.Write("Config File Written.");
        }
        public void WriteWaveFile(string wavFilename)
        {
            m_MessageQueue.Write("Writing WAV File... (" + wavFilename + ")");

            const int SamplingRate = 8000;
            const int BitPerSample = 16;
            int scale = (int)(Math.Pow(2.0, BitPerSample - 1)) - 1;

            WaveFile waveFile = new WaveFile();
            waveFile.Info = new WaveFileInfo();
            waveFile.Info.FileFormat = WaveFileFormat.Wave;
            waveFile.Info.WaveFormat = new WaveFormat(WaveEncoding.Pcm, 1, SamplingRate, BitPerSample, 0);

            float[] floatData = WaveReconstructor.ReconstructWave(m_BnvibFile, SamplingRate);
            waveFile.Data = new WaveSampleData(1, floatData.Length);
            for (int i = 0; i < floatData.Length; i++)
            {
                waveFile.Data.Samples[0][i] = (int)(floatData[i] * scale);
            }

            try
            {
                WaveFileWriter.WriteWaveFile(wavFilename, waveFile);
            }
            catch (System.Exception ex)
            {
                m_MessageQueue.WriteError(ex.Message);
                m_MessageQueue.Finish();
                return;
            }
        }
    }
}
