﻿// 設定ファイルを使用する
//#define ENABLE_CONFIG_FILE

// 最近使ったファイルを使用する
//#define ENABLE_RECENT_FILE

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace Nintendo.NX.IrsensorNfcFirmwareUpdater
{
    using PairType = KeyValuePair<string, Color>;

    public partial class MainForm : Form
    {
        private const int ProgressSmoother = 10;

        private const string ImageDataDirectory = "data";

        /// <summary>
        /// STMFlashLoader.exe のパス
        /// </summary>
        private string FlashLoaderPath { get; set; }

        /// <summary>
        /// ポート一覧
        /// </summary>
        private List<Util.PortInfo> PortList { get; set; }

        /// <summary>
        /// FW イメージ情報
        /// </summary>
        private ImageInfo FwInfo { get; set; }

        /// <summary>
        /// データ受信イベントハンドラ
        /// </summary>
        private DataReceivedEventHandler DataReceivedHandler { get; set; }

        private Process WriterProcess { get; set; }

        public MainForm()
        {
            InitializeComponent();

            PortList = new List<Util.PortInfo>();
            FwInfo = new ImageInfo();
            DataReceivedHandler = null;
        }

        /// <summary>
        /// STMFlashLoader のインストール先を確認
        /// </summary>
        /// <returns></returns>
        private bool CheckStmFlashLoader()
        {
            const string FlashLoaderFilename = "STMFlashLoader.exe";

            var searchPaths = new string[]
            {
                Path.Combine(Application.StartupPath, "bin"),  // ローカルに置いてある前提
            };

            foreach (var path in searchPaths)
            {
                if (!Directory.Exists(path))
                {
                    continue;
                }

                foreach (var filepath in Directory.GetFiles(path))
                {
                    var filename = Path.GetFileName(filepath);
                    if (string.Equals(filename, FlashLoaderFilename, StringComparison.CurrentCultureIgnoreCase))
                    {
                        FlashLoaderPath = filepath;
                        Console.WriteLine(FlashLoaderPath);
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// シリアルポート一覧の更新
        /// </summary>
        private void UpdatePortList()
        {
            PortList = Util.GetDetailedPortNames();

            serialPortBox.BeginUpdate();
            serialPortBox.Items.Clear();

            foreach (var port in PortList)
            {
                var portName = Util.RegexPortNum.Replace(port.Name, "");
                var comName = string.Format(
                    string.IsNullOrEmpty(portName) ? "{0}" : "{0}: ",
                    port.Id);
                serialPortBox.Items.Add(comName + portName);
            }

            if (serialPortBox.Items.Count > 0)
            {
                serialPortBox.SelectedIndex = 0;
            }

            serialPortBox.EndUpdate();
        }

        /// <summary>
        /// FW イメージファイル一覧の更新
        /// </summary>
        private void UpdateImageFileList()
        {
            string imageListPath = Path.Combine(Application.StartupPath, ImageDataDirectory);
            if (!Directory.Exists(imageListPath))
            {
                return;
            }

            comboImageFile.BeginUpdate();
            comboImageFile.Items.Clear();
            foreach (var filepath in Directory.GetFiles(imageListPath, "*.hex"))
            {
                comboImageFile.Items.Add(Path.GetFileName(filepath));
            }
            comboImageFile.EndUpdate();

            UpdateFirmwareListVisibility();

#if ENABLE_RECENT_FILE
            var config = Config.GetInstance();
            if (config.RecentFileList.Count > 0)
            {
                comboImageFile.Text = config.RecentFileList[0];
            }
#else
            if (comboImageFile.Items.Count > 0)
            {
                comboImageFile.SelectedIndex = 0;
            }
#endif
        }

        private void UpdateFirmwareListVisibility()
        {
            bool existsSingleImageFile = comboImageFile.Items.Count == 1;
            labelImage.Visible =
                comboImageFile.Enabled =
                comboImageFile.Visible =
                buttonBrowseImage.Enabled =
                buttonBrowseImage.Visible =
                !existsSingleImageFile;

            if (existsSingleImageFile)
            {
                labelProgress.Text = "Press start button to update";

                var minimumSize = MinimumSize;
                minimumSize.Height = 160;
                this.MinimumSize = minimumSize;
#if !_DEBUG
                this.FormBorderStyle = FormBorderStyle.FixedSingle;
                this.MaximizeBox = false;
#endif
            }
        }

        /// <summary>
        /// フォームロード
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void MainForm_Load(object sender, EventArgs e)
        {
            if (!CheckStmFlashLoader())
            {
                MessageBox.Show(
                    "Could not find \"STMFlashLoader\".",
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
                Close();
                return;
            }

            WriterProcess = new Process();
            UpdatePortList();

#if ENABLE_CONFIG_FILE
            var config = Config.GetInstance();
            config.Load();
            for (int i = 0; i < PortList.Count; ++i)
            {
                if (PortList[i].Number == config.LastPortNumber)
                {
                    serialPortBox.SelectedIndex = i;
                    break;
                }
            }
            checkVerify.Checked = config.VerifyEnabled;
#endif

            UpdateImageFileList();

            Height = MinimumSize.Height;
#if !DEBUG
            textProcessLog.Visible = false;
#endif
        }

        /// <summary>
        /// フォームクローズ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void MainForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            try
            {
#if ENABLE_CONFIG_FILE
                Config.GetInstance().Save();
#endif
                WriterProcess.Kill();
            }
            catch
            {
                // 失敗しても無視
            }
        }

        /// <summary>
        /// ファイルドラッグ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void MainForm_DragOver(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                e.Effect = DragDropEffects.Copy;
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }
        }

        /// <summary>
        /// ファイルドロップ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void MainForm_DragDrop(object sender, DragEventArgs e)
        {
            if (!e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                return;
            }

            var files = (string[])e.Data.GetData(DataFormats.FileDrop);
            comboImageFile.Text = files[0];
        }

        /// <summary>
        /// ポート選択
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void serialPortBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            var config = Config.GetInstance();
            var index = ((ComboBox)sender).SelectedIndex;
            if (index >= 0)
            {
                config.LastPortNumber = PortList[index].Number;
            }
            else
            {
                config.LastPortNumber = 0;
            }
        }

        /// <summary>
        /// イメージファイル選択
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void buttonBrowseImage_Click(object sender, EventArgs e)
        {
            if (openImageDialog.ShowDialog(this) != DialogResult.OK)
            {
                return;
            }

            comboImageFile.Text = openImageDialog.FileName;
        }

        /// <summary>
        /// Verify 有無変更
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void checkVerify_CheckedChanged(object sender, EventArgs e)
        {
            var control = (CheckBox)sender;
            var config = Config.GetInstance();
            config.VerifyEnabled = control.Checked;
        }

        /// <summary>
        /// アップデートを続けるか判定
        /// </summary>
        /// <param name="log"></param>
        /// <returns></returns>
        private bool IsContinueUpdate(string log)
        {
            // エラー系のログが出た場合は中断
            var killRegex = new Regex(@"^\s*Press any key to continue");
            var failRegex = new Regex(@"\[KO\]");
            if (killRegex.IsMatch(log) || failRegex.IsMatch(log))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Erase 開始ログの確認
        /// </summary>
        /// <param name="log"></param>
        private void CheckEraseLog(string log)
        {
            var eraseRegex = new Regex(@"^\s*ERASING\s*\.+");
            if (eraseRegex.IsMatch(log))
            {
                BeginInvoke(new Action(() => labelProgress.Text = "Erasing flash..."));
            }
        }

        /// <summary>
        /// Download (Write) の進捗確認
        /// </summary>
        /// <param name="log"></param>
        private void CheckDownloadLog(string log)
        {
            {
                // これは STMFlashLoader を修正して追加したもの
                var sectorsRegex = new Regex(@"^\[DOWNLOAD\] Total sectors = (?<sectors>\d+)");
                var match = sectorsRegex.Match(log);
                if (match.Success)
                {
                    FwInfo.Sectors = int.Parse(match.Groups["sectors"].Value);
                    BeginInvoke(new Action(() =>
                    {
                        progressWrite.Maximum = (FwInfo.Sectors * (FwInfo.NeedVerify ? 2 : 1) + 1) * ProgressSmoother;
                        progressWrite.Value = 10;
                        labelProgress.Text = string.Format("Writing... (1/{0} sectors)", FwInfo.Sectors);
                    }));
                }
            }

            {
                var downloadRegex = new Regex(@"^downloading\s+page/sector\s+(?<sector>\d+)");
                var match = downloadRegex.Match(log);
                if (match.Success)
                {
                    var sector = int.Parse(match.Groups["sector"].Value) + 2;
                    BeginInvoke(new Action(() =>
                    {
                        progressWrite.Value = sector * ProgressSmoother;
                        if (sector <= FwInfo.Sectors)
                        {
                            labelProgress.Text = string.Format("Writing... ({0}/{1} sectors)", sector, FwInfo.Sectors);
                        }
                    }));
                }
            }
        }

        /// <summary>
        /// Verify の進捗確認
        /// </summary>
        /// <param name="log"></param>
        private void CheckVerifyLog(string log)
        {
            {
                var verifyStartStr = new Regex(@"^\s*VERIFYING\s*\.+");
                if (verifyStartStr.IsMatch(log))
                {
                    BeginInvoke(new Action(() =>
                    {
                        progressWrite.Value = (FwInfo.Sectors + 1) * ProgressSmoother;
                        labelProgress.Text = string.Format("Verifying... (1/{0} sectors)", FwInfo.Sectors);
                    }));
                }
            }

            {
                var verifyStr = new Regex(@"^verifying\s+page/sector\s+(?<sector>\d+)");
                var match = verifyStr.Match(log);
                if (match.Success)
                {
                    var sector = int.Parse(match.Groups["sector"].Value) + 2;
                    BeginInvoke(new Action(() =>
                    {
                        progressWrite.Value = (FwInfo.Sectors + sector) * ProgressSmoother;
                        if (sector <= FwInfo.Sectors)
                        {
                            labelProgress.Text = string.Format("Verifying... ({0}/{1} sectors)", sector, FwInfo.Sectors);
                        }
                    }));
                }
            }
        }

        /// <summary>
        /// STMFlashLoader のログを表示
        /// </summary>
        /// <param name="log"></param>
        private void PrintLog(string log)
        {
            BeginInvoke(new Action(() =>
            {
                textProcessLog.Focus();

                Debug.WriteLine(log);
                textProcessLog.AppendText(log);
                textProcessLog.AppendText(Environment.NewLine);

                var hilightList = new List<PairType>()
                    {
                        new PairType("[OK]", Color.DodgerBlue),
                        new PairType("[KO]", Color.Crimson)
                    };
                int currentSelectionStart = textProcessLog.SelectionStart;
                int currentSelectionLength = textProcessLog.SelectionLength;
                int pos = currentSelectionStart;
                while (pos >= 0)
                {
                    bool isFound = false;
                    foreach (var pair in hilightList)
                    {
                        int newPos = textProcessLog.Find(pair.Key, pos, RichTextBoxFinds.None);
                        if (newPos >= 0)
                        {
                            textProcessLog.SelectionColor = pair.Value;
                            isFound = true;
                            pos = newPos + 1;
                            break;
                        }
                    }

                    if (!isFound)
                    {
                        pos = -1;
                    }
                }
                textProcessLog.Select(currentSelectionStart, currentSelectionLength);
            }));
        }

        /// <summary>
        /// プロセスからの stdout/stderr 受信ハンドラ
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Process_DataReceived(object sender, DataReceivedEventArgs e)
        {
            var log = e.Data;
            if (Disposing || IsDisposed || log == null)
            {
                return;
            }

            if (!IsContinueUpdate(log))
            {
                var process = (Process)sender;
                if (!process.HasExited)
                {
                    process.Kill();
                }
                Debug.WriteLine("KILLED");
                return;
            }

            CheckEraseLog(log);
            CheckDownloadLog(log);
            CheckVerifyLog(log);
            PrintLog(log);
        }

        /// <summary>
        /// プロセス起動情報の設定
        /// </summary>
        /// <param name="process">設定対象のプロセス</param>
        private void SetupProcessInfo(Process process, string imagePath)
        {
            if (DataReceivedHandler == null)
            {
                DataReceivedHandler = Process_DataReceived;
            };

            var portNumber = PortList[serialPortBox.SelectedIndex].Number;

            FwInfo.Filename = imagePath;
            FwInfo.NeedVerify = checkVerify.Checked;

            var startInfo = process.StartInfo;
            startInfo.FileName = FlashLoaderPath;
            startInfo.Arguments = string.Format(
                "-c --pn {0} --br 115200 -i STM32F4_11_512K -e --all -d --fn \"{1}\" {2}",
                portNumber,
                FwInfo.Filename,
                FwInfo.NeedVerify ? "--v" : "");

            // コンソール出力をリダイレクトするために必要な設定
            startInfo.CreateNoWindow = true;
            startInfo.UseShellExecute = false;
            startInfo.RedirectStandardError = true;
            startInfo.RedirectStandardOutput = true;

            process.OutputDataReceived += DataReceivedHandler;
            process.ErrorDataReceived += DataReceivedHandler;

            textProcessLog.Clear();
        }

        /// <summary>
        /// コントロールの有効状態を設定
        /// </summary>
        /// <param name="enable"></param>
        private void SetControlEnabled(bool enable)
        {
            serialPortBox.Enabled = enable;
            buttonBrowseImage.Enabled = enable;
            buttonStart.Enabled = enable;
            checkVerify.Enabled = enable;
            comboImageFile.Enabled = enable;

            waitIcon.Visible = !enable;
        }

        private string GetActualImagePath(string filepath)
        {
            if (File.Exists(filepath))
            {
                return filepath;
            }

            var pathInData = Path.Combine(ImageDataDirectory, filepath);
            if (File.Exists(pathInData))
            {
                return pathInData;
            }

            return null;
        }

        /// <summary>
        /// 開始ボタン
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private async void buttonStart_Click(object sender, EventArgs e)
        {
            if (serialPortBox.SelectedIndex < 0)
            {
                MessageBox.Show(
                    "COM port is not selected.",
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
                return;
            }

            var imagePath = GetActualImagePath(comboImageFile.Text);
            if (!File.Exists(imagePath))
            {
                MessageBox.Show(
                    "Specified image is not found.",
                    "Error",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
                return;
            }
            //else if (!WriterProcess.HasExited)
            //{
            //    MessageBox.Show(
            //        "Process is already running.",
            //        "Error",
            //        MessageBoxButtons.OK,
            //        MessageBoxIcon.Error);
            //    return;
            //}

            SetControlEnabled(false);
            progressWrite.Value = 0;
            labelProgress.Text = "Preparing...";

            // COM ポートに残っているゴミを掃除
            CleanComPort(PortList[serialPortBox.SelectedIndex].Id);

            Config.GetInstance().RegisterRecentFile(comboImageFile.Text);

            SetupProcessInfo(WriterProcess, imagePath);

            await Task.Run(() =>
            {
                WriterProcess.Start();

                // コンソール出力のリダイレクト開始
                WriterProcess.BeginOutputReadLine();
                WriterProcess.BeginErrorReadLine();

                WriterProcess.WaitForExit();

                // リダイレクト終了
                WriterProcess.CancelErrorRead();
                WriterProcess.CancelOutputRead();
                WriterProcess.OutputDataReceived -= DataReceivedHandler;
                WriterProcess.ErrorDataReceived -= DataReceivedHandler;
            });

            WriterProcess.Close();

            if (progressWrite.Value < progressWrite.Maximum)
            {
                labelProgress.Text = "FAILED...";
                labelProgress.ForeColor = Color.Red;
            }
            else
            {
                labelProgress.Text = "Finished!!";
            }
            progressWrite.Value = progressWrite.Maximum;

            //SetControlEnabled(true);
            waitIcon.Visible = false;
            buttonStart.Text = "Exit";
            buttonStart.Image = null;
            buttonStart.Click -= buttonStart_Click;
            buttonStart.Click += buttonStart_ClickForExit;
            buttonStart.Enabled = true;
        }

        private void buttonStart_ClickForExit(object sender, EventArgs e)
        {
            Close();
        }

        /// <summary>
        /// COM ポートに残っているデータを捨てる
        /// </summary>
        /// <param name="portName">対象の COM ポート</param>
        private void CleanComPort(string portName)
        {
            using (var comPort = new System.IO.Ports.SerialPort())
            {
                comPort.PortName = portName;
                comPort.BaudRate = 115200;
                comPort.Open();

                // 送受信バッファ内のデータを全削除
                comPort.DiscardInBuffer();
                comPort.DiscardOutBuffer();

                comPort.Close();
            }
        }
    }
}
