﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Runtime.Serialization;
    using System.Windows.Forms;
    using System.Xml.Linq;

    /// <summary>
    /// ユーザ設定を扱うクラスです。
    /// </summary>
    internal class UserConfig
    {
        private const string Extention = ".user.config";

        private static readonly Lazy<string> FilePath =
            new Lazy<string>(GetFilePath);

        private readonly object syncObject = new object();

        private readonly UserConfigData data = Load();

        /// <summary>
        /// UserConfig クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal UserConfig()
        {
        }

        /// <summary>
        /// ユーザ設定を構成ファイルにセーブします。
        /// </summary>
        internal void Save()
        {
            var document = new XDocument();

            using (var writer = document.CreateWriter())
            {
                var serializer =
                    new DataContractSerializer(typeof(UserConfigData));

                lock (this.syncObject)
                {
                    serializer.WriteObject(writer, this.data);
                }
            }

            string path = FilePath.Value;

            document.Save(path);
        }

        /// <summary>
        /// ボーダークロッシング設定を返します。
        /// </summary>
        /// <returns>ボーダークロッシング設定です。</returns>
        internal BorderCrossingConfig GetBorderCrossingConfig()
        {
            lock (this.syncObject)
            {
                return new BorderCrossingConfig(this.data.BorderCrossing);
            }
        }

        /// <summary>
        /// ボーダークロッシング設定を設定します。
        /// </summary>
        /// <param name="config">ボーダークロッシング設定です。</param>
        internal void SetBorderCrossingConfig(BorderCrossingConfig config)
        {
            lock (this.syncObject)
            {
                this.data.BorderCrossing = new BorderCrossingConfig(config);
            }
        }

        /// <summary>
        /// ホットキー設定を返します。
        /// </summary>
        /// <returns>ホットキー設定です。</returns>
        internal HotKeyConfig GetHotKeyConfig()
        {
            lock (this.syncObject)
            {
                return new HotKeyConfig(this.data.HotKey);
            }
        }

        /// <summary>
        /// ホットキー設定を設定します。
        /// </summary>
        /// <param name="config">ホットキー設定です。</param>
        internal void SetHotKeyConfig(HotKeyConfig config)
        {
            lock (this.syncObject)
            {
                this.data.HotKey = new HotKeyConfig(config);
            }
        }

        /// <summary>
        /// 通信設定を返します。
        /// </summary>
        /// <returns>通信設定です。</returns>
        internal ConnectionConfig GetConnectionConfig()
        {
            lock (this.syncObject)
            {
                return new ConnectionConfig(this.data.Connection);
            }
        }

        /// <summary>
        /// 通信設定を設定します。
        /// </summary>
        /// <param name="config">通信設定です。</param>
        internal void SetConnectionConfig(ConnectionConfig config)
        {
            lock (this.syncObject)
            {
                this.data.Connection = new ConnectionConfig(config);
            }
        }

        /// <summary>
        /// ゲームパッド設定を返します。
        /// </summary>
        /// <param name="name">ゲームパッドの名前です。</param>
        /// <returns>ゲームパッド設定です。</returns>
        internal GamePadConfig GetGamePadConfig(string name)
        {
            lock (this.syncObject)
            {
                GamePadConfig config =
                    this.data.GamePads.FirstOrDefault(x => x.Name == name);

                return (config == null) ? null : new GamePadConfig(config);
            }
        }

        /// <summary>
        /// ゲームパッド設定を設定します。
        /// </summary>
        /// <param name="config">ゲームパッド設定です。</param>
        internal void SetGamePadConfig(GamePadConfig config)
        {
            lock (this.syncObject)
            {
                this.data.GamePads.RemoveAll(x => x.Name == config.Name);

                this.data.GamePads.Add(new GamePadConfig(config));
            }
        }

        private static string GetFilePath()
        {
            Assembly assembly = Assembly.GetEntryAssembly();

            string path = Path.GetDirectoryName(assembly.Location);

            path = Path.Combine(path, Application.ProductName);

            path = path + Extention;

            return path;
        }

        private static UserConfigData Load()
        {
            string path = FilePath.Value;

            if (File.Exists(path))
            {
                try
                {
                    var document = XDocument.Load(path);

                    using (var reader = document.CreateReader())
                    {
                        var deserializer =
                            new DataContractSerializer(typeof(UserConfigData));

                        return (UserConfigData)deserializer.ReadObject(reader);
                    }
                }
                catch
                {
                    // 何もしない
                }
            }

            return new UserConfigData();
        }

        [DataContract(Name = "UserConfig")]
        private sealed class UserConfigData
        {
            internal UserConfigData()
            {
            }

            [DataMember]
            internal BorderCrossingConfig BorderCrossing { get; set; } =
                new BorderCrossingConfig();

            [DataMember]
            internal HotKeyConfig HotKey { get; set; } = new HotKeyConfig();

            [DataMember]
            internal ConnectionConfig Connection { get; set; } =
                new ConnectionConfig();

            [DataMember]
            internal List<GamePadConfig> GamePads { get; set; } =
                new List<GamePadConfig>();
        }
    }
}
