﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector.Services
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Runtime.InteropServices;
    using System.Threading;
    using System.Threading.Tasks;

    /// <summary>
    /// TouchScreen サービスを提供します。
    /// </summary>
    internal sealed class TouchScreenService : ISessionService
    {
        private const int TouchCountMax = 16;

        private static readonly TimeSpan SamplerInterval =
            TimeSpan.FromMilliseconds(7);

        private readonly object syncObject = new object();

        private TouchScreenState hostState = new TouchScreenState();

        private TouchScreenState autoState = new TouchScreenState();

        private TouchScreenState lastState = new TouchScreenState();

        private uint port = HidShellPortAccessor.DefaultPort;

        private ISessionWriter writer = null;

        private CancellationTokenSource tokenSource = null;

        private Task task = null;

        /// <summary>
        /// TouchScreenService クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal TouchScreenService()
        {
        }

        /// <summary>
        /// セッションの情報を設定します。
        /// </summary>
        public SessionInfo Info { private get; set; }

        /// <summary>
        /// サービスを開始します。
        /// </summary>
        /// <param name="writer">セッションへの書き込みを扱うライターです。</param>
        public void Start(ISessionWriter writer)
        {
            lock (this.syncObject)
            {
                this.hostState = new TouchScreenState();
                this.autoState = this.hostState;
                this.lastState = this.hostState;
            }

            this.writer = writer;

            var ioPortManager = this.Info.IoPortManager;

            this.port = ioPortManager.GetPort(this.Info.Name);

            ioPortManager.Polling += this.OnPolling;

            this.tokenSource = new CancellationTokenSource();

            this.task = Task.Run(() => this.Sample(this.tokenSource.Token));
        }

        /// <summary>
        /// サービスを停止します。
        /// </summary>
        public void Stop()
        {
            this.Cancel();
        }

        /// <summary>
        /// サービスをキャンセルします。
        /// </summary>
        public void Cancel()
        {
            this.tokenSource.Cancel();

            this.task.Wait();
            this.task.Dispose();
            this.task = null;

            this.tokenSource.Dispose();
            this.tokenSource = null;

            var ioPortManager = this.Info.IoPortManager;
            ioPortManager.Polling -= this.OnPolling;

            this.writer = null;
        }

        /// <summary>
        /// メッセージを受け付けます。
        /// </summary>
        /// <param name="message">メッセージです。</param>
        public void AcceptMessage(byte[] message)
        {
            if (message.Length != Session.MessageSize)
            {
                return;
            }

            var messageType = (MessageType)message[0];

            if (messageType != MessageType.TouchBegan &&
                messageType != MessageType.TouchMoved &&
                messageType != MessageType.TouchEnded)
            {
                return;
            }

            var chunk = StructConverter.FromBytes<TouchChunk>(message);

            switch (messageType)
            {
                case MessageType.TouchBegan:
                    this.AcceptTouchBeganMessage(
                        chunk.FingerId, chunk.X, chunk.Y);
                    break;

                case MessageType.TouchMoved:
                    this.AcceptTouchMovedMessage(
                        chunk.FingerId, chunk.X, chunk.Y);
                    break;

                case MessageType.TouchEnded:
                    this.AcceptTouchEndedMessage(
                        chunk.FingerId);
                    break;

                default:
                    break;
            }
        }

        private static void GenerateChunks(
            List<byte[]> chunks,
            ref TouchScreenState nextState,
            ref TouchScreenState lastState)
        {
            int nextIndex = 0;
            int lastIndex = 0;

            while (true)
            {
                if (lastIndex == lastState.Count)
                {
                    for (int i = nextIndex; i < nextState.Count; ++i)
                    {
                        TouchState touch = nextState.Touches[i];

                        var chunk = new TouchChunk
                        {
                            Type = (byte)MessageType.TouchBegan,
                            FingerId = (byte)touch.FingerId,
                            X = (short)touch.X,
                            Y = (short)touch.Y,
                        };

                        chunks.Add(StructConverter.ToBytes(chunk));
                    }

                    return;
                }

                if (nextIndex == nextState.Count)
                {
                    for (int i = lastIndex; i < lastState.Count; ++i)
                    {
                        TouchState touch = lastState.Touches[i];

                        var chunk = new TouchChunk
                        {
                            Type = (byte)MessageType.TouchEnded,
                            FingerId = (byte)touch.FingerId,
                        };

                        chunks.Add(StructConverter.ToBytes(chunk));
                    }

                    return;
                }

                TouchState nextTouch = nextState.Touches[nextIndex];
                TouchState lastTouch = lastState.Touches[lastIndex];

                if (nextTouch.FingerId != lastTouch.FingerId)
                {
                    var chunk = new TouchChunk
                    {
                        Type = (byte)MessageType.TouchEnded,
                        FingerId = (byte)lastTouch.FingerId,
                    };

                    chunks.Add(StructConverter.ToBytes(chunk));

                    ++lastIndex;
                }
                else
                {
                    if (nextTouch.X != lastTouch.X ||
                        nextTouch.Y != lastTouch.Y)
                    {
                        var chunk = new TouchChunk
                        {
                            Type = (byte)MessageType.TouchMoved,
                            FingerId = (byte)nextTouch.FingerId,
                            X = (short)nextTouch.X,
                            Y = (short)nextTouch.Y,
                        };

                        chunks.Add(StructConverter.ToBytes(chunk));
                    }

                    ++nextIndex;
                    ++lastIndex;
                }
            }
        }

        private static void SetTouchScreenState(
            IntPtr handle, ref TouchScreenState state, uint port)
        {
            var result = (HidShellResult)
                Native.SetHidShellTouchScreenState(
                    handle, ref state, port, HidShellPortAccessor.Out);

            if (result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }
        }

        private static void GetTouchScreenAutoPilotState(
            IntPtr handle, out TouchScreenState outState, uint port)
        {
            var result = (HidShellResult)
                Native.GetHidShellTouchScreenState(
                    handle, out outState, port, HidShellPortAccessor.In);

            switch (result)
            {
                case HidShellResult.Success:
                    return;

                case HidShellResult.StateNotSet:
                    outState = new TouchScreenState();
                    return;

                default:
                    throw new HidShellException(result);
            }
        }

        private void AcceptTouchBeganMessage(int fingerId, int x, int y)
        {
            var touch = new TouchState()
            {
                FingerId = fingerId,
                X = x,
                Y = y,
            };

            lock (this.syncObject)
            {
                var count = this.hostState.Count;

                if (count < TouchCountMax)
                {
                    if (Enumerable.Range(0, count).All(i =>
                            this.hostState.Touches[i].FingerId != fingerId))
                    {
                        this.hostState.Touches[count] = touch;

                        this.hostState.Count += 1;
                    }
                }
            }
        }

        private void AcceptTouchMovedMessage(int fingerId, int x, int y)
        {
            var touch = new TouchState()
            {
                FingerId = fingerId,
                X = x,
                Y = y,
            };

            lock (this.syncObject)
            {
                foreach (int i in Enumerable.Range(0, this.hostState.Count))
                {
                    if (this.hostState.Touches[i].FingerId == fingerId)
                    {
                        this.hostState.Touches[i] = touch;
                    }
                }
            }
        }

        private void AcceptTouchEndedMessage(int fingerId)
        {
            var state = new TouchScreenState();

            lock (this.syncObject)
            {
                foreach (int i in Enumerable.Range(0, this.hostState.Count))
                {
                    if (this.hostState.Touches[i].FingerId != fingerId)
                    {
                        state.Touches[state.Count] = this.hostState.Touches[i];

                        state.Count += 1;
                    }
                }

                this.hostState = state;
            }
        }

        private void OnPolling(object sender, HidShellPortAccessor accessor)
        {
            lock (this.syncObject)
            {
                GetTouchScreenAutoPilotState(
                    accessor.Handle, out this.autoState, this.port);

                SetTouchScreenState(
                    accessor.Handle, ref this.hostState, this.port);
            }
        }

        private void Sample(CancellationToken token)
        {
            var works = true;

            var stopwatch = new Stopwatch();

            var chunks = new List<byte[]>();

            var autoState = new TouchScreenState();

            while (works && !token.IsCancellationRequested)
            {
                stopwatch.Start();

                lock (this.syncObject)
                {
                    autoState = this.autoState;
                }

                GenerateChunks(chunks, ref autoState, ref this.lastState);

                foreach (var chunk in chunks)
                {
                    try
                    {
                        this.writer.Write(chunk, 0, chunk.Length, token);
                    }
                    catch
                    {
                        works = false;

                        break;
                    }
                }

                chunks.Clear();

                this.lastState = autoState;

                stopwatch.Stop();

                if (stopwatch.Elapsed < SamplerInterval &&
                    works && !token.IsCancellationRequested)
                {
                    token.WaitHandle.WaitOne(
                        SamplerInterval - stopwatch.Elapsed);
                }

                stopwatch.Reset();
            }
        }

        [StructLayout(LayoutKind.Explicit)]
        private struct TouchChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(4)]
            public byte FingerId;

            [FieldOffset(8)]
            public short X;

            [FieldOffset(10)]
            public short Y;
        }

        [StructLayout(LayoutKind.Sequential, Pack = 8)]
        private struct TouchState
        {
            internal long DeltaTime;
            internal uint Attributes;
            internal int FingerId;
            internal int X;
            internal int Y;
            internal int DiameterX;
            internal int DiameterY;
            internal int RotationAngle;
        }

        private struct TouchesState
        {
            private TouchState touch0;
            private TouchState touch1;
            private TouchState touch2;
            private TouchState touch3;
            private TouchState touch4;
            private TouchState touch5;
            private TouchState touch6;
            private TouchState touch7;
            private TouchState touch8;
            private TouchState touch9;
            private TouchState touch10;
            private TouchState touch11;
            private TouchState touch12;
            private TouchState touch13;
            private TouchState touch14;
            private TouchState touch15;

            internal TouchState this[int i]
            {
                get
                {
                    switch (i)
                    {
                        case 0: return touch0;
                        case 1: return touch1;
                        case 2: return touch2;
                        case 3: return touch3;
                        case 4: return touch4;
                        case 5: return touch5;
                        case 6: return touch6;
                        case 7: return touch7;
                        case 8: return touch8;
                        case 9: return touch9;
                        case 10: return touch10;
                        case 11: return touch11;
                        case 12: return touch12;
                        case 13: return touch13;
                        case 14: return touch14;
                        case 15: return touch15;
                        default: throw new IndexOutOfRangeException();
                    }
                }

                set
                {
                    switch (i)
                    {
                        case 0: touch0 = value; break;
                        case 1: touch1 = value; break;
                        case 2: touch2 = value; break;
                        case 3: touch3 = value; break;
                        case 4: touch4 = value; break;
                        case 5: touch5 = value; break;
                        case 6: touch6 = value; break;
                        case 7: touch7 = value; break;
                        case 8: touch8 = value; break;
                        case 9: touch9 = value; break;
                        case 10: touch10 = value; break;
                        case 11: touch11 = value; break;
                        case 12: touch12 = value; break;
                        case 13: touch13 = value; break;
                        case 14: touch14 = value; break;
                        case 15: touch15 = value; break;
                        default: throw new IndexOutOfRangeException();
                    }
                }
            }
        }

        private struct TouchScreenState
        {
            internal int Count;
            internal TouchesState Touches;
        }

        private static class Native
        {
            [DllImport("HidShellLibrary.dll")]
            internal static extern int GetHidShellTouchScreenState(
                IntPtr handle,
                out TouchScreenState outState, uint port, uint direction);

            [DllImport("HidShellLibrary.dll")]
            internal static extern int SetHidShellTouchScreenState(
                IntPtr handle,
                ref TouchScreenState state, uint port, uint direction);
        }
    }
}
