﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector.Services
{
    using System;
    using System.Threading;
    using System.Threading.Tasks;

    /// <summary>
    /// PingPong サービスを提供します。
    /// </summary>
    internal sealed class PingPongService : ISessionService
    {
        private static readonly TimeSpan MonitoringOverhead =
            TimeSpan.FromMilliseconds(2000);

        private static readonly TimeSpan MonitoringInterval =
            TimeSpan.FromMilliseconds(1000);

        private static readonly byte[] PingChunk = new byte[
            Session.MessageSize]
        {
            (byte)MessageType.Ping, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
        };

        private readonly object syncObject = new object();

        private ISessionWriter writer = null;

        private bool waits = false;

        private bool isTimedOut = false;

        private CancellationTokenSource tokenSource = null;

        private Task task = null;

        /// <summary>
        /// PingPongService クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal PingPongService()
        {
        }

        /// <summary>
        /// セッションの情報を設定します。
        /// </summary>
        public SessionInfo Info { private get; set; }

        /// <summary>
        /// タイムアウトが発生したか否かを表す値を返します。
        /// </summary>
        internal bool IsTimedOut
        {
            get
            {
                lock (this.syncObject)
                {
                    return this.isTimedOut;
                }
            }
        }

        /// <summary>
        /// サービスを開始します。
        /// </summary>
        /// <param name="writer">セッションへの書き込みを扱うライターです。</param>
        public void Start(ISessionWriter writer)
        {
            lock (this.syncObject)
            {
                this.writer = writer;

                this.waits = false;

                this.isTimedOut = false;
            }

            this.tokenSource = new CancellationTokenSource();

            CancellationToken token = this.tokenSource.Token;

            this.task = Task.Run(() => this.Monitor(token));
        }

        /// <summary>
        /// サービスを停止します。
        /// </summary>
        public void Stop()
        {
            this.Cancel();
        }

        /// <summary>
        /// サービスをキャンセルします。
        /// </summary>
        public void Cancel()
        {
            this.tokenSource.Cancel();

            this.task.Wait();
            this.task.Dispose();
            this.task = null;

            this.tokenSource.Dispose();
            this.tokenSource = null;

            lock (this.syncObject)
            {
                this.writer = null;
                this.waits = false;
                this.isTimedOut = false;
            }
        }

        /// <summary>
        /// メッセージを受け付けます。
        /// </summary>
        /// <param name="message">メッセージです。</param>
        public void AcceptMessage(byte[] message)
        {
            if (message.Length != Session.MessageSize)
            {
                return;
            }

            if (message[0] == (byte)MessageType.Pong)
            {
                lock (this.syncObject)
                {
                    this.waits = false;
                }
            }
        }

        private void Monitor(CancellationToken token)
        {
            ISessionWriter writer = null;

            lock (this.syncObject)
            {
                writer = this.writer;
            }

            var isFirst = true;

            while (!token.IsCancellationRequested)
            {
                if (isFirst)
                {
                    isFirst = false;

                    token.WaitHandle.WaitOne(MonitoringOverhead);
                }
                else
                {
                    token.WaitHandle.WaitOne(MonitoringInterval);
                }

                ConnectionConfig config =
                    this.Info.UserConfig.GetConnectionConfig();

                if (!config.IsTimeoutEnabled)
                {
                    continue;
                }

                lock (this.syncObject)
                {
                    this.waits = true;
                }

                Task task = Task.Run(() =>
                {
                    try
                    {
                        writer.Write(PingChunk, 0, PingChunk.Length, token);
                    }
                    catch
                    {
                        // 何もしない
                    }
                });

                try
                {
                    if (task.Wait(config.Timeout))
                    {
                        token.WaitHandle.WaitOne(config.PollingInterval);
                    }
                }
                catch
                {
                    break;
                }

                lock (this.syncObject)
                {
                    if (this.waits)
                    {
                        this.isTimedOut = true;

                        break;
                    }
                }
            }
        }
    }
}
