﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector.Services
{
    using System;
    using System.Collections.Generic;
    using System.Drawing;
    using System.Runtime.InteropServices;
    using System.Threading;
    using System.Threading.Tasks;

    /// <summary>
    /// Capture サービスを提供します。
    /// </summary>
    internal sealed class CaptureService : ISessionService
    {
        private readonly object syncObject = new object();

        private ISessionWriter writer = null;

        private CaptureSession captureSession = null;

        /// <summary>
        /// CaptureService クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal CaptureService()
        {
        }

        /// <summary>
        /// セッションの情報を設定します。
        /// </summary>
        public SessionInfo Info { private get; set; }

        /// <summary>
        /// サービスを開始します。
        /// </summary>
        /// <param name="writer">セッションへの書き込みを扱うライターです。</param>
        public void Start(ISessionWriter writer)
        {
            lock (this.syncObject)
            {
                this.writer = writer;
            }
        }

        /// <summary>
        /// サービスを停止します。
        /// </summary>
        public void Stop()
        {
            this.Cancel();
        }

        /// <summary>
        /// サービスをキャンセルします。
        /// </summary>
        public void Cancel()
        {
            lock (this.syncObject)
            {
                if (this.captureSession != null)
                {
                    this.captureSession.OnCaptureCancel();

                    this.captureSession = null;
                }

                this.writer = null;
            }
        }

        /// <summary>
        /// メッセージを受け付けます。
        /// </summary>
        /// <param name="message">メッセージです。</param>
        public void AcceptMessage(byte[] message)
        {
            lock (this.syncObject)
            {
                if (this.captureSession != null)
                {
                    this.captureSession.OnAcceptMessage(message);

                    if (this.captureSession.IsCompleted)
                    {
                        this.captureSession = null;
                    }
                }
            }
        }

        /// <summary>
        /// Capture サービスのセッションを取得します。
        /// </summary>
        internal ICaptureSession GetCaptureSession()
        {
            lock (this.syncObject)
            {
                if (this.captureSession == null && this.writer != null)
                {
                    this.captureSession = new CaptureSession(this.writer);
                }

                return this.captureSession;
            }
        }

        private sealed class CaptureSession : ICaptureSession
        {
            private readonly ISessionWriter writer;

            private readonly object syncObject = new object();

            private bool isCompleted = false;

            private bool isStoppedByMouse = false;

            private Point position = new Point();

            private MouseButtons buttons = MouseButtons.None;

            private readonly Queue<byte[]> queue = new Queue<byte[]>();

            private readonly ManualResetEvent queuingEvent;

            private bool isTaskCancelled = false;

            private readonly CancellationTokenSource tokenSource;

            private readonly Task task;

            internal CaptureSession(ISessionWriter writer)
            {
                this.writer = writer;

                this.queuingEvent = new ManualResetEvent(false);

                this.tokenSource = new CancellationTokenSource();

                CancellationToken token = this.tokenSource.Token;

                this.task = Task.Run(() => this.Monitor(token));
            }

            public bool IsCompleted
            {
                get
                {
                    lock (this.syncObject)
                    {
                        return this.isCompleted;
                    }
                }
            }

            public bool IsStoppedByMouse
            {
                get
                {
                    lock (this.syncObject)
                    {
                        return this.isStoppedByMouse;
                    }
                }
            }

            public Point Position
            {
                get
                {
                    lock (this.syncObject)
                    {
                        return this.position;
                    }
                }
            }

            public void OnCaptureStart(
                bool isTriggerdByMouse,
                BorderType border, Rectangle screenSize, Point position)
            {
                var borderLength = new Func<ushort>(() =>
                {
                    switch (border)
                    {
                        case BorderType.Right:
                        case BorderType.Left:
                            return (ushort)(
                                screenSize.Bottom - screenSize.Top);
                        case BorderType.Top:
                        case BorderType.Bottom:
                            return (ushort)(
                                screenSize.Right - screenSize.Left);
                        default:
                            return 0;
                    }
                })();

                lock (this.syncObject)
                {
                    var chunk = new CaptureChunk()
                    {
                        Type = (byte)MessageType.CaptureStart,
                        BorderType = (byte)border,
                        IsTriggerdByMouse = (byte)(isTriggerdByMouse ? 1 : 0),
                        BorderLength = borderLength,
                        PositionX = (short)position.X,
                        PositionY = (short)position.Y,
                    };

                    byte[] bytes = StructConverter.ToBytes(chunk);

                    if (!this.isCompleted)
                    {
                        this.position = position;

                        this.queue.Enqueue(bytes);

                        this.queuingEvent.Set();
                    }
                }
            }

            public void OnCaptureStop()
            {
                lock (this.syncObject)
                {
                    var chunk = new CaptureChunk()
                    {
                        Type = (byte)MessageType.CaptureStop,
                    };

                    byte[] bytes = StructConverter.ToBytes(chunk);

                    if (!this.isCompleted)
                    {
                        this.CompleteCapture();

                        this.queue.Enqueue(bytes);

                        this.queuingEvent.Set();
                    }
                }
            }

            public void OnKeyDown(KeyboardEventArgs args)
            {
                lock (this.syncObject)
                {
                    var chunk = new KeyboardChunk()
                    {
                        Type = (byte)MessageType.KeyDown,
                        UsageId = (byte)args.UsageId,
                    };

                    byte[] bytes = StructConverter.ToBytes(chunk);

                    if (!this.isCompleted)
                    {
                        this.queue.Enqueue(bytes);

                        this.queuingEvent.Set();
                    }
                }
            }

            public void OnKeyUp(KeyboardEventArgs args)
            {
                lock (this.syncObject)
                {
                    var chunk = new KeyboardChunk()
                    {
                        Type = (byte)MessageType.KeyUp,
                        UsageId = (byte)args.UsageId,
                    };

                    byte[] bytes = StructConverter.ToBytes(chunk);

                    if (!this.isCompleted)
                    {
                        this.queue.Enqueue(bytes);

                        this.queuingEvent.Set();
                    }
                }
            }

            public void OnMouseMove(MouseEventArgs args)
            {
                lock (this.syncObject)
                {
                    var chunk = new MouseChunk()
                    {
                        Type = (byte)MessageType.MouseMove,
                        DeltaX = (short)args.X,
                        DeltaY = (short)args.Y,
                    };

                    byte[] bytes = StructConverter.ToBytes(chunk);

                    if (!this.isCompleted)
                    {
                        this.queue.Enqueue(bytes);

                        this.queuingEvent.Set();
                    }
                }
            }

            public void OnMouseDown(MouseEventArgs args)
            {
                lock (this.syncObject)
                {
                    this.buttons |= args.Button;

                    var chunk = new MouseChunk()
                    {
                        Type = (byte)MessageType.MouseButton,
                        Buttons = (byte)this.buttons,
                    };

                    byte[] bytes = StructConverter.ToBytes(chunk);

                    if (!this.isCompleted)
                    {
                        this.queue.Enqueue(bytes);

                        this.queuingEvent.Set();
                    }
                }
            }

            public void OnMouseUp(MouseEventArgs args)
            {
                lock (this.syncObject)
                {
                    this.buttons &= ~args.Button;

                    var chunk = new MouseChunk()
                    {
                        Type = (byte)MessageType.MouseButton,
                        Buttons = (byte)this.buttons,
                    };

                    byte[] bytes = StructConverter.ToBytes(chunk);

                    if (!this.isCompleted)
                    {
                        this.queue.Enqueue(bytes);

                        this.queuingEvent.Set();
                    }
                }
            }

            public void OnMouseWheel(MouseEventArgs args)
            {
                lock (this.syncObject)
                {
                    var chunk = new MouseChunk()
                    {
                        Type = (byte)MessageType.MouseWheel,
                        WheelDelta = (short)args.WheelDelta,
                    };

                    byte[] bytes = StructConverter.ToBytes(chunk);

                    if (!this.isCompleted)
                    {
                        this.queue.Enqueue(bytes);

                        this.queuingEvent.Set();
                    }
                }
            }

            internal void OnCaptureCancel()
            {
                lock (this.syncObject)
                {
                    this.CompleteCapture();
                }

                this.CancelTask();
            }

            internal void OnAcceptMessage(byte[] message)
            {
                if (message.Length != Session.MessageSize)
                {
                    return;
                }

                var messageType = (MessageType)message[0];

                if (messageType != MessageType.CaptureStop)
                {
                    return;
                }

                var chunk = StructConverter.FromBytes<CaptureChunk>(message);

                var position = new Point(chunk.PositionX, chunk.PositionY);

                lock (this.syncObject)
                {
                    if (!this.isCompleted)
                    {
                        this.isStoppedByMouse = true;

                        this.position = position;

                        this.CompleteCapture();
                    }
                }

                this.CancelTask();
            }

            private void CompleteCapture()
            {
                this.isCompleted = true;

                this.queue.Clear();
            }

            private void CancelTask()
            {
                if (!this.isTaskCancelled)
                {
                    this.isTaskCancelled = true;

                    this.tokenSource.Cancel();

                    this.task.Wait();
                    this.task.Dispose();

                    this.tokenSource.Dispose();
                }
            }

            private void Monitor(CancellationToken token)
            {
                while (!token.IsCancellationRequested)
                {
                    byte[] bytes = null;

                    lock (this.syncObject)
                    {
                        if (this.queue.Count > 0)
                        {
                            bytes = this.queue.Dequeue();
                        }
                        else if (this.isCompleted)
                        {
                            break;
                        }
                    }

                    if (bytes == null)
                    {
                        WaitHandle.WaitAny(new[]
                        {
                            token.WaitHandle, this.queuingEvent
                        });
                    }
                    else
                    {
                        try
                        {
                            this.writer.Write(bytes, 0, bytes.Length, token);
                        }
                        catch
                        {
                            lock (this.syncObject)
                            {
                                if (!this.isCompleted)
                                {
                                    this.CompleteCapture();
                                }
                            }

                            break;
                        }
                    }
                }
            }
        }

        [StructLayout(LayoutKind.Explicit)]
        private struct CaptureChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(4)]
            public byte BorderType;

            [FieldOffset(5)]
            public byte IsTriggerdByMouse;

            [FieldOffset(6)]
            public ushort BorderLength;

            [FieldOffset(8)]
            public short PositionX;

            [FieldOffset(10)]
            public short PositionY;
        }

        [StructLayout(LayoutKind.Explicit)]
        private struct KeyboardChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(4)]
            public byte UsageId;

            [FieldOffset(8)]
            private int Padding;
        }

        [StructLayout(LayoutKind.Explicit)]
        private struct MouseChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(4)]
            public byte Buttons;

            [FieldOffset(6)]
            public short WheelDelta;

            [FieldOffset(8)]
            public short DeltaX;

            [FieldOffset(10)]
            public short DeltaY;
        }
    }
}
