﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector.Services
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Runtime.InteropServices;
    using System.Threading;
    using System.Threading.Tasks;

    /// <summary>
    /// AbstractedPad サービスを提供します。
    /// </summary>
    internal sealed class AbstractedPadService : ISessionService
    {
        private const int AbstractedPadCountMax = 8;

        private static readonly TimeSpan MonitorInterval =
            TimeSpan.FromMilliseconds(500);

        private static readonly TimeSpan SamplerInterval =
            TimeSpan.FromMilliseconds(7);

        private readonly object syncObject = new object();

        private readonly IDictionary<int, GamePadConfig> configMap =
            new Dictionary<int, GamePadConfig>();

        private readonly AbstractedPadState[] hostStates =
            new AbstractedPadState[AbstractedPadCountMax];

        private readonly AbstractedPadState[] targetStates =
            new AbstractedPadState[AbstractedPadCountMax];

        private readonly AbstractedPadState[] autoStates =
            new AbstractedPadState[AbstractedPadCountMax];

        private readonly AbstractedPadState[] lastStates =
            new AbstractedPadState[AbstractedPadCountMax];

        private readonly AbstractedPadState[] prevStates =
            new AbstractedPadState[AbstractedPadCountMax];

        private uint port = HidShellPortAccessor.DefaultPort;

        private ISessionWriter writer = null;

        private CancellationTokenSource monitorTokenSource = null;

        private Task monitorTask = null;

        private CancellationTokenSource samplerTokenSource = null;

        private Task samplerTask = null;

        /// <summary>
        /// AbstractedPad クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal AbstractedPadService()
        {
        }

        [Flags]
        private enum AbstractedPadAttribute : uint
        {
            None                    = 0,
            IsConnected             = 1u,
            IsEnableSixAxisSensor   = 1u << 1,
        }

        private enum AbstractedPadBatteryLevel : byte
        {
            Empty       = 0,
            Critical    = 1,
            Low         = 2,
            Medium      = 3,
            High        = 4,
        }

        private enum AbstractedPadColorType : byte
        {
            Main = 0,
            Sub = 1,
        }

        private enum AbstractedPadInterfaceType : byte
        {
            Unknown = 0,
            Bluetooth = 1,
            Rail = 2,
            Usb = 3,
        }

        private enum AbstractedPadDeviceType : byte
        {
            SwitchProController = 0,
            HandheldJoyLeft = 2,
            HandheldJoyRight = 3,
            JoyConLeft = 4,
            JoyConRight = 5,
        }

        [Flags]
        private enum AbstractedPadButtons : uint
        {
            None        = 0,
            A           = 1u,
            B           = 1u << 1,
            X           = 1u << 2,
            Y           = 1u << 3,
            StickL      = 1u << 4,
            StickR      = 1u << 5,
            L           = 1u << 6,
            R           = 1u << 7,
            ZL          = 1u << 8,
            ZR          = 1u << 9,
            Start       = 1u << 10,
            Select      = 1u << 11,
            Left        = 1u << 12,
            Up          = 1u << 13,
            Right       = 1u << 14,
            Down        = 1u << 15,
            SL          = 1u << 16,
            SR          = 1u << 17,
            Home        = 1u << 18,
            Shot        = 1u << 19,
            StickLLeft  = 1u << 20,
            StickLUp    = 1u << 21,
            StickLRight = 1u << 22,
            StickLDown  = 1u << 23,
            StickRLeft  = 1u << 24,
            StickRUp    = 1u << 25,
            StickRRight = 1u << 26,
            StickRDown  = 1u << 27,
        };

        /// <summary>
        /// セッションの情報を設定します。
        /// </summary>
        public SessionInfo Info { private get; set; }

        /// <summary>
        /// サービスを開始します。
        /// </summary>
        /// <param name="writer">セッションへの書き込みを扱うライターです。</param>
        public void Start(ISessionWriter writer)
        {
            lock (this.syncObject)
            {
                this.configMap.Clear();

                for (int i = 0; i < AbstractedPadCountMax; ++i)
                {
                    this.targetStates[i] = new AbstractedPadState();
                    this.hostStates[i] = this.targetStates[i];
                    this.autoStates[i] = this.targetStates[i];
                    this.lastStates[i] = this.targetStates[i];
                    this.prevStates[i] = this.targetStates[i];
                }
            }

            this.writer = writer;

            var ioPortManager = this.Info.IoPortManager;

            this.port = ioPortManager.GetPort(this.Info.Name);

            ioPortManager.Polling += this.OnPolling;

            var gamePadManager = this.Info.GamePadManager;
            gamePadManager.ButtonDown += this.OnButtonDown;
            gamePadManager.ButtonUp += this.OnButtonUp;
            gamePadManager.StickMoved += this.OnStickMoved;

            this.monitorTokenSource = new CancellationTokenSource();
            this.monitorTask = Task.Run(
                () => this.Monitor(this.monitorTokenSource.Token));

            this.samplerTokenSource = new CancellationTokenSource();
            this.samplerTask = Task.Run(
                () => this.Sample(this.samplerTokenSource.Token));
        }

        /// <summary>
        /// サービスを停止します。
        /// </summary>
        public void Stop()
        {
            this.Cancel();
        }

        /// <summary>
        /// サービスをキャンセルします。
        /// </summary>
        public void Cancel()
        {
            this.samplerTokenSource.Cancel();

            this.monitorTokenSource.Cancel();

            this.samplerTask.Wait();
            this.samplerTask.Dispose();
            this.samplerTask = null;

            this.monitorTask.Wait();
            this.monitorTask.Dispose();
            this.monitorTask = null;

            this.samplerTokenSource.Dispose();
            this.samplerTokenSource = null;

            this.monitorTokenSource.Dispose();
            this.monitorTokenSource = null;

            var gamePadManager = this.Info.GamePadManager;
            gamePadManager.ButtonDown -= this.OnButtonDown;
            gamePadManager.ButtonUp -= this.OnButtonUp;
            gamePadManager.StickMoved -= this.OnStickMoved;

            var ioPortManager = this.Info.IoPortManager;
            ioPortManager.Polling -= this.OnPolling;

            this.writer = null;

            // エミュレート状態をリセット
            using (var accessor = new HidShellPortAccessor())
            {
                var clearState = new AbstractedPadState();

                for (uint i = 0; i < AbstractedPadCountMax; ++i)
                {
                    SetAbstractedPadAutoPilotState(
                        accessor.Handle, ref clearState, i, this.port);

                    SetAbstractedPadState(
                        accessor.Handle, ref clearState, i, this.port);
                }
            }
        }

        /// <summary>
        /// メッセージを受け付けます。
        /// </summary>
        /// <param name="message">メッセージです。</param>
        public void AcceptMessage(byte[] message)
        {
            if (message.Length != Session.MessageSize)
            {
                return;
            }

            switch ((MessageType)message[0])
            {
                case MessageType.AbstractedPadButton:
                    {
                        var chunk = StructConverter.FromBytes<
                            AbstractedPadButtonChunk>(message);

                        this.AcceptAbstractedPadButtonMessage(chunk.Id, chunk.Buttons);

                        break;
                    }

                case MessageType.AbstractedPadStick:
                    {
                        var chunk = StructConverter.FromBytes<
                            AbstractedPadStickChunk>(message);

                        var stickType = (GamePadStickType)chunk.StickType;

                        this.AcceptAbstractedPadStickMessage(
                            chunk.Id, stickType, chunk.X, chunk.Y);

                        break;
                    }

                case MessageType.AbstractedPadPowerState:
                    {
                        var chunk = StructConverter.FromBytes<
                            AbstractedPadPowerChunk>(message);

                        var batteryLevel = (AbstractedPadBatteryLevel)chunk.BatteryLevel;

                        this.AcceptAbstractedPadPowerMessage(chunk.Id, chunk.IsPowered, chunk.IsCharging, batteryLevel);

                        break;
                    }

                case MessageType.AbstractedPadDeviceData:
                    {
                        var chunk = StructConverter.FromBytes<
                            AbstractedPadDeviceDataChunk>(message);

                        var deviceType = (AbstractedPadDeviceType)chunk.DeviceType;
                        var interfaceType = (AbstractedPadInterfaceType)chunk.InterfaceType;
                        var attribute = (AbstractedPadAttribute)chunk.Attribute;

                        this.AcceptAbstractedPadDeviceDataMessage(chunk.Id, deviceType, interfaceType, attribute);

                        break;
                    }

                case MessageType.AbstractedPadColor:
                    {
                        var chunk = StructConverter.FromBytes<
                            AbstractedPadColorChunk>(message);

                        var colorType = (AbstractedPadColorType)chunk.ColorType;
                        var colorValue = new AbstractedPadColorValue();
                        colorValue.Red = chunk.Red;
                        colorValue.Green = chunk.Green;
                        colorValue.Blue = chunk.Blue;
                        colorValue.Alpha = chunk.Alpha;

                        this.AcceptAbstractedPadColorMessage(chunk.Id, colorType, colorValue);

                        break;
                    }

                default:
                    break;
            }
        }

        private static void GenerateChunks(
            List<byte[]> chunks,
            int deviceNumber,
            ref AbstractedPadState nextState,
            ref AbstractedPadState lastState)
        {
            // デバイス色（メイン）
            if (nextState.Color.Main.Red != lastState.Color.Main.Red
                || nextState.Color.Main.Green != lastState.Color.Main.Green
                || nextState.Color.Main.Blue != lastState.Color.Main.Blue
                || nextState.Color.Main.Alpha != lastState.Color.Main.Alpha)
            {
                var chunk = new AbstractedPadColorChunk()
                {
                    Type = (byte)MessageType.AbstractedPadColor,
                    Id = (byte)deviceNumber,
                    ColorType = (byte)AbstractedPadColorType.Main,
                    Red = nextState.Color.Main.Red,
                    Green = nextState.Color.Main.Green,
                    Blue = nextState.Color.Main.Blue,
                    Alpha = nextState.Color.Main.Alpha,
                };
                chunks.Add(StructConverter.ToBytes(chunk));
            }

            // デバイス色（サブ）
            if (nextState.Color.Sub.Red != lastState.Color.Sub.Red
                || nextState.Color.Sub.Green != lastState.Color.Sub.Green
                || nextState.Color.Sub.Blue != lastState.Color.Sub.Blue
                || nextState.Color.Sub.Alpha != lastState.Color.Sub.Alpha)
            {
                var chunk = new AbstractedPadColorChunk()
                {
                    Type = (byte)MessageType.AbstractedPadColor,
                    Id = (byte)deviceNumber,
                    ColorType = (byte)AbstractedPadColorType.Sub,
                    Red = nextState.Color.Sub.Red,
                    Green = nextState.Color.Sub.Green,
                    Blue = nextState.Color.Sub.Blue,
                    Alpha = nextState.Color.Sub.Alpha,
                };
                chunks.Add(StructConverter.ToBytes(chunk));
            }

            // ボタン
            if (nextState.Buttons != lastState.Buttons)
            {
                var chunk = new AbstractedPadButtonChunk()
                {
                    Type = (byte)MessageType.AbstractedPadButton,
                    Id = (byte)deviceNumber,
                    Buttons = (uint)nextState.Buttons,
                };
                chunks.Add(StructConverter.ToBytes(chunk));
            }

            // L スティック
            if (nextState.StickL.X != lastState.StickL.X ||
                nextState.StickL.Y != lastState.StickL.Y)
            {
                var chunk = new AbstractedPadStickChunk()
                {
                    Type = (byte)MessageType.AbstractedPadStick,
                    Id = (byte)deviceNumber,
                    StickType = (byte)GamePadStickType.Left,
                    X = (short)nextState.StickL.X,
                    Y = (short)nextState.StickL.Y,
                };

                chunks.Add(StructConverter.ToBytes(chunk));
            }

            // R スティック
            if (nextState.StickR.X != lastState.StickR.X ||
                nextState.StickR.Y != lastState.StickR.Y)
            {
                var chunk = new AbstractedPadStickChunk()
                {
                    Type = (byte)MessageType.AbstractedPadStick,
                    Id = (byte)deviceNumber,
                    StickType = (byte)GamePadStickType.Right,
                    X = (short)nextState.StickR.X,
                    Y = (short)nextState.StickR.Y,
                };

                chunks.Add(StructConverter.ToBytes(chunk));
            }

            // 電源状態
            if (nextState.PowerInfo.IsPowered != lastState.PowerInfo.IsPowered
                || nextState.PowerInfo.IsCharging != lastState.PowerInfo.IsCharging
                || nextState.PowerInfo.BatteryLevel != lastState.PowerInfo.BatteryLevel)
            {
                var chunk = new AbstractedPadPowerChunk()
                {
                    Type = (byte)MessageType.AbstractedPadPowerState,
                    Id = (byte)deviceNumber,
                    IsPowered = (byte)Convert.ToByte(nextState.PowerInfo.IsPowered),
                    IsCharging = (byte)Convert.ToByte(nextState.PowerInfo.IsCharging),
                    BatteryLevel = (byte)nextState.PowerInfo.BatteryLevel,
                };
                chunks.Add(StructConverter.ToBytes(chunk));
            }

            // attribute フラグセット
            if (nextState.Attribute != lastState.Attribute
               || nextState.DeviceType != lastState.DeviceType
               || nextState.InterfaceType != lastState.InterfaceType)
            {
                var chunk = new AbstractedPadDeviceDataChunk()
                {
                    Type = (byte)MessageType.AbstractedPadDeviceData,
                    Id = (byte)deviceNumber,
                    InterfaceType = (byte)nextState.InterfaceType,
                    DeviceType = (byte)nextState.DeviceType,
                    Attribute = (uint)(nextState.Attribute),
                };
                chunks.Add(StructConverter.ToBytes(chunk));
            }
        }

        private static void SetAbstractedPadState(
            IntPtr handle, ref AbstractedPadState state, uint id, uint port)
        {
            var result = (HidShellResult)
                Native.SetHidShellAbstractedPadState(
                    handle, ref state, id, port, HidShellPortAccessor.Out);

            if (result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }
        }

        private static void GetAbstractedPadAutoPilotState(
            IntPtr handle, out AbstractedPadState outState, uint id, uint port)
        {
            var result = (HidShellResult)
                Native.GetHidShellAbstractedPadState(
                    handle, out outState, id, port, HidShellPortAccessor.In);

            switch (result)
            {
                case HidShellResult.Success:
                    return;

                case HidShellResult.StateNotSet:
                    outState = new AbstractedPadState();
                    return;

                default:
                    throw new HidShellException(result);
            }
        }

        private static void SetAbstractedPadAutoPilotState(
            IntPtr handle, ref AbstractedPadState state, uint id, uint port)
        {
            var result = (HidShellResult)Native.SetHidShellAbstractedPadState(
                handle, ref state, id, port, HidShellPortAccessor.In);

            if( result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }
        }

        private static bool AreEqual(
            ref AbstractedPadColorValue lhs, ref AbstractedPadColorValue rhs)
        {
            if (lhs.Red != rhs.Red)
            {
                return false;
            }

            if (lhs.Green != rhs.Green)
            {
                return false;
            }

            if (lhs.Blue != rhs.Blue)
            {
                return false;
            }

            if (lhs.Alpha != rhs.Alpha)
            {
                return false;
            }

            return true;
        }

        private static bool AreEqual(
            ref AbstractedPadColor lhs, ref AbstractedPadColor rhs)
        {
            if (!AreEqual(ref lhs.Main, ref rhs.Main))
            {
                return false;
            }

            if (!AreEqual(ref lhs.Sub, ref rhs.Sub))
            {
                return false;
            }

            return true;
        }

        private static void SelectAbstractedPadState(
            out AbstractedPadState outState,
            ref AbstractedPadState majorState,
            ref AbstractedPadState minorState)
        {
            var isConnected = AbstractedPadAttribute.IsConnected;

            if (!minorState.Attribute.HasFlag(isConnected) ||
                majorState.Buttons != AbstractedPadButtons.None ||
                majorState.StickL.X != 0 || majorState.StickL.Y != 0 ||
                majorState.StickR.X != 0 || majorState.StickR.Y != 0)
            {
                outState = majorState;
            }
            else
            {
                outState = minorState;
            }
        }

        private void AcceptAbstractedPadButtonMessage(int id, uint buttons)
        {
            if(id < 0 && id >= AbstractedPadCountMax)
            {
                return;
            }
            lock (this.syncObject)
            {
                this.targetStates[id].Buttons = (AbstractedPadButtons)buttons;
            }
        }

        private void AcceptAbstractedPadStickMessage(
            int id, GamePadStickType stickType, int x, int y)
        {
            if (id < 0 && id >= AbstractedPadCountMax)
            {
                return;
            }
            lock (this.syncObject)
            {
                switch (stickType)
                {
                    case GamePadStickType.Left:
                        this.targetStates[id].StickL.X = x;
                        this.targetStates[id].StickL.Y = y;
                        break;

                    case GamePadStickType.Right:
                        this.targetStates[id].StickR.X = x;
                        this.targetStates[id].StickR.Y = y;
                        break;

                    default:
                        break;
                }
            }
        }

        private void AcceptAbstractedPadPowerMessage(
            int id, byte isPowered, byte isCharging, AbstractedPadBatteryLevel batteryLevel)
        {
            if (id < 0 && id >= AbstractedPadCountMax)
            {
                return;
            }
            lock (this.syncObject)
            {
                AbstractedPadPowerInfo info;
                info.IsPowered = isPowered;
                info.IsCharging = isCharging;
                info.BatteryLevel = batteryLevel;
                this.targetStates[id].PowerInfo = info;
            }
        }

        private void AcceptAbstractedPadDeviceDataMessage(
            int id, AbstractedPadDeviceType deviceType, AbstractedPadInterfaceType interfaceType, AbstractedPadAttribute attribute)
        {
            if (id < 0 && id >= AbstractedPadCountMax)
            {
                return;
            }

            lock (this.syncObject)
            {
                if(attribute.HasFlag(AbstractedPadAttribute.IsConnected))
                {
                    this.targetStates[id].DeviceType = deviceType;
                    this.targetStates[id].InterfaceType = interfaceType;
                    this.targetStates[id].Attribute = attribute;
                }
                else
                {
                    this.targetStates[id] = new AbstractedPadState();
                    this.targetStates[id].Attribute = attribute;
                }
            }
        }

        private void AcceptAbstractedPadColorMessage(
            int id, AbstractedPadColorType colorType, AbstractedPadColorValue colorValue)
        {
            if (id < 0 && id >= AbstractedPadCountMax)
            {
                return;
            }

            lock (this.syncObject)
            {
                switch(colorType)
                {
                    case AbstractedPadColorType.Main:
                        this.targetStates[id].Color.Main = colorValue;
                        break;
                    case AbstractedPadColorType.Sub:
                        this.targetStates[id].Color.Sub = colorValue;
                        break;
                    default:
                        break;
                }
            }
        }

        private void OnButtonDown(object sender, GamePadEventArgs args)
        {
            lock (this.syncObject)
            {
                GamePadConfig config;

                if (this.configMap.TryGetValue(args.SamplerId, out config))
                {
                    int playerNumber = config.PlayerNumber;

                    if (this.hostStates[playerNumber].Attribute.HasFlag(
                            AbstractedPadAttribute.IsConnected))
                    {
                        this.hostStates[playerNumber].Buttons |=
                            (AbstractedPadButtons)args.Button;
                    }
                }
            }
        }

        private void OnButtonUp(object sender, GamePadEventArgs args)
        {
            lock (this.syncObject)
            {
                GamePadConfig config;

                if (this.configMap.TryGetValue(args.SamplerId, out config))
                {
                    int playerNumber = config.PlayerNumber;

                    if (this.hostStates[playerNumber].Attribute.HasFlag(
                            AbstractedPadAttribute.IsConnected))
                    {
                        this.hostStates[playerNumber].Buttons &=
                            ~(AbstractedPadButtons)args.Button;
                    }
                }
            }
        }

        private void OnStickMoved(object sender, GamePadEventArgs args)
        {
            lock (this.syncObject)
            {
                GamePadConfig config;

                if (this.configMap.TryGetValue(args.SamplerId, out config))
                {
                    int playerNumber = config.PlayerNumber;

                    if (this.hostStates[playerNumber].Attribute.HasFlag(
                            AbstractedPadAttribute.IsConnected))
                    {
                        switch (args.StickType)
                        {
                            case GamePadStickType.Left:
                                this.hostStates[playerNumber].StickL.X =
                                    args.Stick.X;
                                this.hostStates[playerNumber].StickL.Y =
                                    args.Stick.Y;
                                break;

                            case GamePadStickType.Right:
                                this.hostStates[playerNumber].StickR.X =
                                    args.Stick.X;
                                this.hostStates[playerNumber].StickR.Y =
                                    args.Stick.Y;
                                break;

                            default:
                                break;
                        }
                    }
                }
            }
        }

        private void OnPolling(object sender, HidShellPortAccessor accessor)
        {
            lock (this.syncObject)
            {
                for (uint i = 0; i < AbstractedPadCountMax; ++i)
                {
                    GetAbstractedPadAutoPilotState(
                        accessor.Handle, out this.autoStates[i], i, this.port);

                    SetAbstractedPadState(
                        accessor.Handle, ref this.targetStates[i], i, this.port);
                }
            }
        }

        private void Monitor(CancellationToken token)
        {
            var configMap = new Dictionary<int, GamePadConfig>();

            while (!token.IsCancellationRequested)
            {
                List<GamePadInfo> infos =
                    this.Info.GamePadManager.GetGamePadInfos();

                configMap.Clear();

                IList<string> names = this.Info.GetSessionNames();

                foreach (GamePadInfo info in infos)
                {
                    GamePadConfig config =
                        this.Info.UserConfig.GetGamePadConfig(info.Name) ??
                        new GamePadConfig()
                        {
                            Name = info.Name,
                            PlayerNumber = info.SamplerId
                        };

                    bool isAvailable = config.IsEnabled && (
                        names.Contains(config.TargetName)
                            ? (config.TargetName == this.Info.Name)
                            : (names.Count > 0 && names[0] == this.Info.Name));

                    if (isAvailable) { configMap[info.SamplerId] = config; }
                }

                var playerNumbers = configMap.Values
                    .Select(x => x.PlayerNumber).Distinct().ToArray<int>();

                lock (this.syncObject)
                {
                    this.configMap.Clear();

                    foreach (var pair in configMap)
                    {
                        this.configMap.Add(pair);
                    }

                    for (int i = 0; i < AbstractedPadCountMax; ++i)
                    {
                        if (!playerNumbers.Contains(i))
                        {
                            this.hostStates[i] = new AbstractedPadState();
                        }
                        else
                        {
                            this.hostStates[i].DeviceType =
                                AbstractedPadDeviceType.SwitchProController;
                            this.hostStates[i].Attribute =
                                AbstractedPadAttribute.IsConnected;
                            this.hostStates[i].Color.Main.Red = 45;
                            this.hostStates[i].Color.Main.Green = 45;
                            this.hostStates[i].Color.Main.Blue = 45;
                            this.hostStates[i].Color.Main.Alpha = 255;
                            this.hostStates[i].Color.Sub.Red = 230;
                            this.hostStates[i].Color.Sub.Green = 230;
                            this.hostStates[i].Color.Sub.Blue = 230;
                            this.hostStates[i].Color.Sub.Alpha = 255;
                            this.hostStates[i].InterfaceType =
                                AbstractedPadInterfaceType.Bluetooth;
                            this.hostStates[i].PowerInfo.BatteryLevel =
                                AbstractedPadBatteryLevel.High;
                        }
                    }
                }

                token.WaitHandle.WaitOne(MonitorInterval);
            }
        }

        private void Sample(CancellationToken token)
        {
            var works = true;

            var stopwatch = new Stopwatch();

            var chunks = new List<byte[]>();

            var bareState = new AbstractedPadState();

            var hostState = new AbstractedPadState();

            var lastState = new AbstractedPadState();

            var hostStates = new AbstractedPadState[AbstractedPadCountMax];

            var autoStates = new AbstractedPadState[AbstractedPadCountMax];

            while (works && !token.IsCancellationRequested)
            {
                stopwatch.Start();

                lock (this.syncObject)
                {
                    Array.Copy(
                        this.hostStates, 0, hostStates, 0, AbstractedPadCountMax);

                    Array.Copy(
                        this.autoStates, 0, autoStates, 0, AbstractedPadCountMax);
                }

                for (int i = 0; i < AbstractedPadCountMax; ++i)
                {
                    SelectAbstractedPadState(
                        out lastState,
                        ref this.lastStates[i], ref this.prevStates[i]);

                    SelectAbstractedPadState(
                        out hostState,
                        ref this.hostStates[i], ref this.autoStates[i]);

                    if (hostState.DeviceType == lastState.DeviceType &&
                        AreEqual(ref hostState.Color, ref lastState.Color))
                    {
                        GenerateChunks(
                            chunks, i, ref hostState, ref lastState);
                    }
                    else
                    {
                        var isConnected = AbstractedPadAttribute.IsConnected;

                        if (lastState.Attribute.HasFlag(isConnected))
                        {
                            GenerateChunks(
                                chunks, i, ref bareState, ref lastState);
                        }

                        if (hostState.Attribute.HasFlag(isConnected))
                        {
                            GenerateChunks(
                                chunks, i, ref hostState, ref bareState);
                        }
                    }
                }

                foreach (var chunk in chunks)
                {
                    try
                    {
                        this.writer.Write(chunk, 0, chunk.Length, token);
                    }
                    catch
                    {
                        works = false;

                        break;
                    }
                }

                chunks.Clear();

                Array.Copy(hostStates, 0, this.lastStates, 0, AbstractedPadCountMax);

                Array.Copy(autoStates, 0, this.prevStates, 0, AbstractedPadCountMax);

                stopwatch.Stop();

                if (stopwatch.Elapsed < SamplerInterval &&
                    works && !token.IsCancellationRequested)
                {
                    token.WaitHandle.WaitOne(
                        SamplerInterval - stopwatch.Elapsed);
                }

                stopwatch.Reset();
            }
        }

        private struct AbstractedPadColorValue
        {
            internal byte Red;
            internal byte Green;
            internal byte Blue;
            internal byte Alpha;
        }

        private struct AbstractedPadStickState
        {
            internal int X;
            internal int Y;
        }

        [StructLayout(LayoutKind.Explicit)]
        private struct AbstractedPadPowerInfo
        {
            [FieldOffset(0)]
            public byte IsPowered;
            [FieldOffset(1)]
            public byte IsCharging;
            [FieldOffset(3)]
            public AbstractedPadBatteryLevel   BatteryLevel;
        }

        private struct AbstractedPadColor
        {
            internal AbstractedPadColorValue Main;
            internal AbstractedPadColorValue Sub;
        }

        // デバイスタイプ・インターフェース
        [StructLayout(LayoutKind.Explicit)]
        private struct AbstractedPadDeviceDataChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(4)]
            public byte Id;

            [FieldOffset(5)]
            public byte InterfaceType;

            [FieldOffset(6)]
            public byte DeviceType;

            [FieldOffset(8)]
            public uint Attribute;
        }

        // デバイスカラー
        [StructLayout(LayoutKind.Explicit)]
        private struct AbstractedPadColorChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(4)]
            public byte Id;

            [FieldOffset(5)]
            public byte ColorType;

            [FieldOffset(8)]
            public byte Red;

            [FieldOffset(9)]
            public byte Green;

            [FieldOffset(10)]
            public byte Blue;

            [FieldOffset(11)]
            public byte Alpha;
        }

        // PowerInfo
        [StructLayout(LayoutKind.Explicit)]
        private struct AbstractedPadPowerChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(4)]
            public byte Id;

            [FieldOffset(5)]
            public byte IsPowered;

            [FieldOffset(6)]
            public byte IsCharging;

            [FieldOffset(7)]
            public byte BatteryLevel;

            [FieldOffset(8)]
            public uint Padding;
        }

        // Buttons
        [StructLayout(LayoutKind.Explicit)]
        private struct AbstractedPadButtonChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(4)]
            public byte Id;

            [FieldOffset(8)]
            public uint Buttons;
        }

        // Stick
        [StructLayout(LayoutKind.Explicit)]
        private struct AbstractedPadStickChunk
        {
            [FieldOffset(0)]
            public byte Type;

            [FieldOffset(4)]
            public byte Id;

            [FieldOffset(5)]
            public byte StickType;

            [FieldOffset(8)]
            public short X;

            [FieldOffset(10)]
            public short Y;
        }

        private struct AbstractedPadState
        {
            internal AbstractedPadDeviceType    DeviceType;
            internal AbstractedPadAttribute     Attribute;
            internal AbstractedPadColor         Color;
            internal AbstractedPadInterfaceType InterfaceType;
            internal AbstractedPadPowerInfo     PowerInfo;
            internal AbstractedPadButtons       Buttons;
            internal AbstractedPadStickState    StickL;
            internal AbstractedPadStickState    StickR;
        }

        private static class Native
        {
            [DllImport("HidShellLibrary.dll")]
            internal static extern int GetHidShellAbstractedPadState(
                IntPtr handle,
                out AbstractedPadState outState, uint id, uint port, uint direction);

            [DllImport("HidShellLibrary.dll")]
            internal static extern int SetHidShellAbstractedPadState(
                IntPtr handle,
                ref AbstractedPadState state, uint id, uint port, uint direction);
        }
    }
}
