﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    /// <summary>
    /// マウスイベントのデータを提供します。
    /// </summary>
    internal class MouseEventArgs
    {
        /// <summary>
        /// MouseEventArgs クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="x">マウスカーソルの x 座標です。</param>
        /// <param name="y">マウスカーソルの y 座標です。</param>
        internal MouseEventArgs(int x, int y)
        {
            this.X = x;
            this.Y = y;
            this.Button = MouseButtons.None;
            this.WheelDelta = 0;
        }

        /// <summary>
        /// MouseEventArgs クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="x">マウスカーソルの x 座標です。</param>
        /// <param name="y">マウスカーソルの y 座標です。</param>
        /// <param name="button">マウスボタンです。</param>
        internal MouseEventArgs(int x, int y, MouseButtons button)
        {
            this.X = x;
            this.Y = y;
            this.Button = button;
            this.WheelDelta = 0;
        }

        /// <summary>
        /// MouseEventArgs クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="x">マウスカーソルの x 座標です。</param>
        /// <param name="y">マウスカーソルの y 座標です。</param>
        /// <param name="wheelDelta">マウスホイールの回転差分です。</param>
        internal MouseEventArgs(int x, int y, int wheelDelta)
        {
            this.X = x;
            this.Y = y;
            this.Button = MouseButtons.None;
            this.WheelDelta = wheelDelta;
        }

        /// <summary>
        /// マウスカーソルの x 座標を取得します。
        /// </summary>
        internal int X { get; private set; }

        /// <summary>
        /// マウスカーソルの y 座標を取得します。
        /// </summary>
        internal int Y { get; private set; }

        /// <summary>
        /// マウスボタンを取得します。
        /// </summary>
        internal MouseButtons Button { get; private set; }

        /// <summary>
        /// マウスホイールの回転差分を取得します。
        /// </summary>
        internal int WheelDelta { get; private set; }

        /// <summary>
        /// マウスイベントのデータの印字可能な表現を返します。
        /// </summary>
        /// <returns>マウスイベントのデータの印字可能な表現です。</returns>
        public override string ToString()
        {
            return string.Format(
                "({0,4}, {1,4}) {2,-7} {3,4}",
                this.X, this.Y, this.Button, this.WheelDelta);
        }
    }
}
