﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector
{
    using System;
    using System.Collections.Generic;
    using System.Drawing;
    using System.Globalization;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Management;
    using System.Runtime.InteropServices;
    using System.Text.RegularExpressions;
    using System.Threading;
    using System.Threading.Tasks;

    /// <summary>
    /// ゲームパッドを管理します。
    /// </summary>
    internal sealed class GamePadManager : IDisposable
    {
        private const int GamePadCountMax = 16;

        private const string MmapName =
            "NN_HID_DETAIL_WINDOWSGENERICPADENTRIES";

        private const int GamePadStickMaxValue = short.MaxValue;

        private const uint JOYERR_NOERROR = 0;
        private const uint JOYERR_UNPLUGGED = 167;

        private const uint JOY_POVCENTERED = 65535;
        private const uint JOY_POVFORWARD = 0;
        private const uint JOY_POVRIGHT = 9000;
        private const uint JOY_POVBACKWARD = 18000;
        private const uint JOY_POVLEFT = 27000;

        private const uint JOY_USEDEADZONE = 0x0800;
        private const uint JOY_RETURNALL = 0x00ff;

        private const uint JOYCAPS_HASZ = 0x0001;
        private const uint JOYCAPS_HASR = 0x0002;
        private const uint JOYCAPS_HASU = 0x0004;
        private const uint JOYCAPS_HASV = 0x0008;
        private const uint JOYCAPS_HASPOV = 0x0010;

        private const int XINPUT_GAMEPAD_LEFT_THUMB_DEADZONE = 7849;
        private const int XINPUT_GAMEPAD_RIGHT_THUMB_DEADZONE = 8689;

        private readonly object syncObject = new object();

        private readonly GamePadHolder gamePadHolder = new GamePadHolder();

        private readonly GamePadBroker gamePadBroker;

        private readonly List<GamePadSampler> samplers =
            new List<GamePadSampler>();

        private IDictionary<int, string> gamePadNameMap =
            new Dictionary<int, string>();

        private bool isDisposed = false;

        private bool isStarted = false;

        /// <summary>
        /// GamePadManager クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal GamePadManager()
        {
            this.gamePadBroker = new GamePadBroker(this.gamePadHolder);

            for (int i = 0; i < GamePadCountMax / 2; ++i)
            {
                this.samplers.Add(new GamePadSampler(i, this.gamePadHolder));
            }

            this.Connected += this.OnConnected;

            this.Disconnected += this.OnDisconnected;
        }

        ~GamePadManager()
        {
            this.Dispose(false);
        }

        /// <summary>
        /// ゲームパッドが接続されると発生します。
        /// </summary>
        internal event EventHandler<GamePadEventArgs> Connected
        {
            add
            {
                lock (this.syncObject)
                {
                    foreach (GamePadSampler sampler in this.samplers)
                    {
                        sampler.Connected += value;
                    }
                }
            }

            remove
            {
                lock (this.syncObject)
                {
                    foreach (GamePadSampler sampler in this.samplers)
                    {
                        sampler.Connected -= value;
                    }
                }
            }
        }

        /// <summary>
        /// ゲームパッドが切断されると発生します。
        /// </summary>
        internal event EventHandler<GamePadEventArgs> Disconnected
        {
            add
            {
                lock (this.syncObject)
                {
                    foreach (GamePadSampler sampler in this.samplers)
                    {
                        sampler.Disconnected += value;
                    }
                }
            }

            remove
            {
                lock (this.syncObject)
                {
                    foreach (GamePadSampler sampler in this.samplers)
                    {
                        sampler.Disconnected -= value;
                    }
                }
            }
        }

        /// <summary>
        /// ゲームパッドのボタンが押されると発生します。
        /// </summary>
        internal event EventHandler<GamePadEventArgs> ButtonDown
        {
            add
            {
                lock (this.syncObject)
                {
                    foreach (GamePadSampler sampler in this.samplers)
                    {
                        sampler.ButtonDown += value;
                    }
                }
            }

            remove
            {
                lock (this.syncObject)
                {
                    foreach (GamePadSampler sampler in this.samplers)
                    {
                        sampler.ButtonDown -= value;
                    }
                }
            }
        }

        /// <summary>
        /// ゲームパッドのボタンが離されると発生します。
        /// </summary>
        internal event EventHandler<GamePadEventArgs> ButtonUp
        {
            add
            {
                lock (this.syncObject)
                {
                    foreach (GamePadSampler sampler in this.samplers)
                    {
                        sampler.ButtonUp += value;
                    }
                }
            }

            remove
            {
                lock (this.syncObject)
                {
                    foreach (GamePadSampler sampler in this.samplers)
                    {
                        sampler.ButtonUp -= value;
                    }
                }
            }
        }

        /// <summary>
        /// ゲームパッドのスティックが動かされると発生します。
        /// </summary>
        internal event EventHandler<GamePadEventArgs> StickMoved
        {
            add
            {
                lock (this.syncObject)
                {
                    foreach (GamePadSampler sampler in this.samplers)
                    {
                        sampler.StickMoved += value;
                    }
                }
            }

            remove
            {
                lock (this.syncObject)
                {
                    foreach (GamePadSampler sampler in this.samplers)
                    {
                        sampler.StickMoved -= value;
                    }
                }
            }
        }

        [Flags]
        private enum GamePadAbilityFlags : uint
        {
            None = 0,
            IsXinput = 1 << 0,
            IsJoystick = 1 << 1,
            HasPov = 1 << 4,
            HasZ = 1 << 8,
            HasR = 1 << 9,
            HasU = 1 << 10,
            HasV = 1 << 11,
        }

        /// <summary>
        /// アンマネージドリソースを開放します。
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);

            GC.SuppressFinalize(this);
        }

        /// <summary>
        /// ゲームパッドの管理を開始します。
        /// </summary>
        internal void Start()
        {
            this.gamePadBroker.Start();

            foreach (GamePadSampler sampler in this.samplers)
            {
                sampler.Start();
            }

            this.isStarted = true;
        }

        /// <summary>
        /// ゲームパッドの管理を停止します。
        /// </summary>
        internal void Stop()
        {
            this.isStarted = false;

            foreach (GamePadSampler sampler in this.samplers)
            {
                sampler.Stop();
            }

            this.gamePadBroker.Stop();
        }

        /// <summary>
        /// ゲームパッドの情報を返します。
        /// </summary>
        internal List<GamePadInfo> GetGamePadInfos()
        {
            var infos = new List<GamePadInfo>();

            lock (this.syncObject)
            {
                foreach (var pair in this.gamePadNameMap)
                {
                    infos.Add(new GamePadInfo(pair.Key, pair.Value));
                }
            }

            infos.Sort((lhs, rhs) => lhs.Name.CompareTo(rhs.Name));

            var xinputs = new List<GamePadInfo>();

            var options = new List<GamePadInfo>();

            foreach (var info in infos)
            {
                if (info.Name.StartsWith("XInput"))
                {
                    xinputs.Add(info);
                }
                else
                {
                    options.Add(info);
                }
            }

            xinputs.AddRange(options);

            return xinputs;
        }

        [DllImport("winmm.dll")]
        private static extern uint joyGetPosEx(uint uJoyID, ref JOYINFOEX pji);

        [DllImport("winmm.dll", CharSet = CharSet.Ansi)]
        private static extern uint joyGetDevCaps(
            uint uJoyID, ref JOYCAPS pjc, uint cbjc);

        [DllImport("XInput9_1_0.dll")]
        private static extern uint XInputGetState(
            uint dwUserIndex, ref XINPUT_STATE pState);

        private void Dispose(bool isDisposing)
        {
            if (!this.isDisposed)
            {
                this.isDisposed = true;

                if (isDisposing)
                {
                    if (this.isStarted)
                    {
                        this.Stop();
                    }

                    this.gamePadBroker.Dispose();

                    this.gamePadHolder.Dispose();
                }
            }
        }

        private void OnConnected(object sender, GamePadEventArgs args)
        {
            lock (this.syncObject)
            {
                this.gamePadNameMap[args.SamplerId] = args.Name;
            }
        }

        private void OnDisconnected(object sender, GamePadEventArgs args)
        {
            lock (this.syncObject)
            {
                this.gamePadNameMap.Remove(args.SamplerId);
            }
        }

        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
        private struct JOYCAPS
        {
            internal ushort wMid;
            internal ushort wPid;
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
            internal string szPname;
            internal uint wXmin;
            internal uint wXmax;
            internal uint wYmin;
            internal uint wYmax;
            internal uint wZmin;
            internal uint wZmax;
            internal uint wNumButtons;
            internal uint wPeriodMin;
            internal uint wPeriodMax;
            internal uint wRmin;
            internal uint wRmax;
            internal uint wUmin;
            internal uint wUmax;
            internal uint wVmin;
            internal uint wVmax;
            internal uint wCaps;
            internal uint wMaxAxes;
            internal uint wNumAxes;
            internal uint wMaxButtons;
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
            internal string szRegKey;
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 260)]
            internal string szOEMVxD;
        }

        private struct JOYINFOEX
        {
            internal uint dwSize;
            internal uint dwFlags;
            internal uint dwXpos;
            internal uint dwYpos;
            internal uint dwZpos;
            internal uint dwRpos;
            internal uint dwUpos;
            internal uint dwVpos;
            internal uint dwButtons;
            internal uint dwButtonNumber;
            internal uint dwPOV;
            internal uint dwReserved1;
            internal uint dwReserved2;
        }

        private struct XINPUT_GAMEPAD
        {
            internal ushort wButtons;
            internal byte bLeftTrigger;
            internal byte bRightTrigger;
            internal short sThumbLX;
            internal short sThumbLY;
            internal short sThumbRX;
            internal short sThumbRY;
        }

        [StructLayout(LayoutKind.Sequential, Pack = 4)]
        private struct XINPUT_STATE
        {
            internal uint dwPacketNumber;
            internal XINPUT_GAMEPAD Gamepad;
        }

        private struct GamePadAbility
        {
            internal GamePadAbilityFlags flags;
            internal uint index;
            internal ushort vid;
            internal ushort pid;
            internal uint buttonCount;
            internal uint xMax;
            internal uint xMin;
            internal uint yMax;
            internal uint yMin;
            internal uint zMax;
            internal uint zMin;
            internal uint rMax;
            internal uint rMin;
            internal uint uMax;
            internal uint uMin;
            internal uint vMax;
            internal uint vMin;
        }

        [StructLayout(LayoutKind.Sequential, Pack = 8)]
        private struct GamePadClient
        {
            ulong clientId;
            long timeStamp;
            int userCount0;
            int userCount1;
            int userCount2;
            int userCount3;
            int userCount4;
            int userCount5;
            int userCount6;
            int userCount7;
            int userCount8;
            int userCount9;
            int userCount10;
            int userCount11;
            int userCount12;
            int userCount13;
            int userCount14;
            int userCount15;
        }

        [StructLayout(LayoutKind.Sequential, Pack = 8)]
        private struct GamePadMap
        {
            GamePadAbility ability0;
            GamePadAbility ability1;
            GamePadAbility ability2;
            GamePadAbility ability3;
            GamePadAbility ability4;
            GamePadAbility ability5;
            GamePadAbility ability6;
            GamePadAbility ability7;
            GamePadAbility ability8;
            GamePadAbility ability9;
            GamePadAbility ability10;
            GamePadAbility ability11;
            GamePadAbility ability12;
            GamePadAbility ability13;
            GamePadAbility ability14;
            GamePadAbility ability15;
            long latestClientId;
            GamePadClient client0;
            GamePadClient client1;
            GamePadClient client2;
            GamePadClient client3;
            GamePadClient client4;
            GamePadClient client5;
            GamePadClient client6;
            GamePadClient client7;
            GamePadClient client8;
            GamePadClient client9;
            GamePadClient client10;
            GamePadClient client11;
            GamePadClient client12;
            GamePadClient client13;
            GamePadClient client14;
            GamePadClient client15;
            GamePadClient client16;
            GamePadClient client17;
            GamePadClient client18;
            GamePadClient client19;
            GamePadClient client20;
            GamePadClient client21;
            GamePadClient client22;
            GamePadClient client23;
            GamePadClient client24;
            GamePadClient client25;
            GamePadClient client26;
            GamePadClient client27;
            GamePadClient client28;
            GamePadClient client29;
            GamePadClient client30;
            GamePadClient client31;
            GamePadClient client32;
            GamePadClient client33;
            GamePadClient client34;
            GamePadClient client35;
            GamePadClient client36;
            GamePadClient client37;
            GamePadClient client38;
            GamePadClient client39;
            GamePadClient client40;
            GamePadClient client41;
            GamePadClient client42;
            GamePadClient client43;
            GamePadClient client44;
            GamePadClient client45;
            GamePadClient client46;
            GamePadClient client47;
            GamePadClient client48;
            GamePadClient client49;
            GamePadClient client50;
            GamePadClient client51;
            GamePadClient client52;
            GamePadClient client53;
            GamePadClient client54;
            GamePadClient client55;
            GamePadClient client56;
            GamePadClient client57;
            GamePadClient client58;
            GamePadClient client59;
            GamePadClient client60;
            GamePadClient client61;
            GamePadClient client62;
            GamePadClient client63;
        }

        private sealed class GamePadHolder : IDisposable
        {
            private readonly Mmap mmap;

            private bool isDisposed = false;

            internal GamePadHolder()
            {
                this.mmap = new Mmap(
                    MmapName, Marshal.SizeOf(typeof(GamePadMap)));
            }

            ~GamePadHolder()
            {
                this.Dispose(false);
            }

            public void Dispose()
            {
                this.Dispose(true);

                GC.SuppressFinalize(this);
            }

            internal GamePadAbility GetGamePadAbility(int id)
            {
                GamePadAbility[] abilities = null;

                this.mmap.Invoke((
                    UnmanagedMemoryAccessor accessor, bool exists) =>
                {
                    abilities = GetGamePadAbilities(accessor, exists);
                });

                return abilities[id];
            }

            internal void UpdateGamePadAbilities(
                IEnumerable<GamePadAbility> newies)
            {
                this.mmap.Invoke((
                    UnmanagedMemoryAccessor accessor, bool exists) =>
                {
                    GamePadAbility[] oldies = GetGamePadAbilities(
                           accessor, exists);

                    RemoveDetachedGamePadAbilities(oldies, newies);

                    AppendAttachedGamePadAbilities(oldies, newies);

                    SetGamePadAbilities(accessor, oldies);
                });
            }

            private static GamePadAbility[] GetGamePadAbilities(
                UnmanagedMemoryAccessor accessor, bool exists)
            {
                var abilities = new GamePadAbility[GamePadCountMax];

                if (exists)
                {
                    accessor.ReadArray(0, abilities, 0, abilities.Length);
                }
                else
                {
                    var gamePadMap = new GamePadMap();

                    accessor.Write(0, ref gamePadMap);
                }

                return abilities;
            }

            private static void SetGamePadAbilities(
                UnmanagedMemoryAccessor accessor, GamePadAbility[] entries)
            {
                accessor.WriteArray(0, entries, 0, entries.Length);
            }

            private static void RemoveDetachedGamePadAbilities(
                GamePadAbility[] oldies, IEnumerable<GamePadAbility> newies)
            {
                for (int i = 0; i < oldies.Length; ++i)
                {
                    GamePadAbility oldie = oldies[i];

                    if (oldie.flags.HasFlag(GamePadAbilityFlags.IsXinput))
                    {
                        if (!newies.Any(
                                x => x.flags.HasFlag(
                                         GamePadAbilityFlags.IsXinput) &&
                                     x.index == oldie.index))
                        {
                            oldie = new GamePadAbility();
                        }
                    }
                    else if (
                        oldie.flags.HasFlag(GamePadAbilityFlags.IsJoystick))
                    {
                        if (!newies.Any(
                                x => x.flags.HasFlag(
                                         GamePadAbilityFlags.IsJoystick) &&
                                     x.index == oldie.index))
                        {
                            oldie = new GamePadAbility();
                        }
                    }

                    oldies[i] = oldie;
                }
            }

            private static void AppendAttachedGamePadAbilities(
                GamePadAbility[] oldies, IEnumerable<GamePadAbility> newies)
            {
                foreach (GamePadAbility newie in newies)
                {
                    int position = -1;

                    var exists = false;

                    for (int i = 0; i < oldies.Length; ++i)
                    {
                        GamePadAbility oldie = oldies[i];

                        if (position == -1 &&
                            oldie.flags == GamePadAbilityFlags.None)
                        {
                            position = i;
                        }
                        else if (newie.index == oldie.index)
                        {
                            if ((newie.flags.HasFlag(
                                     GamePadAbilityFlags.IsXinput) &&
                                 oldie.flags.HasFlag(
                                     GamePadAbilityFlags.IsXinput)) ||
                                (newie.flags.HasFlag(
                                     GamePadAbilityFlags.IsJoystick) &&
                                 oldie.flags.HasFlag(
                                     GamePadAbilityFlags.IsJoystick)))
                            {
                                position = i;

                                exists = true;

                                break;
                            }
                        }
                    }

                    if (!exists &&
                        newie.flags.HasFlag(GamePadAbilityFlags.IsXinput) &&
                        newie.index < oldies.Length &&
                        oldies[newie.index].flags == GamePadAbilityFlags.None)
                    {
                        position = (int)newie.index;
                    }

                    if (0 <= position)
                    {
                        oldies[position] = newie;
                    }
                }
            }

            private void Dispose(bool isDisposing)
            {
                if (!this.isDisposed)
                {
                    this.isDisposed = true;

                    if (isDisposing)
                    {
                        this.mmap.Dispose();
                    }
                }
            }
        }

        private sealed class GamePadBroker : IDisposable
        {
            private static readonly TimeSpan Interval =
                TimeSpan.FromMilliseconds(25);

            private static readonly Regex XInputPattern = new Regex(
                "VID_([0-9A-F]{4})&PID_([0-9A-F]{4})", RegexOptions.Compiled);

            private readonly ManagementClass pnpEntity =
                new ManagementClass("Win32_PnPEntity");

            private readonly GamePadHolder gamePadHolder;

            private bool isDisposed = false;

            private uint focusedId = 0;

            private readonly List<GamePadAbility> abilities =
                new List<GamePadAbility>();

            private bool isStarted = false;

            private CancellationTokenSource tokenSource = null;

            private Task task = null;

            internal GamePadBroker(GamePadHolder gamePadHolder)
            {
                this.gamePadHolder = gamePadHolder;
            }

            ~GamePadBroker()
            {
                this.Dispose(false);
            }

            private enum GamePadId : uint
            {
                XInput1,
                XInput2,
                XInput3,
                XInput4,
                Joystick1,
                Joystick2,
                Joystick3,
                Joystick4,
                Joystick5,
                Joystick6,
                Joystick7,
                Joystick8,
                Joystick9,
                Joystick10,
                Joystick11,
                Joystick12,
                Joystick13,
                Joystick14,
                Joystick15,
                Joystick16,
            };

            public void Dispose()
            {
                this.Dispose(true);

                GC.SuppressFinalize(this);
            }

            internal void Start()
            {
                this.focusedId = 0;

                this.tokenSource = new CancellationTokenSource();

                CancellationToken token = this.tokenSource.Token;

                this.task = Task.Run(() => this.Sample(token));

                this.isStarted = true;
            }

            internal void Stop()
            {
                this.isStarted = false;

                this.tokenSource.Cancel();

                this.task.Wait();

                this.task.Dispose();
                this.task = null;

                this.tokenSource.Dispose();
                this.tokenSource = null;
            }

            private static void ProcessXInput(
                List<GamePadAbility> abilities, uint index)
            {
                var state = new XINPUT_STATE();

                if (XInputGetState(index, ref state) == 0)
                {
                    var ability = new GamePadAbility();
                    ability.flags |= GamePadAbilityFlags.IsXinput;
                    ability.index = index;

                    abilities.Add(ability);
                }
            }

            private static void ProcessJoystick(
                List<GamePadAbility> abilities, uint index)
            {
                var state = new JOYINFOEX();
                state.dwSize = (uint)Marshal.SizeOf(state);
                state.dwFlags = JOY_RETURNALL | JOY_USEDEADZONE;

                if (joyGetPosEx(index, ref state) == JOYERR_NOERROR)
                {
                    var caps = new JOYCAPS();

                    if (joyGetDevCaps(
                            index, ref caps, (uint)Marshal.SizeOf(caps)) ==
                        JOYERR_NOERROR)
                    {
                        var ability = new GamePadAbility();
                        ability.flags |= GamePadAbilityFlags.IsJoystick;
                        ability.index = index;
                        ability.vid = caps.wMid;
                        ability.pid = caps.wPid;
                        ability.buttonCount = caps.wNumButtons;
                        ability.xMax = caps.wXmax;
                        ability.xMin = caps.wXmin;
                        ability.yMax = caps.wYmax;
                        ability.yMin = caps.wYmin;
                        ability.zMax = caps.wZmax;
                        ability.zMin = caps.wZmin;
                        ability.rMax = caps.wRmax;
                        ability.rMin = caps.wRmin;
                        ability.uMax = caps.wUmax;
                        ability.uMin = caps.wUmin;
                        ability.vMax = caps.wVmax;
                        ability.vMin = caps.wVmin;

                        if ((caps.wCaps & JOYCAPS_HASZ) != 0)
                        {
                            ability.flags |= GamePadAbilityFlags.HasZ;
                        }

                        if ((caps.wCaps & JOYCAPS_HASR) != 0)
                        {
                            ability.flags |= GamePadAbilityFlags.HasR;
                        }

                        if ((caps.wCaps & JOYCAPS_HASU) != 0)
                        {
                            ability.flags |= GamePadAbilityFlags.HasU;
                        }

                        if ((caps.wCaps & JOYCAPS_HASV) != 0)
                        {
                            ability.flags |= GamePadAbilityFlags.HasV;
                        }

                        if ((caps.wCaps & JOYCAPS_HASPOV) != 0)
                        {
                            ability.flags |= GamePadAbilityFlags.HasPov;
                        }

                        abilities.Add(ability);
                    }
                }
            }

            private static void FilterNintendoDevice(
                List<GamePadAbility> abilities)
            {
                uint NintendoManufactureId = 0x057e;

                abilities.RemoveAll(x => x.vid == NintendoManufactureId);
            }

            private static void FilterXInputPad(
                List<GamePadAbility> abilities, ManagementClass pnpEntity)
            {
                ManagementObjectCollection objs = pnpEntity.GetInstances();

                foreach (ManagementObject obj in objs)
                {
                    object prop = obj.GetPropertyValue("DeviceID");

                    if (prop == null)
                    {
                        continue;
                    }

                    var deviceId = prop.ToString();

                    if (!deviceId.Contains("IG_"))
                    {
                        continue;
                    }

                    Match match = XInputPattern.Match(deviceId);

                    if (!match.Success)
                    {
                        continue;
                    }

                    var vid = ushort.Parse(
                        match.Groups[1].Value, NumberStyles.HexNumber);

                    var pid = ushort.Parse(
                        match.Groups[2].Value, NumberStyles.HexNumber);

                    abilities.RemoveAll(x => x.vid == vid && x.pid == pid);
                }
            }

            private void Dispose(bool isDisposing)
            {
                if (!this.isDisposed)
                {
                    this.isDisposed = true;

                    if (isDisposing)
                    {
                        if (this.isStarted)
                        {
                            this.Stop();
                        }

                        this.pnpEntity.Dispose();
                    }
                }
            }

            private void Sample(CancellationToken token)
            {
                var stopwatch = new Stopwatch();

                while (!token.IsCancellationRequested)
                {
                    stopwatch.Start();

                    if (this.focusedId == 0)
                    {
                        this.abilities.Clear();
                    }

                    switch ((GamePadId)this.focusedId)
                    {
                        case GamePadId.XInput1:
                        case GamePadId.XInput2:
                        case GamePadId.XInput3:
                        case GamePadId.XInput4:
                            ProcessXInput(
                                this.abilities,
                                this.focusedId - (uint)GamePadId.XInput1);
                            this.focusedId += 1;
                            break;

                        case GamePadId.Joystick1:
                        case GamePadId.Joystick2:
                        case GamePadId.Joystick3:
                        case GamePadId.Joystick4:
                        case GamePadId.Joystick5:
                        case GamePadId.Joystick6:
                        case GamePadId.Joystick7:
                        case GamePadId.Joystick8:
                        case GamePadId.Joystick9:
                        case GamePadId.Joystick10:
                        case GamePadId.Joystick11:
                        case GamePadId.Joystick12:
                        case GamePadId.Joystick13:
                        case GamePadId.Joystick14:
                        case GamePadId.Joystick15:
                        case GamePadId.Joystick16:
                            ProcessJoystick(
                                this.abilities,
                                this.focusedId - (uint)GamePadId.Joystick1);
                            this.focusedId += 1;
                            break;

                        default:
                            FilterNintendoDevice(this.abilities);
                            FilterXInputPad(this.abilities, this.pnpEntity);
                            this.gamePadHolder
                                .UpdateGamePadAbilities(this.abilities);
                            this.focusedId = 0;
                            break;
                    }

                    stopwatch.Stop();

                    if (stopwatch.Elapsed < Interval)
                    {
                        token.WaitHandle.WaitOne(Interval - stopwatch.Elapsed);
                    }

                    stopwatch.Reset();
                }
            }
        }

        private sealed class GamePadSampler
        {
            private static readonly TimeSpan Interval =
                TimeSpan.FromMilliseconds(8);

            private readonly int samplerId;

            private readonly GamePadHolder gamePadHolder;

            private bool isConnected = false;

            private List<GamePadButtons> buttons = new List<GamePadButtons>();

            private Point stickL = Point.Empty;

            private Point stickR = Point.Empty;

            private CancellationTokenSource tokenSource = null;

            private Task task = null;

            internal GamePadSampler(int samplerId, GamePadHolder gamePadHolder)
            {
                this.samplerId = samplerId;

                this.gamePadHolder = gamePadHolder;
            }

            internal event EventHandler<GamePadEventArgs>
                Connected = delegate { };

            internal event EventHandler<GamePadEventArgs>
                Disconnected = delegate { };

            internal event EventHandler<GamePadEventArgs>
                ButtonDown = delegate { };

            internal event EventHandler<GamePadEventArgs>
                ButtonUp = delegate { };

            internal event EventHandler<GamePadEventArgs>
                StickMoved = delegate { };

            internal void Start()
            {
                this.tokenSource = new CancellationTokenSource();

                CancellationToken token = this.tokenSource.Token;

                this.task = Task.Run(() => this.Sample(token));
            }

            internal void Stop()
            {
                this.tokenSource.Cancel();

                this.task.Wait();

                this.task.Dispose();
                this.task = null;

                this.tokenSource.Dispose();
                this.tokenSource = null;
            }

            private static List<GamePadButtons> ListXInputButtons(
                ushort buttons)
            {
                var list = new List<GamePadButtons>();

                if ((buttons & 0x0001) != 0)
                {
                    list.Add(GamePadButtons.Up);
                }

                if ((buttons & 0x0002) != 0)
                {
                    list.Add(GamePadButtons.Down);
                }

                if ((buttons & 0x0004) != 0)
                {
                    list.Add(GamePadButtons.Left);
                }

                if ((buttons & 0x0008) != 0)
                {
                    list.Add(GamePadButtons.Right);
                }

                if ((buttons & 0x0010) != 0)
                {
                    list.Add(GamePadButtons.Plus);
                }

                if ((buttons & 0x0020) != 0)
                {
                    list.Add(GamePadButtons.Minus);
                }

                if ((buttons & 0x0040) != 0)
                {
                    list.Add(GamePadButtons.StickL);
                }

                if ((buttons & 0x0080) != 0)
                {
                    list.Add(GamePadButtons.StickR);
                }

                if ((buttons & 0x0100) != 0)
                {
                    list.Add(GamePadButtons.L);
                }

                if ((buttons & 0x0200) != 0)
                {
                    list.Add(GamePadButtons.R);
                }

                if ((buttons & 0x1000) != 0)
                {
                    list.Add(GamePadButtons.B);
                }

                if ((buttons & 0x2000) != 0)
                {
                    list.Add(GamePadButtons.A);
                }

                if ((buttons & 0x4000) != 0)
                {
                    list.Add(GamePadButtons.Y);
                }

                if ((buttons & 0x8000) != 0)
                {
                    list.Add(GamePadButtons.X);
                }

                return list;
            }

            private static List<GamePadButtons> ListJoystickButtons(
                uint buttons)
            {
                var list = new List<GamePadButtons>();

                if ((buttons & 0x00000001) != 0)
                {
                    list.Add(GamePadButtons.Y);
                }

                if ((buttons & 0x00000002) != 0)
                {
                    list.Add(GamePadButtons.B);
                }

                if ((buttons & 0x00000004) != 0)
                {
                    list.Add(GamePadButtons.A);
                }

                if ((buttons & 0x00000008) != 0)
                {
                    list.Add(GamePadButtons.X);
                }

                if ((buttons & 0x00000010) != 0)
                {
                    list.Add(GamePadButtons.L);
                }

                if ((buttons & 0x00000020) != 0)
                {
                    list.Add(GamePadButtons.R);
                }

                if ((buttons & 0x00000040) != 0)
                {
                    list.Add(GamePadButtons.ZL);
                }

                if ((buttons & 0x00000080) != 0)
                {
                    list.Add(GamePadButtons.ZR);
                }

                if ((buttons & 0x00000100) != 0)
                {
                    list.Add(GamePadButtons.Minus);
                }

                if ((buttons & 0x00000200) != 0)
                {
                    list.Add(GamePadButtons.Plus);
                }

                if ((buttons & 0x00000400) != 0)
                {
                    list.Add(GamePadButtons.StickL);
                }

                if ((buttons & 0x00000800) != 0)
                {
                    list.Add(GamePadButtons.StickR);
                }

                return list;
            }

            private static Point ClampGamePadStick(
                int x, int y, int min, int max)
            {
                bool isXPositive = (x >= 0);
                bool isYPositive = (y >= 0);

                x = isXPositive ? x : -x;
                y = isYPositive ? y : -y;

                x = Math.Min(x, max);
                y = Math.Min(y, max);

                x = (x <= min) ? 0 : x - min;
                y = (y <= min) ? 0 : y - min;

                if (x == 0 && y == 0)
                {
                    return Point.Empty;
                }

                var r = (double)(max - min);

                var d2 = (double)x * x + (double)y * y;

                if (d2 <= r * r)
                {
                    x = (int)(GamePadStickMaxValue * x / r);
                    y = (int)(GamePadStickMaxValue * y / r);
                }
                else
                {
                    var d = Math.Sqrt(d2);
                    x = (int)(GamePadStickMaxValue * x / d);
                    y = (int)(GamePadStickMaxValue * y / d);
                }

                x = isXPositive ? x : -x;
                y = isYPositive ? y : -y;

                return new Point(x, y);
            }

            private static Point GetXInputStickL(short x, short y)
            {
                return ClampGamePadStick(
                    x, y,
                    XINPUT_GAMEPAD_LEFT_THUMB_DEADZONE, GamePadStickMaxValue);
            }

            private static Point GetXInputStickR(short x, short y)
            {
                return ClampGamePadStick(
                    x, y,
                    XINPUT_GAMEPAD_RIGHT_THUMB_DEADZONE, GamePadStickMaxValue);
            }

            private static uint GetJoystickStickMaxValue(uint min, uint max)
            {
                var range = max - min + 1;

                return range / 2 - ((range % 2 == 0 && range > 1) ? 1u : 0);
            }

            private static Point GetJoystickStickL(
                uint x, uint xMin, uint xMax,
                uint y, uint yMin, uint yMax)
            {
                xMax = GetJoystickStickMaxValue(xMin, xMax);
                yMax = GetJoystickStickMaxValue(yMin, yMax);

                var max = (int)Math.Min(xMax, yMax);

                return ClampGamePadStick(
                    (int)((long)x - (xMax + xMin)),
                    (int)((long)(yMax + yMin) - y),
                    XINPUT_GAMEPAD_LEFT_THUMB_DEADZONE *
                    max / GamePadStickMaxValue,
                    max);
            }

            private static Point GetJoystickStickR(
                uint z, uint zMin, uint zMax,
                uint r, uint rMin, uint rMax)
            {
                zMax = GetJoystickStickMaxValue(zMin, zMax);
                rMax = GetJoystickStickMaxValue(rMin, rMax);

                var max = (int)Math.Min(zMax, zMax);

                return ClampGamePadStick(
                    (int)((long)z - (zMax + zMin)),
                    (int)((long)(rMax + rMin) - r),
                    XINPUT_GAMEPAD_RIGHT_THUMB_DEADZONE *
                    max / GamePadStickMaxValue,
                    max);
            }

            private void Sample(CancellationToken token)
            {
                this.isConnected = false;

                this.buttons.Clear();

                this.stickL = Point.Empty;
                this.stickR = Point.Empty;

                var isFirst = true;

                var stopwatch = new Stopwatch();

                while (!token.IsCancellationRequested)
                {
                    stopwatch.Start();

                    GamePadAbility ability =
                        this.gamePadHolder.GetGamePadAbility(this.samplerId);

                    if (ability.flags.HasFlag(GamePadAbilityFlags.IsXinput))
                    {
                        this.ProcessXInput(isFirst, ability);
                    }
                    else if (
                        ability.flags.HasFlag(GamePadAbilityFlags.IsJoystick))
                    {
                        this.ProcessJoystick(isFirst, ability);
                    }
                    else
                    {
                        this.ProcessDisconnection(isFirst);
                    }

                    isFirst = false;

                    stopwatch.Stop();

                    if (stopwatch.Elapsed < Interval)
                    {
                        token.WaitHandle.WaitOne(Interval - stopwatch.Elapsed);
                    }

                    stopwatch.Reset();
                }
            }

            private void ProcessDisconnection(bool isFirst)
            {
                if (isFirst || this.isConnected)
                {
                    var args = new GamePadEventArgs(this.samplerId);

                    this.Disconnected(this, args);

                    this.isConnected = false;

                    this.buttons.Clear();

                    this.stickL = Point.Empty;
                    this.stickR = Point.Empty;
                }
            }

            private void ProcessXInput(bool isFirst, GamePadAbility ability)
            {
                var state = new XINPUT_STATE();

                if (XInputGetState(ability.index, ref state) != 0)
                {
                    this.ProcessDisconnection(isFirst);
                }
                else
                {
                    var buttons = ListXInputButtons(state.Gamepad.wButtons);

                    var stickL = GetXInputStickL(
                        state.Gamepad.sThumbLX, state.Gamepad.sThumbLY);

                    var stickR = GetXInputStickR(
                        state.Gamepad.sThumbRX, state.Gamepad.sThumbRY);

                    if (this.buttons.Contains(GamePadButtons.ZL))
                    {
                        if (state.Gamepad.bLeftTrigger >= 25)
                        {
                            buttons.Add(GamePadButtons.ZL);
                        }
                    }
                    else
                    {
                        if (state.Gamepad.bLeftTrigger >= 30)
                        {
                            buttons.Add(GamePadButtons.ZL);
                        }
                    }

                    if (this.buttons.Contains(GamePadButtons.ZR))
                    {
                        if (state.Gamepad.bRightTrigger >= 25)
                        {
                            buttons.Add(GamePadButtons.ZR);
                        }
                    }
                    else
                    {
                        if (state.Gamepad.bRightTrigger >= 30)
                        {
                            buttons.Add(GamePadButtons.ZR);
                        }
                    }

                    if (!this.isConnected)
                    {
                        this.isConnected = true;

                        this.Connected(this, new GamePadEventArgs(
                            this.samplerId, string.Format(
                                "XInput Controller #{0}", ability.index + 1)));
                    }
                    else
                    {
                        foreach (var button in buttons.Except(this.buttons))
                        {
                            this.ButtonDown(this, new GamePadEventArgs(
                                this.samplerId, button));
                        }

                        foreach (var button in this.buttons.Except(buttons))
                        {
                            this.ButtonUp(this, new GamePadEventArgs(
                                this.samplerId, button));
                        }

                        if (stickL != this.stickL)
                        {
                            this.StickMoved(this, new GamePadEventArgs(
                                this.samplerId,
                                GamePadStickType.Left, stickL));
                        }

                        if (stickR != this.stickR)
                        {
                            this.StickMoved(this, new GamePadEventArgs(
                                this.samplerId,
                                GamePadStickType.Right, stickR));
                        }
                    }

                    this.buttons = buttons;

                    this.stickL = stickL;
                    this.stickR = stickR;
                }
            }

            private void ProcessJoystick(bool isFirst, GamePadAbility ability)
            {
                var state = new JOYINFOEX();
                state.dwSize = (uint)Marshal.SizeOf(state);
                state.dwFlags = JOY_RETURNALL | JOY_USEDEADZONE;

                if (joyGetPosEx(ability.index, ref state) != JOYERR_NOERROR)
                {
                    this.ProcessDisconnection(isFirst);
                }
                else
                {
                    var buttons = ListJoystickButtons(state.dwButtons);

                    var stickL = GetJoystickStickL(
                        state.dwXpos, ability.xMin, ability.xMax,
                        state.dwYpos, ability.yMin, ability.yMax);

                    var stickR = Point.Empty;

                    if (ability.flags.HasFlag(GamePadAbilityFlags.HasZ) &&
                        ability.flags.HasFlag(GamePadAbilityFlags.HasR))
                    {
                        stickR = GetJoystickStickR(
                            state.dwZpos, ability.zMin, ability.zMax,
                            state.dwRpos, ability.rMin, ability.rMax);
                    }

                    if (ability.flags.HasFlag(GamePadAbilityFlags.HasPov) &&
                        state.dwPOV != JOY_POVCENTERED)
                    {
                        if (JOY_POVBACKWARD < state.dwPOV)
                        {
                            buttons.Add(GamePadButtons.Left);
                        }

                        if (state.dwPOV < JOY_POVRIGHT ||
                            state.dwPOV > JOY_POVLEFT)
                        {
                            buttons.Add(GamePadButtons.Up);
                        }

                        if (state.dwPOV > JOY_POVFORWARD &&
                            state.dwPOV < JOY_POVBACKWARD)
                        {
                            buttons.Add(GamePadButtons.Right);
                        }

                        if (state.dwPOV > JOY_POVRIGHT &&
                            state.dwPOV < JOY_POVLEFT)
                        {
                            buttons.Add(GamePadButtons.Down);
                        }
                    }

                    if (!this.isConnected)
                    {
                        this.isConnected = true;

                        var index = ability.index + 1;

                        this.Connected(this, new GamePadEventArgs(
                            this.samplerId, string.Format(
                                "Generic Controller #{0}", index)));
                    }
                    else
                    {
                        foreach (var button in buttons.Except(this.buttons))
                        {
                            this.ButtonDown(this, new GamePadEventArgs(
                                this.samplerId, button));
                        }

                        foreach (var button in this.buttons.Except(buttons))
                        {
                            this.ButtonUp(this, new GamePadEventArgs(
                                this.samplerId, button));
                        }

                        if (stickL != this.stickL)
                        {
                            this.StickMoved(this, new GamePadEventArgs(
                                this.samplerId,
                                GamePadStickType.Left, stickL));
                        }

                        if (stickR != this.stickR)
                        {
                            this.StickMoved(this, new GamePadEventArgs(
                                this.samplerId,
                                GamePadStickType.Right, stickR));
                        }
                    }

                    this.buttons = buttons;

                    this.stickL = stickL;
                    this.stickR = stickR;
                }
            }
        }
    }
}
