﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace InputDirector.Forms
{
    using System.Collections.Generic;
    using System.Drawing;
    using System.Linq;
    using System.Windows.Forms;

    /// <summary>
    /// 設定フォームです。
    /// </summary>
    internal sealed partial class SettingsForm : Form
    {
        private readonly InputDirector inputDirector;

        private bool waitsHotKey = false;

        private Keys hotKey;

        private Keys hotKeyModifier;

        private readonly Dictionary<string, GamePadConfig>
            gamePadConfigOldMap = new Dictionary<string, GamePadConfig>();

        private readonly Dictionary<string, GamePadConfig>
            gamePadConfigNewMap = new Dictionary<string, GamePadConfig>();

        /// <summary>
        /// SettingsForm クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="inputDirector">InputDirector です。</param>
        internal SettingsForm(InputDirector inputDirector)
        {
            InitializeComponent();

            this.inputDirector = inputDirector;

            this.LoadBorderCrossingSettings();
            this.LoadHotKeySettings();
            this.LoadConnectionSettings();
            this.LoadGamePadSettings();
        }

        private void UpdateBorderCrossingForm()
        {
            this.radioButton1.Enabled = this.checkBox1.Checked;
            this.radioButton2.Enabled = this.checkBox1.Checked;
            this.radioButton3.Enabled = this.checkBox1.Checked;
            this.radioButton4.Enabled = this.checkBox1.Checked;

            this.groupBox4.Enabled = this.checkBox1.Checked;

            if (this.radioButton1.Checked)
            {
                this.pictureBoxPC.Location = new Point(26, 40);
                this.pictureBoxTV.Location = new Point(102, 40);
            }
            else if (this.radioButton2.Checked)
            {
                this.pictureBoxPC.Location = new Point(102, 40);
                this.pictureBoxTV.Location = new Point(26, 40);
            }
            else if (this.radioButton3.Checked)
            {
                this.pictureBoxPC.Location = new Point(64, 69);
                this.pictureBoxTV.Location = new Point(64, 12);
            }
            else if (this.radioButton4.Checked)
            {
                this.pictureBoxPC.Location = new Point(64, 12);
                this.pictureBoxTV.Location = new Point(64, 69);
            }
        }

        private void UpdateHotKeyForm()
        {
            this.label3.Enabled = this.checkBox2.Checked;
            this.label4.Enabled = this.checkBox2.Checked;
            this.label5.Enabled = this.checkBox2.Checked;

            this.textBox1.Enabled = this.checkBox2.Checked;
            this.textBox1.Text = ((this.hotKeyModifier == Keys.None) ?
                "" : this.hotKeyModifier.ToString() + " + ") +
                this.hotKey.ToString();
        }

        private void UpdateConnectionForm()
        {
            this.numericUpDown2.Enabled = this.checkBox4.Checked;
        }

        private void UpdateGamePadForm()
        {
            if (this.comboBox1.Items.Count == 0)
            {
                this.comboBox1.Enabled = false;

                this.checkBox5.Enabled = false;

                this.label8.Enabled = false;

                this.comboBox2.Enabled = false;

                this.label7.Enabled = false;

                this.numericUpDown3.Enabled = false;
            }
            else
            {
                var key = (string)this.comboBox1.SelectedItem;

                GamePadConfig config = this.gamePadConfigNewMap[key];

                this.checkBox5.Checked = config.IsEnabled;

                string targetName = config.TargetName;

                int index = string.IsNullOrEmpty(targetName) ?
                    -1 : this.comboBox2.Items.IndexOf(targetName);

                if (0 <= index)
                {
                    this.comboBox2.SelectedIndex = index;
                }

                this.numericUpDown3.Value = config.PlayerNumber;

                this.label8.Enabled = config.IsEnabled;

                this.comboBox2.Enabled =
                    config.IsEnabled && this.comboBox2.Items.Count > 0;

                this.label7.Enabled = config.IsEnabled;

                this.numericUpDown3.Enabled = config.IsEnabled;
            }
        }

        private void LoadBorderCrossingSettings()
        {
            BorderCrossingConfig config =
                this.inputDirector.UserConfig.GetBorderCrossingConfig();

            this.checkBox1.Checked = config.IsEnabled;

            switch (config.Border)
            {
                case BorderType.Left:
                    this.radioButton2.Checked = true;
                    break;
                case BorderType.Top:
                    this.radioButton3.Checked = true;
                    break;
                case BorderType.Bottom:
                    this.radioButton4.Checked = true;
                    break;
                default:
                    this.radioButton1.Checked = true;
                    break;
            }

            switch (config.Trigger)
            {
                case BorderCrossingTriggerType.Click:
                    this.radioButton6.Checked = true;
                    break;
                case BorderCrossingTriggerType.DoubleClick:
                    this.radioButton7.Checked = true;
                    break;
                default:
                    this.radioButton5.Checked = true;
                    break;
            }

            this.UpdateBorderCrossingForm();
        }

        private void LoadHotKeySettings()
        {
            HotKeyConfig config =
                this.inputDirector.UserConfig.GetHotKeyConfig();

            this.checkBox2.Checked = config.IsEnabled;

            this.hotKey = config.Key;

            this.hotKeyModifier = config.Modifier;

            this.UpdateHotKeyForm();
        }

        private void LoadConnectionSettings()
        {
            ConnectionConfig config =
                this.inputDirector.UserConfig.GetConnectionConfig();

            this.checkBox3.Checked = config.IsAutoConnectionEnabled;

            this.checkBox4.Checked = config.IsTimeoutEnabled;

            this.numericUpDown1.Value = (decimal)config.PollingInterval;

            this.numericUpDown2.Value = (decimal)(config.Timeout / 1000);

            this.UpdateConnectionForm();
        }

        private void LoadGamePadSettings()
        {
            IList<string> sessionNames = this.inputDirector.GetSessionNames();

            foreach (var sessionName in sessionNames)
            {
                this.comboBox2.Items.Add(sessionName);
            }

            if (this.comboBox2.Items.Count > 0)
            {
                this.comboBox2.SelectedIndex = 0;
            }

            List<GamePadInfo> infos = this.inputDirector.GetGamePadInfos();

            foreach (var info in infos)
            {
                string name = info.Name;

                this.comboBox1.Items.Add(name);

                GamePadConfig config =
                    this.inputDirector.UserConfig.GetGamePadConfig(name) ??
                    new GamePadConfig()
                    {
                        Name = name, PlayerNumber = info.SamplerId
                    };

                this.gamePadConfigOldMap[name] = config;

                this.gamePadConfigNewMap[name] = new GamePadConfig(config);
            }

            if (this.comboBox1.Items.Count > 0)
            {
                this.comboBox1.SelectedIndex = 0;
            }
            else
            {
                this.UpdateGamePadForm();
            }
        }

        private bool SaveBorderCrossingSettings()
        {
            BorderCrossingConfig config =
                this.inputDirector.UserConfig.GetBorderCrossingConfig();

            var isModified = false;

            bool isEnabled = this.checkBox1.Checked;

            if (isEnabled != config.IsEnabled)
            {
                config.IsEnabled = isEnabled;

                isModified = true;
            }

            var border = BorderType.Right;

            if (this.radioButton2.Checked)
            {
                border = BorderType.Left;
            }
            else if (this.radioButton3.Checked)
            {
                border = BorderType.Top;
            }
            else if (this.radioButton4.Checked)
            {
                border = BorderType.Bottom;
            }

            if (border != config.Border)
            {
                config.Border = border;

                isModified = true;
            }

            var trigger = BorderCrossingTriggerType.Auto;

            if (this.radioButton6.Checked)
            {
                trigger = BorderCrossingTriggerType.Click;
            }
            else if (this.radioButton7.Checked)
            {
                trigger = BorderCrossingTriggerType.DoubleClick;
            }

            if (trigger != config.Trigger)
            {
                config.Trigger = trigger;

                isModified = true;
            }

            if (isModified)
            {
                this.inputDirector.UserConfig
                    .SetBorderCrossingConfig(config);
            }

            return isModified;
        }

        private bool SaveHotKeySettings()
        {
            HotKeyConfig config =
                this.inputDirector.UserConfig.GetHotKeyConfig();

            var isModified = false;

            bool isEnabled = this.checkBox2.Checked;

            if (isEnabled != config.IsEnabled)
            {
                config.IsEnabled = isEnabled;

                isModified = true;
            }

            if (this.hotKey != config.Key)
            {
                config.Key = this.hotKey;

                isModified = true;
            }

            if (this.hotKeyModifier != config.Modifier)
            {
                config.Modifier = this.hotKeyModifier;

                isModified = true;
            }

            if (isModified)
            {
                this.inputDirector.UserConfig.SetHotKeyConfig(config);
            }

            return isModified;
        }

        private bool SaveConnectionSettings()
        {
            ConnectionConfig config =
                this.inputDirector.UserConfig.GetConnectionConfig();

            var isModified = false;

            bool isAutoConnectionEnabled = this.checkBox3.Checked;

            if (isAutoConnectionEnabled != config.IsAutoConnectionEnabled)
            {
                config.IsAutoConnectionEnabled = isAutoConnectionEnabled;

                isModified = true;
            }

            bool isTimeoutEnabled = this.checkBox4.Checked;

            if (isTimeoutEnabled != config.IsTimeoutEnabled)
            {
                config.IsTimeoutEnabled = isTimeoutEnabled;

                isModified = true;
            }

            var interval = (int)this.numericUpDown1.Value;

            if (interval != config.PollingInterval)
            {
                config.PollingInterval = interval;

                isModified = true;
            }

            var timeout = (int)this.numericUpDown2.Value * 1000;

            if (timeout != config.Timeout)
            {
                config.Timeout = timeout;

                isModified = true;
            }

            if (isModified)
            {
                this.inputDirector.UserConfig.SetConnectionConfig(config);
            }

            return isModified;
        }

        private bool SaveGamePadSettings()
        {
            var isModified = false;

            foreach (var item in this.comboBox1.Items)
            {
                var name = (string)item;

                GamePadConfig newConfig = this.gamePadConfigNewMap[name];

                GamePadConfig oldConfig = this.gamePadConfigOldMap[name];

                if (newConfig.IsEnabled    != oldConfig.IsEnabled   ||
                    newConfig.TargetName   != oldConfig.TargetName  ||
                    newConfig.PlayerNumber != oldConfig.PlayerNumber)
                {
                    this.inputDirector.UserConfig.SetGamePadConfig(newConfig);

                    isModified = true;
                }
            }

            return isModified;
        }

        private void SettingsForm_Closed(object sender, System.EventArgs e)
        {
            var isModified = false;

            if (this.SaveBorderCrossingSettings())
            {
                isModified = true;
            }

            if (this.SaveHotKeySettings())
            {
                isModified = true;
            }

            if (this.SaveConnectionSettings())
            {
                isModified = true;
            }

            if (this.SaveGamePadSettings())
            {
                isModified = true;
            }

            if (isModified)
            {
                this.inputDirector.UserConfig.Save();
            }
        }

        private void checkBox1_CheckedChanged(
            object sender, System.EventArgs e)
        {
            this.UpdateBorderCrossingForm();
        }

        private void radioButton1_CheckedChanged(
            object sender, System.EventArgs e)
        {
            this.UpdateBorderCrossingForm();
        }

        private void radioButton2_CheckedChanged(
            object sender, System.EventArgs e)
        {
            this.UpdateBorderCrossingForm();
        }

        private void radioButton3_CheckedChanged(
            object sender, System.EventArgs e)
        {
            this.UpdateBorderCrossingForm();
        }

        private void radioButton4_CheckedChanged(
            object sender, System.EventArgs e)
        {
            this.UpdateBorderCrossingForm();
        }

        private void checkBox2_CheckedChanged(
            object sender, System.EventArgs e)
        {
            this.UpdateHotKeyForm();
        }

        private void label5_Click(object sender, System.EventArgs e)
        {
            this.waitsHotKey = true;
        }

        private void groupBox1_Enter(object sender, System.EventArgs e)
        {
            this.waitsHotKey = false;
        }

        private void groupBox2_Enter(object sender, System.EventArgs e)
        {
            this.waitsHotKey = false;
        }

        private void tabControl1_Click(object sender, System.EventArgs e)
        {
            this.waitsHotKey = false;
        }

        private void SettingsForm_KeyUp(
            object sender, System.Windows.Forms.KeyEventArgs e)
        {
            if (!this.waitsHotKey)
            {
                return;
            }

            this.waitsHotKey = false;

            if (new[] { Keys.ShiftKey, Keys.ControlKey, Keys.Menu }
                    .Contains(e.KeyCode))
            {
                return;
            }

            if (e.KeyCode == Keys.F12 && e.Modifiers == Keys.None)
            {
                return;
            }

            this.hotKey = e.KeyCode;

            this.hotKeyModifier = e.Modifiers;

            this.UpdateHotKeyForm();
        }

        private void checkBox4_CheckedChanged(
            object sender, System.EventArgs e)
        {
            this.UpdateConnectionForm();
        }

        private void comboBox1_SelectedIndexChanged(
            object sender, System.EventArgs e)
        {
            this.UpdateGamePadForm();
        }

        private void numericUpDown3_ValueChanged(
            object sender, System.EventArgs e)
        {
            if (this.comboBox1.Items.Count == 0) { return; }

            var key = (string)this.comboBox1.SelectedItem;

            GamePadConfig config = this.gamePadConfigNewMap[key];

            config.PlayerNumber = (int)this.numericUpDown3.Value;
        }

        private void checkBox5_CheckedChanged(
            object sender, System.EventArgs e)
        {
            if (this.comboBox1.Items.Count == 0) { return; }

            var key = (string)this.comboBox1.SelectedItem;

            GamePadConfig config = this.gamePadConfigNewMap[key];

            config.IsEnabled = this.checkBox5.Checked;

            this.UpdateGamePadForm();
        }

        private void comboBox2_SelectedIndexChanged(
            object sender, System.EventArgs e)
        {
            if (this.comboBox1.Items.Count == 0) { return; }

            var key = (string)this.comboBox1.SelectedItem;

            GamePadConfig config = this.gamePadConfigNewMap[key];

            config.TargetName = (string)this.comboBox2.SelectedItem;
        }
    }
}
