﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace HidShell.Devices
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.Diagnostics;
    using System.Linq;
    using System.Runtime.InteropServices;
    using System.Threading;
    using System.Threading.Tasks;
    using Properties;

    /// <summary>
    /// TouchScreen を扱うクラスです。
    /// </summary>
    [PartCreationPolicy(CreationPolicy.NonShared)]
    internal sealed class TouchScreen : IDevice
    {
        private const string DeviceType = "TouchScreen";

        private const int TouchCountMax = 16;

        private const int TouchScreenCountMax = 1;

        private static readonly TimeSpan Interval =
            TimeSpan.FromMilliseconds(4);

        private readonly List<string> deviceIds = new List<string>();

        private readonly List<EventDumper> eventDumpers;

        private readonly List<EventSender> eventSenders;

        private uint port = PortAccessor.DefaultPort;

        private Logger logger = null;

        /// <summary>
        /// TouchScreen クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal TouchScreen()
        {
            this.eventDumpers = new List<EventDumper>();

            this.eventSenders = new List<EventSender>();


            foreach (int i in Enumerable.Range(0, TouchScreenCountMax))
            {
                var deviceId = i.ToString();

                this.deviceIds.Add(deviceId);

                this.eventDumpers.Add(
                    new EventDumper(deviceId, GetTouchScreenEventDumper(i)));

                this.eventSenders.AddRange(new []
                {
                    new EventSender(
                        deviceId, EventType.TouchBegan,
                        GetTouchBeganEventSender(i)),
                    new EventSender(
                        deviceId, EventType.TouchMoved,
                        GetTouchMovedEventSender(i)),
                    new EventSender(
                        deviceId, EventType.TouchEnded,
                        GetTouchEndedEventSender(i)),
                 });
            }
        }

        /// <summary>
        /// ポートを設定します。
        /// </summary>
        /// <param name="port">ポートです。</param>
        public void SetPort(uint port)
        {
            this.port = port;
        }

        /// <summary>
        /// ロガーを設定します。
        /// </summary>
        /// <param name="logger">ロガーです。</param>
        public void SetLogger(Logger logger)
        {
            this.logger = logger;
        }

        /// <summary>
        /// 指定されたデバイス種別をサポートするか否かを表す値を返します。
        /// </summary>
        /// <param name="deviceType">デバイス種別を表す文字列です。</param>
        /// <returns>指定されたデバイス種別をサポートするか否かを表す値です。</returns>
        public bool Supports(string deviceType)
        {
            return deviceType == DeviceType;
        }

        /// <summary>
        /// デバイス識別子で指定されたデバイスについてイベントをダンプします。
        /// </summary>
        /// <param name="deviceId">デバイス識別子です。</param>
        public void DumpEvent(string deviceId)
        {
            var dumpers = deviceId == null
                ? this.eventDumpers
                : this.eventDumpers.Where(x => x.SupportsDeviceId(deviceId));

            if (deviceId != null && dumpers.Count() == 0)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorDeviceIdUnexpected, deviceId));
            }

            Parallel.ForEach(dumpers, dumper => dumper.Invoke(this));
        }

        /// <summary>
        /// デバイス識別子で指定されたデバイスに対してイベントを発行します。
        /// </summary>
        /// <param name="deviceId">デバイス識別子です。</param>
        /// <param name="eventType">イベント種別です。</param>
        /// <param name="eventArgs">イベント引数です。</param>
        public void SendEvent(
            string deviceId, string eventType, string[] eventArgs)
        {
            var senders =
                this.eventSenders.Where(x => x.SupportsDeviceId(deviceId));

            if (senders.Count() == 0)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorDeviceIdUnexpected, deviceId));
            }

            var sender =
                senders.FirstOrDefault(x => x.SupportsEventType(eventType));

            if (sender == null)
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorEventTypeUnexpected, eventType));
            }

            sender.Invoke(this, eventArgs);
        }

        private static void GetTouchScreenState(
            IntPtr handle, out TouchScreenState outState, uint port)
        {
            var result = (HidShellResult)
                Native.GetHidShellTouchScreenState(
                    handle, out outState, port, PortAccessor.Out);

            if (result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }
        }

        private static void GetTouchScreenAutoPilotState(
            IntPtr handle, out TouchScreenState outState, uint port)
        {
            var result = (HidShellResult)
                Native.GetHidShellTouchScreenState(
                    handle, out outState, port, PortAccessor.In);

            switch (result)
            {
                case HidShellResult.Success:
                    return;

                case HidShellResult.StateNotSet:
                    outState = new TouchScreenState();
                    return;

                default:
                    throw new HidShellException(result);
            }
        }

        private static void SetTouchScreenAutoPilotState(
            IntPtr handle, ref TouchScreenState state, uint port)
        {
            var result = (HidShellResult)
                Native.SetHidShellTouchScreenState(
                    handle, ref state, port, PortAccessor.In);

            if (result != HidShellResult.Success)
            {
                throw new HidShellException(result);
            }
        }

        private static int ParseFingerId(string fingerId)
        {
            var value = new int();

            if (!int.TryParse(fingerId, out value))
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorFingerIdInvalid, fingerId));
            }

            return value;
        }

        private static int ParseCoordinateX(string x)
        {
            var value = new int();

            if (!int.TryParse(x, out value))
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorCoordinateInvalidX, x));
            }

            return value;
        }

        private static int ParseCoordinateY(string y)
        {
            var value = new int();

            if (!int.TryParse(y, out value))
            {
                throw new HidShellException(string.Format(
                    Resources.ErrorCoordinateInvalidY, y));
            }

            return value;
        }

        private static Action<TouchScreen> GetTouchScreenEventDumper(int index)
        {
            return (TouchScreen that) => {
                DumpTouchScreenEvent(
                    that.port, that.logger, that.deviceIds[index]);
            };
        }

        private static void DumpTouchScreenEvent(
            uint port, Logger logger, string deviceId)
        {
            var stopwatch = new Stopwatch();

            var nextState = new TouchScreenState();
            var lastState = new TouchScreenState();

            using (var accessor = new PortAccessor())
            {
                GetTouchScreenState(accessor.Handle, out lastState, port);
            }

            while (true)
            {
                stopwatch.Start();

                using (var accessor = new PortAccessor())
                {
                    GetTouchScreenState(accessor.Handle, out nextState, port);
                }

                int nextIndex = 0;
                int lastIndex = 0;

                while (true)
                {
                    if (lastIndex == lastState.Count)
                    {
                        for (int i = nextIndex; i < nextState.Count; ++i)
                        {
                            logger.WriteLine(
                                "{0,-16}{1,-16}{2} {3,3} {4,5} {5,5}",
                                DeviceType, deviceId,
                                EventType.TouchBegan,
                                nextState.Touches[i].FingerId,
                                nextState.Touches[i].X,
                                nextState.Touches[i].Y);
                        }

                        break;
                    }

                    if (nextIndex == nextState.Count)
                    {
                        for (int i = lastIndex; i < lastState.Count; ++i)
                        {
                            logger.WriteLine(
                                "{0,-16}{1,-16}{2} {3,3}",
                                DeviceType, deviceId,
                                EventType.TouchEnded,
                                lastState.Touches[i].FingerId);
                        }

                        break;
                    }

                    TouchState nextTouch = nextState.Touches[nextIndex];
                    TouchState lastTouch = lastState.Touches[lastIndex];

                    if (nextTouch.FingerId != lastTouch.FingerId)
                    {
                        logger.WriteLine(
                            "{0,-16}{1,-16}{2} {3,3}",
                            DeviceType, deviceId,
                            EventType.TouchEnded,
                            lastTouch.FingerId);

                        ++lastIndex;
                    }
                    else
                    {
                        if (nextTouch.X != lastTouch.X ||
                            nextTouch.Y != lastTouch.Y)
                        {
                            logger.WriteLine(
                                "{0,-16}{1,-16}{2} {3,3} {4,5} {5,5}",
                                DeviceType, deviceId,
                                EventType.TouchMoved,
                                nextTouch.FingerId,
                                nextTouch.X,
                                nextTouch.Y);
                        }

                        ++nextIndex;
                        ++lastIndex;
                    }
                }

                lastState = nextState;

                stopwatch.Stop();

                if (stopwatch.Elapsed < Interval)
                {
                    Thread.Sleep(Interval - stopwatch.Elapsed);
                }

                stopwatch.Reset();
            }
        }

        private static Action<TouchScreen, string[]> GetTouchBeganEventSender(
            int index)
        {
            var state = new TouchScreenState();

            return (TouchScreen that, string[] args) =>
            {
                if (args.Count() != 3)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.TouchBegan));
                }

                var touch = new TouchState()
                {
                    FingerId = ParseFingerId(args[0]),
                    X = ParseCoordinateX(args[1]),
                    Y = ParseCoordinateY(args[2]),
                };

                using (var accessor = new PortAccessor())
                {
                    GetTouchScreenAutoPilotState(
                        accessor.Handle, out state, that.port);

                    int count = state.Count;

                    if (count < TouchCountMax)
                    {
                        if (Enumerable.Range(0, count).All(i =>
                                touch.FingerId != state.Touches[i].FingerId))
                        {
                            state.Touches[count] = touch;

                            state.Count += 1;
                        }
                    }

                    SetTouchScreenAutoPilotState(
                        accessor.Handle, ref state, that.port);
                }
            };
        }

        private static Action<TouchScreen, string[]> GetTouchMovedEventSender(
            int index)
        {
            var state = new TouchScreenState();

            return (TouchScreen that, string[] args) =>
            {
                if (args.Count() != 3)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.TouchMoved));
                }

                var touch = new TouchState()
                {
                    FingerId = ParseFingerId(args[0]),
                    X = ParseCoordinateX(args[1]),
                    Y = ParseCoordinateY(args[2]),
                };

                using (var accessor = new PortAccessor())
                {
                    GetTouchScreenAutoPilotState(
                        accessor.Handle, out state, that.port);

                    for (int i = 0; i < state.Count; ++i)
                    {
                        if (touch.FingerId == state.Touches[i].FingerId)
                        {
                            state.Touches[i] = touch;
                        }
                    }

                    SetTouchScreenAutoPilotState(
                        accessor.Handle, ref state, that.port);
                }
            };
        }

        private static Action<TouchScreen, string[]> GetTouchEndedEventSender(
            int index)
        {
            var lastState = new TouchScreenState();

            return (TouchScreen that, string[] args) =>
            {
                if (args.Count() != 1)
                {
                    throw new HidShellException(string.Format(
                        Resources.ErrorEventArgCountInvalid,
                        EventType.TouchEnded));
                }

                int fingerId = ParseFingerId(args[0]);

                var nextState = new TouchScreenState();

                using (var accessor = new PortAccessor())
                {
                    GetTouchScreenAutoPilotState(
                        accessor.Handle, out lastState, that.port);

                    for (int i = 0; i < lastState.Count; ++i)
                    {
                        TouchState touch = lastState.Touches[i];

                        if (fingerId != touch.FingerId)
                        {
                            nextState.Touches[nextState.Count] = touch;

                            nextState.Count += 1;
                        }
                    }

                    SetTouchScreenAutoPilotState(
                        accessor.Handle, ref nextState, that.port);
                }
            };
        }

        [StructLayout(LayoutKind.Sequential, Pack = 8)]
        private struct TouchState
        {
            internal long DeltaTime;
            internal uint Attributes;
            internal int FingerId;
            internal int X;
            internal int Y;
            internal int DiameterX;
            internal int DiameterY;
            internal int RotationAngle;
        }

        private struct TouchesState
        {
            private TouchState touch0;
            private TouchState touch1;
            private TouchState touch2;
            private TouchState touch3;
            private TouchState touch4;
            private TouchState touch5;
            private TouchState touch6;
            private TouchState touch7;
            private TouchState touch8;
            private TouchState touch9;
            private TouchState touch10;
            private TouchState touch11;
            private TouchState touch12;
            private TouchState touch13;
            private TouchState touch14;
            private TouchState touch15;

            internal TouchState this[int i]
            {
                get
                {
                    switch (i)
                    {
                        case 0: return touch0;
                        case 1: return touch1;
                        case 2: return touch2;
                        case 3: return touch3;
                        case 4: return touch4;
                        case 5: return touch5;
                        case 6: return touch6;
                        case 7: return touch7;
                        case 8: return touch8;
                        case 9: return touch9;
                        case 10: return touch10;
                        case 11: return touch11;
                        case 12: return touch12;
                        case 13: return touch13;
                        case 14: return touch14;
                        case 15: return touch15;
                        default: throw new IndexOutOfRangeException();
                    }
                }

                set
                {
                    switch (i)
                    {
                        case 0: touch0 = value; break;
                        case 1: touch1 = value; break;
                        case 2: touch2 = value; break;
                        case 3: touch3 = value; break;
                        case 4: touch4 = value; break;
                        case 5: touch5 = value; break;
                        case 6: touch6 = value; break;
                        case 7: touch7 = value; break;
                        case 8: touch8 = value; break;
                        case 9: touch9 = value; break;
                        case 10: touch10 = value; break;
                        case 11: touch11 = value; break;
                        case 12: touch12 = value; break;
                        case 13: touch13 = value; break;
                        case 14: touch14 = value; break;
                        case 15: touch15 = value; break;
                        default: throw new IndexOutOfRangeException();
                    }
                }
            }
        }

        private struct TouchScreenState
        {
            internal int Count;
            internal TouchesState Touches;
        }

        private static class Native
        {
            [DllImport("HidShellLibrary.dll")]
            internal static extern int GetHidShellTouchScreenState(
                IntPtr handle,
                out TouchScreenState outState, uint port, uint direction);

            [DllImport("HidShellLibrary.dll")]
            internal static extern int SetHidShellTouchScreenState(
                IntPtr handle,
                ref TouchScreenState state, uint port, uint direction);
        }

        private static class EventType
        {
            internal const string TouchBegan = "TouchBegan";

            internal const string TouchMoved = "TouchMoved";

            internal const string TouchEnded = "TouchEnded";
        }

        private sealed class EventDumper
        {
            private string deviceId;

            private Action<TouchScreen> dumper;

            internal EventDumper(string deviceId, Action<TouchScreen> dumper)
            {
                this.deviceId = deviceId;

                this.dumper = dumper;
            }

            internal bool SupportsDeviceId(string deviceId)
            {
                return deviceId == this.deviceId;
            }

            internal void Invoke(TouchScreen that)
            {
                this.dumper(that);
            }
        }

        private sealed class EventSender
        {
            private string deviceId;

            private string eventType;

            private Action<TouchScreen, string[]> sender;

            internal EventSender(
                string deviceId,
                string eventType,
                Action<TouchScreen, string[]> sender)
            {
                this.deviceId = deviceId;

                this.eventType = eventType;

                this.sender = sender;
            }

            internal bool SupportsDeviceId(string deviceId)
            {
                return deviceId == this.deviceId;
            }

            internal bool SupportsEventType(string eventType)
            {
                return eventType == this.eventType;
            }

            internal void Invoke(TouchScreen that, string[] eventArgs)
            {
                this.sender(that, eventArgs);
            }
        }
    }
}
