﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE

#include <nw/types.h>

#if defined(NW_PLATFORM_WIN32)
#include <winext/cafe/env.h>
#include <winext/cafe/fs.h>
#else
#include <cafe/env.h>
#include <cafe/fs.h>
#endif
#include <nw/demo.h>
#include <nw/gfnd.h>
#include <nw/dw.h>

#if defined(NW_PLATFORM_WIN32)
#include <nw/dev/win/dev_PadWin.h>
#include <nw/dev/win/dev_WPadWin.h>
#include <nw/dev/win/dev_VPadWin.h>
#include <nw/dev/win/dev_MouseWin.h>
#include <windows.h>
#else
#include <nw/dev/cafe/dev_PadCafe.h>
#include <nw/dev/cafe/dev_WPadCafe.h>
#include <nw/dev/cafe/dev_VPadCafe.h>
#include <nw/dev/cafe/dev_MouseCafe.h>
#endif
#include <nw/lyt.h>
#include <nw/vwrlyt.h>

namespace {

const u32 SCREEN_WIDTH = 1280;
const u32 SCREEN_HEIGHT = 720;

#if defined(NW_PLATFORM_CAFE)
const s32 PAD_CHANNEL = 0;
const s32 WPAD_CHANNEL = 0;
#else
const s32 JOYPAD_NUMBER = 0;
#endif

#if defined(NW_PLATFORM_CAFE) && defined(NW_MCS_ENABLE)
#define USE_MCS_INPUT
#endif

#if defined(USE_MCS_INPUT)
const u32 MCS_INPUT_DEVICE_BUF_SIZE = 64 * 1024; //!< mcsHID のバッファサイズです。
u8* s_InputDeviceBuf = NULL;                     //!< mcsHID のバッファです。
#endif

//-----------------------------------------------------------------------------
//! @brief        メモリの初期化を実行します。
//!
//! @param[out]   heap          メインメモリのヒープです。
//-----------------------------------------------------------------------------
void
InitializeMemory( nw::ut::MemoryAllocator* heap )
{
    const u32 MEM2_SIZE         = 256 * 1024 * 1024;

  #if defined(NW_PLATFORM_WIN32)
    void* memory = malloc( MEM2_SIZE );
  #else
    void* memory = MEMAllocFromDefaultHeap( MEM2_SIZE );
  #endif

    heap->Initialize( memory, MEM2_SIZE );
}

//-----------------------------------------------------------------------------
//! @brief        デモシステムです。
//-----------------------------------------------------------------------------
class TestDemoSystem : public nw::demo::DemoSystem
{
public:
    typedef nw::ut::FixedSafeString<NW_DEV_FS_MAX_FULLPATH_SIZE> PathString;

    //! @brief コンストラクタに渡す生成パラメータです。
    struct CreateArg : public nw::demo::DemoSystem::CreateArg
    {
        //! CAFE_CONTENT_DIR環境変数が示すディレクトリのFSにおけるパスです。
        const char* nwRootPath;

#if defined(NW_PLATFORM_CAFE)
        //! レイアウトの描画に使用するシェーダのパスです。
        const char* lytShaderPath;
#endif

        CreateArg()
            : nwRootPath(NULL)
#if defined(NW_PLATFORM_CAFE)
            , lytShaderPath(NULL)
#endif
        {}
    };

    //-------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //-------------------------------------------------------------------------
    /* ctor */
    TestDemoSystem(const CreateArg& arg)
      : nw::demo::DemoSystem( arg )
      , m_Arg(arg)
    {
        NW_ASSERT(s_pInstance == NULL);
        s_pInstance = this;
    }

    //-------------------------------------------------------------------------
    //! @brief        nw::dev::FileDeviceManagerを初期化します。
    //-------------------------------------------------------------------------
    static void InitFileDeviceManager(nw::ut::IAllocator* pAllocator)
    {
        nw::dev::FileDeviceManager::CreateArg createArg;
        createArg.allocator = pAllocator;
        createArg.useMount = true;
        nw::dev::FileDeviceManager* pFileSystem = NW_INSTANCE(nw::dev::FileDeviceManager);
        pFileSystem->Initialize(createArg);
    }

    //-------------------------------------------------------------------------
    //! @brief        nw::dev::FileDeviceManagerを解放します。
    //-------------------------------------------------------------------------
    static void FinalizeFileDeviceManager()
    {
        nw::dev::FileDeviceManager* pFileSystem = NW_INSTANCE(nw::dev::FileDeviceManager);
        pFileSystem->Finalize();
    }

    //-------------------------------------------------------------------------
    //! @brief        CAFE_CONTENT_DIR環境変数の値を取得します。
    //!
    //! @param[out] nwRoot  値が格納されます。
    //!
    //! @return       適切なパスが取得できた場合には true を返します。
    //!
    //! @details
    //! 環境変数の値を FS のパスに変換して nwRoot に格納します。
    //-------------------------------------------------------------------------
    static bool GetNwRoot(nw::ut::BufferedSafeString& nwRoot)
    {
#if defined(NW_VWRLYT_ENABLE)
    #if defined(NW_PLATFORM_CAFE)
        PathString buff;

        if (0 != ENVGetEnvironmentVariable("CAFE_CONTENT_DIR", buff.getBuffer(), buff.getBufferSize()))
        {
            return false;
        }

        nw::dev::FileDeviceManager* pFileSystem = NW_INSTANCE(nw::dev::FileDeviceManager);
        return nw::vwrlyt::Viewer::ConvertHostToFsPath(&nwRoot, pFileSystem->GetFSMountPath(), buff.c_str());
    #elif defined(NW_PLATFORM_WIN32)
        char buff[MAX_PATH + 5];
        GetCurrentDirectoryA(MAX_PATH, buff);
        strcat(buff, "\\data");
        nw::dev::FileDeviceManager* pFileSystem = NW_INSTANCE(nw::dev::FileDeviceManager);
        return nw::vwrlyt::Viewer::ConvertHostToFsPath(&nwRoot, pFileSystem->GetFSMountPath(), buff);
    #endif
#else
        NW_UNUSED_VARIABLE(nwRoot);
        return false;
#endif
    }

    //-------------------------------------------------------------------------
    //! @brief        決め打ちのサンプルデータを読み込みます。
    //-------------------------------------------------------------------------
    void LoadSampleData();

    //-------------------------------------------------------------------------
    //! @brief        デモの初期化処理を行います。
    //-------------------------------------------------------------------------
    virtual void ProcInit()
    {
        nw::dev::FileDeviceManager* pFileSystem = NW_INSTANCE(nw::dev::FileDeviceManager);

#if defined( NW_PLATFORM_CAFE )
        // 入力デバイスを初期化します。
        {
            nw::dev::PadDeviceCafe::GetInstance()->Initialize();
            nw::dev::WPadDeviceCafe::GetInstance()->Initialize( this->GetAllocator() );
            nw::dev::KeyboardMouseDeviceCafe::GetInstance()->Initialize();
        }

        // パッドを初期化します。
        {
            // デバッグ用パッドにデバイスを設定します。
            m_Pad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::PadCafe ) ) ) nw::dev::PadCafe( PAD_CHANNEL );
            m_Pad->SetPadDevice( nw::dev::PadDeviceCafe::GetInstance() );
            // WPad にデバイスを設定します。
            m_WPad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::WPadCafe ) ) ) nw::dev::WPadCafe( WPAD_CHANNEL );
            m_WPad->SetWPadDevice( nw::dev::WPadDeviceCafe::GetInstance() );
            // VPad にデバイスを設定します。
            m_VPad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::VPadCafe ) ) ) nw::dev::VPadCafe();
            m_VPad->SetVPadDevice( nw::dev::VPadDeviceCafe::GetInstance() );
            // Mouse にデバイスを設定します。
            m_Mouse = new( m_Arg.allocator->Alloc( sizeof( nw::dev::MouseCafe ) ) ) nw::dev::MouseCafe();
            m_Mouse->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceCafe::GetInstance() );
            m_Mouse->SetPointerBound( nw::math::VEC2( 0.f, 0.f ), nw::math::VEC2( static_cast<f32>( m_Arg.width ), static_cast<f32>( m_Arg.height ) ) );
        }

#else
        // 入力デバイスを初期化します。
        {
            // ジョイパッドデバイスを初期化します。
            nw::dev::JoyPadDeviceWin::GetInstance()->Initialize();

            // キーボード・マウスデバイスを初期化します。
            nw::dev::KeyboardMouseDeviceWin::GetInstance()->Initialize();
            // マウス入力を取得するウィンドウのハンドルを設定します。
            nw::dev::KeyboardMouseDeviceWin::GetInstance()->SetMainWindowHandle( this->GetWindowHandle() );
        }

        // パッドを初期化します。
        {
            // デバッグ用パッドにデバイスを設定します。
            m_Pad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::PadWin ) ) ) nw::dev::PadWin( JOYPAD_NUMBER );
            m_Pad->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
            m_Pad->SetJoyPadDevice( nw::dev::JoyPadDeviceWin::GetInstance() );

            // WPad にデバイスを設定します。
            m_WPad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::WPadWin ) ) ) nw::dev::WPadWin();
            m_WPad->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
            // ポインタの座標を中心原点にします。
            m_WPad->SetPointerCenterOrigin( true );

            // VPad にデバイスを設定します。
            m_VPad = new( m_Arg.allocator->Alloc( sizeof( nw::dev::VPadWin ) ) ) nw::dev::VPadWin( JOYPAD_NUMBER );
            m_VPad->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
            m_VPad->SetJoyPadDevice( nw::dev::JoyPadDeviceWin::GetInstance() );
            // Mouse にデバイスを設定します。
            m_Mouse = new( m_Arg.allocator->Alloc( sizeof( nw::dev::MouseWin ) ) ) nw::dev::MouseWin();
            m_Mouse->SetKeyboardMouseDevice( nw::dev::KeyboardMouseDeviceWin::GetInstance() );
        }
#endif

        // Cafeではシェーダバイナリを読み込んで初期化する必要があります。
        nw::ut::MemoryRange fontShaderBinary;
        nw::ut::MemoryRange lytShaderBinary;

        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.allocator = this->GetAllocator();

        PathString path;

        // フォントシェーダの読み込み
#if defined(NW_PLATFORM_CAFE)
        {
            loadArg.path = m_Arg.fontShaderPath;
            u8* binary = pFileSystem->Load(loadArg);
            fontShaderBinary = nw::ut::MakeMemoryRange(binary, loadArg.readSize);
        }
#endif

        // フォントの読み込み
        {
            nw::dev::FileDevice::LoadArg myLoadArg = loadArg;
            myLoadArg.path = m_Arg.fontPath;
            myLoadArg.alignment = nw::font::RESOURCE_ALIGNMENT;
            u8* binary = pFileSystem->Load(myLoadArg);
            m_FontBinary = nw::ut::MakeMemoryRange(binary, myLoadArg.readSize);
        }

#if defined(NW_PLATFORM_CAFE)
        // レイアウトシェーダの読み込み
        {
            loadArg.path = m_Arg.lytShaderPath;
            u8* binary = pFileSystem->Load(loadArg);
            lytShaderBinary = nw::ut::MakeMemoryRange(binary, loadArg.readSize);
        }
#endif

        NW_INSTANCE(nw::gfnd::Graphics)->LockDrawContext();
        {
            const u32 CharMax = 512;

#if defined(NW_PLATFORM_CAFE)
            m_GraphicsResource.Setup(CharMax, fontShaderBinary);
            m_ShaderSetupHelper.Setup(&m_GraphicsResource, lytShaderBinary);
#endif

#if defined(NW_PLATFORM_WIN32)
            m_GraphicsResource.Setup(CharMax);
#endif
        }
        NW_INSTANCE(nw::gfnd::Graphics)->UnlockDrawContext();

        m_DwTextRenderer.Initialize(
            *this->GetAllocator(),
            m_FontBinary.Begin(),
            m_FontBinary.Size(),
            fontShaderBinary.Begin(),
            fontShaderBinary.Size());
        m_DwUIRenderer.SetTextRenderer(&m_DwTextRenderer);
        m_DwUIRenderer.SetPrimitiveRenderer(NW_INSTANCE(nw::dev::PrimitiveRenderer));

        pFileSystem->Unload(loadArg, fontShaderBinary.Begin());
        pFileSystem->Unload(loadArg, lytShaderBinary.Begin());

#if defined(NW_VWRLYT_ENABLE)

        m_LayoutViewer
            .SetUIRenderer(&m_DwUIRenderer)
            .SetDefaultPreviewPath(m_Arg.nwRootPath)
            .Initialize(
                &m_GraphicsResource,
                this->GetAllocator(),
                nw::lyt::Size(f32(this->GetWidth()), f32(this->GetHeight())));

        m_Inputs
            .SetMainViewportSize(this->GetWidth(), this->GetHeight())
            .SetVPadSrc(this->m_VPad)
            .SetWPadSrc(this->m_WPad)
            .SetPadSrc(this->m_Pad)
            .SetMouseSrc(this->m_Mouse);
#endif // NW_VWRLYT_ENABLE

        this->LoadSampleData();

        m_GfxContext.SetDepthEnable(false, false);
        m_GfxContext.SetCullingMode(nw::gfnd::Graphics::CULLING_MODE_NONE);

        // 負荷メーターの枠の色を変えます。
        this->GetMeterDrawer().SetBorderColor(nw::ut::Color4u8(nw::ut::Color4u8::X_DARK_GRAY));
    }

    //-------------------------------------------------------------------------
    //! @brief        デモの終了処理です。
    //-------------------------------------------------------------------------
    virtual void ProcFin()
    {
        // 入力デバイスの終了処理を行います。
        {
            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.allocator = this->GetAllocator();
            nw::dev::FileDeviceManager* pFileSystem = NW_INSTANCE(nw::dev::FileDeviceManager);
            pFileSystem->Unload(loadArg, m_FontBinary.Begin());

#if defined( NW_PLATFORM_CAFE )
            nw::dev::PadDeviceCafe::GetInstance()->Finalize();
            nw::dev::WPadDeviceCafe::GetInstance()->Finalize();
            nw::dev::KeyboardMouseDeviceCafe::GetInstance()->Finalize();
#else
            nw::dev::JoyPadDeviceWin::GetInstance()->Finalize();
            nw::dev::KeyboardMouseDeviceWin::GetInstance()->Finalize();
#endif

            nw::ut::SafeFreeWithDestruct(m_Pad, m_Arg.allocator);
            nw::ut::SafeFreeWithDestruct(m_VPad, m_Arg.allocator);
            nw::ut::SafeFreeWithDestruct(m_WPad, m_Arg.allocator);
            nw::ut::SafeFreeWithDestruct(m_Mouse, m_Arg.allocator);
        }

        m_GraphicsResource.Finalize();

#if defined(NW_PLATFORM_CAFE)
        m_ShaderSetupHelper.Finalize();
#endif

        m_DwTextRenderer.Finalize();

#if defined(NW_VWRLYT_ENABLE)
        m_LayoutViewer.Finalize();
#endif
    }

    //-------------------------------------------------------------------------
    //! @brief        デモの計算処理です。
    //-------------------------------------------------------------------------
    virtual void ProcCalc()
    {
#if defined(NW_VWRLYT_ENABLE)

        // デバイスを更新します。
        {
#if defined(NW_PLATFORM_CAFE)
            nw::dev::PadDeviceCafe::GetInstance()->Update();
            nw::dev::WPadDeviceCafe::GetInstance()->Update();
            nw::dev::VPadDeviceCafe::GetInstance()->Update();
            nw::dev::KeyboardMouseDeviceCafe::GetInstance()->Update();
#else
            nw::dev::KeyboardMouseDeviceWin::GetInstance()->Update();
            nw::dev::JoyPadDeviceWin::GetInstance()->Update();
#endif
           // パッドを更新します。
            m_Pad->Update();
            m_WPad->Update();
            m_VPad->Update();
            m_Mouse->Update();
        }

#if defined(NW_PLATFORM_WIN32)
        if (m_LayoutViewer.IsScreenShotDone())
        {
            m_Exit = true;
            return;
        }
#endif
        m_LayoutViewer.UpdateSystem();
        m_Inputs.Update();
        m_LayoutViewer.UpdateInputs(m_Inputs);
        m_LayoutViewer.UpdateMenu();

        this->GetMeterCalc().BeginMeasure();
        m_LayoutViewer.UpdateLayout();
        this->GetMeterCalc().EndMeasure();

        // FPSの切り替え
        nw::vwrlyt::Viewer::Fps fps = m_LayoutViewer.GetFps();
        if (m_Fps != fps)
        {
            m_Fps = fps;

            bool isFps60 = (fps == nw::vwrlyt::Viewer::FPS_60);

#if defined(NW_PLATFORM_WIN32)
            this->SetFps(isFps60? 60.f : 30.f);
#endif

#if defined(NW_PLATFORM_CAFE)
            this->SetVBlankWaitInterval(isFps60? 1 : 2);
#endif
        }

#endif // NW_VWRLYT_ENABLE
    }

    //-------------------------------------------------------------------------
    //! @brief        TV画面の描画処理です。
    //-------------------------------------------------------------------------
    virtual void ProcDraw()
    {
#if defined(NW_VWRLYT_ENABLE)

        {
            nw::math::Matrix44 projMatrix;
            projMatrix.SetOrtho(0.f, f32(this->GetWidth()), f32(this->GetHeight()), 0.f, 0.0f, 1.f);
            m_DwUIRenderer.SetProjectionMatrix(projMatrix);
        }


        this->SetViewport(0.f, 0.f, f32(this->GetWidth()), f32(this->GetHeight()), 0.f, 1.f);
        this->SetScissor(0.f, 0.f, f32(this->GetWidth()), f32(this->GetHeight()));
        this->ClearFrameBuffers();

        this->GetMeterGPU().BeginMeasure();
        this->GetMeterDraw().BeginMeasure();
        m_LayoutViewer.DrawLayout();
        this->GetMeterDraw().EndMeasure();
        this->GetMeterGPU().EndMeasure();

        m_LayoutViewer.DrawSystem();

        if (m_Arg.usePointerCursor &&
            m_Inputs.GetMouse() &&
            m_Inputs.GetMouse()->IsPointerOn())
        {
            nw::math::Vector2 pos = m_Inputs.GetMouse()->GetPointer();
            this->DrawPointerCursor(
                pos.x * 2.f / this->GetWidth() -1.f,
                pos.y * 2.f / this->GetHeight() - 1.f,
                f32(this->GetWidth()),
                f32(this->GetHeight()));
        }

        this->DrawLoadMeters();

        m_DrawOnce = true;

#endif // NW_VWRLYT_ENABLE
    }

#if defined(NW_PLATFORM_WIN32)
    static LRESULT CALLBACK WindowMessageCallback(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
    {
        NW_UNUSED_VARIABLE(hWnd);
        NW_UNUSED_VARIABLE(message);
        NW_UNUSED_VARIABLE(wParam);
        NW_UNUSED_VARIABLE(lParam);
#if defined(NW_VWRLYT_ENABLE)
        if (GetInstance()->m_LayoutViewer.PreviewByMessageProcCallback(hWnd, message, wParam, lParam))
        {
            return 1;
        }
#endif

        return 0;
    }
#endif

    //-------------------------------------------------------------------------
    //! @brief        インスタンスを取得します。
    //-------------------------------------------------------------------------
    static TestDemoSystem* GetInstance()
    {
        return s_pInstance;
    }

private:
    //! グラフィックスコンテキストです。
    nw::gfnd::GraphicsContext   m_GfxContext;

    //! レイアウトグラフィックスリソースです。
    nw::lyt::GraphicsResource m_GraphicsResource;

    //! フォントバイナリです。
    nw::ut::MemoryRange m_FontBinary;

#if defined(NW_PLATFORM_CAFE)
    //! レイアウトシェーダ・セットアップヘルパーです。
    nw::lyt::ShaderSetupHelper m_ShaderSetupHelper;
#endif

#if defined(NW_VWRLYT_ENABLE)
    //! レイアウトビューアです。
    nw::vwrlyt::Viewer m_LayoutViewer;
    //! 入力情報です。
    nw::vwrlyt::Inputs m_Inputs;
    //!
    nw::vwrlyt::Viewer::Fps m_Fps;
#endif

    //!
    bool m_DrawOnce;

    nw::internal::dw::NwTextRenderer m_DwTextRenderer;
    nw::internal::dw::NwUIRenderer m_DwUIRenderer;
    nw::internal::dw::ControlWindow m_DwControlWindow;
    CreateArg m_Arg;

#if defined( NW_PLATFORM_CAFE )
    nw::dev::PadCafe*          m_Pad;    // デバッグ用パッドを管理します。
    nw::dev::WPadCafe*         m_WPad;   // WPAD を管理します。
    nw::dev::VPadCafe*         m_VPad;   // VPAD を管理します。
    nw::dev::MouseCafe*        m_Mouse;  // Mouse を管理します。
#else
    nw::dev::PadWin*           m_Pad;    // デバッグ用パッドを管理します。
    nw::dev::WPadWin*          m_WPad;   // WPAD をマウスを用いて PC で再現します。
    nw::dev::VPadWin*          m_VPad;   // VPAD を PC で再現します。
    nw::dev::MouseWin*         m_Mouse;  // Mouse を PC で再現します。
#endif

    static TestDemoSystem* s_pInstance;
};

TestDemoSystem* TestDemoSystem::s_pInstance = NULL;

//-----------------------------------------------------------------------------
void
TestDemoSystem::LoadSampleData()
{
}

//-----------------------------------------------------------------------------

} // anonymous namespace

//-----------------------------------------------------------------------------
//! @brief        メイン関数です。
//-----------------------------------------------------------------------------
int
NwDemoMain(int argc, char **argv)
{
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);

    nw::ut::MemoryAllocator demoAllocator;
    InitializeMemory( &demoAllocator );

    nw::ut::MemoryAllocator* allocator = &demoAllocator;

    nw::lyt::Initialize(allocator);

    // ファイルシステムを初期化します。
#if defined( NW_PLATFORM_WIN32 )
    SetEnvironmentVariableA("CAFE_CONTENT_DIR", "UNUSED"); // Win32では CAFE_CONTENT_DIR を用いないため、適当な文字列を設定しています。
#endif
    TestDemoSystem::InitFileDeviceManager(allocator);

    // CAFE_CONTENT_DIR 環境変数を取得します。
    TestDemoSystem::PathString nwRoot;
    if (!TestDemoSystem::GetNwRoot(nwRoot))
    {
        NW_LOG("Can not get valid CAFE_CONTENT_DIR environment variable.");
    	NW_HALT;
    }

#if defined(USE_MCS_INPUT)
    // マウス入力キャプチャのために mcs を初期化します。
    {
        nw::mcs::Mcs_Initialize();

        u32 errorCode = nw::mcs::Mcs_Open();
        NW_ASSERT( errorCode == nw::mcs::MCS_ERROR_SUCCESS );

        // mcsHID を初期化します。
        nw::mcs::McsHID_Initialize();
        if ( nw::mcs::McsHID_GetRegisteredBuffer() == NULL )
        {
            // 登録済みのバッファがない場合には、バッファを生成して登録します。
            s_InputDeviceBuf = reinterpret_cast<u8*>( allocator->Alloc( MCS_INPUT_DEVICE_BUF_SIZE ) );
            nw::mcs::McsHID_RegisterBuffer( s_InputDeviceBuf, MCS_INPUT_DEVICE_BUF_SIZE );
        }
    }
#endif // #if defined(USE_MCS_INPUT)

    {
        // デモシステムの初期化を行います。
        TestDemoSystem::CreateArg demoSystemArg;
        demoSystemArg.allocator = allocator;
        demoSystemArg.width     = SCREEN_WIDTH;
        demoSystemArg.height    = SCREEN_HEIGHT;
        demoSystemArg.drawMeter = true;
#if defined( NW_PLATFORM_CAFE )
        demoSystemArg.usePointerCursor = true;
#endif
#if defined( NW_PLATFORM_WIN32 )
        demoSystemArg.primitiveRendererInitialize = true;
#endif

        demoSystemArg.nwRootPath = nwRoot.c_str();

        TestDemoSystem::PathString fontPath;
        fontPath.format(
            "%s/nintendo_NTLG-DB_002_Nw4f.bffnt",
            nwRoot.c_str());
        demoSystemArg.fontPath = fontPath.c_str();

#if defined( NW_PLATFORM_CAFE )
        TestDemoSystem::PathString fontShaderPath;
        fontShaderPath.format(
            "%s/font_BuildinShader.gsh",
            nwRoot.c_str());
        demoSystemArg.fontShaderPath = fontShaderPath.c_str();

        TestDemoSystem::PathString lytShaderPath;
        lytShaderPath.format(
            "%s/lyt_BuildinShader.gsh",
            nwRoot.c_str());
        demoSystemArg.lytShaderPath = lytShaderPath.c_str();

        TestDemoSystem::PathString primRendererPath;
        primRendererPath.format(
            "%s/dev_PrimitiveRenderer.gsh",
            nwRoot.c_str());
        demoSystemArg.primitiveRendererShaderPath = primRendererPath.c_str();
#endif
        void* pDemoSystemBuffer = allocator->Alloc(sizeof(TestDemoSystem));
        TestDemoSystem* pDemoSystem = new (pDemoSystemBuffer) TestDemoSystem( demoSystemArg );
        NW_ASSERT_NOT_NULL(pDemoSystem);

        // CAFE版ビルド時のワーニング抑制
        (void)TestDemoSystem::GetInstance();

        pDemoSystem->Initialize();
        pDemoSystem->InitializeGraphicsSystem();

#if defined( NW_PLATFORM_WIN32 )
        // プレビュー要求のWindowsメッセージを受信するために、
        // ウィンドウタイトルに"Layout Viewer"を含むように設定します。
        pDemoSystem->SetWindowTitle("Layout Viewer");
        pDemoSystem->SetMsgProcCallback(TestDemoSystem::WindowMessageCallback);
#endif

        // 自動テスト用のブロックです。
        NW_DEMO_TEST_BLOCK( allocator )
        {
            pDemoSystem->ProcInit();

            // 描画ループです。
            do
            {
                pDemoSystem->BeginFrame();

                pDemoSystem->ProcCalc();

                nw::gfnd::Graphics::GetInstance()->LockDrawContext();
                {
#if defined(NW_PLATFORM_CAFE)
                    // GPUのreadキャッシュを全域クリアする
                    GX2Invalidate(static_cast<GX2InvalidateType>(GX2_INVALIDATE_ATTRIB_BUFFER | GX2_INVALIDATE_TEXTURE | GX2_INVALIDATE_CONSTANT_BUFFER | GX2_INVALIDATE_SHADER),
                        0x00000000, 0xffffffff );
#endif
                    pDemoSystem->ProcDraw();
                }
                nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

                pDemoSystem->SwapBuffer();

                pDemoSystem->EndFrame();

                pDemoSystem->WaitForVBlank();
            } while ( ! pDemoSystem->IsExiting() );

            pDemoSystem->ProcFin();
        }

        // デモシステムの終了処理を行います。
        pDemoSystem->FinalizeGraphicsSystem();
        pDemoSystem->Finalize();
        nw::ut::SafeFree(pDemoSystemBuffer, allocator);

#if defined(USE_MCS_INPUT)
    // mcs の終了処理を行います。
    {
        if ( s_InputDeviceBuf != NULL && nw::mcs::McsHID_GetRegisteredBuffer() == s_InputDeviceBuf )
        {
            nw::mcs::McsHID_UnregisterBuffer();
            allocator->Free( s_InputDeviceBuf );
        }

        nw::mcs::McsHID_Finalize();
        nw::mcs::Mcs_Finalize();
    }
#endif // #if defined(USE_MCS_INPUT)

        TestDemoSystem::FinalizeFileDeviceManager();
        nw::ut::SafeFreeWithDestruct(pDemoSystem, allocator);
    }

    return 0;
}
