﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace InputCapture.Interop
{
    using System;
    using System.Runtime.InteropServices;

    /// <summary>
    /// マウス入力のフック処理を行うクラスです。
    /// </summary>
    public class MouseHook
    {
        [StructLayout(LayoutKind.Sequential)]
        public struct POINT
        {
            public int x;
            public int y;

            public POINT(int x, int y)
            {
                this.x = x;
                this.y = y;
            }
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct MSLLHOOKSTRUCT
        {
            public POINT pt;
            public int mouseData;
            public int flags;
            public int time;
            public IntPtr dwExtraInfo;
        }

        //-----------------------------------------------------------------

        public const int HC_ACTION = 0;
        public const int WH_MOUSE_LL = 14;

        public enum WM : int
        {
            WM_MOUSEMOVE = 0x0200,
            WM_LBUTTONDOWN = 0x0201,
            WM_LBUTTONUP = 0x0202,
            WM_LBUTTONDBLCLK = 0x0203,
            WM_RBUTTONDOWN = 0x0204,
            WM_RBUTTONUP = 0x0205,
            WM_RBUTTONDBLCLK = 0x0206,
            WM_MBUTTONDOWN = 0x0207,
            WM_MBUTTONUP = 0x0208,
            WM_MBUTTONDBLCLK = 0x0209,
            WM_MOUSEWHEEL = 0x020A,
            WM_XBUTTONDOWN = 0x020B,
            WM_XBUTTONUP = 0x020C,
            WM_XBUTTONDBLCLK = 0x020D,
            WM_MOUSEHWHEEL = 0x020E,
        }

        //-----------------------------------------------------------------

        private IntPtr hookHandle;
        private User32.HookProc mouseProc;
        private readonly MouseHookEventArgs eventArgs = new MouseHookEventArgs();

        //-----------------------------------------------------------------

        public event EventHandler<MouseHookEventArgs> MouseDown;
        public event EventHandler<MouseHookEventArgs> MouseUp;
        public event EventHandler<MouseHookEventArgs> MouseMove;
        public event EventHandler<MouseHookEventArgs> MouseWheel;

        //-----------------------------------------------------------------

        public MouseHook()
        {
            AppDomain.CurrentDomain.DomainUnload +=
                (sender, e) => this.Stop();

            // GC に回収されないようにメンバ変数に保持しておきます。
            this.mouseProc = new User32.HookProc(this.HookProc);
        }

        //-----------------------------------------------------------------

        public bool IsCapturing
        {
            get { return this.hookHandle != IntPtr.Zero; }
        }

        //-----------------------------------------------------------------

        public void Start()
        {
            this.hookHandle = User32.SetWindowsHookEx(
                WH_MOUSE_LL,
                this.mouseProc,
                Marshal.GetHINSTANCE(typeof(MouseHook).Module),
                0);
        }

        public void Stop()
        {
            if (this.hookHandle == IntPtr.Zero)
            {
                return;
            }

            User32.UnhookWindowsHookEx(this.hookHandle);
            this.hookHandle = IntPtr.Zero;
        }

        private void RaiseEvent(EventHandler<MouseHookEventArgs> eventHandler, MouseHookEventArgs args)
        {
            if (eventHandler != null)
            {
                eventHandler(this, args);
            }
        }

        private IntPtr HookProc(int nCode, IntPtr wParam, IntPtr lParam)
        {
            if (nCode != HC_ACTION)
            {
                return User32.CallNextHookEx(hookHandle, nCode, wParam, lParam);
            }

            // HookProc は頻繁にコールされるため、
            // 毎回 eventArgs のインスタンスを生成しないようにしています。
            eventArgs.Initialize((int)wParam, lParam);

            switch ((WM)wParam.ToInt32())
            {
                case WM.WM_LBUTTONDBLCLK:
                    break;
                case WM.WM_LBUTTONDOWN:
                case WM.WM_MBUTTONDOWN:
                case WM.WM_RBUTTONDOWN:
                case WM.WM_XBUTTONDOWN:
                    this.RaiseEvent(MouseDown, this.eventArgs);
                    break;

                case WM.WM_LBUTTONUP:
                case WM.WM_MBUTTONUP:
                case WM.WM_RBUTTONUP:
                case WM.WM_XBUTTONUP:
                    this.RaiseEvent(MouseUp, this.eventArgs);
                    break;

                case WM.WM_MOUSEMOVE:
                    this.RaiseEvent(MouseMove, this.eventArgs);
                    break;

                case WM.WM_MOUSEWHEEL:
                    this.RaiseEvent(MouseWheel, this.eventArgs);
                    break;
            }

            if (eventArgs.IsParameterUpdated)
            {
                wParam = (IntPtr)eventArgs.WParam;

                unsafe
                {
                    *(MSLLHOOKSTRUCT*)lParam = eventArgs.LParam;
                }
            }

            if (eventArgs.IsCanceled)
            {
                return (IntPtr)1;
            }

            return User32.CallNextHookEx(hookHandle, nCode, wParam, lParam);
        }
    }

    public sealed class MouseHookEventArgs : EventArgs
    {
        public enum MouseButtons
        {
            None,
            Left,
            Right,
            Middle,
            XButton1,
            XButton2
        }

        public const int WHEEL_DELTA = 120;

        //-----------------------------------------------------------------

        private bool isCanceled;
        private bool isParameterUpdated;
        private int wParam;
        private MouseHook.MSLLHOOKSTRUCT lParam;
        private MouseButtons button;
        private int delta;

        //-----------------------------------------------------------------

        internal MouseHookEventArgs()
        {
        }

        internal void Initialize(int wParam, IntPtr lParam)
        {
            this.isCanceled = false;
            this.isParameterUpdated = false;
            this.wParam = wParam;
            this.button = MouseButtons.None;
            this.delta = 0;

            unsafe
            {
                this.lParam = *(MouseHook.MSLLHOOKSTRUCT*)lParam;
            }

            switch ((MouseHook.WM)wParam)
            {
                case MouseHook.WM.WM_LBUTTONDOWN:
                case MouseHook.WM.WM_LBUTTONUP:
                    this.button = MouseButtons.Left;
                    break;

                case MouseHook.WM.WM_MBUTTONDOWN:
                case MouseHook.WM.WM_MBUTTONUP:
                    this.button = MouseButtons.Middle;
                    break;

                case MouseHook.WM.WM_RBUTTONDOWN:
                case MouseHook.WM.WM_RBUTTONUP:
                    this.button = MouseButtons.Right;
                    break;

                case MouseHook.WM.WM_XBUTTONDOWN:
                case MouseHook.WM.WM_XBUTTONUP:
                    switch (this.lParam.mouseData >> 16)
                    {
                        case 0x0001:
                            this.button = MouseButtons.XButton1;
                            break;

                        case 0x0002:
                            this.button = MouseButtons.XButton2;
                            break;
                    }
                    break;

                case MouseHook.WM.WM_MOUSEWHEEL:
                    this.button = MouseButtons.None;
                    this.delta = this.lParam.mouseData >> 16;    // 上位16ビットを抽出
                    break;
            }
        }

        public MouseButtons Button
        {
            get { return this.button; }
        }

        /// <summary>
        /// マウス操作をキャンセルします。
        /// </summary>
        public bool IsCanceled
        {
            get { return this.isCanceled; }
            set { this.isCanceled = value; }
        }

        public int X
        {
            get
            {
                return this.lParam.pt.x;
            }

            set
            {
                this.lParam.pt = new MouseHook.POINT(value, this.lParam.pt.y);
                this.isParameterUpdated = true;
            }
        }

        public int Y
        {
            get
            {
                return this.lParam.pt.y;
            }

            set
            {
                this.lParam.pt = new MouseHook.POINT(this.lParam.pt.x, value);
                this.isParameterUpdated = true;
            }
        }

        public int Delta
        {
            get { return this.delta; }
        }

        public int Time
        {
            get { return this.lParam.time; }
        }

        public int WParam
        {
            get
            {
                return this.wParam;
            }

            set
            {
                this.wParam = value;
                this.isParameterUpdated = true;
            }
        }

        public MouseHook.MSLLHOOKSTRUCT LParam
        {
            get
            {
                return this.lParam;
            }

            set
            {
                this.lParam = value;
                this.isParameterUpdated = true;
            }
        }

        internal bool IsParameterUpdated
        {
            get { return this.isParameterUpdated; }
        }
    }
}
