﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace InputCapture.Interop
{
    using System;
    using System.Runtime.InteropServices;

    public class KeyboardHook
    {
        public const int HC_ACTION = 0;
        public const int WH_KEYBOARD_LL = 13;

        [StructLayout(LayoutKind.Sequential)]
        public struct KBDLLHOOKSTRUCT
        {
            public int vkCode;
            public int scanCode;
            public uint flags;
            public int time;
            public IntPtr dwExtraInfo;
        }

        public enum WM : int
        {
            WM_KEYDOWN    = 0x0100,
            WM_KEYUP      = 0x0101,
            WM_SYSKEYDOWN = 0x0104,
            WM_SYSKEYUP   = 0x0105,
        }

        //-----------------------------------------------------------------

        private IntPtr hookHandle;
        private User32.HookProc keyboardProc;
        private readonly KeyboardHookEventArgs eventArgs = new KeyboardHookEventArgs();

        //-----------------------------------------------------------------

        public event EventHandler<KeyboardHookEventArgs> KeyDown;
        public event EventHandler<KeyboardHookEventArgs> KeyUp;
        public event EventHandler<KeyboardHookEventArgs> SystemKeyDown;
        public event EventHandler<KeyboardHookEventArgs> SystemKeyUp;

        //-----------------------------------------------------------------

        public KeyboardHook()
        {
            AppDomain.CurrentDomain.DomainUnload +=
                (sender, e) => this.Stop();

            // GC に回収されないようにメンバ変数に保持しておきます。
            this.keyboardProc = new User32.HookProc(this.HookProc);
        }

        //-----------------------------------------------------------------

        public bool IsCapturing
        {
            get { return this.hookHandle != IntPtr.Zero; }
        }

        //-----------------------------------------------------------------

        public void Start()
        {
            this.hookHandle = User32.SetWindowsHookEx(
                WH_KEYBOARD_LL,
                this.keyboardProc,
                Marshal.GetHINSTANCE(typeof(KeyboardHook).Module),
                0);
        }

        public void Stop()
        {
            if (this.hookHandle == IntPtr.Zero)
            {
                return;
            }

            User32.UnhookWindowsHookEx(this.hookHandle);
            this.hookHandle = IntPtr.Zero;
        }

        private void RaiseEvent(EventHandler<KeyboardHookEventArgs> eventHandler, KeyboardHookEventArgs args)
        {
            if (eventHandler != null)
            {
                eventHandler(this, args);
            }
        }

        private IntPtr HookProc(int nCode, IntPtr wParam, IntPtr lParam)
        {
            if (nCode != HC_ACTION)
            {
                return User32.CallNextHookEx(hookHandle, nCode, wParam, lParam);
            }

            // HookProc は頻繁にコールされるため、
            // 毎回 eventArgs のインスタンスを生成しないようにしています。
            eventArgs.Initialize((int)wParam, lParam);

            switch ((WM)wParam.ToInt32())
            {
                case WM.WM_KEYDOWN:
                    this.RaiseEvent(KeyDown, this.eventArgs);
                    break;

                case WM.WM_KEYUP:
                    this.RaiseEvent(KeyUp, this.eventArgs);
                    break;

                case WM.WM_SYSKEYDOWN:
                    this.RaiseEvent(SystemKeyDown, this.eventArgs);
                    break;

                case WM.WM_SYSKEYUP:
                    this.RaiseEvent(SystemKeyUp, this.eventArgs);
                    break;
            }

            if (eventArgs.IsParameterUpdated)
            {
                wParam = (IntPtr)eventArgs.WParam;

                unsafe
                {
                    *(KBDLLHOOKSTRUCT*)lParam = eventArgs.LParam;
                }
            }

            if(eventArgs.IsCanceled)
            {
                return (IntPtr)1;
            }

            return User32.CallNextHookEx(hookHandle, nCode, wParam, lParam);
        }
    }

    public sealed class KeyboardHookEventArgs : EventArgs
    {
        private bool isCanceled;
        private bool isParameterUpdated;
        private int wParam;
        private KeyboardHook.KBDLLHOOKSTRUCT lParam;

        //-----------------------------------------------------------------

        internal KeyboardHookEventArgs() { }

        internal void Initialize(int wParam, IntPtr lParam)
        {
            this.isCanceled = false;
            this.isParameterUpdated = false;
            this.wParam = wParam;

            unsafe
            {
                this.lParam = *(KeyboardHook.KBDLLHOOKSTRUCT*)lParam;
            }
        }

        public bool IsSystemKey
        {
            get
            {
                switch ((KeyboardHook.WM)wParam)
                {
                    case KeyboardHook.WM.WM_SYSKEYDOWN:
                    case KeyboardHook.WM.WM_SYSKEYUP:
                        return true;
                }

                return false;
            }
        }

        public int KeyCode
        {
            get
            {
                return this.lParam.vkCode;
            }

            set
            {
                this.lParam.vkCode = value;
                this.isParameterUpdated = true;
            }
        }

        public int ScanCode
        {
            get
            {
                return this.lParam.scanCode;
            }

            set
            {
                this.lParam.scanCode = value;
                this.isParameterUpdated = true;
            }
        }

        public uint Flags
        {
            get { return this.lParam.flags; }
        }

        public int Time
        {
            get { return this.lParam.time; }
        }

        public bool IsCanceled
        {
            get { return this.isCanceled; }
            set { this.isCanceled = value; }
        }

        public int WParam
        {
            get
            {
                return this.wParam;
            }

            set
            {
                this.wParam = value;
                this.isParameterUpdated = true;
            }
        }

        public KeyboardHook.KBDLLHOOKSTRUCT LParam
        {
            get
            {
                return this.lParam;
            }

            set
            {
                this.lParam = value;
                this.isParameterUpdated = true;
            }
        }

        internal bool IsParameterUpdated
        {
            get { return this.isParameterUpdated; }
        }
    }
}
