﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace InputCapture.Communication
{
    using System;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using InputCapture.Binary;

    /// <summary>
    /// 送受信するメッセージの種類を表します。
    /// </summary>
    public enum MessageType
    {
        Ping,
        Ack,

        CaptureStart,
        CaptureStop,

        KeyDown,
        KeyUp,

        MouseMove,
        MouseDown,
        MouseUp,
        MouseDoubleClick,
        MouseWheel
    }

    [Flags]
    public enum KeyModifier : byte
    {
        None = 0x00,
        Shift = 0x01,
        Control = 0x02,
        Alt = 0x04,
    }

    [Flags]
    public enum KeyAttribute : byte
    {
        NumPad = 0x01,  // 数値キーパッドの数字
    }

    public enum KeyCategory : byte
    {
        Function,
        System,
        Alphabet,
        Numeric,
        Symbol,
    }

    /// <summary>
    /// マウスボタンを表します。
    /// </summary>
    [Flags]
    public enum MouseButton
    {
        None = 0x00,
        Left = 0x01,
        Right = 0x02,
        Middle = 0x04,
        X1 = 0x08,
        X2 = 0x10,
    }

    /// <summary>
    /// 送信メッセージに変換するための基底クラスです。
    /// </summary>
    public abstract class SendMessageArg
    {
        public abstract void ToStream(Stream stream);

        protected static void WritePadding(NetBinaryWriter bw, int bytes)
        {
            for (int i = 0; i < bytes; ++i)
            {
                byte padding = 0;
                bw.Write(padding);
            }
        }
    }

    /// <summary>
    /// キャプチャー開始イベントを送信形式に変換するクラスです。
    /// </summary>
    public class CaptureEventArg : SendMessageArg
    {
        public BorderType borderType;
        public bool captureOnMousePosition;
        public uint borderLength;
        public Point pointerPos;

        public override void ToStream(Stream stream)
        {
            NetBinaryWriter bw = new NetBinaryWriter(stream);
            bw.Write((byte)borderType);
            bw.Write((byte)(captureOnMousePosition ? 1 : 0));
            bw.Write((ushort)borderLength);
            bw.Write((short)pointerPos.X);
            bw.Write((short)pointerPos.Y);
            bw.Flush();
        }

        public override string ToString()
        {
            return "CaptureEventArg";
        }

        public string ToInfoStr()
        {
            return "";
        }
    }

    /// <summary>
    /// キーイベントを送信形式に変換するクラスです。
    /// </summary>
    public class KeyEventArg : SendMessageArg
    {
        public KeyModifier modifier;
        public KeyAttribute attribute;
        public KeyCategory category;
        public byte indexValue;
        public uint keyboardLanguageId;

        public override void ToStream(Stream stream)
        {
            NetBinaryWriter bw = new NetBinaryWriter(stream);
            bw.Write((byte)modifier);
            bw.Write((byte)attribute);
            bw.Write((byte)category);
            bw.Write(indexValue);
            bw.Write(keyboardLanguageId);
            bw.Flush();
        }

        public override string ToString()
        {
            return "McsKeyEventArg: " + ToInfoStr();
        }

        public string ToInfoStr()
        {
            return string.Format("{0}, {1}, index {2:X2}, {3}, {4}", attribute, category, indexValue, modifier, keyboardLanguageId);
        }
    }

    /// <summary>
    /// マウスイベントを送信形式に変換するクラスです。
    /// </summary>
    public class MouseEventArg : SendMessageArg
    {
        public MouseButton button = MouseButton.None;
        public short wheel = 0;
        public short scrollLines = 0;

        public override void ToStream(Stream stream)
        {
            NetBinaryWriter bw = new NetBinaryWriter(stream);
            bw.Write((byte)button);
            bw.Write((byte)0);
            bw.Write((short)wheel);
            bw.Write((short)scrollLines);
            WritePadding(bw, 2);
            bw.Flush();
        }

        public override string ToString()
        {
            return "MouseEventArg: " + ToInfoStr();
        }

        public string ToInfoStr()
        {
            return string.Format("wheel = {0}, scrollLines = {1}", wheel, scrollLines);
        }
    }

    /// <summary>
    /// ポインターイベントを送信形式に変換するクラスです。
    /// </summary>
    public class PointerEventArg : SendMessageArg
    {
        public Point point;

        public override void ToStream(Stream stream)
        {
            NetBinaryWriter bw = new NetBinaryWriter(stream);
            bw.Write((short)point.X);
            bw.Write((short)point.Y);
            WritePadding(bw, 4);
            bw.Flush();
        }

        public override string ToString()
        {
            return "PointerEventArg: " + ToInfoStr();
        }

        public string ToInfoStr()
        {
            return string.Format("[X, Y] [{0,4:d},{1,4:d}]", point.X, point.Y);
        }
    }

    //-----------------------------------------------------------------

    /// <summary>
    /// メッセージ送信を行うクラスです。
    /// </summary>
    public class MessageSender
    {
        public const int MsgSize = 12;

        private Connecter _connecter = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public MessageSender(Connecter connecter)
        {
            Debug.Assert(connecter != null);
            _connecter = connecter;
        }

        /// <summary>
        /// キャプチャした入力データを書き込みます。
        /// </summary>
        public bool WriteInputData(MessageType eventType, SendMessageArg arg)
        {
            MemoryStream memStream = new MemoryStream();
            WriteEventType(memStream, eventType);
            arg.ToStream(memStream);

            if (!_connecter.WriteStream(memStream))
            {
                return false;
            }

            return true;
        }

        private void WriteEventType(Stream stream, MessageType eventType)
        {
            BinaryWriter bw = new BinaryWriter(stream);
            byte byteData = (byte)eventType;
            bw.Write(byteData);
            byteData = 0;
            bw.Write(byteData);
            bw.Write(byteData);
            bw.Write(byteData);
            bw.Flush();
        }
    }
}
