﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// DecodeTga OutputTgaFile

//=============================================================================
// include
//=============================================================================
#include "ImageConvert.h"

using namespace std;

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

namespace
{

//-----------------------------------------------------------------------------
//! @brief decode TGA bitmap
//-----------------------------------------------------------------------------
void DecodeTgaBitmap(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int imageW,
    const int imageH
)
{
    //-----------------------------------------------------------------------------
    // ヘッダを解析します。
    const int idSize = static_cast<int>(pSrc[0x00]);
    size_t iBuf = RImage::TGA_HEADER_SIZE + idSize;

    const bool hasPal = (pSrc[0x01] != 0);
    const int imageType = pSrc[0x02];
    const int bitPerPixel = pSrc[0x10];

    // decode image descriptor
    //const int attrBitSize = pSrc[0x11] & 0x0f;
    const bool rightToLeft  = ((pSrc[0x11] & 0x10) != 0);
    const bool upperToLower = ((pSrc[0x11] & 0x20) != 0);
    const int interLeave    =  (pSrc[0x11] & 0xc0) >> 6; // 0=none, 1=two, 2=four

    //-----------------------------------------------------------------------------
    // パレットをデコードします。
    uint8_t* pPalBuf = NULL;
    if (hasPal)
    {
        const int ciMin   = pSrc[0x03] | (pSrc[0x04] << 8);
        const int colSize = pSrc[0x05] | (pSrc[0x06] << 8);
        const int ciMax = ciMin + colSize;
        const int colBitSize = pSrc[0x07];
        pPalBuf = new uint8_t[ciMax * R_RGBA_BYTES];
        int iCol;
        for (iCol = 0; iCol < ciMin; ++iCol)
        {
            // ciMin までを黒で埋めます。
            int ofs = iCol * R_RGBA_BYTES;
            pPalBuf[ofs    ] = pPalBuf[ofs + 1] = pPalBuf[ofs + 2] = 0x00;
            pPalBuf[ofs + 3] = 0xff;
        }
        for ( ; iCol < ciMax; ++iCol)
        {
            uint8_t cr, cg, cb, ca;
            if (colBitSize == 8) // GRAY 8 bit
            {
                cr = cg = cb = ca = pSrc[iBuf++];
            }
            else if (colBitSize == 15 ||    // RGB(A) 15/16 bit
                     colBitSize == 16)
            {
                int val = pSrc[iBuf] | (pSrc[iBuf + 1] << 8);
                iBuf += 2;
                cr = static_cast<uint8_t>((val >> 10) & 0x1f);
                cg = static_cast<uint8_t>((val >>  5) & 0x1f);
                cb = static_cast<uint8_t>((val      ) & 0x1f);
                cr = (cr << 3) | (cr >> 2);
                cg = (cg << 3) | (cg >> 2);
                cb = (cb << 3) | (cb >> 2);
                ca = 0xff;
                //if (attrBitSize == 0)
                //{
                //  ca = 0xff;
                //}
                //else
                //{
                //  ca = (val & 0x8000) ? 0xff : 0x00;
                //}
            }
            else if (colBitSize == 24) // RGB 24 bit
            {
                cb = pSrc[iBuf    ];
                cg = pSrc[iBuf + 1];
                cr = pSrc[iBuf + 2];
                ca = 0xff;
                iBuf += 3;
            }
            else if (colBitSize == 32) // RGBA 32 bit
            {
                cb = pSrc[iBuf    ];
                cg = pSrc[iBuf + 1];
                cr = pSrc[iBuf + 2];
                ca = pSrc[iBuf + 3];
                iBuf += 4;
            }
            else
            {
                cr = cg = cb = ca = 0;
            }
            const int palOfs = iCol * R_RGBA_BYTES;
            pPalBuf[palOfs    ] = cr;
            pPalBuf[palOfs + 1] = cg;
            pPalBuf[palOfs + 2] = cb;
            pPalBuf[palOfs + 3] = ca;
        }
    }

    //-----------------------------------------------------------------------------
    // ビットマップをデコードします。
    const bool rleFlag = (imageType == 9 || imageType == 10 || imageType == 11);
    int rleCount = 0;
    bool sameFlag = false;
    uint8_t cr = 0, cg = 0, cb = 0, ca = 0;
    int iyOfs = 0;
    int iyCur = iyOfs;
    for (int iy = 0; iy < imageH; ++iy)
    {
        const int iyDst = (upperToLower) ? iyCur : imageH - 1 - iyCur;
        int dstOfs = iyDst * imageW * R_RGBA_BYTES;
        if (rightToLeft)
        {
            dstOfs += (imageW - 1) * R_RGBA_BYTES;
        }
        for (int ix = 0; ix < imageW; ++ix)
        {
            bool getColorFlag = true;
            if (rleFlag)
            {
                if (rleCount == 0)
                {
                    int count = pSrc[iBuf++];
                    if (count & 0x80)
                    {
                        rleCount = count - 0x80;
                    }
                    else
                    {
                        rleCount = count;
                    }
                    sameFlag = ((count & 0x80) != 0);
                }
                else
                {
                    --rleCount;
                    getColorFlag = !sameFlag;
                }
            }

            if (getColorFlag)
            {
                if (hasPal)                 // CI 8/16 bit
                {
                    int palOfs;
                    if (bitPerPixel == 8)
                    {
                        palOfs = pSrc[iBuf++] * R_RGBA_BYTES;
                    }
                    else
                    {
                        palOfs = (pSrc[iBuf] | (pSrc[iBuf + 1] << 8)) * R_RGBA_BYTES;
                        iBuf += 2;
                    }
                    cr = pPalBuf[palOfs    ];
                    cg = pPalBuf[palOfs + 1];
                    cb = pPalBuf[palOfs + 2];
                    ca = pPalBuf[palOfs + 3];
                }
                else if (bitPerPixel == 8)  // GRAY 8 bit
                {
                    cr = cg = cb = pSrc[iBuf++];
                    ca = 0xff;
                }
                else if (bitPerPixel == 15 ||
                         bitPerPixel == 16) // RGB(A) 16 bit
                {
                    int val = pSrc[iBuf] | (pSrc[iBuf + 1] << 8);
                    iBuf += 2;
                    cr = static_cast<uint8_t>((val >> 10) & 0x1f);
                    cg = static_cast<uint8_t>((val >>  5) & 0x1f);
                    cb = static_cast<uint8_t>((val      ) & 0x1f);
                    cr = (cr << 3) | (cr >> 2);
                    cg = (cg << 3) | (cg >> 2);
                    cb = (cb << 3) | (cb >> 2);
                    ca = 0xff;
                    //if (attrBitSize == 0)
                    //{
                    //  ca = 0xff;
                    //}
                    //else
                    //{
                    //  ca = (val & 0x8000) ? 0xff : 0x00;
                    //}
                }
                else if (bitPerPixel == 24) // RGB 24 bit
                {
                    cb = pSrc[iBuf    ];
                    cg = pSrc[iBuf + 1];
                    cr = pSrc[iBuf + 2];
                    ca = 0xff;
                    iBuf += 3;
                }
                else if (bitPerPixel == 32) // RGBA 32 bit
                {
                    cb = pSrc[iBuf    ];
                    cg = pSrc[iBuf + 1];
                    cr = pSrc[iBuf + 2];
                    ca = pSrc[iBuf + 3];
                    iBuf += 4;
                }
            }
            pDst[dstOfs    ] = cr;
            pDst[dstOfs + 1] = cg;
            pDst[dstOfs + 2] = cb;
            pDst[dstOfs + 3] = ca;
            if (rightToLeft)
            {
                dstOfs -= 4;
            }
            else
            {
                dstOfs += 4;
            }
        }
        if (interLeave == 2)        // 4
        {
            iyCur += 4;
        }
        else if (interLeave == 1)   // 2
        {
            iyCur += 2;
        }
        else                        // 0
        {
            ++iyCur;
        }
        if (iyCur >= imageH)
        {
            iyCur = ++iyOfs;
        }
    }

    //-----------------------------------------------------------------------------
    // パレットを解放します。
    if (pPalBuf != NULL)
    {
        delete[] pPalBuf;
    }
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief TGA ファイルの 2 つのカラーが同じなら true を返します。
//-----------------------------------------------------------------------------
inline bool IsSameTgaColor(const uint8_t* pCol0, const uint8_t* pCol1, const size_t pixelBytes)
{
    return (
        pCol0[0] == pCol1[0] &&
        pCol0[1] == pCol1[1] &&
        pCol0[2] == pCol1[2] &&
        (pixelBytes == R_RGB_BYTES || pCol0[3] == pCol1[3]));
}

//-----------------------------------------------------------------------------
//! @brief TGA ファイルのランレングスを取得します。
//!
//! @param[in] pSrc 処理中のピクセルのポインタです。
//! @param[in] ix 処理中のピクセルの水平方向の座標です。
//! @param[in] imageW 幅です。
//! @param[in] pixelBytes ピクセルあたりのバイト数です。
//!
//! @return 同じカラーが連続するなら「連続する長さ」、
//!         同じカラーが連続しないなら「連続しない長さ * (-1)」を返します。
//-----------------------------------------------------------------------------
int GetTgaRunLength(
    const uint8_t* pSrc,
    const int ix,
    const int imageW,
    const size_t pixelBytes
)
{
    //-----------------------------------------------------------------------------
    // 残り 1 ピクセルのとき
    if (ix == imageW - 1)
    {
        return -1;
    }

    //-----------------------------------------------------------------------------
    // 最初の 2 つのカラーを比較します。
    const uint8_t* pCol0 = pSrc;
    const uint8_t* pCol1 = pSrc + pixelBytes;
    const bool isSame = IsSameTgaColor(pCol0, pCol1, pixelBytes);

    //-----------------------------------------------------------------------------
    // get count
    const int RLE_COUNT_MAX = 0x80;
    int length = 2;
    while (ix + length < imageW)
    {
        pCol0 += pixelBytes;
        pCol1 += pixelBytes;
        if (IsSameTgaColor(pCol0, pCol1, pixelBytes) != isSame)
        {
            if (!isSame) // same でないときに同じピクセルが現れたら 1 つ前までをランとします。
            {
                --length;
            }
            break;
        }
        if (length >= RLE_COUNT_MAX) // same でないときのためにここでチェックします。
        {
            break;
        }
        ++length;
    }
    return (isSame) ? length : -length;
}

//-----------------------------------------------------------------------------
//! @brief TGA ファイルのイメージデータを RLE 圧縮します。
//!
//! @param[out] pCompressed 圧縮後のデータを格納するバッファへのポインタです。
//! @param[in] pBitmap 圧縮前のデータを格納したバッファへのポインタです。
//!                    BGR(A) の順に格納しておきます。
//! @param[in] imageW 幅です。
//! @param[in] imageH 高さです。
//! @param[in] hasAlpha アルファ成分を持つなら true です。
//!
//! @return 圧縮後のサイズを返します。
//-----------------------------------------------------------------------------
size_t CompressTgaRle(
    uint8_t* pCompressed,
    const uint8_t* pBitmap,
    const int imageW,
    const int imageH,
    const bool hasAlpha
)
{
    const size_t pixelBytes = (hasAlpha) ? R_RGBA_BYTES : R_RGB_BYTES;
    const size_t pitch = pixelBytes * imageW; // 1 ラインあたりのバイト数です。
    uint8_t* pDst = pCompressed;
    for (int iy = 0; iy < imageH; ++iy)
    {
        //const int realrow = imageH - 1 - iy;
        const int realrow = iy;
        const uint8_t* pSrc = pBitmap + realrow * pitch;
        for (int ix = 0; ix < imageW; )
        {
            int length = GetTgaRunLength(pSrc, ix, imageW, pixelBytes);
            if (length > 0)
            {
                *pDst++ = static_cast<uint8_t>(0x80 + length - 1);
                memcpy(pDst, pSrc, pixelBytes);
                pDst += pixelBytes;
            }
            else if (length < 0)
            {
                length = -length;
                *pDst++ = static_cast<uint8_t>(length - 1);
                memcpy(pDst, pSrc, length * pixelBytes);
                pDst += length * pixelBytes;
            }
            pSrc += length * pixelBytes;
            ix += length;
        }
    }
    return pDst - pCompressed;
}

} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief TGA ファイルのデータをデコードします。
//-----------------------------------------------------------------------------
RStatus RImage::DecodeTga(const uint8_t* fileBuf, const size_t fileSize)
{
    //-----------------------------------------------------------------------------
    // check header
    const int imageType = fileBuf[0x02];
    if (imageType !=  1 &&  // CI
        imageType !=  2 &&  // RGB
        imageType !=  3 &&  // GRAY
        imageType !=  9 &&  // CI (RLE)
        imageType != 10 &&  // RGB (RLE)
        imageType != 11)    // GRAY (RLE)
    {
        return RStatus(RStatus::FAILURE, "TGA file is wrong: (invalid image type): " + m_FilePath); // RShowError
    }

    const int bitPerPixel = fileBuf[0x10];
    if (bitPerPixel !=  8 &&
        bitPerPixel != 15 &&
        bitPerPixel != 16 &&
        bitPerPixel != 24 &&
        bitPerPixel != 32)
    {
        return RStatus(RStatus::FAILURE, "TGA file is wrong: (invalid bit per pixel): " + m_FilePath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // get size
    m_ImageW = fileBuf[0x0c] + (fileBuf[0x0d] << 8);
    m_ImageH = fileBuf[0x0e] + (fileBuf[0x0f] << 8);

    //-----------------------------------------------------------------------------
    // get alpha flag
    if (fileBuf[0x01] != 0) // color palette flag
    {
        // CI
        m_HasAlpha = (fileBuf[0x07] == 32); // palette color bit size
    }
    else
    {
        // RGB or GRAY
        const int attrBitSize = fileBuf[0x11] & 0x0f;
        m_HasAlpha = (attrBitSize != 0);
        if ((imageType == 3 || imageType == 11) && // GRAY
            bitPerPixel == 8)
        {
            m_HasAlpha = false;
        }
    }

    //-----------------------------------------------------------------------------
    // decode bitmap
    m_ImageDataSize = static_cast<size_t>(R_RGBA_BYTES) * m_ImageW * m_ImageH;
    m_pImageData = new uint8_t[m_ImageDataSize];
    DecodeTgaBitmap(m_pImageData, fileBuf, m_ImageW, m_ImageH);

    R_UNUSED_VARIABLE(fileSize);

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief TGA ファイルを出力します。
//-----------------------------------------------------------------------------
RStatus RImage::OutputTgaFile(
    const std::string& filePath,
    const bool isRLE,
    const bool isCubeVC,
    const bool outputsMip
) const
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // TGA ファイルの幅と高さを決定します。
    int dstW = m_ImageW;
    int dstH = m_ImageH;
    if (IsCubeMap())
    {
        if (!isCubeVC)
        {
            dstW *= CUBE_HC_COUNT_H;
            dstH *= CUBE_HC_COUNT_V;
        }
        else
        {
            dstW *= CUBE_VC_COUNT_H;
            dstH *= CUBE_VC_COUNT_V;
        }
        dstH *= RMax(static_cast<int>(m_ImageD / CUBE_FACE_COUNT), 1);
    }
    else if (m_Dimension == FtxDimension_3d      ||
             m_Dimension == FtxDimension_1dArray ||
             m_Dimension == FtxDimension_2dArray)
    {
        dstH *= m_ImageD;
    }

    const int baseW = dstW;
    int outMipCount = (outputsMip) ? m_MipCount : 1;
    for (int level = 1; level < outMipCount; ++level)
    {
        dstW += RMax(baseW >> level, 1);
    }

    //-----------------------------------------------------------------------------
    // create bitmap
    const size_t pixelBytes = (m_HasAlpha) ? R_RGBA_BYTES : R_RGB_BYTES;
    const size_t bitmapSize = pixelBytes * dstW * dstH;
    uint8_t* pBitmap = new uint8_t[bitmapSize];
    memset(pBitmap, 0xff, bitmapSize);

    const uint8_t* pSrc = m_pImageData;
    int ofsIx = 0;
    for (int level = 0; level < outMipCount; ++level)
    {
        const int curW = RMax(m_ImageW >> level, 1);
        const int curH = RMax(m_ImageH >> level, 1);
        const int curD = (m_Dimension == FtxDimension_3d) ?
            RMax(m_ImageD >> level, 1) : m_ImageD;
        for (int iz = 0; iz < curD; ++iz)
        {
            int iFace = iz % CUBE_FACE_COUNT;
            int dstIx = 0;
            int dstIy = m_ImageH * iz;
            if (IsCubeMap())
            {
                // 画像のピクセルデータは +X, -X, +Y, -Y, -Z, +Z の順に格納されているので
                // -Z と +Z を入れ替えます。
                if (iFace == CUBE_FACE_PZ)
                {
                    iFace = CUBE_FACE_NZ;
                }
                else if (iFace == CUBE_FACE_NZ)
                {
                    iFace = CUBE_FACE_PZ;
                }
                const int* cubePoss = (!isCubeVC) ?
                    CubeHcPositions[iFace] : CubeVcPositions[iFace];
                dstIx = cubePoss[0] * curW;
                dstIy = cubePoss[1] * curH;
                dstIy += (iz / CUBE_FACE_COUNT) * m_ImageH * ((!isCubeVC) ? CUBE_HC_COUNT_V : CUBE_VC_COUNT_V);
            }
            uint8_t* pDstTop = pBitmap + (ofsIx + dstIx + dstIy * dstW) * pixelBytes;
            uint8_t* pDst = pDstTop;
            for (int iy = 0; iy < curH; ++iy)
            {
                for (int ix = 0; ix < curW; ++ix)
                {
                    // TGA のビットマップは BGRA の順に格納します。
                    *pDst++ = pSrc[2]; // B
                    *pDst++ = pSrc[1]; // G
                    *pDst++ = pSrc[0]; // R
                    if (m_HasAlpha)
                    {
                        *pDst++ = pSrc[3]; // A
                    }
                    pSrc += R_RGBA_BYTES;
                }
                pDst += (dstW - curW) * pixelBytes;
            }

            // 垂直十字キューブマップは +Z 方向を上下左右反転します。
            if (IsCubeMap() && isCubeVC && iFace == CUBE_FACE_PZ)
            {
                RFlipFaceHv(pDstTop, curW, curH, dstW * pixelBytes, pixelBytes);
            }
        }
        ofsIx += RMax(baseW >> level, 1);
    }

    //-----------------------------------------------------------------------------
    // イメージデータを上下反転します（Photoshop 標準 TGA とあわせるため）。
    RFlipFaceV(pBitmap, dstW, dstH, dstW * pixelBytes, pixelBytes);

    //-----------------------------------------------------------------------------
    // TGA ファイルを出力します。
    ofstream ofs(filePath.c_str(), ios_base::binary);
    if (!ofs)
    {
        delete[] pBitmap;
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + filePath); // RShowError
    }

    uint8_t header[TGA_HEADER_SIZE] =
    {
        0x00,       // id size
        0x00,       // palette flag
        static_cast<uint8_t>((isRLE) ? 0x0a : 0x02), // image type (0x02 = RGB, 0x0a = RLE RGB)
        0x00,       // color index min L
        0x00,       // color index min H
        0x00,       // palette color size L
        0x00,       // palette color size H
        0x00,       // palette color bit size
        0x00,       // x origin of image L
        0x00,       // x origin of image H
        0x00,       // y origin of image L
        0x00,       // y origin of image H
        static_cast<uint8_t>(dstW & 0xff), // width L
        static_cast<uint8_t>(dstW >> 8  ), // width H
        static_cast<uint8_t>(dstH & 0xff), // height L
        static_cast<uint8_t>(dstH >> 8  ), // height H
        static_cast<uint8_t>((m_HasAlpha) ? 0x20 : 0x18), // pixel bit size
        static_cast<uint8_t>((m_HasAlpha) ? 0x08 : 0x00), // descriptor (0x20 = upper to lower, 0x08 = 32bit RGBA)
    };
    ofs.write(reinterpret_cast<const char*>(header), sizeof(header));
    if (!isRLE)
    {
        ofs.write(reinterpret_cast<const char*>(pBitmap), bitmapSize);
    }
    else
    {
        // ランレングス圧縮します。
        // 最悪のケースでは圧縮前より 1/3 or 1/4 大きくなることを考慮します。
        uint8_t* pCompressed = new uint8_t[(pixelBytes + 1) * dstW * dstH];
        const size_t compressedSize = CompressTgaRle(pCompressed, pBitmap, dstW, dstH, m_HasAlpha);
        ofs.write(reinterpret_cast<const char*>(pCompressed), compressedSize);
        delete[] pCompressed;
    }

    //-----------------------------------------------------------------------------
    // free bitmap
    delete[] pBitmap;

    return status;
} // NOLINT(readability/fn_size)

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

