﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// variables
// Process ParseArgument MainProc
// ConvertOnMemoryTest ConvertBitmapTest

//=============================================================================
// include
//=============================================================================
#include "Common.h"
#include "TextureConverter.h"

using namespace std;
using namespace nn::gfx::tool::texcvtr;

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//=============================================================================
// constants
//=============================================================================
const char* DllName = "TextureConverter.dll"; //!< メインの DLL ファイル名です。
const char* ConverterName = "TextureConverter"; //!< コンバーター名です。

//=============================================================================
// variables
//=============================================================================

//-----------------------------------------------------------------------------
// dll functions
GetCvtrVersionFunc GetCvtrVersion;
ShowUsageFunc ShowUsage;
CheckGpuEncodingFunc CheckGpuEncoding;
SetComModeFunc SetComMode;
IsGlobalOptionArgFunc IsGlobalOptionArg;
IsNextArgGlobalOptionValueFunc IsNextArgGlobalOptionValue;
IsNextArgJobOptionValueFunc IsNextArgJobOptionValue;
SetOptionsFunc SetOptions;
ClearFunc Clear;
ReadInputFileFunc ReadInputFile;
ReadFtxDataFunc ReadFtxData;
ReadBitmapDataFunc ReadBitmapData;
ConvertToFileFunc ConvertToFile;
ConvertToDataFunc ConvertToData;
GetErrorStringFunc GetErrorString;
GetErrorShiftJisStringFunc GetErrorShiftJisString;

//=============================================================================
//! @brief DLL から関数を取得するためのマクロです。
//=============================================================================
#define PROC_ADDRESS(handle, name)                                        \
name = reinterpret_cast<name##Func>(GetProcAddress(handle, #name));       \
    if (name == nullptr)                                                  \
    {                                                                     \
        RShowErrorArgs("Cannot find function: %s (%s)", #name, DllName);  \
        return 1;                                                         \
    }                                                                     \

//-----------------------------------------------------------------------------
//! @brief エラーを表示します。
//-----------------------------------------------------------------------------
void RShowError(const char* message)
{
    cerr << ConverterName << ": Error: " << message << endl;
}

//-----------------------------------------------------------------------------
//! @brief エラーを表示します（printf 形式）。
//-----------------------------------------------------------------------------
void RShowErrorArgs(const char* format, ...)
{
    const int STR_SIZE = 1024;
    char* str = new char[STR_SIZE];
    va_list args;
    va_start(args, format);
    _vsnprintf_s(str, STR_SIZE, _TRUNCATE, format, args);
    va_end(args);
    RShowError(str);
    delete[] str;
}

//-----------------------------------------------------------------------------
//! @brief 変換処理を実行します。
//!
//! @param[in] convArg コンバーター引数です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool Process(const ConverterArgument& convArg)
{
    //-----------------------------------------------------------------------------
    // 必要になった際にコンバーター内で COM を初期化するように設定します。
    SetComMode(true, COINIT_APARTMENTTHREADED);

    //-----------------------------------------------------------------------------
    // グローバルオプションを設定します。
    if (!SetOptions(RUnicodeCArray(convArg.globalOptions).GetCArray()))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // ジョブについてループします。
    bool success = true;
    for (size_t jobIdx = 0; jobIdx < convArg.jobArgs.size(); ++jobIdx)
    {
        const JobArgument& jobArg = convArg.jobArgs[jobIdx];
        if (!ReadInputFile(
            RUnicodeCArray(jobArg.filePaths).GetCArray(),
            RUnicodeCArray(jobArg.options).GetCArray()))
        {
            success = false;
            break;
        }

        if (!ConvertToFile())
        {
            success = false;
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // テクスチャーコンバーターのメモリーをクリアします。
    Clear();

    return success;
}

//-----------------------------------------------------------------------------
//! @brief 引数を解析します。
//!
//! @param[out] pConvArg コンバーター引数へのポインタです。
//! @param[in] argc コマンドラインで指定された引数の数です。
//! @param[in] argv コマンドラインで指定された引数の配列です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool ParseArgument(ConverterArgument* pConvArg, int argc, char* argv[])
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // コマンド名を取得します。
    const std::wstring converterName = RGetUnicodeFromShiftJis(ConverterName);
    const std::wstring cmdName = RGetUnicodeFromShiftJis(RGetFileNameFromFilePath(argv[0]));

    //-----------------------------------------------------------------------------
    // 引数ファイルで指定された引数を展開します。
    RStringArray srcArgs;
    for (int argIdx = 1; argIdx < argc; ++argIdx)
    {
        srcArgs.push_back(std::string(argv[argIdx]));
    }
    RStringArray args;
    status = ExpandArguments(&args, srcArgs);
    if (!status)
    {
        RShowError(status.GetMessage().c_str());
        return false;
    }

    //-----------------------------------------------------------------------------
    // 引数を解析します。
    ConverterArgument& convArg = *pConvArg;
    JobArgument commonJobArg;
    RStringArray jobListFilePaths;
    for (int argIdx = 0; argIdx < static_cast<int>(args.size()); ++argIdx)
    {
        const std::string& arg = args[argIdx];
        const std::wstring argW = RGetUnicodeFromShiftJis(arg);
        if (arg == "--version")
        {
            ShowUsage(cout, converterName.c_str(), cmdName.c_str(), true);
            exit(0);
        }
        else if (arg == "-h" || arg == "--help")
        {
            ShowUsage(cout, converterName.c_str(), cmdName.c_str(), false);
            exit(0);
        }
        else if (arg == "--job-list" || arg.find("--job-list=") == 0 ||
            arg == "--convert-list" || arg.find("--convert-list=") == 0)
        {
            std::string jobListFilePath;
            status = GetOptionArgValue(&jobListFilePath, &argIdx, args);
            if (status)
            {
                jobListFilePaths.push_back(jobListFilePath);
            }
        }
        else if (arg == "--check-gpu-encoding")
        {
            exit(CheckGpuEncoding() ? 0 : 1);
        }
        else if (arg.find('-') == 0)
        {
            if (IsGlobalOptionArg(argW.c_str()))
            {
                convArg.globalOptions.push_back(arg);
                if (IsNextArgGlobalOptionValue(argW.c_str()))
                {
                    ++argIdx;
                    if (argIdx < static_cast<int>(args.size()))
                    {
                        convArg.globalOptions.push_back(args[argIdx]);
                    }
                    else
                    {
                        status = RStatus(RStatus::FAILURE, "Option has no value: " + arg); // RShowError
                    }
                }
            }
            else
            {
                commonJobArg.options.push_back(arg);
                if (IsNextArgJobOptionValue(argW.c_str()))
                {
                    ++argIdx;
                    if (argIdx < static_cast<int>(args.size()))
                    {
                        commonJobArg.options.push_back(args[argIdx]);
                    }
                    else
                    {
                        status = RStatus(RStatus::FAILURE, "Option has no value: " + arg); // RShowError
                    }
                }
            }
        }
        else
        {
            const std::string inputPath = RDequoteString(arg);
            commonJobArg.filePaths.push_back(RGetFullFilePath(inputPath, true));
        }

        if (!status)
        {
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // ジョブリストファイル群を解析します。
    if (status)
    {
        if (!jobListFilePaths.empty())
        {
            for (size_t jobIdx = 0; jobIdx < jobListFilePaths.size(); ++jobIdx)
            {
                status = ParseJobListFile(&convArg, commonJobArg, jobListFilePaths[jobIdx],
                    IsGlobalOptionArg, IsNextArgJobOptionValue);
                if (!status)
                {
                    break;
                }
            }
            if (status && convArg.jobArgs.empty())
            {
                status = RStatus(RStatus::FAILURE, "Job list is empty"); // RShowError
            }
        }
        else if (!commonJobArg.filePaths.empty())
        {
            convArg.jobArgs.push_back(commonJobArg);
        }
    }

    if (!status)
    {
        RShowError(status.GetMessage().c_str());
        return false;
    }

    if (convArg.jobArgs.empty())
    {
        ShowUsage(cout, converterName.c_str(), cmdName.c_str(), false);
        return false;
    }

    return true;
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 構造化例外の変換関数です。
//-----------------------------------------------------------------------------
void TranslateSe(unsigned int code, struct _EXCEPTION_POINTERS* ep)
{
    throw ep; // 標準 C++ の例外を発生させます。
    R_UNUSED_VARIABLE(code);
}

//-----------------------------------------------------------------------------
//! @brief メイン関数です。
//-----------------------------------------------------------------------------
int main(int argc, char* argv[]) // MainProc
{
    //-----------------------------------------------------------------------------
    // 構造化例外の変換関数を設定します。
    _set_se_translator(TranslateSe); // このスレッドでのみ有効です。

    //-----------------------------------------------------------------------------
    // DLL をロードします。
    const std::string exeFolderPath = RGetFolderFromFilePath(RGetModuleFileName(nullptr)) + "\\";
    const std::string dllPath = exeFolderPath + DllName;
    //cerr << "dll: " << dllPath << endl;
    HINSTANCE hDLL = LoadLibrary(dllPath.c_str());
    if (hDLL == nullptr)
    {
        RShowErrorArgs("Cannot load dll: %s (%s)", dllPath.c_str(), GetWindowsLastErrorMessage().c_str());
        return 1;
    }

    //-----------------------------------------------------------------------------
    // DLL 解放用のオブジェクトを作成します。
    struct Deleter
    {
        HINSTANCE m_hDLL;
        explicit Deleter(HINSTANCE hDLL) : m_hDLL(hDLL) {}
        ~Deleter()
        {
            BOOL ret = FreeLibrary(m_hDLL);
            if (!ret)
            {
                RShowErrorArgs("Cannot release dll: %s", DllName);
            }
        }
    } deleter(hDLL);

    //-----------------------------------------------------------------------------
    // DLL の関数を取得します。
    PROC_ADDRESS(hDLL, GetCvtrVersion);
    PROC_ADDRESS(hDLL, ShowUsage);
    PROC_ADDRESS(hDLL, CheckGpuEncoding);
    PROC_ADDRESS(hDLL, SetComMode);
    PROC_ADDRESS(hDLL, IsGlobalOptionArg);
    PROC_ADDRESS(hDLL, IsNextArgGlobalOptionValue);
    PROC_ADDRESS(hDLL, IsNextArgJobOptionValue);
    PROC_ADDRESS(hDLL, SetOptions);
    PROC_ADDRESS(hDLL, Clear);
    PROC_ADDRESS(hDLL, ReadInputFile);
    PROC_ADDRESS(hDLL, ReadFtxData);
    PROC_ADDRESS(hDLL, ReadBitmapData);
    PROC_ADDRESS(hDLL, ConvertToFile);
    PROC_ADDRESS(hDLL, ConvertToData);
    PROC_ADDRESS(hDLL, GetErrorString);
    PROC_ADDRESS(hDLL, GetErrorShiftJisString);

    //-----------------------------------------------------------------------------
    // parse argument
    ConverterArgument convArg;
    if (!ParseArgument(&convArg, argc, argv))
    {
        return 1;
    }

    //-----------------------------------------------------------------------------
    // process
    if (!Process(convArg))
    {
        const std::string msg = GetErrorShiftJisString();
        if (!msg.empty())
        {
            RShowError(msg.c_str());
        }
        return 1;
    }
    return 0;
}

