﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//=============================================================================
// include
//=============================================================================
#include "REncoder.h"

using namespace std;

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//=============================================================================
//! @brief DLL から関数を取得するためのマクロです。
//=============================================================================
#define PROC_ADDRESS(handle, name)                                   \
*reinterpret_cast<void**>(&name) = GetProcAddress(handle, #name);    \
    if (name == nullptr)                                             \
    {                                                                \
        cerr << "Error: Cannot find function: " << #name << endl;    \
    }                                                                \

//-----------------------------------------------------------------------------
//! @brief 初期化します。
//-----------------------------------------------------------------------------
RStatus REncoder::Initialize(const char* dllPath)
{
    //-----------------------------------------------------------------------------
    // 初期化済みなら終了処理をします。
    if (IsInitialized())
    {
        Finalize();
    }

    //-----------------------------------------------------------------------------
    // DLL ファイルのパスを設定します。
    if (dllPath != nullptr)
    {
        m_DllPath = dllPath;
    }
    else
    {
        dllPath = m_DllPath.c_str();
    }
    //cerr << "REncoder: initialize: " << dllPath << endl;

    //-----------------------------------------------------------------------------
    // DLL をロードします。
    // エンコーダの DLL ファイルと同じフォルダに存在する PVRTexLib.dll を
    // 優先的にロードするため LOAD_WITH_ALTERED_SEARCH_PATH を指定します。
    m_hDll = LoadLibraryEx(dllPath, nullptr, LOAD_WITH_ALTERED_SEARCH_PATH);
    if (m_hDll == nullptr)
    {
        return RStatus(RStatus::FAILURE, "Cannot load dll: " + std::string(dllPath) + // RShowError
            " (" + GetWindowsLastErrorMessage() + ")");
    }

    //-----------------------------------------------------------------------------
    // 関数アドレスを取得します。
    PROC_ADDRESS(m_hDll, GetDataSize);
    PROC_ADDRESS(m_hDll, ConvertFormat);
    PROC_ADDRESS(m_hDll, IsGpuEncodingAvailable);

    if (GetDataSize            == nullptr ||
        ConvertFormat          == nullptr ||
        IsGpuEncodingAvailable == nullptr)
    {
        Finalize();
        return RStatus(RStatus::FAILURE, "Cannot find function: " + std::string(dllPath)); // RShowError
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 終了します。
//-----------------------------------------------------------------------------
void REncoder::Finalize()
{
    //cerr << "REncoder: finalize: " << m_hDll << endl;

    //-----------------------------------------------------------------------------
    // DLL を解放します。
    if (m_hDll != nullptr)
    {
        FreeLibrary(m_hDll);
        m_hDll = nullptr;
    }

    //-----------------------------------------------------------------------------
    // 関数アドレスをクリアします。
    GetDataSize            = nullptr;
    ConvertFormat          = nullptr;
    IsGpuEncodingAvailable = nullptr;
}

//-----------------------------------------------------------------------------
//! @brief エンコーダ用のイメージの次元を取得します（static 関数）。
//-----------------------------------------------------------------------------
texenc::ImageDimension REncoder::GetImageDimension(const FtxDimension dimension)
{
    switch (dimension)
    {
    case FtxDimension_1d:                 return texenc::ImageDimension_1d;
    case FtxDimension_2d:                 return texenc::ImageDimension_2d;
    case FtxDimension_3d:                 return texenc::ImageDimension_3d;
    case FtxDimension_CubeMap:            return texenc::ImageDimension_CubeMap;
    case FtxDimension_1dArray:            return texenc::ImageDimension_1dArray;
    case FtxDimension_2dArray:            return texenc::ImageDimension_2dArray;
    case FtxDimension_2dMultisample:      return texenc::ImageDimension_2dMultisample;
    case FtxDimension_2dMultisampleArray: return texenc::ImageDimension_2dMultisampleArray;
    case FtxDimension_CubeMapArray:       return texenc::ImageDimension_CubeMapArray;
    default:                              return texenc::ImageDimension_2d;
    }
}

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

