﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// RGetFloat16Value
// RFlipFaceH RCropFace RResizeFreeFace RResizeFree3d
// RAdjustTransparentRgbFace

// ResizeFaceByLinear ResizeFaceByCubic

//=============================================================================
// include
//=============================================================================
#include "ImageEdit.h"

using namespace std;

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//-----------------------------------------------------------------------------
//! @brief float 型の成分値を T 型に変換します（単純なキャスト）。
//!
//! @return T 型に変換した値を返します。
//-----------------------------------------------------------------------------
template <typename T>
T ConvertFloatCompValue(const float v)
{
    return static_cast<T>(v);
}

//-----------------------------------------------------------------------------
//! @brief float 型の成分値を uint8_t 型に変換します（四捨五入して uint8_t の範囲に抑えます）。
//!
//! @return uint8_t 型に変換した値を返します。
//-----------------------------------------------------------------------------
template <>
uint8_t ConvertFloatCompValue<uint8_t>(const float v)
{
    return static_cast<uint8_t>(RClampValue(0x00, 0xff, RRound(v)));
}

//-----------------------------------------------------------------------------
//! @brief float 型の成分値をクランプして T 型に変換します。
//!
//! @param[in] clamps01 [0,1] にクランプするなら true、クランプしないなら false を指定します。
//!
//! @return T 型に変換した値を返します。
//-----------------------------------------------------------------------------
template <typename T>
T ClampFloatCompValue(const float v, const bool clamps01)
{
    return static_cast<T>((clamps01) ? RClampValue(0.0f, 1.0f, v) : v);
}

//-----------------------------------------------------------------------------
//! @brief float 型の成分値をクランプして uint8_t 型に変換します。
//!
//! @param[in] clamps01 未使用です。常に [0,255] にクランプします。
//!
//! @return uint8_t 型に変換した値を返します。
//-----------------------------------------------------------------------------
template <>
uint8_t ClampFloatCompValue(const float v, const bool clamps01)
{
    R_UNUSED_VARIABLE(clamps01);
    return static_cast<uint8_t>(RClampValue(0x00, 0xff, RRound(v)));
}

//-----------------------------------------------------------------------------
// float_16 変換用テーブルです。
//-----------------------------------------------------------------------------
const uint16_t Float16ELut[1 << 9] =
{
    // 以下、OpenEXR の eLut.h からコピーしたデータです。

        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,  1024,  2048,  3072,  4096,  5120,  6144,  7168,
     8192,  9216, 10240, 11264, 12288, 13312, 14336, 15360,
    16384, 17408, 18432, 19456, 20480, 21504, 22528, 23552,
    24576, 25600, 26624, 27648, 28672, 29696,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0, 33792, 34816, 35840, 36864, 37888, 38912, 39936,
    40960, 41984, 43008, 44032, 45056, 46080, 47104, 48128,
    49152, 50176, 51200, 52224, 53248, 54272, 55296, 56320,
    57344, 58368, 59392, 60416, 61440, 62464,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
        0,     0,     0,     0,     0,     0,     0,     0,
};

//-----------------------------------------------------------------------------
//! @brief float 型を int 型としてリードした値を float_16 の値に変換します。
//-----------------------------------------------------------------------------
uint16_t ConvertToFloat16Value(const int i)
{
    // 以下、OpenEXR の half.cpp の convert() からコピーしたコードです。

    int s =  (i >> 16) & 0x00008000;
    int e = ((i >> 23) & 0x000000ff) - (127 - 15);
    int m =   i        & 0x007fffff;
    if (e <= 0)
    {
        if (e < -10)
        {
            return static_cast<uint16_t>(s);
        }

        m = m | 0x00800000;

        int t = 14 - e;
        int a = (1 << (t - 1)) - 1;
        int b = (m >> t) & 1;

        m = (m + a + b) >> t;
        return static_cast<uint16_t>(s | m);
    }
    else if (e == 0xff - (127 - 15))
    {
        if (m == 0)
        {
            return static_cast<uint16_t>(s | 0x7c00);
        }
        else
        {
            m >>= 13;
            return static_cast<uint16_t>(s | 0x7c00 | m | (m == 0));
        }
    }
    else
    {
        m = m + 0x00000fff + ((m >> 13) & 1);

        if (m & 0x00800000)
        {
            m =  0;
            e += 1;
        }

        if (e > 30)
        {
            // overflow
            return static_cast<uint16_t>(s | 0x7c00);
        }

        return static_cast<uint16_t>(s | (e << 10) | (m >> 13));
    }
}

//-----------------------------------------------------------------------------
//! @brief 全ピクセルの RGBA 成分値が [0,1] の範囲内なら true を返します。
//!
//! @param[in] pSrc ビットマップデータです。
//! @param[in] pixCount ピクセル数です。
//!
//! @return 全ピクセルの RGBA 成分値が [0,1] の範囲内なら true を返します。
//-----------------------------------------------------------------------------
template <typename T>
bool IsAllPixelsInRange01(const T* pSrc, const size_t pixCount)
{
    const size_t valCount = pixCount * R_RGBA_COUNT;
    for (size_t iVal = 0; iVal < valCount; ++iVal)
    {
        const T val = *pSrc++;
        if (val < 0 || val > 1)
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 4 ピクセルのアルファがすべて 0 なら true を返します。
//!
//! @param[in] pSrc0 ピクセル 0 の R 成分へのポインタです。
//! @param[in] pSrc1 ピクセル 1 の R 成分へのポインタです。
//! @param[in] pSrc2 ピクセル 2 の R 成分へのポインタです。
//! @param[in] pSrc3 ピクセル 3 の R 成分へのポインタです。
//!
//! @return 4 ピクセルのアルファがすべて 0 なら true を返します。
//-----------------------------------------------------------------------------
template <typename T>
inline bool IsAllTransparent(
    const T* pSrc0,
    const T* pSrc1,
    const T* pSrc2,
    const T* pSrc3
)
{
    return (pSrc0[3] == 0 && pSrc1[3] == 0 && pSrc2[3] == 0 && pSrc3[3] == 0);
}

//-----------------------------------------------------------------------------
//! @brief サンプリング位置が透明ピクセル上なら true を返します。
//!
//! @param[in] pS0 ピクセル 0 の R 成分へのポインタです。
//! @param[in] pS1 ピクセル 1 の R 成分へのポインタです。
//! @param[in] r サンプリング位置です（[0,1]）。
//!
//! @return サンプリング位置が透明ピクセル上なら true を返します。
//-----------------------------------------------------------------------------
template <typename T>
inline bool IsOnTransparent(const T* pS0, const T* pS1, const float r)
{
    return
        (pS0[3] == 0 && (r < 0.5f || pS1[3] == 0)) ||
        (pS1[3] == 0 && (r > 0.5f               ));
}

//-----------------------------------------------------------------------------
//! @brief サンプリング位置が透明ピクセル上なら true を返します。
//!
//! @param[in] pS00 左上のピクセルの R 成分へのポインタです。
//! @param[in] pS10 右上のピクセルの R 成分へのポインタです。
//! @param[in] pS01 左下のピクセルの R 成分へのポインタです。
//! @param[in] pS11 右下のピクセルの R 成分へのポインタです。
//! @param[in] rx 水平方向のサンプリング位置です（[0,1]）。
//! @param[in] ry 垂直方向のサンプリング位置です（[0,1]）。
//!
//! @return サンプリング位置が透明ピクセル上なら true を返します。
//-----------------------------------------------------------------------------
template <typename T>
inline bool IsOnTransparent(
    const T* pS00,
    const T* pS10,
    const T* pS01,
    const T* pS11,
    const float rx,
    const float ry
)
{
    if (rx < 0.5f)
    {
        return IsOnTransparent(pS00, pS01, ry);
    }
    else if (rx > 0.5f)
    {
        return IsOnTransparent(pS10, pS11, ry);
    }
    else // rx == 0.5f
    {
        return
            (ry <  0.5f && pS00[3] == 0 && pS10[3] == 0) ||
            (ry >  0.5f && pS01[3] == 0 && pS11[3] == 0) ||
            (ry == 0.5f && IsAllTransparent(pS00, pS10, pS01, pS11));
    }
}

//-----------------------------------------------------------------------------
//! @brief サンプリング位置がウェイト 0 のピクセル上なら true を返します。
//!
//! @param[in] w0 ピクセル 0 のウェイトです。
//! @param[in] w1 ピクセル 1 のウェイトです。
//! @param[in] r サンプリング位置です（[0,1]）。
//!
//! @return サンプリング位置がウェイト 0 のピクセル上なら true を返します。
//-----------------------------------------------------------------------------
inline bool IsOnZeroWeight(const float w0, const float w1, const float r)
{
    return
        (w0 == 0.0f && (r < 0.5f || w1 == 0.0f)) ||
        (w1 == 0.0f && (r > 0.5f              ));
}

//-----------------------------------------------------------------------------
//! @brief フェースを任意の幅と高さにポイントサンプリングでサイズ変更します。
//!
//! @param[out] pDst サイズ変更後のビットマップデータを格納します。
//! @param[in] pSrc サイズ変更前のビットマップデータです。
//! @param[in] dstW サイズ変更後の幅です。
//! @param[in] dstH サイズ変更後の高さです。
//! @param[in] srcW サイズ変更前の幅です。
//! @param[in] srcH サイズ変更前の高さです。
//-----------------------------------------------------------------------------
template <typename T>
void ResizeFaceByPoint(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int srcW,
    const int srcH
)
{
    const float invScaleX = static_cast<float>(srcW) / dstW;
    const float invScaleY = static_cast<float>(srcH) / dstH;
    for (int iy = 0; iy < dstH; ++iy)
    {
        const float fy = RMax((iy + 0.5f) * invScaleY, 0.0f);
        const int y0 = RMin(static_cast<int>(fy), srcH - 1);
        const int y0Ofs = y0 * srcW;
        for (int ix = 0; ix < dstW; ++ix)
        {
            const float fx = RMax((ix + 0.5f) * invScaleX, 0.0f);
            const int x0 = RMin(static_cast<int>(fx), srcW - 1);
            const T* pS00 = pSrc + (x0 + y0Ofs) * R_RGBA_COUNT;
            for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
            {
                *pDst++ = *pS00++;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief フェースを任意の幅と高さにポイントサンプリングでサイズ変更します。
//!
//! @param[out] pDst サイズ変更後のビットマップデータを格納します。
//! @param[in] pSrc サイズ変更前のビットマップデータです。
//! @param[in] dstW サイズ変更後の幅です。
//! @param[in] dstH サイズ変更後の高さです。
//! @param[in] dstD サイズ変更後の奥行きです。
//! @param[in] srcW サイズ変更前の幅です。
//! @param[in] srcH サイズ変更前の高さです。
//! @param[in] srcD サイズ変更前の奥行きです。
//-----------------------------------------------------------------------------
template <typename T>
void Resize3dByPoint(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int dstD,
    const int srcW,
    const int srcH,
    const int srcD
)
{
    const float invScaleX = static_cast<float>(srcW) / dstW;
    const float invScaleY = static_cast<float>(srcH) / dstH;
    const float invScaleZ = static_cast<float>(srcD) / dstD;
    const size_t srcFacePixCount = srcW * srcH;
    for (int iz = 0; iz < dstD; ++iz)
    {
        const float fz = RMax((iz + 0.5f) * invScaleZ, 0.0f);
        const int z0 = RMin(static_cast<int>(fz), srcD - 1);
        const size_t z0Ofs = srcFacePixCount * z0;
        for (int iy = 0; iy < dstH; ++iy)
        {
            const float fy = RMax((iy + 0.5f) * invScaleY, 0.0f);
            const int y0 = RMin(static_cast<int>(fy), srcH - 1);
            const int y0Ofs = y0 * srcW;
            for (int ix = 0; ix < dstW; ++ix)
            {
                const float fx = RMax((ix + 0.5f) * invScaleX, 0.0f);
                const int x0 = RMin(static_cast<int>(fx), srcW - 1);
                const T* pS00 = pSrc + (x0 + y0Ofs + z0Ofs) * R_RGBA_COUNT;
                for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                {
                    *pDst++ = *pS00++;
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief フェースを任意の幅と高さに線形補間でサイズ変更します。
//!
//! @param[out] pDst サイズ変更後のビットマップデータを格納します。
//! @param[in] pSrc サイズ変更前のビットマップデータです。
//! @param[in] dstW サイズ変更後の幅です。
//! @param[in] dstH サイズ変更後の高さです。
//! @param[in] srcW サイズ変更前の幅です。
//! @param[in] srcH サイズ変更前の高さです。
//! @param[in] ignoresXpaRgb 透明ピクセルの RGB 成分を補間計算から除外するなら true です。
//-----------------------------------------------------------------------------
template <typename T>
void ResizeFaceByLinear(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int srcW,
    const int srcH,
    const bool ignoresXpaRgb
)
{
    const float invScaleX = static_cast<float>(srcW) / dstW;
    const float invScaleY = static_cast<float>(srcH) / dstH;
    const bool isInterpX = (invScaleX <= 2.0f);
    const bool isInterpY = (invScaleY <= 2.0f);
    if (isInterpX && isInterpY)
    {
        //-----------------------------------------------------------------------------
        // 幅と高さのスケールが共に 0.5 以上の場合は単純な線形補間で計算します。
        for (int iy = 0; iy < dstH; ++iy)
        {
            const float fy = RMax((iy + 0.5f) * invScaleY - 0.5f, 0.0f);
            const int y0 = RMin(static_cast<int>(fy), srcH - 1);
            const int y1 = RMin(y0 + 1, srcH - 1);
            const int y0Ofs = y0 * srcW;
            const int y1Ofs = y1 * srcW;
            const float ry = fy - static_cast<float>(y0);
            const float oneMinusRy = 1.0f - ry;
            for (int ix = 0; ix < dstW; ++ix)
            {
                const float fx = RMax((ix + 0.5f) * invScaleX - 0.5f, 0.0f);
                const int x0 = RMin(static_cast<int>(fx), srcW - 1);
                const int x1 = RMin(x0 + 1, srcW - 1);
                const float rx = fx - static_cast<float>(x0);
                const float oneMinusRx = 1.0f - rx;
                const T* pS00 = pSrc + (x0 + y0Ofs) * R_RGBA_COUNT;
                const T* pS10 = pSrc + (x1 + y0Ofs) * R_RGBA_COUNT;
                const T* pS01 = pSrc + (x0 + y1Ofs) * R_RGBA_COUNT;
                const T* pS11 = pSrc + (x1 + y1Ofs) * R_RGBA_COUNT;
                if (!ignoresXpaRgb || IsAllTransparent(pS00, pS10, pS01, pS11))
                {
                    //-----------------------------------------------------------------------------
                    // RGBA 成分を単純に計算
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        const float v =
                            oneMinusRy * (oneMinusRx * (*pS00++) + rx * (*pS10++)) +
                                    ry * (oneMinusRx * (*pS01++) + rx * (*pS11++));
                        *pDst++ = ConvertFloatCompValue<T>(v);
                    }
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // 透明ピクセルの RGB 成分を計算から除外
                    const float w00 = (pS00[3] != 0) ? oneMinusRy * oneMinusRx : 0.0f;
                    const float w10 = (pS10[3] != 0) ? oneMinusRy *         rx : 0.0f;
                    const float w01 = (pS01[3] != 0) ?         ry * oneMinusRx : 0.0f;
                    const float w11 = (pS11[3] != 0) ?         ry *         rx : 0.0f;
                    const float wsum = w00 + w10 + w01 + w11;
                    const float wsumInv = (wsum != 0.0f) ? 1.0f / wsum : 0.0f;
                    const bool isZeroA = IsOnTransparent(pS00, pS10, pS01, pS11, rx, ry);
                    //cerr << "rfbl: " << ix << "," << iy << ": " << x0 << "," << y0 << ": " << rx << "," << ry << ": " << w00 << "," << w10 << "," << w01 << "," << w11 << endl;
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        float v = wsumInv *
                            (w00 * (*pS00++) + w10 * (*pS10++) + w01 * (*pS01++) + w11 * (*pS11++));
                        if (iRgba == 3 && isZeroA)
                        {
                            v = 0.0f;
                        }
                        *pDst++ = ConvertFloatCompValue<T>(v);
                    }
                }
            }
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // 幅または高さのスケールが 0.5 未満の場合、水平／垂直方向の数ピクセルを平均します。
        for (int iy = 0; iy < dstH; ++iy)
        {
            const float fy = RMax(0.0f, (isInterpY) ?
                (iy + 0.5f) * invScaleY - 0.5f :
                iy * invScaleY);
            const int y0 = RMin(srcH - 1, static_cast<int>(fy));
            const int y1 = RMin(srcH - 1, (isInterpY) ?
                y0 + 1 :
                static_cast<int>((iy + 1) * invScaleY) - 1);
            const int blockH = y1 - y0 + 1;
            const int y0Ofs = y0 * srcW;
            const int y1Ofs = y1 * srcW;
            const float ry = fy - static_cast<float>(y0);
            const float oneMinusRy = 1.0f - ry;
            for (int ix = 0; ix < dstW; ++ix)
            {
                const float fx = RMax(0.0f, (isInterpX) ?
                    (ix + 0.5f) * invScaleX - 0.5f :
                    ix * invScaleX);
                const int x0 = RMin(srcW - 1, static_cast<int>(fx));
                const int x1 = RMin(srcW - 1, (isInterpX) ?
                    x0 + 1 :
                    static_cast<int>((ix + 1) * invScaleX) - 1);
                const int blockW = x1 - x0 + 1;
                const int blockPixCount = blockW * blockH;
                const float rx = fx - static_cast<float>(x0);
                const float oneMinusRx = 1.0f - rx;
                const T* pS00 = pSrc + (x0 + y0Ofs) * R_RGBA_COUNT;
                const T* pS10 = pSrc + (x1 + y0Ofs) * R_RGBA_COUNT;
                const T* pS01 = pSrc + (x0 + y1Ofs) * R_RGBA_COUNT;
                const float r0 = (isInterpY) ? oneMinusRy / blockW : oneMinusRx / blockH;
                const float r1 = (isInterpY) ?         ry / blockW :         rx / blockH;
                if (!ignoresXpaRgb)
                {
                    //-----------------------------------------------------------------------------
                    // RGBA 成分を単純に計算
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        float v = 0.0f;
                        if (!isInterpX && !isInterpY) // 水平／垂直方向ともに平均
                        {
                            const T* pS = pS00;
                            for (int by = 0; by < blockH; ++by)
                            {
                                for (int bx = 0; bx < blockW; ++bx)
                                {
                                    v += *pS;
                                    pS += R_RGBA_COUNT;
                                }
                                pS += (srcW - blockW) * R_RGBA_COUNT;
                            }
                            v /= blockPixCount;
                        }
                        else
                        {
                            float v0 = 0.0f;
                            float v1 = 0.0f;
                            if (isInterpY) // 水平方向は平均、垂直方向は線形補間
                            {
                                const T* pRow0 = pS00;
                                const T* pRow1 = pS01;
                                for (int bx = 0; bx < blockW; ++bx)
                                {
                                    v0 += *pRow0;
                                    v1 += *pRow1;
                                    pRow0 += R_RGBA_COUNT;
                                    pRow1 += R_RGBA_COUNT;
                                }
                            }
                            else // 水平方向は線形補間、垂直方向は平均
                            {
                                const T* pCol0 = pS00;
                                const T* pCol1 = pS10;
                                for (int by = 0; by < blockH; ++by)
                                {
                                    v0 += *pCol0;
                                    v1 += *pCol1;
                                    pCol0 += srcW * R_RGBA_COUNT;
                                    pCol1 += srcW * R_RGBA_COUNT;
                                }
                            }
                            v = r0 * v0 + r1 * v1;
                        }
                        *pDst++ = ConvertFloatCompValue<T>(v);
                        ++pS00;
                        ++pS10;
                        ++pS01;
                    }
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // 透明ピクセルの RGB 成分を計算から除外
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        float v = 0.0f;
                        if (!isInterpX && !isInterpY) // 水平／垂直方向ともに平均
                        {
                            const T* pS = pS00;
                            float wv = 0.0f;
                            float wsum = 0.0f;
                            for (int by = 0; by < blockH; ++by)
                            {
                                for (int bx = 0; bx < blockW; ++bx)
                                {
                                    v += *pS;
                                    const float w = (pS[3 - iRgba] != 0) ? 1.0f : 0.0f;
                                    wv += w * (*pS);
                                    wsum += w;
                                    pS += R_RGBA_COUNT;
                                }
                                pS += (srcW - blockW) * R_RGBA_COUNT;
                            }
                            v = (wsum != 0.0f) ? wv / wsum : v / blockPixCount;
                        }
                        else
                        {
                            float v0 = 0.0f;
                            float v1 = 0.0f;
                            float wv0 = 0.0f;
                            float wv1 = 0.0f;
                            float wsum0 = 0.0f;
                            float wsum1 = 0.0f;
                            bool isZeroA = false;
                            if (isInterpY) // 水平方向は平均、垂直方向は線形補間
                            {
                                const T* pRow0 = pS00;
                                const T* pRow1 = pS01;
                                for (int bx = 0; bx < blockW; ++bx)
                                {
                                    v0 += *pRow0;
                                    v1 += *pRow1;
                                    const float w0 = (pRow0[3 - iRgba] != 0) ? 1.0f : 0.0f;
                                    const float w1 = (pRow1[3 - iRgba] != 0) ? 1.0f : 0.0f;
                                    wv0 += w0 * (*pRow0);
                                    wv1 += w1 * (*pRow1);
                                    wsum0 += w0;
                                    wsum1 += w1;
                                    pRow0 += R_RGBA_COUNT;
                                    pRow1 += R_RGBA_COUNT;
                                }
                                isZeroA = (iRgba == 3 && IsOnZeroWeight(wsum0, wsum1, ry));
                            }
                            else // 水平方向は線形補間、垂直方向は平均
                            {
                                const T* pCol0 = pS00;
                                const T* pCol1 = pS10;
                                for (int by = 0; by < blockH; ++by)
                                {
                                    v0 += *pCol0;
                                    v1 += *pCol1;
                                    const float w0 = (pCol0[3 - iRgba] != 0) ? 1.0f : 0.0f;
                                    const float w1 = (pCol1[3 - iRgba] != 0) ? 1.0f : 0.0f;
                                    wv0 += w0 * (*pCol0);
                                    wv1 += w1 * (*pCol1);
                                    wsum0 += w0;
                                    wsum1 += w1;
                                    pCol0 += srcW * R_RGBA_COUNT;
                                    pCol1 += srcW * R_RGBA_COUNT;
                                }
                                isZeroA = (iRgba == 3 && IsOnZeroWeight(wsum0, wsum1, rx));
                            }

                            if (isZeroA)
                            {
                                v = 0.0f; // サンプリングする位置が透明ピクセル上なら A 成分を強制的に 0 にします。
                            }
                            else if (wsum0 == 0.0f && wsum1 == 0.0f)
                            {
                                v = r0 * v0 + r1 * v1; // すべて透明ピクセルなら単純に計算します。
                            }
                            else
                            {
                                wv0 = (wsum0 != 0.0f) ? wv0 / wsum0 : wv0;
                                wv1 = (wsum1 != 0.0f) ? wv1 / wsum1 : wv1;
                                const float wr0 = (wsum0 != 0.0f) ? ((isInterpY) ? oneMinusRy : oneMinusRx) : 0.0f;
                                const float wr1 = (wsum1 != 0.0f) ? ((isInterpY) ?         ry :         rx) : 0.0f;
                                const float wrInv = (wr0 + wr1 != 0.0f) ? 1.0f / (wr0 + wr1) : 1.0f;
                                v = (wr0 * wv0 + wr1 * wv1) * wrInv;
                            }
                        }
                        *pDst++ = ConvertFloatCompValue<T>(v);
                        ++pS00;
                        ++pS10;
                        ++pS01;
                    }
                }
            }
        }
    }
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief 3D テクスチャを任意のサイズに線形補間でサイズ変更します。
//!
//! @param[out] pDst サイズ変更後のビットマップデータを格納します。
//! @param[in] pSrc サイズ変更前のビットマップデータです。
//! @param[in] dstW サイズ変更後の幅です。
//! @param[in] dstH サイズ変更後の高さです。
//! @param[in] dstD サイズ変更後の奥行きです。
//! @param[in] srcW サイズ変更前の幅です。
//! @param[in] srcH サイズ変更前の高さです。
//! @param[in] srcD サイズ変更前の奥行きです。
//! @param[in] ignoresXpaRgb 透明ピクセルの RGB 成分を補間計算から除外するなら true です。
//-----------------------------------------------------------------------------
template <typename T>
void Resize3dByLinear(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int dstD,
    const int srcW,
    const int srcH,
    const int srcD,
    const bool ignoresXpaRgb
)
{
    //-----------------------------------------------------------------------------
    // 幅と高さをサイズ変更します。
    const size_t srcFacePixCount = srcW * srcH;
    const size_t dstFacePixCount = dstW * dstH;
    const size_t srcFaceCompCount = srcFacePixCount * R_RGBA_COUNT;
    const size_t dstFaceCompCount = dstFacePixCount * R_RGBA_COUNT;
    T* pReduced2d = new T[dstFaceCompCount * srcD];
    for (int iz = 0; iz < srcD; ++iz)
    {
        ResizeFaceByLinear(
            pReduced2d + dstFaceCompCount * iz,
            pSrc       + srcFaceCompCount * iz,
            dstW, dstH, srcW, srcH, ignoresXpaRgb);
    }

    //-----------------------------------------------------------------------------
    // 奥行きをサイズ変更します。
    const float invScaleZ = static_cast<float>(srcD) / dstD;
    if (dstD == srcD)
    {
        memcpy(pDst, pReduced2d, dstFaceCompCount * srcD);
    }
    else if (invScaleZ <= 2.0f)
    {
        //-----------------------------------------------------------------------------
        // 奥行きのスケールが 0.5 以上の場合は単純な線形補間で計算します。
        for (int iz = 0; iz < dstD; ++iz)
        {
            const float fz = RMax((iz + 0.5f) * invScaleZ - 0.5f, 0.0f);
            const int z0 = RMin(srcD - 1, static_cast<int>(fz));
            const int z1 = RMin(srcD - 1, z0 + 1);
            const float rz = fz - static_cast<float>(z0);
            const float oneMinusRz = 1.0f - rz;
            const T* pS0 = pReduced2d + z0 * dstFaceCompCount;
            const T* pS1 = pReduced2d + z1 * dstFaceCompCount;
            for (size_t iPix = 0; iPix < dstFacePixCount; ++iPix)
            {
                if (!ignoresXpaRgb || (pS0[3] == 0 && pS1[3] == 0))
                {
                    //-----------------------------------------------------------------------------
                    // RGBA 成分を単純に計算
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        const float v = oneMinusRz * (*pS0++) + rz * (*pS1++);
                        *pDst++ = ConvertFloatCompValue<T>(v);
                    }
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // 透明ピクセルの RGB 成分を計算から除外
                    const float w0 = (pS0[3] != 0) ? oneMinusRz : 0.0f;
                    const float w1 = (pS1[3] != 0) ?         rz : 0.0f;
                    const float wsum = w0 + w1;
                    const float wsumInv = (wsum != 0.0f) ? 1.0f / wsum : 0.0f;
                    const bool isZeroA = IsOnTransparent(pS0, pS1, rz);
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        float v = wsumInv * (w0 * (*pS0++) + w1 * (*pS1++));
                        if (isZeroA && iRgba == 3)
                        {
                            v = 0.0f;
                        }
                        *pDst++ = ConvertFloatCompValue<T>(v);
                    }
                }
            }
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // 奥行きのスケールが 0.5 未満の場合、奥行き方向の数ピクセルを平均します。
        for (int iz = 0; iz < dstD; ++iz)
        {
            const float fz = iz * invScaleZ;
            const int z0 = RMin(srcD - 1, static_cast<int>(fz));
            const int z1 = RMin(srcD - 1, static_cast<int>((iz + 1) * invScaleZ) - 1);
            const int blockD = z1 - z0 + 1;
            const T* pS0 = pReduced2d + z0 * dstFaceCompCount;
            for (size_t iPix = 0; iPix < dstFacePixCount; ++iPix)
            {
                if (!ignoresXpaRgb)
                {
                    //-----------------------------------------------------------------------------
                    // RGBA 成分を単純に計算
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        float v = 0.0f;
                        const T* pS = pS0 + iRgba;
                        for (int bz = 0; bz < blockD; ++bz)
                        {
                            v += *pS;
                            pS += dstFaceCompCount;
                        }
                        v /= blockD;
                        *pDst++ = ConvertFloatCompValue<T>(v);
                    }
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // 透明ピクセルの RGB 成分を計算から除外
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        float v = 0.0f;
                        float wv = 0.0f;
                        float wsum = 0.0f;
                        const T* pS = pS0 + iRgba;
                        for (int bz = 0; bz < blockD; ++bz)
                        {
                            v += *pS;
                            const float w = (pS[3 - iRgba] != 0) ? 1.0f : 0.0f;
                            wv += w * (*pS);
                            wsum += w;
                            pS += dstFaceCompCount;
                        }
                        v = (wsum != 0.0f) ? wv / wsum : v / blockD;
                        *pDst++ = ConvertFloatCompValue<T>(v);
                    }
                }
                pS0 += R_RGBA_COUNT;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 幅と高さをサイズ変更したビットマップデータを解放します。
    delete[] pReduced2d;
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief キュービック補間（滑らか）のフィルタのパラメータ C を取得します。
//!
//! @param[in] scale スケールです。
//!
//! @return フィルタのパラメータ C を返します。
//-----------------------------------------------------------------------------
inline float GetCubicSmoothFilterC(const float scale)
{
    return
        (scale >= 1.00f) ? 0.625f                :
        (scale >= 0.25f) ? 1.125f - 0.5f * scale :
        1.0f;
}

//-----------------------------------------------------------------------------
//! @brief キュービック補間（シャープ）のフィルタのパラメータ C を取得します。
//!
//! @param[in] scale スケールです。
//!
//! @return フィルタのパラメータ C を返します。
//-----------------------------------------------------------------------------
inline float GetCubicSharpFilterC(const float scale)
{
    return
        (scale >= 1.00f) ? 1.0f                :
        (scale >= 0.25f) ? 2.6f - 1.6f * scale :
        2.2f;
}

//-----------------------------------------------------------------------------
//! @brief キュービック補間のフィルタのパラメータ C を取得します。
//!
//! @param[in] resizeFilter サイズ変更フィルタです。
//! @param[in] scale スケールです。
//!
//! @return フィルタのパラメータ C を返します。
//-----------------------------------------------------------------------------
float GetCubicFilterC(const ResizeFilter resizeFilter, const float scale)
{
    const float FilterCStandard = 0.75f;

    return
        (resizeFilter == ResizeFilter_Cubic      ) ? FilterCStandard              :
        (resizeFilter == ResizeFilter_CubicSmooth) ? GetCubicSmoothFilterC(scale) :
        GetCubicSharpFilterC(scale);
}

//-----------------------------------------------------------------------------
//! @brief キュービック補間のブラー係数を取得します。
//!
//! @param[in] resizeFilter サイズ変更フィルタです。
//!
//! @return ブラー係数を返します。
//-----------------------------------------------------------------------------
float GetCubicBlur(const ResizeFilter resizeFilter)
{
    const float BlurStandard = 1.00f;
    const float BlurSmooth   = 1.15f;
    const float BlurSharp    = 1.05f;

    return
        (resizeFilter == ResizeFilter_Cubic      ) ? BlurStandard :
        (resizeFilter == ResizeFilter_CubicSmooth) ? BlurSmooth   :
        BlurSharp;
}

//-----------------------------------------------------------------------------
//! @brief キュービック補間の距離に対するウェイトを取得します。
//!
//! @param[in] d 対象ピクセルまでの距離です。
//! @param[in] c Mitchell-Netravali フィルタのパラメータ C です（大きいほどシャープになります）。
//!
//! @return 距離に対するウェイトを返します。
//-----------------------------------------------------------------------------
inline float GetCubicInterpWeight(const float d, const float c)
{
    if (d >= 2.0f)
    {
        return 0.0f;
    }
    else
    {
        const float d2 = d * d;
        const float d3 = d2 * d;
        if (d < 1.0f)
        {
            return (2.0f - c) * d3 + (-3.0f + c) * d2 + 1.0f;
            //return d3 - 2.0f * d2 + 1.0f; // c = 1
        }
        else
        {
            return (-c) * (d3 - 5.0f * d2 + 8.0f * d - 4.0f);
            //return -d3 + 5.0f * d2 - 8.0f * d + 4.0f; // c = 1
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief フェースを任意の幅と高さにキュービック補間（各方向 4 ピクセルを重み付け平均）でサイズ変更します。
//!
//! @param[out] pDst サイズ変更後のビットマップデータを格納します。
//! @param[in] pSrc サイズ変更前のビットマップデータです。
//! @param[in] dstW サイズ変更後の幅です。
//! @param[in] dstH サイズ変更後の高さです。
//! @param[in] srcW サイズ変更前の幅です。
//! @param[in] srcH サイズ変更前の高さです。
//! @param[in] ignoresXpaRgb 透明ピクセルの RGB 成分を補間計算から除外するなら true です。
//! @param[in] filterCx 水平方向のフィルタのパラメータ C です。
//! @param[in] filterCy 垂直方向のフィルタのパラメータ C です。
//-----------------------------------------------------------------------------
template <typename T>
void ResizeFaceBySimpleCubic(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int srcW,
    const int srcH,
    const bool ignoresXpaRgb,
    const float filterCx,
    const float filterCy
)
{
    const int lineValCount = srcW * R_RGBA_COUNT;
    const float invScaleX = static_cast<float>(srcW) / dstW;
    const float invScaleY = static_cast<float>(srcH) / dstH;
    const bool clamps01 = IsAllPixelsInRange01(pSrc, static_cast<size_t>(srcW) * srcH);

    const int SampleCount = 4;
    for (int iy = 0; iy < dstH; ++iy)
    {
        const float fy = RMax((iy + 0.5f) * invScaleY - 0.5f, 0.0f);
        const int y0 = RMin(static_cast<int>(fy), srcH - 1);
        const float ry = fy - static_cast<float>(y0);
        const int yofss[SampleCount] =
        {
            ((y0 > 0) ? y0 - 1 : 0) * lineValCount,
            RMin(y0    , srcH - 1) * lineValCount,
            RMin(y0 + 1, srcH - 1) * lineValCount,
            RMin(y0 + 2, srcH - 1) * lineValCount
        };
        float weightsY[SampleCount];
        if (0 < y0 && y0 + 2 < srcH)
        {
            weightsY[0] = GetCubicInterpWeight(1.0f + ry, filterCy);
            weightsY[1] = GetCubicInterpWeight(       ry, filterCy);
            weightsY[2] = GetCubicInterpWeight(1.0f - ry, filterCy);
            weightsY[3] = GetCubicInterpWeight(2.0f - ry, filterCy);
        }
        else // 画像の端の部分は部分的にリニア補間
        {
            weightsY[0] = 0.0f;
            weightsY[1] = 1.0f - ry;
            weightsY[2] = ry;
            weightsY[3] = 0.0f;
        }
        //if (y0 == 1) cerr << "wy: " << ry << ": " << weightsY[0] << " " << weightsY[1] << " " << weightsY[2] << " " << weightsY[3] << endl;
        for (int ix = 0; ix < dstW; ++ix)
        {
            const float fx = RMax((ix + 0.5f) * invScaleX - 0.5f, 0.0f);
            const int x0 = RMin(static_cast<int>(fx), srcW - 1);
            const float rx = fx - static_cast<float>(x0);
            const int xofss[SampleCount] =
            {
                ((x0 > 0) ? x0 - 1 : 0) * R_RGBA_COUNT,
                RMin(x0    , srcW - 1) * R_RGBA_COUNT,
                RMin(x0 + 1, srcW - 1) * R_RGBA_COUNT,
                RMin(x0 + 2, srcW - 1) * R_RGBA_COUNT
            };
            float weightsX[SampleCount];
            if (0 < x0 && x0 + 2 < srcW)
            {
                weightsX[0] = GetCubicInterpWeight(1.0f + rx, filterCx);
                weightsX[1] = GetCubicInterpWeight(       rx, filterCx);
                weightsX[2] = GetCubicInterpWeight(1.0f - rx, filterCx);
                weightsX[3] = GetCubicInterpWeight(2.0f - rx, filterCx);
            }
            else // 画像の端の部分は部分的にリニア補間
            {
                weightsX[0] = 0.0f;
                weightsX[1] = 1.0f - rx;
                weightsX[2] = rx;
                weightsX[3] = 0.0f;
            }
            if (!ignoresXpaRgb)
            {
                //-----------------------------------------------------------------------------
                // RGBA 成分を単純に計算
                for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                {
                    float v = 0.0f;
                    for (int sy = 0; sy < SampleCount; ++sy)
                    {
                        const float weightY = weightsY[sy];
                        for (int sx = 0; sx < SampleCount; ++sx)
                        {
                            v += weightY * weightsX[sx] * pSrc[yofss[sy] + xofss[sx] + iRgba];
                        }
                    }
                    *pDst++ = ClampFloatCompValue<T>(v, clamps01);
                }
            }
            else
            {
                //-----------------------------------------------------------------------------
                // 透明ピクセルの RGB 成分を計算から除外
                const bool isZeroA = IsOnTransparent(
                    pSrc + yofss[1] + xofss[1],
                    pSrc + yofss[1] + xofss[2],
                    pSrc + yofss[2] + xofss[1],
                    pSrc + yofss[2] + xofss[2],
                    rx, ry);
                for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                {
                    float v = 0.0f;
                    if (!(isZeroA && iRgba == 3))
                    {
                        float wv = 0.0f;
                        float wsum = 0.0f;
                        float wvAll = 0.0f;
                        for (int sy = 0; sy < SampleCount; ++sy)
                        {
                            const float weightY = weightsY[sy];
                            for (int sx = 0; sx < SampleCount; ++sx)
                            {
                                const T src = pSrc[yofss[sy] + xofss[sx] + iRgba];
                                const float w = weightY * weightsX[sx];
                                if (pSrc[yofss[sy] + xofss[sx] + 3] != 0)
                                {
                                    wv += w * src;
                                    wsum += w;
                                }
                                wvAll += w * src;
                            }
                        }
                        v = (wsum != 0.0f) ? wv / wsum : wvAll;
                    }
                    *pDst++ = ClampFloatCompValue<T>(v, clamps01);
                }
            }
        }
    }
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief フェースを任意の幅と高さにキュービック補間（各方向 5 ピクセル以上を重み付け平均）でサイズ変更します。
//!
//! @param[out] pDst サイズ変更後のビットマップデータを格納します。
//! @param[in] pSrc サイズ変更前のビットマップデータです。
//! @param[in] dstW サイズ変更後の幅です。
//! @param[in] dstH サイズ変更後の高さです。
//! @param[in] srcW サイズ変更前の幅です。
//! @param[in] srcH サイズ変更前の高さです。
//! @param[in] ignoresXpaRgb 透明ピクセルの RGB 成分を補間計算から除外するなら true です。
//! @param[in] filterCx 水平方向のフィルタのパラメータ C です。
//! @param[in] filterCy 垂直方向のフィルタのパラメータ C です。
//! @param[in] blur ブラー係数です（大きいほどボケます。ブラーなしなら 1.0 です。大きいほど計算時間が増えます）。
//-----------------------------------------------------------------------------
template <typename T>
void ResizeFaceByAdvancedCubic(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int srcW,
    const int srcH,
    const bool ignoresXpaRgb,
    const float filterCx,
    const float filterCy,
    const float blur
)
{
    const float Epsilon = 0.0001f; // ceil / floor のための微小な値です。

    const int lineValCount = srcW * R_RGBA_COUNT;
    const float invScaleX = static_cast<float>(srcW) / dstW;
    const float invScaleY = static_cast<float>(srcH) / dstH;
    const bool clamps01 = IsAllPixelsInRange01(pSrc, static_cast<size_t>(srcW) * srcH);

    float factorX = (invScaleX <= 1.0f) ? 1.0f : invScaleX;
    float factorY = (invScaleY <= 1.0f) ? 1.0f : invScaleY;
    factorX *= blur;
    factorY *= blur;

    const int maxPixCountX = 1 + 2 * static_cast<int>(ceil(2.0f * factorX));
    const int maxPixCountY = 1 + 2 * static_cast<int>(ceil(2.0f * factorY));
    RFloatArray weightsX(maxPixCountX);
    RFloatArray weightsY(maxPixCountY);
    RIntArray xofss(maxPixCountX);
    RIntArray yofss(maxPixCountY);
    for (int iy = 0; iy < dstH; ++iy)
    {
        const float fy = (iy + 0.5f) * invScaleY - 0.5f;
        const int y0 = static_cast<int>(ceil (fy - 2.0f * factorY - Epsilon));
        const int y1 = static_cast<int>(floor(fy + 2.0f * factorY + Epsilon));
        int blockH = 0;
        for (int sy = y0; sy <= y1; ++sy)
        {
            const float dy = abs(sy - fy) / factorY;
            const float wy = GetCubicInterpWeight(dy, filterCy);
            if (wy != 0.0f)
            {
                weightsY[blockH] = wy;
                yofss[blockH] = RClampValue(0, srcH - 1, sy) * lineValCount;
                ++blockH;
            }
        }
        //cerr << "blockY: " << iy << ", " << fy << ", " << y0 << " - " << y1 << ", " << blockH << endl;
        for (int ix = 0; ix < dstW; ++ix)
        {
            const float fx = (ix + 0.5f) * invScaleX - 0.5f;
            const int x0 = static_cast<int>(ceil (fx - 2.0f * factorX - Epsilon));
            const int x1 = static_cast<int>(floor(fx + 2.0f * factorX + Epsilon));
            int blockW = 0;
            for (int sx = x0; sx <= x1; ++sx)
            {
                const float dx = abs(sx - fx) / factorX;
                const float wx = GetCubicInterpWeight(dx, filterCx);
                if (wx != 0.0f)
                {
                    weightsX[blockW] = wx;
                    xofss[blockW] = RClampValue(0, srcW - 1, sx) * R_RGBA_COUNT;
                    ++blockW;
                }
            }
            for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
            {
                float wv = 0.0f;
                float wsum = 0.0f;
                float wvAll = 0.0f;
                float wsumAll = 0.0f;
                for (int by = 0; by < blockH; ++by)
                {
                    const float weightY = weightsY[by];
                    for (int bx = 0; bx < blockW; ++bx)
                    {
                        const T src = pSrc[yofss[by] + xofss[bx] + iRgba];
                        const float w = weightY * weightsX[bx];
                        if (!ignoresXpaRgb || iRgba == 3 || pSrc[yofss[by] + xofss[bx] + 3] != 0)
                        {
                            wv += w * src;
                            wsum += w;
                        }
                        wvAll += w * src;
                        wsumAll += w;
                    }
                }
                const float v = (wsum != 0.0f) ? wv / wsum :
                    ((wsumAll != 0.0f) ? wvAll / wsumAll : 0.0f);
                *pDst++ = ClampFloatCompValue<T>(v, clamps01);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief フェースを任意の幅と高さにキュービック補間でサイズ変更します。
//!
//! @param[out] pDst サイズ変更後のビットマップデータを格納します。
//! @param[in] pSrc サイズ変更前のビットマップデータです。
//! @param[in] dstW サイズ変更後の幅です。
//! @param[in] dstH サイズ変更後の高さです。
//! @param[in] srcW サイズ変更前の幅です。
//! @param[in] srcH サイズ変更前の高さです。
//! @param[in] resizeFilter サイズ変更フィルタです。
//! @param[in] ignoresXpaRgb 透明ピクセルの RGB 成分を補間計算から除外するなら true です。
//-----------------------------------------------------------------------------
template <typename T>
void ResizeFaceByCubic(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int srcW,
    const int srcH,
    const ResizeFilter resizeFilter,
    const bool ignoresXpaRgb
)
{
    //-----------------------------------------------------------------------------
    // 幅または高さのスケールが 0.25 未満の場合、4 倍のサイズまで線形補間で計算後、
    // キュービック補間で 0.25 倍にサイズ変更します。
    // （サンプリングするピクセルを減らして高速化するため。Photoshop も同様）
    if (dstW * 4 < srcW || dstH * 4 < srcH)
    {
        const int interW = (dstW * 4 < srcW) ? dstW * 4 : srcW;
        const int interH = (dstH * 4 < srcH) ? dstH * 4 : srcH;
        //cerr << "inter: " << interW << " x " << interH << endl;
        T* pInterData = new T[interW * interH * R_RGBA_COUNT];
        ResizeFaceByLinear(pInterData, pSrc, interW, interH, srcW, srcH, ignoresXpaRgb);
        ResizeFaceByCubic(pDst, pInterData, dstW, dstH, interW, interH, resizeFilter, ignoresXpaRgb);
        delete[] pInterData;
        return;
    }

    //-----------------------------------------------------------------------------
    // 補間パラメータを決定します。
    const float scaleX = static_cast<float>(dstW) / srcW;
    const float scaleY = static_cast<float>(dstH) / srcH;
    const float filterCx = GetCubicFilterC(resizeFilter, scaleX);
    const float filterCy = GetCubicFilterC(resizeFilter, scaleY);
    const float blur = GetCubicBlur(resizeFilter);

    //-----------------------------------------------------------------------------
    if (scaleX >= 1.0f && scaleY >= 1.0f && blur == 1.0f)
    {
        // 水平／垂直方向ともに拡大かつブラーなしの場合、
        // 各方向 4 ピクセルを重み付け平均します。
        ResizeFaceBySimpleCubic(pDst, pSrc, dstW, dstH, srcW, srcH, ignoresXpaRgb,
            filterCx, filterCy);
    }
    else
    {
        // 水平／垂直方向のいずれかが縮小の場合またはブラーありの場合、
        // 各方向 5 ピクセル以上を重み付け平均します。
        ResizeFaceByAdvancedCubic(pDst, pSrc, dstW, dstH, srcW, srcH, ignoresXpaRgb,
            filterCx, filterCy, blur);
    }
}

//-----------------------------------------------------------------------------
//! @brief 3D テクスチャを任意のサイズにキュービック補間でサイズ変更します。
//!
//! @param[out] pDst サイズ変更後のビットマップデータを格納します。
//! @param[in] pSrc サイズ変更前のビットマップデータです。
//! @param[in] dstW サイズ変更後の幅です。
//! @param[in] dstH サイズ変更後の高さです。
//! @param[in] dstD サイズ変更後の奥行きです。
//! @param[in] srcW サイズ変更前の幅です。
//! @param[in] srcH サイズ変更前の高さです。
//! @param[in] srcD サイズ変更前の奥行きです。
//! @param[in] resizeFilter サイズ変更フィルタです。
//! @param[in] ignoresXpaRgb 透明ピクセルの RGB 成分を補間計算から除外するなら true です。
//-----------------------------------------------------------------------------
template <typename T>
void Resize3dByCubic(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int dstD,
    const int srcW,
    const int srcH,
    const int srcD,
    const ResizeFilter resizeFilter,
    const bool ignoresXpaRgb
)
{
    //-----------------------------------------------------------------------------
    // 奥行きのスケールが 0.25 未満の場合、4 倍のサイズまで線形補間で計算後、
    // キュービック補間で 0.25 倍にサイズ変更します。
    if (dstD * 4 < srcD)
    {
        const int interD = dstD * 4;
        //cerr << "inter: " << interD << endl;
        T* pInterData = new T[srcW * srcH * interD * R_RGBA_COUNT];
        Resize3dByLinear(pInterData, pSrc, srcW, srcH, interD, srcW, srcH, srcD, ignoresXpaRgb);
        Resize3dByCubic(pDst, pInterData, dstW, dstH, dstD, srcW, srcH, interD, resizeFilter, ignoresXpaRgb);
        delete[] pInterData;
        return;
    }

    //-----------------------------------------------------------------------------
    // 幅と高さをサイズ変更します。
    const size_t srcFacePixCount = srcW * srcH;
    const size_t dstFacePixCount = dstW * dstH;
    const size_t srcFaceCompCount = srcFacePixCount * R_RGBA_COUNT;
    const size_t dstFaceCompCount = dstFacePixCount * R_RGBA_COUNT;
    T* pReduced2d = new T[dstFaceCompCount * srcD];
    for (int iz = 0; iz < srcD; ++iz)
    {
        ResizeFaceByCubic(
            pReduced2d + dstFaceCompCount * iz,
            pSrc       + srcFaceCompCount * iz,
            dstW, dstH, srcW, srcH, resizeFilter, ignoresXpaRgb);
    }

    //-----------------------------------------------------------------------------
    // 補間パラメータを決定します。
    const float scaleZ = static_cast<float>(dstD) / srcD;
    const float filterCz = GetCubicFilterC(resizeFilter, scaleZ);
    const float blur = GetCubicBlur(resizeFilter);

    //-----------------------------------------------------------------------------
    // 奥行きをサイズ変更します。
    const bool clamps01 = IsAllPixelsInRange01(pReduced2d, static_cast<size_t>(dstW) * dstH * srcD);
    const float invScaleZ = static_cast<float>(srcD) / dstD;
    if (dstD == srcD)
    {
        memcpy(pDst, pReduced2d, dstFaceCompCount * srcD);
    }
    else if (scaleZ >= 1.0f && blur == 1.0f)
    {
        //-----------------------------------------------------------------------------
        // 拡大かつブラーなしの場合、奥行き方向 4 ピクセルを重み付け平均します。
        const int SampleCount = 4;
        for (int iz = 0; iz < dstD; ++iz)
        {
            const float fz = RMax((iz + 0.5f) * invScaleZ - 0.5f, 0.0f);
            const int z0 = RMin(static_cast<int>(fz), srcD - 1);
            const float rz = fz - static_cast<float>(z0);
            const size_t zofss[SampleCount] =
            {
                ((z0 > 0) ? z0 - 1 : 0) * dstFaceCompCount,
                RMin(z0    , srcD - 1) * dstFaceCompCount,
                RMin(z0 + 1, srcD - 1) * dstFaceCompCount,
                RMin(z0 + 2, srcD - 1) * dstFaceCompCount
            };
            float weightsZ[SampleCount];
            if (0 < z0 && z0 + 2 < srcD)
            {
                weightsZ[0] = GetCubicInterpWeight(1.0f + rz, filterCz);
                weightsZ[1] = GetCubicInterpWeight(       rz, filterCz);
                weightsZ[2] = GetCubicInterpWeight(1.0f - rz, filterCz);
                weightsZ[3] = GetCubicInterpWeight(2.0f - rz, filterCz);
            }
            else // 画像の端の部分は部分的にリニア補間
            {
                weightsZ[0] = 0.0f;
                weightsZ[1] = 1.0f - rz;
                weightsZ[2] = rz;
                weightsZ[3] = 0.0f;
            }

            for (size_t iPix = 0; iPix < dstFacePixCount; ++iPix)
            {
                const size_t pixOfs = iPix * R_RGBA_COUNT;
                if (!ignoresXpaRgb)
                {
                    //-----------------------------------------------------------------------------
                    // RGBA 成分を単純に計算
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        float v = 0.0f;
                        for (int sz = 0; sz < SampleCount; ++sz)
                        {
                            v += weightsZ[sz] * pReduced2d[zofss[sz] + pixOfs + iRgba];
                        }
                        *pDst++ = ClampFloatCompValue<T>(v, clamps01);
                    }
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // 透明ピクセルの RGB 成分を計算から除外
                    const bool isZeroA = IsOnTransparent(
                        pReduced2d + zofss[1] + pixOfs,
                        pReduced2d + zofss[2] + pixOfs,
                        rz);
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        float v = 0.0f;
                        if (!(isZeroA && iRgba == 3))
                        {
                            float wv = 0.0f;
                            float wsum = 0.0f;
                            float wvAll = 0.0f;
                            for (int sz = 0; sz < SampleCount; ++sz)
                            {
                                const T src = pReduced2d[zofss[sz] + pixOfs + iRgba];
                                const float w = weightsZ[sz];
                                if (pReduced2d[zofss[sz] + pixOfs + 3] != 0)
                                {
                                    wv += w * src;
                                    wsum += w;
                                }
                                wvAll += w * src;
                            }
                            v = (wsum != 0.0f) ? wv / wsum : wvAll;
                        }
                        *pDst++ = ClampFloatCompValue<T>(v, clamps01);
                    }
                }
            }
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // 縮小の場合またはブラーありの場合、奥行き方向 5 ピクセル以上を重み付け平均します。
        const float Epsilon = 0.0001f; // ceil / floor のための微小な値です。

        float factorZ = (invScaleZ <= 1.0f) ? 1.0f : invScaleZ;
        factorZ *= blur;

        const int maxPixCountZ = 1 + 2 * static_cast<int>(ceil(2.0f * factorZ));
        RFloatArray weightsZ(maxPixCountZ);
        std::vector<size_t> zofss(maxPixCountZ);

        for (int iz = 0; iz < dstD; ++iz)
        {
            const float fz = (iz + 0.5f) * invScaleZ - 0.5f;
            const int z0 = static_cast<int>(ceil (fz - 2.0f * factorZ - Epsilon));
            const int z1 = static_cast<int>(floor(fz + 2.0f * factorZ + Epsilon));
            int blockD = 0;
            for (int sz = z0; sz <= z1; ++sz)
            {
                const float dz = abs(sz - fz) / factorZ;
                const float wz = GetCubicInterpWeight(dz, filterCz);
                if (wz != 0.0f)
                {
                    weightsZ[blockD] = wz;
                    zofss[blockD] = RClampValue(0, srcD - 1, sz) * dstFaceCompCount;
                    ++blockD;
                }
            }

            for (size_t iPix = 0; iPix < dstFacePixCount; ++iPix)
            {
                const size_t pixOfs = iPix * R_RGBA_COUNT;
                for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                {
                    float wv = 0.0f;
                    float wsum = 0.0f;
                    float wvAll = 0.0f;
                    float wsumAll = 0.0f;
                    for (int bz = 0; bz < blockD; ++bz)
                    {
                        const T src = pReduced2d[zofss[bz] + pixOfs + iRgba];
                        const float w = weightsZ[bz];
                        if (!ignoresXpaRgb || iRgba == 3 || pReduced2d[zofss[bz] + pixOfs + 3] != 0)
                        {
                            wv += w * src;
                            wsum += w;
                        }
                        wvAll += w * src;
                        wsumAll += w;
                    }
                    const float v = (wsum != 0.0f) ? wv / wsum :
                        ((wsumAll != 0.0f) ? wvAll / wsumAll : 0.0f);
                    *pDst++ = ClampFloatCompValue<T>(v, clamps01);
                }
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 幅と高さをサイズ変更したビットマップデータを解放します。
    delete[] pReduced2d;
} // NOLINT(readability/fn_size)

//-----------------------------------------------------------------------------
//! @brief 隣接する 4 ピクセルを取得します。
//!
//! @param[out] adjs 隣接する 4 ピクセルの RGBA データを格納します。
//! @param[in] pSrc ビットマップデータです。
//! @param[in] ix 中央のピクセルの X 座標です。
//! @param[in] iy 中央のピクセルの Y 座標です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] isDiagonal 上下左右のピクセルを取得するなら false、斜め 4 方向のピクセルを取得するなら true を指定します。
//!
//! @return アルファが 0 でない隣接ピクセルがあれば true を返します。
//-----------------------------------------------------------------------------
template <typename T>
bool GetAdjacent4Pixels(
    T adjs[4][R_RGBA_COUNT],
    const T* pSrc,
    const int ix,
    const int iy,
    const int imageW,
    const int imageH,
    const bool isDiagonal
)
{
    const int PositionCount = 4;

    //   0
    // 1 * 2
    //   3
    static const int s_Poss0[PositionCount][2] =
    {
        {  0, -1 }, // 0
        { -1,  0 }, // 1
        {  1,  0 }, // 2
        {  0,  1 }, // 3
    };

    // 0   1
    //   *
    // 2   3
    static const int s_Poss1[PositionCount][2] =
    {
        { -1, -1 }, // 0
        {  1, -1 }, // 1
        { -1,  1 }, // 2
        {  1,  1 }, // 3
    };

    const size_t pixelBytes = sizeof(T) * R_RGBA_COUNT;
    const int rowOfs = imageW * R_RGBA_COUNT;
    const int (*pPoss)[2] = (!isDiagonal) ? s_Poss0 : s_Poss1;

    if (0 < ix && ix < imageW - 1 &&
        0 < iy && iy < imageH - 1)
    {
        // 中央のピクセルが画像の端でない場合
        for (int iPos = 0; iPos < PositionCount; ++iPos)
        {
            const int px = ix + pPoss[iPos][0];
            const int py = iy + pPoss[iPos][1];
            memcpy(adjs[iPos], pSrc + px * R_RGBA_COUNT + py * rowOfs, pixelBytes);
        }
    }
    else
    {
        // 中央のピクセルが画像の端の場合
        memset(adjs, 0x00, pixelBytes * PositionCount);
        for (int iPos = 0; iPos < PositionCount; ++iPos)
        {
            const int px = ix + pPoss[iPos][0];
            const int py = iy + pPoss[iPos][1];
            if (0 <= px && px < imageW &&
                0 <= py && py < imageH)
            {
                memcpy(adjs[iPos], pSrc + px * R_RGBA_COUNT + py * rowOfs, pixelBytes);
            }
        }
    }

    return (
        adjs[0][3] != 0 ||
        adjs[1][3] != 0 ||
        adjs[2][3] != 0 ||
        adjs[3][3] != 0);
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief float 型の値から float_16 の値を取得します。
//-----------------------------------------------------------------------------
uint16_t RGetFloat16Value(const float valueF32)
{
    const float f = valueF32;

    // 以下、OpenEXR の half.h の half::half(float f) からコピーしたコードです。

    union uif
    {
        uint32_t i;
        float f;
    };

    uif x;
    x.f = f;

    uint16_t hv = 0;
    if (f == 0)
    {
        hv = static_cast<uint16_t>(x.i >> 16);
    }
    else
    {
        int e = (x.i >> 23) & 0x000001ff;
        e = Float16ELut[e];
        if (e)
        {
            int m = x.i & 0x007fffff;
            hv = static_cast<uint16_t>(e + ((m + 0x00000fff + ((m >> 13) & 1)) >> 13));
        }
        else
        {
            hv = ConvertToFloat16Value(static_cast<int>(x.i));
        }
    }
    return hv;
}

//-----------------------------------------------------------------------------
//! @brief フェースを水平に反転します。
//-----------------------------------------------------------------------------
void RFlipFaceH(
    void* pDst,
    const int dstW,
    const int dstH,
    const size_t stride,
    const size_t pixelBytes
)
{
    uint8_t* pDstTop = reinterpret_cast<uint8_t*>(pDst);
    uint8_t* pDstLine = pDstTop;
    uint8_t* pPixelBuf = new uint8_t[pixelBytes];
    const int halfW = dstW / 2;
    for (int iy = 0; iy < dstH; ++iy)
    {
        for (int ix = 0; ix < halfW; ++ix)
        {
            const int jx = dstW - 1 - ix;
            uint8_t* pPixI = pDstLine + ix * pixelBytes;
            uint8_t* pPixJ = pDstLine + jx * pixelBytes;
            memcpy(pPixelBuf, pPixI    , pixelBytes);
            memcpy(pPixI    , pPixJ    , pixelBytes);
            memcpy(pPixJ    , pPixelBuf, pixelBytes);
        }
        pDstLine += stride;
    }
    delete[] pPixelBuf;
}

//-----------------------------------------------------------------------------
//! @brief フェースを垂直に反転します。
//-----------------------------------------------------------------------------
void RFlipFaceV(
    void* pDst,
    const int dstW,
    const int dstH,
    const size_t stride,
    const size_t pixelBytes
)
{
    uint8_t* pDstTop = reinterpret_cast<uint8_t*>(pDst);
    const int halfH = dstH / 2;
    const size_t lineBytes = pixelBytes * dstW;
    uint8_t* pLineBuf = new uint8_t[lineBytes];
    for (int iy = 0; iy < halfH; ++iy)
    {
        const int jy = dstH - 1 - iy;
        uint8_t* pLineI = pDstTop + iy * stride;
        uint8_t* pLineJ = pDstTop + jy * stride;
        memcpy(pLineBuf, pLineI  , lineBytes);
        memcpy(pLineI  , pLineJ  , lineBytes);
        memcpy(pLineJ  , pLineBuf, lineBytes);
    }
    delete[] pLineBuf;
}

//-----------------------------------------------------------------------------
//! @brief フェースを水平垂直に反転します。
//-----------------------------------------------------------------------------
void RFlipFaceHv(
    void* pDst,
    const int dstW,
    const int dstH,
    const size_t stride,
    const size_t pixelBytes
)
{
    RFlipFaceH(pDst, dstW, dstH, stride, pixelBytes);
    RFlipFaceV(pDst, dstW, dstH, stride, pixelBytes);
}

//-----------------------------------------------------------------------------
//! @brief フェースを切り抜きます。
//-----------------------------------------------------------------------------
template <typename T>
void RCropFace(
    T* pDst,
    const T* pSrc,
    const RCropRect& cropRect,
    const int srcW
)
{
    const size_t lineBytes = static_cast<size_t>(cropRect.m_W) * R_RGBA_COUNT * sizeof(T);
    const T* pS = pSrc + (cropRect.m_X + cropRect.m_Y * srcW) * R_RGBA_COUNT;
    for (int iy = 0; iy < cropRect.m_H; ++iy)
    {
        memcpy(pDst, pS, lineBytes);
        pS += srcW * R_RGBA_COUNT;
        pDst += cropRect.m_W * R_RGBA_COUNT;
    }
}

//-----------------------------------------------------------------------------
// テンプレートを明示的に実体化します。
template void RCropFace<uint8_t>(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const RCropRect& cropRect,
    const int srcW
);

template void RCropFace<float>(
    float* pDst,
    const float* pSrc,
    const RCropRect& cropRect,
    const int srcW
);

//-----------------------------------------------------------------------------
//! @brief フェースを任意の幅と高さにサイズ変更します。
//-----------------------------------------------------------------------------
template <typename T>
void RResizeFreeFace(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int srcW,
    const int srcH,
    const ResizeFilter resizeFilter,
    const bool ignoresXpaRgb
)
{
    if (resizeFilter == ResizeFilter_Point)
    {
        ResizeFaceByPoint(pDst, pSrc, dstW, dstH, srcW, srcH);
    }
    else if (resizeFilter == ResizeFilter_Linear ||
             resizeFilter == ResizeFilter_Default)
    {
        ResizeFaceByLinear(pDst, pSrc, dstW, dstH, srcW, srcH, ignoresXpaRgb);
    }
    else
    {
        ResizeFaceByCubic(pDst, pSrc, dstW, dstH, srcW, srcH, resizeFilter, ignoresXpaRgb);
    }
}

//-----------------------------------------------------------------------------
// テンプレートを明示的に実体化します。
template void RResizeFreeFace<uint8_t>(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int dstW,
    const int dstH,
    const int srcW,
    const int srcH,
    const ResizeFilter resizeFilter,
    const bool ignoresXpaRgb
);

template void RResizeFreeFace<float>(
    float* pDst,
    const float* pSrc,
    const int dstW,
    const int dstH,
    const int srcW,
    const int srcH,
    const ResizeFilter resizeFilter,
    const bool ignoresXpaRgb
);

//-----------------------------------------------------------------------------
//! @brief 3D テクスチャを任意のサイズにサイズ変更します。
//-----------------------------------------------------------------------------
template <typename T>
void RResizeFree3d(
    T* pDst,
    const T* pSrc,
    const int dstW,
    const int dstH,
    const int dstD,
    const int srcW,
    const int srcH,
    const int srcD,
    const ResizeFilter resizeFilter,
    const bool ignoresXpaRgb
)
{
    if (resizeFilter == ResizeFilter_Point)
    {
        Resize3dByPoint(pDst, pSrc, dstW, dstH, dstD, srcW, srcH, srcD);
    }
    else if (resizeFilter == ResizeFilter_Linear ||
             resizeFilter == ResizeFilter_Default)
    {
        Resize3dByLinear(pDst, pSrc, dstW, dstH, dstD, srcW, srcH, srcD, ignoresXpaRgb);
    }
    else
    {
        Resize3dByCubic(pDst, pSrc, dstW, dstH, dstD, srcW, srcH, srcD, resizeFilter, ignoresXpaRgb);
    }
}

//-----------------------------------------------------------------------------
// テンプレートを明示的に実体化します。
template void RResizeFree3d<uint8_t>(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int dstW,
    const int dstH,
    const int dstD,
    const int srcW,
    const int srcH,
    const int srcD,
    const ResizeFilter resizeFilter,
    const bool ignoresXpaRgb
);

template void RResizeFree3d<float>(
    float* pDst,
    const float* pSrc,
    const int dstW,
    const int dstH,
    const int dstD,
    const int srcW,
    const int srcH,
    const int srcD,
    const ResizeFilter resizeFilter,
    const bool ignoresXpaRgb
);

//-----------------------------------------------------------------------------
//! @brief フェースの透明ピクセルの RGB 成分を調整します。
//-----------------------------------------------------------------------------
template <typename T>
void RAdjustTransparentRgbFace(T* pDst, const int dstW, const int dstH)
{
    T* pCur = pDst;
    T adjs[4][R_RGBA_COUNT];
    for (int iy = 0; iy < dstH; ++iy)
    {
        for (int ix = 0; ix < dstW; ++ix)
        {
            if (pCur[3] == 0)
            {
                for (int iTry = 0; iTry < 2; ++iTry)
                {
                    const bool isDiagonal = (iTry == 1);
                    if (GetAdjacent4Pixels(adjs, pDst, ix, iy, dstW, dstH, isDiagonal))
                    {
                        const float alphaSum = static_cast<float>(
                            adjs[0][3] +
                            adjs[1][3] +
                            adjs[2][3] +
                            adjs[3][3]);
                        const float alphaSumInv = 1.0f / alphaSum;
                        for (int iRgb = 0; iRgb < R_RGB_COUNT; ++iRgb)
                        {
                            const float sum =
                                adjs[0][iRgb] * static_cast<float>(adjs[0][3]) +
                                adjs[1][iRgb] * static_cast<float>(adjs[1][3]) +
                                adjs[2][iRgb] * static_cast<float>(adjs[2][3]) +
                                adjs[3][iRgb] * static_cast<float>(adjs[3][3]);
                            const float average = sum * alphaSumInv;
                            pCur[iRgb] = ConvertFloatCompValue<T>(average);
                        }
                        break;
                    }
                }
            }
            pCur += R_RGBA_COUNT;
        }
    }
}

//-----------------------------------------------------------------------------
// テンプレートを明示的に実体化します。
template void RAdjustTransparentRgbFace<uint8_t>(uint8_t* pDst, const int dstW, const int dstH);
template void RAdjustTransparentRgbFace<float>  (float*   pDst, const int dstW, const int dstH);

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

