﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// TexConverterBase FtxOpt ConverterUtility
// RImage ROriginalImageT RMergeInfo ExtraImageAttr

//=============================================================================
// include
//=============================================================================
#include "ImageEdit.h"
#include "NXUtility.h"
#include "REncoder.h"
#include "ROpenExr.h"
#include "WicUtility.h"

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//=============================================================================
// constants
//=============================================================================
extern const int LinearRgbaFlags[]; //!< RGBA 成分に対するリニア変換フラグ配列です。

//=============================================================================
//! @brief ftx 変換オプションのクラスです。
//=============================================================================
class FtxOpt
{
public:
    //! @brief GPU エンコーディングモードを表す列挙型です。
    enum GpuEncoding
    {
        GpuEncoding_Disable, //!< 常に CPU でエンコードします。
        GpuEncoding_Enable,  //!< 可能なら GPU でエンコードし、不可能ならエラーにします。
        GpuEncoding_Auto,    //!< 可能なら GPU でエンコードし、不可能なら CPU でエンコードします。
        GpuEncoding_Invalid  //!< 無効な GPU エンコーディングモードです。
    };

    static const std::string DCC_PRESET_DEFAULT; //!< デフォルトの DCC プリセット名（指定なし）です。

    static const std::string HINT_DEFAULT; //!< デフォルトのヒント情報（指定なし）です。

    //! @brief リニア変換フラグを表す列挙型です。
    enum LinearFlag
    {
        LINEAR_NONE    = 0,        //!< sRGB からリニアに変換しません。
        LINEAR_R       = (1 << 3), //!< R 成分を sRGB からリニアに変換します。
        LINEAR_G       = (1 << 2), //!< G 成分を sRGB からリニアに変換します。
        LINEAR_B       = (1 << 1), //!< B 成分を sRGB からリニアに変換します。
        LINEAR_A       = (1 << 0), //!< A 成分を sRGB からリニアに変換します。
        LINEAR_RGB     = LINEAR_R | LINEAR_G | LINEAR_B, //!< RGB 成分を sRGB からリニアに変換します。
        LINEAR_RGBA    = LINEAR_RGB | LINEAR_A,          //!< RGBA 成分を sRGB からリニアに変換します。
        LINEAR_INVALID = -2,       //!< 無効なリニア変換です。
        LINEAR_DEFAULT = -1        //!< デフォルトのリニア変換（指定なし）です。
    };

    static const int DIM_INVALID = -2; //!< 無効な次元です。
    static const int DIM_DEFAULT = -1; //!< デフォルトの次元（指定なし）です。

    //! @brief 特別なフォーマットを表す列挙型です。
    enum SpecialFormat
    {
        FORMAT_INVALID =  0, //!< 無効なフォーマット（FtxFormat_Invalid）です。
        FORMAT_DEFAULT = -1, //!< デフォルトのフォーマット（指定なし）です。
        FORMAT_BC_AUTO = -2  //!< 画像の状況に応じて bc1/3/4/5 を自動的に選択します。
    };

    static const int DEPTH_DEFAULT = -1; //!< デフォルトの奥行き（指定なし）です。

    static const int MipCountMax = 15; //!< ミップマップのレベル数の最大値です。
    static const int MipCountDefault = 0; //!< デフォルトのミップマップのレベル数（指定なし）です。

    //! @brief ミップマップ生成フィルタを表す列挙型です。
    enum MipGenFilter
    {
        MIP_GEN_POINT,  //!< ポイントサンプルです。
        MIP_GEN_LINEAR, //!< リニア補間です。
        MIP_GEN_CUBIC,  //!< キュービック補間です。
        MIP_GEN_INVALID = -2, //!< 無効なミップマップ生成フィルタです。
        MIP_GEN_DEFAULT = -1  //!< デフォルトのミップマップ生成フィルタ（指定なし）です。
    };

    static const int COMP_SEL_INVALID = -2; //!< 無効な成分選択です。
    static const int COMP_SEL_DEFAULT = -1; //!< デフォルトの成分選択（指定なし）です。

    static const int WEIGHTED_COMPRESS_DEFAULT = -1; //!< デフォルトの圧縮時の誤差の重み付け（指定なし）です。
    static const int WEIGHTED_COMPRESS_DISABLE =  0; //!< 圧縮時の誤差の重み付けを無効にします。
    static const int WEIGHTED_COMPRESS_ENABLE  =  1; //!< 圧縮時の誤差の重み付けを有効にします。

    //! @brief ディザモードを表す列挙型です。
    enum Dither
    {
        Dither_Disable, //!< 使用しません。
        Dither_Enable,  //!< 使用します（エンコーダーが対応していれば）。
        Dither_Auto,    //!< ゆるやかにカラーが変化する画像なら使用します（エンコーダーが対応していれば）。
        Dither_Invalid  //!< 無効なディザモードです。
    };

    static const int SWIZZLE_DEFAULT = -1; //!< デフォルトのスウィズル値（指定なし）です。
    static const int SWIZZLE_MIN = 0; //!< スウィズル値の最小値です。
    static const int SWIZZLE_MAX = 7; //!< スウィズル値の最大値です。

    //! @brief キューブマップのフェースを表す列挙型です。
    enum CubeFace
    {
        CUBE_FACE_PX,   //!< +X 面です。
        CUBE_FACE_NX,   //!< -X 面です。
        CUBE_FACE_PY,   //!< +Y 面です。
        CUBE_FACE_NY,   //!< -Y 面です。
        CUBE_FACE_PZ,   //!< +Z 面です。
        CUBE_FACE_NZ,   //!< -Z 面です。
        CUBE_FACE_COUNT //!< フェースの最大数です。
    };

    static const std::string COMMENT_TEXT_DEFAULT; //!< デフォルトの編集用コメント文（指定なし）です。

    //! @brief 入力画像ファイルのパス配列です。
    //!        キューブマップの 6 面を個別に指定する場合は前面の入力画像ファイルのパスを設定します。
    RStringArray m_InputPaths;

    std::string m_OutputPath; //!< 出力ファイルのパスです。
    std::string m_OutputOriginalPath; //!< 元画像ファイルの出力パスです（空文字なら出力しない）（非公開）。
    bool m_OutputsBntx; //!< bntx ファイルを出力するなら true です。
    bool m_IsImageToBntx; //!< 画像ファイルから直接 bntx ファイルに変換するなら true です。
    std::string m_MergePath; //!< マージする ftx ファイル（拡張子 ftxa または ftxb）のパスです。
    bool m_DisablesMergeError; //!< マージする ftx ファイルが存在しない場合のエラーを抑制するなら true です。
    bool m_NoEncoding; //!< 再エンコードしないで入力ファイルのエンコード済みデータを出力するなら true です。
    GpuEncoding m_GpuEncoding; //!< GPU エンコーディングモードです。
    std::string m_DccPreset; //!< DCC プリセット名です。
    std::string m_Hint; //!< ヒント情報です。
    int m_LinearFlag; //!< リニア変換フラグです。
    int m_Dimension; //!< 次元（FtxDimension）です。
    int m_Format; //!< テクスチャーフォーマット（FtxFormat または SpecialFormat）です。
    int m_Depth; //!< 奥行きです。
    int m_MipCount; //!< ミップマップのレベル数です。
    MipGenFilter m_MipGenFilter; //!< ミップマップ生成フィルタです。
    int m_CompSel; //!< 成分選択です。
    std::string m_Quality; //!< エンコード品質です（空文字ならデフォルト）（非公開）。
    int m_WeightedCompress; //!< 圧縮時の RGB 成分の誤差の重み付けです。
    Dither m_Dither; //!< ディザモードです（非公開）。
    bool m_UsesEncoderProcess; //!< サーバープロセスでエンコーダーを使用するなら true です（非公開）。
    int m_InitialSwizzle; //!< 初期スウィズル値です。
    std::string m_CubeBackPath;   //!< キューブマップ後の入力画像ファイルのパスです。
    std::string m_CubeLeftPath;   //!< キューブマップ左の入力画像ファイルのパスです。
    std::string m_CubeRightPath;  //!< キューブマップ右の入力画像ファイルのパスです。
    std::string m_CubeTopPath;    //!< キューブマップ上の入力画像ファイルのパスです。
    std::string m_CubeBottomPath; //!< キューブマップ下の入力画像ファイルのパスです。

    //! @brief ミップマップの入力画像を個別指定する場合のパスです。
    //!        レベル 0 のパスは使用されません。
    std::string m_MipPaths[MipCountMax];

    //! @brief キューブマップのミップマップの入力画像を個別指定する場合のパスです。
    //!        各フェースのレベル 0 のパスは使用されません。
    std::string m_CubeMipPathss[CUBE_FACE_COUNT][MipCountMax];

    std::string m_CommentText; //!< 編集用コメント文です。
    RCropRect m_CropRect; //!< 切り抜き矩形です。
    int m_ReductionLevel; //!< 縮小レベル数です。
    int m_ResizeW; //!< サイズ変更後の幅のピクセル数です（0 なら指定なし）。
    int m_ResizeH; //!< サイズ変更後の高さのピクセル数です（0 なら指定なし）。
    float m_ResizePercentW; //!< サイズ変更後の幅のパーセンテージです（0 なら指定なし）。
    float m_ResizePercentH; //!< サイズ変更後の高さのパーセンテージです（0 なら指定なし）。
    ResizeFilter m_ResizeFilter; //!< サイズ変更フィルタです。
    bool m_ResizesInLinear; //!< リニア空間で縮小計算するなら true です。
    bool m_IgnoresAlpha; //!< 入力画像のアルファ成分を無視するなら true です。
    bool m_AdjustsXpaRgb; //!< 透明ピクセルの RGB 成分を調整するなら true です。
    std::string m_ReplacePaths[R_RGBA_COUNT]; //!< チャンネル差し替え画像ファイルのパス配列です。
    std::string m_AfterReplacePaths[R_RGBA_COUNT]; //!< 入力ファイルのサイズ変更後にチャンネル差し替えする画像ファイルのパス配列です。
    int m_ReplaceChanIdxs[R_RGBA_COUNT]; //!< チャンネル差し替え元のチャンネルインデックス配列です。
    bool m_IsSilent; //!< 処理内容の表示を抑制するなら true です。
    std::string m_ProjectRootPath; //!< プロジェクトのルートフォルダーのパスです。
    bool m_DisablesFileInfo; //!< <file_info> を更新しないなら true です。
    bool m_DisablesOriginalImage; //!< <original_image> を出力しないなら true です。
    bool m_DisablesMemoryPool; //!< bntx ファイルでメモリープールを無効にするなら true です。
    std::string m_BinTileMode; //!< bntx ファイルのタイルモードです（空文字なら linear）。
    FtxTileOptimize m_TileOptimize; //!< タイリング最適化です。
    int m_TileSizeThreshold; //!< タイリング最適化時のデータ削減率のしきい値です。
    bool m_IsSparseTiled; //!< スパーステクスチャー向けタイリングなら true です。
    bool m_IsTgaRle; //!< TGA ファイルをランレングス圧縮するなら true です。
    bool m_ForcesDdsDx10; //!< DDS ファイルを強制的に DX10 形式で出力するなら true です。

    // 以下、すべて非公開オプションです。
    bool m_SupportsFtx; //!< ftx ファイルに対応するなら true です。
    bool m_DisablesSizeError; //!< ftx ファイル出力時のサイズエラー判定を無効にするなら true です。
    bool m_IsCSource; //!< C 言語のソース形式で出力するなら true です。

    std::string m_ToolName; //!< ftx ファイルに出力するツール名です。

    //! @brief テンポラリーフォルダーに出力してからコピーする場合の最終出力フォルダーのパスです。
    //!        実際のファイルは m_OutputPath で指定されたパスに出力されますが、
    //!        <original_image> の original_path は m_FinalFolderPath からの相対パスになります。
    //!        空文字なら m_OutputPath を最終出力フォルダーとみなします。
    std::string m_FinalFolderPath;

public:
    //! コンストラクターです。
    FtxOpt()
    : m_OutputsBntx(false),
      m_IsImageToBntx(false),
      m_DisablesMergeError(false),
      m_NoEncoding(false),
      m_GpuEncoding(GpuEncoding_Enable),
      m_DccPreset(DCC_PRESET_DEFAULT),
      m_Hint(HINT_DEFAULT),
      m_LinearFlag(LINEAR_DEFAULT),
      m_Dimension(DIM_DEFAULT),
      m_Format(FORMAT_DEFAULT),
      m_Depth(DEPTH_DEFAULT),
      m_MipCount(MipCountDefault),
      m_MipGenFilter(MIP_GEN_DEFAULT),
      m_CompSel(COMP_SEL_DEFAULT),
      m_WeightedCompress(WEIGHTED_COMPRESS_DEFAULT),
      m_Dither(Dither_Auto),
      m_UsesEncoderProcess(false),
      m_InitialSwizzle(SWIZZLE_DEFAULT),
      m_CommentText(COMMENT_TEXT_DEFAULT),
      m_ReductionLevel(0),
      m_ResizeW(0),
      m_ResizeH(0),
      m_ResizePercentW(0.0f),
      m_ResizePercentH(0.0f),
      m_ResizeFilter(ResizeFilter_Default),
      m_ResizesInLinear(true),
      m_IgnoresAlpha(false),
      m_AdjustsXpaRgb(false),
      m_IsSilent(false),
      m_DisablesFileInfo(false),
      m_DisablesOriginalImage(false),
      m_DisablesMemoryPool(false),
      m_TileOptimize(FtxTileOptimize_Performance),
      m_TileSizeThreshold(FtxDefaultTileSizeThreshold),
      m_IsSparseTiled(false),
      m_IsTgaRle(true),
      m_ForcesDdsDx10(false),
      m_SupportsFtx(false),
      m_DisablesSizeError(false),
      m_IsCSource(false)
    {
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            m_ReplaceChanIdxs[iRgba] = 0;
        }
    }

    //! デストラクターです。
    virtual ~FtxOpt()
    {
    }

    //! 初期化します。
    void Init()
    {
        *this = FtxOpt();
    }

    //! @brief ジョブオプションを設定します。
    //!
    //! @param[in] options ジョブオプションのユニコード文字列配列です。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus SetJobOptions(const wchar_t* options[]);

    //! @brief 入力ファイルパス配列を設定します。
    //!
    //! @param[in] paths 入力ファイルパスのユニコード文字列配列です。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus SetInputPaths(const wchar_t* paths[]);

    //! @brief キューブマップ個別指定、ミップマップ個別指定を含めた
    //!        全入力ファイルのパス配列を取得します。
    //!
    //! @param[out] allInputPaths 全入力ファイルのパス配列を格納します。
    //!
    void GetAllInputPaths(RStringArray& allInputPaths) const;

    //! @brief キューブマップ個別指定なら true を返します。
    bool IsCubeMapSeparete() const
    {
        return (
            !m_CubeBackPath.empty()  ||
            !m_CubeLeftPath.empty()  ||
            !m_CubeRightPath.empty() ||
            !m_CubeTopPath.empty()   ||
            !m_CubeBottomPath.empty());
    }

    //! @brief キューブマップなら true を返します。
    bool IsCubeMap() const;

    //! @brief ミップマップ画像個別指定なら true を返します。
    //!
    //! @param[in] numMips 実際に生成するミップマップのレベル数です。
    //!
    //! @return ミップマップ画像個別指定なら true を返します。
    //!
    bool IsMipImageSpecified(const int numMips) const;

    //! @brief サイズ変更が指定されているなら true を返します。
    //!
    //! @return サイズ変更が指定されているなら true を返します。
    //!
    bool IsResizeSpecified() const
    {
        return (m_ReductionLevel != 0    ||
                m_ResizeW != 0           ||
                m_ResizeH != 0           ||
                m_ResizePercentW != 0.0f ||
                m_ResizePercentH != 0.0f);
    }

    //! @brief チャンネル差し替えが指定されているなら true を返します。
    //!
    //! @return チャンネル差し替えが指定されているなら true を返します。
    //!
    bool IsReplaceSpecified() const
    {
        for (int rgbaIdx = 0; rgbaIdx < R_RGBA_COUNT; ++rgbaIdx)
        {
            if (!m_ReplacePaths[rgbaIdx].empty())
            {
                return true;
            }
        }
        return false;
    }

    //! @brief サイズ変更後のチャンネル差し替えが指定されているなら true を返します。
    //!
    //! @return サイズ変更後のチャンネル差し替えが指定されているなら true を返します。
    //!
    bool IsAfterReplaceSpecified() const
    {
        for (int rgbaIdx = 0; rgbaIdx < R_RGBA_COUNT; ++rgbaIdx)
        {
            if (!m_AfterReplacePaths[rgbaIdx].empty())
            {
                return true;
            }
        }
        return false;
    }

    //! @brief 再エンコードが必要なら true を返します。
    //!
    //! @return 再エンコードが必要なら true を返します。
    //!
    bool NeedsEncoding() const;

    //! @brief 入力ファイルのデータをそのまま出力可能なら true を返します。
    //!
    //! @return 入力ファイルのデータをそのまま出力可能なら true を返します。
    //!
    bool IsPassThrough() const;

    //! @brief 中間ファイルの文字列からミップマップ生成フィルタを取得します。
    //!
    //! @param[in] str ミップマップ生成フィルタを表す文字列です。
    //! @param[in] linearIfInvalid 文字列が不正な場合にリニアを返すなら true です。
    //!
    //! @return ミップマップ生成フィルタを返します。
    //!
    static MipGenFilter GetMipGenFilterFromString(const std::string& str, const bool linearIfInvalid);

    //! @brief ftx 変換オプションのミップマップ生成フィルタから中間ファイルの文字列を取得します。
    //!
    //! @param[in] mipGenFilter ミップマップ生成フィルタです。
    //!
    //! @return ミップマップ生成フィルタを表す文字列を返します。
    //!
    static std::string GetMipGenFilterString(const MipGenFilter mipGenFilter);

    //! @brief 入力ファイルと出力ファイルの情報を表示します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] dstPath 出力ファイルのパスです。
    //! @param[in] dstFinalPath 最終出力ファイルのパスです。
    //!
    void DisplayFileInfo(
        std::ostream& os,
        const std::string& dstPath,
        const std::string& dstFinalPath
    ) const;
};

//! @brief 変換オプション配列の定義です。
typedef std::vector<FtxOpt> FtxOptArray;

//=============================================================================
//! @brief 変換ユーティリティーのクラスです。
//=============================================================================
class ConverterUtility
{
private:
    REncoder* m_pEncoder; //!< エンコーダーへのポインタです。
    WicUtility* m_pWicUtility; //!< WIC ユーティリティーのポインタです。
    ROpenExr* m_pOpenExr; //!< OpenEXR ユーティリティーのポインタです。
    NXUtility* m_pNXUtility; //!< NX ユーティリティーへのポインタです。

public:
    //! コンストラクターです。
    ConverterUtility()
    {
        m_pEncoder = nullptr;
        m_pWicUtility = nullptr;
        m_pOpenExr = nullptr;
        m_pNXUtility = nullptr;
    }

    //! デストラクターです。
    virtual ~ConverterUtility()
    {
    }

    //! エンコーダーへのポインタを取得します。
    REncoder* GetEncoder() const
    {
        return m_pEncoder;
    }

    //! エンコーダーへのポインタを設定します。
    void SetEncoder(REncoder* pEncoder)
    {
        m_pEncoder = pEncoder;
    }

    //! WIC ユーティリティーへのポインタを取得します。
    WicUtility* GetWicUtility() const
    {
        return m_pWicUtility;
    }

    //! WIC ユーティリティーへのポインタを設定します。
    void SetWicUtility(WicUtility* pWicUtility)
    {
        m_pWicUtility = pWicUtility;
    }

    //! OpenEXR ユーティリティーへのポインタを取得します。
    ROpenExr* GetOpenExr() const
    {
        return m_pOpenExr;
    }

    //! OpenEXR ユーティリティーへのポインタを設定します。
    void SetOpenExr(ROpenExr* pOpenExr)
    {
        m_pOpenExr = pOpenExr;
    }

    //! NX ユーティリティーへのポインタを取得します。
    NXUtility* GetNXUtility() const
    {
        return m_pNXUtility;
    }

    //! NX ユーティリティーへのポインタを設定します。
    void SetNXUtility(NXUtility* pNXUtility)
    {
        m_pNXUtility = pNXUtility;
    }
};

//=============================================================================
//! @brief 画像用追加属性のクラスです。
//=============================================================================
class ExtraImageAttr
{
public:
    //! 値の型を表す列挙型です。
    enum Type
    {
        Type_Integer,       //!< 整数です。
        Type_IntegerArray,  //!< 整数配列です。
        Type_Float,         //!< 浮動小数点数です。
        Type_FloatArray,    //!< 浮動小数点数配列です。
        Type_String,        //!< 文字列です。
        Type_StringArray,   //!< 文字列配列です。
        Type_ByteArray      //!< バイト配列です。
    };

public:
    //! @brief 既定のコンストラクタです。
    //!
    ExtraImageAttr()
    : m_Type(Type_IntegerArray)
    {
    }

    //! @brief 整数用のコンストラクタです。
    //!
    //! @param[in] value 整数値です。
    //!
    explicit ExtraImageAttr(const int value)
    : m_Type(Type_Integer)
    {
        m_IntegerValues.push_back(value);
    }

    //! @brief 整数配列用のコンストラクタです。
    //!
    //! @param[in] array 整数配列です。
    //!
    explicit ExtraImageAttr(const RIntArray& array)
    : m_Type(Type_IntegerArray),
      m_IntegerValues(array)
    {
    }

    //! @brief 浮動小数点数用のコンストラクタです。
    //!
    //! @param[in] value 浮動小数点数値です。
    //!
    explicit ExtraImageAttr(const float value)
    : m_Type(Type_Float)
    {
        m_FloatValues.push_back(value);
    }

    //! @brief 浮動小数点数配列用のコンストラクタです。
    //!
    //! @param[in] array 浮動小数点数配列です。
    //!
    explicit ExtraImageAttr(const RFloatArray& array)
    : m_Type(Type_FloatArray),
      m_FloatValues(array)
    {
    }

    //! @brief 文字列用のコンストラクタです。
    //!
    //! @param[in] value 文字列値です。
    //!
    explicit ExtraImageAttr(const std::string& value)
    : m_Type(Type_String)
    {
        m_StringValues.push_back(value);
    }

    //! @brief 文字列配列用のコンストラクタです。
    //!
    //! @param[in] array 文字列配列です。
    //!
    explicit ExtraImageAttr(const RStringArray& array)
    : m_Type(Type_StringArray),
      m_StringValues(array)
    {
    }

    //! @brief バイト配列用のコンストラクタです。
    //!
    //! @param[in] array バイト配列です。
    //!
    explicit ExtraImageAttr(const std::vector<uint8_t>& array)
    : m_Type(Type_ByteArray),
      m_ByteValues(array)
    {
    }

    //! @brief バイト配列用のコンストラクタです。
    //!
    //! @param[in] array バイト配列の先頭へのポインタです。
    //! @param[in] bytes バイト数です。
    //!
    ExtraImageAttr(const void* array, const size_t bytes)
    : m_Type(Type_ByteArray)
    {
        const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(array);
        m_ByteValues = std::vector<uint8_t>(pSrc, pSrc + bytes);
    }

    //! @brief 値の型を取得します。
    //!
    //! @return 値の型を返します。
    //!
    Type GetType() const
    {
        return m_Type;
    }

    //! @brief 整数値を取得します。
    //!
    //! @return 整数値を返します。整数型でなければ 0 を返します。
    //!
    int GetInteger() const
    {
        return (m_Type == Type_Integer && !m_IntegerValues.empty()) ? m_IntegerValues[0] : 0;
    }

    //! @brief 整数配列を取得します。
    //!
    //! @return 整数配列を返します。整数配列型でなければ空の配列を返します。
    //!
    const RIntArray& GetIntegerArray() const
    {
        return m_IntegerValues;
    }

    //! @brief 浮動小数点数値を取得します。
    //!
    //! @return 浮動小数点数値を返します。浮動小数点数型でなければ 0 を返します。
    //!
    float GetFloat() const
    {
        return (m_Type == Type_Float && !m_FloatValues.empty()) ? m_FloatValues[0] : 0.0f;
    }

    //! @brief 浮動小数点数配列を取得します。
    //!
    //! @return 浮動小数点数配列を返します。浮動小数点数配列型でなければ空の配列を返します。
    //!
    const RFloatArray& GetFloatArray() const
    {
        return m_FloatValues;
    }

    //! @brief 文字列値を取得します。
    //!
    //! @return 文字列値を返します。文字列型でなければ空文字を返します。
    //!
    std::string GetString() const
    {
        return (m_Type == Type_String && !m_StringValues.empty()) ? m_StringValues[0] : "";
    }

    //! @brief 文字列配列を取得します。
    //!
    //! @return 文字列配列を返します。文字列配列型でなければ空の配列を返します。
    //!
    const RStringArray& GetStringArray() const
    {
        return m_StringValues;
    }

    //! @brief バイト配列を取得します。
    //!
    //! @return バイト配列を返します。バイト配列型でなければ空の配列を返します。
    //!
    const std::vector<uint8_t>& GetByteArray() const
    {
        return m_ByteValues;
    }

    //! @brief ストリームに出力します（動作確認用）。
    //!
    friend std::ostream& operator<<(std::ostream& os, const ExtraImageAttr& attr);

protected:
    Type m_Type; //!< 値の型です。
    RIntArray m_IntegerValues; //!< 整数配列です。
    RFloatArray m_FloatValues; //!< 浮動小数点数配列です。
    RStringArray m_StringValues; //!< 文字列配列です。
    std::vector<uint8_t> m_ByteValues; //!< バイト配列です。
};

//! @brief 文字列と画像用追加属性のマップの定義です。
typedef std::map<std::string, ExtraImageAttr> ExtraImageAttrMap;

//=============================================================================
// 他のクラスから参照するため、元画像配列を先に定義します。
//=============================================================================
class ROriginalImage;

//! @brief 元画像配列の定義です。
typedef std::vector<ROriginalImage> ROriginalImageArray;

//=============================================================================
//! @brief 画像のクラスです。
//=============================================================================
class RImage
{
public:
    static const int WidthHeightMin = 1; //!< 画像の幅と高さの最小値です。
    static const int WidthHeightMax = 16384; //!< 画像の幅と高さの最大値です。
    static const int MipCountMax = 15; //!< ミップマップのレベル数の最大値です。

    static const int CUBE_HC_COUNT_H = 4; //!< 水平十字キューブマップの水平方向のフェース数です。
    static const int CUBE_HC_COUNT_V = 3; //!< 水平十字キューブマップの垂直方向のフェース数です。
    static const int CUBE_VC_COUNT_H = 3; //!< 垂直十字キューブマップの水平方向のフェース数です。
    static const int CUBE_VC_COUNT_V = 4; //!< 垂直十字キューブマップの垂直方向のフェース数です。

    //! @brief 透明モードを表す列挙型です。
    enum TransparencyMode
    {
        OPA,    //!< 不透明です（アルファがすべて 255）。
        MASK,   //!< 抜きありです（アルファが 0 または 255）。
        XLU,    //!< 半透明です（1 以上 254 以下のアルファが存在）。
    };

    //! @brief キューブマップのフェースを表す列挙型です。
    enum CubeFace
    {
        CUBE_FACE_PX,   //!< +X 面です。
        CUBE_FACE_NX,   //!< -X 面です。
        CUBE_FACE_PY,   //!< +Y 面です。
        CUBE_FACE_NY,   //!< -Y 面です。
        CUBE_FACE_PZ,   //!< +Z 面です。
        CUBE_FACE_NZ,   //!< -Z 面です。
        CUBE_FACE_COUNT //!< フェースの最大数です。
    };

    static const int CubeHcPositions[CUBE_FACE_COUNT][2]; //!< 水平十字キューブマップの各フェースの位置配列です。
    static const int CubeVcPositions[CUBE_FACE_COUNT][2]; //!< 垂直十字キューブマップの各フェースの位置配列です。

    static const int TGA_HEADER_SIZE = 0x12; //!< TGA ファイルのヘッダサイズです。

    //! @brief リードフラグを表す列挙型です。
    enum ReadFlag
    {
        ReadFlag_Original = 0x1 << 0, //!< 元画像データを取得します。
        ReadFlag_Encoded  = 0x1 << 1, //!< エンコードされたデータを取得します。
        ReadFlag_Both = ReadFlag_Original | ReadFlag_Encoded
    };

protected:
    std::string m_Name; //!< 名前です。
    std::string m_FilePath; //!< ファイルパスです。
    std::string m_DccPreset; //!< DCC プリセット名です。
    std::string m_Hint; //!< ヒント情報です。
    int m_LinearFlag; //!< リニア変換フラグです。
    int m_Dimension; //!< 次元（FtxDimension）です。
    int m_Format; //!< テクスチャーフォーマット（FtxFormat または FtxOpt::SpecialFormat）です。
    int m_ImageW; //!< 幅です。
    int m_ImageH; //!< 高さです。
    int m_ImageD; //!< 奥行きです。
    int m_MipCount; //!< ミップマップのレベル数です。
    FtxOpt::MipGenFilter m_MipGenFilter; //!< ミップマップ生成フィルタです。
    int m_CompSel; //!< 成分選択です。
    int m_WeightedCompress; //!< 圧縮時の RGB 成分の誤差の重み付けです。
    FtxTileMode m_TileMode; //!< ftx の タイルモードです。
    bool m_UsesTextureLayout; //!< テクスチャーレイアウトを使用するなら true です。
    FtxTextureLayout m_TextureLayout; //!< テクスチャーレイアウトです。
    FtxTileOptimize m_TileOptimize; //!< タイリング最適化です。
    int m_TileSizeThreshold; //!< タイリング最適化時のデータ削減率のしきい値です。
    bool m_IsSparseTiled; //!< スパーステクスチャー向けタイリングなら true です。
    int m_InitialSwizzle; //!< ftx の初期スウィズル値です。
    int m_Swizzle; //!< ftx のスウィズル値です。
    int m_Alignment; //!< ftx のアライメントです。
    int m_Pitch; //!< ftx のピッチです。
    size_t m_LevelOffsets[MipCountMax]; //!< 各ミップマップデータのイメージデータ先頭からのオフセット配列です。
    std::string m_OriginalImageHash; //!< ftx の元画像のハッシュ文字列です。
    bool m_HasAlpha; //!< アルファ成分を持つなら true です。
    bool m_IsFloat; //!< イメージデータのデータ型が浮動小数点数なら true、整数なら false です。
    bool m_IsEncoded; //!< イメージデータがエンコードされたデータなら true、エンコードされていないデータなら false です。
    uint8_t* m_pImageData; //!< イメージデータへのポインタです。RGBA の順に左上から右下へ格納されます。
    size_t m_ImageDataSize; //!< イメージデータ全体（ベース + ミップマップ）のサイズ（バイト数）です。
    size_t m_MipDataSize; //!< イメージデータ中のミップマップデータのサイズ（バイト数）です。
    size_t m_ExtraDataSize; //!< イメージデータ後に格納した追加データのサイズ（バイト数）です。
    uint8_t* m_pEncodedData; //!< エンコードされたデータへのポインタです。
    size_t m_EncodedDataSize; //!< エンコードされたデータのサイズです。
    bool m_IsDecodedOriginal; //!< ftx リード時にエンコードされたデータをデコードしてイメージデータとしたなら true です。
    size_t m_BinaryDataOffset; //!< テクスチャーバイナリーファイル内のデータのオフセットです。
    RStringArray m_OriginalPaths; //!< ftx のオリジナルパス配列です。
    std::string m_CreateInfo; //!< ftx の <create> 要素の内容です。
    std::string m_ModifyInfo; //!< ftx の <modify> 要素の内容です。
    RUserDataArray m_UserDatas; //!< ftx のユーザーデータ配列です。
    std::string m_CommentLabel; //!< 編集用コメントラベルです。
    std::string m_CommentCol; //!< 編集用コメントカラー文字列です。
    std::string m_CommentText; //!< 編集用コメント文です。
    std::string m_ToolData; //!< ftx の <tool_data> 要素の内容です。
    std::string m_UserToolData; //!< ftx の <user_tool_data> 要素の内容です。
    ExtraImageAttrMap m_ExtraAttrMap; //!< 追加属性のマップです。
    int m_WorkIndex; //!< 作業用インデックスです（バイナリー変換で使用）。

public:
    //! @brief テクスチャーレイアウトをクリアします。
    void ClearTextureLayout()
    {
        memset(&m_TextureLayout, 0x00, sizeof(m_TextureLayout));
    }

    //! @brief 各ミップマップデータのイメージデータ先頭からのオフセット配列を初期化します。
    void ClearLevelOffsets()
    {
        memset(m_LevelOffsets, 0x00, sizeof(m_LevelOffsets));
    }

    //! @brief m_Name 以外の全パラメーターを初期化します。
    void InitializeParams()
    {
        m_FilePath.clear();
        m_DccPreset = FtxOpt::DCC_PRESET_DEFAULT;
        m_Hint = FtxOpt::HINT_DEFAULT;
        m_LinearFlag = FtxOpt::LINEAR_DEFAULT;
        m_Dimension = FtxOpt::DIM_DEFAULT;
        m_Format = FtxOpt::FORMAT_INVALID;
        m_ImageW = 0;
        m_ImageH = 0;
        m_ImageD = 1;
        m_MipCount = 1;
        m_MipGenFilter = FtxOpt::MIP_GEN_DEFAULT;
        m_CompSel = FtxOpt::COMP_SEL_DEFAULT;
        m_WeightedCompress = FtxOpt::WEIGHTED_COMPRESS_DEFAULT;
        m_TileMode = FtxTileMode_Linear;
        m_UsesTextureLayout = false;
        m_TileOptimize = FtxTileOptimize_Performance;
        m_TileSizeThreshold = FtxDefaultTileSizeThreshold;
        m_IsSparseTiled = false;
        m_InitialSwizzle = FtxOpt::SWIZZLE_DEFAULT;
        m_Swizzle = 0;
        m_Alignment = 1;
        m_Pitch = 0;
        m_OriginalImageHash.clear();
        m_HasAlpha = false;
        m_IsFloat = false;
        m_IsEncoded = false;
        m_ImageDataSize = 0;
        m_MipDataSize = 0;
        m_ExtraDataSize = 0;
        m_EncodedDataSize = 0;
        m_IsDecodedOriginal = false;
        m_BinaryDataOffset = 0;
        m_OriginalPaths.clear();
        m_CreateInfo.clear();
        m_ModifyInfo.clear();
        m_UserDatas.clear();
        m_CommentLabel.clear();
        m_CommentCol.clear();
        m_CommentText.clear();
        m_ToolData.clear();
        m_UserToolData.clear();
        m_WorkIndex = 0;

        ClearTextureLayout();
        ClearLevelOffsets();
        DeleteAllExtraAttr();
    }

    //! @brief コンストラクターです。
    NN_IMPLICIT RImage(const char* name = nullptr)
    : m_pImageData(nullptr),
      m_pEncodedData(nullptr)
    {
        if (name != nullptr)
        {
            m_Name = name;
        }
        InitializeParams();
    }

    //! @brief メモリーを解放し、m_Name 以外の全パラメーターを初期化します。
    void Clear()
    {
        RFreeAndClearArray(m_pImageData);
        RFreeAndClearArray(m_pEncodedData);
        InitializeParams();
    }

    //! @brief デストラクターです。
    virtual ~RImage()
    {
        Clear();
    }

    //! エンコードされたデータをイメージデータに移動します。
    void MoveEncodedDataToImageData()
    {
        if (m_pEncodedData != nullptr)
        {
            if (m_pImageData != nullptr)
            {
                delete[] m_pImageData;
            }
            m_IsEncoded = true;
            m_pImageData = m_pEncodedData;
            m_ImageDataSize = m_EncodedDataSize;
            m_MipDataSize = (m_MipCount >= 2) ? m_ImageDataSize - m_LevelOffsets[1] : 0;
            m_ExtraDataSize = 0;
            m_pEncodedData = nullptr;
            m_EncodedDataSize = 0;
        }
    }

    //! @brief エンコードされたデータをデコードしてイメージデータに格納します。
    //!
    //! @param[in,out] pEncoder エンコーダーへのポインタです。
    //! @param[in] decodesMip ミップマップもデコードするなら true、
    //!                       レベル 0 のみデコードするなら false を指定します。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool DecodeEncodedDataToImageData(REncoder* pEncoder, const bool decodesMip);

    //! @brief ファイルから画像データをリードします。
    //!        現在 TGA、PNG、DDS、EXR、ftx ファイルに対応しています。
    //!
    //! @param[in] filePath ファイルのパスです。
    //! @param[in] readFlags リードフラグです。
    //! @param[in] removesAlpha アルファ成分を削除するなら true です。
    //! @param[in] convUtil 変換ユーティリティーです。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus ReadFile(
        const std::string& filePath,
        const int readFlags,
        const bool removesAlpha,
        const ConverterUtility& convUtil
    );

    //! @brief TGA ファイルのデータをデコードします。
    //!
    //! @param[in] fileBuf ファイルをリードしたバッファです。
    //! @param[in] fileSize ファイルのサイズです。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus DecodeTga(const uint8_t* fileBuf, const size_t fileSize);

    //! @brief TGA ファイルを出力します。
    //!        unorm_8_8_8_8 フォーマットのみ対応しています。
    //!
    //! @param[in] filePath TGA ファイルのパスです。
    //! @param[in] isRLE ランレングス圧縮するなら true を指定します。
    //! @param[in] isCubeVC 垂直十字キューブマップにするなら true、
    //!                     水平十字キューブマップにするなら false を指定します。
    //! @param[in] outputsMip ミップマップを並べて出力するなら true を指定します。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus OutputTgaFile(
        const std::string& filePath,
        const bool isRLE,
        const bool isCubeVC,
        const bool outputsMip
    ) const;

    //! @brief PNG ファイルのデータをデコードします。
    //!
    //! @param[in,out] pWicUtility WIC ユーティリティーのポインタです。
    //! @param[in] fileBuf ファイルをリードしたバッファです。
    //! @param[in] fileSize ファイルのサイズです。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus DecodePng(WicUtility* pWicUtility, const uint8_t* fileBuf, const size_t fileSize);

    //! @brief PNG ファイルを出力します。
    //!
    //! @param[in,out] pWicUtility WIC ユーティリティーのポインタです。
    //! @param[in] filePath PNG ファイルのパスです。
    //! @param[in] isCubeVC 垂直十字キューブマップにするなら true、
    //!                     水平十字キューブマップにするなら false を指定します。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus OutputPngFile(
        WicUtility* pWicUtility,
        const std::string& filePath,
        const bool isCubeVC
    ) const;

    //! @brief DDS ファイルのデータをデコードします。
    //!        キューブマップのイメージデータは右手座標系で +X、-X、+Y、-Y、+Z、-Z の順になるように取得します。
    //!
    //! @param[in,out] pEncoder エンコーダーへのポインタです。
    //! @param[in] fileBuf ファイルをリードしたバッファです。
    //! @param[in] fileSize ファイルのサイズです。
    //! @param[in] readFlags リードフラグです。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus DecodeDds(
        REncoder* pEncoder,
        const uint8_t* fileBuf,
        const size_t fileSize,
        const int readFlags
    );

    //! @brief DDS ファイルを出力します。
    //!
    //! @param[in] filePath DDS ファイルのパスです。
    //! @param[in] outputsEncoded エンコードされたデータを出力するなら true を指定します。
    //! @param[in] outputsMip ミップマップを出力するなら true を指定します。
    //! @param[in] swapsCubeZ キューブマップの場合に Z 方向のイメージデータをスワップして出力するなら true を指定します。
    //!                       イメージデータが右手座標系で +X、-X、+Y、-Y、+Z、-Z の順なら true を指定します。
    //! @param[in] forcesDx10 DX9 形式で出力可能な画像でも強制的に DX10 形式で出力するなら true を指定します。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus OutputDdsFile(
        const std::string& filePath,
        const bool outputsEncoded,
        const bool outputsMip,
        const bool swapsCubeZ,
        const bool forcesDx10
    ) const;

    //! @brief EXR ファイルのデータをデコードします。
    //!
    //! @param[in,out] pOpenExr OpenEXR ユーティリティーのポインタです。
    //! @param[in] fileBuf ファイルをリードしたバッファです。
    //! @param[in] fileSize ファイルのサイズです。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus DecodeExr(ROpenExr* pOpenExr, const uint8_t* fileBuf, const size_t fileSize);

    //! @brief EXR ファイルを出力します。
    //!        float_32_32_32_32 フォーマットのみ対応しています。
    //!
    //! @param[in,out] pOpenExr OpenEXR ユーティリティーのポインタです。
    //! @param[in] filePath EXR ファイルのパスです。
    //! @param[in] isCubeVC 垂直十字キューブマップにするなら true、
    //!                     水平十字キューブマップにするなら false を指定します。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus OutuptExrFile(
        ROpenExr* pOpenExr,
        const std::string& filePath,
        const bool isCubeVC
    ) const;

    //! @brief ftx ファイルのデータをデコードします。
    //!
    //! @param[in,out] pEncoder エンコーダーへのポインタです。
    //! @param[in] fileBuf ファイルをリードしたバッファです。
    //! @param[in] fileSize ファイルのサイズです。
    //! @param[in] isBinary バイナリー形式の ftx ファイルなら true を指定します。
    //! @param[in] readFlags リードフラグです。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus DecodeFtx(
        REncoder* pEncoder,
        const uint8_t* fileBuf,
        const size_t fileSize,
        const bool isBinary,
        const int readFlags
    );

    //! @brief ftx ファイルを出力します。
    //!        内部でメモリーを解放する機能があるため const 関数ではありません。
    //!
    //! @param[out] pOutputData ftx ファイルのデータを格納するバッファへのポインタです。
    //!                         ファイルに出力する場合は nullptr を指定します。
    //! @param[in,out] pOriginalImages 元画像配列へのポインタです。
    //! @param[in] filePath ftx ファイルのパスです。
    //! @param[in] opt ftx 変換オプションです。
    //! @param[in] toolVersionStr ツールバージョン文字列です。
    //! @param[in] freesMemory メモリーの最大使用量を減らすために内部で画像と元画像の
    //!                        メモリーを解放するなら true です。
    //!
    RStatus OutputFtxFile(
        std::string* pOutputData,
        ROriginalImageArray* pOriginalImages,
        const std::string& filePath,
        const FtxOpt& opt,
        const std::string& toolVersionStr,
        const bool freesMemory
    );

    //! @brief ビットマップデータから画像データをリードします。
    //!
    //! @param[in] pData ビットマップデータです。
    //!                  1 ピクセルあたり 4 または 16 バイトで、RGBA の順に格納します。画像の左上が先頭です。
    //!                  キューブマップ（6 面個別指定）の場合、+X、-X、+Y、-Y、+Z、-Z 面の順に格納します。
    //!                  ビットマップデータのサイズは width * height * depth * (4 または 16) となります。
    //! @param[in] dimension 次元（FtxDimension）です。DIM_DEFAULT は指定なしを意味します。
    //! @param[in] width ビットマップの幅です。
    //! @param[in] height ビットマップの高さです。
    //! @param[in] depth ビットマップの深さです。
    //! @param[in] hasAlpha アルファ成分を持つなら true です。
    //! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
    //! @param[in] filePaths 画像ファイルのパス配列です。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus ReadBitmap(
        const void* pData,
        const int dimension,
        const int width,
        const int height,
        const int depth,
        const bool hasAlpha,
        const bool isFloat,
        const RStringArray& filePaths
    );

    //! @brief データ型を変更します。元のデータ型と同じなら何もしません。
    //!
    //! @param[in] isFloat データ型を浮動小数点数にするなら true、整数にするなら false です。
    //! @param[in] changesLevelOffsets 各ミップマップデータのイメージデータ先頭からの
    //!                                オフセット配列も変更するなら true です。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus ChangeDataType(const bool isFloat, const bool changesLevelOffsets);

    //! @brief 画像の色空間を sRGB からリニアに変換します。
    //!
    //! @param[in] dstLinearFlag リニア変換フラグです。
    //!
    void ConvertToLinear(const int dstLinearFlag);

    //! @brief 画像の色空間をリニアから sRGB に変換します。
    //!
    //! @param[in] dstLinearFlag リニア変換フラグです。
    //!
    void ConvertToSrgb(const int dstLinearFlag);

    //! @brief イメージデータを符号なし 8 ビットから符号付き 8 ビットに変換します。
    //!        イメージデータが浮動小数点数型なら何もしません。
    //!
    //! @param[in] limitsMinus127 下限を -127 (0x81) とするなら true、-128 (0x80) とするなら false です。
    //!
    void ConvertToSigned8(const bool limitsMinus127);

    //! @brief イメージデータを符号付き 8 ビットから符号なし 8 ビットに変換します。
    //!        イメージデータが浮動小数点数型なら何もしません。
    //!
    void ConvertToUnsigned8();

    //! @brief A 成分を削除します。
    //!
    void RemoveAlpha();

    //! @brief A 成分を G 成分にコピーします。
    //!
    void CopyAToG();

    //! @brief 指定した成分選択に応じてイメージデータの RGBA 成分をスワップします。
    //!        成分選択に応じて A 成分存在フラグも設定します。
    //!
    //! @param[in] compSel 成分選択です。
    //!
    void SwapRgbaComponent(const int compSel);

    //! @brief チャンネルを差し替えます。
    //!
    //! @param[in] dstChanIdx 差し替え先のチャンネルインデックスです。
    //! @param[in] srcImg 差し替え元の画像です。
    //! @param[in] dstChanIdx 差し替え元のチャンネルインデックスです。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus ReplaceChannel(
        const int dstChanIdx,
        const RImage& srcImg,
        const int srcChanIdx
    );

    //! @brief キューブマップのイメージデータの +Z 方向と -Z 方向のデータを交換します。
    //!        キューブマップおよびキューブマップ配列以外なら何もしません。
    //!
    void SwapCubeFaceZ();

    //! @brief 切り抜きます。
    //!
    //! @param[in] cropRect 切り抜き矩形です。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus Crop(const RCropRect& cropRect);

    //! @brief サイズ変更後の幅／高さ／奥行きを取得します。
    //!
    //! @param[out] pDstW 変更後の幅を格納します（nullptr なら格納しません）。
    //! @param[out] pDstH 変更後の高さを格納します（nullptr なら格納しません）。
    //! @param[out] pDstD 変更後の奥行きを格納します（nullptr なら格納しません）。
    //! @param[in] reductionLevel 縮小レベル数です。
    //! @param[in] resizeW サイズ変更後の幅のピクセル数です（0 なら指定なし）。
    //! @param[in] resizeH サイズ変更後の高さのピクセル数です（0 なら指定なし）。
    //! @param[in] percentW サイズ変更後の幅のパーセンテージです（0 なら指定なし）。
    //! @param[in] percentH サイズ変更後の高さのパーセンテージです（0 なら指定なし）。
    //!
    //! @return サイズが変わるなら true、変わらないなら false を返します。
    //!
    bool GetResizedWhd(
        int* pDstW,
        int* pDstH,
        int* pDstD,
        const int reductionLevel,
        const int resizeW,
        const int resizeH,
        const float percentW,
        const float percentH
    ) const;

    //! @brief サイズ変更します。
    //!
    //! @param[in] dstW 変更後の幅です。
    //! @param[in] dstH 変更後の高さです。
    //! @param[in] dstD 変更後の奥行きです。
    //! @param[in] linearFlag リニアー変換フラグです。
    //! @param[in] resizeFilter サイズ変更フィルターです。
    //! @param[in] ignoresXpaRgb 透明ピクセルの RGB 成分を補間計算から除外するなら true です。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus Resize(
        const int dstW,
        const int dstH,
        const int dstD,
        const int linearFlag,
        const ResizeFilter resizeFilter,
        const bool ignoresXpaRgb
    );

    //! 情報を表示します。
    void DisplayInfo(std::ostream& os) const;

    //! ダンプ出力します（動作確認用）。
    void DumpToFile(const std::string& filePath) const;

    //! @brief NX 向けのタイリング変換をします。
    //!
    //! @param[in,out] pNXUtility NX ユーティリティーへのポインタです。
    //! @param[in] isTiling NX 向けのタイリング変換をするなら true、タイリング変換を解除するなら false です。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus ConvertNXTiling(NXUtility* pNXUtility, const bool isTiling);

    //-----------------------------------------------------------------------------
    // アクセサ

    //! 名前を取得します。
    const std::string& GetName() const
    {
        return m_Name;
    }

    //! 名前を設定します。
    void SetName(const std::string& name)
    {
        m_Name = name;
    }

    //! 画像ファイルのパスを取得します。
    const std::string& GetFilePath() const
    {
        return m_FilePath;
    }

    //! 画像ファイルのパスを設定します。
    void SetFilePath(const std::string& path)
    {
        m_FilePath = path;
    }

    //! DCC プリセット名を返します。
    const std::string& GetDccPreset() const
    {
        return m_DccPreset;
    }

    //! DCC プリセット名を設定します。
    void SetDccPreset(const std::string& dccPreset)
    {
        m_DccPreset = dccPreset;
    }

    //! ヒント情報を返します。
    const std::string& GetHint() const
    {
        return m_Hint;
    }

    //! ヒント情報を設定します。
    void SetHint(const std::string& hint)
    {
        m_Hint = hint;
    }

    //! リニア変換フラグを返します。
    int GetLinearFlag() const
    {
        return m_LinearFlag;
    }

    //! リニア変換フラグを設定します。
    void SetLinearFlag(const int linearFlag)
    {
        m_LinearFlag = linearFlag;
    }

    //! 次元を返します。
    int GetDimension() const
    {
        return m_Dimension;
    }

    //! 次元を設定します。
    void SetDimension(const int dimension)
    {
        m_Dimension = dimension;
    }

    //! 配列テクスチャーなら true を返します。
    bool IsArray() const
    {
        return RIsArrayDimension(static_cast<FtxDimension>(m_Dimension));
    }

    //! キューブマップまたはキューブマップ配列なら true を返します。
    bool IsCubeMap() const
    {
        return RIsCubMapDimension(static_cast<FtxDimension>(m_Dimension));
    }

    //! フォーマットを返します。
    int GetFormat() const
    {
        return m_Format;
    }

    //! フォーマットを設定します。
    void SetFormat(const int format)
    {
        m_Format = format;
    }

    //! 幅を返します。
    int GetImageW() const
    {
        return m_ImageW;
    }

    //! 幅を設定します。
    void SetImageW(const int imageW)
    {
        m_ImageW = imageW;
    }

    //! 高さを返します。
    int GetImageH() const
    {
        return m_ImageH;
    }

    //! 高さを設定します。
    void SetImageH(const int imageH)
    {
        m_ImageH = imageH;
    }

    //! 奥行きを返します。
    int GetImageD() const
    {
        return m_ImageD;
    }

    //! 奥行きを設定します。
    void SetImageD(const int imageD)
    {
        m_ImageD = imageD;
    }

    //! ミップマップのレベル数を返します。
    int GetMipCount() const
    {
        return m_MipCount;
    }

    //! ミップマップのレベル数を設定します。
    void SetMipCount(const int mipCount)
    {
        m_MipCount = mipCount;
    }

    //! ミップマップ生成フィルタを返します。
    FtxOpt::MipGenFilter GetMipGenFilter() const
    {
        return m_MipGenFilter;
    }

    //! ミップマップ生成フィルタを設定します。
    void SetMipGenFilter(const FtxOpt::MipGenFilter mipGenFilter)
    {
        m_MipGenFilter = mipGenFilter;
    }

    //! 成分選択を返します。
    int GetCompSel() const
    {
        return m_CompSel;
    }

    //! 成分選択を設定します。
    void SetCompSel(const int compSel)
    {
        m_CompSel = compSel;
    }

    //! 圧縮時の RGB 成分の誤差の重み付けを返します。
    int GetWeightedCompress() const
    {
        return m_WeightedCompress;
    }

    //! 成分選択を設定します。
    void SetWeightedCompress(const int weightedCompress)
    {
        m_WeightedCompress = weightedCompress;
    }

    //! ftx のタイルモードを返します。
    FtxTileMode GetTileMode() const
    {
        return m_TileMode;
    }

    //! ftx のタイルモードを設定します。
    void SetTileMode(const FtxTileMode tileMode)
    {
        m_TileMode = tileMode;
    }

    //! テクスチャーレイアウトを使用するなら true を返します。
    bool UsesTextureLayout() const
    {
        return m_UsesTextureLayout;
    }

    //! テクスチャーレイアウトを使用するかどうか設定します。
    void SetUsesTextureLayout(const bool usesTextureLayout)
    {
        m_UsesTextureLayout = usesTextureLayout;
    }

    //! テクスチャーレイアウトを返します。
    FtxTextureLayout GetTextureLayout() const
    {
        return m_TextureLayout;
    }

    //! テクスチャーレイアウトを設定します。
    void SetTextureLayout(const FtxTextureLayout& textureLayout)
    {
        m_TextureLayout = textureLayout;
    }

    //! タイリング最適化を返します。
    FtxTileOptimize GetTileOptimize() const
    {
        return m_TileOptimize;
    }

    //! タイリング最適化を設定します。
    void SetTileOptimize(const FtxTileOptimize tileOptimize)
    {
        m_TileOptimize = tileOptimize;
    }

    //! タイリング最適化時のデータ削減率のしきい値を返します。
    int GetTileSizeThreshold() const
    {
        return m_TileSizeThreshold;
    }

    //! タイリング最適化時のデータ削減率のしきい値を設定します。
    void SetTileSizeThreshold(const int tileSizeThreshold)
    {
        m_TileSizeThreshold = tileSizeThreshold;
    }

    //! スパーステクスチャー向けタイリングなら true を返します。
    bool IsSparseTiled() const
    {
        return m_IsSparseTiled;
    }

    //! スパーステクスチャー向けタイリングフラグを設定します。
    void SetSparseTiled(const bool isSparseTiled)
    {
        m_IsSparseTiled = isSparseTiled;
    }

    //! ftx の初期スウィズル値を返します。
    int GetInitialSwizzle() const
    {
        return m_InitialSwizzle;
    }

    //! ftx の初期スウィズル値を設定します。
    void SetInitialSwizzle(const int initialSwizzle)
    {
        m_InitialSwizzle = initialSwizzle;
    }

    //! ftx のスウィズル値を返します。
    int GetSwizzle() const
    {
        return m_Swizzle;
    }

    //! ftx のスウィズル値を設定します。
    void SetSwizzle(const int swizzle)
    {
        m_Swizzle = swizzle;
    }

    //! ftx のアライメントを返します。
    int GetAlignment() const
    {
        return m_Alignment;
    }

    //! ftx のアライメントを設定します。
    void SetAlignment(const int alignment)
    {
        m_Alignment = alignment;
    }

    //! ftx のピッチを返します。
    int GetPitch() const
    {
        return m_Pitch;
    }

    //! ftx のピッチを設定します。
    void SetPitch(const int pitch)
    {
        m_Pitch = pitch;
    }

    //! 指定レベルのイメージデータ先頭からのオフセットを返します。
    size_t GetLevelOffset(const int level) const
    {
        return m_LevelOffsets[level];
    }

    //! 指定レベルのイメージデータ先頭からのオフセットを設定します。
    void SetLevelOffset(const int level, const size_t offset)
    {
        m_LevelOffsets[level] = offset;
    }

    //! 各ミップマップデータのイメージデータ先頭からのオフセット配列を設定します。
    //!
    //! @param[in] levelOffsets 各ミップマップデータのイメージデータ先頭からのオフセット配列です。
    //!
    void SetLevelOffsets(const std::vector<size_t>& levelOffsets)
    {
        ClearLevelOffsets();
        const int mipCount = RMin(static_cast<int>(levelOffsets.size()), MipCountMax);
        for (int level = 0; level < mipCount; ++level)
        {
            m_LevelOffsets[level] = levelOffsets[level];
        }
    }

    //! ftx の元画像のハッシュ文字列を返します。
    const std::string& GetOriginalImageHash() const
    {
        return m_OriginalImageHash;
    }

    //! ftx の元画像のハッシュ文字列を設定します。
    void SetOriginalImageHash(const std::string& hash)
    {
        m_OriginalImageHash = hash;
    }

    //! アルファ成分が存在するなら true を返します。
    bool HasAlpha() const
    {
        return m_HasAlpha;
    }

    //! アルファ成分存在フラグを設定します。
    void SetHasAlpha(const bool hasAlpha)
    {
        m_HasAlpha = hasAlpha;
    }

    //! イメージデータのデータ型が浮動小数点数なら true、整数なら false を返します。
    bool IsFloat() const
    {
        return m_IsFloat;
    }

    //! イメージデータのデータ型の浮動小数点数フラグを設定します。
    void SetIsFloat(const bool isFloat)
    {
        m_IsFloat = isFloat;
    }

    //! イメージデータがエンコードされたデータなら true、エンコードされていないデータなら false を返します。
    bool IsEncoded() const
    {
        return m_IsEncoded;
    }

    //! イメージデータのエンコード済みフラグを設定します。
    void SetIsEncoded(const bool isEncoded)
    {
        m_IsEncoded = isEncoded;
    }

    //! イメージデータへのポインタを返します。
    const uint8_t* GetImageData() const
    {
        return m_pImageData;
    }

    //! イメージデータへのポインタを返します。
    uint8_t* GetImageData()
    {
        return m_pImageData;
    }

    //! イメージデータへのポインタを設定します。
    void SetImageData(uint8_t* pImageData)
    {
        m_pImageData = pImageData;
    }

    //! イメージデータデータ全体のサイズを返します。
    size_t GetImageDataSize() const
    {
        return m_ImageDataSize;
    }

    //! イメージデータデータ全体のサイズを設定します。
    void SetImageDataSize(const size_t imageDataSize)
    {
        m_ImageDataSize = imageDataSize;
    }

    //! ミップマップデータのサイズを返します。
    size_t GetMipDataSize() const
    {
        return m_MipDataSize;
    }

    //! ミップマップデータのサイズを設定します。
    void SetMipDataSize(const size_t mipDataSize)
    {
        m_MipDataSize = mipDataSize;
    }

    //! 追加データのサイズを返します。
    size_t GetExtraDataSize() const
    {
        return m_ExtraDataSize;
    }

    //! 追加データのサイズを設定します。
    void SetExtraDataSize(const size_t extraDataSize)
    {
        m_ExtraDataSize = extraDataSize;
    }

    //! エンコードされたへのポインタを返します。
    const uint8_t* GetEncodedData() const
    {
        return m_pEncodedData;
    }

    //! エンコードされたデータのサイズを返します。
    size_t GetEncodedDataSize() const
    {
        return m_EncodedDataSize;
    }

    //! ftx リード時にエンコードされたデータをデコードしてイメージデータとしたなら true を返します。
    bool IsDecodedOriginal() const
    {
        return m_IsDecodedOriginal;
    }

    //! ftx リード時にエンコードされたデータをデコードしてイメージデータとしたフラグを設定します。
    void SetIsDecodedOriginal(const bool isDecodedOriginal)
    {
        m_IsDecodedOriginal = isDecodedOriginal;
    }

    //! テクスチャーバイナリーファイル内のデータのオフセットを返します。
    size_t GetBinaryDataOffset() const
    {
        return m_BinaryDataOffset;
    }

    //! テクスチャーバイナリーファイル内のデータのオフセットを設定します。
    void SetBinaryDataOffset(const size_t binaryDataOffset)
    {
        m_BinaryDataOffset = binaryDataOffset;
    }

    //! ftx のオリジナルパス配列を返します。
    const RStringArray& GetOriginalPaths() const
    {
        return m_OriginalPaths;
    }

    //! ftx のオリジナルパスをフルパスで返します。
    const std::string GetOriginalFullPath(const int iPath) const
    {
        std::string fullPath;
        if (iPath < static_cast<int>(m_OriginalPaths.size()))
        {
            fullPath = m_OriginalPaths[iPath];
            if (!RIsFullFilePath(fullPath))
            {
                // ftx からの相対パスならフルパスに変換します。
                fullPath = RGetFullFilePath(
                    RGetFolderFromFilePath(m_FilePath) + "/" + fullPath, true);
            }
        }
        return fullPath;
    }

    //! ftx の <create> 要素の内容を返します。
    const std::string& GetCreateInfo() const
    {
        return m_CreateInfo;
    }

    //! ftx の <modify> 要素の内容を返します。
    const std::string& GetModifyInfo() const
    {
        return m_ModifyInfo;
    }

    //! ftx のユーザーデータを持つなら true を返します。
    bool HasUserData() const
    {
        return !m_UserDatas.empty();
    }

    //! ftx のユーザーデータ配列を返します。
    const RUserDataArray& GetUserDatas() const
    {
        return m_UserDatas;
    }

    //! ftx のユーザーデータ配列を返します。
    RUserDataArray& GetUserDatas()
    {
        return m_UserDatas;
    }

    //! ftx のユーザーデータ配列を設定します。
    void SetUserDatas(const RUserDataArray& userDatas)
    {
        m_UserDatas = userDatas;
    }

    //! 編集用コメントラベルを返します。
    const std::string& GetCommentLabel() const
    {
        return m_CommentLabel;
    }

    //! 編集用コメントラベルを設定します。
    void SetCommentLabel(const std::string& commentLabel)
    {
        m_CommentLabel = commentLabel;
    }

    //! 編集用コメントカラー文字列を返します。
    const std::string& GetCommentCol() const
    {
        return m_CommentCol;
    }

    //! 編集用コメントカラー文字列を設定します。
    void SetCommentCol(const std::string& commentCol)
    {
        m_CommentCol = commentCol;
    }

    //! 編集用コメント文を返します。
    const std::string& GetCommentText() const
    {
        return m_CommentText;
    }

    //! 編集用コメントカラー文を設定します。
    void SetCommentText(const std::string& commentText)
    {
        m_CommentText = commentText;
    }

    //! ftx の <tool_data> 要素の内容を返します。
    const std::string& GetToolData() const
    {
        return m_ToolData;
    }

    //! ftx の <tool_data> 要素の内容を設定します。
    void SetToolData(const std::string& toolData)
    {
        m_ToolData = toolData;
    }

    //! ftx の <user_tool_data> 要素の内容を返します。
    const std::string& GetUserToolData() const
    {
        return m_UserToolData;
    }

    //! ftx の <user_tool_data> 要素の内容を設定します。
    void SetUserToolData(const std::string& userToolData)
    {
        m_UserToolData = userToolData;
    }

    //! @brief 追加属性のマップを取得します。
    //!
    //! @return 追加属性のマップを返します。
    //!
    const ExtraImageAttrMap& GetExtraAttrMap() const
    {
        return m_ExtraAttrMap;
    }

    //! @brief 追加属性へのポインタを取得します。
    //!
    //! @param[in] name 追加属性名です。
    //!
    //! @return 追加属性へのポインタ（存在しなければ nullptr）を返します。
    //!
    const ExtraImageAttr* GetExtraAttr(const std::string& name) const
    {
        auto attrIt = m_ExtraAttrMap.find(name);
        return (attrIt != m_ExtraAttrMap.end()) ? &attrIt->second : nullptr;
    }

    //! @brief 追加属性のマップを設定します。
    //!
    //! @param[in] map 追加属性のマップです。
    //!
    void SetExtraAttrMap(const ExtraImageAttrMap& map)
    {
        m_ExtraAttrMap = map;
    }

    //! @brief 追加属性を設定します。
    //!
    //! @param[in] name 追加属性名です。
    //! @param[in] attr 追加属性です。
    //!
    void SetExtraAttr(const std::string& name, const ExtraImageAttr& attr)
    {
        m_ExtraAttrMap[name] = attr;
    }

    //! @brief 追加属性をすべて削除します。
    //!
    void DeleteAllExtraAttr()
    {
        m_ExtraAttrMap.clear();
    }

    //! @brief 追加属性を削除します。
    //!
    //! @param[in] name 追加属性名です。
    //!
    void DeleteExtraAttr(const std::string& name)
    {
        m_ExtraAttrMap.erase(name);
    }

    //! 作業用インデックスを返します。
    int GetWorkIndex() const
    {
        return m_WorkIndex;
    }

    //! 作業用インデックスを設定します。
    void SetWorkIndex(const int workIndex)
    {
        m_WorkIndex = workIndex;
    }

    //-----------------------------------------------------------------------------
    // static メンバ関数

    //! @brief 水平十字キューブマップの幅と高さなら true を返します。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return 水平十字キューブマップの幅と高さなら true を返します。
    //!
    static bool IsCubeHCWH(const int imageW, const int imageH);

    //! @brief 垂直十字キューブマップの幅と高さなら true を返します。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return 垂直十字キューブマップの幅と高さなら true を返します。
    //!
    static bool IsCubeVCWH(const int imageW, const int imageH);

private:
    RImage(const RImage& other);
    RImage& operator=(const RImage& other);
};

//=============================================================================
//! @brief 元画像のクラスです。
//=============================================================================
class ROriginalImage // ROriginalImageT
{
public:
    //! @brief キューブ面の方向を表す列挙型です。
    enum Face
    {
        NONE,       //!< キューブマップ以外です。
        POSITIVE_X, //!< +X 面です。
        NEGATIVE_X, //!< -X 面です。
        POSITIVE_Y, //!< +Y 面です。
        NEGATIVE_Y, //!< -Y 面です。
        POSITIVE_Z, //!< +Z 面です。
        NEGATIVE_Z  //!< -Z 面です。
    };

    //! @brief フォーマットを表す列挙型です。
    enum Format { RGBA8, RGB8, RGBA32F, RGB32F };

    //! @brief 3d の場合は奥行きのインデックス、
    //!        1d_array、2d_array、cube_array の場合は配列のインデックスです。
    int m_SliceIndex;

    Face m_Face; //!< キューブ面の方向です。
    Format m_Format; //!< フォーマットです。
    int m_Width; //!< 幅です。
    int m_Height; //!< 高さです。
    std::string m_OriginalPath; //!< ftx ファイルのフォルダーを基準とした元画像ファイルへの相対パスです。

    // internal
    std::string m_DccPreset; //!< DCC プリセット名です。
    std::string m_Hint; //!< ヒント情報です。
    int m_LinearFlag; //!< リニア変換フラグです。
    int m_OriginalFormat; //!< 元画像ファイルのフォーマット（FtxFormat または FtxOpt::SpecialFormat）です。
    int m_MipCount; //!< ミップマップのレベル数です。
    FtxOpt::MipGenFilter m_MipGenFilter; //!< ミップマップ生成フィルタです。
    int m_CompSel; //!< 成分選択です。
    int m_WeightedCompress; //!< 圧縮時の RGB 成分の誤差の重み付けです。
    int m_InitialSwizzle; //!< 初期スウィズル値です。
    std::string m_CreateInfo; //!< 元画像ファイルが ftx ファイルの場合の <create> 要素の内容です。
    std::string m_ModifyInfo; //!< 元画像ファイルが ftx ファイルの場合の <modify> 要素の内容です。
    uint8_t* m_pImageData; //!< イメージデータへのポインタです。RGBA の順に左上から右下へ格納されます。

    // 【注意】メンバを追加した際は代入演算子の処理も追加！

public:
    //! @brief コンストラクターです。
    //!
    //! @param[in] sliceIndex 奥行きまたは配列のインデックスです。
    //! @param[in] face キューブ面の方向です。
    //! @param[in] width 幅です。
    //! @param[in] height 高さです。
    //! @param[in] originalPath 元画像ファイルへの相対パスです。
    //! @param[in] pImageData イメージデータへのポインタです。RGBA の順に左上から右下へ格納されている必要があります。
    //! @param[in] pSourceImage 元画像データへのポインタです。
    //!
    ROriginalImage(
        const int sliceIndex = 0,
        const Face face = NONE,
        const int width = 0,
        const int height = 0,
        const char* originalPath = "",
        const uint8_t* pImageData = nullptr,
        const RImage* pSourceImage = nullptr
    )
    : m_SliceIndex(sliceIndex),
      m_Face(face),
      m_Format(RGBA8),
      m_Width(width),
      m_Height(height),
      m_OriginalPath(originalPath),
      m_DccPreset(FtxOpt::DCC_PRESET_DEFAULT),
      m_Hint(FtxOpt::HINT_DEFAULT),
      m_LinearFlag(FtxOpt::LINEAR_DEFAULT),
      m_OriginalFormat(FtxOpt::FORMAT_INVALID),
      m_MipCount(1),
      m_MipGenFilter(FtxOpt::MIP_GEN_DEFAULT),
      m_CompSel(FtxOpt::COMP_SEL_DEFAULT),
      m_WeightedCompress(FtxOpt::WEIGHTED_COMPRESS_DEFAULT),
      m_InitialSwizzle(FtxOpt::SWIZZLE_DEFAULT),
      m_pImageData(nullptr)
    {
        //cerr << "ROriginalImage()" << endl;

        //-----------------------------------------------------------------------------
        // 元画像データから情報を取得します。
        if (pSourceImage != nullptr)
        {
            const RImage& rimg = *pSourceImage;
            m_Format           = (rimg.IsFloat()) ?
                (rimg.HasAlpha() ? RGBA32F : RGB32F) :
                (rimg.HasAlpha() ? RGBA8   : RGB8  );
            m_DccPreset        = rimg.GetDccPreset();
            m_Hint             = rimg.GetHint();
            m_LinearFlag       = rimg.GetLinearFlag();
            m_OriginalFormat   = rimg.GetFormat();
            m_MipCount         = rimg.GetMipCount();
            m_MipGenFilter     = rimg.GetMipGenFilter();
            m_CompSel          = rimg.GetCompSel();
            m_WeightedCompress = rimg.GetWeightedCompress();
            m_InitialSwizzle   = rimg.GetInitialSwizzle();
            m_CreateInfo       = rimg.GetCreateInfo();
            m_ModifyInfo       = rimg.GetModifyInfo();
        }

        //-----------------------------------------------------------------------------
        // イメージデータを確保してコピーします。
        if (pImageData != nullptr)
        {
            const size_t colBytes = (IsFloat()) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
            const size_t size = colBytes * m_Width * m_Height;
            m_pImageData = new uint8_t[size];
            memcpy(m_pImageData, pImageData, size);
        }
    }

    //! コピーコンストラクターです。
    ROriginalImage(const ROriginalImage& other)
    : m_pImageData(nullptr)
    {
        //cerr << "ROriginalImage(copy)" << endl;
        operator=(other);
    }

    //! メモリーを解放します。
    void Clear()
    {
        if (m_pImageData != nullptr)
        {
            delete[] m_pImageData;
            m_pImageData = nullptr;
        }
    }

    //! デストラクターです。
    virtual ~ROriginalImage()
    {
        //cerr << "~ROriginalImage()" << endl;
        Clear();
    }

    //! 代入演算子です。
    ROriginalImage& operator=(const ROriginalImage& other)
    {
        //cerr << "ROriginalImage::operator=" << endl;
        if (this != &other)
        {
            Clear();

            m_SliceIndex       = other.m_SliceIndex;
            m_Face             = other.m_Face;
            m_Format           = other.m_Format;
            m_Width            = other.m_Width;
            m_Height           = other.m_Height;
            m_OriginalPath     = other.m_OriginalPath;
            m_DccPreset        = other.m_DccPreset;
            m_Hint             = other.m_Hint;
            m_LinearFlag       = other.m_LinearFlag;
            m_OriginalFormat   = other.m_OriginalFormat;
            m_MipCount         = other.m_MipCount;
            m_MipGenFilter     = other.m_MipGenFilter;
            m_CompSel          = other.m_CompSel;
            m_WeightedCompress = other.m_WeightedCompress;
            m_InitialSwizzle   = other.m_InitialSwizzle;
            m_CreateInfo       = other.m_CreateInfo;
            m_ModifyInfo       = other.m_ModifyInfo;

            if (other.m_pImageData != nullptr)
            {
                const size_t colBytes = (IsFloat()) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
                const size_t size = colBytes * m_Width * m_Height;
                m_pImageData = new uint8_t[size];
                memcpy(m_pImageData, other.m_pImageData, size);
            }
        }
        return *this;
    }

    //! アルファ成分を持つなら true を返します。
    bool HasAlpha() const
    {
        return (m_Format == RGBA8 || m_Format == RGBA32F);
    }

    //! データ型が浮動小数点数なら true、整数なら false を返します。
    bool IsFloat() const
    {
        return (m_Format == RGBA32F || m_Format == RGB32F);
    }

    //! 出力サイズ（バイト数）を返します。
    size_t GetSize() const
    {
        const size_t colBytes =
            (m_Format == RGBA8  ) ? R_RGBA_BYTES       :
            (m_Format == RGB8   ) ? R_RGB_BYTES        :
            (m_Format == RGBA32F) ? R_RGBA_FLOAT_BYTES :
            (m_Format == RGB32F ) ? R_RGB_FLOAT_BYTES  :
            R_RGBA_BYTES;
        return colBytes * m_Width * m_Height;
    }

    //! イメージデータバッファのサイズ（バイト数）を返します。
    size_t GetImageDataSize() const
    {
        const size_t colBytes = (IsFloat()) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
        return colBytes * m_Width * m_Height;
    }

    //! @brief 出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] tabCount <original_image> 要素のインデントに必要なタブの数です。
    //! @param[in] index <original_image_array> 内でのインデックスです。
    //! @param[in] streamIndex データ列のインデックスです。
    //!
    void Output(
        std::ostream& os,
        const int tabCount,
        const int index,
        const int streamIndex
    ) const;


    //! @brief データ列配列にデータ列を追加します。
    //!
    //! @param[in,out] dataStreams データ列配列です。
    //!
    void AppendStream(RDataStreamArray& dataStreams) const;

    //! グレースケールなら true を返します。
    bool IsGray() const;

    //! 透明モードを返します。
    RImage::TransparencyMode GetTransparencyMode() const;
};

//=============================================================================
//! @brief マージ情報のクラスです。
//=============================================================================
class RMergeInfo
{
public:
    bool m_IsEnable; //!< マージが有効なら true です。
    std::string m_FilePath; //!< マージするファイルのパスです。
    std::string m_Hint; //!< ヒント情報です。
    int m_LinearFlag; //!< リニア変換フラグです。
    int m_Dimension; //!< 次元（FtxDimension）です。
    int m_Format; //!< テクスチャーフォーマット（FtxFormat または FtxOpt::SpecialFormat）です。
    int m_MipCount; //!< ミップマップのレベル数です。
    FtxOpt::MipGenFilter m_MipGenFilter; //!< ミップマップ生成フィルタです。
    int m_CompSel; //!< 成分選択です。
    int m_WeightedCompress; //!< 圧縮時の RGB 成分の誤差の重み付けです。
    int m_InitialSwizzle; //!< 初期スウィズル値です。
    int m_Width; //!< 幅です。
    int m_Height; //!< 高さです。
    RUserDataArray m_UserDatas; //!< ユーザーデータ配列です。
    std::string m_CommentLabel; //!< 編集用コメントラベルです。
    std::string m_CommentCol; //!< 編集用コメントカラー文字列です。
    std::string m_CommentText; //!< 編集用コメント文です。
    std::string m_ToolData; //!< <tool_data> 要素の内容です。
    std::string m_UserToolData; //!< <user_tool_data> 要素の内容です。

public:
    //! コンストラクターです。
    RMergeInfo()
    : m_IsEnable(false),
      m_Dimension(FtxOpt::DIM_DEFAULT),
      m_Hint(FtxOpt::HINT_DEFAULT),
      m_LinearFlag(FtxOpt::LINEAR_DEFAULT),
      m_Format(FtxOpt::FORMAT_DEFAULT),
      m_MipCount(FtxOpt::MipCountDefault),
      m_MipGenFilter(FtxOpt::MIP_GEN_DEFAULT),
      m_CompSel(FtxOpt::COMP_SEL_DEFAULT),
      m_WeightedCompress(FtxOpt::WEIGHTED_COMPRESS_DEFAULT),
      m_InitialSwizzle(FtxOpt::SWIZZLE_DEFAULT),
      m_Width(1),
      m_Height(1)
    {
    }

    //! クリアします。
    void Clear()
    {
        *this = RMergeInfo();
    }

    //! @brief ファイルをリードしてマージ情報を取得します。
    //!
    //! @param[in] ファイルのパスです。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus ReadFile(const std::string& filePath);

private:
    //! @brief ftx ファイルのデータからマージ情報を取得します。
    //!
    //! @param[in] fileBuf ファイルをリードしたバッファです。
    //! @param[in] fileSize ファイルのサイズです。
    //! @param[in] isBinary バイナリー形式の ftx ファイルなら true を指定します。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus ParseFtx(
        const uint8_t* fileBuf,
        const size_t fileSize,
        const bool isBinary
    );
};

//=============================================================================
//! @brief テクスチャーコンバーターの基底クラスです。
//=============================================================================
class TexConverterBase
{
protected:
    //-----------------------------------------------------------------------------
    // DLL 関連
    HINSTANCE m_hDLL; //!< テクスチャーコンバーターの DLL のインスタンスハンドルです。

    //-----------------------------------------------------------------------------
    // バージョン情報関連
    int m_VersionMajor; //!< メジャーバージョンです。
    int m_VersionMinor; //!< マイナーバージョンです。
    int m_VersionMicro; //!< マイクロバージョンです。
    int m_VersionBugfix; //!< バグフィックスバージョンです。

    //-----------------------------------------------------------------------------
    // 外部ユーティリティー関連
    REncoder m_Encoder; //!< エンコーダーです。
    NXUtility m_NXUtility; //!< NX ユーティリティーです。
    ROpenExr m_OpenExr; //!< OpenEXR ユーティリティーです。
    WicUtility m_WicUtility; //!< WIC ユーティリティーです。
    ConverterUtility m_ConverterUtility; //!< 変換ユーティリティーです。

    //-----------------------------------------------------------------------------
    // オプション関連
    bool m_SupportsFtx; //!< ftx ファイルに対応するなら true です。
    bool m_IsSilent; //!< 処理内容の表示を抑制するなら true を指定します。
    bool m_IsCSource; //!< C 言語のソース形式で出力するなら true を指定します。
    std::string m_DefaultToolName; //!< ftx ファイルに出力するツール名のデフォルト値です。
    std::string m_ToolName; //!< ftx ファイルに出力するツール名を指定します。
    std::string m_FinalFolderPath; //!< 最終出力フォルダーのパス指定です。空文字なら出力フォルダーに出力します。

    FtxOpt m_Opt; //!< ftx 変換オプションです。

    //-----------------------------------------------------------------------------
    // 画像データ関連
    RImage m_WorkImage; //!< 作業用画像です。キューブマップは左手座標系で +X、-X、+Y、-Y、+Z、-Z の順です。
    ROriginalImageArray m_OriginalImages; //!< 元画像配列です。
    RMergeInfo m_MergeInfo; //!< マージ情報です。

    //-----------------------------------------------------------------------------
    // 変換先関連
    std::string m_DstPath; //!< 変換先のパスです。
    std::string m_DstFinalPath; //!< 変換先の最終出力パスです。
    FtxFormat m_DstFormat; //!< 変換先のフォーマットです。
    int m_DstMipCount; //!< 変換先のミップマップのレベル数です。
    FtxOpt::MipGenFilter m_DstMipGenFilter; //!< 変換先のミップマップ生成フィルタです。
    FtxCompSel m_DstCompSel; //!< 変換先の成分選択です。
    bool m_DstWeightedCompress; //!< 圧縮時の RGB 成分の誤差の重み付けです。
    std::string m_DstDccPreset; //!< 変換先の DCC プリセット名です。
    std::string m_DstHint; //!< 変換先のヒント情報です。
    int m_DstLinearFlag; //!< 変換先のリニア変換フラグです。
    int m_DstInitialSwizzle; //!< 変換先の初期スウィズル値です。
    RUserDataArray m_DstUserDatas; //!< 変換先のユーザーデータ配列です。
    std::string m_DstCommentLabel; //!< 変換先の編集用コメントラベルです。
    std::string m_DstCommentCol; //!< 変換先の編集用コメントカラー文字列です。
    std::string m_DstCommentText; //!< 変換先の編集用コメント文です。
    std::string m_DstToolData; //!< 変換先のツールデータです。
    std::string m_DstUserToolData; //!< 変換先のユーザーツールデータです。
    std::string m_DstData; //!< 変換後のデータ（ftx ファイルの内容）です。
    bool m_ConvertsToTga; //!< TGA ファイルに変換するなら true です。
    bool m_ConvertsToPng; //!< PNG ファイルに変換するなら true です。
    bool m_ConvertsToDds; //!< DDS ファイルに変換するなら true です。
    bool m_ConvertsToExr; //!< EXR ファイルに変換するなら true です。
    bool m_ConvertsToEncodedDds; //!< エンコードされた DDS ファイルに変換するなら true です。
    bool m_ConvertsToMipDds; //!< ミップマップを持つ DDS ファイル（エンコードなし）に変換するなら true です。

    //-----------------------------------------------------------------------------
    // エラー関連
    std::string m_ErrorString; //!< エラーの Shift JIS 文字列です。
    std::wstring m_ErrorWString; //!< エラーのユニコード文字列です。

public:
    //! @brief コンストラクターです。
    //!
    //! @param[in] hDll DLL のインスタンスハンドルです。
    //! @param[in] versionMajor メジャーバージョンです。
    //! @param[in] versionMinor マイナーバージョンです。
    //! @param[in] versionMicro マイクロバージョンです。
    //! @param[in] versionBugfix バグフィックスバージョンです。
    //! @param[in] supportsFtx ftx ファイルに対応するなら true です。
    //!
    TexConverterBase(
        HINSTANCE hDll,
        const int versionMajor,
        const int versionMinor,
        const int versionMicro,
        const int versionBugfix,
        const bool supportsFtx
    );

    //! デストラクターです。
    virtual ~TexConverterBase();

    //! コンバーターのバージョンを整数値で返します。
    int GetCvtrVersion() const;

    //! @brief コンバーターのバージョン文字列を取得します。
    //!
    //! @param[in] includesBugFix バグフィックスバージョンも含めるなら true を指定します。
    //!
    //! @brief コンバーターのバージョン文字列を返します。
    //!
    std::string GetConverterVersionString(const bool includesBugFix) const;

    //! コンバーターの使用方法を表示します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in] cvtrName コンバーター名です。
    //! @param[in] cmdName コンバーターのコマンド名です。
    //! @param[in] isVersionOnly バージョンのみ表示するなら true を指定します。
    //!
    void ShowUsage(
        std::ostream& os,
        const wchar_t* cvtrName,
        const wchar_t* cmdName,
        const bool isVersionOnly
    ) const;

    //! @brief 現在の環境で GPU によるエンコーディングが可能か判定します。
    //!
    //! @return GPU によるエンコーディングが可能なら true を返します。
    //!
    bool CheckGpuEncoding();

    //! @brief COM のモードを設定します。
    //!        コンバーター内で COM を初期化する場合、
    //!        ReadInputFile を呼ぶ前に呼ぶ必要があります。
    //!
    //! @param[in] initializesCom 必要になった際にコンバーター内で COM を初期化するなら true、
    //!                           外部で初期化するなら false を指定します。
    //!                           デフォルトでは false の動作です。
    //! @param[in] coInit CoInitializeEx で指定する COM の初期化オプションです。
    //!
    void SetComMode(const bool initializesCom, const int coInit);

    //! @brief 引数がグローバルオプションなら true を返します。
    //!
    //! @param[in] 引数です。
    //!
    //! @return 引数がグローバルオプションなら true を返します。
    //!
    bool IsGlobalOptionArg(const wchar_t* arg) const;

    //! @brief 指定した引数の次の引数がグローバルオプションの値なら true を返します。
    //!
    //! @param[in] 引数です。
    //!
    //! @return 次の引数がグローバルオプションの値なら true を返します。
    //!
    bool IsNextArgGlobalOptionValue(const wchar_t* arg) const;

    //! @brief 指定した引数の次の引数がジョブオプションの値なら true を返します。
    //!
    //! @param[in] 引数です。
    //!
    //! @return 次の引数がジョブオプションの値なら true を返します。
    //!
    bool IsNextArgJobOptionValue(const wchar_t* arg) const;

    //! @brief グローバルオプションを設定します。
    //!
    //! @param[in] グローバルオプションのユニコード文字列配列です。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool SetOptions(const wchar_t* options[]);

    //! @brief メモリーをクリアし、ジョブオプションを初期化します。
    //!        元画像、出力データなどをすべてクリアします。
    //!
    void Clear()
    {
        InitJobOptions();

        m_WorkImage.Clear();
        m_OriginalImages.clear();
        m_MergeInfo.Clear();

        m_DstUserDatas.clear();
        m_DstCommentLabel.clear();
        m_DstCommentCol.clear();
        m_DstCommentText.clear();
        m_DstToolData.clear();
        m_DstUserToolData.clear();
        m_DstData.clear();
    }

    //! @brief エラー文字列をクリアします。
    //!
    void ClearError()
    {
        m_ErrorString.clear();
        m_ErrorWString.clear();
    }

    //! @brief 入力ファイルをリードします。
    //!
    //! @param[in] paths 入力ファイルのパス配列です。
    //!                  キューブマップの各面のファイルのパスは options で指定します。
    //! @param[in] options ジョブオプション文字列配列です。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool ReadInputFile(const wchar_t* paths[], const wchar_t* options[]);

    //! @brief メモリー上の ftx ファイルのデータをリードします。
    //!
    //! @param[in] pData ftx ファイルのデータです。
    //! @param[in] dataSize ftx ファイルのデータサイズです。
    //! @param[in] path ftx ファイルのパスです。
    //!                 このパスの拡張子からバイナリー形式かアスキー形式か判別します。
    //! @param[in] options ジョブオプション文字列配列です。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool ReadFtxData(
        const void* pData,
        const size_t dataSize,
        const wchar_t* path,
        const wchar_t* options[]
    );

    //! @brief メモリー上のビットマップデータをリードします。
    //!
    //! @param[in] pData ビットマップデータです。
    //!                  1 ピクセルあたり 4 または 16 バイトで、RGBA の順に格納します。画像の左上が先頭です。
    //!                  キューブマップ（6 面個別指定）の場合、+X、-X、+Y、-Y、+Z、-Z 面の順に格納します。
    //!                  ビットマップデータのサイズは width * height * depth * (4 または 16) となります。
    //! @param[in] width ビットマップの幅です。
    //! @param[in] height ビットマップの高さです。
    //! @param[in] depth ビットマップの深さです。
    //!                  水平十字／垂直十字キューブマップの場合は 1 を指定します。
    //! @param[in] hasAlpha アルファ成分を持つなら true です。
    //! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
    //! @param[in] paths 入力ファイルのパス配列です。
    //! @param[in] originalPaths オリジナルファイルのパス配列です。
    //!                          キューブマップ（6 面個別指定）の場合、+X、-X、+Y、-Y、+Z、-Z 面の順に指定します。
    //!                          入力ファイルが ftx 以外の場合、paths と同じ値を指定します。
    //! @param[in] options ジョブオプション文字列配列です。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool ReadBitmapData(
        const void* pData,
        const int width,
        const int height,
        const int depth,
        const bool hasAlpha,
        const bool isFloat,
        const wchar_t* paths[],
        const wchar_t* originalPaths[],
        const wchar_t* options[]
    );

    //! @brief 変換して出力ファイルをセーブします。
    //!        リード時のジョブオプションで指定した出力パスにセーブします。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool ConvertToFile();

    //! @brief 変換して出力ファイルをメモリーに格納します。
    //!        リード時のジョブオプションで指定した出力パスによって
    //!        出力ファイル形式が決定されます。
    //!
    //! @param[out] ppData 出力ファイルを格納したメモリーのアドレスを格納します。
    //! @param[out] pDataSize 出力ファイルのサイズを格納します。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool ConvertToData(const void** ppData, size_t* pDataSize);

    //! エラーのユニコード文字列を返します。
    const wchar_t* GetErrorString() const
    {
        return m_ErrorWString.c_str();
    }

    //! エラーの Shift JIS 文字列を返します。
    const char* GetErrorShiftJisString() const
    {
        return m_ErrorString.c_str();
    }

    //! @brief 作成情報を設定します。
    //!        ビットマップデータをリードした後に必要なら呼びます。
    //!
    //! @param[in] createInfo 作成情報の XML 文字列です。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool SetCreateInfo(const wchar_t* createInfo);

    //! @brief ユーザーデータを設定します。
    //!        ビットマップデータをリードした後に必要なら呼びます。
    //!
    //! @param[in] pPackedUserData パックされたデータへのポインタです。
    //! @param[in] packedUserDataSize パックされたデータのサイズです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool SetUserData(const void* pPackedUserData, const size_t packedUserDataSize);

    //! @brief 編集用コメントラベルを設定します。
    //!        ビットマップデータをリードした後に必要なら呼びます。
    //!
    //! @param[in] pCommentLabel 編集用コメントラベルへのポインタです。文字コードは Shift-JIS です。
    //! @param[in] commentLabelSize 編集用コメントラベルのサイズです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool SetCommentLabel(const void* pCommentLabel, const size_t commentLabelSize);

    //! @brief 編集用コメントカラー文字列を設定します。
    //!        ビットマップデータをリードした後に必要なら呼びます。
    //!
    //! @param[in] pCommentCol 編集用コメントカラー文字列へのポインタです。文字コードは Shift-JIS です。
    //! @param[in] commentColSize 編集用コメントカラー文字列のサイズです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool SetCommentColor(const void* pCommentCol, const size_t commentColSize);

    //! @brief ツールデータを設定します。
    //!        ビットマップデータをリードした後に必要なら呼びます。
    //!
    //! @param[in] pToolData ツールデータ文字列へのポインタです。文字コードは Shift-JIS です。
    //! @param[in] toolDataSize ツールデータ文字列のサイズです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool SetToolData(const void* pToolData, const size_t toolDataSize);

    //! @brief ユーザーツールデータを設定します。
    //!        ビットマップデータをリードした後に必要なら呼びます。
    //!
    //! @param[in] pUserToolData ユーザーツールデータ文字列へのポインタです。文字コードは Shift-JIS です。
    //! @param[in] userToolDataSize ユーザーツールデータ文字列のサイズです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool SetUserToolData(const void* pUserToolData, const size_t userToolDataSize);

    //! @brief ftx ファイルをサポートしていない場合のエラーステータスを取得します。
    //!
    //! @param[in] filePath アクセスを試みた ftx ファイルのパスです。
    //!
    //! @return ftx ファイルをサポートしていない場合のエラーステータスを返します。
    //!
    static RStatus GetFtxUnsupportedStatus(const std::string& filePath);

protected:
    //! グローバルオプションを初期化します。
    void InitGlobalOptions()
    {
        m_IsSilent = false;
        m_IsCSource = false;
        m_ToolName = m_DefaultToolName;
        m_FinalFolderPath.clear();
    };

    //! グローバルオプションを ftx 変換オプションにコピーします。
    void CopyGlobalOptions()
    {
        m_Opt.m_SupportsFtx      = m_SupportsFtx;
        m_Opt.m_IsSilent         = m_IsSilent;
        m_Opt.m_IsCSource        = m_IsCSource;
        m_Opt.m_ToolName         = m_ToolName;
        m_Opt.m_FinalFolderPath  = m_FinalFolderPath;
    };

    //! ジョブオプションを初期化します。
    void InitJobOptions()
    {
        m_Opt.Init();
        CopyGlobalOptions();
    };

    //! 処理結果ステータスを解析します。
    bool ParseStatus(const RStatus& status)
    {
        if (!status)
        {
            m_ErrorString = status.GetMessage();
            m_ErrorWString = RGetUnicodeFromShiftJis(m_ErrorString);
        }
        return static_cast<bool>(status);
    }

    //! @brief リード前の共通処理です。
    RStatus PrepareToRead(const wchar_t* paths[], const wchar_t* options[]);

    //! @brief 入力画像を編集します。
    RStatus EditInputImage();

    //! @brief 入力画像を切り抜きます。
    RStatus CropInputImage();

    //! @brief 入力画像をサイズ変更します。
    //!
    //! @param[out] サイズ変更に使用したリニアー変換フラグを格納します。
    //!
    //! @return 処理結果を返します。
    //!
    RStatus ResizeInputImage(int* pLinearFlagForResize);

    //! @brief 入力画像の透明ピクセルの RGB 成分を調整します。
    RStatus AdjustTransparentRgb();

    //! @brief 変換先の各パラメータを決定します。
    RStatus DecideDstParam();

    //! @brief エンコードされたデータをリードします。
    RStatus ReadEncodedData();

    //! @brief 変換します。
    //!
    //! @param[in] isToFile ファイルにセーブするなら true、
    //!                     m_DstData に格納するなら fale を指定します。
    //!
    RStatus ConvertSub(const bool isToFile);

    //! @brief 元画像の画像ファイルを出力します。
    //!
    //! @param[in] originalImagePath 元画像の画像ファイルのパスです。
    //!
    RStatus OutputOriginalImageFile(const std::string& originalImagePath);
};

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

