﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// SetOptions SetJobOptions ShowUsage
// ReadInputFile ReadFtxData ReadBitmapData
// ConvertToFile ConvertToData

// PrepareToRead ReadImageFile RImage_ReadFile ReadEncodedData
// ParseFtxData ParseBitmapData RImage_ReadBitmap
// DecideDstParam SetWorkImageDimension DecideFormat EditInputImage

// ConvertSub CreateMipmap GenerateMipLevels OverwriteMipImage ConvertToLinear
// ConvertFormat ConvertByEncoder ConvertNXTiling
// NormalizeSurfaceNormal

// OutputBntxFile

// FtxOpt
// RImage ROriginalImage RMergeInfo ExtraImageAttr

//=============================================================================
// 処理選択用マクロです。
//=============================================================================
#define MIPMAP_CUBIC_AFTER_LINEAR // ミップマップ生成時に上位レベルまでリニア補間してからキュービック補間するなら定義します。

//=============================================================================
// include
//=============================================================================
#include "ImageConvert.h"
#include "BinaryFormat.h"

using namespace std;
using namespace nn::gfx::tool;

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//=============================================================================
// constants
//=============================================================================
const int LinearRgbaFlags[R_RGBA_COUNT] = //!< RGBA 成分に対するリニア変換フラグ配列です。
{
    FtxOpt::LINEAR_R,
    FtxOpt::LINEAR_G,
    FtxOpt::LINEAR_B,
    FtxOpt::LINEAR_A,
};

const std::string FtxOpt::DCC_PRESET_DEFAULT = "<default>"; //!< デフォルトの DCC プリセット名（指定なし）です。
const std::string FtxOpt::HINT_DEFAULT = "<default>"; //!< デフォルトのヒント情報（指定なし）です。
const std::string FtxOpt::COMMENT_TEXT_DEFAULT = "<?default?>"; //!< デフォルトの編集用コメント文（指定なし）です。

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//-----------------------------------------------------------------------------
// debug
#ifdef _DEBUG
int VERBOSE_LEVEL = 2; //!< 処理状況表示の冗長さのレベルです。
#else
int VERBOSE_LEVEL = 1; //!< 処理状況表示の冗長さのレベルです。
#endif

//-----------------------------------------------------------------------------
// dimension

//! @brief 次元モードを表す列挙型です。
enum RDimMode
{
    R_DIM_MODE_1D,             //!< 1 次元テクスチャーです。
    R_DIM_MODE_2D,             //!< 2 次元テクスチャーです。
    R_DIM_MODE_3D,             //!< 3 次元テクスチャーです。
    R_DIM_MODE_3D_INPUT,       //!< 入力ファイルが 3 次元テクスチャーです。
    R_DIM_MODE_CUBE_SEPARATE,  //!< キューブマップ個別指定です。
    R_DIM_MODE_CUBE_HC,        //!< 水平十字キューブマップです。
    R_DIM_MODE_CUBE_VC,        //!< 垂直十字キューブマップです。
    R_DIM_MODE_CUBE_INPUT,     //!< 入力ファイルがキューブマップです。
    R_DIM_MODE_1D_ARRAY,       //!< 1 次元テクスチャー配列です。
    R_DIM_MODE_1D_ARRAY_INPUT, //!< 入力ファイルが 1 次元テクスチャー配列です。
    R_DIM_MODE_2D_ARRAY,       //!< 2 次元テクスチャー配列です。
    R_DIM_MODE_2D_ARRAY_INPUT, //!< 入力ファイルが 2 次元テクスチャー配列です。
};

//-----------------------------------------------------------------------------
// color
const int OpaqueAlphaBorder = 0x80; //!< 1bit アルファフォーマットでアルファがこの値以上なら不透明とみなします。

//-----------------------------------------------------------------------------
// リニア変換
const float SrgbGammaValue = 2.2f;

//-----------------------------------------------------------------------------
//! @brief オプション文字列からリニア変換フラグを取得します。
//!
//! @param[in] str リニア変換フラグを表すオプション文字列です。
//!
//! @return リニア変換フラグを返します。
//-----------------------------------------------------------------------------
int GetLinearFlagFromOptString(const std::string& str)
{
    if (str.size() == R_RGBA_COUNT)
    {
        int linearFlag = FtxOpt::LINEAR_NONE;
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            const char c = str[iRgba];
            if (c == '0')
            {

            }
            else if (c == '1')
            {
                linearFlag |= LinearRgbaFlags[iRgba];
            }
            else
            {
                return FtxOpt::LINEAR_INVALID;
            }
        }
        return linearFlag;
    }
    return FtxOpt::LINEAR_INVALID;
}

//-----------------------------------------------------------------------------
//! @brief リニア変換フラグからオプション文字列を取得します。
//!
//! @param[in] linearFlag リニア変換フラグです。
//!
//! @return オプション文字列を返します。
//-----------------------------------------------------------------------------
std::string GetOptStringFromLinearFlag(const int linearFlag)
{
    if (linearFlag < 0)
    {
        return "0000";
    }
    else
    {
        char strBuf[R_RGBA_COUNT];
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            strBuf[iRgba] = ((linearFlag & LinearRgbaFlags[iRgba]) != 0) ? '1' : '0';
        }
        return std::string(strBuf, sizeof(strBuf));
    }
}

//-----------------------------------------------------------------------------
//! @brief オプション文字列から次元を取得します。
//!
//! @param[in] str 次元を表すオプション文字列です。
//!
//! @return 次元を返します。
//-----------------------------------------------------------------------------
int GetDimensionFromOptString(const std::string& str)
{
    if (str == "1d"         ||
        str == "2d"         ||
        str == "3d"         ||
        str == "1d_array"   ||
        str == "2d_array"   ||
        str == "cube"       ||
        str == "cube_array")
    {
        return RGetTextureDimensionFromString(str);
    }
    else
    {
        return FtxOpt::DIM_INVALID;
    }
}

//-----------------------------------------------------------------------------
//! @brief オプション文字列からフォーマットを取得します。
//!
//! @param[in] str フォーマットを表すオプション文字列です。
//!
//! @return フォーマットを返します。
//-----------------------------------------------------------------------------
int GetFormatFromOptString(const std::string& str)
{
    if (str == "bc_auto")
    {
        return FtxOpt::FORMAT_BC_AUTO;
    }
    else
    {
        return RGetTextureFormatFromString(str);
    }
}

//-----------------------------------------------------------------------------
//! @brief オプション文字列から成分選択を取得します。
//!
//! @param[in] str 成分選択を表すオプション文字列です。
//!
//! @return 成分選択を返します。
//-----------------------------------------------------------------------------
int GetCompSelFromOptString(const std::string& str)
{
    const char CompSelChars[FtxComponent_Count] =
    {
        'r', 'g', 'b', 'a', '0', '1'
    };

    int comps[R_RGBA_COUNT];
    int validCount = 0;
    if (str.size() == 4)
    {
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            const char c = str[iRgba];
            for (int iComp = 0; iComp < FtxComponent_Count; ++iComp)
            {
                if (c == CompSelChars[iComp])
                {
                    comps[iRgba] = iComp;
                    ++validCount;
                    break;
                }
            }
        }
    }

    return (validCount == R_RGBA_COUNT) ?
        FTX_GET_COMP_SEL(comps[0], comps[1], comps[2], comps[3]) :
        FtxOpt::COMP_SEL_INVALID;
}

//-----------------------------------------------------------------------------
// サイズ変更フィルタの文字列と値の配列です。
//-----------------------------------------------------------------------------
const RStringAndValue<ResizeFilter> g_ResizeFilterStringValues[] =
{
    { "point"       , ResizeFilter_Point       },
    { "linear"      , ResizeFilter_Linear      },
    { "cubic"       , ResizeFilter_Cubic       },
    { "cubic_smooth", ResizeFilter_CubicSmooth },
    { "cubic_sharp" , ResizeFilter_CubicSharp  },
    { nullptr       , ResizeFilter_Invalid     },
};

//-----------------------------------------------------------------------------
//! @brief ミップマップ画像個別指定の引数なら true を返します。
//!
//! @param[out] level レベルを格納します。
//! @param[in] token 引数のトークンです。
//!
//! @return ミップマップ画像個別指定の引数なら true を返します。
//-----------------------------------------------------------------------------
bool IsMipImageArgToken(int& level, const std::string& token)
{
    level = 0;
    for (int iTest = 0; iTest < 2; ++iTest)
    {
        const std::string top = (iTest == 0) ? "-mi" : "--mip_image";
        if (token.find(top) == 0)
        {
            for (int lv = 1; lv < FtxOpt::MipCountMax; ++lv)
            {
                if (token.find(top + RGetNumberString(lv) + "=") == 0)
                {
                    level = lv;
                    return true;
                }
            }
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief キューブマップのミップマップ画像個別指定の引数なら true を返します。
//!
//! @param[out] iFace +X, -X, +Y, -Y, +Z, -Z の順のフェースインデックスを格納します。
//! @param[out] level レベルを格納します。
//! @param[in] token 引数のトークンです。
//!
//! @return キューブマップのミップマップ画像個別指定の引数なら true を返します。
//-----------------------------------------------------------------------------
bool IsCubeMipImageArgToken(int& iFace, int& level, const std::string& token)
{
    static const char* const s_OptionTops[FtxOpt::CUBE_FACE_COUNT][2] =
    {
        { "-crt", "--cubemap_right"  },
        { "-clt", "--cubemap_left"   },
        { "-ctp", "--cubemap_top"    },
        { "-cbm", "--cubemap_bottom" },
        { "-cft", "--cubemap_front"  },
        { "-cbk", "--cubemap_back"   },
    };

    iFace = 0;
    level = 0;
    for (int ifc = 0; ifc < FtxOpt::CUBE_FACE_COUNT; ++ifc)
    {
        for (int iTest = 0; iTest < 2; ++iTest)
        {
            const std::string top = s_OptionTops[ifc][iTest];
            if (token.find(top) == 0)
            {
                for (int lv = 1; lv < FtxOpt::MipCountMax; ++lv)
                {
                    if (token.find(top + RGetNumberString(lv) + "=") == 0)
                    {
                        iFace = ifc;
                        level = lv;
                        return true;
                    }
                }
            }
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 切り抜き矩形を引数から取得します。
//!
//! @param[out] cropRect 切り抜き矩形を格納します。
//! @param[in] arg 引数です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus GetCropRectArg(RCropRect& cropRect, const std::string& arg)
{
    RStringArray vals;
    const int valCount = RTokenizeString(vals, arg, ",");
    if (valCount == 4)
    {
        const int x = atoi(vals[0].c_str());
        const int y = atoi(vals[1].c_str());
        const int w = atoi(vals[2].c_str());
        const int h = atoi(vals[3].c_str());
        if (x >= 0 && y >= 0 && w >= 1 && h >= 1)
        {
            cropRect = RCropRect(x, y, w, h);
            return RStatus::SUCCESS;
        }
    }
    return RStatus(RStatus::FAILURE, "Crop is wrong: " + arg); // RShowError
}

//-----------------------------------------------------------------------------
//! @brief チャンネル差し替え画像ファイルのパスを引数から取得します。
//!
//! @param[out] path チャンネル差し替え画像ファイルのパスを格納します。
//! @param[out] chanIdx チャンネル差し替え元のチャンネルインデックスを格納します。
//! @param[in] arg 引数です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus GetReplacePathArg(std::string& path, int& chanIdx, const std::string& arg)
{
    path = arg;
    chanIdx = 0;
    if (arg.size() > 2 && arg[arg.size() - 2] == ':')
    {
        const char chanC = RGetLowerCaseString(arg.substr(arg.size() - 1))[0];
        const size_t iSrcChan = std::string("rgba").find(chanC);
        if (iSrcChan == std::string::npos)
        {
            return RStatus(RStatus::FAILURE, "Replace is wrong: " + arg); // RShowError
        }
        chanIdx = static_cast<int>(iSrcChan);
        path = arg.substr(0, arg.size() - 2);
    }
    path = RGetFullFilePath(path, true);
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief ファイル名の末尾の数値をインクリメントしたパスを追加します。
//!
//! @param[in,out] paths ファイルパス配列です。
//! @param[in] basePath ベースとなるファイルのパスです。
//! @param[in] addCount 追加するパス数です。
//!
//! @return ベースとなるファイル名の末尾に数値があれば true、なければ false を返します。
//-----------------------------------------------------------------------------
bool AddNumberIncrementedPath(
    RStringArray& paths,
    const std::string basePath,
    const int addCount
)
{
    std::string prefix = RGetNoExtensionFilePath(basePath);
    const std::string suffix = "." + RGetExtensionFromFilePath(basePath);
    int digitCount = 0;
    for (size_t ic = prefix.size() - 1; ic >= 0; --ic)
    {
        const char c = prefix[ic];
        if ('0' <= c && c <= '9')
        {
            ++digitCount;
        }
        else
        {
            break;
        }
    }
    if (digitCount == 0)
    {
        return false;
    }
    int curN = atoi(prefix.substr(prefix.size() - digitCount).c_str());
    prefix = prefix.substr(0, prefix.size() - digitCount);
    const std::string numFmt = (digitCount == 1) ?
        std::string("%d") : "%0" + RGetNumberString(digitCount) + "d";
    for (int iPath = 0; iPath < addCount; ++iPath)
    {
        ++curN;
        const std::string incPath = prefix + RGetNumberString(curN, numFmt.c_str()) + suffix;
        paths.push_back(incPath);
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief レベルに対応したイメージデータのポインタを取得します。
//!
//! @param[out] pFaceBytes 1 フェースのデータサイズを格納します。nullptr なら格納しません。
//! @param[in] image 画像です。
//! @param[in] level レベルです。
//! @param[in] iDepth 奥行きインデックスです。
//!
//! @return イメージデータのポインタを返します。
//-----------------------------------------------------------------------------
void* GetLevelImageData(
    size_t* pFaceBytes,
    RImage& image,
    const int level,
    const int iDepth
)
{
    const size_t ofs = image.GetLevelOffset(level);
    uint8_t* top = image.GetImageData() + ofs;
    const size_t levelBytes =
        (image.GetMipCount() == 1        ) ? image.GetImageDataSize()       :
        (level == image.GetMipCount() - 1) ? image.GetImageDataSize() - ofs :
        image.GetLevelOffset(level + 1) - ofs;
    const int curD = (image.GetDimension() == FtxDimension_3d) ?
        RMax(image.GetImageD() >> level, 1) : image.GetImageD();
    const size_t faceBytes = levelBytes / curD;
    if (pFaceBytes != nullptr)
    {
        *pFaceBytes = faceBytes;
    }
    return top + faceBytes * iDepth;
}

//-----------------------------------------------------------------------------
//! @brief イメージデータを水平方向にフリップします。
//!
//! @param[in,out] pImage イメージデータです。
//! @param[in] w 幅です。
//! @param[in] h 高さです。
//! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
//-----------------------------------------------------------------------------
void FlipImageH(void* pImage, const int w, const int h, const bool isFloat)
{
    const int halfW = w / 2;
    if (!isFloat)
    {
        uint32_t* pLineU32 = reinterpret_cast<uint32_t*>(pImage);
        for (int iy = 0; iy < h; ++iy)
        {
            for (int ix = 0; ix < halfW; ++ix)
            {
                const int jx = w - 1 - ix;
                const uint32_t rgbaI = *(pLineU32 + ix);
                *(pLineU32 + ix) = *(pLineU32 + jx);
                *(pLineU32 + jx) = rgbaI;
            }
            pLineU32 += w;
        }
    }
    else
    {
        float rgbaF32[R_RGBA_COUNT];
        float* pLineF32 = reinterpret_cast<float*>(pImage);
        for (int iy = 0; iy < h; ++iy)
        {
            for (int ix = 0; ix < halfW; ++ix)
            {
                const int jx = w - 1 - ix;
                memcpy(rgbaF32                     , pLineF32 + ix * R_RGBA_COUNT, sizeof(rgbaF32));
                memcpy(pLineF32 + ix * R_RGBA_COUNT, pLineF32 + jx * R_RGBA_COUNT, sizeof(rgbaF32));
                memcpy(pLineF32 + jx * R_RGBA_COUNT, rgbaF32                     , sizeof(rgbaF32));
            }
            pLineF32 += w * R_RGBA_COUNT;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief イメージデータを垂直方向にフリップします。
//!
//! @param[in,out] pImage イメージデータです。
//! @param[in] w 幅です。
//! @param[in] h 高さです。
//! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
//-----------------------------------------------------------------------------
void FlipImageV(void* pImage, const int w, const int h, const bool isFloat)
{
    uint8_t* img = reinterpret_cast<uint8_t*>(pImage);
    const size_t colBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t lineBytes = colBytes * w;
    uint8_t* lineBuf = new uint8_t[lineBytes];
    const int halfH = h / 2;
    for (int iy = 0; iy < halfH; ++iy)
    {
        const int jy = h - 1 - iy;
        uint8_t* lineI = img + iy * lineBytes;
        uint8_t* lineJ = img + jy * lineBytes;
        memcpy(lineBuf, lineI  , lineBytes);
        memcpy(lineI  , lineJ  , lineBytes);
        memcpy(lineJ  , lineBuf, lineBytes);
    }
    delete[] lineBuf;
}

//-----------------------------------------------------------------------------
//! @brief ミップマップの最大レベル数を取得します。
//!
//! @param[in] dimension 次元です。
//! @param[in] w 画像の幅です。
//! @param[in] h 画像の高さです。
//! @param[in] d 画像の奥行きです。
//!
//! @return ミップマップの最大レベル数を返します。
//-----------------------------------------------------------------------------
int GetMaxMipmapLevels(const FtxDimension dimension, const int w, const int h, const int d)
{
    int maxWHD = (dimension == FtxDimension_3d) ?
        RMax(w, RMax(h, d)) : RMax(w, h);
    int maxLevel = 1;
    while (maxWHD >= 2)
    {
        ++maxLevel;
        maxWHD >>= 1;
    }
    return RMin(maxLevel, RImage::MipCountMax);
}

//-----------------------------------------------------------------------------
//! @brief ミップマップの最大レベル数を取得します。
//!
//! @param[in] image 画像です。
//!
//! @return ミップマップの最大レベル数を返します。
//-----------------------------------------------------------------------------
int GetMaxMipmapLevels(const RImage& image)
{
    return GetMaxMipmapLevels(static_cast<FtxDimension>(image.GetDimension()),
        image.GetImageW(), image.GetImageH(), image.GetImageD());
}

//-----------------------------------------------------------------------------
//! @brief 元画像配列に適したフォーマットを取得します。
//!
//! @param[in] originalImages 元画像配列です。
//!
//! @return フォーマットを返します。
//-----------------------------------------------------------------------------
FtxFormat GetSuitableFormatForOriginalImages(const ROriginalImageArray& originalImages)
{
    //-----------------------------------------------------------------------------
    // グレースケールか判定します。
    const int imgCount = static_cast<int>(originalImages.size());
    bool isGray = true;
    for (int iImg = 0; iImg < imgCount; ++iImg)
    {
        if (!originalImages[iImg].IsGray())
        {
            isGray = false;
            break;
        }
    }

    //-----------------------------------------------------------------------------
    // 透明モードを取得します。
    RImage::TransparencyMode xpaMode = RImage::OPA;
    for (int iImg = 0; iImg < imgCount; ++iImg)
    {
        const RImage::TransparencyMode faceXpaMode = originalImages[iImg].GetTransparencyMode();
        if (faceXpaMode == RImage::XLU)
        {
            xpaMode = RImage::XLU;
            break;
        }
        else if (faceXpaMode == RImage::MASK)
        {
            xpaMode = RImage::MASK;
            // 残りの画像に中間的なアルファが存在する可能性があるので判定を継続します。
        }
    }
    //cerr << "TransparencyMode: " << xpaMode << endl;

    //-----------------------------------------------------------------------------
    // フォーマットを決定します。
    if (isGray)
    {
        return (xpaMode == RImage::OPA) ?
            FtxFormat_Unorm_Bc4 :
            FtxFormat_Unorm_Bc3;
    }
    else // color
    {
        return (xpaMode == RImage::XLU) ?
            FtxFormat_Unorm_Bc3 :
            FtxFormat_Unorm_Bc1;
    }
}

//-----------------------------------------------------------------------------
//! @brief 整数型の入力ファイルから変換可能なテクスチャーフォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 整数型の入力ファイルから変換可能なテクスチャーフォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsConvertibleTextureFormatFromInteger(const FtxFormat format)
{
    return !RIsRealNumberTextureFormat(format);
}

//-----------------------------------------------------------------------------
//! @brief 実数型の入力ファイルから変換可能なテクスチャーフォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 実数型の入力ファイルから変換可能なテクスチャーフォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsConvertibleTextureFormatFromRealNumber(const FtxFormat format)
{
    return (
        RIsRealNumberTextureFormat(format) ||
        format == FtxFormat_Unorm_8_8_8_8  ||
        format == FtxFormat_Snorm_8_8_8_8  ||
        format == FtxFormat_Srgb_8_8_8_8   ||
        RIsBcTextureFormat(format)         ||
        RIsAstcTextureFormat(format));
}

//-----------------------------------------------------------------------------
//! @brief フォーマットを決定します。
//!        マージ指定の場合はマージするファイルからフォーマットを取得します。
//!
//! @param[out] pDstFormat フォーマットを格納します（nullptr なら格納しません）。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] mergeInfo マージ情報です。
//! @param[in] originalImages 元画像配列です。
//! @param[in] pReferenceImage 元画像配列以外に参照する画像へのポインターです（nullptr なら参照しません）。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus DecideFormat(
    FtxFormat* pDstFormat,
    const FtxOpt& opt,
    const RMergeInfo& mergeInfo,
    const ROriginalImageArray& originalImages,
    const RImage* pReferenceImage
)
{
    //-----------------------------------------------------------------------------
    // 最初の元画像へのポインタを取得します。
    const ROriginalImage* pFirstOrgImg = (!originalImages.empty()) ?
        &originalImages[0] : nullptr;

    //-----------------------------------------------------------------------------
    // フォーマットを決定します。
    FtxFormat dstFormat = (pFirstOrgImg != nullptr && pFirstOrgImg->IsFloat()) ?
        FtxFormat_Float_16_16_16_16 :
        FtxFormat_Unorm_8_8_8_8;
    if (opt.m_Format == FtxOpt::FORMAT_BC_AUTO)
    {
        dstFormat = GetSuitableFormatForOriginalImages(originalImages);
    }
    else if (opt.m_Format != FtxOpt::FORMAT_DEFAULT)
    {
        dstFormat = static_cast<FtxFormat>(opt.m_Format);
    }
    else if (mergeInfo.m_IsEnable && mergeInfo.m_Format != FtxOpt::FORMAT_DEFAULT)
    {
        dstFormat = static_cast<FtxFormat>(mergeInfo.m_Format);
    }
    else if (pFirstOrgImg != nullptr &&
             pFirstOrgImg->m_OriginalFormat != FtxOpt::FORMAT_INVALID)
    {
        dstFormat = static_cast<FtxFormat>(pFirstOrgImg->m_OriginalFormat);
    }
    else if (pReferenceImage != nullptr &&
             pReferenceImage->GetFormat() != FtxOpt::FORMAT_INVALID)
    {
        dstFormat = static_cast<FtxFormat>(pReferenceImage->GetFormat());
    }

    if (dstFormat == FtxFormat_Invalid)
    {
        return RStatus(RStatus::FAILURE, "Format is wrong: " + opt.m_InputPaths[0]); // RShowError
    }

    if (pFirstOrgImg != nullptr)
    {
        const std::string errorInfo = opt.m_InputPaths[0] + " (" + RGetTextureFormatString(dstFormat) + ")";
        if (!pFirstOrgImg->IsFloat() && !IsConvertibleTextureFormatFromInteger(dstFormat))
        {
            return RStatus(RStatus::FAILURE, "Cannot convert the integer data to the format: " + errorInfo); // RShowError
        }
        if (pFirstOrgImg->IsFloat() && !IsConvertibleTextureFormatFromRealNumber(dstFormat))
        {
            return RStatus(RStatus::FAILURE, "Cannot convert the real number data to the format: " + errorInfo); // RShowError
        }
    }

    if (pDstFormat != nullptr)
    {
        *pDstFormat = dstFormat;
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief ミップマップのレベル数を決定します。
//!
//! @param[in] opt ftx 変換オプションです。
//! @param[in] mergeInfo マージ情報です。
//! @param[in] originalImages 元画像配列です。
//!
//! @return ミップマップのレベル数を返します。
//-----------------------------------------------------------------------------
int DecideMipCount(
    const FtxOpt& opt,
    const RMergeInfo& mergeInfo,
    const ROriginalImageArray& originalImages
)
{
    const ROriginalImage* pFirstOrgImg = (!originalImages.empty()) ?
        &originalImages[0] : nullptr;

    int dstMipCount;
    if (opt.m_MipCount != FtxOpt::MipCountDefault)
    {
        dstMipCount = opt.m_MipCount;
    }
    else if (mergeInfo.m_IsEnable && mergeInfo.m_MipCount != FtxOpt::MipCountDefault)
    {
        dstMipCount = mergeInfo.m_MipCount;
        if (mergeInfo.m_MipCount >= 2 && pFirstOrgImg != nullptr)
        {
            // マージする ftx ファイルの mip_level が 2 以上の場合、
            // 最小レベルの幅と高さが、マージするファイルの最小レベルの幅と高さ以下になる
            // レベルまでミップマップを作成します。
            int curW = pFirstOrgImg->m_Width;
            int curH = pFirstOrgImg->m_Height;
            const int minShift = (mergeInfo.m_MipCount - 1);
            const int minW = max(mergeInfo.m_Width  >> minShift, 1);
            const int minH = max(mergeInfo.m_Height >> minShift, 1);
            dstMipCount = 1;
            while (curW > minW || curH > minH)
            {
                ++dstMipCount;
                curW >>= 1;
                curH >>= 1;
            }
        }
    }
    else
    {
        dstMipCount = FtxOpt::MipCountMax;
        if (pFirstOrgImg != nullptr)
        {
            const std::string ext = RGetExtensionFromFilePath(opt.m_InputPaths[0]);
            if (ext == DdsExtension  ||
                ext == FtxaExtension ||
                ext == FtxbExtension)
            {
                dstMipCount = pFirstOrgImg->m_MipCount;
            }
        }
    }
    return dstMipCount;
}

//-----------------------------------------------------------------------------
//! @brief リニア変換フラグを決定します。
//!
//! @param[out] pSpecified オプションまたは入力画像によって
//!                        リニア変換フラグが指定されたなら true を格納します。
//!                        nullptr なら格納しません。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] mergeInfo マージ情報です。
//! @param[in] originalImages 元画像配列です。
//! @param[in] pReferenceImage 元画像配列以外に参照する画像へのポインターです（nullptr なら参照しません）。
//! @param[in] format 変換先のフォーマットです。
//!
//! @return リニア変換フラグを返します。
//-----------------------------------------------------------------------------
int DecideLinearFlag(
    bool* pSpecified,
    const FtxOpt& opt,
    const RMergeInfo& mergeInfo,
    const ROriginalImageArray& originalImages,
    const RImage* pReferenceImage,
    const FtxFormat format
)
{
    int linearFlag = FtxOpt::LINEAR_NONE;
    bool specified = false;
    if (opt.m_LinearFlag != FtxOpt::LINEAR_DEFAULT)
    {
        linearFlag = opt.m_LinearFlag;
        specified = true;
    }
    else if (mergeInfo.m_IsEnable && mergeInfo.m_LinearFlag != FtxOpt::LINEAR_DEFAULT)
    {
        linearFlag = mergeInfo.m_LinearFlag;
        specified = true;
    }
    else if (!originalImages.empty() &&
             originalImages[0].m_LinearFlag != FtxOpt::LINEAR_DEFAULT)
    {
        linearFlag = originalImages[0].m_LinearFlag;
        specified = true;
    }
    else if (pReferenceImage != nullptr &&
             pReferenceImage->GetLinearFlag() != FtxOpt::LINEAR_DEFAULT)
    {
        linearFlag = pReferenceImage->GetLinearFlag();
        specified = true;
    }

    if (RIsSrgbFetchTextureFormat(format))
    {
        linearFlag |= FtxOpt::LINEAR_RGB;
        specified = true;
    }

    if (pSpecified != nullptr)
    {
        *pSpecified = specified;
    }
    return linearFlag;
}

//-----------------------------------------------------------------------------
//! @brief キューブマップ個別指定の各面の入力ファイルが指定されているかチェックします。
//!
//! @param[in] opt ftx 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus CheckCubeMapSeparete(const FtxOpt& opt)
{
    const char* faceName =
        (opt.m_CubeBackPath.empty()  ) ? "back"   :
        (opt.m_CubeLeftPath.empty()  ) ? "left"   :
        (opt.m_CubeRightPath.empty() ) ? "right"  :
        (opt.m_CubeTopPath.empty()   ) ? "top"    :
        (opt.m_CubeBottomPath.empty()) ? "bottom" :
        nullptr;
    if (faceName == nullptr)
    {
        return RStatus::SUCCESS;
    }
    else
    {
        return RStatus(RStatus::FAILURE, "Cube map face is not specified (" + std::string(faceName) + "): " + opt.m_InputPaths[0]); // RShowError
    }
}

//-----------------------------------------------------------------------------
//! @brief メモリーの値を表示します。
//-----------------------------------------------------------------------------
inline void DisplayMemory(std::ostream& os, const void* mem, const int size, const char* title)
{
    os << title << " ----" << endl;

    const char fillBak = os.fill('0');
    const long flagBak = os.setf(ios_base::hex, ios_base::basefield);
    os.setf(ios_base::uppercase);
    os.setf(ios_base::right, ios_base::adjustfield);

    const uint8_t* array = reinterpret_cast<const uint8_t*>(mem);
    const int column = 16;
    int ic = 0;
    for (int ival = 0; ival < size; ++ival)
    {
        os << setw(2) << static_cast<unsigned int>(array[ival]);
        if (ival < size - 1)
        {
            if (ic == column - 1)
            {
                os << endl;
            }
            else
            {
                os << (((ic & 3) == 3) ? "  " : " ");
            }
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
    os << endl;

    os.fill(fillBak);
    os.flags(flagBak);
}

//-----------------------------------------------------------------------------
//! @brief 画像の次元を表示します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] image 画像です。
//! @param[in] dimMode 次元モードです。
//! @param[in] opt ftx 変換オプションです。
//-----------------------------------------------------------------------------
void DisplayImageDimension(
    std::ostream& os,
    const RImage& image,
    const RDimMode dimMode,
    const FtxOpt& opt
)
{
    if (!opt.m_IsSilent)
    {
        os << "Dim     : " << RGetTextureDimensionString(
            static_cast<FtxDimension>(image.GetDimension()));
        if (dimMode == R_DIM_MODE_CUBE_HC)
        {
            os << " (horizontal cross)";
        }
        if (dimMode == R_DIM_MODE_CUBE_VC)
        {
            os << " (virtical cross)";
        }
        os << endl;
    }
}

//-----------------------------------------------------------------------------
//! @brief 入力画像の次元をキューブマップに変換するなら true を返します。
//!
//! @param[in] rimg 入力画像です。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] mergeInfo マージ情報です。
//!
//! @return 入力画像の次元をキューブマップに変換するなら true を返します。
//-----------------------------------------------------------------------------
bool IsToCubeDimension(
    const RImage& rimg,
    const FtxOpt& opt,
    const RMergeInfo& mergeInfo
)
{
    if (!rimg.IsCubeMap())
    {
        if (opt.IsCubeMapSeparete())
        {
            return true;
        }
        const FtxDimension dimension = static_cast<FtxDimension>(
            (opt.m_Dimension                               != FtxOpt::DIM_DEFAULT) ? opt.m_Dimension       :
            (mergeInfo.m_IsEnable && mergeInfo.m_Dimension != FtxOpt::DIM_DEFAULT) ? mergeInfo.m_Dimension :
            rimg.GetDimension());
        if (RIsCubMapDimension(dimension))
        {
            return (RImage::IsCubeHCWH(rimg.GetImageW(), rimg.GetImageH()) ||
                    RImage::IsCubeVCWH(rimg.GetImageW(), rimg.GetImageH()));
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 作業用画像の次元を入力画像と変換オプションから設定します。
//!
//! @param[out] dimMode 次元モードを格納します。
//! @param[in,out] workImage 作業用画像です。次元と配列テクスチャーフラグのみ格納します。
//! @param[out] isDimModeChanged 元画像の次元モードから変更されたなら true を格納します。
//! @param[in] rimg 入力画像です。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] mergeInfo マージ情報です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus SetWorkImageDimension(
    RDimMode& dimMode,
    RImage& workImage,
    bool& isDimModeChanged,
    const RImage& rimg,
    const FtxOpt& opt,
    const RMergeInfo& mergeInfo
)
{
    //-----------------------------------------------------------------------------
    // 入力画像から次元モードを決定します。
    dimMode = R_DIM_MODE_2D;
    if (rimg.GetDimension() == FtxDimension_1d)
    {
        dimMode = R_DIM_MODE_1D;
    }
    else if (rimg.GetDimension() == FtxDimension_3d)
    {
        dimMode = R_DIM_MODE_3D_INPUT;
    }
    else if (rimg.IsCubeMap())
    {
        dimMode = R_DIM_MODE_CUBE_INPUT;
    }
    else if (rimg.GetDimension() == FtxDimension_1dArray)
    {
        dimMode = R_DIM_MODE_1D_ARRAY_INPUT;
    }
    else if (rimg.GetDimension() == FtxDimension_2dArray)
    {
        dimMode = R_DIM_MODE_2D_ARRAY_INPUT;
    }
    const RDimMode inputDimMode = dimMode;

    //-----------------------------------------------------------------------------
    // 変換オプションおよびマージ情報に従って次元モードを変更します。
    if (opt.IsCubeMapSeparete())
    {
        dimMode = R_DIM_MODE_CUBE_SEPARATE;
    }
    else if (opt.m_Dimension != FtxOpt::DIM_DEFAULT)
    {
        if (opt.m_Dimension == FtxDimension_1d)
        {
            dimMode = R_DIM_MODE_1D;
        }
        else if (opt.m_Dimension == FtxDimension_2d)
        {
            dimMode = R_DIM_MODE_2D;
        }
        else if (opt.m_Dimension == FtxDimension_3d && dimMode != R_DIM_MODE_3D_INPUT)
        {
            dimMode = (dimMode == R_DIM_MODE_1D_ARRAY_INPUT || dimMode == R_DIM_MODE_2D_ARRAY_INPUT) ?
                R_DIM_MODE_3D_INPUT : R_DIM_MODE_3D;
        }
        else if (RIsCubMapDimension(static_cast<FtxDimension>(opt.m_Dimension)) && dimMode != R_DIM_MODE_CUBE_INPUT)
        {
            if (RImage::IsCubeHCWH(rimg.GetImageW(), rimg.GetImageH()))
            {
                dimMode = R_DIM_MODE_CUBE_HC;
            }
            else if (RImage::IsCubeVCWH(rimg.GetImageW(), rimg.GetImageH()))
            {
                dimMode = R_DIM_MODE_CUBE_VC;
            }
        }
        else if (opt.m_Dimension == FtxDimension_1dArray && dimMode != R_DIM_MODE_1D_ARRAY_INPUT)
        {
            dimMode = (dimMode == R_DIM_MODE_3D_INPUT || dimMode == R_DIM_MODE_2D_ARRAY_INPUT) ?
                R_DIM_MODE_1D_ARRAY_INPUT : R_DIM_MODE_1D_ARRAY;
        }
        else if (opt.m_Dimension == FtxDimension_2dArray && dimMode != R_DIM_MODE_2D_ARRAY_INPUT)
        {
            dimMode = (dimMode == R_DIM_MODE_3D_INPUT || dimMode == R_DIM_MODE_1D_ARRAY_INPUT) ?
                R_DIM_MODE_2D_ARRAY_INPUT : R_DIM_MODE_2D_ARRAY;
        }
    }
    else if (mergeInfo.m_IsEnable && mergeInfo.m_Dimension != FtxOpt::DIM_DEFAULT)
    {
        if (mergeInfo.m_Dimension == FtxDimension_1d)
        {
            dimMode = R_DIM_MODE_1D;
        }
        else if (mergeInfo.m_Dimension == FtxDimension_2d)
        {
            dimMode = R_DIM_MODE_2D;
        }
        else if (mergeInfo.m_Dimension == FtxDimension_3d && dimMode != R_DIM_MODE_3D_INPUT)
        {
            dimMode = (dimMode == R_DIM_MODE_1D_ARRAY_INPUT || dimMode == R_DIM_MODE_2D_ARRAY_INPUT) ?
                R_DIM_MODE_3D_INPUT : R_DIM_MODE_3D;
        }
        else if (RIsCubMapDimension(static_cast<FtxDimension>(mergeInfo.m_Dimension)) &&
            dimMode != R_DIM_MODE_CUBE_INPUT)
        {
            if (RImage::IsCubeHCWH(rimg.GetImageW(), rimg.GetImageH()))
            {
                dimMode = R_DIM_MODE_CUBE_HC;
            }
            else if (RImage::IsCubeVCWH(rimg.GetImageW(), rimg.GetImageH()))
            {
                dimMode = R_DIM_MODE_CUBE_VC;
            }
        }
        else if (mergeInfo.m_Dimension == FtxDimension_1dArray && dimMode != R_DIM_MODE_1D_ARRAY_INPUT)
        {
            dimMode = (dimMode == R_DIM_MODE_3D_INPUT || dimMode == R_DIM_MODE_2D_ARRAY_INPUT) ?
                R_DIM_MODE_1D_ARRAY_INPUT : R_DIM_MODE_1D_ARRAY;
        }
        else if (mergeInfo.m_Dimension == FtxDimension_2dArray && dimMode != R_DIM_MODE_2D_ARRAY_INPUT)
        {
            dimMode = (dimMode == R_DIM_MODE_3D_INPUT || dimMode == R_DIM_MODE_1D_ARRAY_INPUT) ?
                R_DIM_MODE_2D_ARRAY_INPUT : R_DIM_MODE_2D_ARRAY;
        }
    }
    else if (opt.m_Depth != FtxOpt::DEPTH_DEFAULT)
    {
        if (dimMode == R_DIM_MODE_2D)
        {
            dimMode = R_DIM_MODE_3D;
        }
    }

    isDimModeChanged = (dimMode != inputDimMode);

    //-----------------------------------------------------------------------------
    // 作業用画像の次元を設定します。
    switch (dimMode)
    {
        case R_DIM_MODE_1D:
            workImage.SetDimension(FtxDimension_1d);
            break;

        default:
        case R_DIM_MODE_2D:
            workImage.SetDimension(FtxDimension_2d);
            break;

        case R_DIM_MODE_3D:
        case R_DIM_MODE_3D_INPUT:
            workImage.SetDimension(FtxDimension_3d);
            break;

        case R_DIM_MODE_CUBE_SEPARATE:
        case R_DIM_MODE_CUBE_HC:
        case R_DIM_MODE_CUBE_VC:
        case R_DIM_MODE_CUBE_INPUT:
        {
            bool isArray = false;
            if (opt.m_Depth != FtxOpt::DEPTH_DEFAULT ||
                (opt.m_Dimension != FtxOpt::DIM_DEFAULT && RIsArrayDimension(static_cast<FtxDimension>(opt.m_Dimension))))
            {
                isArray = true;
            }
            else if (dimMode == R_DIM_MODE_CUBE_INPUT && rimg.GetImageD() > RImage::CUBE_FACE_COUNT)
            {
                isArray = true;
            }
            else if (mergeInfo.m_IsEnable)
            {
                isArray = RIsArrayDimension(static_cast<FtxDimension>(mergeInfo.m_Dimension));
            }
            else if (dimMode == R_DIM_MODE_CUBE_INPUT)
            {
                isArray = rimg.IsArray();
            }
            workImage.SetDimension((!isArray) ? FtxDimension_CubeMap : FtxDimension_CubeMapArray);
            break;
        }

        case R_DIM_MODE_1D_ARRAY:
        case R_DIM_MODE_1D_ARRAY_INPUT:
            workImage.SetDimension(FtxDimension_1dArray);
            break;

        case R_DIM_MODE_2D_ARRAY:
        case R_DIM_MODE_2D_ARRAY_INPUT:
            workImage.SetDimension(FtxDimension_2dArray);
            break;
    }

    return RStatus::SUCCESS;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 作業用画像の幅と高さを入力画像と次元モードから設定します。
//!
//! @param[in,out] workImage 作業用画像です。
//! @param[in] rimg 入力画像です。
//! @param[in] dimMode 次元モードです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus SetWorkImageWidthHeight(
    RImage& workImage,
    const RImage& rimg,
    const RDimMode dimMode
)
{
    workImage.SetImageW(rimg.GetImageW());
    workImage.SetImageH(rimg.GetImageH());

    if (dimMode == R_DIM_MODE_1D             ||
        dimMode == R_DIM_MODE_1D_ARRAY       ||
        dimMode == R_DIM_MODE_1D_ARRAY_INPUT)
    {
        workImage.SetImageH(1);
    }
    else if (dimMode == R_DIM_MODE_CUBE_HC)
    {
        workImage.SetImageW(workImage.GetImageW() / RImage::CUBE_HC_COUNT_H);
        workImage.SetImageH(workImage.GetImageH() / RImage::CUBE_HC_COUNT_V);
        if (workImage.GetImageW() * RImage::CUBE_HC_COUNT_H != rimg.GetImageW() ||
            workImage.GetImageH() * RImage::CUBE_HC_COUNT_V != rimg.GetImageH())
        {
            return RStatus(RStatus::FAILURE, "Cube map horizontal cross width or height is wrong: " + rimg.GetFilePath()); // RShowError
        }
    }
    else if (dimMode == R_DIM_MODE_CUBE_VC)
    {
        workImage.SetImageW(workImage.GetImageW() / RImage::CUBE_VC_COUNT_H);
        workImage.SetImageH(workImage.GetImageH() / RImage::CUBE_VC_COUNT_V);
        if (workImage.GetImageW() * RImage::CUBE_VC_COUNT_H != rimg.GetImageW() ||
            workImage.GetImageH() * RImage::CUBE_VC_COUNT_V != rimg.GetImageH())
        {
            return RStatus(RStatus::FAILURE, "Cube map vertical cross width or height is wrong: " + rimg.GetFilePath()); // RShowError
        }
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 元画像ファイルへの相対パスを取得します。
//!
//! @param[in] from ftx ファイルのパスです。
//! @param[in] to 元画像ファイルのパスです。
//!
//! @return 元画像ファイルへの相対パスを返します。
//-----------------------------------------------------------------------------
std::string GetOriginalFilePath(
    const std::string& from,
    const std::string& to
)
{
    // ドライブをまたぐ場合は、拡張子付きのファイル名を返します。
    if (from.size() >= 2 && to.size() >= 2)
    {
        if (from.c_str()[1] == ':' &&
            to.c_str()[1]   == ':')
        {
            if (!RIsSameStringNoCase(from.substr(0, 2), to.substr(0, 2)))
            {
                return RGetFileNameFromFilePath(to);
            }
        }
    }

    // 相対パスを取得します。
    const std::string fromW = RGetWindowsFilePath(from);
    const std::string toW   = RGetWindowsFilePath(to);
    TCHAR path[MAX_PATH];
    #ifdef UNICODE
    if (::PathRelativePathTo(path,
        RGetUnicodeFromShiftJis(fromW).c_str(), FILE_ATTRIBUTE_ARCHIVE,
        RGetUnicodeFromShiftJis(toW).c_str()  , FILE_ATTRIBUTE_ARCHIVE))
    {
        const std::string relPath = RGetUnixFilePath(RGetShiftJisFromUnicode(path));
        if (RGetFolderFromFilePath(relPath) == ".")
        {
            return RGetFileNameFromFilePath(relPath);
        }
        else
        {
            return relPath;
        }
    }
    else
    {
        return RGetFileNameFromFilePath(to);
    }
    #else
    if (::PathRelativePathTo(path,
        fromW.c_str(), FILE_ATTRIBUTE_ARCHIVE,
        toW.c_str()  , FILE_ATTRIBUTE_ARCHIVE))
    {
        const std::string relPath = RGetUnixFilePath(path);
        if (RGetFolderFromFilePath(relPath) == ".")
        {
            return RGetFileNameFromFilePath(relPath);
        }
        else
        {
            return relPath;
        }
    }
    else
    {
        return RGetFileNameFromFilePath(to);
    }
    #endif
}

//-----------------------------------------------------------------------------
//! @brief 入力画像から元画像を作成して追加します。
//!
//! @param[in,out] originalImages 元画像配列です。
//! @param[in] rimg 入力画像です。
//! @param[in] sliceIndex インデックスです。
//! @param[in] face です。
//! @param[in] finalOutputPath 最終出力パスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
void AppendOriginalImage(
    ROriginalImageArray& originalImages,
    const RImage& rimg,
    const int sliceIndex,
    const ROriginalImage::Face face,
    const std::string& finalOutputPath
)
{
    std::string originalPath = (!rimg.GetOriginalPaths().empty()) ?
        rimg.GetOriginalFullPath(0) :
        rimg.GetFilePath();
    originalPath = GetOriginalFilePath(finalOutputPath, originalPath);

    originalImages.push_back(ROriginalImage(
        sliceIndex,
        face,
        rimg.GetImageW(),
        rimg.GetImageH(),
        originalPath.c_str(),
        rimg.GetImageData(),
        &rimg));
}

//-----------------------------------------------------------------------------
//! @brief 出力イメージデータでのキューブマップのフェースインデックスを取得します。
//!
//! @param[in] iFace +X, -X, +Y, -Y, +Z, -Z の順のフェースインデックスです。
//!
//! @return +X, -X, +Y, -Y, -Z, +Z の順のフェースインデックスを返します。
//-----------------------------------------------------------------------------
int GetCubeMapDstFaceIdx(const int iFace)
{
    return
        (iFace == RImage::CUBE_FACE_PZ) ? 5 :
        (iFace == RImage::CUBE_FACE_NZ) ? 4 :
        iFace;
}

//-----------------------------------------------------------------------------
//! @brief 水平十字／垂直十字キューブマップのイメージを画像から設定します。
//!
//! @param[in,out] workImage 作業用画像です。
//! @param[in,out] originalImages 元画像配列です。
//! @param[in] rimg 画像です。
//! @param[in] dimMode 次元モードです。
//! @param[in] finalOutputPath 最終出力パスです。
//-----------------------------------------------------------------------------
void SetCrossCubeImage(
    RImage& workImage,
    ROriginalImageArray& originalImages,
    const int iCube,
    const RImage& rimg,
    const RDimMode dimMode,
    const std::string& finalOutputPath
)
{
    const std::string originalPath =
        GetOriginalFilePath(finalOutputPath, rimg.GetFilePath());

    const size_t colBytes = (rimg.IsFloat()) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t lineBytes = colBytes * workImage.GetImageW();
    const size_t faceBytes = colBytes * workImage.GetImageW() * workImage.GetImageH();
    const size_t cubeBytes = faceBytes * RImage::CUBE_FACE_COUNT;
    const int (*facePoss)[2] = (dimMode == R_DIM_MODE_CUBE_HC) ?
        RImage::CubeHcPositions : RImage::CubeVcPositions;
    for (int iFace = 0; iFace < RImage::CUBE_FACE_COUNT; ++iFace)
    {
        const int srcIx = facePoss[iFace][0] * workImage.GetImageW();
        const int srcIy = facePoss[iFace][1] * workImage.GetImageH();
        const uint8_t* pSrc = rimg.GetImageData() +
            (srcIx + srcIy * rimg.GetImageW()) * colBytes;

        uint8_t* pDst = workImage.GetImageData() +
            iCube * cubeBytes + GetCubeMapDstFaceIdx(iFace) * faceBytes;
        uint8_t* pFaceTop = pDst;
        for (int iy = 0; iy < workImage.GetImageH(); ++iy)
        {
            memcpy(pDst, pSrc, lineBytes);
            pSrc += colBytes * rimg.GetImageW();
            pDst += lineBytes;
        }

        // 垂直十字キューブマップは +Z 方向を上下左右反転します。
        if (dimMode == R_DIM_MODE_CUBE_VC && iFace == RImage::CUBE_FACE_PZ)
        {
            FlipImageH(pFaceTop, workImage.GetImageW(), workImage.GetImageH(), rimg.IsFloat());
            FlipImageV(pFaceTop, workImage.GetImageW(), workImage.GetImageH(), rimg.IsFloat());
        }

        originalImages.push_back(ROriginalImage(
            iCube,
            static_cast<ROriginalImage::Face>(ROriginalImage::POSITIVE_X + iFace),
            workImage.GetImageW(),
            workImage.GetImageH(),
            originalPath.c_str(),
            pFaceTop,
            &rimg));
    }
}

//-----------------------------------------------------------------------------
//! @brief 作業用画像のイメージデータを入力画像から設定します。
//!
//! @param[in,out] workImage 作業用画像です。
//! @param[in,out] originalImages 元画像配列です。
//! @param[in] rimg 入力画像です。
//! @param[in] dimMode 次元モードです。
//! @param[in] finalOutputPath 最終出力パスです。
//-----------------------------------------------------------------------------
void SetWorkImageData(
    RImage& workImage,
    ROriginalImageArray& originalImages,
    const RImage& rimg,
    const RDimMode dimMode,
    const std::string& finalOutputPath
)
{
    const size_t colBytes = (rimg.IsFloat()) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t faceBytes = colBytes * workImage.GetImageW() * workImage.GetImageH();

    if (dimMode == R_DIM_MODE_3D_INPUT       ||
        dimMode == R_DIM_MODE_1D_ARRAY_INPUT ||
        dimMode == R_DIM_MODE_2D_ARRAY_INPUT)
    {
        //-----------------------------------------------------------------------------
        // 入力画像がすでに 3D または配列の場合
        const size_t srcFaceBytes = colBytes * rimg.GetImageW() * rimg.GetImageH();
        const uint8_t* pSrc = rimg.GetImageData();
        uint8_t* pDst = workImage.GetImageData();
        for (int iDepth = 0; iDepth < workImage.GetImageD(); ++iDepth)
        {
            memcpy(pDst, pSrc, faceBytes);

            std::string originalPath = (!rimg.GetOriginalPaths().empty()) ?
                rimg.GetOriginalFullPath(iDepth) :
                rimg.GetFilePath();
            originalPath = GetOriginalFilePath(finalOutputPath, originalPath);
            originalImages.push_back(ROriginalImage(
                iDepth, ROriginalImage::NONE, rimg.GetImageW(), rimg.GetImageH(),
                originalPath.c_str(), pSrc, &rimg));

            pSrc += srcFaceBytes;
            pDst += faceBytes;
        }
    }
    else if (dimMode == R_DIM_MODE_CUBE_INPUT)
    {
        //-----------------------------------------------------------------------------
        // 入力画像がすでにキューブマップの場合
        const uint8_t* pSrc = rimg.GetImageData();
        for (int iDepth = 0; iDepth < rimg.GetImageD(); ++iDepth)
        {
            const int iCube = iDepth / RImage::CUBE_FACE_COUNT;
            const int iFace = iDepth % RImage::CUBE_FACE_COUNT;
            const int baseZ = iCube * RImage::CUBE_FACE_COUNT;
            uint8_t* pDst = workImage.GetImageData() +
                (baseZ + GetCubeMapDstFaceIdx(iFace)) * faceBytes;
            memcpy(pDst, pSrc, faceBytes);

            std::string originalPath = (!rimg.GetOriginalPaths().empty()) ?
                rimg.GetOriginalFullPath(iDepth) :
                rimg.GetFilePath();
            originalPath = GetOriginalFilePath(finalOutputPath, originalPath);

            originalImages.push_back(ROriginalImage(
                iCube,
                static_cast<ROriginalImage::Face>(ROriginalImage::POSITIVE_X + iFace),
                workImage.GetImageW(),
                workImage.GetImageH(),
                originalPath.c_str(),
                pSrc,
                &rimg));
            pSrc += faceBytes;
        }
    }
    else if (dimMode == R_DIM_MODE_CUBE_HC || dimMode == R_DIM_MODE_CUBE_VC)
    {
        //-----------------------------------------------------------------------------
        // 水平十字／垂直十字キューブマップの場合
        SetCrossCubeImage(workImage, originalImages, 0, rimg, dimMode, finalOutputPath);
    }
    else
    {
        //-----------------------------------------------------------------------------
        // 1D、2D、キューブマップ 6 面個別指定の場合
        // キューブマップは最初の面のみ設定します。
        memcpy(workImage.GetImageData(), rimg.GetImageData(), faceBytes);

        const ROriginalImage::Face orgFace = (workImage.IsCubeMap()) ?
            ROriginalImage::POSITIVE_X : ROriginalImage::NONE;
        AppendOriginalImage(originalImages, rimg, 0, orgFace, finalOutputPath);
    }
}

//-----------------------------------------------------------------------------
//! @brief 作業用画像にミップマップデータを設定します。
//!
//! @param[in,out] workImage 作業用画像です。
//! @param[in] rimg 入力画像です。
//-----------------------------------------------------------------------------
void SetWorkImageMipData(RImage& workImage, const RImage& rimg)
{
    const uint8_t* pSrc = rimg.GetImageData() + workImage.GetImageDataSize();
    uint8_t* pDst = workImage.GetImageData() + workImage.GetImageDataSize();
    if (!workImage.IsCubeMap())
    {
        memcpy(pDst, pSrc, rimg.GetMipDataSize());
    }
    else // CUBE
    {
        // rimg は右手座標系で +X、-X、+Y、-Y、+Z、-Z 面の順なので、
        // 左手座標系で +X、-X、+Y、-Y、+Z、-Z 面の順になるようにイメージデータに格納します。
        const size_t colBytes = (rimg.IsFloat()) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
        for (int level = 1; level < rimg.GetMipCount(); ++level)
        {
            const int curW = RMax(rimg.GetImageW() >> level, 1);
            const int curH = RMax(rimg.GetImageH() >> level, 1);
            const size_t faceBytes = colBytes * curW * curH;
            for (int iDepth = 0; iDepth < rimg.GetImageD(); ++iDepth)
            {
                const int iCube = iDepth / RImage::CUBE_FACE_COUNT;
                const int iFace = iDepth % RImage::CUBE_FACE_COUNT;
                const int baseZ = iCube * RImage::CUBE_FACE_COUNT;
                uint8_t* pDstFace = pDst + (baseZ + GetCubeMapDstFaceIdx(iFace)) * faceBytes;
                memcpy(pDstFace, pSrc, faceBytes);
                pSrc += faceBytes;
            }
            pDst += rimg.GetImageD() * faceBytes;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 入力画像から作業用画像をセットアップします。
//!
//! @param[out] dimMode 次元モードを格納します。
//! @param[out] workImage 作業用画像を格納します。
//! @param[out] originalImages 元画像配列を格納します。
//! @param[in] rimg 入力画像です。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] mergeInfo マージ情報です。
//! @param[in] finalOutputPath 最終出力パスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus SetupWorkImage(
    RDimMode& dimMode,
    RImage& workImage,
    ROriginalImageArray& originalImages,
    const RImage& rimg,
    const FtxOpt& opt,
    const RMergeInfo& mergeInfo,
    const std::string& finalOutputPath
)
{
    //-----------------------------------------------------------------------------
    // 画像ファイルのパスを設定します。
    workImage.SetFilePath(opt.m_InputPaths[0]);

    //-----------------------------------------------------------------------------
    // 次元を決定します。
    dimMode = R_DIM_MODE_2D;
    bool isDimModeChanged;
    RStatus status = SetWorkImageDimension(dimMode, workImage, isDimModeChanged, rimg, opt, mergeInfo);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // 作業用画像のメンバを設定します。
    status = SetWorkImageWidthHeight(workImage, rimg, dimMode);
    RCheckStatus(status);

    if (workImage.IsCubeMap())
    {
        if (dimMode == R_DIM_MODE_CUBE_INPUT)
        {
            workImage.SetImageD(static_cast<int>(rimg.GetImageD() * opt.m_InputPaths.size()));
        }
        else
        {
            workImage.SetImageD(static_cast<int>(RImage::CUBE_FACE_COUNT * opt.m_InputPaths.size()));
        }
    }
    else if (dimMode == R_DIM_MODE_3D       ||
             dimMode == R_DIM_MODE_1D_ARRAY ||
             dimMode == R_DIM_MODE_2D_ARRAY)
    {
        workImage.SetImageD(static_cast<int>(opt.m_InputPaths.size()));
    }
    else if (dimMode == R_DIM_MODE_3D_INPUT       ||
             dimMode == R_DIM_MODE_1D_ARRAY_INPUT ||
             dimMode == R_DIM_MODE_2D_ARRAY_INPUT)
    {
        workImage.SetImageD(rimg.GetImageD());
    }
    else
    {
        workImage.SetImageD(1);
    }

    // ミップマップデータのサイズを決定します。
    const size_t mipDataSize =
        (!isDimModeChanged                   &&
         dimMode != R_DIM_MODE_CUBE_SEPARATE &&
         dimMode != R_DIM_MODE_CUBE_HC       &&
         dimMode != R_DIM_MODE_CUBE_VC       &&
         !(workImage.IsCubeMap() && workImage.GetImageD() > RImage::CUBE_FACE_COUNT)) ?
        rimg.GetMipDataSize() : 0;

    workImage.SetMipCount(1);
    workImage.SetIsFloat(rimg.IsFloat());
    workImage.SetFormat((rimg.IsFloat()) ? FtxFormat_Float_32_32_32_32 : FtxFormat_Unorm_8_8_8_8);
    const size_t colBytes = (rimg.IsFloat()) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t faceBytes = colBytes * workImage.GetImageW() * workImage.GetImageH();
    workImage.SetImageDataSize(faceBytes * workImage.GetImageD());
    uint8_t* pImageData = new uint8_t[workImage.GetImageDataSize() + mipDataSize];
    workImage.SetImageData(pImageData);
    workImage.SetExtraDataSize(mipDataSize);
    workImage.SetTileMode(FtxTileMode_Linear);
    SetWorkImageData(workImage, originalImages, rimg, dimMode, finalOutputPath);
    workImage.SetIsDecodedOriginal(rimg.IsDecodedOriginal());

    //-----------------------------------------------------------------------------
    // 入力画像にミップマップデータがあればイメージデータの後半に格納します。
    if (mipDataSize > 0)
    {
        SetWorkImageMipData(workImage, rimg);
    }

    //-----------------------------------------------------------------------------
    // ユーザーデータ、編集用コメント、ツールデータ、ユーザーツールデータを設定します。
    workImage.SetUserDatas(rimg.GetUserDatas());
    workImage.SetCommentLabel(rimg.GetCommentLabel());
    workImage.SetCommentCol(rimg.GetCommentCol());
    workImage.SetCommentText(rimg.GetCommentText());
    workImage.SetToolData(rimg.GetToolData());
    workImage.SetUserToolData(rimg.GetUserToolData());

    //-----------------------------------------------------------------------------
    // 追加属性を設定します。
    workImage.SetExtraAttrMap(rimg.GetExtraAttrMap());

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 元画像配列に 1 つでもアルファ成分があれば true を返します。
//!
//! @param[in] originalImages 元画像配列です。
//!
//! @return 元画像配列に 1 つでもアルファ成分があれば true を返します。
//-----------------------------------------------------------------------------
bool OriginalImagesHasAlpha(const ROriginalImageArray& originalImages)
{
    for (size_t iOrgImg = 0; iOrgImg < originalImages.size(); ++iOrgImg)
    {
        if (originalImages[iOrgImg].HasAlpha())
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief キューブマップ個別指定の場合の残りのファイルをリードします。
//!
//! @param[in,out] workImage 作業用画像です。
//! @param[in,out] originalImages 元画像配列です。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] mergeInfo マージ情報です。
//! @param[in] finalOutputPath 最終出力パスです。
//! @param[in] rimg 1 つ目のファイルの画像です。
//! @param[in] faceBytes 1 フェースのデータサイズ（バイト数）です。
//! @param[in] convUtil 変換ユーティリティーです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ReadCubeMapSeparateFiles(
    RImage& workImage,
    ROriginalImageArray& originalImages,
    const FtxOpt& opt,
    const std::string& finalOutputPath,
    const RImage& rimg,
    const size_t faceBytes,
    const ConverterUtility& convUtil
)
{
    //-----------------------------------------------------------------------------
    // 各フェースのパス配列を設定します。
    const std::string* pFacePaths[] =
    {
        &opt.m_CubeRightPath,   // +X
        &opt.m_CubeLeftPath,    // -X
        &opt.m_CubeTopPath,     // +Y
        &opt.m_CubeBottomPath,  // -Y
        &opt.m_InputPaths[0],   // +Z
        &opt.m_CubeBackPath,    // -Z
    };

    const int cubeCount = static_cast<int>(opt.m_InputPaths.size());
    RStringArray facePathss[RImage::CUBE_FACE_COUNT];
    for (int iFace = 0; iFace < RImage::CUBE_FACE_COUNT; ++iFace)
    {
        const std::string basePath = *pFacePaths[iFace];
        facePathss[iFace].push_back(basePath);
        if (cubeCount >= 2)
        {
            if (!AddNumberIncrementedPath(facePathss[iFace], basePath, cubeCount - 1))
            {
                return RStatus(RStatus::FAILURE, "Input file name does not have digits for depth: " + basePath); // RShowError
            }
        }
    }

    //-----------------------------------------------------------------------------
    // 各フェースのファイルをリードします。
    const size_t cubeBytes = faceBytes * RImage::CUBE_FACE_COUNT;
    for (int iCube = 0; iCube < cubeCount; ++iCube)
    {
        uint8_t* pDstTop = workImage.GetImageData() + iCube * cubeBytes;
        for (int iFace = 0; iFace < RImage::CUBE_FACE_COUNT; ++iFace)
        {
            if (iCube == 0 && iFace == 0) // 最初のキューブの +X 方向はリード済みなのでスキップします。
            {
                continue;
            }
            const std::string& facePath = facePathss[iFace][iCube];
            RImage faceImg;
            RStatus status = faceImg.ReadFile(facePath, RImage::ReadFlag_Original, opt.m_IgnoresAlpha, convUtil);
            RCheckStatus(status);
            if (faceImg.GetImageW() != workImage.GetImageW() ||
                faceImg.GetImageH() != workImage.GetImageH())
            {
                return RStatus(RStatus::FAILURE, "Cube map width or height is not identical: " + facePath); // RShowError
            }
            else if (faceImg.IsFloat() != rimg.IsFloat())
            {
                return RStatus(RStatus::FAILURE, "Cube map format is different from other face: " + facePath); // RShowError
            }

            uint8_t* pDst = pDstTop + GetCubeMapDstFaceIdx(iFace) * faceBytes;
            memcpy(pDst, faceImg.GetImageData(), faceBytes);

            const ROriginalImage::Face orgFace = static_cast<ROriginalImage::Face>(
                ROriginalImage::POSITIVE_X + iFace);
            AppendOriginalImage(originalImages, faceImg, iCube, orgFace, finalOutputPath);
        }
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief キューブマップ配列の 2 つ目以降のファイルをリードします。
//!
//! @param[in,out] workImage 作業用画像です。
//! @param[in,out] originalImages 元画像配列です。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] mergeInfo マージ情報です。
//! @param[in] finalOutputPath 最終出力パスです。
//! @param[in] rimg 1 つ目のファイルの画像です。
//! @param[in] dimMode 次元モードです。
//! @param[in] faceBytes 1 フェースのデータサイズ（バイト数）です。
//! @param[in] convUtil 変換ユーティリティーです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ReadCubeMapArrayFiles(
    RImage& workImage,
    ROriginalImageArray& originalImages,
    const FtxOpt& opt,
    const std::string& finalOutputPath,
    const RImage& rimg,
    const RDimMode dimMode,
    const size_t faceBytes,
    const ConverterUtility& convUtil
)
{
    const int cubeCount = static_cast<int>(opt.m_InputPaths.size());
    const size_t cubeBytes = faceBytes * RImage::CUBE_FACE_COUNT;
    for (int iCube = 1; iCube < cubeCount; ++iCube)
    {
        const std::string& cubePath = opt.m_InputPaths[iCube];
        RImage cubeImg;
        RStatus status = cubeImg.ReadFile(cubePath, RImage::ReadFlag_Original, opt.m_IgnoresAlpha, convUtil);
        RCheckStatus(status);
        if (cubeImg.IsFloat() != rimg.IsFloat())
        {
            return RStatus(RStatus::FAILURE, "Cube map format is different from other: " + cubePath); // RShowError
        }

        if (dimMode == R_DIM_MODE_CUBE_INPUT)
        {
            if (!cubeImg.IsCubeMap())
            {
                return RStatus(RStatus::FAILURE, "Input is not cube map: " + cubePath); // RShowError
            }
            else if (cubeImg.GetImageW() != workImage.GetImageW() ||
                     cubeImg.GetImageH() != workImage.GetImageH())
            {
                return RStatus(RStatus::FAILURE, "Cube map width or height is not identical: " + cubePath); // RShowError
            }
            const uint8_t* pSrc = cubeImg.GetImageData();
            for (int iFace = 0; iFace < RImage::CUBE_FACE_COUNT; ++iFace)
            {
                uint8_t* pDst = workImage.GetImageData() + iCube * cubeBytes +
                    GetCubeMapDstFaceIdx(iFace) * faceBytes;
                memcpy(pDst, pSrc, faceBytes);

                std::string originalPath = (!cubeImg.GetOriginalPaths().empty()) ?
                    cubeImg.GetOriginalFullPath(iFace) :
                    cubeImg.GetFilePath();
                originalPath = GetOriginalFilePath(finalOutputPath, originalPath);
                originalImages.push_back(ROriginalImage(
                    iCube,
                    static_cast<ROriginalImage::Face>(ROriginalImage::POSITIVE_X + iFace),
                    workImage.GetImageW(),
                    workImage.GetImageH(),
                    originalPath.c_str(),
                    pSrc,
                    &rimg));
                pSrc += faceBytes;
            }
        }
        else // 水平十字／垂直十字キューブマップの場合
        {
            if (dimMode == R_DIM_MODE_CUBE_HC)
            {
                if (workImage.GetImageW() * RImage::CUBE_HC_COUNT_H != cubeImg.GetImageW() ||
                    workImage.GetImageH() * RImage::CUBE_HC_COUNT_V != cubeImg.GetImageH())
                {
                    return RStatus(RStatus::FAILURE, "Cube map horizontal cross width or height is wrong: " + cubePath); // RShowError
                }
            }
            else
            {
                if (workImage.GetImageW() * RImage::CUBE_VC_COUNT_H != cubeImg.GetImageW() ||
                    workImage.GetImageH() * RImage::CUBE_VC_COUNT_V != cubeImg.GetImageH())
                {
                    return RStatus(RStatus::FAILURE, "Cube map vertical cross width or height is wrong: " + cubePath); // RShowError
                }
            }
            SetCrossCubeImage(workImage, originalImages, iCube, cubeImg, dimMode, finalOutputPath);
        }
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 差し替え画像ファイルをリードして各チャンネルを差し替えます。
//!        入力ファイルと差し替え画像ファイルのサイズが異なる場合は、
//!        差し替え画像ファイルのサイズを入力ファイルにあわせてから差し替えます。
//!        入力ファイルをサイズ変更する場合、サイズ変更前とサイズ変更後に呼ばれます。
//!        サイズ変更前に呼ばれた際は、差し替え画像ファイルのサイズが入力ファイルと
//!        同じなら差し替えます。
//!
//! @param[in,out] pImage 差し替え先の画像へのポインターです。
//! @param[in,out] afterReplacePaths 入力ファイルのサイズ変更前に呼ばれた場合、
//!                                  サイズ変更後に差し替える画像ファイルのパス配列を格納します
//!                                  （インデックスは RGBA 成分）。
//!                                  サイズ変更前にすべて差し替える場合は nullptr を指定します。
//!                                  サイズ変更後に呼ばれた場合、この引数で指定されたパスが
//!                                  空でなければ差し替えます（値は変更しません）。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] isAfterResize 入力ファイルのサイズ変更前に呼ぶ場合は false、
//!                          サイズ変更後に呼ぶ場合は true を指定します。
//! @param[in] usesLinearFlagForResize サイズ変更時に引数で指定したリニアー変換フラグを
//!                                    使用するなら true、差し替え画像のリニアー変換フラグを
//!                                    使用するなら false を指定します。
//! @param[in] linearFlagForResize サイズ変更時に使用するリニアー変換フラグです。
//! @param[in] convUtil 変換ユーティリティーです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ReplaceChannels(
    RImage* pImage,
    std::string* afterReplacePaths,
    const FtxOpt& opt,
    const bool isAfterResize,
    const bool usesLinearFlagForResize,
    const int linearFlagForResize,
    const ConverterUtility& convUtil
)
{
    RStatus status;
    std::string lastReplacePath;
    RImage replaceImg;
    int linearFlag = pImage->GetLinearFlag();
    for (int dstRgbaIdx = 0; dstRgbaIdx < R_RGBA_COUNT; ++dstRgbaIdx)
    {
        const std::string& replacePath = (afterReplacePaths != nullptr && isAfterResize) ?
            afterReplacePaths[dstRgbaIdx] :
            opt.m_ReplacePaths[dstRgbaIdx];
        if (!replacePath.empty())
        {
            //-----------------------------------------------------------------------------
            // 差し替え画像ファイルをリードします。
            if (replacePath != lastReplacePath)
            {
                status = replaceImg.ReadFile(replacePath, RImage::ReadFlag_Original, false, convUtil);
                RCheckStatus(status);
                lastReplacePath = replacePath;
            }

            //-----------------------------------------------------------------------------
            // ftx ファイルまたは DDS ファイルならリニアー変換フラグを差し替えます。
            const int srcRgbaIdx = opt.m_ReplaceChanIdxs[dstRgbaIdx];
            bool isDstChanLinear = false;
            if (IsFtxFilePath(replacePath) ||
                RGetExtensionFromFilePath(replacePath) == DdsExtension)
            {
                const int replaceLinear = replaceImg.GetLinearFlag();
                if (replaceLinear != FtxOpt::LINEAR_DEFAULT)
                {
                    if (linearFlag == FtxOpt::LINEAR_DEFAULT)
                    {
                        linearFlag = FtxOpt::LINEAR_NONE;
                    }
                    linearFlag &= ~LinearRgbaFlags[dstRgbaIdx];
                    if ((replaceLinear & LinearRgbaFlags[srcRgbaIdx]) != 0)
                    {
                        linearFlag |= LinearRgbaFlags[dstRgbaIdx];
                        isDstChanLinear = true;
                    }
                }
            }

            //-----------------------------------------------------------------------------
            // 差し替え画像ファイルのサイズが異なるならサイズ変更します。
            if (replaceImg.GetImageW() != pImage->GetImageW() ||
                replaceImg.GetImageH() != pImage->GetImageH() ||
                replaceImg.GetImageD() != pImage->GetImageD())
            {
                if (afterReplacePaths != nullptr && !isAfterResize)
                {
                    // 入力ファイルのサイズ変更前に呼ばれた場合、
                    // サイズ変更後に差し替えるパスを設定して差し替え処理をスキップします。
                    afterReplacePaths[dstRgbaIdx] = replacePath;
                    continue;
                }
                const int srcLinearFlagForResize =
                    (usesLinearFlagForResize && ((linearFlagForResize & LinearRgbaFlags[dstRgbaIdx]) != 0)) ||
                    (!usesLinearFlagForResize && isDstChanLinear) ?
                    FtxOpt::LINEAR_RGBA : FtxOpt::LINEAR_NONE;
                status = replaceImg.Resize(pImage->GetImageW(), pImage->GetImageH(), pImage->GetImageD(),
                    srcLinearFlagForResize, opt.m_ResizeFilter, false);
                RCheckStatus(status);
                lastReplacePath = ""; // リニアー変換フラグが異なる場合のために、次のループで常に再読み込みします。
            }

            if (afterReplacePaths != nullptr && !isAfterResize)
            {
                afterReplacePaths[dstRgbaIdx].clear();
            }

            //-----------------------------------------------------------------------------
            // チャンネルを差し替えます。
            //cerr << "replace" << dstRgbaIdx << ": " << replacePath << " (" << srcRgbaIdx << ")" << endl;
            status = pImage->ReplaceChannel(dstRgbaIdx, replaceImg, srcRgbaIdx);
            RCheckStatus(status);
        }
    }
    if (!isAfterResize)
    {
        pImage->SetLinearFlag(linearFlag);
    }
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 画像ファイルをリードして作業用画像と元画像を設定します。
//!
//! @param[in,out] workImage 作業用画像です。
//! @param[out] originalImages 元画像配列です。
//! @param[in,out] pOpt ftx 変換オプションへのポインターです。
//! @param[in] mergeInfo マージ情報です。
//! @param[in] finalOutputPath 最終出力パスです。
//! @param[in] convUtil 変換ユーティリティーです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ReadImageFile(
    RImage& workImage,
    ROriginalImageArray& originalImages,
    FtxOpt* pOpt,
    const RMergeInfo& mergeInfo,
    const std::string& finalOutputPath,
    const ConverterUtility& convUtil
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // 最初の入力ファイルをリードします。
    FtxOpt& opt = *pOpt;
    const std::string& firstPath = (opt.IsCubeMapSeparete()) ?
        opt.m_CubeRightPath : opt.m_InputPaths[0];
    RImage rimg;
    status = rimg.ReadFile(firstPath, RImage::ReadFlag_Original, opt.m_IgnoresAlpha, convUtil);
    RCheckStatus(status);
    //cerr << "rimg format: " << rimg.GetFormat() << endl;

    //-----------------------------------------------------------------------------
    // 差し替え画像ファイルをリードして各チャンネルを差し替えます。
    bool usesLinearFlagForResize = true;
    int linearFlagForResize = FtxOpt::LINEAR_NONE;
    if (opt.m_ResizesInLinear)
    {
        ROriginalImageArray dummyOriginalImages;
        FtxFormat formatForResize = FtxFormat_Unorm_8_8_8_8;
        DecideFormat(&formatForResize, opt, mergeInfo, dummyOriginalImages, &rimg);
        linearFlagForResize = DecideLinearFlag(&usesLinearFlagForResize,
            opt, mergeInfo, dummyOriginalImages, &rimg, formatForResize);
    }
    std::string* afterReplacePaths = (
        opt.IsResizeSpecified()                  &&
        !opt.m_CropRect.IsEnabled()              &&
        !IsToCubeDimension(rimg, opt, mergeInfo)) ?
        opt.m_AfterReplacePaths : nullptr;
    status = ReplaceChannels(&rimg, afterReplacePaths,
        opt, false, usesLinearFlagForResize, linearFlagForResize, convUtil);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // 作業用画像をセットアップします。
    RDimMode dimMode;
    SetupWorkImage(dimMode, workImage, originalImages,
        rimg, opt, mergeInfo, finalOutputPath);
    DisplayImageDimension(cout, workImage, dimMode, opt);

    //-----------------------------------------------------------------------------
    // キューブマップ個別指定の場合の残りのファイルをリードします。
    const size_t colBytes = (rimg.IsFloat()) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t faceBytes = colBytes * workImage.GetImageW() * workImage.GetImageH();
    if (dimMode == R_DIM_MODE_CUBE_SEPARATE)
    {
        status = ReadCubeMapSeparateFiles(workImage, originalImages, opt, finalOutputPath, rimg, faceBytes, convUtil);
        RCheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // 3D、1D 配列、2D 配列の場合の残りのファイルをリードします。
    if (dimMode == R_DIM_MODE_3D       ||
        dimMode == R_DIM_MODE_1D_ARRAY ||
        dimMode == R_DIM_MODE_2D_ARRAY)
    {
        const std::string errTop = (dimMode == R_DIM_MODE_3D) ? "3D" : "Array";
        for (int iDepth = 1; iDepth < workImage.GetImageD(); ++iDepth)
        {
            const std::string& facePath = opt.m_InputPaths[iDepth];
            RImage faceImg;
            status = faceImg.ReadFile(facePath, RImage::ReadFlag_Original, opt.m_IgnoresAlpha, convUtil);
            RCheckStatus(status);
            if (faceImg.GetImageW() != static_cast<int>(workImage.GetImageW()) ||
                (dimMode != R_DIM_MODE_1D_ARRAY && faceImg.GetImageH() != workImage.GetImageH()))
            {
                return RStatus(RStatus::FAILURE, errTop + " width or height is not identical: " + facePath); // RShowError
            }
            else if (faceImg.IsFloat() != rimg.IsFloat())
            {
                return RStatus(RStatus::FAILURE, errTop + " format is different from other face: " + facePath); // RShowError
            }

            uint8_t* dstPtr = workImage.GetImageData() + iDepth * faceBytes;
            memcpy(dstPtr, faceImg.GetImageData(), faceBytes);

            AppendOriginalImage(originalImages, faceImg, iDepth, ROriginalImage::NONE, finalOutputPath);
        }
    }

    //-----------------------------------------------------------------------------
    // キューブマップ配列の場合の残りのファイルをリードします。
    if (workImage.IsCubeMap() &&
        dimMode != R_DIM_MODE_CUBE_SEPARATE &&
        opt.m_InputPaths.size() >= 2)
    {
        status = ReadCubeMapArrayFiles(workImage, originalImages, opt, finalOutputPath, rimg, dimMode, faceBytes, convUtil);
        RCheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // アルファ成分フラグを設定します。
    workImage.SetHasAlpha(OriginalImagesHasAlpha(originalImages));

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルのデータを解析して作業用画像と元画像を設定します。
//!
//! @param[in,out] workImage 作業用画像です。
//! @param[out] originalImages 元画像配列です。
//! @param[in,out] pEncoder エンコーダーへのポインタです。
//! @param[in] pData ftx ファイルのデータです。
//! @param[in] dataSize ftx ファイルのデータサイズです。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] mergeInfo マージ情報です。
//! @param[in] finalOutputPath 最終出力パスです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseFtxData(
    RImage& workImage,
    ROriginalImageArray& originalImages,
    REncoder* pEncoder,
    const void* pData,
    const size_t dataSize,
    const FtxOpt& opt,
    const RMergeInfo& mergeInfo,
    const std::string& finalOutputPath
)
{
    //-----------------------------------------------------------------------------
    // ftx ファイルのデータをデコードします。
    RImage rimg;
    rimg.SetFilePath(opt.m_InputPaths[0]);
    RStatus status = rimg.DecodeFtx(pEncoder,
        reinterpret_cast<const uint8_t*>(pData), dataSize,
        IsFtxBinaryFilePath(opt.m_InputPaths[0]), RImage::ReadFlag_Original);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // アルファ成分が不要なら削除します。
    if (opt.m_IgnoresAlpha)
    {
        rimg.RemoveAlpha();
    }

    //-----------------------------------------------------------------------------
    // 作業用画像をセットアップします。
    RDimMode dimMode;
    SetupWorkImage(dimMode, workImage, originalImages,
        rimg, opt, mergeInfo, finalOutputPath);
    DisplayImageDimension(cout, workImage, dimMode, opt);

    //-----------------------------------------------------------------------------
    // アルファ成分フラグを設定します。
    workImage.SetHasAlpha(OriginalImagesHasAlpha(originalImages));

    return status;
}

//-----------------------------------------------------------------------------
//! @brief ビットマップデータを解析して作業用画像と元画像を設定します。
//!
//! @param[in,out] workImage 作業用画像です。
//! @param[out] originalImages 元画像配列です。
//! @param[in] pData ビットマップデータです。
//!                  1 ピクセルあたり 4 または 16 バイトで、RGBA の順に格納します。画像の左上が先頭です。
//!                  キューブマップ（6 面個別指定）の場合、+X、-X、+Y、-Y、+Z、-Z 面の順に格納します。
//!                  ビットマップデータのサイズは width * height * depth * (4 または 16) となります。
//! @param[in] width ビットマップの幅です。
//! @param[in] height ビットマップの高さです。
//! @param[in] depth ビットマップの深さです。
//!                  水平十字／垂直十字キューブマップの場合は 1 を指定します。
//! @param[in] hasAlpha アルファ成分を持つなら true です。
//! @param[in] isFloat 浮動小数点数データなら true です。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] mergeInfo マージ情報です。
//! @param[in] finalOutputPath 最終出力パスです。
//! @param[in] originalPaths オリジナルファイルのパス配列です。
//!                          キューブマップ（6 面個別指定）の場合、+X、-X、+Y、-Y、+Z、-Z 面の順に指定します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ParseBitmapData(
    RImage& workImage,
    ROriginalImageArray& originalImages,
    const void* pData,
    const int width,
    const int height,
    const int depth,
    const bool hasAlpha,
    const bool isFloat,
    const FtxOpt& opt,
    const RMergeInfo& mergeInfo,
    const std::string& finalOutputPath,
    const wchar_t* originalPaths[]
)
{
    //-----------------------------------------------------------------------------
    // ビットマップデータをデコードします。
    RImage rimg;
    const int bitmapDim = (RIsCubMapDimension(static_cast<FtxDimension>(opt.m_Dimension)) && depth == 1) ?
        FtxDimension_2d : opt.m_Dimension;
    RStatus status = rimg.ReadBitmap(pData, bitmapDim, width, height, depth,
        hasAlpha && !opt.m_IgnoresAlpha, isFloat, opt.m_InputPaths);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // 作業用画像をセットアップします。
    RDimMode dimMode;
    SetupWorkImage(dimMode, workImage, originalImages,
        rimg, opt, mergeInfo, finalOutputPath);
    DisplayImageDimension(cout, workImage, dimMode, opt);

    //-----------------------------------------------------------------------------
    // アルファ成分フラグを設定します。
    workImage.SetHasAlpha(OriginalImagesHasAlpha(originalImages));

    //-----------------------------------------------------------------------------
    // 元画像にオリジナルファイルのパスを設定します。
    RStringArray orgPaths;
    const size_t orgPathCount = RGetStringArray(orgPaths, originalPaths);
    for (size_t iOrgImg = 0; iOrgImg < originalImages.size(); ++iOrgImg)
    {
        const std::string originalPath = (iOrgImg < orgPathCount) ?
            orgPaths[iOrgImg] : orgPaths[0];
        originalImages[iOrgImg].m_OriginalPath =
            GetOriginalFilePath(finalOutputPath, originalPath);
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief bntx ファイルのタイルモードの表示用文字列を取得します。
//!
//! @param[in] binTileMode bntx ファイルのタイルモードです。
//!
//! @return bntx ファイルのタイルモードの表示用文字列を返します。
//-----------------------------------------------------------------------------
std::string GetBinTileModeString(const std::string& binTileMode)
{
    if (binTileMode == "nx")
    {
        return "NX";
    }
    return binTileMode;
}

//-----------------------------------------------------------------------------
//! @brief bntx ファイルを出力します。
//!
//! @param[out] pOutputData bntx ファイルのデータを格納します。
//!                         ファイルに出力する場合は nullptr を指定します。
//! @param[in,out] pImages 画像へのポインタの配列です。
//! @param[in] bntxPath bntx ファイルのパスです。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] convUtil 変換ユーティリティーです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OutputBntxFile(
    std::string* pOutputData,
    const std::vector<RImage*>& pImages,
    const std::string& bntxPath,
    const FtxOpt& opt,
    const ConverterUtility& convUtil
)
{
    //-----------------------------------------------------------------------------
    // 各画像に変換オプションを設定します。
    for (size_t texIdx = 0; texIdx < pImages.size(); ++texIdx)
    {
        RImage* pImage = pImages[texIdx];
        pImage->SetTileOptimize(opt.m_TileOptimize);
        pImage->SetTileSizeThreshold(opt.m_TileSizeThreshold);
        pImage->SetSparseTiled(opt.m_IsSparseTiled);
    }

    //-----------------------------------------------------------------------------
    // bntx ファイルをオープンします。
    std::ofstream ofs;
    if (pOutputData == nullptr)
    {
        ofs.open(bntxPath.c_str(), ios::binary);
        if (!ofs)
        {
            return RStatus(RStatus::FAILURE, "Cannot open the file: " + bntxPath); // RShowError
        }
    }

    //-----------------------------------------------------------------------------
    // bntx ファイルのデータを出力します。
    std::ostringstream oss;
    std::ostream* pOs = (pOutputData == nullptr) ? &ofs : reinterpret_cast<std::ostream*>(&oss);
    std::ostream& os = *pOs;
    const std::string binName = RGetNoExtensionFilePath(RGetFileNameFromFilePath(bntxPath));
    RStatus status = OutputBntxData(os, pImages, binName,
        opt.m_BinTileMode, !opt.m_DisablesMemoryPool, convUtil);
    if (pOutputData != nullptr)
    {
        *pOutputData = (status) ? oss.str() : "";
    }
    return status;
}

//-----------------------------------------------------------------------------
//! @brief ftx / dds ファイル群をリードして bntx ファイルを出力します。
//!
//! @param[out] pOutputData bntx ファイルのデータを格納します。
//!                         ファイルに出力する場合は nullptr を指定します。
//! @param[in] bntxPath bntx ファイルのパスです。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] convUtil 変換ユーティリティーです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OutputBntxFromFtxDdsFiles(
    std::string* pOutputData,
    const std::string& bntxPath,
    const FtxOpt& opt,
    const ConverterUtility& convUtil
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // ftx / dds ファイル群をリードします。
    RStringArray imageNames;
    std::vector<RImage*> pImages;
    for (size_t inIdx = 0; inIdx < opt.m_InputPaths.size(); ++inIdx)
    {
        const std::string& inputPath = opt.m_InputPaths[inIdx];
        if (!opt.m_IsSilent)
        {
            cout << "Input   : " << inputPath << endl;
        }
        RImage* pImage = new RImage();
        pImages.push_back(pImage);
        status = pImage->ReadFile(inputPath, RImage::ReadFlag_Encoded, false, convUtil); // エンコードされたデータのみ取得します。
        if (!status)
        {
            break;
        }

        // 同名の入力ファイルが複数指定されていればエラーにします。
        const std::string imageName = pImage->GetName();
        if (RFindValueInArray(imageNames, imageName) != -1)
        {
            status = RStatus(RStatus::FAILURE, "It is impossible to convert same name textures: " // RShowError
                + imageName);
            break;
        }

        // 成分選択オプションを反映します。
        if (opt.m_CompSel != FtxOpt::COMP_SEL_DEFAULT)
        {
            pImage->SetCompSel(opt.m_CompSel);
        }

        imageNames.push_back(imageName);
    }

    //-----------------------------------------------------------------------------
    // bntx ファイルを出力します。
    if (status)
    {
        if (!opt.m_IsSilent)
        {
            cout << "Output  : " << bntxPath << endl;
            if (!opt.m_BinTileMode.empty())
            {
                cout << "TileMode: " << GetBinTileModeString(opt.m_BinTileMode) << endl;
            }
        }
        status = OutputBntxFile(pOutputData, pImages, bntxPath, opt, convUtil);
    }

    //-----------------------------------------------------------------------------
    // 画像を解放します。
    for (size_t imgIdx = 0; imgIdx < pImages.size(); ++imgIdx)
    {
        delete pImages[imgIdx];
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 作業用画像から bntx ファイルを出力します。
//!
//! @param[out] pOutputData bntx ファイルのデータを格納します。
//!                         ファイルに出力する場合は nullptr を指定します。
//! @param[in,out] pWorkImage 作業用画像へのポインタです。
//! @param[in] bntxPath bntx ファイルのパスです。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] convUtil 変換ユーティリティーです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OutputBntxFromWorkImage(
    std::string* pOutputData,
    RImage* pWorkImage,
    const std::string& bntxPath,
    const FtxOpt& opt,
    const ConverterUtility& convUtil
)
{
    pWorkImage->SetName(RGetNoExtensionFilePath(
        RGetFileNameFromFilePath(opt.m_InputPaths[0])));
    std::vector<RImage*> pImages;
    pImages.push_back(pWorkImage);
    return OutputBntxFile(pOutputData, pImages, bntxPath, opt, convUtil);
}

//-----------------------------------------------------------------------------
//! @brief C 言語のソース形式でテクスチャーを出力します。
//!
//! @param[in] filePath 出力ファイルのパスです。
//! @param[in] workImage 作業用画像です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OutTextureAsCSource(
    const std::string& filePath,
    const RImage& workImage
)
{
    //-----------------------------------------------------------------------------
    // open file
    ofstream ofs(filePath.c_str(), ios_base::binary);
    if (!ofs)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + filePath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // get texture data stream
    RDataStream texStream(workImage.GetImageData(), workImage.GetImageDataSize(), 16);

    //-----------------------------------------------------------------------------
    // out
    std::ostream& os = ofs;
    os << "uint8_t " << "s_TexData0" << "[] =" << R_ENDL;
    os << "{" << R_ENDL;
    ROutCSourceData(os, 0, texStream.m_ByteValues, 16);
    os << "};" << R_ENDL;
    os << R_ENDL;

    //-----------------------------------------------------------------------------
    // close file
    ofs.close();

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief uint8_t 型のカラーを sRGB からリニアに変換するテーブルを作成します。
//!
//! @param[out] pTable 長さ 256 の配列です。
//! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
//-----------------------------------------------------------------------------
void CreateLinearTable(uint8_t* pTable, const bool isFloat)
{
    if (!isFloat)
    {
        // (a/b)^n = (a^n) / (b^n)
        // (v/255)^2.2 = (v^2.2) / (255^2.2)
        const float linearMul = 1.0f / powf(255.0f, SrgbGammaValue) * 255.0f;
        for (int i = 0; i < 256; ++i)
        {
            pTable[i] = static_cast<uint8_t>(powf(static_cast<float>(i), SrgbGammaValue) * linearMul);
        }
    }
    else
    {
        // 浮動小数点数型ならテーブルを使用しないのでゼロクリアしておきます。
        memset(pTable, 0x00, 256);
    }
}

//-----------------------------------------------------------------------------
//! @brief uint8_t 型のカラーをリニアから sRGB に変換するテーブルを作成します。
//!
//! @param[out] pTable 長さ 256 の配列です。
//! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
//-----------------------------------------------------------------------------
void CreateSrgbTable(uint8_t* pTable, const bool isFloat)
{
    if (!isFloat)
    {
        // (a/b)^n = (a^n) / (b^n)
        // (v/255)^(1/2.2) = (v^(1/2.2)) / (255^(1/2.2))
        const float linearMul = 1.0f / powf(255.0f, 1.0f / SrgbGammaValue) * 255.0f;
        for (int i = 0; i < 256; ++i)
        {
            pTable[i] = static_cast<uint8_t>(powf(static_cast<float>(i), 1.0f / SrgbGammaValue) * linearMul);
        }
    }
    else
    {
        // 浮動小数点数型ならテーブルを使用しないのでゼロクリアしておきます。
        memset(pTable, 0x00, 256);
    }
}

//-----------------------------------------------------------------------------
//! @brief ポイントサンプルで縮小したミップマップデータを画像に設定します。
//!
//! @param[in,out] pImage ミップマップのメモリー確保済みの画像のポインタです。
//!                       フォーマットは unorm_8_8_8_8 または
//!                       float_32_32_32_32 である必要があります。
//! @param[in] level 設定するレベルです。
//! @param[in] iDepth 奥行きインデックスです。
//-----------------------------------------------------------------------------
void ReduceByPoint(RImage* pImage, const int level, const int iDepth)
{
    const int srcLevel = level - 1;
    const int srcW = RMax(static_cast<int>(pImage->GetImageW()) >> srcLevel, 1);
    const int srcH = RMax(static_cast<int>(pImage->GetImageH()) >> srcLevel, 1);
    const int curW = RMax(static_cast<int>(pImage->GetImageW()) >>    level, 1);
    const int curH = RMax(static_cast<int>(pImage->GetImageH()) >>    level, 1);
    const int srcZ = (pImage->GetDimension() == FtxDimension_3d) ? iDepth * 2 : iDepth;
    const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(GetLevelImageData(nullptr, *pImage, srcLevel, srcZ));
    uint8_t* pDst = reinterpret_cast<uint8_t*>(GetLevelImageData(nullptr, *pImage, level, iDepth));
    const size_t colBytes = (pImage->IsFloat()) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    for (int iy = 0; iy < curH; ++iy)
    {
        const int y0 = RMin(iy * 2, srcH - 1);
        const int y0Ofs = y0 * srcW;
        for (int ix = 0; ix < curW; ++ix)
        {
            const int x0 = RMin(ix * 2, srcW - 1);
            memcpy(pDst, pSrc + (x0 + y0Ofs) * colBytes, colBytes);
            pDst += colBytes;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief リニア補間で縮小したミップマップデータを画像に設定します。
//!
//! @param[in,out] pImage ミップマップのメモリー確保済みの画像のポインタです。
//!                       フォーマットは unorm_8_8_8_8 または
//!                       float_32_32_32_32 である必要があります。
//! @param[in] level 設定するレベルです。
//! @param[in] iDepth 奥行きインデックスです。
//-----------------------------------------------------------------------------
void ReduceByLinear(RImage* pImage, const int level, const int iDepth)
{
    const int srcLevel = level - 1;
    const int srcW = RMax(static_cast<int>(pImage->GetImageW()) >> srcLevel, 1);
    const int srcH = RMax(static_cast<int>(pImage->GetImageH()) >> srcLevel, 1);
    const int curW = RMax(static_cast<int>(pImage->GetImageW()) >>    level, 1);
    const int curH = RMax(static_cast<int>(pImage->GetImageH()) >>    level, 1);
    uint8_t* pDst = reinterpret_cast<uint8_t*>(GetLevelImageData(nullptr, *pImage, level, iDepth));
    const bool isFloat = pImage->IsFloat();
    if (pImage->GetDimension() != FtxDimension_3d)
    {
        //-----------------------------------------------------------------------------
        // 3D 以外の場合は 2 x 2 ピクセルを平均します。
        const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(
            GetLevelImageData(nullptr, *pImage, srcLevel, iDepth));
        if (!isFloat)
        {
            for (int iy = 0; iy < curH; ++iy)
            {
                const int y0 = RMin(iy * 2, srcH - 1);
                const int y1 = RMin(y0 + 1, srcH - 1);
                const int y0Ofs = y0 * srcW;
                const int y1Ofs = y1 * srcW;
                for (int ix = 0; ix < curW; ++ix)
                {
                    const int x0 = RMin(ix * 2, srcW - 1);
                    const int x1 = RMin(x0 + 1, srcW - 1);
                    const uint8_t* pS0 = pSrc + (x0 + y0Ofs) * R_RGBA_COUNT;
                    const uint8_t* pS1 = pSrc + (x1 + y0Ofs) * R_RGBA_COUNT;
                    const uint8_t* pS2 = pSrc + (x0 + y1Ofs) * R_RGBA_COUNT;
                    const uint8_t* pS3 = pSrc + (x1 + y1Ofs) * R_RGBA_COUNT;
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        const int sum = (*pS0++) + (*pS1++) + (*pS2++) + (*pS3++);
                        *pDst++ = static_cast<uint8_t>(RMin(RRound(sum * 0.25f), 0xff));
                        //*pDst++ = static_cast<uint8_t>(sum / 4);
                    }
                }
            }
        }
        else
        {
            float* pDstF32 = reinterpret_cast<float*>(pDst);
            for (int iy = 0; iy < curH; ++iy)
            {
                const int y0 = RMin(iy * 2, srcH - 1);
                const int y1 = RMin(y0 + 1, srcH - 1);
                const int y0Ofs = y0 * srcW;
                const int y1Ofs = y1 * srcW;
                for (int ix = 0; ix < curW; ++ix)
                {
                    const int x0 = RMin(ix * 2, srcW - 1);
                    const int x1 = RMin(x0 + 1, srcW - 1);
                    const float* pS0 = reinterpret_cast<const float*>(pSrc) + (x0 + y0Ofs) * R_RGBA_COUNT;
                    const float* pS1 = reinterpret_cast<const float*>(pSrc) + (x1 + y0Ofs) * R_RGBA_COUNT;
                    const float* pS2 = reinterpret_cast<const float*>(pSrc) + (x0 + y1Ofs) * R_RGBA_COUNT;
                    const float* pS3 = reinterpret_cast<const float*>(pSrc) + (x1 + y1Ofs) * R_RGBA_COUNT;
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        *pDstF32++ = ((*pS0++) + (*pS1++) + (*pS2++) + (*pS3++)) / 4.0f;
                    }
                }
            }
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // 3D の場合は 2 x 2 x 2 ピクセルを平均します。
        const int srcD = RMax(static_cast<int>(pImage->GetImageD()) >> srcLevel, 1);
        const int srcZ0 = RMin(iDepth * 2 + 0, srcD - 1);
        const int srcZ1 = RMin(iDepth * 2 + 1, srcD - 1);
        const uint8_t* pSrc0 = reinterpret_cast<const uint8_t*>(
            GetLevelImageData(nullptr, *pImage, srcLevel, srcZ0));
        const uint8_t* pSrc1 = reinterpret_cast<const uint8_t*>(
            GetLevelImageData(nullptr, *pImage, srcLevel, srcZ1));
        if (!isFloat)
        {
            for (int iy = 0; iy < curH; ++iy)
            {
                const int y0 = RMin(iy * 2, srcH - 1);
                const int y1 = RMin(y0 + 1, srcH - 1);
                const int y0Ofs = y0 * srcW;
                const int y1Ofs = y1 * srcW;
                for (int ix = 0; ix < curW; ++ix)
                {
                    const int x0 = RMin(ix * 2, srcW - 1);
                    const int x1 = RMin(x0 + 1, srcW - 1);
                    const uint8_t* pS00 = pSrc0 + (x0 + y0Ofs) * R_RGBA_COUNT;
                    const uint8_t* pS01 = pSrc0 + (x1 + y0Ofs) * R_RGBA_COUNT;
                    const uint8_t* pS02 = pSrc0 + (x0 + y1Ofs) * R_RGBA_COUNT;
                    const uint8_t* pS03 = pSrc0 + (x1 + y1Ofs) * R_RGBA_COUNT;
                    const uint8_t* pS10 = pSrc1 + (x0 + y0Ofs) * R_RGBA_COUNT;
                    const uint8_t* pS11 = pSrc1 + (x1 + y0Ofs) * R_RGBA_COUNT;
                    const uint8_t* pS12 = pSrc1 + (x0 + y1Ofs) * R_RGBA_COUNT;
                    const uint8_t* pS13 = pSrc1 + (x1 + y1Ofs) * R_RGBA_COUNT;
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        const int sum = (*pS00++) + (*pS01++) + (*pS02++) + (*pS03++) +
                                        (*pS10++) + (*pS11++) + (*pS12++) + (*pS13++);
                        *pDst++ = static_cast<uint8_t>(RMin(RRound(sum * 0.125f), 0xff));
                        //*pDst++ = static_cast<uint8_t>(sum / 8);
                    }
                }
            }
        }
        else
        {
            float* pDstF32 = reinterpret_cast<float*>(pDst);
            for (int iy = 0; iy < curH; ++iy)
            {
                const int y0 = RMin(iy * 2, srcH - 1);
                const int y1 = RMin(y0 + 1, srcH - 1);
                const int y0Ofs = y0 * srcW;
                const int y1Ofs = y1 * srcW;
                for (int ix = 0; ix < curW; ++ix)
                {
                    const int x0 = RMin(ix * 2, srcW - 1);
                    const int x1 = RMin(x0 + 1, srcW - 1);
                    const float* pS00 = reinterpret_cast<const float*>(pSrc0) + (x0 + y0Ofs) * R_RGBA_COUNT;
                    const float* pS01 = reinterpret_cast<const float*>(pSrc0) + (x1 + y0Ofs) * R_RGBA_COUNT;
                    const float* pS02 = reinterpret_cast<const float*>(pSrc0) + (x0 + y1Ofs) * R_RGBA_COUNT;
                    const float* pS03 = reinterpret_cast<const float*>(pSrc0) + (x1 + y1Ofs) * R_RGBA_COUNT;
                    const float* pS10 = reinterpret_cast<const float*>(pSrc1) + (x0 + y0Ofs) * R_RGBA_COUNT;
                    const float* pS11 = reinterpret_cast<const float*>(pSrc1) + (x1 + y0Ofs) * R_RGBA_COUNT;
                    const float* pS12 = reinterpret_cast<const float*>(pSrc1) + (x0 + y1Ofs) * R_RGBA_COUNT;
                    const float* pS13 = reinterpret_cast<const float*>(pSrc1) + (x1 + y1Ofs) * R_RGBA_COUNT;
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        *pDstF32++ = (
                            (*pS00++) + (*pS01++) + (*pS02++) + (*pS03++) +
                            (*pS10++) + (*pS11++) + (*pS12++) + (*pS13++)) / 8.0f;
                    }
                }
            }
        }
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief キュービック補間で縮小したミップマップデータを画像に設定します。
//!        【参考にしたサイト】
//!        http://koujinz.cocolog-nifty.com/blog/2009/05/bicubic-a97c.html
//!        http://csharpimage.blog60.fc2.com/blog-entry-20.html
//!
//! @param[in,out] pImage ミップマップのメモリー確保済みの画像のポインタです。
//!                       フォーマットは unorm_8_8_8_8 または
//!                       float_32_32_32_32 である必要があります。
//! @param[in] level 設定するレベルです。
//! @param[in] iDepth 奥行きインデックスです。
//-----------------------------------------------------------------------------
void ReduceByCubic(RImage* pImage, const int level, const int iDepth)
{
    // bicubic_weight(d) = 1 - 2 * d^2 + d^3         (d < 1.0)
    //                     4 - 8 * d + 5 * d^2 - d^3 (1.0 ≦ d < 2.0)
    //                     0                         (2.0 ≦ d)

    // d = 0.5 ->  0.625 ( 5/8)
    // d = 1.5 -> -0.125 (-1/8)

    const int SAMPLE_COUNT = 4;
    static const float s_Linear1d[SAMPLE_COUNT] = { 0.0f, 0.5f, 0.5f, 0.0f };
    static const float s_Cubic1d[SAMPLE_COUNT] = { -0.125f, 0.625f, 0.625f, -0.125f }; // 通常のキュービック（シャープ）
    //static const float s_Cubic1d[SAMPLE_COUNT] = { -0.1f, 0.6f, 0.6f, -0.1f }; // 少しシャープ
    //static const float s_Cubic1d[SAMPLE_COUNT] = { 0.125f, 0.375f, 0.375f, 0.125f }; // リニアよりぼかす（NVIDIA DDS の Cubic に近い）

    const int srcLevel = level - 1;
    //const int srcLevel = 0; // 最上位レベルから縮小するテスト
    const int srcRate = 1 << (level - srcLevel);
    const int srcOfs = (srcRate >> 1) - 1; // 上位レベルのどこを使用するかのオフセットです。
        // 1/4  なら 1 -> 0 (1) 2 3
        // 1/8  なら 3 -> 0 1 [2 (3) 4 5] 6 7
        // 1/16 なら 7 -> 0 1 2 3 4 5 [6 (7) 8 9] A B C D E F

    const int srcW = RMax(static_cast<int>(pImage->GetImageW()) >> srcLevel, 1);
    const int srcH = RMax(static_cast<int>(pImage->GetImageH()) >> srcLevel, 1);
    const int curW = RMax(static_cast<int>(pImage->GetImageW()) >>    level, 1);
    const int curH = RMax(static_cast<int>(pImage->GetImageH()) >>    level, 1);
    uint8_t* pDst = reinterpret_cast<uint8_t*>(GetLevelImageData(nullptr, *pImage, level, iDepth));
    const bool isFloat = pImage->IsFloat();

    float* pDstF32 = reinterpret_cast<float*>(pDst);
    const int lineValCount = srcW * R_RGBA_COUNT;
    if (pImage->GetDimension() != FtxDimension_3d)
    {
        //-----------------------------------------------------------------------------
        // 3D 以外の場合は 4 x 4 ピクセルを重み付け平均します。

        // バイキュービック用の重みを計算します。
        //float cubic2dSum = 0.0f;
        float cubic2d[SAMPLE_COUNT][SAMPLE_COUNT];
        for (int sy = 0; sy < SAMPLE_COUNT; ++sy)
        {
            for (int sx = 0; sx < SAMPLE_COUNT; ++sx)
            {
                cubic2d[sy][sx] = s_Cubic1d[sy] * s_Cubic1d[sx];
                //cubic2dSum += cubic2d[sy][sx]; // s_Cubic1d の合計が 1.0 なら cubic2d の合計も 1.0
            }
        }

        const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(
            GetLevelImageData(nullptr, *pImage, srcLevel, iDepth));
        const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
        //float edgeWeightsY[SAMPLE_COUNT];
        for (int iy = 0; iy < curH; ++iy)
        {
            const int iy2 = iy * srcRate + srcOfs;
            const int yofss[SAMPLE_COUNT] =
            {
                ((iy2 > 0) ? iy2 - 1 : 0) * lineValCount,
                RMin(iy2    , srcH - 1) * lineValCount,
                RMin(iy2 + 1, srcH - 1) * lineValCount,
                RMin(iy2 + 2, srcH - 1) * lineValCount
            };
            const float* pWeightsY = (0 < iy2 && iy2 + 2 < srcH) ? s_Cubic1d : s_Linear1d;
            //const float* pWeightsY = s_Cubic1d;
            //if (iy2 == 0 || iy2 + 1 >= srcH)
            //{
            //  pWeightsY = edgeWeightsY;
            //  memcpy(edgeWeightsY, s_Cubic1d, sizeof(edgeWeightsY));
            //  if (iy2 == 0)
            //  {
            //      edgeWeightsY[0] = 0.0f;
            //      //edgeWeightsY[1] = 0.5f;
            //  }
            //  if (iy2 + 1 >= srcH)
            //  {
            //      edgeWeightsY[2] = 0.5f;
            //      //edgeWeightsY[3] = 0.0f;
            //  }
            //}

            for (int ix = 0; ix < curW; ++ix)
            {
                const int ix2 = ix * srcRate + srcOfs;
                const int xofss[SAMPLE_COUNT] =
                {
                    ((ix2 > 0) ? ix2 - 1 : 0) * R_RGBA_COUNT,
                    RMin(ix2    , srcW - 1) * R_RGBA_COUNT,
                    RMin(ix2 + 1, srcW - 1) * R_RGBA_COUNT,
                    RMin(ix2 + 2, srcW - 1) * R_RGBA_COUNT
                };
                const float* pWeightsX = (0 < ix2 && ix2 + 2 < srcW) ? s_Cubic1d : s_Linear1d;

                if (!isFloat)
                {
                    //-----------------------------------------------------------------------------
                    // uint8_t 型
                    if (pWeightsX == s_Cubic1d && pWeightsY == s_Cubic1d)
                    {
                        // イメージの端でない部分はすべてキュービック補間
                        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                        {
                            float sum = 0.0f;
                            for (int sy = 0; sy < SAMPLE_COUNT; ++sy)
                            {
                                for (int sx = 0; sx < SAMPLE_COUNT; ++sx)
                                {
                                    sum += cubic2d[sy][sx] * pSrc[yofss[sy] + xofss[sx] + iRgba];
                                }
                            }
                            *pDst++ = static_cast<uint8_t>(RClampValue(0x00, 0xff, RRound(sum)));
                        }
                    }
                    else
                    {
                        // イメージの端の部分は部分的にリニア補間
                        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                        {
                            float sum = 0.0f;
                            for (int sy = 0; sy < SAMPLE_COUNT; ++sy)
                            {
                                const float weightY = pWeightsY[sy];
                                for (int sx = 0; sx < SAMPLE_COUNT; ++sx)
                                {
                                    sum += weightY * pWeightsX[sx] * pSrc[yofss[sy] + xofss[sx] + iRgba];
                                }
                            }
                            *pDst++ = static_cast<uint8_t>(RClampValue(0x00, 0xff, RRound(sum)));
                        }
                    }
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // float 型
                    if (pWeightsX == s_Cubic1d && pWeightsY == s_Cubic1d)
                    {
                        // イメージの端でない部分はすべてキュービック補間
                        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                        {
                            float sum = 0.0f;
                            for (int sy = 0; sy < SAMPLE_COUNT; ++sy)
                            {
                                for (int sx = 0; sx < SAMPLE_COUNT; ++sx)
                                {
                                    sum += cubic2d[sy][sx] * pSrcF32[yofss[sy] + xofss[sx] + iRgba];
                                }
                            }
                            *pDstF32++ = RMax(sum, 0.0f);
                        }
                    }
                    else
                    {
                        // イメージの端の部分は部分的にリニア補間
                        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                        {
                            float sum = 0.0f;
                            for (int sy = 0; sy < SAMPLE_COUNT; ++sy)
                            {
                                const float weightY = pWeightsY[sy];
                                for (int sx = 0; sx < SAMPLE_COUNT; ++sx)
                                {
                                    sum += weightY * pWeightsX[sx] * pSrcF32[yofss[sy] + xofss[sx] + iRgba];
                                }
                            }
                            *pDstF32++ = RMax(sum, 0.0f);
                        }
                    }
                } // isFloat
            } // ix
        } // iy
    }
    else
    {
        //-----------------------------------------------------------------------------
        // 3D の場合は 4 x 4 x 4 ピクセルを重み付け平均します。
        const int srcD = RMax(static_cast<int>(pImage->GetImageD()) >> srcLevel, 1);
        const int iz2 = iDepth * srcRate + srcOfs;
        const int zofss[SAMPLE_COUNT] =
        {
            (iz2 > 0) ? iz2 - 1 : 0,
            RMin(iz2    , srcD - 1),
            RMin(iz2 + 1, srcD - 1),
            RMin(iz2 + 2, srcD - 1)
        };
        const float* pWeightsZ = (0 < iz2 && iz2 + 2 < srcD) ? s_Cubic1d : s_Linear1d;
        const uint8_t* pSrcZs[SAMPLE_COUNT] =
        {
            reinterpret_cast<const uint8_t*>(GetLevelImageData(nullptr, *pImage, srcLevel, zofss[0])),
            reinterpret_cast<const uint8_t*>(GetLevelImageData(nullptr, *pImage, srcLevel, zofss[1])),
            reinterpret_cast<const uint8_t*>(GetLevelImageData(nullptr, *pImage, srcLevel, zofss[2])),
            reinterpret_cast<const uint8_t*>(GetLevelImageData(nullptr, *pImage, srcLevel, zofss[3]))
        };

        for (int iy = 0; iy < curH; ++iy)
        {
            const int iy2 = iy * srcRate + srcOfs;
            const int yofss[SAMPLE_COUNT] =
            {
                ((iy2 > 0) ? iy2 - 1 : 0) * lineValCount,
                RMin(iy2    , srcH - 1) * lineValCount,
                RMin(iy2 + 1, srcH - 1) * lineValCount,
                RMin(iy2 + 2, srcH - 1) * lineValCount
            };
            const float* pWeightsY = (0 < iy2 && iy2 + 2 < srcH) ? s_Cubic1d : s_Linear1d;

            for (int ix = 0; ix < curW; ++ix)
            {
                const int ix2 = ix * srcRate + srcOfs;
                const int xofss[SAMPLE_COUNT] =
                {
                    ((ix2 > 0) ? ix2 - 1 : 0) * R_RGBA_COUNT,
                    RMin(ix2    , srcW - 1) * R_RGBA_COUNT,
                    RMin(ix2 + 1, srcW - 1) * R_RGBA_COUNT,
                    RMin(ix2 + 2, srcW - 1) * R_RGBA_COUNT
                };
                const float* pWeightsX = (0 < ix2 && ix2 + 2 < srcW) ? s_Cubic1d : s_Linear1d;

                if (!isFloat)
                {
                    //-----------------------------------------------------------------------------
                    // uint8_t 型
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        float sumZ = 0.0f;
                        for (int sz = 0; sz < SAMPLE_COUNT; ++sz)
                        {
                            const uint8_t* pSrcZ = pSrcZs[sz];
                            float sumXY = 0.0f;
                            for (int sy = 0; sy < SAMPLE_COUNT; ++sy)
                            {
                                const float weightY = pWeightsY[sy];
                                for (int sx = 0; sx < SAMPLE_COUNT; ++sx)
                                {
                                    sumXY += weightY * pWeightsX[sx] * pSrcZ[yofss[sy] + xofss[sx] + iRgba];
                                }
                            }
                            sumZ += pWeightsZ[sz] * sumXY;
                        }
                        *pDst++ = static_cast<uint8_t>(RClampValue(0x00, 0xff, RRound(sumZ)));
                    }
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // float 型
                    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
                    {
                        float sumZ = 0.0f;
                        for (int sz = 0; sz < SAMPLE_COUNT; ++sz)
                        {
                            const float* pSrcZF32 = reinterpret_cast<const float*>(pSrcZs[sz]);
                            float sumXY = 0.0f;
                            for (int sy = 0; sy < SAMPLE_COUNT; ++sy)
                            {
                                const float weightY = pWeightsY[sy];
                                for (int sx = 0; sx < SAMPLE_COUNT; ++sx)
                                {
                                    sumXY += weightY * pWeightsX[sx] * pSrcZF32[yofss[sy] + xofss[sx] + iRgba];
                                }
                            }
                            sumZ += pWeightsZ[sz] * sumXY;
                        }
                        *pDstF32++ = RMax(sumZ, 0.0f);
                    }
                } // isFloat
            } // ix
        } // iy
    } // is3D
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 1 フェースの上位レベルを縮小したミップマップデータを画像に設定します。
//!
//! @param[in,out] pImage ミップマップのメモリー確保済みの画像のポインタです。
//!                       フォーマットは unorm_8_8_8_8 または
//!                       float_32_32_32_32 である必要があります。
//! @param[in] level 設定するレベルです。
//! @param[in] iDepth 奥行きインデックスです。
//! @param[in] mipGenFilter ミップマップ生成フィルタです。
//-----------------------------------------------------------------------------
void ReduceFaceFromUpperLevel(
    RImage* pImage,
    const int level,
    const int iDepth,
    const FtxOpt::MipGenFilter mipGenFilter
)
{
    if (level != 0)
    {
        switch (mipGenFilter)
        {
            case FtxOpt::MIP_GEN_POINT:
                ReduceByPoint(pImage, level, iDepth);
                break;
            default:
            case FtxOpt::MIP_GEN_LINEAR:
                ReduceByLinear(pImage, level, iDepth);
                break;
            case FtxOpt::MIP_GEN_CUBIC:
                ReduceByCubic(pImage, level, iDepth);
                break;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 上位レベルを縮小したミップマップデータを画像に設定します。
//!
//! @param[in,out] pImage ミップマップのメモリー確保済みの画像のポインタです。
//!                       フォーマットは unorm_8_8_8_8 または
//!                       float_32_32_32_32 である必要があります。
//! @param[in] level 設定するレベルです。
//! @param[in] mipGenFilter ミップマップ生成フィルタです。
//-----------------------------------------------------------------------------
void ReduceFromUpperLevel(
    RImage* pImage,
    const int level,
    const FtxOpt::MipGenFilter mipGenFilter
)
{
    const int curD = (pImage->GetDimension() == FtxDimension_3d) ?
        RMax(pImage->GetImageD() >> level, 1) : pImage->GetImageD();
    for (int iDepth = 0; iDepth < curD; ++iDepth)
    {
        ReduceFaceFromUpperLevel(pImage, level, iDepth, mipGenFilter);
    }
}

//-----------------------------------------------------------------------------
//! @brief 各レベルのミップマップデータを作成します。
//!
//! @param[in,out] pImage 画像のポインタです。
//! @param[in] minLevelWhd 最小レベルの幅または高さまたは奥行きです。
//! @param[in] mipGenFilter ミップマップ生成フィルタです。
//! @param[in] originalMipCount 元画像ファイルのミップマップのレベル数です。
//-----------------------------------------------------------------------------
void GenerateMipLevels(
    RImage* pImage,
    const int minLevelWhd,
    const FtxOpt::MipGenFilter mipGenFilter,
    const int originalMipCount
)
{
    //-----------------------------------------------------------------------------
    // ミップマップのレベル数を求めます。
    int curWhd = (pImage->GetDimension() == FtxDimension_3d) ?
        RMax(pImage->GetImageW(), RMax(pImage->GetImageH(), pImage->GetImageD())) :
        RMax(pImage->GetImageW(), pImage->GetImageH());
    int mipCount = 1;
    while (curWhd >= 2 && curWhd > minLevelWhd)
    {
        ++mipCount;
        curWhd >>= 1;
    }
    mipCount = RMin(mipCount, RImage::MipCountMax);

    //-----------------------------------------------------------------------------
    // 変換後のサイズを求めます。
    const bool isFloat = pImage->IsFloat();
    const size_t colBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    size_t dstImageDataSize = 0;
    pImage->ClearLevelOffsets();
    for (int level = 0; level < mipCount; ++level)
    {
        pImage->SetLevelOffset(level, dstImageDataSize);
        const int curW = RMax(pImage->GetImageW() >> level, 1);
        const int curH = RMax(pImage->GetImageH() >> level, 1);
        const int curD = (pImage->GetDimension() == FtxDimension_3d) ?
            RMax(pImage->GetImageD() >> level, 1) : pImage->GetImageD();
        const size_t levelDataSize = colBytes * curW * curH * curD;
        dstImageDataSize += levelDataSize;
    }

    //-----------------------------------------------------------------------------
    // 変換先のイメージデータを確保します。
    uint8_t* pDstImageData = new uint8_t[dstImageDataSize];

    //-----------------------------------------------------------------------------
    // レベル 0 のデータをコピーします。
    const uint8_t* pSrcImageData = pImage->GetImageData();
    const size_t srcImageDataSize = pImage->GetImageDataSize();
    memcpy(pDstImageData, pSrcImageData, srcImageDataSize);

    //-----------------------------------------------------------------------------
    // 画像のパラメータを設定します。
    pImage->SetMipCount(mipCount);
    pImage->SetImageData(pDstImageData);
    pImage->SetImageDataSize(dstImageDataSize);
    pImage->SetMipDataSize((mipCount >= 2) ? dstImageDataSize - pImage->GetLevelOffset(1) : 0);

    //-----------------------------------------------------------------------------
    // 各レベルのデータを生成します。
    // キュービック補間の場合、まずリニア補間のデータを生成して、
    // サイズの小さいレベルから順に上位レベルのデータをキュービック補間して設定します。
    if (pImage->GetExtraDataSize() == 0 || originalMipCount != mipCount)
    {
        #ifdef MIPMAP_CUBIC_AFTER_LINEAR
        const bool isMipGenCubic = (mipGenFilter == FtxOpt::MIP_GEN_CUBIC);
        const FtxOpt::MipGenFilter firstFilter = (isMipGenCubic) ? FtxOpt::MIP_GEN_LINEAR : mipGenFilter;
        #else
        const FtxOpt::MipGenFilter firstFilter = mipGenFilter;
        #endif
        for (int level = 1; level < mipCount; ++level)
        {
            ReduceFromUpperLevel(pImage, level, firstFilter);
        }

        #ifdef MIPMAP_CUBIC_AFTER_LINEAR
        if (isMipGenCubic)
        {
            for (int level = mipCount - 1; level >= 1; --level)
            {
                ReduceFromUpperLevel(pImage, level, mipGenFilter);
            }
        }
        #endif
    }

    //-----------------------------------------------------------------------------
    // 変換オプションでミップマップのレベル数が指定されてなくて、
    // 元画像がミップマップデータを持つならミップマップデータの内容を上書きします。
    if (pImage->GetExtraDataSize() != 0 && originalMipCount <= mipCount)
    {
        memcpy(pDstImageData + pImage->GetLevelOffset(1),
            reinterpret_cast<const uint8_t*>(pSrcImageData) + srcImageDataSize,
            pImage->GetExtraDataSize());
    }
    pImage->SetExtraDataSize(0);

    //-----------------------------------------------------------------------------
    // 元画像のイメージデータを解放します。
    delete[] pSrcImageData;
}

//-----------------------------------------------------------------------------
//! @brief ミップマップ画像を上書きします。
//!
//! @param[in,out] pImage ミップマップ作成済みの画像のポインタです。
//!                       フォーマットは unorm_8_8_8_8 または
//!                       float_32_32_32_32 である必要があります。
//! @param[in] mipGenFilter ミップマップ生成フィルタです。
//! @param[in] dstLinearFlag リニア変換フラグです。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] convUtil 変換ユーティリティーです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OverwriteMipImage(
    RImage* pImage,
    const FtxOpt::MipGenFilter mipGenFilter,
    const int dstLinearFlag,
    const FtxOpt& opt,
    const ConverterUtility& convUtil
)
{
    const bool isFloat = pImage->IsFloat();
    const size_t colBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;

    #ifdef MIPMAP_CUBIC_AFTER_LINEAR
    const bool isMipGenCubic = (mipGenFilter == FtxOpt::MIP_GEN_CUBIC);
    const FtxOpt::MipGenFilter firstFilter = (isMipGenCubic) ? FtxOpt::MIP_GEN_LINEAR : mipGenFilter;
    #else
    const FtxOpt::MipGenFilter firstFilter = mipGenFilter;
    #endif

    if (!opt.IsCubeMapSeparete())
    {
        //-----------------------------------------------------------------------------
        // キューブマップ個別指定でない場合
        RIntArray reducedLevels;
        int lastOverwriteLevel = FtxOpt::MipCountMax;
        for (int level = 1; level < pImage->GetMipCount(); ++level)
        {
            const int curD = (pImage->GetDimension() == FtxDimension_3d) ?
                RMax(pImage->GetImageD() >> level, 1) : pImage->GetImageD();
            const std::string& mipPath = opt.m_MipPaths[level];
            if (!mipPath.empty())
            {
                //-----------------------------------------------------------------------------
                // ミップマップ画像をリードします。
                RImage mipImg;
                RStatus status = mipImg.ReadFile(mipPath, RImage::ReadFlag_Original, opt.m_IgnoresAlpha, convUtil);
                RCheckStatus(status);
                mipImg.ChangeDataType(isFloat, true);

                //-----------------------------------------------------------------------------
                // ミップマップ画像の作業用画像を作成し、
                // 次元、幅、高さ、奥行きをチェックします。
                RImage mipWork("MipWork");
                RDimMode dimMode;
                ROriginalImageArray mipOriginalImages;
                FtxOpt mipOpt;
                mipOpt.m_InputPaths.push_back(mipPath);
                if (opt.m_Dimension == FtxDimension_1d ||
                    RIsCubMapDimension(static_cast<FtxDimension>(opt.m_Dimension)))
                {
                    mipOpt.m_Dimension = opt.m_Dimension;
                }
                else if (opt.m_Dimension == FtxDimension_1dArray)
                {
                    mipOpt.m_Dimension = FtxDimension_1d;
                }
                SetupWorkImage(dimMode, mipWork, mipOriginalImages,
                    mipImg, mipOpt, RMergeInfo(), "");
                //mipWork.DisplayInfo(cout);

                const int curW = RMax(pImage->GetImageW() >> level, 1);
                const int curH = RMax(pImage->GetImageH() >> level, 1);
                if ((pImage->GetDimension() == FtxDimension_1d && mipWork.GetDimension() != pImage->GetDimension()) ||
                    (pImage->GetDimension() == FtxDimension_2d && mipWork.GetDimension() != pImage->GetDimension()) ||
                    (pImage->IsCubeMap() && mipWork.IsCubeMap() != pImage->IsCubeMap()))
                {
                    return RStatus(RStatus::FAILURE, "Mipmap image dimension is different from base image: " + mipPath); // RShowError
                }
                else if (static_cast<int>(mipWork.GetImageW()) != curW ||
                         static_cast<int>(mipWork.GetImageH()) != curH)
                {
                    return RStatus(RStatus::FAILURE, "Mipmap image width or height is wrong: " + mipPath); // RShowError
                }
                else if (static_cast<int>(mipWork.GetImageD()) > curD)
                {
                    return RStatus(RStatus::FAILURE, "Mipmap image depth is wrong: " + mipPath); // RShowError
                }

                //-----------------------------------------------------------------------------
                // 必要があれば sRGB からリニアに変換します。
                if (dstLinearFlag != FtxOpt::LINEAR_NONE)
                {
                    mipWork.ConvertToLinear(dstLinearFlag);
                }

                //-----------------------------------------------------------------------------
                // イメージデータをコピーします。
                memcpy(GetLevelImageData(nullptr, *pImage, level, 0), mipWork.GetImageData(), mipWork.GetImageDataSize());
                lastOverwriteLevel = level;
            }
            else
            {
                //-----------------------------------------------------------------------------
                // ミップマップ画像が指定されていない場合、
                // 1 つ上のレベルを縮小したイメージデータを設定します。
                if (level > lastOverwriteLevel)
                {
                    ReduceFromUpperLevel(pImage, level, firstFilter);
                    reducedLevels.push_back(level);
                }
            }
        } // level

        //-----------------------------------------------------------------------------
        // キュービック補間の場合、まずリニア補間のデータを生成して、
        // サイズの小さいレベルから順に上位レベルのデータをキュービック補間して設定します。
        #ifdef MIPMAP_CUBIC_AFTER_LINEAR
        if (isMipGenCubic)
        {
            const int reducedCount = static_cast<int>(reducedLevels.size());
            for (int iReduced = reducedCount - 1; iReduced >= 0; --iReduced)
            {
                ReduceFromUpperLevel(pImage, reducedLevels[iReduced], mipGenFilter);
            }
        }
        #endif
    }
    else
    {
        //-----------------------------------------------------------------------------
        // キューブマップ個別指定の場合
        for (int iFace = 0; iFace < FtxOpt::CUBE_FACE_COUNT; ++iFace)
        {
            // 作業用画像のピクセルデータは +X, -X, +Y, -Y, -Z, +Z の順に格納されているので
            // フェースインデックスを奥行きインデックスに変換する際に -Z と +Z を入れ替えます。
            int iDepth = iFace;
            if (iFace == FtxOpt::CUBE_FACE_PZ)
            {
                iDepth = FtxOpt::CUBE_FACE_NZ;
            }
            else if (iFace == FtxOpt::CUBE_FACE_NZ)
            {
                iDepth = FtxOpt::CUBE_FACE_PZ;
            }

            RIntArray reducedLevels;
            int lastOverwriteLevel = FtxOpt::MipCountMax;
            for (int level = 1; level < pImage->GetMipCount(); ++level)
            {
                const std::string& mipPath = opt.m_CubeMipPathss[iFace][level];
                if (!mipPath.empty())
                {
                    //-----------------------------------------------------------------------------
                    // ミップマップ画像をリードします。
                    RImage mipImg;
                    RStatus status = mipImg.ReadFile(mipPath, RImage::ReadFlag_Original, opt.m_IgnoresAlpha, convUtil);
                    RCheckStatus(status);
                    mipImg.ChangeDataType(isFloat, true);
                    //if (mipImg.IsFloat() != isFloat)
                    //{
                    //  return RStatus(RStatus::FAILURE, "Mipmap image format is different from base image: " + mipPath); // RShowError
                    //}

                    //-----------------------------------------------------------------------------
                    // 次元、幅、高さをチェックします。
                    const int curW = RMax(pImage->GetImageW() >> level, 1);
                    const int curH = RMax(pImage->GetImageH() >> level, 1);
                    if (mipImg.GetDimension() != FtxOpt::DIM_DEFAULT &&
                        mipImg.GetDimension() != FtxDimension_2d)
                    {
                        return RStatus(RStatus::FAILURE, "Cube mipmap image dimension is not 2D: " + mipPath); // RShowError
                    }
                    else if (mipImg.GetImageW() != static_cast<int>(curW) ||
                             mipImg.GetImageH() != static_cast<int>(curH))
                    {
                        return RStatus(RStatus::FAILURE, "Mipmap image width or height is wrong: " + mipPath); // RShowError
                    }

                    //-----------------------------------------------------------------------------
                    // 必要があれば sRGB からリニアに変換します。
                    if (dstLinearFlag != FtxOpt::LINEAR_NONE)
                    {
                        mipImg.ConvertToLinear(dstLinearFlag);
                    }

                    //-----------------------------------------------------------------------------
                    // イメージデータをコピーします。
                    const size_t faceBytes = colBytes * curW * curH;
                    memcpy(GetLevelImageData(nullptr, *pImage, level, iDepth), mipImg.GetImageData(), faceBytes);
                    lastOverwriteLevel = level;
                }
                else
                {
                    //-----------------------------------------------------------------------------
                    // ミップマップ画像が指定されていない場合、
                    // 1 つ上のレベルを縮小したイメージデータを設定します。
                    if (level > lastOverwriteLevel)
                    {
                        ReduceFaceFromUpperLevel(pImage, level, iDepth, firstFilter);
                        reducedLevels.push_back(level);
                    }
                }
            } // level

            //-----------------------------------------------------------------------------
            // キュービック補間の場合、まずリニア補間のデータを生成して、
            // サイズの小さいレベルから順に上位レベルのデータをキュービック補間して設定します。
            #ifdef MIPMAP_CUBIC_AFTER_LINEAR
            if (isMipGenCubic)
            {
                const int reducedCount = static_cast<int>(reducedLevels.size());
                for (int iReduced = reducedCount - 1; iReduced >= 0; --iReduced)
                {
                    ReduceFaceFromUpperLevel(pImage, reducedLevels[iReduced], iDepth, mipGenFilter);
                }
            }
            #endif
        } // iFace
    }

    return RStatus::SUCCESS;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief ミップマップテクスチャーを作成します。
//!
//! @param[in,out] pImage 画像です。
//! @param[in] dstFormat 出力フォーマットです。
//! @param[in] dstMipCount 作成するミップマップのレベル数です。
//! @param[in] dstMipGenFilter ミップマップ生成フィルタです。
//! @param[in] dstLinearFlag リニア変換フラグです。
//! @param[in] originalMipCount 元画像ファイルのミップマップのレベル数です。
//! @param[in] opt ftx 変換オプションです。
//! @param[in] convUtil 変換ユーティリティーです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus CreateMipmap(
    RImage* pImage,
    const FtxFormat dstFormat,
    const int dstMipCount,
    const FtxOpt::MipGenFilter dstMipGenFilter,
    const int dstLinearFlag,
    const int originalMipCount,
    const FtxOpt& opt,
    const ConverterUtility& convUtil
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // 必要があれば sRGB からリニアに変換します。
    const bool isFloat = pImage->IsFloat();
    if (dstLinearFlag != FtxOpt::LINEAR_NONE)
    {
        //-----------------------------------------------------------------------------
        // データ型が整数なら浮動小数点数フォーマットに変換します。
        if (!isFloat)
        {
            status = pImage->ChangeDataType(true, true);
            RCheckStatus(status);
        }

        //-----------------------------------------------------------------------------
        // sRGB からリニアに変換します。
        pImage->ConvertToLinear(dstLinearFlag);
    }

    //-----------------------------------------------------------------------------
    // 各レベルのミップマップデータを作成します。
    const int maxWHD = (pImage->GetDimension() == FtxDimension_3d) ?
        RMax(pImage->GetImageW(), RMax(pImage->GetImageH(), pImage->GetImageD())) :
        RMax(pImage->GetImageW(), pImage->GetImageH());
    const int minLevelWhd = maxWHD >> (dstMipCount - 1);
    GenerateMipLevels(pImage, minLevelWhd, dstMipGenFilter, originalMipCount);

    //-----------------------------------------------------------------------------
    // ミップマップ画像個別指定ならミップマップデータの内容を上書きします。
    if (opt.IsMipImageSpecified(dstMipCount))
    {
        status = OverwriteMipImage(pImage, dstMipGenFilter, dstLinearFlag, opt, convUtil);
        RCheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // 必要があればリニアから sRGB に変換し、整数フォーマットに変換します。
    if (dstLinearFlag != FtxOpt::LINEAR_NONE)
    {
        //-----------------------------------------------------------------------------
        // リニアから sRGB に変換します。
        if (RIsSrgbFetchTextureFormat(dstFormat))
        {
            // 出力フォーマットが sRGB フェッチなら RGB 成分をリニアから sRGB に変換します。
            pImage->ConvertToSrgb(dstLinearFlag & FtxOpt::LINEAR_RGB);
        }

        //-----------------------------------------------------------------------------
        // 元のデータ型が整数フォーマットなら整数フォーマットに変換します。
        if (!isFloat)
        {
            status = pImage->ChangeDataType(false, true);
            RCheckStatus(status);
        }
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 整数型カラーから法線を取得します。
//!
//! @param[out] n 法線です（長さ 3 の配列を指定します）。
//! @param[in] c 整数型カラーです（長さ 3 の配列を指定します）。
//-----------------------------------------------------------------------------
inline void GetNormalFromColor(float* n, const uint8_t* c)
{
    n[0] = (c[0] == 0) ? -1.0f : (c[0] - 128) / 127.0f;
    n[1] = (c[1] == 0) ? -1.0f : (c[1] - 128) / 127.0f;
    n[2] = (c[2] == 0) ? -1.0f : (c[2] - 128) / 127.0f;
}

//-----------------------------------------------------------------------------
//! @brief 法線から整数型カラーを取得します。
//!
//! @param[out] c 整数型カラーです（長さ 3 の配列を指定します）。
//! @param[in] n 法線です（長さ 3 の配列を指定します）。
//-----------------------------------------------------------------------------
inline void GetColorFromNormal(uint8_t* c, const float* n)
{
    for (int iXyz = 0; iXyz < 3; ++iXyz)
    {
        const float nn = n[iXyz];
        if (nn <= -1.0f)
        {
            c[iXyz] = 0x01;
        }
        else if (nn >= 1.0f)
        {
            c[iXyz] = 0xff;
        }
        else
        {
            c[iXyz] = static_cast<uint8_t>(RRound(nn * 127.0f) + 128);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 法線を正規化します。
//!
//! @param[in,out] n 法線です（長さ 3 の配列を指定します）。
//! @param[in] isPositiveZ Z 成分が 0 以上になるように調整するなら true を指定します。
//-----------------------------------------------------------------------------
inline void NormalizeNormal(float* n, const bool isPositiveZ)
{
    if (isPositiveZ && n[2] < 0.0f)
    {
        n[2] = 0.0f;
    }

    const float len = sqrtf(n[0] * n[0] + n[1] * n[1] + n[2] * n[2]);
    if (len != 0.0f)
    {
        const float invLen = 1.0f / len;
        n[0] *= invLen;
        n[1] *= invLen;
        n[2] *= invLen;
    }
}

//-----------------------------------------------------------------------------
//! @brief 法線マップ用画像の法線の Z 成分が 0 以上かつ
//!        長さが 1.0 になるように調整します。
//!
//! @param[in,out] pImage 画像のポインタです。
//-----------------------------------------------------------------------------
void NormalizeSurfaceNormal(RImage* pImage)
{
    if (!pImage->IsFloat())
    {
        uint8_t* pDstU8 = pImage->GetImageData();
        const int pixCount = static_cast<int>(pImage->GetImageDataSize() / R_RGBA_BYTES);
        for (int iPix = 0; iPix < pixCount; ++iPix)
        {
            float n[3];
            GetNormalFromColor(n, pDstU8);
            NormalizeNormal(n, true);
            GetColorFromNormal(pDstU8, n);
            pDstU8 += R_RGBA_COUNT;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief エンコーダー DLL で指定されたフォーマットに変換します。
//!
//! @param[in,out] pImage 画像のポインタです。
//! @param[in,out] encoder エンコーダーです。
//! @param[in] dstFormat フォーマットです。
//! @param[in] weightedCompress 圧縮時の RGB 成分の誤差の重み付けです。
//! @param[in] dstHint ヒント情報です。
//! @param[in] opt ftx 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertByEncoder(
    RImage* pImage,
    REncoder& encoder,
    const FtxFormat dstFormat,
    const bool weightedCompress,
    const std::string& dstHint,
    const FtxOpt& opt
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // エンコーダーを初期化します。
    if (!encoder.IsInitialized())
    {
        status = encoder.Initialize(nullptr);
        RCheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // 必要なら符号付き 8 ビットに変換します。
    if (RIsSignedTextureFormat(dstFormat))
    {
        pImage->ConvertToSigned8(true);
    }

    //-----------------------------------------------------------------------------
    // 元画像のパラメータを取得します。
    const FtxFormat srcFormat = static_cast<FtxFormat>(pImage->GetFormat());
    const uint8_t* pSrcImageData = pImage->GetImageData();
    std::vector<size_t> srcLevelOffsets;
    for (int level = 0; level < pImage->GetMipCount(); ++level)
    {
        srcLevelOffsets.push_back(pImage->GetLevelOffset(level));
    }

    //-----------------------------------------------------------------------------
    // 変換後のサイズを求めます。
    const std::wstring dstFormatW = RGetUnicodeFromShiftJis(RGetTextureFormatString(dstFormat));
    const texenc::ImageDimension encoderDim =
        REncoder::GetImageDimension(static_cast<FtxDimension>(pImage->GetDimension()));
    size_t dstImageDataSize = 0;
    pImage->ClearLevelOffsets();
    for (int level = 0; level < pImage->GetMipCount(); ++level)
    {
        pImage->SetLevelOffset(level, dstImageDataSize);
        const int curW = RMax(pImage->GetImageW() >> level, 1);
        const int curH = RMax(pImage->GetImageH() >> level, 1);
        const int curD = (pImage->GetDimension() == FtxDimension_3d) ?
            RMax(pImage->GetImageD() >> level, 1) : pImage->GetImageD();
        const size_t levelDataSize =
            encoder.GetDataSize(dstFormatW.c_str(), encoderDim, curW, curH, curD, 1);
        dstImageDataSize += levelDataSize;
    }

    //-----------------------------------------------------------------------------
    // 変換先のイメージデータを確保します。
    uint8_t* pDstImageData = new uint8_t[dstImageDataSize];

    //-----------------------------------------------------------------------------
    // 画像のパラメータを設定します。
    pImage->SetFormat(dstFormat);
    pImage->SetIsFloat(RIsRealNumberTextureFormat(dstFormat));
    pImage->SetIsEncoded(true);
    pImage->SetImageData(pDstImageData);
    pImage->SetImageDataSize(dstImageDataSize);
    pImage->SetMipDataSize((pImage->GetMipCount() >= 2) ? dstImageDataSize - pImage->GetLevelOffset(1) : 0);

    //-----------------------------------------------------------------------------
    // エンコード前のフォーマット文字列を決定します。
    const std::wstring srcFormatW = RIsFloatTextureFormat(srcFormat) ?
        L"float_32_32_32_32" :
        ((RIsSignedTextureFormat(dstFormat)) ? L"snorm_8_8_8_8" : L"unorm_8_8_8_8");

    //-----------------------------------------------------------------------------
    // エンコード品質を決定します。
    std::string quality = opt.m_Quality;
    if (quality.empty())
    {
        quality = "1";
    }
    if (RIsBc123TextureFormat(dstFormat) ||
        RIsEtcTextureFormat(dstFormat))
    {
        if (weightedCompress && quality.find("perceptual") == std::string::npos)
        {
            quality += "_perceptual";
        }
    }
    const std::wstring qualityW = RGetUnicodeFromShiftJis(quality);

    //-----------------------------------------------------------------------------
    // エンコードフラグを決定します。
    int encodeFlag = texenc::EncodeFlag_ReverseRgba;
    //int encodeFlag = texenc::EncodeFlag_Default;
    if (opt.m_GpuEncoding != FtxOpt::GpuEncoding_Disable)
    {
        encodeFlag |= texenc::EncodeFlag_GpuEncoding;
        if (opt.m_GpuEncoding == FtxOpt::GpuEncoding_Auto)
        {
            encodeFlag |= texenc::EncodeFlag_GpuAuto;
        }
    }
    if (RIsAstcTextureFormat(dstFormat) && dstHint == RGetHintForNormal())
    {
        encodeFlag |= texenc::EncodeFlag_NormalMapLa;
    }
    if (opt.m_Dither == FtxOpt::Dither_Enable)
    {
        encodeFlag |= texenc::EncodeFlag_Dither;
    }
    else if (opt.m_Dither == FtxOpt::Dither_Auto)
    {
        encodeFlag |= texenc::EncodeFlag_AutoDither;
    }
    if (opt.m_UsesEncoderProcess)
    {
        encodeFlag |= texenc::EncodeFlag_ServerProcess;
    }

    //-----------------------------------------------------------------------------
    // 変換します。
    if (!encoder.ConvertFormat(pDstImageData, pSrcImageData,
        dstFormatW.c_str(), srcFormatW.c_str(), qualityW.c_str(), encodeFlag,
        encoderDim, pImage->GetImageW(), pImage->GetImageH(), pImage->GetImageD(),
        pImage->GetMipCount()))
    {
        const bool isGpuEncodingFormat = (
            RIsBc123TextureFormat(dstFormat) ||
            RIsBc67TextureFormat(dstFormat)  ||
            RIsAstcTextureFormat(dstFormat));
        const std::string gpuMsg = (opt.m_GpuEncoding == FtxOpt::GpuEncoding_Enable && isGpuEncodingFormat) ?
            " (GPU encoding is not available)" : "";
        status = RStatus(RStatus::FAILURE, "Cannot convert to destination format" + gpuMsg + ": " + // RShowError
            opt.m_InputPaths[0] + " (" + RGetTextureFormatString(dstFormat) + ")");
    }

    //-----------------------------------------------------------------------------
    // 元画像のイメージデータを解放します。
    delete[] pSrcImageData;

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief LA (Luminance + Alpha) 形式に適したフォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return LA 形式に適したフォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsFormatForLuminanceAlpha(const FtxFormat format)
{
    return (format == FtxFormat_Unorm_4_4       ||
            format == FtxFormat_Unorm_8_8       ||
            format == FtxFormat_Unorm_Bc5       ||
            format == FtxFormat_Unorm_Eac_11_11);
}

//-----------------------------------------------------------------------------
//! @brief 指定されたフォーマットに変換します。
//!
//! @param[in,out] pImage 画像のポインタです。
//! @param[in,out] encoder エンコーダーです。
//! @param[in] dstFormat フォーマットです。
//! @param[in] weightedCompress 圧縮時の RGB 成分の誤差の重み付けです。
//! @param[in] dstHint ヒント情報です。
//! @param[in] dstLinearFlag リニア変換フラグです。
//! @param[in] opt ftx 変換オプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertFormat(
    RImage* pImage,
    REncoder& encoder,
    const FtxFormat dstFormat,
    const bool weightedCompress,
    const std::string& dstHint,
    const int dstLinearFlag,
    const FtxOpt& opt
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // ミップマップなしならここで sRGB からリニアに変換します。
    if (pImage->GetMipCount() == 1  &&
        dstLinearFlag != FtxOpt::LINEAR_NONE)
    {
        if (!RIsSrgbFetchTextureFormat(dstFormat))
        {
            pImage->ConvertToLinear(dstLinearFlag);
        }
        else // 出力フォーマットが sRGB フェッチなら A 成分のみリニアに変換します。
        {
            pImage->ConvertToLinear(dstLinearFlag & FtxOpt::LINEAR_A);
        }
    }

    //-----------------------------------------------------------------------------
    // LA (Luminance + Alpha) 形式に適したフォーマット
    // （unorm_4_4、unorm_8_8、unorm_bc5、unorm_eac_11_11）で法線マップでない場合、
    // 画像ファイルに A 成分があれば G 成分にコピーします。
    if (IsFormatForLuminanceAlpha(dstFormat))
    {
        if (dstHint != RGetHintForNormal() && pImage->HasAlpha())
        {
            pImage->CopyAToG();
        }
    }

    //-----------------------------------------------------------------------------
    // 法線用テクスチャーの場合、法線の Z 成分が 0 以上かつ長さが 1.0 になるように調整します。
    if (dstHint == RGetHintForNormal())
    {
        NormalizeSurfaceNormal(pImage);
    }

    //-----------------------------------------------------------------------------
    // 変換します。
    return ConvertByEncoder(pImage, encoder, dstFormat, weightedCompress, dstHint, opt);
}

//-----------------------------------------------------------------------------
//! @brief 成分選択を維持するなら true を返します。
//!
//! @param[in] srcFormat 入力画像のフォーマットです。
//! @param[in] dstFormat 出力テクスチャーのフォーマットです。
//!
//! @return 成分選択を維持するなら true を返します。
//-----------------------------------------------------------------------------
bool KeepsCompSel(const FtxFormat srcFormat, const FtxFormat dstFormat)
{
    return RGetComponentCount(srcFormat) == RGetComponentCount(dstFormat) &&
        RIsSignedTextureFormat(srcFormat) == RIsSignedTextureFormat(dstFormat);
}

//-----------------------------------------------------------------------------
//! @brief 画像から元画像配列を再設定します。
//!
//! @param[in,out] originalImages 元画像配列です。
//! @param[in] image 画像です。
//-----------------------------------------------------------------------------
void UpdateOriginalImagesFromImage(
    ROriginalImageArray& originalImages,
    const RImage& image
)
{
    if (!originalImages.empty())
    {
        if (static_cast<int>(originalImages.size()) != image.GetImageD())
        {
            originalImages.resize(image.GetImageD());
        }

        const size_t faceBytes = image.GetImageDataSize() / image.GetImageD();
        const uint8_t* pSrc = image.GetImageData();
        for (int iz = 0; iz < image.GetImageD(); ++iz)
        {
            int orgImgIdx = iz;
            if (image.IsCubeMap())
            {
                const int cubeIdx = iz / RImage::CUBE_FACE_COUNT;
                const int faceIdx = iz % RImage::CUBE_FACE_COUNT;
                const int baseZ = cubeIdx * RImage::CUBE_FACE_COUNT;
                orgImgIdx = baseZ + GetCubeMapDstFaceIdx(faceIdx);
            }
            ROriginalImage& originalImage = originalImages[orgImgIdx];
            if (originalImage.GetImageDataSize() != faceBytes)
            {
                delete[] originalImage.m_pImageData;
                originalImage.m_pImageData = new uint8_t[faceBytes];
            }
            memcpy(originalImage.m_pImageData, pSrc, faceBytes);
            originalImage.m_Width  = image.GetImageW();
            originalImage.m_Height = image.GetImageH();
            if (image.HasAlpha())
            {
                originalImage.m_Format = (originalImage.IsFloat()) ?
                    ROriginalImage::RGBA32F : ROriginalImage::RGBA8;
            }
            pSrc += faceBytes;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータ群をアンパックします。
//!
//! @param[out] userDatas ユーザーデータ配列です。
//! @param[in] pPackedUserData パックされたデータへのポインタです。
//! @param[in] packedUserDataSize パックされたデータのサイズです。
//-----------------------------------------------------------------------------
RStatus UnpackUserDataArray(
    RUserDataArray& userDatas,
    const uint8_t* pPackedUserData,
    const size_t packedUserDataSize
)
{
    RStatus status;

    const uint8_t* pSrc = pPackedUserData;
    const uint8_t* pEnd = pSrc + packedUserDataSize;

    uint32_t userDataCount;
    RGetMemLittle(userDataCount, pSrc);
    pSrc += sizeof(userDataCount);
    //NoteTrace("ud count: %d", userDataCount);

    while (pSrc < pEnd)
    {
        //-----------------------------------------------------------------------------
        // uint32_t nameSize;   // ends は含みません
        // char name[nameSize]; // ends は含みません
        uint32_t nameSize;
        RGetMemLittle(nameSize, pSrc);
        pSrc += sizeof(nameSize);
        std::string name(reinterpret_cast<const char*>(pSrc), nameSize);
        pSrc += nameSize;

        //-----------------------------------------------------------------------------
        // uint32_t type;
        uint32_t type;
        RGetMemLittle(type, pSrc);
        pSrc += sizeof(type);

        //-----------------------------------------------------------------------------
        // uint32_t contentSize;
        uint32_t contentSize;
        RGetMemLittle(contentSize, pSrc);
        pSrc += sizeof(contentSize);

        //-----------------------------------------------------------------------------
        // uint8_t content[contentSize];
        //NoteTrace("ud: %s: %d", name.c_str(), type);
        RUserData userData(name, static_cast<RUserData::Type>(type));

        const uint8_t* pContent = pSrc;
        const uint8_t* pContentEnd = pContent + contentSize;
        if (type == RUserData::FLOAT || type == RUserData::INT)
        {
            RGetMemLittle(userData.m_NumberValueCount, pContent);
            if (userData.m_NumberValueCount != 0)
            {
                userData.m_NumberValuesText = std::string(
                    reinterpret_cast<const char*>(pContent + sizeof(userData.m_NumberValueCount)),
                    contentSize - sizeof(userData.m_NumberValueCount));
                //NoteTrace("ud num: %d: [%s]", userData.m_NumberValueCount, userData.m_NumberValuesText.c_str());
            }
        }
        else if (type == RUserData::STRING || type == RUserData::WSTRING)
        {
            uint32_t stringCount;
            RGetMemLittle(stringCount, pContent);
            pContent += sizeof(stringCount);
            for (uint32_t iString = 0; iString < stringCount && pContent < pContentEnd; ++iString)
            {
                uint32_t stringSize;
                RGetMemLittle(stringSize, pContent);
                pContent += sizeof(stringSize);

                userData.m_StringValues.push_back(
                    std::string(reinterpret_cast<const char*>(pContent), stringSize));
                pContent += stringSize;
                //NoteTrace("ud string: %s", userData.m_StringValues[userData.m_StringValues.size() - 1].c_str());
            }
        }
        else
        {
            userData.m_StreamValues.resize(contentSize);
            for (uint32_t iByte = 0; iByte < contentSize; ++iByte)
            {
                userData.m_StreamValues[iByte] = *pContent++;
            }
        }
        userDatas.push_back(userData);
        pSrc += contentSize;
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 有効な bntx ファイルのタイルモードなら true を返します。
//!
//! @param[in] binTileMode bntx ファイルのタイルモードです。
//!
//! @return 有効な bntx ファイルのタイルモードなら true を返します。
//-----------------------------------------------------------------------------
bool IsValidBinTileMode(const std::string& binTileMode)
{
    static const char* const ValidTileModes[] =
    {
        "linear",
        "nx",
        nullptr
    };

    for (int tileModxIdx = 0; ValidTileModes[tileModxIdx] != nullptr; ++tileModxIdx)
    {
        if (binTileMode == ValidTileModes[tileModxIdx])
        {
            return true;
        }
    }
    return false;
}

//=============================================================================
// オプション引数情報配列です。
//=============================================================================
const OptionArgInfo GlobalOptionArgInfos[] =
{
    { ' ', "version"           , false },
    { 'h', "help"              , false },
    { 's', "silent"            , false },
    { ' ', "args-file"         , true  },
    { ' ', "job-list"          , true  },
    { ' ', "check-gpu-encoding", false },
    { ' ', "convert-list"      , true  }, // 非公開オプションです（廃止予定）。
    { ' ', "csource"           , false }, // 非公開オプションです。
    { ' ', "tool-name"         , true  }, // 非公開オプションです。
    { ' ', "final-folder"      , true  }, // 非公開オプションです。
    { ' ', nullptr, false }, // 配列の最後を表します。
};

const OptionArgInfo JobOptionArgInfos[] =
{
    { ' ', "project-root"          , true  },
    { ' ', "disable-file-info"     , false },
    { ' ', "disable-original-image", false },
    { ' ', "disable-memory-pool"   , false },
    { ' ', "tile-mode"             , true  },
    { ' ', "tile-optimize"         , true  },
    { ' ', "tile-size-threshold"   , true  },
    { ' ', "sparse-tiled"          , true  },
    { 'o', "output"                , true  },
    { ' ', "output-original"       , true  }, // 非公開オプションです。
    { 'm', "merge"                 , true  },
    { ' ', "disable-merge-error"   , false },
    { ' ', "no-encoding"           , false },
    { ' ', "gpu-encoding"          , true  },
    { ' ', "dcc-preset"            , true  },
    { 'n', "hint"                  , true  },
    { 'l', "linear"                , true  },
    { 'd', "dimension"             , true  },
    { 'f', "format"                , true  },
    { 't', "depth"                 , true  },
    { 'i', "mip-level"             , true  },
    { ' ', "mip-gen-filter"        , true  },
    { 'c', "comp-sel"              , true  },
    { 'q', "quality"               , true  }, // 非公開オプションです。
    { 'w', "weighted-compress"     , true  },
    { ' ', "dither"                , true  }, // 非公開オプションです。
    { ' ', "encoder-process"       , true  }, // 非公開オプションです。
    { ' ', "swizzle"               , true  },
    { ' ', "cubemap-back"          , true  },
    { ' ', "cubemap-left"          , true  },
    { ' ', "cubemap-right"         , true  },
    { ' ', "cubemap-top"           , true  },
    { ' ', "cubemap-bottom"        , true  },
    { ' ', "mip-image#"            , true  },
    { ' ', "cubemap-front#"        , true  },
    { ' ', "cubemap-back#"         , true  },
    { ' ', "cubemap-left#"         , true  },
    { ' ', "cubemap-right#"        , true  },
    { ' ', "cubemap-top#"          , true  },
    { ' ', "cubemap-bottom#"       , true  },
    { ' ', "comment"               , true  },
    { ' ', "crop"                  , true  },
    { 'e', "reduce-level"          , true  },
    { 'x', "resize-w"              , true  },
    { 'y', "resize-h"              , true  },
    { ' ', "resize-filter"         , true  },
    { ' ', "resize-gamma"          , true  },
    { ' ', "ignore-alpha"          , true  },
    { 'u', "adjust-transparent-rgb", true  },
    { 'r', "replace-r"             , true  },
    { 'g', "replace-g"             , true  },
    { 'b', "replace-b"             , true  },
    { 'a', "replace-a"             , true  },
    { ' ', "no-tga-rle"            , false },
    { ' ', "dds-dx10"              , false }, // 非公開オプションです。
    { ' ', "no-tiling"             , false }, // 非公開オプションです（現在は効果なし）。
    { ' ', "disable-size-error"    , false }, // 非公開オプションです。
    { ' ', nullptr, false }, // 配列の最後を表します。
};

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 引数がグローバルオプションなら true を返します。
//-----------------------------------------------------------------------------
bool TexConverterBase::IsGlobalOptionArg(const wchar_t* arg) const
{
    const std::string argA = RGetShiftJisFromUnicode(arg);
    bool isLongName;
    int suffixNum;
    return (GetOptionArgInfo(&isLongName, &suffixNum, GlobalOptionArgInfos, argA) != nullptr);
}

//-----------------------------------------------------------------------------
//! @brief 指定した引数の次の引数がグローバルオプションの値なら true を返します。
//-----------------------------------------------------------------------------
bool TexConverterBase::IsNextArgGlobalOptionValue(const wchar_t* arg) const
{
    const std::string argA = RGetShiftJisFromUnicode(arg);
    return IsNextArgOptionValue(GlobalOptionArgInfos, argA);
}

//-----------------------------------------------------------------------------
//! @brief 指定した引数の次の引数がジョブオプションの値なら true を返します。
//-----------------------------------------------------------------------------
bool TexConverterBase::IsNextArgJobOptionValue(const wchar_t* arg) const
{
    const std::string argA = RGetShiftJisFromUnicode(arg);
    return IsNextArgOptionValue(JobOptionArgInfos, argA);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターにグローバルオプションを設定します。
//-----------------------------------------------------------------------------
bool TexConverterBase::SetOptions(const wchar_t* options[])
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // グローバルオプションを初期化します。
    InitGlobalOptions();

    //-----------------------------------------------------------------------------
    // グローバルオプションのユニコード文字列配列を解析します。
    RStringArray tokens;
    const int tokenCount = RGetStringArray(tokens, options);
    for (int iToken = 0; iToken < tokenCount; ++iToken)
    {
        const std::string& curArg = tokens[iToken];
        const OptionArgInfo* pInfo;
        int suffixNum;
        std::string value;
        status = ParseOptionArg(&pInfo, &suffixNum, &value, &iToken, tokens,
            GlobalOptionArgInfos);
        if (!status)
        {
            break;
        }

        const std::string longName = pInfo->longName;
        if (longName == "silent")
        {
            m_IsSilent = true;
        }
        else if (longName == "csource") // 非公開オプションです。
        {
            m_IsCSource = true;
        }
        else if (longName == "tool-name") // 非公開オプションです。
        {
            m_ToolName = value;
        }
        else if (longName == "final-folder") // 非公開オプションです。
        {
            m_FinalFolderPath = RGetFullFilePath(value, true);
        }
        else if (longName == "version"            ||
                 longName == "help"               ||
                 longName == "check-gpu-encoding")
        {
            // exe 側で処理するオプションです。
        }
        else
        {
            status = RStatus(RStatus::FAILURE, "Invalid global option: " + curArg); // RShowError
            break;
        }
    }

    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief ftx 変換オプションがキューブマップなら true を返します。
//-----------------------------------------------------------------------------
bool FtxOpt::IsCubeMap() const
{
    return (RIsCubMapDimension(static_cast<FtxDimension>(m_Dimension)) || IsCubeMapSeparete());
}

//-----------------------------------------------------------------------------
//! @brief ftx 変換オプションがミップマップ画像個別指定なら true を返します。
//-----------------------------------------------------------------------------
bool FtxOpt::IsMipImageSpecified(const int mipCount) const
{
    for (int level = 1; level < mipCount; ++level)
    {
        if (!IsCubeMapSeparete())
        {
            if (!m_MipPaths[level].empty())
            {
                return true;
            }
        }
        else
        {
            for (int iFace = 0; iFace < CUBE_FACE_COUNT; ++iFace)
            {
                if (!m_CubeMipPathss[iFace][level].empty())
                {
                    return true;
                }
            }
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief ftx 変換オプションが再エンコードが必要なら true を返します。
//-----------------------------------------------------------------------------
bool FtxOpt::NeedsEncoding() const
{
    return (!m_MergePath.empty()                            ||
            m_LinearFlag != LINEAR_DEFAULT                  ||
            m_Dimension != DIM_DEFAULT                      ||
            m_Format != FORMAT_DEFAULT                      ||
            m_Depth != DEPTH_DEFAULT                        ||
            m_MipCount != MipCountDefault                   ||
            m_MipGenFilter != MIP_GEN_DEFAULT               ||
            !m_Quality.empty()                              ||
            m_WeightedCompress != WEIGHTED_COMPRESS_DEFAULT ||
            m_Dither != Dither_Auto                         ||
            IsCubeMapSeparete()                             ||
            IsMipImageSpecified(MipCountMax)                ||
            m_CropRect.IsEnabled()                          ||
            IsResizeSpecified()                             ||
            m_IgnoresAlpha                                  ||
            m_AdjustsXpaRgb                                 ||
            IsReplaceSpecified());
}

//-----------------------------------------------------------------------------
//! @brief ftx 変換オプションが入力ファイルのデータをそのまま出力可能なら true を返します。
//-----------------------------------------------------------------------------
bool FtxOpt::IsPassThrough() const
{
    return (!NeedsEncoding()                         &&
            m_DccPreset      == DCC_PRESET_DEFAULT   &&
            m_Hint           == HINT_DEFAULT         &&
            m_CompSel        == COMP_SEL_DEFAULT     &&
            m_InitialSwizzle == SWIZZLE_DEFAULT      &&
            m_CommentText    == COMMENT_TEXT_DEFAULT);
}

//-----------------------------------------------------------------------------
//! @brief ftx 変換オプションにジョブオプションを設定します。
//-----------------------------------------------------------------------------
RStatus FtxOpt::SetJobOptions(const wchar_t* options[])
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // ジョブオプションのユニコード文字列配列を解析します。
    RStringArray tokens;
    const int tokenCount = RGetStringArray(tokens, options);
    for (int iToken = 0; iToken < tokenCount; ++iToken)
    {
        const std::string& curArg = tokens[iToken];
        const OptionArgInfo* pInfo;
        int suffixNum;
        std::string value;
        status = ParseOptionArg(&pInfo, &suffixNum, &value, &iToken, tokens,
            JobOptionArgInfos);
        RCheckStatus(status);

        const std::string longName = pInfo->longName;
        const bool isMipLevelSuffix = (1 <= suffixNum && suffixNum < MipCountMax);
        const bool isBoolValue = (value == "true" || value == "false");
        const bool boolValue = (value == "true");
        if (longName == "project-root" && m_SupportsFtx)
        {
            m_ProjectRootPath = RGetFilePathWithEndingSlash(RGetFullFilePath(value, true));
        }
        else if (longName == "disable-file-info" && m_SupportsFtx)
        {
            m_DisablesFileInfo = true;
        }
        else if (longName == "disable-original-image" && m_SupportsFtx)
        {
            m_DisablesOriginalImage = true;
        }
        else if (longName == "disable-memory-pool")
        {
            m_DisablesMemoryPool = true;
        }
        else if (longName == "tile-mode")
        {
            m_BinTileMode = RGetLowerCaseString(value);
            if (!m_BinTileMode.empty() && !IsValidBinTileMode(m_BinTileMode))
            {
                status = RStatus(RStatus::FAILURE, "Tile mode is wrong: " + value); // RShowError
            }
        }
        else if (longName == "tile-optimize")
        {
            m_TileOptimize = (value == "performance") ? FtxTileOptimize_Performance :
                             (value == "size"       ) ? FtxTileOptimize_Size        :
                             (value == "size_auto"  ) ? FtxTileOptimize_SizeAuto    :
                             FtxTileOptimize_Invalid;
            if (m_TileOptimize == FtxTileOptimize_Invalid)
            {
                status = RStatus(RStatus::FAILURE, "Tile optimize is wrong: " + value); // RShowError
            }
        }
        else if (longName == "tile-size-threshold")
        {
            m_TileSizeThreshold = atoi(value.c_str());
            if (m_TileSizeThreshold < 0 || 100 < m_TileSizeThreshold)
            {
                status = RStatus(RStatus::FAILURE, "Tile size threshold is wrong: " + value); // RShowError
            }
        }
        else if (longName == "sparse-tiled")
        {
            if (isBoolValue)
            {
                m_IsSparseTiled = boolValue;
            }
            else
            {
                status = RStatus(RStatus::FAILURE, "Sparse tiled is wrong: " + value); // RShowError
            }
        }
        else if (longName == "output")
        {
            m_OutputPath = RGetFullFilePath(value, true);
        }
        else if (longName == "output-original")
        {
            m_OutputOriginalPath = RGetFullFilePath(value, true);
        }
        else if (longName == "merge" && m_SupportsFtx)
        {
            m_MergePath = RGetFullFilePath(value, true);
        }
        else if (longName == "disable-merge-error" && m_SupportsFtx)
        {
            m_DisablesMergeError = true;
        }
        else if (longName == "no-encoding")
        {
            m_NoEncoding = true;
        }
        else if (longName == "gpu-encoding")
        {
            m_GpuEncoding = (value == "false") ? GpuEncoding_Disable :
                            (value == "true" ) ? GpuEncoding_Enable  :
                            (value == "auto" ) ? GpuEncoding_Auto    :
                            GpuEncoding_Invalid;
            if (m_GpuEncoding == GpuEncoding_Invalid)
            {
                status = RStatus(RStatus::FAILURE, "GPU encoding is wrong: " + value); // RShowError
            }
        }
        else if (longName == "dcc-preset" && m_SupportsFtx)
        {
            m_DccPreset = value;
        }
        else if (longName == "hint")
        {
            m_Hint = value;
        }
        else if (longName == "linear")
        {
            m_LinearFlag = GetLinearFlagFromOptString(value);
            if (m_LinearFlag == LINEAR_INVALID)
            {
                status = RStatus(RStatus::FAILURE, "Linear is wrong: " + value); // RShowError
            }
        }
        else if (longName == "dimension")
        {
            m_Dimension = GetDimensionFromOptString(value);
            if (m_Dimension == DIM_INVALID)
            {
                status = RStatus(RStatus::FAILURE, "Dimension is wrong: " + value); // RShowError
            }
        }
        else if (longName == "format")
        {
            m_Format = GetFormatFromOptString(value);
            if (m_Format == FORMAT_INVALID)
            {
                status = RStatus(RStatus::FAILURE, "Format is wrong: " + value); // RShowError
            }
        }
        else if (longName == "depth")
        {
            m_Depth = atoi(value.c_str());
            if (m_Depth <= 0)
            {
                status = RStatus(RStatus::FAILURE, "Depth is wrong: " + value); // RShowError
            }
        }
        else if (longName == "mip-level")
        {
            m_MipCount = atoi(value.c_str());
            if (m_MipCount < 1)
            {
                m_MipCount = 1;
            }
        }
        else if (longName == "mip-gen-filter")
        {
            m_MipGenFilter = GetMipGenFilterFromString(value, false);
            if (m_MipGenFilter == MIP_GEN_INVALID)
            {
                status = RStatus(RStatus::FAILURE, "Mipmap generation filter is wrong: " + value); // RShowError
            }
        }
        else if (longName == "comp-sel")
        {
            m_CompSel = GetCompSelFromOptString(value);
            if (m_CompSel == COMP_SEL_INVALID)
            {
                status = RStatus(RStatus::FAILURE, "Component selection is wrong: " + value); // RShowError
            }
        }
        else if (longName == "quality") // 現在、非公開オプション
        {
            m_Quality = value;
        }
        else if (longName == "weighted-compress")
        {
            if (value == "false")
            {
                m_WeightedCompress = WEIGHTED_COMPRESS_DISABLE;
            }
            else if (value == "true")
            {
                m_WeightedCompress = WEIGHTED_COMPRESS_ENABLE;
            }
            else
            {
                status = RStatus(RStatus::FAILURE, "Weighted compress is wrong: " + value); // RShowError
            }
        }
        else if (longName == "dither") // 非公開オプションです。
        {
            m_Dither = (value == "false") ? Dither_Disable :
                       (value == "true" ) ? Dither_Enable  :
                       (value == "auto" ) ? Dither_Auto    :
                       Dither_Invalid;
            if (m_Dither == Dither_Invalid)
            {
                status = RStatus(RStatus::FAILURE, "Dither is wrong: " + value); // RShowError
            }
        }
        else if (longName == "encoder-process") // 非公開オプションです。
        {
            if (isBoolValue)
            {
                m_UsesEncoderProcess = boolValue;
            }
            else
            {
                status = RStatus(RStatus::FAILURE, "Encoder process is wrong: " + value); // RShowError
            }
        }
        else if (longName == "swizzle")
        {
            m_InitialSwizzle = atoi(value.c_str());
            if (m_InitialSwizzle < SWIZZLE_MIN ||
                m_InitialSwizzle > SWIZZLE_MAX)
            {
                status = RStatus(RStatus::FAILURE, "Swizzle is wrong: " + value); // RShowError
            }
        }
        else if (longName == "cubemap-back")
        {
            m_CubeBackPath = RGetFullFilePath(value, true);
        }
        else if (longName == "cubemap-left")
        {
            m_CubeLeftPath = RGetFullFilePath(value, true);
        }
        else if (longName == "cubemap-right")
        {
            m_CubeRightPath = RGetFullFilePath(value, true);
        }
        else if (longName == "cubemap-top")
        {
            m_CubeTopPath = RGetFullFilePath(value, true);
        }
        else if (longName == "cubemap-bottom")
        {
            m_CubeBottomPath = RGetFullFilePath(value, true);
        }
        else if (longName == "mip-image#" && isMipLevelSuffix)
        {
            m_MipPaths[suffixNum] = RGetFullFilePath(value, true);
        }
        else if (longName == "cubemap-right#" && isMipLevelSuffix)
        {
            m_CubeMipPathss[0][suffixNum] = RGetFullFilePath(value, true);
        }
        else if (longName == "cubemap-left#" && isMipLevelSuffix)
        {
            m_CubeMipPathss[1][suffixNum] = RGetFullFilePath(value, true);
        }
        else if (longName == "cubemap-top#" && isMipLevelSuffix)
        {
            m_CubeMipPathss[2][suffixNum] = RGetFullFilePath(value, true);
        }
        else if (longName == "cubemap-bottom#" && isMipLevelSuffix)
        {
            m_CubeMipPathss[3][suffixNum] = RGetFullFilePath(value, true);
        }
        else if (longName == "cubemap-front#" && isMipLevelSuffix)
        {
            m_CubeMipPathss[4][suffixNum] = RGetFullFilePath(value, true);
        }
        else if (longName == "cubemap-back#" && isMipLevelSuffix)
        {
            m_CubeMipPathss[5][suffixNum] = RGetFullFilePath(value, true);
        }
        else if (longName == "comment" && m_SupportsFtx)
        {
            m_CommentText = value;
        }
        else if (longName == "crop")
        {
            status = GetCropRectArg(m_CropRect, value);
        }
        else if (longName == "reduce-level")
        {
            m_ReductionLevel = atoi(value.c_str());
            if (m_ReductionLevel < 0)
            {
                status = RStatus(RStatus::FAILURE, "Reduce level is wrong: " + value); // RShowError
            }
        }
        else if (longName == "resize-w")
        {
            if (!value.empty() && value[value.size() - 1] == '%')
            {
                m_ResizeW = 0;
                m_ResizePercentW = strtof(value.c_str(), nullptr);
            }
            else
            {
                m_ResizeW = atoi(value.c_str());
                m_ResizePercentW = 0.0f;
            }
            if (m_ResizeW <= 0 && m_ResizePercentW <= 0.0f)
            {
                status = RStatus(RStatus::FAILURE, "Resize width is wrong: " + value); // RShowError
            }
        }
        else if (longName == "resize-h")
        {
            if (!value.empty() && value[value.size() - 1] == '%')
            {
                m_ResizeH = 0;
                m_ResizePercentH = strtof(value.c_str(), nullptr);
            }
            else
            {
                m_ResizeH = atoi(value.c_str());
                m_ResizePercentH = 0.0f;
            }
            if (m_ResizeH <= 0 && m_ResizePercentH <= 0.0f)
            {
                status = RStatus(RStatus::FAILURE, "Resize height is wrong: " + value); // RShowError
            }
        }
        else if (longName == "resize-filter")
        {
            m_ResizeFilter = RGetValueFromString(g_ResizeFilterStringValues, value, ResizeFilter_Invalid);
            if (m_ResizeFilter == ResizeFilter_Invalid)
            {
                status = RStatus(RStatus::FAILURE, "Resize filter is wrong: " + value); // RShowError
            }
        }
        else if (longName == "resize-gamma")
        {
            if (value == "false")
            {
                m_ResizesInLinear = false;
            }
            else if (value == "true")
            {
                m_ResizesInLinear = true;
            }
            else
            {
                status = RStatus(RStatus::FAILURE, "Resize gamma is wrong: " + value); // RShowError
            }
        }
        else if (longName == "ignore-alpha")
        {
            if (isBoolValue)
            {
                m_IgnoresAlpha = boolValue;
            }
            else
            {
                status = RStatus(RStatus::FAILURE, "Ignore alpha is wrong: " + value); // RShowError
            }
        }
        else if (longName == "adjust-transparent-rgb")
        {
            if (value == "false")
            {
                m_AdjustsXpaRgb = false;
            }
            else if (value == "true")
            {
                m_AdjustsXpaRgb = true;
            }
            else
            {
                status = RStatus(RStatus::FAILURE, "Adjust transparent rgb is wrong: " + value); // RShowError
            }
        }
        else if (longName == "replace-r")
        {
            status = GetReplacePathArg(m_ReplacePaths[0], m_ReplaceChanIdxs[0], value);
        }
        else if (longName == "replace-g")
        {
            status = GetReplacePathArg(m_ReplacePaths[1], m_ReplaceChanIdxs[1], value);
        }
        else if (longName == "replace-b")
        {
            status = GetReplacePathArg(m_ReplacePaths[2], m_ReplaceChanIdxs[2], value);
        }
        else if (longName == "replace-a")
        {
            status = GetReplacePathArg(m_ReplacePaths[3], m_ReplaceChanIdxs[3], value);
        }
        else if (longName == "no-tga-rle")
        {
            m_IsTgaRle = false;
        }
        else if (longName == "dds-dx10") // 非公開オプションです。
        {
            m_ForcesDdsDx10 = true;
        }
        else if (longName == "no-tiling") // 非公開オプションです。
        {
            // 現在は効果なし。
        }
        else if (longName == "disable-size-error") // 非公開オプションです。
        {
            m_DisablesSizeError = true;
        }
        else
        {
            status = RStatus(RStatus::FAILURE, "Option is wrong: " + curArg); // RShowError
        }
        RCheckStatus(status);
    }

    // bntx ファイル出力フラグを設定します。
    const std::string outExt = RGetExtensionFromFilePath(m_OutputPath);
    m_OutputsBntx = (outExt == BntxExtension || (!m_SupportsFtx && outExt.empty()));

    return RStatus::SUCCESS;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief ftx 変換オプションに入力ファイルパス配列を設定します。
//-----------------------------------------------------------------------------
RStatus FtxOpt::SetInputPaths(const wchar_t* paths[])
{
    //-----------------------------------------------------------------------------
    // 入力ファイル数と拡張子をチェックします。
    RStringArray inPaths;
    const int pathCount = RGetStringArray(inPaths, paths);
    if (pathCount == 0)
    {
        return RStatus(RStatus::FAILURE, "No input file"); // RShowError
    }
    else if (m_OutputsBntx)
    {
        int inFtxCount = 0;
        int inDdsCount = 0;
        int inOtherImageCount = 0;
        for (int pathIdx = 0; pathIdx < pathCount; ++pathIdx)
        {
            const std::string& inPath = inPaths[pathIdx];
            const std::string ext = RGetExtensionFromFilePath(inPath);
            if (ext == FtxaExtension || ext == FtxbExtension)
            {
                ++inFtxCount;
            }
            else if (ext == DdsExtension)
            {
                ++inDdsCount;
            }
            else if (ext == BntxExtension)
            {
                return RStatus(RStatus::FAILURE, "Input file type is wrong: " + inPath); // RShowError
            }
            else
            {
                ++inOtherImageCount;
            }
        }

        const int ftxDdsCount = inFtxCount + inDdsCount;
        if (inOtherImageCount > 0 && ftxDdsCount > 0)
        {
            if (m_SupportsFtx)
            {
                return RStatus(RStatus::FAILURE, "Image file and ftx (DDS) file are specified"); // RShowError
            }
            else
            {
                return RStatus(RStatus::FAILURE, "DDS file and other image file are specified"); // RShowError
            }
        }
        else if (inOtherImageCount >= 2)
        {
            return RStatus(RStatus::FAILURE, "Multi input image files are specified"); // RShowError
        }

        if (inDdsCount == 1 && ftxDdsCount == 1)
        {
            // 入力ファイルが DDS ファイル 1 つで特定のオプションが指定されている場合、
            // 他の画像ファイルからのバイナリー変換と同様に処理します。
            m_IsImageToBntx = (!IsPassThrough());
        }
        else
        {
            m_IsImageToBntx = (inOtherImageCount > 0);
        }
    }
    else if (pathCount >= 2)
    {
        return RStatus(RStatus::FAILURE, "Multi input files are specified: " + inPaths[1]); // RShowError
    }

    //-----------------------------------------------------------------------------
    // 入力ファイルが DDS ファイル 1 つで再エンコード不要なら
    // --no-encoding が指定されていなくても再エンコードしないようにします。
    if (!m_OutputsBntx || m_IsImageToBntx)
    {
        if (pathCount == 1 &&
            RGetExtensionFromFilePath(inPaths[0]) == DdsExtension &&
            !NeedsEncoding())
        {
            const std::string outExt = RGetExtensionFromFilePath(m_OutputPath);
            if (outExt != TgaExtension &&
                outExt != PngExtension &&
                outExt != ExrExtension)
            {
                m_NoEncoding = true;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // ftx ファイルから ftx ファイルへの変換で、
    // --disable-original-image が指定されていて再エンコード不要の場合、
    // --no-encoding が指定されていなくても再エンコードしないようにします。
    if (pathCount == 1              &&
        IsFtxFilePath(inPaths[0])   &&
        IsFtxFilePath(m_OutputPath) &&
        m_DisablesOriginalImage     &&
        !NeedsEncoding())
    {
        m_NoEncoding = true;
    }

    //-----------------------------------------------------------------------------
    // 引数のパス配列をフルパスに変換します。
    for (int iPath = 0; iPath < pathCount; ++iPath)
    {
        std::string inPath = inPaths[iPath];
        inPath = (inPath == "untitled") ? inPath : RGetFullFilePath(inPath, true);
        m_InputPaths.push_back(inPath);
    }

    //-----------------------------------------------------------------------------
    // 奥行き指定があれば最初のファイル名の末尾の数値をインクリメントしたパスを
    // 入力ファイルパス配列に追加します。
    if (!m_OutputsBntx || m_IsImageToBntx)
    {
        if (m_Depth != DEPTH_DEFAULT && m_Depth >= 2)
        {
            if (!AddNumberIncrementedPath(m_InputPaths, m_InputPaths[0], m_Depth - 1))
            {
                return RStatus(RStatus::FAILURE, "Input file name does not have digits for depth"); // RShowError
            }
        }
    }

    return RStatus::SUCCESS;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief ftx 変換オプションのキューブマップ個別指定、ミップマップ個別指定を含めた
//!        全入力ファイルのパス配列を取得します。
//-----------------------------------------------------------------------------
void FtxOpt::GetAllInputPaths(RStringArray& allInputPaths) const
{
    allInputPaths = m_InputPaths;
    if (!IsCubeMapSeparete())
    {
        for (int level = 1; level < MipCountMax; ++level)
        {
            const std::string& mipPath = m_MipPaths[level];
            if (!mipPath.empty())
            {
                allInputPaths.push_back(mipPath);
            }
        }
    }
    else
    {
        allInputPaths.push_back(m_CubeBackPath);
        allInputPaths.push_back(m_CubeLeftPath);
        allInputPaths.push_back(m_CubeRightPath);
        allInputPaths.push_back(m_CubeTopPath);
        allInputPaths.push_back(m_CubeBottomPath);

        for (int iFace = 0; iFace < CUBE_FACE_COUNT; ++iFace)
        {
            for (int level = 1; level < MipCountMax; ++level)
            {
                const std::string& mipPath = m_CubeMipPathss[iFace][level];
                if (!mipPath.empty())
                {
                    allInputPaths.push_back(mipPath);
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列から ftx 変換オプションのミップマップ生成フィルタを取得します。
//-----------------------------------------------------------------------------
FtxOpt::MipGenFilter FtxOpt::GetMipGenFilterFromString(const std::string& str, const bool linearIfInvalid)
{
    return
        (str == "point" ) ? MIP_GEN_POINT  :
        (str == "linear") ? MIP_GEN_LINEAR :
        (str == "cubic" ) ? MIP_GEN_CUBIC  :
        (linearIfInvalid) ? MIP_GEN_LINEAR :
        MIP_GEN_INVALID;
}

//-----------------------------------------------------------------------------
//! @brief ftx 変換オプションのミップマップ生成フィルタから中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string FtxOpt::GetMipGenFilterString(const MipGenFilter mipGenFilter)
{
    switch (mipGenFilter)
    {
        case MIP_GEN_POINT:  return "point";
        case MIP_GEN_LINEAR: return "linear";
        case MIP_GEN_CUBIC:  return "cubic";
        default:             return "invalid";
    }
}

//-----------------------------------------------------------------------------
//! @brief ftx 変換オプションの入力ファイルと出力ファイルの情報を表示します。
//-----------------------------------------------------------------------------
void FtxOpt::DisplayFileInfo(
    std::ostream& os,
    const std::string& dstPath,
    const std::string& dstFinalPath
) const
{
    if (IsCubeMapSeparete())
    {
        os << "Front   : " << m_InputPaths[0] << endl;
        os << "Back    : " << m_CubeBackPath << endl;
        os << "Left    : " << m_CubeLeftPath << endl;
        os << "Right   : " << m_CubeRightPath << endl;
        os << "Top     : " << m_CubeTopPath << endl;
        os << "Bottom  : " << m_CubeBottomPath << endl;
    }
    else
    {
        os << "Input   : " << m_InputPaths[0] << endl;
        //for (size_t iInput = 1; iInput < m_InputPaths.size(); ++iInput)
        //{
        //  os << "        : " << m_InputPaths[iInput] << endl;
        //}
    }

    os << "Output  : " << dstPath << endl;

    if (dstFinalPath != dstPath)
    {
        os << "Final   : " << dstFinalPath << endl;
    }

    if (!m_OutputOriginalPath.empty())
    {
        os << "Original: " << m_OutputOriginalPath << endl;
    }

    if (!m_BinTileMode.empty() && m_OutputsBntx)
    {
        os << "TileMode: " << GetBinTileModeString(m_BinTileMode) << endl;
    }

    if (!m_MergePath.empty())
    {
        os << "Merge   : " << m_MergePath << endl;
    }

    if (!IsCubeMapSeparete())
    {
        for (int level = 1; level < MipCountMax; ++level)
        {
            const std::string& mipPath = m_MipPaths[level];
            if (!mipPath.empty())
            {
                os << "Mip" << RGetNumberString(level, "%-2d") << "   : " << mipPath << endl;
            }
        }
    }
    else
    {
        static const int s_DispCubeFaceIdxs[CUBE_FACE_COUNT] =
        {
            CUBE_FACE_PZ,
            CUBE_FACE_NZ,
            CUBE_FACE_NX,
            CUBE_FACE_PX,
            CUBE_FACE_PY,
            CUBE_FACE_NY,
        };

        static const char* const s_DispCubeFaceStrs[CUBE_FACE_COUNT] =
        {
            "Front", "Back", "Left", "Right", "Top", "Bottom",
        };

        const long flagBak = os.setf(ios_base::left, ios_base::adjustfield);
        for (int iFace = 0; iFace < CUBE_FACE_COUNT; ++iFace)
        {
            const int iFaceDisp = s_DispCubeFaceIdxs[iFace];
            for (int level = 1; level < MipCountMax; ++level)
            {
                const std::string& mipPath = m_CubeMipPathss[iFaceDisp][level];
                if (!mipPath.empty())
                {
                    const std::string dispStr = s_DispCubeFaceStrs[iFace] + RGetNumberString(level);
                    os << setw(8) << dispStr << ": " << mipPath << endl;
                }
            }
        }
        os.flags(flagBak);
    }

    static const char s_RgbaStr[] = "RGBA";
    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
    {
        const std::string& replacePath = m_ReplacePaths[iRgba];
        if (!replacePath.empty())
        {
            os << "Replace" << s_RgbaStr[iRgba] << ": " << replacePath
               << " (" << s_RgbaStr[m_ReplaceChanIdxs[iRgba]] << ")" << endl;
        }
    }
}

//-----------------------------------------------------------------------------
// @brief 画像用追加属性をストリームに出力します（動作確認用）。
//-----------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& os, const ExtraImageAttr& attr)
{
    switch (attr.GetType())
    {
    case ExtraImageAttr::Type_Integer:
        os << attr.GetInteger();
        break;
    case ExtraImageAttr::Type_IntegerArray:
        os << "{";
        for (size_t valueIdx = 0; valueIdx < attr.m_IntegerValues.size(); ++valueIdx)
        {
            os << ((valueIdx > 0) ? ", " : "") << attr.m_IntegerValues[valueIdx];
        }
        os << "}";
        break;
    case ExtraImageAttr::Type_Float:
        os << attr.GetFloat();
        break;
    case ExtraImageAttr::Type_FloatArray:
        os << "{";
        for (size_t valueIdx = 0; valueIdx < attr.m_FloatValues.size(); ++valueIdx)
        {
            os << ((valueIdx > 0) ? ", " : "") << attr.m_FloatValues[valueIdx];
        }
        os << "}";
        break;
    case ExtraImageAttr::Type_String:
        os << attr.GetString();
        break;
    case ExtraImageAttr::Type_StringArray:
        os << "{";
        for (size_t valueIdx = 0; valueIdx < attr.m_StringValues.size(); ++valueIdx)
        {
            os << ((valueIdx > 0) ? ", " : "") << attr.m_StringValues[valueIdx];
        }
        os << "}";
        break;
    case ExtraImageAttr::Type_ByteArray:
        os << "{";
        for (size_t valueIdx = 0; valueIdx < attr.m_ByteValues.size(); ++valueIdx)
        {
            os << ((valueIdx > 0) ? ", " : "") << "0x"
               << RGetNumberString(attr.m_ByteValues[valueIdx], "%02x");
        }
        os << "}";
        break;
    default:
        break;
    }
    return os;
}

//-----------------------------------------------------------------------------
// キューブマップ関連の定数です。
//-----------------------------------------------------------------------------
const int RImage::CubeHcPositions[CUBE_FACE_COUNT][2] =
{
    //      [+Y]
    // [-X] [-Z] [+X] [+Z]
    //      [-Y]

    { 2, 1 }, // +X
    { 0, 1 }, // -X
    { 1, 0 }, // +Y
    { 1, 2 }, // -Y
    { 3, 1 }, // +Z
    { 1, 1 }, // -Z
};

const int RImage::CubeVcPositions[CUBE_FACE_COUNT][2] =
{
    //      [+Y]
    // [-X] [-Z] [+X]
    //      [-Y]
    //      [+Z]

    { 2, 1 }, // +X
    { 0, 1 }, // -X
    { 1, 0 }, // +Y
    { 1, 2 }, // -Y
    { 1, 3 }, // +Z
    { 1, 1 }, // -Z
};

//-----------------------------------------------------------------------------
//! @brief 水平十字キューブマップの幅と高さなら true を返します。
//-----------------------------------------------------------------------------
bool RImage::IsCubeHCWH(const int imageW, const int imageH)
{
    if (imageW % CUBE_HC_COUNT_H == 0 &&
        imageH % CUBE_HC_COUNT_V == 0)
    {
        const int faceW = imageW / CUBE_HC_COUNT_H;
        const int faceH = imageH / CUBE_HC_COUNT_V;
        return (faceW == faceH && faceW >= WidthHeightMin);
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 垂直十字キューブマップの幅と高さなら true を返します。
//-----------------------------------------------------------------------------
bool RImage::IsCubeVCWH(const int imageW, const int imageH)
{
    if (imageW % CUBE_VC_COUNT_H == 0 &&
        imageH % CUBE_VC_COUNT_V == 0)
    {
        const int faceW = imageW / CUBE_VC_COUNT_H;
        const int faceH = imageH / CUBE_VC_COUNT_V;
        return (faceW == faceH && faceW >= WidthHeightMin);
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 元画像がグレースケールなら true を返します。
//-----------------------------------------------------------------------------
bool ROriginalImage::IsGray() const
{
    const int pixCount = m_Width * m_Height;
    if (!IsFloat())
    {
        const uint8_t* pSrc = m_pImageData;
        for (int iPix = 0; iPix < pixCount; ++iPix)
        {
            if (pSrc[0] != pSrc[1] || pSrc[0] != pSrc[2])
            {
                return false;
            }
            pSrc += R_RGBA_COUNT;
        }
    }
    else
    {
        const float* pSrcF32 = reinterpret_cast<const float*>(m_pImageData);
        for (int iPix = 0; iPix < pixCount; ++iPix)
        {
            if (pSrcF32[0] != pSrcF32[1] || pSrcF32[0] != pSrcF32[2])
            {
                return false;
            }
            pSrcF32 += R_RGBA_COUNT;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 元画像の透明モードを返します。
//-----------------------------------------------------------------------------
RImage::TransparencyMode ROriginalImage::GetTransparencyMode() const
{
    RImage::TransparencyMode xpaMode = RImage::OPA;
    const int pixCount = m_Width * m_Height;
    if (m_Format == RGBA8)
    {
        const uint8_t* pSrc = m_pImageData;
        for (int iPix = 0; iPix < pixCount; ++iPix)
        {
            const uint8_t alpha = pSrc[3];
            if (alpha == 0x00)
            {
                xpaMode = RImage::MASK;
                // 残りの部分に中間的なアルファが存在する可能性があるので
                // 判定を継続します。
            }
            else if (alpha < 0xff)
            {
                xpaMode = RImage::XLU;
                break;
            }
            pSrc += R_RGBA_COUNT;
        }
    }
    else if (m_Format == RGBA32F)
    {
        const float* pSrcF32 = reinterpret_cast<const float*>(m_pImageData);
        for (int iPix = 0; iPix < pixCount; ++iPix)
        {
            const float alpha = pSrcF32[3];
            if (alpha == 0.0f)
            {
                xpaMode = RImage::MASK;
                // 残りの部分に中間的なアルファが存在する可能性があるので
                // 判定を継続します。
            }
            else if (alpha < 1.0f)
            {
                xpaMode = RImage::XLU;
                break;
            }
            pSrcF32 += R_RGBA_COUNT;
        }
    }
    return xpaMode;
}

//-----------------------------------------------------------------------------
//! @brief 画像データの情報を表示します。
//-----------------------------------------------------------------------------
void RImage::DisplayInfo(std::ostream& os) const
{
    const long flagBak = os.setf(ios_base::right, ios_base::adjustfield);

    std::ostringstream sizeOss;
    sizeOss << m_ImageW << " x " << m_ImageH << " x " << m_ImageD;

    os << setw(9) << left << m_Name << ": "
       << RGetTextureDimensionString(static_cast<FtxDimension>(m_Dimension)) << " "
       << setw(13) << left << RGetTextureFormatString(static_cast<FtxFormat>(m_Format)) << " "
       << setw(12) << sizeOss.str() << " "
       << setw(2) << right << m_MipCount << " "
       << setw(6) << right << m_ImageDataSize - m_MipDataSize << " "
       << setw(6) << m_MipDataSize << " "
       << setw(6) << m_EncodedDataSize << " "
       << m_Pitch << " "
       << endl;

    std::string levelOfsStr;
    for (int level = 0; level < m_MipCount; ++level)
    {
        if (level > 0)
        {
            levelOfsStr += " ";
        }
        levelOfsStr += RGetNumberString(static_cast<uint32_t>(m_LevelOffsets[level]));
    }
    os << "  levelOffsets: " << levelOfsStr << endl;

    os.flags(flagBak);
}

//-----------------------------------------------------------------------------
//! @brief 画像データをダンプ出力します（動作確認用）。
//-----------------------------------------------------------------------------
void RImage::DumpToFile(const std::string& filePath) const
{
    std::ofstream ofs(filePath.c_str(), ios_base::binary);
    if (ofs)
    {
        cerr << "Dump: " << m_Name << ": " << filePath << endl;
        if (m_pImageData != nullptr)
        {
            ofs.write(reinterpret_cast<const char*>(m_pImageData), m_ImageDataSize);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 画像のエンコードされたデータをデコードしてイメージデータに格納します。
//-----------------------------------------------------------------------------
bool RImage::DecodeEncodedDataToImageData(REncoder* pEncoder, const bool decodesMip)
{
    //-----------------------------------------------------------------------------
    // イメージデータを確保します。
    RFreeAndClearArray(m_pImageData);
    const int dstMipCount = (decodesMip) ? m_MipCount : 1;
    const FtxFormat srcFormat = static_cast<FtxFormat>(m_Format);
    const bool isSigned = RIsSignedTextureFormat(srcFormat);
    const FtxFormat dstFormat = (m_IsFloat) ? FtxFormat_Float_32_32_32_32 :
                                (isSigned ) ? FtxFormat_Snorm_8_8_8_8     :
                                FtxFormat_Unorm_8_8_8_8;

    std::vector<size_t> originalLevelOfss;
    m_ImageDataSize = RGetTextureDataSize(&originalLevelOfss,
        dstFormat, static_cast<FtxDimension>(m_Dimension),
        m_ImageW, m_ImageH, m_ImageD, dstMipCount);
    m_pImageData = new uint8_t[m_ImageDataSize];
    m_MipDataSize = (dstMipCount >= 2) ? m_ImageDataSize - originalLevelOfss[1] : 0;
    m_ExtraDataSize = 0;

    //-----------------------------------------------------------------------------
    // エンコーダーが初期化されていなければ初期化します。
    if (pEncoder == nullptr || m_pEncodedData == nullptr)
    {
        return false;
    }
    if (!pEncoder->IsInitialized())
    {
        RStatus status = pEncoder->Initialize(nullptr);
        if (!status)
        {
            return false;
        }
    }

    //-----------------------------------------------------------------------------
    // エンコーダーでプレーンな RGBA フォーマットに変換します。
    const texenc::ImageDimension encoderDim = REncoder::GetImageDimension(
        static_cast<FtxDimension>(m_Dimension));
    if (!pEncoder->ConvertFormat(m_pImageData, m_pEncodedData,
        RGetUnicodeFromShiftJis(RGetTextureFormatString(dstFormat)).c_str(),
        RGetUnicodeFromShiftJis(RGetTextureFormatString(srcFormat)).c_str(),
        L"", texenc::EncodeFlag_ReverseRgba,
        encoderDim, m_ImageW, m_ImageH, m_ImageD, dstMipCount))
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // 必要に応じてイメージデータを加工します。
    if (IsCubeMap())
    {
        SwapCubeFaceZ();
    }

    if (dstFormat == FtxFormat_Snorm_8_8_8_8)
    {
        ConvertToUnsigned8();
    }

    const int srcCompCount = RGetComponentCount(srcFormat);
    const bool isHintForNormal = (m_Hint == RGetHintForNormal());
    const bool isAutoLa = IsFormatForLuminanceAlpha(srcFormat) && !isHintForNormal;
    const bool isAstcNormal = RIsAstcTextureFormat(srcFormat) && isHintForNormal;
    if ((srcCompCount == 1 && m_CompSel == FtxCompSelRrr1) ||
        (isAutoLa          && m_CompSel == FtxCompSelRrrg) ||
        (isAstcNormal      && m_CompSel == FtxCompSelRa01))
    {
        SwapRgbaComponent(m_CompSel);
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief ファイルから画像データをリードします。
//-----------------------------------------------------------------------------
RStatus RImage::ReadFile( // RImage_ReadFile
    const std::string& filePath,
    const int readFlags,
    const bool removesAlpha,
    const ConverterUtility& convUtil
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // メモリーを解放します。
    Clear();

    //-----------------------------------------------------------------------------
    // check type
    m_FilePath = filePath;
    const std::string ext = RGetExtensionFromFilePath(m_FilePath);
    if (ext != TgaExtension  &&
        ext != PngExtension  &&
        ext != DdsExtension  &&
        ext != ExrExtension  &&
        ext != FtxaExtension &&
        ext != FtxbExtension)
    {
        return RStatus(RStatus::FAILURE, "Image type is not supported: " + filePath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // ファイルパスから名前を設定します。
    m_Name = RGetNoExtensionFilePath(RGetFileNameFromFilePath(filePath));

    //-----------------------------------------------------------------------------
    // read file
    RFileBuf fileBuf(m_FilePath);
    if (!fileBuf)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + m_FilePath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // decode file
    if (ext == TgaExtension)
    {
        status = DecodeTga(fileBuf.GetBuf(), fileBuf.GetSize());
    }
    else if (ext == PngExtension)
    {
        status = DecodePng(convUtil.GetWicUtility(), fileBuf.GetBuf(), fileBuf.GetSize());
    }
    else if (ext == DdsExtension)
    {
        status = DecodeDds(convUtil.GetEncoder(), fileBuf.GetBuf(), fileBuf.GetSize(),
            readFlags);
    }
    else if (ext == ExrExtension)
    {
        status = DecodeExr(convUtil.GetOpenExr(), fileBuf.GetBuf(), fileBuf.GetSize());
    }
    else if (ext == FtxaExtension || ext == FtxbExtension)
    {
        status = DecodeFtx(convUtil.GetEncoder(), fileBuf.GetBuf(), fileBuf.GetSize(),
            (ext == FtxbExtension), readFlags);
    }

    //-----------------------------------------------------------------------------
    // アルファ成分が不要なら削除します。
    if (status && removesAlpha)
    {
        RemoveAlpha();
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief ビットマップデータから画像データをリードします。
//-----------------------------------------------------------------------------
RStatus RImage::ReadBitmap( // RImage_ReadBitmap
    const void* pData,
    const int dimension,
    const int width,
    const int height,
    const int depth,
    const bool hasAlpha,
    const bool isFloat,
    const RStringArray& filePaths
)
{
    //-----------------------------------------------------------------------------
    // メモリーを解放します。
    Clear();

    //-----------------------------------------------------------------------------
    // set param
    m_FilePath = (filePaths.size() != 0) ? filePaths[0] : "invalid";
    if (dimension != FtxOpt::DIM_DEFAULT)
    {
        m_Dimension = dimension;
    }
    else
    {
        m_Dimension = (depth == FtxOpt::CUBE_FACE_COUNT) ?
            FtxDimension_CubeMap : FtxDimension_2d;
    }
    m_Format = (isFloat) ?
        FtxFormat_Float_32_32_32_32 :
        FtxFormat_Unorm_8_8_8_8;
    m_ImageW = width;
    m_ImageH = height;
    m_ImageD = depth;
    m_MipCount = 1;
    m_OriginalPaths = filePaths;
    m_HasAlpha = hasAlpha;
    m_IsFloat = isFloat;

    //-----------------------------------------------------------------------------
    // copy image data
    const int pixCount = m_ImageW * m_ImageH * m_ImageD;
    const size_t colBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    m_ImageDataSize = colBytes * pixCount;
    m_pImageData = new uint8_t[m_ImageDataSize];
    memcpy(m_pImageData, pData, m_ImageDataSize);

    //-----------------------------------------------------------------------------
    // アルファ成分がなければイメージデータのアルファ成分を 0xff (1.0) でフィルします。
    if (!hasAlpha)
    {
        RemoveAlpha();
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 画像データのデータ型を変更します。元のデータ型と同じなら何もしません。
//-----------------------------------------------------------------------------
RStatus RImage::ChangeDataType(const bool isFloat, const bool changesLevelOffsets)
{
    if (m_IsFloat == isFloat)
    {
        return RStatus::SUCCESS;
    }

    if (m_IsEncoded)
    {
        if (isFloat)
        {
            if (m_Format != FtxFormat_Unorm_8_8_8_8 &&
                m_Format != FtxFormat_Srgb_8_8_8_8  &&
                m_Format != FtxFormat_Invalid)
            {
                return RStatus(RStatus::FAILURE, "Cannot convert to float rgba format: " + m_FilePath); // RShowError
            }
        }
        else
        {
            if (m_Format != FtxFormat_Float_32_32_32_32 &&
                m_Format != FtxFormat_Invalid)
            {
                return RStatus(RStatus::FAILURE, "Cannot convert to unorm rgba format: " + m_FilePath); // RShowError
            }
        }
    }

    const size_t srcValueBytes = (m_IsFloat) ? sizeof(float) : sizeof(uint8_t);
    const size_t dstValueBytes = (isFloat  ) ? sizeof(float) : sizeof(uint8_t);
    const size_t valueCount = (m_ImageDataSize + m_ExtraDataSize) / srcValueBytes;
    const uint8_t* pOldImageData = m_pImageData;
    m_pImageData = new uint8_t[dstValueBytes * valueCount];
    m_IsFloat = isFloat;
    if (isFloat)
    {
        m_Format = FtxFormat_Float_32_32_32_32;
        m_ImageDataSize *= sizeof(float);
        m_MipDataSize   *= sizeof(float);
        m_ExtraDataSize *= sizeof(float);
        if (changesLevelOffsets)
        {
            for (int level = 0; level < m_MipCount; ++level)
            {
                m_LevelOffsets[level] *= sizeof(float);
            }
        }
        float* pDstF32 = reinterpret_cast<float*>(m_pImageData);
        const uint8_t* pSrc = pOldImageData;
        for (size_t valueIdx = 0; valueIdx < valueCount; ++valueIdx)
        {
            *pDstF32++ = *pSrc++ / 255.0f;
        }
    }
    else
    {
        m_Format = FtxFormat_Unorm_8_8_8_8;
        m_ImageDataSize /= sizeof(float);
        m_MipDataSize   /= sizeof(float);
        m_ExtraDataSize /= sizeof(float);
        if (changesLevelOffsets)
        {
            for (int level = 0; level < m_MipCount; ++level)
            {
                m_LevelOffsets[level] /= sizeof(float);
            }
        }
        uint8_t* pDst = m_pImageData;
        const float* pSrcF32 = reinterpret_cast<const float*>(pOldImageData);
        for (size_t valueIdx = 0; valueIdx < valueCount; ++valueIdx)
        {
            *pDst++ = static_cast<uint8_t>(RClampValue(0, 255, RRound(*pSrcF32++ * 255.0f)));
        }
    }
    delete[] pOldImageData;
    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief 画像の色空間を sRGB からリニアに変換します。
//!
//! @param[in] dstLinearFlag リニア変換フラグです。
//-----------------------------------------------------------------------------
void RImage::ConvertToLinear(const int dstLinearFlag)
{
    //-----------------------------------------------------------------------------
    // uint8_t 型のカラーを sRGB からリニアに変換するテーブルを作成します。
    uint8_t linearTable[256];
    CreateLinearTable(linearTable, m_IsFloat);

    //-----------------------------------------------------------------------------
    // 各成分を sRGB からリニアに変換します。
    const size_t colBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const int pixCount = static_cast<int>((m_ImageDataSize + m_ExtraDataSize) / colBytes);
    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
    {
        if ((dstLinearFlag & LinearRgbaFlags[iRgba]) != 0)
        {
            if (!m_IsFloat)
            {
                uint8_t* pDstU8 = m_pImageData + iRgba;
                for (int iPix = 0; iPix < pixCount; ++iPix)
                {
                    *pDstU8 = linearTable[*pDstU8];
                    pDstU8 += R_RGBA_COUNT;
                }
            }
            else
            {
                float* pDstF32 = reinterpret_cast<float*>(m_pImageData) + iRgba;
                for (int iPix = 0; iPix < pixCount; ++iPix)
                {
                    *pDstF32 = powf(RMax(*pDstF32, 0.0f), SrgbGammaValue);
                    pDstF32 += R_RGBA_COUNT;
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 画像の色空間をリニアから sRGB に変換します。
//!
//! @param[in] dstLinearFlag リニア変換フラグです。
//-----------------------------------------------------------------------------
void RImage::ConvertToSrgb(const int dstLinearFlag)
{
    //-----------------------------------------------------------------------------
    // uint8_t 型のカラーをリニアから sRGB に変換するテーブルを作成します。
    uint8_t srgbTable[256];
    CreateSrgbTable(srgbTable, m_IsFloat);

    //-----------------------------------------------------------------------------
    // 各成分をリニアから sRGB に変換します。
    const size_t colBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const int pixCount = static_cast<int>((m_ImageDataSize + m_ExtraDataSize) / colBytes);
    for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
    {
        if ((dstLinearFlag & LinearRgbaFlags[iRgba]) != 0)
        {
            if (!m_IsFloat)
            {
                uint8_t* pDstU8 = m_pImageData + iRgba;
                for (int iPix = 0; iPix < pixCount; ++iPix)
                {
                    *pDstU8 = srgbTable[*pDstU8];
                    pDstU8 += R_RGBA_COUNT;
                }
            }
            else
            {
                float* pDstF32 = reinterpret_cast<float*>(m_pImageData) + iRgba;
                for (int iPix = 0; iPix < pixCount; ++iPix)
                {
                    *pDstF32 = powf(RMax(*pDstF32, 0.0f), 1.0f / SrgbGammaValue);
                    pDstF32 += R_RGBA_COUNT;
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 画像のイメージデータを符号なし 8 ビットから符号付き 8 ビットに変換します。
//!        イメージデータが浮動小数点数型なら何もしません。
//-----------------------------------------------------------------------------
void RImage::ConvertToSigned8(const bool limitsMinus127)
{
    if (!m_IsFloat)
    {
        uint8_t* pDstU8 = m_pImageData;
        if (limitsMinus127)
        {
            for (size_t valueIdx = 0; valueIdx < m_ImageDataSize; ++valueIdx)
            {
                *pDstU8 = static_cast<uint8_t>(
                    (*pDstU8 >= 0x80) ? *pDstU8 - 0x80 : 0x80 + *pDstU8);
                if (*pDstU8 == 0x80)
                {
                    *pDstU8 = 0x81;
                }
                ++pDstU8;
            }
        }
        else
        {
            for (size_t valueIdx = 0; valueIdx < m_ImageDataSize; ++valueIdx)
            {
                *pDstU8 = static_cast<uint8_t>(
                    (*pDstU8 >= 0x80) ? *pDstU8 - 0x80 : 0x80 + *pDstU8);
                ++pDstU8;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 画像のイメージデータを符号付き 8 ビットから符号なし 8 ビットに変換します。
//!        イメージデータが浮動小数点数型なら何もしません。
//-----------------------------------------------------------------------------
void RImage::ConvertToUnsigned8()
{
    if (!m_IsFloat)
    {
        uint8_t* pDstU8 = m_pImageData;
        for (size_t valueIdx = 0; valueIdx < m_ImageDataSize; ++valueIdx)
        {
            const uint8_t value = *pDstU8;
            *pDstU8++ = static_cast<uint8_t>(
                ((value & 0x80) == 0) ? value + 0x80 : value - 0x80);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 画像の A 成分を削除します。
//-----------------------------------------------------------------------------
void RImage::RemoveAlpha()
{
    m_HasAlpha = false;

    // イメージデータの A 成分を 0xff (1.0) でフィルします。
    if (m_pImageData != nullptr)
    {
        const size_t pixelBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
        const size_t pixelCount = (m_ImageDataSize + m_ExtraDataSize) / pixelBytes;
        if (!m_IsFloat)
        {
            uint8_t* pDstU8 = m_pImageData;
            for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
            {
                pDstU8[3] = 0xff;
                pDstU8 += R_RGBA_COUNT;
            }
        }
        else
        {
            float* pDstF32 = reinterpret_cast<float*>(m_pImageData);
            for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
            {
                pDstF32[3] = 1.0f;
                pDstF32 += R_RGBA_COUNT;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 画像の A 成分を G 成分にコピーします。
//-----------------------------------------------------------------------------
void RImage::CopyAToG()
{
    const size_t pixelBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t pixelCount = (m_ImageDataSize + m_ExtraDataSize) / pixelBytes;
    if (!m_IsFloat)
    {
        uint8_t* pDstU8 = m_pImageData;
        for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            pDstU8[1] = pDstU8[3];
            pDstU8 += R_RGBA_COUNT;
        }
    }
    else
    {
        float* pDstF32 = reinterpret_cast<float*>(m_pImageData);
        for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            pDstF32[1] = pDstF32[3];
            pDstF32 += R_RGBA_COUNT;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定した成分選択に応じて画像のイメージデータの RGBA 成分をスワップします。
//!        成分選択に応じて A 成分存在フラグも設定します。
//-----------------------------------------------------------------------------
void RImage::SwapRgbaComponent(const int compSel)
{
    const int compR = ((compSel >> FtxCompSelShiftR) & 0xff) % FtxComponent_Count;
    const int compG = ((compSel >> FtxCompSelShiftG) & 0xff) % FtxComponent_Count;
    const int compB = ((compSel >> FtxCompSelShiftB) & 0xff) % FtxComponent_Count;
    const int compA = ((compSel >> FtxCompSelShiftA) & 0xff) % FtxComponent_Count;
    m_HasAlpha = (compA != FtxComponent_One);

    const size_t pixelBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t pixelCount = (m_ImageDataSize + m_ExtraDataSize) / pixelBytes;
    if (!m_IsFloat)
    {
        uint8_t srcCompsU8[FtxComponent_Count] = { 0 };
        srcCompsU8[FtxComponent_Zero] = 0x00;
        srcCompsU8[FtxComponent_One ] = 0xff;
        uint8_t* pDstU8 = m_pImageData;
        for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            srcCompsU8[FtxComponent_Red  ] = pDstU8[0];
            srcCompsU8[FtxComponent_Green] = pDstU8[1];
            srcCompsU8[FtxComponent_Blue ] = pDstU8[2];
            srcCompsU8[FtxComponent_Alpha] = pDstU8[3];
            pDstU8[0] = srcCompsU8[compR];
            pDstU8[1] = srcCompsU8[compG];
            pDstU8[2] = srcCompsU8[compB];
            pDstU8[3] = srcCompsU8[compA];
            pDstU8 += R_RGBA_COUNT;
        }
    }
    else
    {
        float srcCompsF32[FtxComponent_Count] = { 0.0f };
        srcCompsF32[FtxComponent_Zero] = 0.0f;
        srcCompsF32[FtxComponent_One ] = 1.0f;
        float* pDstF32 = reinterpret_cast<float*>(m_pImageData);
        for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            srcCompsF32[FtxComponent_Red  ] = pDstF32[0];
            srcCompsF32[FtxComponent_Green] = pDstF32[1];
            srcCompsF32[FtxComponent_Blue ] = pDstF32[2];
            srcCompsF32[FtxComponent_Alpha] = pDstF32[3];
            pDstF32[0] = srcCompsF32[compR];
            pDstF32[1] = srcCompsF32[compG];
            pDstF32[2] = srcCompsF32[compB];
            pDstF32[3] = srcCompsF32[compA];
            pDstF32 += R_RGBA_COUNT;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 画像のチャンネルを差し替えます。
//-----------------------------------------------------------------------------
RStatus RImage::ReplaceChannel(
    const int dstChanIdx,
    const RImage& srcImg,
    const int srcChanIdx
)
{
    //-----------------------------------------------------------------------------
    // 幅、高さ、奥行き、データ型をチェックします。
    if (m_ImageW != srcImg.GetImageW() ||
        m_ImageH != srcImg.GetImageH() ||
        m_ImageD != srcImg.GetImageD())
    {
        return RStatus(RStatus::FAILURE, "Replace width or height or depth is not identical: " + srcImg.GetFilePath()); // RShowError
    }
    else if (m_IsFloat != srcImg.IsFloat())
    {
        return RStatus(RStatus::FAILURE, "Replace data type is not identical: " + srcImg.GetFilePath()); // RShowError
    }

    //-----------------------------------------------------------------------------
    // 差し替えます。
    size_t replaceDataSize = m_ImageDataSize - m_MipDataSize;
    if (m_MipDataSize   == srcImg.GetMipDataSize()   &&
        m_ExtraDataSize == srcImg.GetExtraDataSize())
    {
        replaceDataSize += m_MipDataSize + m_ExtraDataSize;
    }
    const size_t pixelBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t pixelCount = replaceDataSize / pixelBytes;

    if (!m_IsFloat)
    {
        uint8_t* pDstU8 = m_pImageData + dstChanIdx;
        const uint8_t* pSrcU8 = srcImg.GetImageData() + srcChanIdx;
        for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            *pDstU8 = *pSrcU8;
            pSrcU8 += R_RGBA_COUNT;
            pDstU8 += R_RGBA_COUNT;
        }
    }
    else
    {
        float* pDstF32 = reinterpret_cast<float*>(m_pImageData) + dstChanIdx;
        const float* pSrcF32 = reinterpret_cast<const float*>(srcImg.GetImageData()) + srcChanIdx;
        for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            *pDstF32 = *pSrcF32;
            pSrcF32 += R_RGBA_COUNT;
            pDstF32 += R_RGBA_COUNT;
        }
    }

    //-----------------------------------------------------------------------------
    // A チャンネルが差し替えられたなら、アルファ成分存在フラグを true にします。
    if (dstChanIdx == 3)
    {
        m_HasAlpha = true;
    }

    return RStatus::SUCCESS;
}

//-----------------------------------------------------------------------------
//! @brief キューブマップのイメージデータの +Z 方向と -Z 方向のデータを交換します。
//!        キューブマップおよびキューブマップ配列以外なら何もしません。
//-----------------------------------------------------------------------------
void RImage::SwapCubeFaceZ()
{
    if (!IsCubeMap())
    {
        return;
    }

    const size_t pixelBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t faceBufSize = pixelBytes * m_ImageW * m_ImageH;
    uint8_t* faceBuf = new uint8_t[faceBufSize];
    uint8_t* pDst = m_pImageData;
    const int mipCount = (m_MipDataSize != 0) ? m_MipCount : 1;
    for (int level = 0; level < mipCount; ++level)
    {
        const int levelW = RMax(m_ImageW >> level, 1);
        const int levelH = RMax(m_ImageH >> level, 1);
        const size_t levelFaceBytes = pixelBytes * levelW * levelH;
        for (int iz = 0; iz < m_ImageD; iz += CUBE_FACE_COUNT)
        {
            uint8_t* pDstPz = pDst + levelFaceBytes * CUBE_FACE_PZ;
            uint8_t* pDstNz = pDst + levelFaceBytes * CUBE_FACE_NZ;
            memcpy(faceBuf, pDstPz , levelFaceBytes);
            memcpy(pDstPz , pDstNz , levelFaceBytes);
            memcpy(pDstNz , faceBuf, levelFaceBytes);
            pDst += levelFaceBytes * CUBE_FACE_COUNT;
        }
    }
    delete[] faceBuf;
}

//-----------------------------------------------------------------------------
//! @brief 画像を切り抜きます。
//-----------------------------------------------------------------------------
RStatus RImage::Crop(const RCropRect& cropRect)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // 切り抜き矩形が画像全体なら何もしません。
    if (cropRect.IsEntireImage(m_ImageW, m_ImageH))
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // 切り抜き矩形が画像の範囲内かチェックします。
    if (!cropRect.IsWithinImage(m_ImageW, m_ImageH))
    {
        return RStatus(RStatus::FAILURE, "Cannot crop image: " + m_FilePath + ": " + // RShowError
             RGetNumberString(cropRect.m_X) + "," +
             RGetNumberString(cropRect.m_Y) + "," +
             RGetNumberString(cropRect.m_W) + "," +
             RGetNumberString(cropRect.m_H));
    }

    //-----------------------------------------------------------------------------
    // 切り抜き後のイメージデータを確保します。
    const size_t pixelBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t dstFaceBytes = pixelBytes * cropRect.m_W * cropRect.m_H;
    const size_t croppedSize = dstFaceBytes * m_ImageD;
    uint8_t* pCroppedData = new uint8_t[croppedSize];

    //-----------------------------------------------------------------------------
    // 各フェースを切り抜きます。
    const size_t srcFaceBytes = pixelBytes * m_ImageW * m_ImageH;
    for (int iz = 0; iz < m_ImageD; ++iz)
    {
        uint8_t* pDst       = pCroppedData + iz * dstFaceBytes;
        const uint8_t* pSrc = m_pImageData + iz * srcFaceBytes;
        if (!m_IsFloat)
        {
            RCropFace(pDst, pSrc, cropRect, m_ImageW);
        }
        else
        {
            float* pDstF32 = reinterpret_cast<float*>(pDst);
            const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
            RCropFace(pDstF32, pSrcF32, cropRect, m_ImageW);
        }
    }

    //-----------------------------------------------------------------------------
    // 切り抜き後のイメージデータを画像に反映します。
    m_ImageW = cropRect.m_W;
    m_ImageH = cropRect.m_H;
    m_MipCount = 1;

    delete[] m_pImageData;
    m_pImageData = pCroppedData;
    m_ImageDataSize = croppedSize;
    m_MipDataSize = 0;
    m_ExtraDataSize = 0;

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 画像サイズ変更後の幅／高さ／奥行きを取得します。
//-----------------------------------------------------------------------------
bool RImage::GetResizedWhd(
    int* pDstW,
    int* pDstH,
    int* pDstD,
    const int reductionLevel,
    const int resizeW,
    const int resizeH,
    const float percentW,
    const float percentH
) const
{
    int dstW = m_ImageW;
    int dstH = m_ImageH;
    int dstD = m_ImageD;

    if (resizeW  != 0    ||
        resizeH  != 0    ||
        percentW != 0.0f ||
        percentH != 0.0f)
    {
        const float scaleW = percentW / 100.0f;
        const float scaleH = percentH / 100.0f;
        dstW = (resizeW  != 0   ) ? resizeW                                                            :
               (percentW != 0.0f) ? RMax(RRound(m_ImageW * scaleW), 1)                                 :
               (resizeH  != 0   ) ? RMax(RRound(static_cast<float>(m_ImageW) / m_ImageH * resizeH), 1) :
               RMax(RRound(m_ImageW * scaleH), 1);
        dstH = (resizeH  != 0   ) ? resizeH                                                            :
               (percentH != 0.0f) ? RMax(RRound(m_ImageH * scaleH), 1)                                 :
               (resizeW  != 0   ) ? RMax(RRound(static_cast<float>(m_ImageH) / m_ImageW * resizeW), 1) :
               RMax(RRound(m_ImageH * scaleW), 1);
    }
    else if (reductionLevel > 0)
    {
        dstW = RMax(m_ImageW >> reductionLevel, 1);
        dstH = RMax(m_ImageH >> reductionLevel, 1);
        dstD = (m_Dimension == FtxDimension_3d) ?
            RMax(m_ImageD >> reductionLevel, 1) : m_ImageD;
    }

    if (pDstW != nullptr)
    {
        *pDstW = dstW;
    }
    if (pDstH != nullptr)
    {
        *pDstH = dstH;
    }
    if (pDstD != nullptr)
    {
        *pDstD = dstD;
    }
    return (dstW != m_ImageW || dstH != m_ImageH || dstD != m_ImageD);
}

//-----------------------------------------------------------------------------
//! @brief 画像をサイズ変更します。
//-----------------------------------------------------------------------------
RStatus RImage::Resize(
    const int dstW,
    const int dstH,
    const int dstD,
    const int linearFlag,
    const ResizeFilter resizeFilter,
    const bool ignoresXpaRgb
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // サイズ変更なしなら何もしません。
    if (m_ImageW == dstW && m_ImageH == dstH && m_ImageD == dstD)
    {
        return status;
    }
    //cerr << "resize: " << RGetFileNameFromFilePath(m_FilePath) << ": "
    //   << m_ImageW << "x" << m_ImageH << "x" << m_ImageD << " -> " << dstW << "x" << dstH << "x" << dstD << ", "
    //   << GetOptStringFromLinearFlag(linearFlag) << ", " << resizeFilter << ", " << ignoresXpaRgb << endl;

    //-----------------------------------------------------------------------------
    // 必要があれば sRGB からリニアに変換します。
    const bool isOriginalFloat = m_IsFloat;
    if (linearFlag != FtxOpt::LINEAR_NONE)
    {
        //-----------------------------------------------------------------------------
        // データ型が整数なら浮動小数点数フォーマットに変換します。
        if (!isOriginalFloat)
        {
            status = ChangeDataType(true, true);
            RCheckStatus(status);
        }

        //-----------------------------------------------------------------------------
        // sRGB からリニアに変換します。
        ConvertToLinear(linearFlag);
    }

    //-----------------------------------------------------------------------------
    // サイズ変更後のイメージデータを確保します。
    const size_t pixelBytes = (m_IsFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t dstFaceBytes = pixelBytes * dstW * dstH;
    const size_t resizedSize = dstFaceBytes * dstD;
    uint8_t* pResizedData = new uint8_t[resizedSize];

    //-----------------------------------------------------------------------------
    // サイズ変更します。
    if (m_Dimension != FtxDimension_3d || m_ImageD == dstD)
    {
        const size_t srcFaceBytes = pixelBytes * m_ImageW * m_ImageH;
        for (int iz = 0; iz < dstD; ++iz)
        {
            const int srcZ = (iz < m_ImageD) ? iz : m_ImageD - 1;
            uint8_t* pDst       = pResizedData + iz   * dstFaceBytes;
            const uint8_t* pSrc = m_pImageData + srcZ * srcFaceBytes;
            if (!m_IsFloat)
            {
                RResizeFreeFace(pDst, pSrc, dstW, dstH, m_ImageW, m_ImageH,
                    resizeFilter, ignoresXpaRgb);
            }
            else
            {
                const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
                float* pDstF32 = reinterpret_cast<float*>(pDst);
                RResizeFreeFace(pDstF32, pSrcF32, dstW, dstH, m_ImageW, m_ImageH,
                    resizeFilter, ignoresXpaRgb);
            }
        }
    }
    else
    {
        uint8_t* pDst       = pResizedData;
        const uint8_t* pSrc = m_pImageData;
        if (!m_IsFloat)
        {
            RResizeFree3d(pDst, pSrc, dstW, dstH, dstD, m_ImageW, m_ImageH, m_ImageD,
                resizeFilter, ignoresXpaRgb);
        }
        else
        {
            const float* pSrcF32 = reinterpret_cast<const float*>(pSrc);
            float* pDstF32 = reinterpret_cast<float*>(pDst);
            RResizeFree3d(pDstF32, pSrcF32, dstW, dstH, dstD, m_ImageW, m_ImageH, m_ImageD,
                resizeFilter, ignoresXpaRgb);
        }
    }

    //-----------------------------------------------------------------------------
    // サイズ変更後のイメージデータを画像に反映します。
    m_ImageW = dstW;
    m_ImageH = dstH;
    m_ImageD = dstD;
    m_MipCount = 1;

    delete[] m_pImageData;
    m_pImageData = pResizedData;
    m_ImageDataSize = resizedSize;
    m_MipDataSize = 0;
    m_ExtraDataSize = 0;

    //-----------------------------------------------------------------------------
    // 必要があればリニアから sRGB に変換し、整数フォーマットに変換します。
    if (linearFlag != FtxOpt::LINEAR_NONE)
    {
        //-----------------------------------------------------------------------------
        // リニアから sRGB に変換します。
        ConvertToSrgb(linearFlag);

        //-----------------------------------------------------------------------------
        // 元のデータ型が整数フォーマットなら整数フォーマットに変換します。
        if (!isOriginalFloat)
        {
            status = ChangeDataType(false, true);
            RCheckStatus(status);
        }
    }
    return status;
}

//-----------------------------------------------------------------------------
//! @brief NX 向けのタイリング変換をします。
//!
//! @param[in,out] pNXUtility NX ユーティリティーへのポインタです。
//! @param[in] isTiling NX 向けのタイリング変換をするなら true、タイリング変換を解除するなら false です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus RImage::ConvertNXTiling(NXUtility* pNXUtility, const bool isTiling)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // NX ユーティリティーが初期化されていなければ初期化します。
    NXUtility& nxUtility = *pNXUtility;
    if (!nxUtility.IsInitialized())
    {
        status = nxUtility.Initialize(nullptr);
        RCheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // タイリング変換をします。
    NXTexture nxTex;
    const std::string dimStr = RGetTextureDimensionString(
        static_cast<FtxDimension>(m_Dimension));
    const std::string formatStr = RGetTextureFormatString(
        static_cast<FtxFormat>(m_Format));
    const int depth = (m_Dimension == FtxDimension_3d) ? m_ImageD : 1;
    const int arrayLength = (IsCubeMap() || IsArray()) ? m_ImageD : 1;
    const uint32_t tilingFlags = NXTilingFlag_Default |
        ((m_IsSparseTiled                           ) ? NXTilingFlag_Sparse           : 0) |
        ((m_TileOptimize == FtxTileOptimize_Size    ) ? NXTilingFlag_OptimizeSize     : 0) |
        ((m_TileOptimize == FtxTileOptimize_SizeAuto) ? NXTilingFlag_OptimizeSizeAuto : 0);
    if (!nxUtility.ConvertTiling(&nxTex, isTiling, m_pImageData, m_ImageDataSize,
        RGetUnicodeFromShiftJis(dimStr).c_str(),
        RGetUnicodeFromShiftJis(formatStr).c_str(),
        m_ImageW, m_ImageH, depth, arrayLength, m_MipCount,
        &m_TextureLayout, tilingFlags, m_TileSizeThreshold))
    {
        return RStatus(RStatus::FAILURE, "Cannot convert NX tiling: " + m_FilePath + // RShowError
            " (" + formatStr + ")");
    }

    //-----------------------------------------------------------------------------
    // テクスチャーデータをコピーします。
    if (m_pImageData != nullptr)
    {
        delete[] m_pImageData;
    }
    m_ImageDataSize = nxTex.dataSize;
    m_pImageData = new uint8_t[m_ImageDataSize];
    m_MipDataSize = (m_MipCount >= 2) ?
        nxTex.dataSize - nxTex.levelOffsets[1] : 0;
    memcpy(m_pImageData, nxTex.pData, nxTex.dataSize);
    m_ExtraDataSize = 0;
    m_TileMode = (isTiling) ? FtxTileMode_NX : FtxTileMode_Linear;
    m_Alignment = nxTex.alignment;
    ClearLevelOffsets();
    for (int level = 0; level < m_MipCount; ++level)
    {
        m_LevelOffsets[level] = nxTex.levelOffsets[level];
        //cerr << "level" << level << ": " << nxTex.levelOffsets[level] << endl;
    }

    m_UsesTextureLayout = isTiling;
    if (isTiling)
    {
        memcpy(&m_TextureLayout, nxTex.textureLayout, sizeof(m_TextureLayout));
    }
    else
    {
        ClearTextureLayout();
    }

    //-----------------------------------------------------------------------------
    // タイリング解除のテストです。
    //if (isTiling)
    //{
    //  NXTexture detiled;
    //  if (!nxUtility.ConvertTiling(&detiled, false, nxTex.pData, nxTex.dataSize,
    //      RGetUnicodeFromShiftJis(dimStr).c_str(),
    //      RGetUnicodeFromShiftJis(formatStr).c_str(),
    //      m_ImageW, m_ImageH, depth, arrayLength, m_MipCount,
    //      &m_TextureLayout, tilingFlags, 0))
    //  {
    //      return RStatus(RStatus::FAILURE, "Cannot detile NX texture: " + m_FilePath + // RShowError
    //          " (" + formatStr + ")");
    //  }
    //  cerr << "detiled: " << detiled.dataSize << endl;
    //  const size_t dumpSize = RMin(detiled.dataSize, static_cast<size_t>(32));
    //  for (size_t i = 0; i < dumpSize; ++i)
    //  {
    //      cerr << RGetNumberString(static_cast<int>(detiled.pData[i]), "%02X");
    //      cerr << ((((i + 1) % 16) == 0 || (i + 1) == dumpSize) ? "\n" : " ");
    //  }
    //    for (int level = 0; level < m_MipCount; ++level)
    //    {
    //      cerr << "detiled level" << level << ": " << detiled.levelOffsets[level] << endl;
    //  }
    //  nxUtility.ReleaseTexture(&detiled);
    //}

    //-----------------------------------------------------------------------------
    // NX テクスチャーを解放します。
    nxUtility.ReleaseTexture(&nxTex);

    return status;
}

//-----------------------------------------------------------------------------
//! @brief ファイルをリードしてマージ情報を取得します。
//-----------------------------------------------------------------------------
RStatus RMergeInfo::ReadFile(const std::string& filePath)
{
    //-----------------------------------------------------------------------------
    // check file type
    m_FilePath = filePath;
    const std::string ext = RGetExtensionFromFilePath(m_FilePath);
    if (ext != FtxaExtension &&
        ext != FtxbExtension)
    {
        return RStatus(RStatus::FAILURE, "Merge file type is not supported: " + m_FilePath); // RShowError
    }
    const bool isBinary = (ext == FtxbExtension);

    //-----------------------------------------------------------------------------
    // read file
    RFileBuf fileBuf(m_FilePath);
    if (!fileBuf)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + m_FilePath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // parse ftx
    return ParseFtx(fileBuf.GetBuf(), fileBuf.GetSize(), isBinary);
}

//-----------------------------------------------------------------------------
//! @brief ftx ファイルをサポートしていない場合のエラーステータスを取得します。
//-----------------------------------------------------------------------------
RStatus TexConverterBase::GetFtxUnsupportedStatus(const std::string& filePath)
{
    const std::string filePathMsg = (!filePath.empty()) ? ": " + filePath : "";
    return RStatus(RStatus::FAILURE, "ftx file is not supported (Please use 3dTextureConverter.exe)" +
        filePathMsg);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターのリード前の共通処理です。
//-----------------------------------------------------------------------------
RStatus TexConverterBase::PrepareToRead(const wchar_t* paths[], const wchar_t* options[])
{
    //-----------------------------------------------------------------------------
    // メモリーをクリアし、ジョブオプションを初期化します。
    Clear();
    ClearError();

    //-----------------------------------------------------------------------------
    // ジョブオプションを設定します。
    RStatus status = m_Opt.SetJobOptions(options);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // 入力ファイルパスを設定します。
    status = m_Opt.SetInputPaths(paths);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // 出力ファイルパスが指定されていなければ、
    // 入力ファイルと同じフォルダーにデフォルトの拡張子で出力します。
    m_DstPath = m_Opt.m_OutputPath;
    if (m_DstPath.empty())
    {
        if (m_SupportsFtx && RGetExtensionFromFilePath(m_Opt.m_InputPaths[0]) == FtxaExtension)
        {
            m_DstPath = m_Opt.m_InputPaths[0]; // ftxa ファイルを上書き
        }
        else
        {
            const std::string defaultExt = (m_SupportsFtx) ? FtxbExtension : BntxExtension;
            m_DstPath = RGetNoExtensionFilePath(m_Opt.m_InputPaths[0]) + "." + defaultExt;
        }
    }

    m_DstFinalPath = (m_Opt.m_FinalFolderPath.empty()) ?
        m_DstPath : m_Opt.m_FinalFolderPath + "/" + RGetFileNameFromFilePath(m_DstPath);

    //-----------------------------------------------------------------------------
    // ftx ファイルをサポートしていない場合に出力ファイルが ftx ファイルならエラーにします。
    if (!m_SupportsFtx)
    {
        if (IsFtxFilePath(m_DstPath))
        {
            return GetFtxUnsupportedStatus(m_DstPath);
        }
    }

    //-----------------------------------------------------------------------------
    // テクスチャー中間ファイルからテクスチャーバイナリーファイルへの変換なら処理を終了します。
    if (m_Opt.m_OutputsBntx && !m_Opt.m_IsImageToBntx)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // キューブマップ個別指定なら各面の入力ファイルが指定されているかチェックします。
    if (m_Opt.IsCubeMapSeparete())
    {
        status = CheckCubeMapSeparete(m_Opt);
        RCheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // 入力ファイルと出力ファイルの情報を表示します。
    if (!m_Opt.m_IsSilent)
    {
        m_Opt.DisplayFileInfo(cout, m_DstPath, m_DstFinalPath);
    }

    //-----------------------------------------------------------------------------
    // 入力ファイルの拡張子をチェックします。
    RStringArray allInputPaths;
    m_Opt.GetAllInputPaths(allInputPaths);

    bool isInputEXR = false;
    bool isInputTGA = false;
    for (size_t iFile = 0; iFile < allInputPaths.size(); ++iFile)
    {
        const std::string ext = RGetExtensionFromFilePath(allInputPaths[iFile]);
        if (ext == ExrExtension)
        {
            isInputEXR = true;
        }
        else if (ext == "tga")
        {
            isInputTGA = true;
        }
        if (isInputEXR && isInputTGA)
        {
            // EXR ファイルと TGA ファイルが入力に混在している場合はエラーです。
            return RStatus(RStatus::FAILURE, "EXR and TGA file is specified"); // RShowError
        }
    }

    //-----------------------------------------------------------------------------
    // マージするファイルをリードします。
    m_MergeInfo.Clear();
    if (!m_Opt.m_MergePath.empty())
    {
        if (m_Opt.m_DisablesMergeError && !RFileExists(m_Opt.m_MergePath))
        {
            // --disable-merge-error オプションが指定されていれば、
            // マージするファイルが存在しなくてもエラーにしません。
        }
        else
        {
            status = m_MergeInfo.ReadFile(m_Opt.m_MergePath);
            RCheckStatus(status);
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 変換先の各パラメータを決定します。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus TexConverterBase::DecideDstParam()
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // 最初の元画像へのポインタを取得します。
    const ROriginalImage* pFirstOrgImg = (!m_OriginalImages.empty()) ?
        &m_OriginalImages[0] : nullptr;

    //-----------------------------------------------------------------------------
    // TGA、PNG、DDS（再エンコードなしまたはミップマップなし）、EXR への変換なら何もしません。
    const std::string dstExt = RGetExtensionFromFilePath(m_DstPath);
    m_ConvertsToTga = (dstExt == TgaExtension);
    m_ConvertsToPng = (dstExt == PngExtension);
    m_ConvertsToDds = (dstExt == DdsExtension);
    m_ConvertsToExr = (dstExt == ExrExtension);
    m_ConvertsToEncodedDds = (m_ConvertsToDds && m_Opt.m_Format != FtxOpt::FORMAT_DEFAULT);
    m_ConvertsToMipDds = (m_ConvertsToDds && !m_ConvertsToEncodedDds && m_Opt.m_MipCount != FtxOpt::MipCountDefault);

    if (m_Opt.m_NoEncoding)
    {
        if (dstExt != FtxaExtension &&
            dstExt != FtxbExtension &&
            dstExt != BntxExtension &&
            !m_ConvertsToDds)
        {
            return RStatus(RStatus::FAILURE, "Output file type is wrong for no encoding: " + m_DstPath); // RShowError
        }
    }

    if (m_ConvertsToTga)
    {
        if (pFirstOrgImg != nullptr && pFirstOrgImg->IsFloat())
        {
            return RStatus(RStatus::FAILURE, "Cannot convert the real number data to TGA: " + m_Opt.m_InputPaths[0]); // RShowError
        }
        return status;
    }
    else if (m_ConvertsToPng)
    {
        return status;
    }
    else if (m_ConvertsToDds)
    {
        if (!m_ConvertsToEncodedDds && !m_ConvertsToMipDds)
        {
            return status;
        }
    }
    else if (m_ConvertsToExr)
    {
        if (pFirstOrgImg != nullptr && !pFirstOrgImg->IsFloat())
        {
            return RStatus(RStatus::FAILURE, "Cannot convert the integer data to EXR: " + m_Opt.m_InputPaths[0]); // RShowError
        }
        return status;
    }
    else if (dstExt != FtxaExtension &&
             dstExt != FtxbExtension &&
             dstExt != BntxExtension)
    {
        return RStatus(RStatus::FAILURE, "Output file type is wrong: " + m_DstPath); // RShowError
    }

    //-----------------------------------------------------------------------------
    // エンコード済みデータを使用する場合は以下のオプションを無視します。
    if (m_Opt.m_NoEncoding)
    {
        const std::string srcExt = RGetExtensionFromFilePath(m_Opt.m_InputPaths[0]);
        if (!m_Opt.m_IsSilent && srcExt == DdsExtension)
        {
            cout << "Format  : " << RGetTextureFormatString(m_DstFormat) << endl;
            if (m_DstMipCount >= 2)
            {
                cout << "Mipmap  : " << m_DstMipCount << endl;
            }
            if (!m_DstHint.empty())
            {
                cout << "Hint    : " << m_DstHint << endl;
            }
            cout << "CompSel : " << RGetCompSelString(m_DstCompSel) << endl;
        }
        return status;
    }

    //-----------------------------------------------------------------------------
    // フォーマットとミップマップのレベル数を決定します。
    m_DstFormat = FtxFormat_Unorm_8_8_8_8;
    status = DecideFormat(&m_DstFormat, m_Opt, m_MergeInfo, m_OriginalImages, nullptr);
    RCheckStatus(status);
    m_DstMipCount = DecideMipCount(m_Opt, m_MergeInfo, m_OriginalImages);

    if (!m_Opt.m_IsSilent && !m_ConvertsToMipDds)
    {
        cout << "Format  : " << RGetTextureFormatString(m_DstFormat) << endl;
    }

    if (!m_Opt.m_IsSilent && m_Opt.m_Depth != FtxOpt::DEPTH_DEFAULT)
    {
        cout << "Depth   : " << m_Opt.m_Depth << endl;
    }

    if (m_DstMipCount >= 2 && m_WorkImage.GetMipCount() <= 1)
    {
        const int maxLevel = GetMaxMipmapLevels(m_WorkImage);
        m_DstMipCount = (m_DstMipCount <= maxLevel) ? m_DstMipCount : maxLevel;

        if (!m_Opt.m_IsSilent)
        {
            cout << "Mipmap  : " << m_DstMipCount << endl;
        }
    }

    //-----------------------------------------------------------------------------
    // ミップマップ生成フィルタを決定します。
    m_DstMipGenFilter = FtxOpt::MIP_GEN_LINEAR;
    bool mipGenFilterSpecified = false;
    if (m_Opt.m_MipGenFilter != FtxOpt::MIP_GEN_DEFAULT)
    {
        m_DstMipGenFilter = m_Opt.m_MipGenFilter;
        mipGenFilterSpecified = true;
    }
    else if (m_MergeInfo.m_IsEnable && m_MergeInfo.m_MipGenFilter != FtxOpt::MIP_GEN_DEFAULT)
    {
        m_DstMipGenFilter = m_MergeInfo.m_MipGenFilter;
        mipGenFilterSpecified = true;
    }
    else if (pFirstOrgImg != nullptr)
    {
        if (pFirstOrgImg->m_MipGenFilter != FtxOpt::MIP_GEN_DEFAULT)
        {
            m_DstMipGenFilter = pFirstOrgImg->m_MipGenFilter;
            mipGenFilterSpecified = true;
        }
    }

    if (!m_Opt.m_IsSilent && mipGenFilterSpecified)
    {
        cout << "MipGen  : " << FtxOpt::GetMipGenFilterString(m_DstMipGenFilter) << endl;
    }

    //-----------------------------------------------------------------------------
    // ヒント情報を決定します。
    m_DstHint = RGetDefaultHint(m_DstFormat);
    bool hintSpecified = false;
    if (m_Opt.m_Hint != FtxOpt::HINT_DEFAULT)
    {
        m_DstHint = m_Opt.m_Hint;
        hintSpecified = true;
    }
    else if (m_MergeInfo.m_IsEnable && m_MergeInfo.m_Hint != FtxOpt::HINT_DEFAULT)
    {
        m_DstHint = m_MergeInfo.m_Hint;
        hintSpecified = true;
    }
    else if (pFirstOrgImg != nullptr)
    {
        if (pFirstOrgImg->m_Hint != FtxOpt::HINT_DEFAULT)
        {
            m_DstHint = pFirstOrgImg->m_Hint;
            hintSpecified = true;
        }
    }

    if (!m_Opt.m_IsSilent && hintSpecified && !m_ConvertsToMipDds)
    {
        cout << "Hint    : " << m_DstHint << endl;
    }

    //-----------------------------------------------------------------------------
    // 成分選択を決定します。
    m_DstCompSel = RGetDefaultCompSel(m_DstFormat);
    bool compSelSpecified = false;
    if (m_Opt.m_CompSel != FtxOpt::COMP_SEL_DEFAULT)
    {
        m_DstCompSel = m_Opt.m_CompSel;
        compSelSpecified = true;
    }
    else if (m_MergeInfo.m_IsEnable && m_MergeInfo.m_CompSel != FtxOpt::COMP_SEL_DEFAULT)
    {
        m_DstCompSel = m_MergeInfo.m_CompSel;
        compSelSpecified = true;
    }
    else if (pFirstOrgImg != nullptr)
    {
        const int originalCompSel = pFirstOrgImg->m_CompSel;
        if (originalCompSel != FtxOpt::COMP_SEL_DEFAULT &&
            KeepsCompSel(static_cast<FtxFormat>(pFirstOrgImg->m_OriginalFormat), m_DstFormat))
        {
            // 入力が ftx ファイルの場合、
            // 成分選択の指定がなくてフォーマットの成分数と符号あり／なしが変わらなければ comp_sel を維持します。
            m_DstCompSel = originalCompSel;
            compSelSpecified = true;
        }
        else if (RIsAstcTextureFormat(m_DstFormat) && m_DstHint == RGetHintForNormal())
        {
            // ASTC フォーマットでヒント情報が normal の場合、
            // 成分選択の指定がなければデフォルトの成分選択を ra01 にします。
            m_DstCompSel = FtxCompSelRa01;
            compSelSpecified = true;
        }
    }

    if (!m_Opt.m_IsSilent && compSelSpecified && !m_ConvertsToDds)
    {
        cout << "CompSel : " << RGetCompSelString(m_DstCompSel) << endl;
    }

    //-----------------------------------------------------------------------------
    // BC 圧縮重み付けを決定します。
    m_DstWeightedCompress = false;
    bool weightedCompressSpecified = false;
    if (m_Opt.m_WeightedCompress != FtxOpt::WEIGHTED_COMPRESS_DEFAULT)
    {
        m_DstWeightedCompress = (m_Opt.m_WeightedCompress == FtxOpt::WEIGHTED_COMPRESS_ENABLE);
        weightedCompressSpecified = true;
    }
    else if (m_MergeInfo.m_IsEnable && m_MergeInfo.m_WeightedCompress != FtxOpt::WEIGHTED_COMPRESS_DEFAULT)
    {
        m_DstWeightedCompress = (m_MergeInfo.m_WeightedCompress == FtxOpt::WEIGHTED_COMPRESS_ENABLE);
        weightedCompressSpecified = true;
    }
    else if (pFirstOrgImg != nullptr)
    {
        const int originalWeightedCompress = pFirstOrgImg->m_WeightedCompress;
        if (originalWeightedCompress != FtxOpt::WEIGHTED_COMPRESS_DEFAULT)
        {
            m_DstWeightedCompress = (originalWeightedCompress == FtxOpt::WEIGHTED_COMPRESS_ENABLE);
            weightedCompressSpecified = true;
        }
    }

    if (!m_Opt.m_IsSilent && !m_Opt.m_Quality.empty() && !m_ConvertsToMipDds)
    {
        cout << "Quality : " << m_Opt.m_Quality << endl;
    }
    if (!m_Opt.m_IsSilent && weightedCompressSpecified && !m_ConvertsToMipDds)
    {
        cout << "WeightBC: " << RBoolStr(m_DstWeightedCompress) << endl;
    }

    //-----------------------------------------------------------------------------
    // DCC プリセット名を決定します。
    m_DstDccPreset = "";
    bool dccPresetSpecified = false;
    if (m_Opt.m_DccPreset != FtxOpt::DCC_PRESET_DEFAULT)
    {
        m_DstDccPreset = m_Opt.m_DccPreset;
        dccPresetSpecified = true;
    }
    else if (pFirstOrgImg != nullptr)
    {
        if (pFirstOrgImg->m_DccPreset != FtxOpt::DCC_PRESET_DEFAULT)
        {
            m_DstDccPreset = pFirstOrgImg->m_DccPreset;
            dccPresetSpecified = true;
        }
    }

    if (!m_Opt.m_IsSilent && dccPresetSpecified && !m_ConvertsToDds)
    {
        cout << "Preset  : " << m_DstDccPreset << endl;
    }

    //-----------------------------------------------------------------------------
    // リニア変換フラグを決定します。
    bool linearFlagSpecified = false;
    m_DstLinearFlag = DecideLinearFlag(&linearFlagSpecified,
        m_Opt, m_MergeInfo, m_OriginalImages, nullptr, m_DstFormat);
    if (!m_Opt.m_IsSilent && linearFlagSpecified)
    {
        cout << "Linear  : " << GetOptStringFromLinearFlag(m_DstLinearFlag) << endl;
    }

    //-----------------------------------------------------------------------------
    // DDS（エンコード済みまたはミップマップあり）への変換なら終了します。
    if (m_ConvertsToDds)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // 初期スウィズル値を決定します。
    m_DstInitialSwizzle = 0;
    bool initialSwizzleSpecified = false;
    if (m_Opt.m_InitialSwizzle != FtxOpt::SWIZZLE_DEFAULT)
    {
        m_DstInitialSwizzle = m_Opt.m_InitialSwizzle;
        initialSwizzleSpecified = true;
    }
    else if (m_MergeInfo.m_IsEnable && m_MergeInfo.m_InitialSwizzle != FtxOpt::SWIZZLE_DEFAULT)
    {
        m_DstInitialSwizzle = m_MergeInfo.m_InitialSwizzle;
        initialSwizzleSpecified = true;
    }
    else if (pFirstOrgImg != nullptr)
    {
        if (pFirstOrgImg->m_InitialSwizzle != FtxOpt::SWIZZLE_DEFAULT)
        {
            m_DstInitialSwizzle = pFirstOrgImg->m_InitialSwizzle;
            initialSwizzleSpecified = true;
        }
    }

    if (!m_Opt.m_IsSilent && initialSwizzleSpecified)
    {
        cout << "Swizzle : " << m_DstInitialSwizzle << endl;
    }

    //-----------------------------------------------------------------------------
    // ユーザーデータを決定します。
    m_DstUserDatas = (m_MergeInfo.m_IsEnable) ?
        m_MergeInfo.m_UserDatas : m_WorkImage.GetUserDatas();

    //-----------------------------------------------------------------------------
    // 編集用コメントを決定します。
    m_DstCommentLabel.clear();
    if (!m_WorkImage.GetCommentLabel().empty())
    {
        // 編集用コメントラベルはマージファイルより入力ファイルの値を優先します。
        m_DstCommentLabel = m_WorkImage.GetCommentLabel();
    }
    else if (m_MergeInfo.m_IsEnable && !m_MergeInfo.m_CommentLabel.empty())
    {
        m_DstCommentLabel = m_MergeInfo.m_CommentLabel;
    }

    m_DstCommentCol.clear();
    if (!m_WorkImage.GetCommentCol().empty())
    {
        // 編集用コメントカラー文字列はマージファイルより入力ファイルの値を優先します。
        m_DstCommentCol = m_WorkImage.GetCommentCol();
    }
    else if (m_MergeInfo.m_IsEnable && !m_MergeInfo.m_CommentCol.empty())
    {
        m_DstCommentCol = m_MergeInfo.m_CommentCol;
    }

    m_DstCommentText.clear();
    bool commentTextSpecified = false;
    if (m_Opt.m_CommentText != FtxOpt::COMMENT_TEXT_DEFAULT)
    {
        m_DstCommentText = m_Opt.m_CommentText;
        commentTextSpecified = true;
    }
    else if (!m_WorkImage.GetCommentText().empty())
    {
        // 編集用コメント文はマージファイルより入力ファイルの値を優先します。
        m_DstCommentText = m_WorkImage.GetCommentText();
        commentTextSpecified = true;
    }
    else if (m_MergeInfo.m_IsEnable && !m_MergeInfo.m_CommentText.empty())
    {
        m_DstCommentText = m_MergeInfo.m_CommentText;
        commentTextSpecified = true;
    }
    if (!m_Opt.m_IsSilent && commentTextSpecified)
    {
        cout << "Comment : " << m_DstCommentText << endl;
    }

    //-----------------------------------------------------------------------------
    // ツールデータ、ユーザーツールデータを決定します。
    m_DstToolData = (m_MergeInfo.m_IsEnable) ?
        m_MergeInfo.m_ToolData : m_WorkImage.GetToolData();

    m_DstUserToolData = (m_MergeInfo.m_IsEnable) ?
        m_MergeInfo.m_UserToolData : m_WorkImage.GetUserToolData();

    //-----------------------------------------------------------------------------
    // オリジナル画像ファイルの情報を表示します。
    if (!m_Opt.m_IsSilent && VERBOSE_LEVEL >= 2)
    {
        m_WorkImage.DisplayInfo(cout);
    }

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief エンコードされたデータをリードします。
//-----------------------------------------------------------------------------
RStatus TexConverterBase::ReadEncodedData()
{
    RStatus status;

    if (!m_Opt.m_IsSilent)
    {
        cout << "No Encoding" << endl;
    }

    //-----------------------------------------------------------------------------
    // 入力ファイルについてループします。
    RImage& workImage = m_WorkImage;
    workImage.SetFilePath(m_Opt.m_InputPaths[0]);
    bool getsOriginal = true;
    uint8_t* pDstImg = nullptr;
    int dstZ = 0;
    int curSlice = 0;
    const int fileCount = static_cast<int>(m_Opt.m_InputPaths.size());
    for (int fileIdx = 0; fileIdx < fileCount; ++fileIdx)
    {
        //-----------------------------------------------------------------------------
        // 拡張子をチェックします。
        const std::string& inputPath = m_Opt.m_InputPaths[fileIdx];
        const std::string ext = RGetExtensionFromFilePath(inputPath);
        if (ext != FtxaExtension &&
            ext != FtxbExtension &&
            ext != DdsExtension)
        {
            return RStatus(RStatus::FAILURE, "Input file is wrong for no encoding: " + inputPath); // RShowError
        }

        //-----------------------------------------------------------------------------
        // 元画像とエンコードされたデータの両方をリードします。
        RImage rimg;
        const int readFlags = (getsOriginal) ?
            RImage::ReadFlag_Both : RImage::ReadFlag_Encoded;
        status = rimg.ReadFile(inputPath, readFlags, false, m_ConverterUtility);
        RCheckStatus(status);

        //-----------------------------------------------------------------------------
        // ftx ファイルに元画像データがない場合、
        // --disable-original-image が指定されているか最初の入力ファイルなら、
        // 元画像データを取得しないようにします。
        if (getsOriginal && rimg.IsDecodedOriginal())
        {
            if (m_Opt.m_DisablesOriginalImage || fileIdx == 0)
            {
                getsOriginal = false;
                m_OriginalImages.clear();
            }
        }

        //-----------------------------------------------------------------------------
        // 成分選択オプションを反映します。
        if (m_Opt.m_CompSel != FtxOpt::COMP_SEL_DEFAULT)
        {
            rimg.SetCompSel(m_Opt.m_CompSel);
        }

        //-----------------------------------------------------------------------------
        // 入力ファイルが ftx ファイルでなければ、ヒント情報、コメントなどの
        // テクスチャーデータに影響しないオプションを反映します。
        if (ext != FtxaExtension &&
            ext != FtxbExtension)
        {
            if (m_Opt.m_DccPreset != FtxOpt::DCC_PRESET_DEFAULT)
            {
                rimg.SetDccPreset(m_Opt.m_DccPreset);
            }

            if (m_Opt.m_Hint != FtxOpt::HINT_DEFAULT)
            {
                rimg.SetHint(m_Opt.m_Hint);
            }

            if (m_Opt.m_InitialSwizzle != FtxOpt::SWIZZLE_DEFAULT)
            {
                rimg.SetInitialSwizzle(m_Opt.m_InitialSwizzle);
            }

            if (m_Opt.m_CommentText != FtxOpt::COMMENT_TEXT_DEFAULT)
            {
                rimg.SetCommentText(m_Opt.m_CommentText);
            }
        }

        //-----------------------------------------------------------------------------
        // 整数型と実数型の元画像が混在しないように最初のファイルの型で統一します。
        if (fileIdx >= 1 && getsOriginal)
        {
            const int formatBak = rimg.GetFormat();
            rimg.ChangeDataType(workImage.IsFloat(), false);
            rimg.SetFormat(formatBak);
        }

        //-----------------------------------------------------------------------------
        // 元画像を追加します。
        if (getsOriginal)
        {
            const size_t orgColBytes = (rimg.IsFloat()) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
            const size_t orgFaceBytes = orgColBytes * rimg.GetImageW() * rimg.GetImageH();
            const uint8_t* pOrgSrc = rimg.GetImageData();
            for (int iz = 0; iz < rimg.GetImageD(); ++iz)
            {
                int iSlice = curSlice + iz;
                int iCubeFace = ROriginalImage::NONE;
                if (rimg.IsCubeMap())
                {
                    iCubeFace = ROriginalImage::POSITIVE_X + (iz % RImage::CUBE_FACE_COUNT);
                    iSlice = curSlice + (iz / RImage::CUBE_FACE_COUNT);
                }
                std::string originalPath = (!rimg.GetOriginalPaths().empty()) ?
                    rimg.GetOriginalFullPath(iz) :
                    rimg.GetFilePath();
                originalPath = GetOriginalFilePath(m_DstFinalPath, originalPath);
                m_OriginalImages.push_back(ROriginalImage(
                    iSlice,
                    static_cast<ROriginalImage::Face>(iCubeFace),
                    rimg.GetImageW(), rimg.GetImageH(),
                    originalPath.c_str(), pOrgSrc, &rimg));
                pOrgSrc += orgFaceBytes;
            }
        }

        //-----------------------------------------------------------------------------
        // 1D / 1D 配列なら高さを 1 に変更します（リード直後は元画像の高さになっています）。
        if (rimg.GetDimension() == FtxDimension_1d      ||
            rimg.GetDimension() == FtxDimension_1dArray)
        {
            rimg.SetImageH(1);
        }

        //-----------------------------------------------------------------------------
        // エンコードされたデータをイメージデータに移動します。
        rimg.MoveEncodedDataToImageData();

        //-----------------------------------------------------------------------------
        const bool weightedCompress = (rimg.GetWeightedCompress() != 0);
        if (fileIdx == 0)
        {
            //-----------------------------------------------------------------------------
            // 最初のファイルならパラメータを決定して、イメージデータを確保します。
            m_DstFormat           = static_cast<FtxFormat>(rimg.GetFormat());
            m_DstMipCount         = rimg.GetMipCount();
            m_DstMipGenFilter     = rimg.GetMipGenFilter();
            m_DstCompSel          = rimg.GetCompSel();
            m_DstWeightedCompress = weightedCompress;
            m_DstDccPreset        = rimg.GetDccPreset();
            m_DstHint             = rimg.GetHint();
            m_DstLinearFlag       = rimg.GetLinearFlag();
            m_DstInitialSwizzle   = rimg.GetInitialSwizzle();
            m_DstUserDatas        = rimg.GetUserDatas();
            m_DstCommentLabel     = rimg.GetCommentLabel();
            m_DstCommentCol       = rimg.GetCommentCol();
            m_DstCommentText      = rimg.GetCommentText();
            m_DstToolData         = rimg.GetToolData();
            m_DstUserToolData     = rimg.GetUserToolData();
            workImage.SetHasAlpha(rimg.HasAlpha());
            workImage.SetIsFloat(rimg.IsFloat());

            workImage.SetDimension(rimg.GetDimension());
            bool isDimValid = true;
            if (m_Opt.m_Dimension == FtxDimension_1dArray)
            {
                if (workImage.GetDimension() == FtxDimension_1d)
                {
                    workImage.SetDimension(FtxDimension_1dArray);
                }
                else
                {
                    isDimValid = false;
                }
            }
            else if (m_Opt.m_Dimension == FtxDimension_2dArray)
            {
                if (workImage.GetDimension() == FtxDimension_2d)
                {
                    workImage.SetDimension(FtxDimension_2dArray);
                }
                else
                {
                    isDimValid = false;
                }
            }
            else if (m_Opt.m_Dimension == FtxDimension_CubeMapArray)
            {
                if (workImage.IsCubeMap())
                {
                    if (rimg.GetImageD() != RImage::CUBE_FACE_COUNT)
                    {
                        return RStatus(RStatus::FAILURE, "Depth is wrong for no encoding: " + inputPath); // RShowError
                    }
                    workImage.SetDimension(FtxDimension_CubeMapArray);
                }
                else
                {
                    isDimValid = false;
                }
            }
            else if (fileCount >= 2 || m_Opt.m_Depth != FtxOpt::DEPTH_DEFAULT)
            {
                if      (workImage.GetDimension() == FtxDimension_1d)
                {
                    workImage.SetDimension(FtxDimension_1dArray);
                }
                else if (workImage.GetDimension() == FtxDimension_2d)
                {
                    workImage.SetDimension(FtxDimension_2dArray);
                }
                else if (workImage.IsCubeMap())
                {
                    if (rimg.GetImageD() != RImage::CUBE_FACE_COUNT)
                    {
                        return RStatus(RStatus::FAILURE, "Depth is wrong for no encoding: " + inputPath); // RShowError
                    }
                    workImage.SetDimension(FtxDimension_CubeMapArray);
                }
                else
                {
                    isDimValid = false;
                }
            }
            if (!isDimValid)
            {
                return RStatus(RStatus::FAILURE, "Dimension is wrong for no encoding: " + inputPath); // RShowError
            }

            workImage.SetImageW(rimg.GetImageW());
            workImage.SetImageH(rimg.GetImageH());
            workImage.SetImageD(rimg.GetImageD() * fileCount);
            workImage.SetMipCount(rimg.GetMipCount());
            workImage.SetFormat(rimg.GetFormat());
            workImage.SetImageDataSize(rimg.GetImageDataSize() * fileCount);
            workImage.SetMipDataSize(rimg.GetMipDataSize() * fileCount);
            workImage.ClearLevelOffsets();
            for (int level = 0; level < workImage.GetMipCount(); ++level)
            {
                workImage.SetLevelOffset(level, rimg.GetLevelOffset(level) * fileCount);
            }
            if (fileCount == 1)
            {
                workImage.SetOriginalImageHash(rimg.GetOriginalImageHash());
            }
            pDstImg = new uint8_t[workImage.GetImageDataSize()];
            workImage.SetImageData(pDstImg);
            workImage.SetIsDecodedOriginal(rimg.IsDecodedOriginal());
        }
        else
        {
            //-----------------------------------------------------------------------------
            // 2 番目以降のファイルならパラメータが同じかチェックします。
            if ((workImage.GetDimension() == FtxDimension_1dArray && rimg.GetDimension() != FtxDimension_1d) ||
                (workImage.GetDimension() == FtxDimension_2dArray && rimg.GetDimension() != FtxDimension_2d) ||
                (workImage.IsCubeMap() && !rimg.IsCubeMap()))
            {
                return RStatus(RStatus::FAILURE, "Dimension is wrong for no encoding: " + inputPath); // RShowError
            }
            else if (rimg.GetImageW() != workImage.GetImageW() ||
                     rimg.GetImageH() != workImage.GetImageH())
            {
                return RStatus(RStatus::FAILURE, "Width or height is not identical: " + inputPath); // RShowError
            }
            else if (rimg.IsCubeMap() && rimg.GetImageD() != RImage::CUBE_FACE_COUNT)
            {
                return RStatus(RStatus::FAILURE, "Depth is wrong for no encoding: " + inputPath); // RShowError
            }
            else if (rimg.GetMipCount() != workImage.GetMipCount())
            {
                return RStatus(RStatus::FAILURE, "Mipmap level is not identical: " + inputPath); // RShowError
            }
            else if (rimg.GetFormat() != workImage.GetFormat())
            {
                return RStatus(RStatus::FAILURE, "Format is not identical: " + inputPath); // RShowError
            }
            else if (static_cast<FtxCompSel>(rimg.GetCompSel()) != m_DstCompSel)
            {
                return RStatus(RStatus::FAILURE, "Component selection is not identical: " + inputPath); // RShowError
            }
            //else if (weightedCompress != m_DstWeightedCompress)
            //{
            //  return RStatus(RStatus::FAILURE, "Weighted compress is not identical: " + inputPath); // RShowError
            //}
            else if (rimg.GetHint() != m_DstHint)
            {
                return RStatus(RStatus::FAILURE, "Hint is not identical: " + inputPath); // RShowError
            }
            else if (rimg.GetLinearFlag() != m_DstLinearFlag)
            {
                return RStatus(RStatus::FAILURE, "Linear is not identical: " + inputPath); // RShowError
            }
        }

        //-----------------------------------------------------------------------------
        // エンコードされたデータを追加します。
        const size_t level0Size = rimg.GetImageDataSize() - rimg.GetMipDataSize();
        memcpy(pDstImg, rimg.GetImageData(), level0Size);
        pDstImg += level0Size;

        if (workImage.GetMipCount() >= 2)
        {
            for (int level = 1; level < rimg.GetMipCount(); ++level)
            {
                const int levelD = (rimg.GetDimension() == FtxDimension_3d) ?
                    RMax(rimg.GetImageD() >> level, 1) : rimg.GetImageD();
                size_t srcFaceBytes;
                size_t dstFaceBytes;
                const void* pSrcLv = GetLevelImageData(&srcFaceBytes, rimg     , level, 0   );
                void* pDstLv       = GetLevelImageData(&dstFaceBytes, workImage, level, dstZ);
                memcpy(pDstLv, pSrcLv, srcFaceBytes * levelD);
            }
        }

        //-----------------------------------------------------------------------------
        // スライス番号をインクリメントします。
        if (rimg.GetDimension() != FtxDimension_3d) // 複数の 3D テクスチャーの結合には非対応です。
        {
            dstZ += rimg.GetImageD();
            curSlice += (workImage.IsCubeMap()) ?
                rimg.GetImageD() / RImage::CUBE_FACE_COUNT : rimg.GetImageD();
        }
    }

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 入力画像を切り抜きます。
//-----------------------------------------------------------------------------
RStatus TexConverterBase::CropInputImage()
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // 切り抜きが指定されていなければ何もしません。
    RCropRect cropRect = m_Opt.m_CropRect;
    if (!cropRect.IsEnabled())
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // 切り抜き矩形が画像の範囲内に収まるように調整します。
    cropRect.AdjustToImage(m_WorkImage.GetImageW(), m_WorkImage.GetImageH());
    if (!m_Opt.m_IsSilent)
    {
        cout << "Crop    : " << cropRect.m_X << " " << cropRect.m_Y << " "
             << cropRect.m_W << " " << cropRect.m_H << endl;
    }

    //-----------------------------------------------------------------------------
    // 切り抜き矩形が画像全体なら何もしません。
    if (cropRect.IsEntireImage(m_WorkImage.GetImageW(), m_WorkImage.GetImageH()))
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // 切り抜きます。
    status = m_WorkImage.Crop(cropRect);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // 画像から元画像配列を再設定します。
    UpdateOriginalImagesFromImage(m_OriginalImages, m_WorkImage);

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 入力画像をサイズ変更します。
//-----------------------------------------------------------------------------
RStatus TexConverterBase::ResizeInputImage(int* pLinearFlagForResize)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // サイズ変更に使用するリニアー変換フラグを決定します。
    FtxFormat format = FtxFormat_Unorm_8_8_8_8;
    DecideFormat(&format, m_Opt, m_MergeInfo, m_OriginalImages, nullptr);
    const int linearFlag = (m_Opt.m_ResizesInLinear) ?
        DecideLinearFlag(nullptr, m_Opt, m_MergeInfo, m_OriginalImages, nullptr, format) :
        FtxOpt::LINEAR_NONE;
    *pLinearFlagForResize = linearFlag;

    //-----------------------------------------------------------------------------
    // サイズ変更が指定されていないか、サイズ変更後のサイズが同じなら何もしません。
    int dstW;
    int dstH;
    int dstD;
    if (!m_WorkImage.GetResizedWhd(&dstW, &dstH, &dstD,
        m_Opt.m_ReductionLevel, m_Opt.m_ResizeW, m_Opt.m_ResizeH,
        m_Opt.m_ResizePercentW, m_Opt.m_ResizePercentH))
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // サイズ変更情報を表示します。
    if (!m_Opt.m_IsSilent)
    {
        const ResizeFilter resizeFilter = (m_Opt.m_ResizeFilter != ResizeFilter_Default) ?
            m_Opt.m_ResizeFilter : ResizeFilter_Linear;
        const std::string resizeFilterStr = "(" +
            RGetStringFromValue(g_ResizeFilterStringValues, resizeFilter) + ")";
        if (m_Opt.m_ResizeW        != 0    ||
            m_Opt.m_ResizeH        != 0    ||
            m_Opt.m_ResizePercentW != 0.0f ||
            m_Opt.m_ResizePercentH != 0.0f)
        {
            cout << "Resize  : " << dstW << " x " << dstH << " " << resizeFilterStr << endl;
        }
        else
        {
            cout << "Reduce  : " << m_Opt.m_ReductionLevel << " " << resizeFilterStr << endl;
        }
    }

    //-----------------------------------------------------------------------------
    // サイズ変更します。
    status = m_WorkImage.Resize(dstW, dstH, dstD,
        linearFlag, m_Opt.m_ResizeFilter, m_Opt.m_AdjustsXpaRgb);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // 画像から元画像配列を再設定します。
    UpdateOriginalImagesFromImage(m_OriginalImages, m_WorkImage);

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 入力画像の透明ピクセルの RGB 成分を調整します。
//-----------------------------------------------------------------------------
RStatus TexConverterBase::AdjustTransparentRgb()
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // 調整が指定されていなければ何もしません。
    if (!m_Opt.m_AdjustsXpaRgb)
    {
        return status;
    }

    //-----------------------------------------------------------------------------
    // 各フェースを調整します。
    if (!m_Opt.m_IsSilent)
    {
        cout << "Adjust  : transparent rgb" << endl;
    }
    RImage& image = m_WorkImage;
    const bool isFloat = image.IsFloat();
    const size_t colBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t faceBytes = colBytes * image.GetImageW() * image.GetImageH();
    for (int iDepth = 0; iDepth < image.GetImageD(); ++iDepth)
    {
        uint8_t* pDst = image.GetImageData() + iDepth * faceBytes;
        if (!isFloat)
        {
            RAdjustTransparentRgbFace(pDst, image.GetImageW(), image.GetImageH());
        }
        else
        {
            float* pDstF32 = reinterpret_cast<float*>(pDst);
            RAdjustTransparentRgbFace(pDstF32, image.GetImageW(), image.GetImageH());
        }
    }

    //-----------------------------------------------------------------------------
    // 元画像の追加データに格納されたミップマップデータは調整しないので、
    // ミップマップを再計算するためにクリアします。
    image.SetExtraDataSize(0);

    //-----------------------------------------------------------------------------
    // 画像から元画像配列を再設定します。
    UpdateOriginalImagesFromImage(m_OriginalImages, image);

    return status;
}

//-----------------------------------------------------------------------------
//! @brief 入力画像を編集します。
//-----------------------------------------------------------------------------
RStatus TexConverterBase::EditInputImage()
{
    //-----------------------------------------------------------------------------
    // 切り抜きます。
    RStatus status = CropInputImage();
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // サイズ変更します。
    int linearFlagForResize = FtxOpt::LINEAR_NONE;
    status = ResizeInputImage(&linearFlagForResize);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // サイズ変更後のチャンネル差し替えを実行します。
    if (m_Opt.IsAfterReplaceSpecified())
    {
        status = ReplaceChannels(&m_WorkImage, m_Opt.m_AfterReplacePaths,
            m_Opt, true, true, linearFlagForResize, m_ConverterUtility);
        RCheckStatus(status);
        UpdateOriginalImagesFromImage(m_OriginalImages, m_WorkImage);
    }

    //-----------------------------------------------------------------------------
    // 透明ピクセルの RGB 成分を調整します。
    return AdjustTransparentRgb();
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターに入力ファイルをリードします。
//-----------------------------------------------------------------------------
bool TexConverterBase::ReadInputFile(const wchar_t* paths[], const wchar_t* options[])
{
    RStatus status = PrepareToRead(paths, options);
    if (status)
    {
        if (!m_Opt.m_OutputsBntx || m_Opt.m_IsImageToBntx)
        {
            if (m_Opt.m_NoEncoding)
            {
                status = ReadEncodedData();
            }
            else
            {
                status = ReadImageFile(m_WorkImage, m_OriginalImages, &m_Opt,
                    m_MergeInfo, m_DstFinalPath, m_ConverterUtility);
                if (status)
                {
                    status = EditInputImage();
                }
            }
            if (status)
            {
                status = DecideDstParam();
            }
        }
    }
    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターにメモリー上の ftx ファイルのデータをリードします。
//-----------------------------------------------------------------------------
bool TexConverterBase::ReadFtxData(
    const void* pData,
    const size_t dataSize,
    const wchar_t* path,
    const wchar_t* options[]
)
{
    const wchar_t* paths[] = { path, nullptr };
    RStatus status = PrepareToRead(paths, options);
    if (status)
    {
        if (m_Opt.m_OutputsBntx)
        {
            m_Opt.m_IsImageToBntx = false;
            status = m_WorkImage.DecodeFtx(&m_Encoder,
                reinterpret_cast<const uint8_t*>(pData), dataSize,
                IsFtxBinaryFilePath(m_Opt.m_InputPaths[0]), RImage::ReadFlag_Encoded);
        }
        else
        {
            status = ParseFtxData(m_WorkImage, m_OriginalImages, &m_Encoder,
                pData, dataSize, m_Opt, m_MergeInfo, m_DstFinalPath);
            if (status)
            {
                status = EditInputImage();
            }
            if (status)
            {
                status = DecideDstParam();
            }
        }
    }
    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターにメモリー上のメモリー上のビットマップデータをリードします。
//-----------------------------------------------------------------------------
bool TexConverterBase::ReadBitmapData(
    const void* pData,
    const int width,
    const int height,
    const int depth,
    const bool hasAlpha,
    const bool isFloat,
    const wchar_t* paths[],
    const wchar_t* originalPaths[],
    const wchar_t* options[]
)
{
    RStatus status = PrepareToRead(paths, options);
    if (status)
    {
        if (m_Opt.m_OutputsBntx)
        {
            m_Opt.m_IsImageToBntx = true;
        }
        status = ParseBitmapData(m_WorkImage, m_OriginalImages,
            pData, width, height, depth, hasAlpha, isFloat,
            m_Opt, m_MergeInfo, m_DstFinalPath, originalPaths);
        if (status)
        {
            status = EditInputImage();
        }
        if (status)
        {
            status = DecideDstParam();
        }
    }
    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief 元画像の画像ファイルを出力します。
//-----------------------------------------------------------------------------
RStatus TexConverterBase::OutputOriginalImageFile(const std::string& originalImagePath)
{
    const std::string ext = RGetExtensionFromFilePath(originalImagePath);
    if (ext == TgaExtension)
    {
        if (m_WorkImage.IsFloat())
        {
            return RStatus(RStatus::FAILURE, "Cannot convert the real number data to TGA: " + originalImagePath); // RShowError
        }
        return m_WorkImage.OutputTgaFile(originalImagePath, m_Opt.m_IsTgaRle, false, false);
    }
    else if (ext == PngExtension)
    {
        return m_WorkImage.OutputPngFile(&m_WicUtility, originalImagePath, false);
    }
    else if (ext == DdsExtension)
    {
        return m_WorkImage.OutputDdsFile(originalImagePath,
            false, false, false, m_Opt.m_ForcesDdsDx10);
    }
    else if (ext == ExrExtension)
    {
        if (!m_WorkImage.IsFloat())
        {
            return RStatus(RStatus::FAILURE, "Cannot convert the integer data to EXR: " + originalImagePath); // RShowError
        }
        return m_WorkImage.OutuptExrFile(&m_OpenExr, originalImagePath, false);
    }
    else
    {
        return RStatus(RStatus::FAILURE, "Original file type is wrong: " + originalImagePath); // RShowError
    }
}

//-----------------------------------------------------------------------------
//! @brief 変換します。
//-----------------------------------------------------------------------------
RStatus TexConverterBase::ConvertSub(const bool isToFile)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // 変換せずに bntx ファイルを出力します。
    std::string* pOutputData = (isToFile) ? nullptr : &m_DstData;
    if (m_Opt.m_OutputsBntx && !m_Opt.m_IsImageToBntx)
    {
        if (m_WorkImage.GetImageData() == nullptr)
        {
            return OutputBntxFromFtxDdsFiles(pOutputData, m_DstPath, m_Opt, m_ConverterUtility);
        }
        else
        {
            return OutputBntxFromWorkImage(pOutputData, &m_WorkImage, m_DstPath, m_Opt, m_ConverterUtility);
        }
    }

    //-----------------------------------------------------------------------------
    // 画像ファイルがリードされているかチェックします。
    if (m_WorkImage.GetImageData() == nullptr)
    {
        return RStatus(RStatus::FAILURE, "Original image is not read"); // RShowError
    }

    //-----------------------------------------------------------------------------
    // 元画像の画像ファイルを出力します。
    if (!m_Opt.m_OutputOriginalPath.empty())
    {
        status = OutputOriginalImageFile(m_Opt.m_OutputOriginalPath);
        RCheckStatus(status);
    }

    //-----------------------------------------------------------------------------
    // TGA、PNG、DDS（再エンコードなしまたはミップマップなし）、EXR ファイルに変換します。
    if (m_ConvertsToTga)
    {
        return m_WorkImage.OutputTgaFile(m_DstPath, m_Opt.m_IsTgaRle, false, false);
    }
    else if (m_ConvertsToPng)
    {
        return m_WorkImage.OutputPngFile(&m_WicUtility, m_DstPath, false);
    }
    else if (m_ConvertsToDds)
    {
        if (!m_ConvertsToEncodedDds && !m_ConvertsToMipDds)
        {
            return m_WorkImage.OutputDdsFile(m_DstPath,
                m_Opt.m_NoEncoding, m_Opt.m_NoEncoding, false, m_Opt.m_ForcesDdsDx10);
        }
    }
    else if (m_ConvertsToExr)
    {
        return m_WorkImage.OutuptExrFile(&m_OpenExr, m_DstPath, false);
    }

    //-----------------------------------------------------------------------------
    // 幅／高さ／奥行きをチェックします。
    if (!m_Opt.m_DisablesSizeError)
    {
        if (m_WorkImage.GetImageW() > RImage::WidthHeightMax ||
            m_WorkImage.GetImageH() > RImage::WidthHeightMax ||
            m_WorkImage.GetImageD() > RImage::WidthHeightMax)
        {
            return RStatus(RStatus::FAILURE, "Image size is large"); // RShowError
        }
    }

    //-----------------------------------------------------------------------------
    // フォーマットをチェックします。
    if (!m_Opt.m_OutputsBntx && !m_ConvertsToDds)
    {
        // 以下のフォーマットは現在中間ファイル仕様が未対応です。
        if (m_DstFormat == FtxFormat_Uint_32_32_32  ||
            m_DstFormat == FtxFormat_Sint_32_32_32  ||
            m_DstFormat == FtxFormat_Float_32_32_32)
        {
            return RStatus(RStatus::FAILURE, "Format is wrong: " + RGetTextureFormatString(m_DstFormat)); // RShowError
        }
    }

    //-----------------------------------------------------------------------------
    // ミップマップテクスチャーを作成します。
    if (!m_Opt.m_NoEncoding && m_DstMipCount >= 2 && m_WorkImage.GetMipCount() <= 1)
    {
        //-----------------------------------------------------------------------------
        // 変換オプションでミップマップのレベル数が指定されていれば、
        // 元画像の追加データに格納されたミップマップデータをクリアします。
        if (m_Opt.m_MipCount != FtxOpt::MipCountDefault ||
            (m_MergeInfo.m_IsEnable && m_MergeInfo.m_MipCount != FtxOpt::MipCountDefault))
        {
            m_WorkImage.SetExtraDataSize(0);
        }

        //-----------------------------------------------------------------------------
        // ミップマップテクスチャーを作成します。
        m_WorkImage.SetName("Mipmapped");
        const ROriginalImage& firstOrgImg = m_OriginalImages[0];
        status = CreateMipmap(&m_WorkImage,
            m_DstFormat, m_DstMipCount, m_DstMipGenFilter, m_DstLinearFlag,
            firstOrgImg.m_MipCount, m_Opt, m_ConverterUtility);
        RCheckStatus(status);

        if (!m_Opt.m_IsSilent && VERBOSE_LEVEL >= 2 && m_WorkImage.GetImageData() != nullptr)
        {
            m_WorkImage.DisplayInfo(cout);
        }
    }

    //-----------------------------------------------------------------------------
    // 以後追加データは使用しないのでクリアします。
    m_WorkImage.SetExtraDataSize(0);

    //-----------------------------------------------------------------------------
    // DDS ファイル（エンコードなし。ミップマップあり）に変換します。
    if (m_ConvertsToMipDds)
    {
        return m_WorkImage.OutputDdsFile(m_DstPath, false, true, false, m_Opt.m_ForcesDdsDx10);
    }

    //-----------------------------------------------------------------------------
    // フォーマットを変換します。
    if (!m_Opt.m_NoEncoding)
    {
        //cerr << "Converting format ..." << endl;
        m_WorkImage.SetName("Converted");
        status = ConvertFormat(&m_WorkImage, m_Encoder,
            m_DstFormat, m_DstWeightedCompress, m_DstHint, m_DstLinearFlag, m_Opt);
        RCheckStatus(status);
        if (!m_Opt.m_IsSilent && VERBOSE_LEVEL >= 2)
        {
            m_WorkImage.DisplayInfo(cout);
        }
    }

    //-----------------------------------------------------------------------------
    // DDS ファイル（エンコード済み）に変換します。
    if (m_ConvertsToDds)
    {
        return m_WorkImage.OutputDdsFile(m_DstPath, true, true, false, m_Opt.m_ForcesDdsDx10);
    }

    //-----------------------------------------------------------------------------
    // 成分選択、BC 圧縮重み付け、ヒント情報、リニア変換フラグ、初期スウィズル値、
    // ユーザーデータ、編集用コメント、ツールデータ、ユーザーツールデータを設定します。
    m_WorkImage.SetMipGenFilter(m_DstMipGenFilter);
    m_WorkImage.SetCompSel(m_DstCompSel);
    m_WorkImage.SetWeightedCompress(m_DstWeightedCompress ?
        FtxOpt::WEIGHTED_COMPRESS_ENABLE : FtxOpt::WEIGHTED_COMPRESS_DISABLE);
    m_WorkImage.SetDccPreset(m_DstDccPreset);
    m_WorkImage.SetHint(m_DstHint);
    m_WorkImage.SetLinearFlag(m_DstLinearFlag);
    m_WorkImage.SetInitialSwizzle(m_DstInitialSwizzle);
    m_WorkImage.SetUserDatas(m_DstUserDatas);
    m_WorkImage.SetCommentLabel(m_DstCommentLabel);
    m_WorkImage.SetCommentCol(m_DstCommentCol);
    m_WorkImage.SetCommentText(m_DstCommentText);
    m_WorkImage.SetToolData(m_DstToolData);
    m_WorkImage.SetUserToolData(m_DstUserToolData);

    //-----------------------------------------------------------------------------
    // ftx ファイルまたは bntx ファイルを出力します。
    if (!m_Opt.m_IsCSource)
    {
        //cerr << "Writing to file ..." << endl;
        if (!m_Opt.m_OutputsBntx)
        {
            status = m_WorkImage.OutputFtxFile(pOutputData, &m_OriginalImages,
                m_DstPath, m_Opt, GetConverterVersionString(false), true);
        }
        else
        {
            status = OutputBntxFromWorkImage(pOutputData, &m_WorkImage, m_DstPath, m_Opt, m_ConverterUtility);
        }
    }

    //-----------------------------------------------------------------------------
    // C 言語のソース形式でテクスチャーを出力します。
    if (status && isToFile && m_Opt.m_IsCSource)
    {
        OutTextureAsCSource(m_DstPath, m_WorkImage);
    }

    return status;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 変換して出力ファイルをセーブします。
//!        リード時のジョブオプションで指定した出力パスにセーブします。
//-----------------------------------------------------------------------------
bool TexConverterBase::ConvertToFile()
{
    RTimeMeasure tm1;

    RStatus status = ConvertSub(true);

    //if (status && !m_Opt.m_IsSilent) cout << "Time   : " << tm1.GetMilliSec() << "ms" << endl;

    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief 変換して出力ファイルをメモリーに格納します。
//-----------------------------------------------------------------------------
bool TexConverterBase::ConvertToData(const void** ppData, size_t* pDataSize)
{
    RStatus status = ConvertSub(false);
    if (status)
    {
        *ppData    = m_DstData.c_str();
        *pDataSize = m_DstData.size();
    }
    else
    {
        *ppData    = nullptr;
        *pDataSize = 0;
    }
    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターに作成情報を設定します。
//-----------------------------------------------------------------------------
bool TexConverterBase::SetCreateInfo(const wchar_t* createInfo)
{
    RStatus status;
    if (!m_OriginalImages.empty())
    {
        m_OriginalImages[0].m_CreateInfo = RGetShiftJisFromUnicode(createInfo);
    }
    else
    {
        status = RStatus(RStatus::FAILURE, "Original image is not read"); // RShowError;
    }
    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターにユーザーデータを設定します。
//-----------------------------------------------------------------------------
bool TexConverterBase::SetUserData(const void* pPackedUserData, const size_t packedUserDataSize)
{
    RStatus status;
    if (!m_MergeInfo.m_IsEnable)
    {
        status = UnpackUserDataArray(m_DstUserDatas,
            reinterpret_cast<const uint8_t*>(pPackedUserData),
            packedUserDataSize);
    }
    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターに編集用コメントラベルを設定します。
//-----------------------------------------------------------------------------
bool TexConverterBase::SetCommentLabel(const void* pCommentLabel, const size_t commentLabelSize)
{
    RStatus status;
    if (!m_MergeInfo.m_IsEnable)
    {
        m_DstCommentLabel = std::string(reinterpret_cast<const char*>(pCommentLabel), commentLabelSize);
    }
    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターに編集用コメントカラー文字列を設定します。
//-----------------------------------------------------------------------------
bool TexConverterBase::SetCommentColor(const void* pCommentCol, const size_t commentColSize)
{
    RStatus status;
    if (!m_MergeInfo.m_IsEnable)
    {
        m_DstCommentCol = std::string(reinterpret_cast<const char*>(pCommentCol), commentColSize);
    }
    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターにツールデータを設定します。
//-----------------------------------------------------------------------------
bool TexConverterBase::SetToolData(const void* pToolData, const size_t toolDataSize)
{
    RStatus status;
    if (!m_MergeInfo.m_IsEnable)
    {
        m_DstToolData = std::string(reinterpret_cast<const char*>(pToolData), toolDataSize);
    }
    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターにユーザーツールデータを設定します。
//-----------------------------------------------------------------------------
bool TexConverterBase::SetUserToolData(const void* pUserToolData, const size_t userToolDataSize)
{
    RStatus status;
    if (!m_MergeInfo.m_IsEnable)
    {
        m_DstUserToolData = std::string(reinterpret_cast<const char*>(pUserToolData), userToolDataSize);
    }
    return ParseStatus(status);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターのバージョンを整数値で返します。
//-----------------------------------------------------------------------------
int TexConverterBase::GetCvtrVersion() const
{
    return (m_VersionMajor  << 24) |
           (m_VersionMinor  << 16) |
           (m_VersionMicro  <<  8) |
           (m_VersionBugfix      );
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターのバージョン文字列を返します。
//-----------------------------------------------------------------------------
std::string TexConverterBase::GetConverterVersionString(const bool includesBugFix) const
{
    const std::string bugFix = (includesBugFix) ?
        "." + RGetNumberString(m_VersionBugfix) : "";
    return
        RGetNumberString(m_VersionMajor) + "." +
        RGetNumberString(m_VersionMinor) + "." +
        RGetNumberString(m_VersionMicro) + bugFix;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターの使用方法を表示します。
//-----------------------------------------------------------------------------
void TexConverterBase::ShowUsage(
    std::ostream& os,
    const wchar_t* cvtrName,
    const wchar_t* cmdName,
    const bool isVersionOnly
) const
{
    os << RGetShiftJisFromUnicode(cvtrName) << " "
       << GetConverterVersionString(true) << endl;
    if (isVersionOnly)
    {
        os << endl;
        os << "Copyright (C)Nintendo All rights reserved." << endl;
        os << endl;
        os << "These coded instructions, statements, and computer programs contain" << endl;
        os << "proprietary information of Nintendo of America Inc. and/or Nintendo" << endl;
        os << "Company Ltd., and are protected by Federal copyright law.  They may" << endl;
        os << "not be disclosed to third parties or copied or duplicated in any form," << endl;
        os << "in whole or in part, without the prior written consent of Nintendo." << endl;
        return;
    }
    os << "Usage: " << RGetShiftJisFromUnicode(cmdName) << " [INPUT_FILE] [Options]" << endl;
    os << endl;
    os << "Options:" << endl;
    os << "  --version                     Display the version information and exit." << endl;
    os << "  -h --help                     Display this help and exit." << endl;
    os << "  -s --silent                   Silent mode." << endl;
    os << "  --args-file=ARGS_FILE         Specify the arguments file." << endl;
    os << "  --job-list=JOB_LIST_FILE      Specify the job list file." << endl;
    if (m_SupportsFtx)
    {
        os << "  --project-root=ROOT_PATH      Specify the project root path." << endl;
        os << "  --disable-file-info           Disable updating <file_info>." << endl;
        os << "  --disable-original-image      Disable outputting original images." << endl;
    }
    os << "  --disable-memory-pool         Disable the memory pool in bntx." << endl;
    os << "  --tile-mode=TILE_MODE         Specify the tiling mode." << endl;
    os << "  --tile-optimize=TILE_OPTIMIZE Specify the tiling optimization." << endl;
    os << "  --tile-size-threshold=VALUE   Specify the threshold for optimization." << endl;
    os << "  --sparse-tiled=BOOL           Specify true to use a sparse tiling." << endl;
    os << "  -o --output=OUTPUT_FILE       Specify the output file." << endl;
    if (m_SupportsFtx)
    {
        os << "                                (ftxb / TGA / PNG / DDS / EXR / bntx)" << endl;
    }
    else
    {
        os << "                                (TGA / PNG / DDS / EXR / bntx)" << endl;
    }
    if (m_SupportsFtx)
    {
        os << "  -m --merge=MERGE_FILE         Specify the ftx file to merge." << endl;
        os << "  --disable-merge-error         Disable the no merge file error." << endl;
        os << "  --dcc-preset=PRESET           Specify the DCC preset name." << endl;
    }
    os << "  -n --hint=HINT                Specify the hint." << endl;
    os << "  -l --linear=LINEAR            Specify the linear conversion flags." << endl;
    os << "                                like 1110 (linear RGB), 1111 (linear RGBA)" << endl;
    os << "  -d --dimension=DIMENSION      Specify the dimension." << endl;
    os << "  -f --format=FORMAT            Specify the format." << endl;
    os << "  -w --weighted-compress=BOOL   Specify the weighted compress." << endl;
    os << "  -t --depth=DEPTH              Specify the depth for 3d & array texture." << endl;
    os << "  --no-encoding                 Uses the encoded data of the input file." << endl;
    os << "  --gpu-encoding=MODE           Specify true to enable GPU encoding." << endl;
    os << "  --check-gpu-encoding          Check whether GPU encoding is available." << endl;
    os << "  -i --mip-level=LEVEL          Specify the max mipmap levels." << endl;
    os << "  --mip-gen-filter=FILTER       Specify the mipmap generation filter." << endl;
    os << "                                point / linear / cubic" << endl;
    os << "  -c --comp-sel=COMP_SEL        Specify the component selection." << endl;
    os << "  --swizzle=SWIZZLE             Specify the initial swizzle [0,7]." << endl;
    os << "  --cubemap-back=BACK           Specify the cube map back file." << endl;
    os << "  --cubemap-left=LEFT           Specify the cube map left file." << endl;
    os << "  --cubemap-right=RIGHT         Specify the cube map right file." << endl;
    os << "  --cubemap-top=TOP             Specify the cube map top file." << endl;
    os << "  --cubemap-bottom=BOTTOM       Specify the cube map bottom file." << endl;
    os << "  --mip-imageM=FILE             Specify the mipmap image for level M." << endl;
    if (m_SupportsFtx)
    {
        os << "  --comment=COMMENT             Specify the comment text." << endl;
    }
    os << "  --crop=X,Y,W,H                Specify the rectangular region." << endl;
    os << "  -e --reduce-level=LEVEL       Specify the reduction level." << endl;
    os << "  -x --resize-w=WIDTH           Specify the resizing width." << endl;
    os << "  -y --resize-h=HEIGHT          Specify the resizing height." << endl;
    os << "  --resize-filter=FILTER        Specify the resizing filter." << endl;
    os << "  --resize-gamma=BOOL           Specify true to resize in linear." << endl;
    os << "  --ignore-alpha=BOOL           Specify true to ignore the alpha channel" << endl;
    os << "                                of the input file." << endl;
    os << "  -u --adjust-transparent-rgb=BOOL Specify true to adjust RGB components" << endl;
    os << "                                   of transparent pixels." << endl;
    os << "  -r --replace-r=FILE:CHAN      Specify the file for R channel." << endl;
    os << "  -g --replace-g=FILE:CHAN      Specify the file for G channel." << endl;
    os << "  -b --replace-b=FILE:CHAN      Specify the file for B channel." << endl;
    os << "  -a --replace-a=FILE:CHAN      Specify the file for A channel." << endl;
    os << "  --no-tga-rle                  No TGA run length encoding." << endl;
    os << endl;
    os << "Supported Input Files:" << endl;
    if (m_SupportsFtx)
    {
        os << "  TGA / PNG / DDS / OpenEXR / ftxa / ftxb" << endl;
    }
    else
    {
        os << "  TGA / PNG / DDS / OpenEXR" << endl;
    }
    os << endl;
    os << "Supported Dimensions:" << endl;
    os << "  1d / 2d / 3d / cube / 1d_array / 2d_array / cube_array" << endl;
    os << endl;
    os << "Supported Formats:" << endl;
    os << "  unorm_8_8_8_8 / snorm_8_8_8_8 / srgb_8_8_8_8" << endl;
    os << "  unorm_bc1 / unorm_bc2 / unorm_bc3 / unorm_bc4 / unorm_bc5 / unorm_bc7" << endl;
    os << "  srgb_bc1 / srgb_bc2 / srgb_bc3 / snorm_bc4 / snorm_bc5 / srgb_bc7" << endl;
    os << "  unorm_astc_4x4 / unorm_astc_5x4 / unorm_astc_5x5 / unorm_astc_6x5" << endl;
    os << "  unorm_astc_6x6 / unorm_astc_8x5 / unorm_astc_8x6 / unorm_astc_8x8" << endl;
    os << "  unorm_astc_10x5 / unorm_astc_10x6 / unorm_astc_10x8" << endl;
    os << "  unorm_astc_10x10 / unorm_astc_12x10 / unorm_astc_12x12" << endl;
    os << "  srgb_astc_4x4 / srgb_astc_5x4 / srgb_astc_5x5 / srgb_astc_6x5" << endl;
    os << "  srgb_astc_6x6 / srgb_astc_8x5 / srgb_astc_8x6 / srgb_astc_8x8" << endl;
    os << "  srgb_astc_10x5 / srgb_astc_10x6 / srgb_astc_10x8" << endl;
    os << "  srgb_astc_10x10 / srgb_astc_12x10 / srgb_astc_12x12" << endl;
    os << endl;
    os << "Supported Formats (Integer input only):" << endl;
    os << "  unorm_5_6_5 / unorm_5_5_5_1 / unorm_4_4_4_4" << endl;
    os << "  unorm_8 / unorm_4_4 / unorm_8_8" << endl;
    os << "  snorm_8 / snorm_8_8" << endl;
    os << "  uint_8 / uint_8_8 / uint_8_8_8_8" << endl;
    os << "  sint_8 / sint_8_8 / sint_8_8_8_8" << endl;
    os << "  unorm_etc1 / unorm_etc2 / unorm_etc2_mask / unorm_etc2_alpha" << endl;
    os << "  srgb_etc2 / srgb_etc2_mask / srgb_etc2_alpha" << endl;
    os << "  unorm_eac_11 / unorm_eac_11_11 / snorm_eac_11 / snorm_eac_11_11" << endl;
    os << "  unorm_pvrtc1_2bpp / unorm_pvrtc1_4bpp" << endl;
    os << "  srgb_pvrtc1_2bpp / srgb_pvrtc1_4bpp" << endl;
    os << "  unorm_pvrtc1_alpha_2bpp / unorm_pvrtc1_alpha_4bpp" << endl;
    os << "  srgb_pvrtc1_alpha_2bpp / srgb_pvrtc1_alpha_4bpp" << endl;
    os << "  unorm_pvrtc2_alpha_2bpp / unorm_pvrtc2_alpha_4bpp" << endl;
    os << "  srgb_pvrtc2_alpha_2bpp / srgb_pvrtc2_alpha_4bpp" << endl;
    os << endl;
    os << "Supported Formats (Real number input only):" << endl;
    os << "  unorm_16 / unorm_16_16 / unorm_16_16_16_16" << endl;
    os << "  snorm_16 / snorm_16_16 / snorm_16_16_16_16" << endl;
    os << "  uint_16 / uint_16_16 / uint_16_16_16_16" << endl;
    os << "  sint_16 / sint_16_16 / sint_16_16_16_16" << endl;
    os << "  uint_32 / uint_32_32 / uint_32_32_32_32" << endl;
    os << "  sint_32 / sint_32_32 / sint_32_32_32_32" << endl;
    os << "  unorm_10_10_10_2 / uint_10_10_10_2" << endl;
    os << "  float_16 / float_32 / float_16_16 / float_32_32" << endl;
    os << "  float_11_11_10 / float_16_16_16_16 / float_32_32_32_32" << endl;
    os << "  ufloat_bc6 / float_bc6" << endl;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターが現在の環境で GPU によるエンコーディングが可能か判定します。
//-----------------------------------------------------------------------------
bool TexConverterBase::CheckGpuEncoding()
{
    REncoder& encoder = m_Encoder;
    if (!encoder.IsInitialized())
    {
        if (!encoder.Initialize(nullptr))
        {
            return false;
        }
    }
    return encoder.IsGpuEncodingAvailable();
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターに COM のモードを設定します。
//-----------------------------------------------------------------------------
void TexConverterBase::SetComMode(const bool initializesCom, const int coInit)
{
    m_WicUtility.SetComMode(initializesCom, coInit);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターのコンストラクターです。
//-----------------------------------------------------------------------------
TexConverterBase::TexConverterBase(
    HINSTANCE hDll,
    const int versionMajor,
    const int versionMinor,
    const int versionMicro,
    const int versionBugfix,
    const bool supportsFtx
)
:   m_hDLL(hDll),
    m_VersionMajor(versionMajor),
    m_VersionMinor(versionMinor),
    m_VersionMicro(versionMicro),
    m_VersionBugfix(versionBugfix),
    m_SupportsFtx(supportsFtx),
    m_WorkImage("Original")
{
    //cerr << "TexConverterBase()" << endl;

    //-----------------------------------------------------------------------------
    // グローバルオプションを初期化します。
    InitGlobalOptions();

    //-----------------------------------------------------------------------------
    // 変換ユーティリティーに各ユーティリティーへのポインタを設定します。
    m_ConverterUtility.SetEncoder(&m_Encoder);
    m_ConverterUtility.SetWicUtility(&m_WicUtility);
    m_ConverterUtility.SetOpenExr(&m_OpenExr);
    m_ConverterUtility.SetNXUtility(&m_NXUtility);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーコンバーターのデストラクターです。
//-----------------------------------------------------------------------------
TexConverterBase::~TexConverterBase()
{
    //cerr << "~TexConverterBase()" << endl;
}

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

