﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// DecodeDdsT DecodeDdsPixelsT ConvertFromDdsPixels
// GetFormatFromDds GetDimensionFromDds
// OutputDdsFile OutputDdsHeader SetDdsPixels

// DdsHeader

//=============================================================================
// include
//=============================================================================
#include "ImageConvert.h"

using namespace std;

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//=============================================================================
// constants
//=============================================================================
/*
DDS ファイル関連の定数は、下記のヘッダとサイトを参考にしました。

DDSD / DDPF / DDSCAPS / DDSCAPS2
http://dench.flatlib.jp/ddsformat

D3DFORMAT
C:\Program Files (x86)\Windows Kits\8.0\Include\shared\d3d9types.h(1373)

DXGI_FORMAT
C:\Program Files (x86)\Windows Kits\8.0\include\um\dxgiformat.h
https://github.com/SaschaWillems/Vulkan/blob/master/external/gli/gli/dx.hpp

D3D10_RESOURCE_DIMENSION
C:\Program Files (x86)\Windows Kits\8.0\include\um\d3d10.h(695)

D3D10_RESOURCE_MISC_FLAG
C:\Program Files (x86)\Windows Kits\8.0\include\um\d3d10.h(761)
*/

const uint32_t DDSD_CAPS        = 0x00000001; // dwCaps/dwCaps2 が有効
const uint32_t DDSD_HEIGHT      = 0x00000002; // dwHeight が有効
const uint32_t DDSD_WIDTH       = 0x00000004; // dwWidth が有効
const uint32_t DDSD_PITCH       = 0x00000008; // dwPitchOrLinearSize が Pitch を表す
const uint32_t DDSD_PIXELFORMAT = 0x00001000; // dwPfSize/dwPfFlags/dwRGB～ 等の直接定義が有効
const uint32_t DDSD_MIPMAPCOUNT = 0x00020000; // dwMipMapCount が有効
const uint32_t DDSD_LINEARSIZE  = 0x00080000; // dwPitchOrLinearSize が LinearSize を表す
const uint32_t DDSD_DEPTH       = 0x00800000; // dwDepth が有効

const uint32_t DDPF_ALPHAPIXELS = 0x00000001; // RGB 以外に alpha が含まれている
const uint32_t DDPF_ALPHA       = 0x00000002; // pixel は Alpha 成分のみ
const uint32_t DDPF_FOURCC      = 0x00000004; // dwFourCC が有効
const uint32_t DDPF_RGB         = 0x00000040; // dwRGBBitCount/dwRBitMask/dwGBitMask/dwBBitMask/dwRGBAlphaBitMask によってフォーマットが定義されている
const uint32_t DDPF_LUMINANCE   = 0x00020000; // 1ch のデータが R G B すべてに展開される
const uint32_t DDPF_BUMPDUDV    = 0x00080000; // pixel が符号付であることを示す（本来は bump 用）

const uint32_t DDSCAPS_ALPHA    = 0x00000002; // Alpha が含まれている場合 (あまり参照されない)
const uint32_t DDSCAPS_COMPLEX  = 0x00000008; // 複数のデータが含まれている場合 Palette/Mipmap/Cube/Volume 等
const uint32_t DDSCAPS_TEXTURE  = 0x00001000; // 常に 1
const uint32_t DDSCAPS_MIPMAP   = 0x00400000; // MipMap が存在する場合

const uint32_t DDSCAPS2_CUBEMAP   = 0x00000200; // Cubemap が存在する場合
const uint32_t DDSCAPS2_CUBEFACES = 0x0000fc00; // Cubemap の各フェースの存在フラグ
const uint32_t DDSCAPS2_VOLUME    = 0x00200000; // VolumeTexture の場合

#define DDS_FOUR_CC_VALUE(c0, c1, c2, c3)                                    \
    ((static_cast<uint32_t>(c0)      ) | (static_cast<uint32_t>(c1) <<  8) | \
     (static_cast<uint32_t>(c2) << 16) | (static_cast<uint32_t>(c3) << 24))

//! @brief DDS ファイルのフォーマット情報を表す列挙型です。
enum DdsFourCc
{
    DdsFourCc_None          =   0,
    DdsFourCc_A8R8G8B8      =  21, // 0x15
    DdsFourCc_X8R8G8B8      =  22, // 0x16
    DdsFourCc_R5G6B5        =  23, // 0x17
    DdsFourCc_A1R5G5B5      =  25, // 0x19
    DdsFourCc_A4R4G4B4      =  26, // 0x1a
    DdsFourCc_A16B16G16R16  =  36, // 0x24
    DdsFourCc_Q8W8V8U8      =  63, // 0x3f
    DdsFourCc_Q16W16V16U16  = 110, // 0x6e
    DdsFourCc_R16F          = 111, // 0x6f
    DdsFourCc_G16R16F       = 112, // 0x70
    DdsFourCc_A16B16G16R16F = 113, // 0x71
    DdsFourCc_R32F          = 114, // 0x72
    DdsFourCc_G32R32F       = 115, // 0x73
    DdsFourCc_A32B32G32R32F = 116, // 0x74
    DdsFourCc_CxV8U8        = 117, // 0x75
    DdsFourCc_Dxt1 = DDS_FOUR_CC_VALUE('D', 'X', 'T', '1'), // 0x31545844
    DdsFourCc_Dxt2 = DDS_FOUR_CC_VALUE('D', 'X', 'T', '2'), // 0x32545844
    DdsFourCc_Dxt3 = DDS_FOUR_CC_VALUE('D', 'X', 'T', '3'), // 0x33545844
    DdsFourCc_Dxt4 = DDS_FOUR_CC_VALUE('D', 'X', 'T', '4'), // 0x34545844
    DdsFourCc_Dxt5 = DDS_FOUR_CC_VALUE('D', 'X', 'T', '5'), // 0x35545844
    DdsFourCc_Ati1 = DDS_FOUR_CC_VALUE('A', 'T', 'I', '1'), // 0x31495441
    DdsFourCc_Ati2 = DDS_FOUR_CC_VALUE('A', 'T', 'I', '2'), // 0x32495441
    DdsFourCc_Bc1  = DDS_FOUR_CC_VALUE('B', 'C', '1', ' '), // 0x20314342
    DdsFourCc_Bc2  = DDS_FOUR_CC_VALUE('B', 'C', '2', ' '), // 0x20324342
    DdsFourCc_Bc3  = DDS_FOUR_CC_VALUE('B', 'C', '3', ' '), // 0x20334342
    DdsFourCc_Bc4U = DDS_FOUR_CC_VALUE('B', 'C', '4', 'U'), // 0x55344342
    DdsFourCc_Bc4S = DDS_FOUR_CC_VALUE('B', 'C', '4', 'S'), // 0x53344342
    DdsFourCc_Bc5U = DDS_FOUR_CC_VALUE('B', 'C', '5', 'U'), // 0x55354342
    DdsFourCc_Bc5S = DDS_FOUR_CC_VALUE('B', 'C', '5', 'S'), // 0x53354342
    DdsFourCc_Dx10 = DDS_FOUR_CC_VALUE('D', 'X', '1', '0')  // 0x30315844
};

#undef DDS_FOUR_CC_VALUE

//! @brief DDS ファイルの dxgi フォーマットを表す列挙型です。
enum DdsDxgi
{
    DdsDxgi_Unknown               =   0, // 0x00
    DdsDxgi_R32G32B32A32_Float    =   2, // 0x02
    DdsDxgi_R32G32B32A32_Uint     =   3, // 0x03
    DdsDxgi_R32G32B32A32_Sint     =   4, // 0x04
    DdsDxgi_R32G32B32_Float       =   6, // 0x06 現在 dds / bntx への変換のみ対応です。
    DdsDxgi_R32G32B32_Uint        =   7, // 0x07 現在 dds / bntx への変換のみ対応です。
    DdsDxgi_R32G32B32_Sint        =   8, // 0x08 現在 dds / bntx への変換のみ対応です。
    DdsDxgi_R16G16B16A16_Float    =  10, // 0x0a
    DdsDxgi_R16G16B16A16_Unorm    =  11, // 0x0b
    DdsDxgi_R16G16B16A16_Uint     =  12, // 0x0c
    DdsDxgi_R16G16B16A16_Snorm    =  13, // 0x0d
    DdsDxgi_R16G16B16A16_Sint     =  14, // 0x0e
    DdsDxgi_R32G32_Float          =  16, // 0x10
    DdsDxgi_R32G32_Uint           =  17, // 0x11
    DdsDxgi_R32G32_Sint           =  18, // 0x12
    DdsDxgi_R10G10B10A2_Unorm     =  24, // 0x18
    DdsDxgi_R10G10B10A2_Uint      =  25, // 0x19
    DdsDxgi_R11G11B10_Float       =  26, // 0x1a
    DdsDxgi_R8G8B8A8_Unorm        =  28, // 0x1c
    DdsDxgi_R8G8B8A8_Unorm_Srgb   =  29, // 0x1d
    DdsDxgi_R8G8B8A8_Uint         =  30, // 0x1e
    DdsDxgi_R8G8B8A8_Snorm        =  31, // 0x1f
    DdsDxgi_R8G8B8A8_Sint         =  32, // 0x20
    DdsDxgi_R16G16_Float          =  34, // 0x22
    DdsDxgi_R16G16_Unorm          =  35, // 0x23
    DdsDxgi_R16G16_Uint           =  36, // 0x24
    DdsDxgi_R16G16_Snorm          =  37, // 0x25
    DdsDxgi_R16G16_Sint           =  38, // 0x26
    DdsDxgi_R32_Float             =  41, // 0x29
    DdsDxgi_R32_Uint              =  42, // 0x2a
    DdsDxgi_R32_Sint              =  43, // 0x2b
    DdsDxgi_R8G8_Unorm            =  49, // 0x31
    DdsDxgi_R8G8_Uint             =  50, // 0x32
    DdsDxgi_R8G8_Snorm            =  51, // 0x33
    DdsDxgi_R8G8_Sint             =  52, // 0x34
    DdsDxgi_R16_Float             =  54, // 0x36
    DdsDxgi_R16_Unorm             =  56, // 0x38
    DdsDxgi_R16_Uint              =  57, // 0x39
    DdsDxgi_R16_Snorm             =  58, // 0x3a
    DdsDxgi_R16_Sint              =  59, // 0x3b
    DdsDxgi_R8_Unorm              =  61, // 0x3d
    DdsDxgi_R8_Uint               =  62, // 0x3e
    DdsDxgi_R8_Snorm              =  63, // 0x3f
    DdsDxgi_R8_Sint               =  64, // 0x40

    DdsDxgi_Bc1_Unorm             =  71, // 0x47
    DdsDxgi_Bc1_Unorm_Srgb        =  72, // 0x48
    DdsDxgi_Bc2_Unorm             =  74, // 0x4a
    DdsDxgi_Bc2_Unorm_Srgb        =  75, // 0x4b
    DdsDxgi_Bc3_Unorm             =  77, // 0x4d
    DdsDxgi_Bc3_Unorm_Srgb        =  78, // 0x4e
    DdsDxgi_Bc4_Unorm             =  80, // 0x50
    DdsDxgi_Bc4_Snorm             =  81, // 0x51
    DdsDxgi_Bc5_Unorm             =  83, // 0x53
    DdsDxgi_Bc5_Snorm             =  84, // 0x54

    DdsDxgi_B5G6R5_Unorm          =  85, // 0x55 unorm_5_6_5 とは並びが異なります。
    DdsDxgi_B5G5R5A1_Unorm        =  86, // 0x56 unorm_5_5_5_1 とは並びが異なります。
    DdsDxgi_B8G8R8A8_Unorm        =  87, // 0x57 unorm_8_8_8_8 とは並びが異なります。
    DdsDxgi_B8G8R8X8_Unorm        =  88, // 0x58 unorm_8_8_8_8 とは並びが異なります。
    DdsDxgi_B8G8R8A8_Unorm_Srgb   =  91, // 0x5b srgb_8_8_8_8 とは並びが異なります。
    DdsDxgi_B8G8R8X8_Unorm_Srgb   =  93, // 0x5d srgb_8_8_8_8 とは並びが異なります。

    DdsDxgi_Bc6H_Uf16             =  95, // 0x5f
    DdsDxgi_Bc6H_Sf16             =  96, // 0x60
    DdsDxgi_Bc7_Unorm             =  98, // 0x62
    DdsDxgi_Bc7_Unorm_Srgb        =  99, // 0x63

    DdsDxgi_B4G4R4A4_Unorm        = 115, // 0x73 unorm_4_4_4_4 とは並びが異なります。

    DdsDxgi_Astc_4x4_Unorm        = 134, // 0x86
    DdsDxgi_Astc_4x4_Unorm_Srgb   = 135, // 0x87
    DdsDxgi_Astc_5x4_Unorm        = 138, // 0x8a
    DdsDxgi_Astc_5x4_Unorm_Srgb   = 139, // 0x8b
    DdsDxgi_Astc_5x5_Unorm        = 142, // 0x8e
    DdsDxgi_Astc_5x5_Unorm_Srgb   = 143, // 0x8f
    DdsDxgi_Astc_6x5_Unorm        = 146, // 0x92
    DdsDxgi_Astc_6x5_Unorm_Srgb   = 147, // 0x93
    DdsDxgi_Astc_6x6_Unorm        = 150, // 0x96
    DdsDxgi_Astc_6x6_Unorm_Srgb   = 151, // 0x97
    DdsDxgi_Astc_8x5_Unorm        = 154, // 0x9a
    DdsDxgi_Astc_8x5_Unorm_Srgb   = 155, // 0x9b
    DdsDxgi_Astc_8x6_Unorm        = 158, // 0x9e
    DdsDxgi_Astc_8x6_Unorm_Srgb   = 159, // 0x9f
    DdsDxgi_Astc_8x8_Unorm        = 162, // 0xa2
    DdsDxgi_Astc_8x8_Unorm_Srgb   = 163, // 0xa3
    DdsDxgi_Astc_10x5_Unorm       = 166, // 0xa6
    DdsDxgi_Astc_10x5_Unorm_Srgb  = 167, // 0xa7
    DdsDxgi_Astc_10x6_Unorm       = 170, // 0xaa
    DdsDxgi_Astc_10x6_Unorm_Srgb  = 171, // 0xab
    DdsDxgi_Astc_10x8_Unorm       = 174, // 0xae
    DdsDxgi_Astc_10x8_Unorm_Srgb  = 175, // 0xaf
    DdsDxgi_Astc_10x10_Unorm      = 178, // 0xb2
    DdsDxgi_Astc_10x10_Unorm_Srgb = 179, // 0xb3
    DdsDxgi_Astc_12x10_Unorm      = 182, // 0xb6
    DdsDxgi_Astc_12x10_Unorm_Srgb = 183, // 0xb7
    DdsDxgi_Astc_12x12_Unorm      = 186, // 0xba
    DdsDxgi_Astc_12x12_Unorm_Srgb = 187, // 0xbb

    DdsDxgi_Force_Uint = 0xffffffff
};

//! @brief DDS ファイルのリソースの次元を表す列挙型です。
enum DdsResDimension
{
    DdsResDimension_Texture1d = 2,
    DdsResDimension_Texture2d = 3,
    DdsResDimension_Texture3d = 4
};

//! @brief DDS ファイルのリソースの misc フラグを表す列挙型です。
enum DdsResMiscFlag
{
    DdsResMisc_TextureCube = 0x04
};

//-----------------------------------------------------------------------------
// DDS ファイルの dxgi フォーマットと対応するテクスチャフォーマットの配列です。
//-----------------------------------------------------------------------------
struct DdsDxgiAndFtxFormat
{
    DdsDxgi dxgiFormat;
    FtxFormat ftxFormat;
};

const DdsDxgiAndFtxFormat DdsDxgiAndFtxFormats[] =
{
    { DdsDxgi_R32G32B32A32_Float   , FtxFormat_Float_32_32_32_32 },
    { DdsDxgi_R32G32B32A32_Uint    , FtxFormat_Uint_32_32_32_32  },
    { DdsDxgi_R32G32B32A32_Sint    , FtxFormat_Sint_32_32_32_32  },
    { DdsDxgi_R32G32B32_Float      , FtxFormat_Float_32_32_32    },
    { DdsDxgi_R32G32B32_Uint       , FtxFormat_Uint_32_32_32     },
    { DdsDxgi_R32G32B32_Sint       , FtxFormat_Sint_32_32_32     },
    { DdsDxgi_R16G16B16A16_Float   , FtxFormat_Float_16_16_16_16 },
    { DdsDxgi_R16G16B16A16_Unorm   , FtxFormat_Unorm_16_16_16_16 },
    { DdsDxgi_R16G16B16A16_Uint    , FtxFormat_Uint_16_16_16_16  },
    { DdsDxgi_R16G16B16A16_Snorm   , FtxFormat_Snorm_16_16_16_16 },
    { DdsDxgi_R16G16B16A16_Sint    , FtxFormat_Sint_16_16_16_16  },
    { DdsDxgi_R32G32_Float         , FtxFormat_Float_32_32       },
    { DdsDxgi_R32G32_Uint          , FtxFormat_Uint_32_32        },
    { DdsDxgi_R32G32_Sint          , FtxFormat_Sint_32_32        },
    { DdsDxgi_R10G10B10A2_Unorm    , FtxFormat_Unorm_10_10_10_2  },
    { DdsDxgi_R10G10B10A2_Uint     , FtxFormat_Uint_10_10_10_2   },
    { DdsDxgi_R11G11B10_Float      , FtxFormat_Float_11_11_10    },
    { DdsDxgi_R8G8B8A8_Unorm       , FtxFormat_Unorm_8_8_8_8     },
    { DdsDxgi_R8G8B8A8_Unorm_Srgb  , FtxFormat_Srgb_8_8_8_8      },
    { DdsDxgi_R8G8B8A8_Uint        , FtxFormat_Uint_8_8_8_8      },
    { DdsDxgi_R8G8B8A8_Snorm       , FtxFormat_Snorm_8_8_8_8     },
    { DdsDxgi_R8G8B8A8_Sint        , FtxFormat_Sint_8_8_8_8      },
    { DdsDxgi_R16G16_Float         , FtxFormat_Float_16_16       },
    { DdsDxgi_R16G16_Unorm         , FtxFormat_Unorm_16_16       },
    { DdsDxgi_R16G16_Uint          , FtxFormat_Uint_16_16        },
    { DdsDxgi_R16G16_Snorm         , FtxFormat_Snorm_16_16       },
    { DdsDxgi_R16G16_Sint          , FtxFormat_Sint_16_16        },
    { DdsDxgi_R32_Float            , FtxFormat_Float_32          },
    { DdsDxgi_R32_Uint             , FtxFormat_Uint_32           },
    { DdsDxgi_R32_Sint             , FtxFormat_Sint_32           },
    { DdsDxgi_R8G8_Unorm           , FtxFormat_Unorm_8_8         },
    { DdsDxgi_R8G8_Uint            , FtxFormat_Uint_8_8          },
    { DdsDxgi_R8G8_Snorm           , FtxFormat_Snorm_8_8         },
    { DdsDxgi_R8G8_Sint            , FtxFormat_Sint_8_8          },
    { DdsDxgi_R16_Float            , FtxFormat_Float_16          },
    { DdsDxgi_R16_Unorm            , FtxFormat_Unorm_16          },
    { DdsDxgi_R16_Uint             , FtxFormat_Uint_16           },
    { DdsDxgi_R16_Snorm            , FtxFormat_Snorm_16          },
    { DdsDxgi_R16_Sint             , FtxFormat_Sint_16           },
    { DdsDxgi_R8_Unorm             , FtxFormat_Unorm_8           },
    { DdsDxgi_R8_Uint              , FtxFormat_Uint_8            },
    { DdsDxgi_R8_Snorm             , FtxFormat_Snorm_8           },
    { DdsDxgi_R8_Sint              , FtxFormat_Sint_8            },

    { DdsDxgi_Bc1_Unorm            , FtxFormat_Unorm_Bc1 },
    { DdsDxgi_Bc1_Unorm_Srgb       , FtxFormat_Srgb_Bc1  },
    { DdsDxgi_Bc2_Unorm            , FtxFormat_Unorm_Bc2 },
    { DdsDxgi_Bc2_Unorm_Srgb       , FtxFormat_Srgb_Bc2  },
    { DdsDxgi_Bc3_Unorm            , FtxFormat_Unorm_Bc3 },
    { DdsDxgi_Bc3_Unorm_Srgb       , FtxFormat_Srgb_Bc3  },
    { DdsDxgi_Bc4_Unorm            , FtxFormat_Unorm_Bc4 },
    { DdsDxgi_Bc4_Snorm            , FtxFormat_Snorm_Bc4 },
    { DdsDxgi_Bc5_Unorm            , FtxFormat_Unorm_Bc5 },
    { DdsDxgi_Bc5_Snorm            , FtxFormat_Snorm_Bc5 },

    { DdsDxgi_B5G6R5_Unorm         , FtxFormat_Unorm_5_6_5   },
    { DdsDxgi_B5G5R5A1_Unorm       , FtxFormat_Unorm_5_5_5_1 },
    { DdsDxgi_B8G8R8A8_Unorm       , FtxFormat_Unorm_8_8_8_8 },
    { DdsDxgi_B8G8R8X8_Unorm       , FtxFormat_Unorm_8_8_8_8 },
    { DdsDxgi_B8G8R8A8_Unorm_Srgb  , FtxFormat_Srgb_8_8_8_8  },
    { DdsDxgi_B8G8R8X8_Unorm_Srgb  , FtxFormat_Srgb_8_8_8_8  },

    { DdsDxgi_Bc6H_Uf16            , FtxFormat_Ufloat_Bc6 },
    { DdsDxgi_Bc6H_Sf16            , FtxFormat_Float_Bc6  },
    { DdsDxgi_Bc7_Unorm            , FtxFormat_Unorm_Bc7  },
    { DdsDxgi_Bc7_Unorm_Srgb       , FtxFormat_Srgb_Bc7   },

    { DdsDxgi_B4G4R4A4_Unorm       , FtxFormat_Unorm_4_4_4_4 },

    { DdsDxgi_Astc_4x4_Unorm       , FtxFormat_Unorm_Astc_4x4   },
    { DdsDxgi_Astc_4x4_Unorm_Srgb  , FtxFormat_Srgb_Astc_4x4    },
    { DdsDxgi_Astc_5x4_Unorm       , FtxFormat_Unorm_Astc_5x4   },
    { DdsDxgi_Astc_5x4_Unorm_Srgb  , FtxFormat_Srgb_Astc_5x4    },
    { DdsDxgi_Astc_5x5_Unorm       , FtxFormat_Unorm_Astc_5x5   },
    { DdsDxgi_Astc_5x5_Unorm_Srgb  , FtxFormat_Srgb_Astc_5x5    },
    { DdsDxgi_Astc_6x5_Unorm       , FtxFormat_Unorm_Astc_6x5   },
    { DdsDxgi_Astc_6x5_Unorm_Srgb  , FtxFormat_Srgb_Astc_6x5    },
    { DdsDxgi_Astc_6x6_Unorm       , FtxFormat_Unorm_Astc_6x6   },
    { DdsDxgi_Astc_6x6_Unorm_Srgb  , FtxFormat_Srgb_Astc_6x6    },
    { DdsDxgi_Astc_8x5_Unorm       , FtxFormat_Unorm_Astc_8x5   },
    { DdsDxgi_Astc_8x5_Unorm_Srgb  , FtxFormat_Srgb_Astc_8x5    },
    { DdsDxgi_Astc_8x6_Unorm       , FtxFormat_Unorm_Astc_8x6   },
    { DdsDxgi_Astc_8x6_Unorm_Srgb  , FtxFormat_Srgb_Astc_8x6    },
    { DdsDxgi_Astc_8x8_Unorm       , FtxFormat_Unorm_Astc_8x8   },
    { DdsDxgi_Astc_8x8_Unorm_Srgb  , FtxFormat_Srgb_Astc_8x8    },
    { DdsDxgi_Astc_10x5_Unorm      , FtxFormat_Unorm_Astc_10x5  },
    { DdsDxgi_Astc_10x5_Unorm_Srgb , FtxFormat_Srgb_Astc_10x5   },
    { DdsDxgi_Astc_10x6_Unorm      , FtxFormat_Unorm_Astc_10x6  },
    { DdsDxgi_Astc_10x6_Unorm_Srgb , FtxFormat_Srgb_Astc_10x6   },
    { DdsDxgi_Astc_10x8_Unorm      , FtxFormat_Unorm_Astc_10x8  },
    { DdsDxgi_Astc_10x8_Unorm_Srgb , FtxFormat_Srgb_Astc_10x8   },
    { DdsDxgi_Astc_10x10_Unorm     , FtxFormat_Unorm_Astc_10x10 },
    { DdsDxgi_Astc_10x10_Unorm_Srgb, FtxFormat_Srgb_Astc_10x10  },
    { DdsDxgi_Astc_12x10_Unorm     , FtxFormat_Unorm_Astc_12x10 },
    { DdsDxgi_Astc_12x10_Unorm_Srgb, FtxFormat_Srgb_Astc_12x10  },
    { DdsDxgi_Astc_12x12_Unorm     , FtxFormat_Unorm_Astc_12x12 },
    { DdsDxgi_Astc_12x12_Unorm_Srgb, FtxFormat_Srgb_Astc_12x12  },
};

//=============================================================================
//! @brief DDS ファイルマスク情報のクラスです。
//=============================================================================
class DdsMaskInfo
{
public:
    uint32_t m_Mask; //!< マスク値です。
    int m_Size; //!< 値のビット数です。
    int m_Shift; //!< シフト数です。

public:
    //! @brief コンストラクタです。
    DdsMaskInfo()
    : m_Mask(0),
      m_Size(0),
      m_Shift(0)
    {
    }

    //! @brief コンストラクタです。
    //!
    //! @param[in] mask マスク値です。
    //!
    explicit DdsMaskInfo(uint32_t mask)
    {
        m_Mask = mask;
        m_Size = m_Shift = 0;
        if (mask != 0)
        {
            while ((mask & 1) == 0 && m_Shift < 32)
            {
                m_Shift += 1;
                mask >>= 1;
            }
            while ((mask & 1) != 0 && m_Shift + m_Size < 32)
            {
                m_Size += 1;
                mask >>= 1;
            }
        }
    }
};

//=============================================================================
//! @brief DDS ファイルヘッダのクラスです。
//=============================================================================
class DdsHeader
{
public:
    // 0x00
    uint32_t dwMagic;             // == 0x20534444  ' SDD'
    uint32_t dwSize;              // == 0x7c で固定
    uint32_t dwFlags;             // ヘッダ内の有効な情報 DDSD_* の組み合わせ
    uint32_t dwHeight;            // 画像の高さ x size
    // 0x10
    uint32_t dwWidth;             // 画像の幅   y size
    uint32_t dwPitchOrLinearSize; // 横1 line の byte 数 (pitch)
                                  // または 1面分の byte 数 (linearsize)
    uint32_t dwDepth;             // 画像の奥行き z size (Volume Texture 用)
    uint32_t dwMipMapCount;       // 含まれている mipmap レベル数
    // 0x20
    uint32_t dwReserved1[11];
    uint32_t dwPfSize;          // == 0x20 で固定
    // 0x50
    uint32_t dwPfFlags;         // pixel フォーマットを表す DDPF_* の組み合わせ
    uint32_t dwFourCC;          // フォーマットが FourCC であらわされる場合のみ
    uint32_t dwRGBBitCount;     // 1 pixel の bit 数
    uint32_t dwRBitMask;        // RGB format 時の mask
    // 0x60
    uint32_t dwGBitMask;        // RGB format 時の mask
    uint32_t dwBBitMask;        // RGB format 時の mask
    uint32_t dwRGBAlphaBitMask; // RGB format 時の mask
    uint32_t dwCaps;            // mipmap 等のフラグ指定用 DDSCAPS_* の組み合わせ
    // 0x70
    uint32_t dwCaps2;           // cube/volume texture 等のフラグ指定用 DDSCAPS2_* の組み合わせ
    uint32_t dwReservedCaps[2];
    uint32_t dwReserved2;

public:
    //! コンストラクタです。
    DdsHeader()
    {
        //cerr << "DdsHeader(): " << sizeof(*this) << endl;
        memset(this, 0x00, sizeof(*this));
        dwMagic  = 0x20534444;
        dwSize   = 0x7c;
        dwPfSize = 0x20;
    }
};

//=============================================================================
//! @brief DDS ファイル DX10 ヘッダのクラスです。
//=============================================================================
class DdsDx10Header
{
public:
    uint32_t dxgiFormat;
    uint32_t resourceDimension;
    uint32_t miscFlag;
    uint32_t arraySize;
    uint32_t reserved;

public:
    //! コンストラクタです。
    DdsDx10Header()
    : dxgiFormat(DdsDxgi_R8G8B8A8_Unorm),
      resourceDimension(DdsResDimension_Texture2d),
      miscFlag(0),
      arraySize(1),
      reserved(0)
    {
    }
};

//-----------------------------------------------------------------------------
//! @brief DDS ファイルからテクスチャフォーマットを取得します。
//!
//! @param[in] dxgiFormat dxgi フォーマットです。
//! @param[in] pfFlags ピクセルフラグです。
//! @param[in] fourCc フォーマット情報です。
//! @param[in] rgbBitCount RGB フォーマットの場合の 1 ピクセルのビット数です。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//!
//! @return テクスチャフォーマットを返します。
//-----------------------------------------------------------------------------
FtxFormat GetFormatFromDds(
    const DdsDxgi dxgiFormat,
    const uint32_t pfFlags,
    const DdsFourCc fourCc,
    const uint32_t rgbBitCount,
    const DdsMaskInfo* maskInfos
)
{
    if (dxgiFormat != DdsDxgi_Unknown)
    {
        const int formatCount = sizeof(DdsDxgiAndFtxFormats) / sizeof(DdsDxgiAndFtxFormats[0]);
        for (int formatIdx = 0; formatIdx < formatCount; ++formatIdx)
        {
            const DdsDxgiAndFtxFormat& formatInfo = DdsDxgiAndFtxFormats[formatIdx];
            if (formatInfo.dxgiFormat == dxgiFormat)
            {
                return formatInfo.ftxFormat;
            }
        }
        return FtxFormat_Invalid;
    }
    else if (fourCc != DdsFourCc_None)
    {
        switch (fourCc)
        {
        case DdsFourCc_A16B16G16R16:  return FtxFormat_Unorm_16_16_16_16;
        case DdsFourCc_Q8W8V8U8:      return FtxFormat_Snorm_8_8_8_8;
        case DdsFourCc_Q16W16V16U16:  return FtxFormat_Snorm_16_16_16_16;
        case DdsFourCc_R16F:          return FtxFormat_Float_16;
        case DdsFourCc_G16R16F:       return FtxFormat_Float_16_16;
        case DdsFourCc_A16B16G16R16F: return FtxFormat_Float_16_16_16_16;
        case DdsFourCc_R32F:          return FtxFormat_Float_32;
        case DdsFourCc_G32R32F:       return FtxFormat_Float_32_32;
        case DdsFourCc_A32B32G32R32F: return FtxFormat_Float_32_32_32_32;
        case DdsFourCc_CxV8U8:        return FtxFormat_Snorm_8_8;
        case DdsFourCc_Dxt1:          return FtxFormat_Unorm_Bc1;
        case DdsFourCc_Dxt2:          return FtxFormat_Unorm_Bc2;
        case DdsFourCc_Dxt3:          return FtxFormat_Unorm_Bc2;
        case DdsFourCc_Dxt4:          return FtxFormat_Unorm_Bc3;
        case DdsFourCc_Dxt5:          return FtxFormat_Unorm_Bc3;
        case DdsFourCc_Ati1:          return FtxFormat_Unorm_Bc4;
        case DdsFourCc_Ati2:          return FtxFormat_Unorm_Bc5;
        case DdsFourCc_Bc1:           return FtxFormat_Unorm_Bc1;
        case DdsFourCc_Bc2:           return FtxFormat_Unorm_Bc2;
        case DdsFourCc_Bc3:           return FtxFormat_Unorm_Bc3;
        case DdsFourCc_Bc4U:          return FtxFormat_Unorm_Bc4;
        case DdsFourCc_Bc4S:          return FtxFormat_Snorm_Bc4;
        case DdsFourCc_Bc5U:          return FtxFormat_Unorm_Bc5;
        case DdsFourCc_Bc5S:          return FtxFormat_Snorm_Bc5;
        default:                      return FtxFormat_Invalid;
        }
    }
    else
    {
        const bool isSigned = ((pfFlags & DDPF_BUMPDUDV) != 0);
        const int rBits = maskInfos[0].m_Size;
        const int gBits = maskInfos[1].m_Size;
        const int aBits = maskInfos[3].m_Size;
        switch (rgbBitCount)
        {
        case 8:
            return (rBits == 8) ? FtxFormat_Unorm_8 :
                   (aBits == 8) ? FtxFormat_Unorm_8 : // A8
                   FtxFormat_Invalid; // A4L4 / P8 は非対応

        case 16:
            return (rBits ==  4 && aBits == 4) ? FtxFormat_Unorm_4_4_4_4 :
                   (rBits ==  5 && gBits == 6) ? FtxFormat_Unorm_5_6_5   :
                   (rBits ==  5 && gBits == 5) ? FtxFormat_Unorm_5_5_5_1 :
                   (rBits ==  8 && isSigned  ) ? FtxFormat_Snorm_8_8     : // V8U8
                   (rBits ==  8              ) ? FtxFormat_Unorm_8_8     :
                   (rBits == 16              ) ? FtxFormat_Unorm_16      :
                   FtxFormat_Invalid;

        case 24:
            return (rBits == 8) ? FtxFormat_Unorm_8_8_8_8 :
                   FtxFormat_Invalid;

        case 32:
            return (rBits ==  8 && isSigned) ? FtxFormat_Snorm_8_8_8_8 : // Q8W8V8U8
                   (rBits ==  8            ) ? FtxFormat_Unorm_8_8_8_8 :
                   (rBits == 16 && isSigned) ? FtxFormat_Snorm_16_16   : // V16U16
                   (rBits == 16            ) ? FtxFormat_Unorm_16_16   :
                   FtxFormat_Invalid;

        default:
            return FtxFormat_Invalid; // P4 は非対応
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルからテクスチャの次元を取得します。
//!        同時に画像の奥行きを調整します。
//!
//! @param[in,out] pImageD DDS ファイルから取得した画像の奥行きへのポインタです。
//! @param[in] fourCc フォーマット情報です。
//! @param[in] caps2 caps2 フラグです。
//! @param[in] resourceDimension リソースの次元です。
//! @param[in] miscFlag misc フラグです。
//! @param[in] arraySize 配列サイズです。
//!
//! @return テクスチャの次元を返します。
//-----------------------------------------------------------------------------
FtxDimension GetDimensionFromDds(
    int* pImageD,
    const DdsFourCc fourCc,
    const uint32_t caps2,
    const DdsResDimension resourceDimension,
    const uint32_t miscFlag,
    const uint32_t arraySize
)
{
    if (fourCc != DdsFourCc_Dx10)
    {
        if ((caps2 & DDSCAPS2_CUBEMAP) != 0)
        {
            *pImageD = RImage::CUBE_FACE_COUNT;
            return FtxDimension_CubeMap;
        }
        else if ((caps2 & DDSCAPS2_VOLUME) != 0)
        {
            return FtxDimension_3d;
        }
        else if (*pImageD >= 2)
        {
            return FtxDimension_2dArray;
        }
        else
        {
            return FtxDimension_2d;
        }
    }
    else
    {
        if ((miscFlag & DdsResMisc_TextureCube) != 0)
        {
            *pImageD = RImage::CUBE_FACE_COUNT * arraySize;
            return (arraySize == 1) ? FtxDimension_CubeMap : FtxDimension_CubeMapArray;
        }
        else
        {
            *pImageD *= arraySize;
            if (resourceDimension == DdsResDimension_Texture1d)
            {
                return (*pImageD == 1) ? FtxDimension_1d : FtxDimension_1dArray;
            }
            else if (resourceDimension == DdsResDimension_Texture2d)
            {
                return (*pImageD == 1) ? FtxDimension_2d : FtxDimension_2dArray;
            }
            else // 3D
            {
                return FtxDimension_3d;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief A8 フォーマットかどうかを取得します。
//!
//! @param[in] rgbBitCount RGB フォーマットの場合の 1 ピクセルのビット数です。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//!
//! @return A8 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsA8Format(const uint32_t rgbBitCount, const DdsMaskInfo* maskInfos)
{
    return (rgbBitCount == 8 && maskInfos[3].m_Size == 8);
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルがアルファ成分を持つかどうかを取得します。
//!
//! @param[in] ftxFormat テクスチャフォーマットです。
//! @param[in] dxgiFormat dxgi フォーマットです。
//! @param[in] fourCc フォーマット情報です。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//!
//! @return アルファ成分を持つなら true を返します。
//-----------------------------------------------------------------------------
bool DdsHasAlpha(
    const FtxFormat ftxFormat,
    const DdsDxgi dxgiFormat,
    const DdsFourCc fourCc,
    const DdsMaskInfo* maskInfos
)
{
    if (dxgiFormat == DdsDxgi_B8G8R8X8_Unorm      ||
        dxgiFormat == DdsDxgi_B8G8R8X8_Unorm_Srgb)
    {
        return false;
    }
    else if (fourCc == DdsFourCc_None)
    {
        return (maskInfos[3].m_Size != 0);
    }
    else
    {
        return (RGetComponentCount(ftxFormat) >= 4);
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのマスク情報を取得します。
//!
//! @param[out] maskInfos RGBA 成分のマスク情報を格納します。長さ 4 の配列を指定します。
//! @param[in] maskR R 成分のマスクです。
//! @param[in] maskG G 成分のマスクです。
//! @param[in] maskB B 成分のマスクです。
//! @param[in] maskA A 成分のマスクです。
//! @param[in] dxgiFormat dxgi フォーマットです。
//-----------------------------------------------------------------------------
void GetDdsMaskInfos(
    DdsMaskInfo* maskInfos,
    const uint32_t maskR,
    const uint32_t maskG,
    const uint32_t maskB,
    const uint32_t maskA,
    const DdsDxgi dxgiFormat
)
{
    uint32_t mr = maskR;
    uint32_t mg = maskG;
    uint32_t mb = maskB;
    uint32_t ma = maskA;

    //-----------------------------------------------------------------------------
    // dxgi フォーマットが RGBA マスクで表現可能なら RGBA マスクを取得します。
    switch (dxgiFormat)
    {
    case DdsDxgi_B5G6R5_Unorm:
        mr = 0x0000f800;
        mg = 0x000007e0;
        mb = 0x0000001f;
        ma = 0x00000000;
        break;

    case DdsDxgi_B5G5R5A1_Unorm:
        mr = 0x00007c00;
        mg = 0x000003e0;
        mb = 0x0000001f;
        ma = 0x00008000;
        break;

    case DdsDxgi_B4G4R4A4_Unorm:
        mr = 0x00000f00;
        mg = 0x000000f0;
        mb = 0x0000000f;
        ma = 0x0000f000;
        break;

    case DdsDxgi_R8G8B8A8_Unorm:
    case DdsDxgi_R8G8B8A8_Unorm_Srgb:
        mr = 0x000000ff;
        mg = 0x0000ff00;
        mb = 0x00ff0000;
        ma = 0xff000000;
        break;

    case DdsDxgi_B8G8R8A8_Unorm:
    case DdsDxgi_B8G8R8A8_Unorm_Srgb:
        mr = 0x00ff0000;
        mg = 0x0000ff00;
        mb = 0x000000ff;
        ma = 0xff000000;
        break;

    case DdsDxgi_B8G8R8X8_Unorm:
    case DdsDxgi_B8G8R8X8_Unorm_Srgb:
        mr = 0x00ff0000;
        mg = 0x0000ff00;
        mb = 0x000000ff;
        ma = 0x00000000;
        break;

    default:
        break;
    };

    //-----------------------------------------------------------------------------
    // マスク情報を作成します。
    maskInfos[0] = DdsMaskInfo(mr);
    maskInfos[1] = DdsMaskInfo(mg);
    maskInfos[2] = DdsMaskInfo(mb);
    maskInfos[3] = DdsMaskInfo(ma);
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの B5G6R5 ピクセルデータを
//!        テクスチャコンバータが扱う形式に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//-----------------------------------------------------------------------------
void ConvertFromDdsB5G6R5(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const DdsMaskInfo* maskInfos
)
{
    uint8_t* pDst = pDstPixels;
    const uint8_t* pSrc = pSrcPixels;
    const size_t pixelCount = dstDataSize / sizeof(uint16_t);
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        const uint32_t s = pSrc[0] | (pSrc[1] << 8);
        const uint32_t r = (s >> maskInfos[0].m_Shift) & 0x1f;
        const uint32_t g = (s >> maskInfos[1].m_Shift) & 0x3f;
        const uint32_t b = (s >> maskInfos[2].m_Shift) & 0x1f;
        const uint32_t d = r | (g << 5) | (b << 11);
        pDst[0] = static_cast<uint8_t>((d     ) & 0xff);
        pDst[1] = static_cast<uint8_t>((d >> 8) & 0xff);
        pDst += sizeof(uint16_t);
        pSrc += sizeof(uint16_t);
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの B5G5R5A1 ピクセルデータを
//!        テクスチャコンバータが扱う形式に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//-----------------------------------------------------------------------------
void ConvertFromDdsB5G5R5A1(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const DdsMaskInfo* maskInfos
)
{
    uint8_t* pDst = pDstPixels;
    const uint8_t* pSrc = pSrcPixels;
    const size_t pixelCount = dstDataSize / sizeof(uint16_t);
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        const uint32_t s = pSrc[0] | (pSrc[1] << 8);
        const uint32_t r = (s >> maskInfos[0].m_Shift) & 0x1f;
        const uint32_t g = (s >> maskInfos[1].m_Shift) & 0x1f;
        const uint32_t b = (s >> maskInfos[2].m_Shift) & 0x1f;
        const uint32_t a = (maskInfos[3].m_Size != 0) ? (s >> maskInfos[3].m_Shift) & 0x01 : 0x01;
        const uint32_t d = r | (g << 5) | (b << 10) | (a << 15);
        pDst[0] = static_cast<uint8_t>((d     ) & 0xff);
        pDst[1] = static_cast<uint8_t>((d >> 8) & 0xff);
        pDst += sizeof(uint16_t);
        pSrc += sizeof(uint16_t);
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの B4G4R4A4 ピクセルデータを
//!        テクスチャコンバータが扱う形式に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//-----------------------------------------------------------------------------
void ConvertFromDdsB4G4R4A4(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const DdsMaskInfo* maskInfos
)
{
    uint8_t* pDst = pDstPixels;
    const uint8_t* pSrc = pSrcPixels;
    const size_t pixelCount = dstDataSize / sizeof(uint16_t);
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        const uint32_t s = pSrc[0] | (pSrc[1] << 8);
        const uint32_t r = (s >> maskInfos[0].m_Shift) & 0x0f;
        const uint32_t g = (s >> maskInfos[1].m_Shift) & 0x0f;
        const uint32_t b = (s >> maskInfos[2].m_Shift) & 0x0f;
        const uint32_t a = (maskInfos[3].m_Size != 0) ? (s >> maskInfos[3].m_Shift) & 0x0f : 0x0f;
        const uint32_t d = r | (g << 4) | (b << 8) | (a << 12);
        pDst[0] = static_cast<uint8_t>((d     ) & 0xff);
        pDst[1] = static_cast<uint8_t>((d >> 8) & 0xff);
        pDst += sizeof(uint16_t);
        pSrc += sizeof(uint16_t);
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの B8G8R8 / B8G8R8A8 / B8G8R8X8 / R8G8B8A8 ピクセルデータを
//!        テクスチャコンバータが扱う形式に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] rgbBitCount RGB フォーマットの場合の 1 ピクセルのビット数です。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//-----------------------------------------------------------------------------
void ConvertFromDdsB8G8R8A8(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const uint32_t rgbBitCount,
    const DdsMaskInfo* maskInfos
)
{
    const int rIdx = maskInfos[0].m_Shift / 8;
    const int gIdx = maskInfos[1].m_Shift / 8;
    const int bIdx = maskInfos[2].m_Shift / 8;
    const int aIdx = maskInfos[3].m_Shift / 8;
    uint8_t* pDst = pDstPixels;
    const uint8_t* pSrc = pSrcPixels;
    const size_t dstPixelBytes = sizeof(uint32_t);
    const size_t srcPixelBytes = (rgbBitCount != 0) ?
        rgbBitCount / 8 : sizeof(uint32_t);
    const size_t pixelCount = dstDataSize / dstPixelBytes;
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        const uint8_t r = pSrc[rIdx];
        const uint8_t g = pSrc[gIdx];
        const uint8_t b = pSrc[bIdx];
        const uint8_t a = (maskInfos[3].m_Size != 0) ? pSrc[aIdx] : 0xff;
        pDst[0] = r;
        pDst[1] = g;
        pDst[2] = b;
        pDst[3] = a;
        pDst += dstPixelBytes;
        pSrc += srcPixelBytes;
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの ATI2 ピクセルデータを
//!        テクスチャコンバータが扱う形式に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//-----------------------------------------------------------------------------
void ConvertFromDdsAti2(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize
)
{
    const size_t BlockBytes = 16;
    const size_t BlockBytesHalf = BlockBytes / 2;
    uint8_t blockBuf[BlockBytes];
    uint8_t* pDst = pDstPixels;
    const uint8_t* pSrc = pSrcPixels;
    const size_t blockCount = dstDataSize / BlockBytes;
    for (size_t blockIdx = 0; blockIdx < blockCount; ++blockIdx)
    {
        // R 成分と G 成分が bc5 とは逆なのでスワップします。
        memcpy(blockBuf, pSrc, BlockBytes);
        memcpy(pDst + BlockBytesHalf, &blockBuf[0             ], BlockBytesHalf);
        memcpy(pDst +              0, &blockBuf[BlockBytesHalf], BlockBytesHalf);
        pDst += BlockBytes;
        pSrc += BlockBytes;
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのピクセルデータをテクスチャコンバータが扱う形式に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] ftxFormat テクスチャフォーマットです。
//! @param[in] fourCc フォーマット情報です。
//! @param[in] rgbBitCount RGB フォーマットの場合の 1 ピクセルのビット数です。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//-----------------------------------------------------------------------------
void ConvertFromDdsPixels(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const FtxFormat ftxFormat,
    const DdsFourCc fourCc,
    const uint32_t rgbBitCount,
    const DdsMaskInfo* maskInfos
)
{
    if (ftxFormat == FtxFormat_Unorm_5_6_5)
    {
        ConvertFromDdsB5G6R5(pDstPixels, pSrcPixels, dstDataSize, maskInfos);
    }
    else if (ftxFormat == FtxFormat_Unorm_5_5_5_1)
    {
        ConvertFromDdsB5G5R5A1(pDstPixels, pSrcPixels, dstDataSize, maskInfos);
    }
    else if (ftxFormat == FtxFormat_Unorm_4_4_4_4)
    {
        ConvertFromDdsB4G4R4A4(pDstPixels, pSrcPixels, dstDataSize, maskInfos);
    }
    else if (ftxFormat == FtxFormat_Unorm_8_8_8_8 ||
             ftxFormat == FtxFormat_Srgb_8_8_8_8)
    {
        ConvertFromDdsB8G8R8A8(pDstPixels, pSrcPixels, dstDataSize, rgbBitCount, maskInfos);
    }
    else if (fourCc == DdsFourCc_Ati2)
    {
        ConvertFromDdsAti2(pDstPixels, pSrcPixels, dstDataSize);
    }
    else
    {
        memcpy(pDstPixels, pSrcPixels, dstDataSize);
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの符号付き 8 ビットのピクセルデータの値を
//!        テクスチャコンバータのオリジナル画像の形式に調整します。
//!        int8_t 型の値 [-128,127] を uint8_t 型の値 [0,255] に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//-----------------------------------------------------------------------------
void AdjustDdsSigned8(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize
)
{
    for (size_t valueIdx = 0; valueIdx < dstDataSize; ++valueIdx)
    {
        const uint8_t s = pSrcPixels[valueIdx];
        pDstPixels[valueIdx] = static_cast<uint8_t>(
            ((s & 0x80) == 0) ? s + 0x80 : s - 0x80);
    }
}

//-----------------------------------------------------------------------------
//! @brief RGBA ピクセルデータの R 成分と G 成分をスワップします。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//-----------------------------------------------------------------------------
void SwapComponentRg(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize
)
{
    uint8_t* pDst = pDstPixels;
    const uint8_t* pSrc = pSrcPixels;
    const size_t pixelCount = dstDataSize / sizeof(uint32_t);
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        const uint8_t r = pSrc[0];
        const uint8_t g = pSrc[1];
        pDst[0] = g;
        pDst[1] = r;
        pDst += sizeof(uint32_t);
        pSrc += sizeof(uint32_t);
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのカラー成分を 8bit に変換した値を返します。
//!
//! @param[in] value 変換元の値です。
//! @param[in] bitSize 変換元のビット数です。
//!
//! @return 8bit に変換した値を返します。
//-----------------------------------------------------------------------------
uint8_t GetDds8Bit(const uint32_t value, const int bitSize)
{
    if (bitSize ==  8)
    {
        return static_cast<uint8_t>(value);
    }
    else if (bitSize ==  6)
    {
        return static_cast<uint8_t>((value << 2) | (value >> 4));
    }
    else if (bitSize ==  5)
    {
        return static_cast<uint8_t>((value << 3) | (value >> 2));
    }
    else if (bitSize ==  4)
    {
        return static_cast<uint8_t>((value << 4) | value);
    }
    else if (bitSize ==  1)
    {
        return static_cast<uint8_t>(value * 0xff);
    }
    else if (bitSize ==  0)
    {
        return 0x00;
    }
    else if (bitSize == 16)
    {
        return static_cast<uint8_t>(value >> 8);
    }
    else if (bitSize == 10)
    {
        return static_cast<uint8_t>(value >> 2);
    }
    else
    {
        return static_cast<uint8_t>(value);
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの float16 を float32 に変換して返します。
//!
//! @param[in] pFloat16 float16 型の値へのポインタです。
//!
//! @return float32 に変換した値を返します。
//-----------------------------------------------------------------------------
float GetDdsF16(const uint8_t* pFloat16)
{
    const uint32_t iv = (pFloat16[1] << 8) | pFloat16[0];
    if (iv == 0)
    {
        return 0.0f;
    }
    const uint32_t s =   iv & 0x8000;
    const uint32_t e = ((iv & 0x7c00) >> 10) - 15 + 127;
    const uint32_t f =   iv & 0x03ff;
    const uint32_t fv = (s << 16) | ((e << 23) & 0x7f800000) | (f << 13);
    return *reinterpret_cast<const float*>(&fv);
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの単純な RGBA ピクセルデータをデコードします。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] pfFlags ピクセルフラグです。
//! @param[in] rgbBitCount RGB フォーマットの場合の 1 ピクセルのビット数です。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//-----------------------------------------------------------------------------
void DecodeDdsSimpleRgbaPixels(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const uint32_t pfFlags,
    const uint32_t rgbBitCount,
    const DdsMaskInfo* maskInfos
)
{
    const bool isLuminance = ((pfFlags & DDPF_LUMINANCE) != 0);
    const bool isSigned    = ((pfFlags & DDPF_BUMPDUDV ) != 0);
    const uint8_t defaultA = (isSigned) ? 0x7f : 0xff;
    const bool isA8 = IsA8Format(rgbBitCount, maskInfos);
    uint8_t* pDst = pDstPixels;
    const uint8_t* pSrc = pSrcPixels;
    const size_t dstPixelBytes = sizeof(uint32_t);
    const size_t srcPixelBytes = (rgbBitCount != 0) ?
        rgbBitCount / 8 : sizeof(uint32_t);
    const size_t pixelCount = dstDataSize / dstPixelBytes;
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        const uint32_t s =
            (rgbBitCount ==  8) ? pSrc[0]                                    :
            (rgbBitCount == 16) ? pSrc[0] | (pSrc[1] << 8)                   :
            (rgbBitCount == 24) ? pSrc[0] | (pSrc[1] << 8) | (pSrc[2] << 16) :
            pSrc[0] | (pSrc[1] << 8) | (pSrc[2] << 16) | (pSrc[3] << 24);
        const uint8_t r = GetDds8Bit((s & maskInfos[0].m_Mask) >> maskInfos[0].m_Shift, maskInfos[0].m_Size);
        const uint8_t g = GetDds8Bit((s & maskInfos[1].m_Mask) >> maskInfos[1].m_Shift, maskInfos[1].m_Size);
        const uint8_t b = GetDds8Bit((s & maskInfos[2].m_Mask) >> maskInfos[2].m_Shift, maskInfos[2].m_Size);
        const uint8_t a = (maskInfos[3].m_Mask != 0) ?
            GetDds8Bit((s & maskInfos[3].m_Mask) >> maskInfos[3].m_Shift, maskInfos[3].m_Size) : defaultA;
        if (!isA8)
        {
            pDst[0] = r;
            pDst[1] = (isLuminance) ? r : g;
            pDst[2] = (isLuminance) ? r : b;
            pDst[3] = a;
        }
        else
        {
            pDst[0] = pDst[1] = pDst[2] = a;
            pDst[3] = defaultA;
        }
        pDst += dstPixelBytes;
        pSrc += srcPixelBytes;
    }

    // 符号付き 8 ビットのピクセルデータを調整します。
    if (isSigned)
    {
        AdjustDdsSigned8(pDstPixels, pDstPixels, dstDataSize);
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの単純な 16bit float 型ピクセルデータをデコードします。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] ftxFormat テクスチャフォーマットです。
//-----------------------------------------------------------------------------
void DecodeDdsSimpleFloat16Pixels(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const FtxFormat ftxFormat
)
{
    const int srcCompCount =
        (ftxFormat == FtxFormat_Float_16   ) ? 1 :
        (ftxFormat == FtxFormat_Float_16_16) ? 2 :
        4;
    float* pDst = reinterpret_cast<float*>(pDstPixels);
    const uint8_t* pSrc = pSrcPixels;
    const size_t dstPixelBytes = sizeof(float) * R_RGBA_COUNT;
    const size_t pixelCount = dstDataSize / dstPixelBytes;
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        pDst[0] = GetDdsF16(pSrc + 0);
        pDst[1] = (srcCompCount >= 2) ? GetDdsF16(pSrc + 2) : 0.0f;
        pDst[2] = (srcCompCount >= 3) ? GetDdsF16(pSrc + 4) : 0.0f;
        pDst[3] = (srcCompCount >= 4) ? GetDdsF16(pSrc + 6) : 1.0f;
        pDst += R_RGBA_COUNT;
        pSrc += sizeof(uint16_t) * srcCompCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの単純な 32bit float 型ピクセルデータをデコードします。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] ftxFormat テクスチャフォーマットです。
//-----------------------------------------------------------------------------
void DecodeDdsSimpleFloat32Pixels(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const FtxFormat ftxFormat
)
{
    const int srcCompCount = RGetComponentCount(ftxFormat);
    float* pDst = reinterpret_cast<float*>(pDstPixels);
    const float* pSrc = reinterpret_cast<const float*>(pSrcPixels);
    const size_t dstPixelBytes = sizeof(float) * R_RGBA_COUNT;
    const size_t pixelCount = dstDataSize / dstPixelBytes;
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        pDst[0] = pSrc[0];
        pDst[1] = (srcCompCount >= 2) ? pSrc[1] : 0.0f;
        pDst[2] = (srcCompCount >= 3) ? pSrc[2] : 0.0f;
        pDst[3] = (srcCompCount >= 4) ? pSrc[3] : 1.0f;
        pDst += R_RGBA_COUNT;
        pSrc += srcCompCount;
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのピクセルデータをエンコーダでデコードします。
//!
//! @param[in,out] pEncoder エンコーダへのポインタです。
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] ftxFormat テクスチャフォーマットです。
//! @param[in] dimension テクスチャの次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] mipCount ミップマップのレベル数です。
//! @param[in] fourCc フォーマット情報です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool DecodeDdsPixelsByEncoder(
    REncoder* pEncoder,
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const FtxFormat ftxFormat,
    const FtxDimension dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount,
    const DdsFourCc fourCc
)
{
    //-----------------------------------------------------------------------------
    // エンコーダが初期化されていなければ初期化します。
    if (pEncoder == nullptr)
    {
        return false;
    }
    if (!pEncoder->IsInitialized())
    {
        RStatus status = pEncoder->Initialize(nullptr);
        if (!status)
        {
            return false;
        }
    }

    //-----------------------------------------------------------------------------
    // エンコーダでプレーンな RGBA フォーマットに変換します。
    const bool isFloat = RIsRealNumberTextureFormat(ftxFormat);
    const bool isSigned = RIsSignedTextureFormat(ftxFormat);
    const std::string dstFormat = (isFloat ) ? "float_32_32_32_32" :
                                  (isSigned) ? "snorm_8_8_8_8"     :
                                  "unorm_8_8_8_8";
    const std::string srcFormat = RGetTextureFormatString(ftxFormat);
    const texenc::ImageDimension encoderDim = REncoder::GetImageDimension(dimension);
    const bool isSucceeded = pEncoder->ConvertFormat(pDstPixels, pSrcPixels,
        RGetUnicodeFromShiftJis(dstFormat).c_str(),
        RGetUnicodeFromShiftJis(srcFormat).c_str(),
        L"", texenc::EncodeFlag_ReverseRgba,
        encoderDim, imageW, imageH, imageD, mipCount);
    if (!isSucceeded)
    {
        return false;
    }

    //-----------------------------------------------------------------------------
    // 符号付き 8 ビットのピクセルデータを調整します。
    if (dstFormat == "snorm_8_8_8_8")
    {
        AdjustDdsSigned8(pDstPixels, pDstPixels, dstDataSize);
    }

    //-----------------------------------------------------------------------------
    // ATI2 フォーマットなら R 成分と G 成分をスワップします。
    if (fourCc == DdsFourCc_Ati2)
    {
        SwapComponentRg(pDstPixels, pDstPixels, dstDataSize);
    }

    return true;
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのピクセルデータをデコードします。
//!
//! @param[in,out] pEncoder エンコーダへのポインタです。
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] ftxFormat テクスチャフォーマットです。
//! @param[in] dimension テクスチャの次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] mipCount ミップマップのレベル数です。
//! @param[in] pfFlags ピクセルフラグです。
//! @param[in] fourCc フォーマット情報です。
//! @param[in] rgbBitCount RGB フォーマットの場合の 1 ピクセルのビット数です。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//-----------------------------------------------------------------------------
void DecodeDdsPixels( // DecodeDdsPixelsT
    REncoder* pEncoder,
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const FtxFormat ftxFormat,
    const FtxDimension dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount,
    const uint32_t pfFlags,
    const DdsFourCc fourCc,
    const uint32_t rgbBitCount,
    const DdsMaskInfo* maskInfos
)
{
    if (ftxFormat == FtxFormat_Unorm_8_8_8_8 ||
        ftxFormat == FtxFormat_Srgb_8_8_8_8)
    {
        ConvertFromDdsB8G8R8A8(pDstPixels, pSrcPixels, dstDataSize, rgbBitCount, maskInfos);
    }
    else if (ftxFormat == FtxFormat_Float_16          ||
             ftxFormat == FtxFormat_Float_16_16       ||
             ftxFormat == FtxFormat_Float_16_16_16_16)
    {
        DecodeDdsSimpleFloat16Pixels(pDstPixels, pSrcPixels, dstDataSize, ftxFormat);
    }
    else if (ftxFormat == FtxFormat_Float_32          ||
             ftxFormat == FtxFormat_Float_32_32       ||
             ftxFormat == FtxFormat_Float_32_32_32    ||
             ftxFormat == FtxFormat_Float_32_32_32_32)
    {
        DecodeDdsSimpleFloat32Pixels(pDstPixels, pSrcPixels, dstDataSize, ftxFormat);
    }
    else if (fourCc == DdsFourCc_None)
    {
        DecodeDdsSimpleRgbaPixels(pDstPixels, pSrcPixels, dstDataSize, pfFlags, rgbBitCount, maskInfos);
    }
    else
    {
        const bool isSucceeded = DecodeDdsPixelsByEncoder(
            pEncoder, pDstPixels, pSrcPixels, dstDataSize,
            ftxFormat, dimension, imageW, imageH, imageD, mipCount, fourCc);
        if (!isSucceeded)
        {
            cerr << "Cannot decode dds pixels: " << RGetTextureFormatString(ftxFormat) << endl;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルの 1 フェースのデータサイズを取得します。
//!
//! @param[in] ftxFormat テクスチャフォーマットです。
//! @param[in] dxgiFormat dxgi フォーマットです。
//! @param[in] rgbBitCount RGB フォーマットの場合の 1 ピクセルのビット数です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] level ミップマップのレベルです。
//!
//! @return 1 フェースのデータサイズを返します。
//-----------------------------------------------------------------------------
size_t GetDdsFaceDataSize(
    const FtxFormat ftxFormat,
    const DdsDxgi dxgiFormat,
    const uint32_t rgbBitCount,
    const int imageW,
    const int imageH,
    const int level
)
{
    const int levelW = RMax(imageW >> level, 1);
    const int levelH = RMax(imageH >> level, 1);
    const size_t pixelCount = levelW * levelH;
    if (dxgiFormat == DdsDxgi_Unknown && rgbBitCount != 0)
    {
        return pixelCount * (rgbBitCount / 8);
    }
    else
    {
        return RGetTextureDataSize(nullptr, ftxFormat, FtxDimension_2d,
            levelW, levelH, 1, 1);
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのピクセルデータをデコードしたプレーンなデータを取得します。
//!
//! @param[in,out] pEncoder エンコーダへのポインタです。
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] dstLevelOfss 変換先の各ミップマップデータのイメージデータ先頭からのオフセット配列です。
//! @param[in] ftxFormat テクスチャフォーマットです。
//! @param[in] dimension テクスチャの次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] dxgiFormat dxgi フォーマットです。
//! @param[in] pfFlags ピクセルフラグです。
//! @param[in] fourCc フォーマット情報です。
//! @param[in] rgbBitCount RGB フォーマットの場合の 1 ピクセルのビット数です。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//-----------------------------------------------------------------------------
void GetDdsDecodedPixels(
    REncoder* pEncoder,
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const std::vector<size_t>& dstLevelOfss,
    const FtxFormat ftxFormat,
    const FtxDimension dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const DdsDxgi dxgiFormat,
    const uint32_t pfFlags,
    const DdsFourCc fourCc,
    const uint32_t rgbBitCount,
    const DdsMaskInfo* maskInfos
)
{
    const int mipCount = static_cast<int>(dstLevelOfss.size());
    const bool isCubeMap = RIsCubMapDimension(dimension);
    if (dimension == FtxDimension_3d || imageD == 1 || (mipCount == 1 && !isCubeMap))
    {
        DecodeDdsPixels(pEncoder, pDstPixels, pSrcPixels, dstDataSize,
            ftxFormat, dimension, imageW, imageH, imageD, mipCount,
            pfFlags, fourCc, rgbBitCount, maskInfos);
    }
    else
    {
        // キューブマップおよび配列テクスチャはミップマップの格納位置を変更します。
        // キューブマップはフェースの格納順も変更します。
        const uint8_t* pSrc = pSrcPixels;
        for (int iz = 0; iz < imageD; ++iz)
        {
            for (int level = 0; level < mipCount; ++level)
            {
                const int levelW = RMax(imageW >> level, 1);
                const int levelH = RMax(imageH >> level, 1);
                const size_t dstLevelOfs = dstLevelOfss[level];
                const size_t dstLevelBytes =
                    ((level < mipCount - 1) ? dstLevelOfss[level + 1] : dstDataSize) - dstLevelOfs;
                const size_t dstFaceBytes = dstLevelBytes / imageD;
                int dstFaceIdx = iz;
                if (isCubeMap)
                {
                    // DDS ファイルは左手座標系で +X、-X、+Y、-Y、+Z、-Z 面の順なので、
                    // 右手座標系で +X、-X、+Y、-Y、+Z、-Z 面の順になるように格納します。
                    int cubeFaceIdx = iz % RImage::CUBE_FACE_COUNT;
                    cubeFaceIdx = (cubeFaceIdx == 4) ? 5 :
                                  (cubeFaceIdx == 5) ? 4 :
                                  cubeFaceIdx;
                    dstFaceIdx = (iz / RImage::CUBE_FACE_COUNT) * RImage::CUBE_FACE_COUNT + cubeFaceIdx;
                }
                uint8_t* pDst = pDstPixels + dstLevelOfs + dstFaceBytes * dstFaceIdx;
                DecodeDdsPixels(pEncoder, pDst, pSrc, dstFaceBytes,
                    ftxFormat, dimension, levelW, levelH, 1, 1,
                    pfFlags, fourCc, rgbBitCount, maskInfos);
                pSrc += GetDdsFaceDataSize(ftxFormat, dxgiFormat, rgbBitCount,
                    imageW, imageH, level);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのエンコード済みピクセルデータを取得します。
//!        必要に応じてテクスチャコンバータが扱う形式に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] dstLevelOfss 変換先の各ミップマップデータのイメージデータ先頭からのオフセット配列です。
//! @param[in] ftxFormat テクスチャフォーマットです。
//! @param[in] dimension テクスチャの次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] dxgiFormat dxgi フォーマットです。
//! @param[in] fourCc フォーマット情報です。
//! @param[in] rgbBitCount RGB フォーマットの場合の 1 ピクセルのビット数です。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//-----------------------------------------------------------------------------
void GetDdsEncodedPixels(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const std::vector<size_t>& dstLevelOfss,
    const FtxFormat ftxFormat,
    const FtxDimension dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const DdsDxgi dxgiFormat,
    const DdsFourCc fourCc,
    const uint32_t rgbBitCount,
    const DdsMaskInfo* maskInfos
)
{
    const int mipCount = static_cast<int>(dstLevelOfss.size());
    if (dimension == FtxDimension_3d || imageD == 1 || mipCount == 1)
    {
        ConvertFromDdsPixels(pDstPixels, pSrcPixels, dstDataSize,
            ftxFormat, fourCc, rgbBitCount, maskInfos);
    }
    else
    {
        // キューブマップおよび配列テクスチャはミップマップの格納位置を変更します。
        const uint8_t* pSrc = pSrcPixels;
        for (int iz = 0; iz < imageD; ++iz)
        {
            for (int level = 0; level < mipCount; ++level)
            {
                const size_t dstLevelOfs = dstLevelOfss[level];
                const size_t dstLevelBytes =
                    ((level < mipCount - 1) ? dstLevelOfss[level + 1] : dstDataSize) - dstLevelOfs;
                const size_t dstFaceBytes = dstLevelBytes / imageD;
                uint8_t* pDst = pDstPixels + dstLevelOfs + dstFaceBytes * iz;
                ConvertFromDdsPixels(pDst, pSrc, dstFaceBytes,
                    ftxFormat, fourCc, rgbBitCount, maskInfos);
                pSrc += GetDdsFaceDataSize(ftxFormat, dxgiFormat, rgbBitCount,
                    imageW, imageH, level);
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのマスク情報で表現できるフォーマットなら true を返します。
//!
//! @param[in] ftxFormat テクスチャフォーマットです。
//!
//! @return マスク情報で表現できるフォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool IsDdsMaskFormat(const FtxFormat ftxFormat)
{
    switch (ftxFormat)
    {
    case FtxFormat_Unorm_8:
    case FtxFormat_Unorm_16:
    case FtxFormat_Unorm_8_8:
    case FtxFormat_Snorm_8_8:
    case FtxFormat_Unorm_16_16:
    case FtxFormat_Snorm_16_16:
    case FtxFormat_Unorm_5_6_5:
    case FtxFormat_Unorm_5_5_5_1:
    case FtxFormat_Unorm_4_4_4_4:
    case FtxFormat_Unorm_8_8_8_8:
    case FtxFormat_Snorm_8_8_8_8:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャフォーマットに対応した DDS ファイルのマスク情報を設定します。
//!
//! @param[out] maskInfos RGBA 成分のマスク情報を格納します。長さ 4 の配列を指定します。
//! @param[in] ftxFormat テクスチャフォーマットです。
//! @param[in] hasAlpha アルファ成分を持つなら true です。
//-----------------------------------------------------------------------------
void SetDdsMaskInfos(
    DdsMaskInfo* maskInfos,
    const FtxFormat ftxFormat,
    const bool hasAlpha
)
{
    uint32_t mr = 0x00000000;
    uint32_t mg = 0x00000000;
    uint32_t mb = 0x00000000;
    uint32_t ma = 0x00000000;

    switch (ftxFormat)
    {
    case FtxFormat_Unorm_8:
        mr = 0x000000ff;
        break;

    case FtxFormat_Unorm_16:
        mr = 0x0000ffff;
        break;

    case FtxFormat_Unorm_8_8:
        mr = 0x000000ff;
        if (hasAlpha)
        {
            ma = 0x0000ff00;
        }
        else
        {
            mg = 0x0000ff00;
        }
        break;

    case FtxFormat_Snorm_8_8:
        mr = 0x000000ff;
        mg = 0x0000ff00;
        break;

    case FtxFormat_Unorm_16_16:
    case FtxFormat_Snorm_16_16:
        mr = 0x0000ffff;
        mg = 0xffff0000;
        break;

    case FtxFormat_Unorm_5_6_5:
        mr = 0x0000f800;
        mg = 0x000007e0;
        mb = 0x0000001f;
        break;

    case FtxFormat_Unorm_5_5_5_1:
        mr = 0x00007c00;
        mg = 0x000003e0;
        mb = 0x0000001f;
        ma = 0x00008000;
        break;

    case FtxFormat_Unorm_4_4_4_4:
        mr = 0x00000f00;
        mg = 0x000000f0;
        mb = 0x0000000f;
        ma = 0x0000f000;
        break;

    case FtxFormat_Unorm_8_8_8_8:
        mr = 0x00ff0000;
        mg = 0x0000ff00;
        mb = 0x000000ff;
        ma = (hasAlpha) ? 0xff000000 : 0x00000000;
        break;

    case FtxFormat_Snorm_8_8_8_8:
        mr = 0x000000ff;
        mg = 0x0000ff00;
        mb = 0x00ff0000;
        ma = 0xff000000;
        break;

    default:
        break;
    };

    maskInfos[0] = DdsMaskInfo(mr);
    maskInfos[1] = DdsMaskInfo(mg);
    maskInfos[2] = DdsMaskInfo(mb);
    maskInfos[3] = DdsMaskInfo(ma);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャフォーマットに対応した DDS ファイルのフォーマット情報を取得します。
//!
//! @param[in] ftxFormat テクスチャフォーマットです。
//!
//! @return フォーマット情報を返します。
//-----------------------------------------------------------------------------
DdsFourCc GetDdsFourCcFormat(const FtxFormat ftxFormat)
{
    switch (ftxFormat)
    {
    case FtxFormat_Unorm_16_16_16_16: return DdsFourCc_A16B16G16R16;
    case FtxFormat_Snorm_16_16_16_16: return DdsFourCc_Q16W16V16U16;
    case FtxFormat_Float_16:          return DdsFourCc_R16F;
    case FtxFormat_Float_16_16:       return DdsFourCc_G16R16F;
    case FtxFormat_Float_16_16_16_16: return DdsFourCc_A16B16G16R16F;
    case FtxFormat_Float_32:          return DdsFourCc_R32F;
    case FtxFormat_Float_32_32:       return DdsFourCc_G32R32F;
    case FtxFormat_Float_32_32_32_32: return DdsFourCc_A32B32G32R32F;
    case FtxFormat_Unorm_Bc1:         return DdsFourCc_Dxt1;
    case FtxFormat_Unorm_Bc2:         return DdsFourCc_Dxt3;
    case FtxFormat_Unorm_Bc3:         return DdsFourCc_Dxt5;
    case FtxFormat_Unorm_Bc4:         return DdsFourCc_Bc4U;
    case FtxFormat_Snorm_Bc4:         return DdsFourCc_Bc4S;
    case FtxFormat_Unorm_Bc5:         return DdsFourCc_Bc5U;
    case FtxFormat_Snorm_Bc5:         return DdsFourCc_Bc5S;
    default:                          return DdsFourCc_None;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャフォーマットに対応した DDS ファイルの dxgi フォーマットを取得します。
//!
//! @param[in] ftxFormat テクスチャフォーマットです。
//!
//! @return dxgi フォーマットを返します。
//-----------------------------------------------------------------------------
DdsDxgi GetDdsDxgiFormat(const FtxFormat ftxFormat)
{
    const int formatCount = sizeof(DdsDxgiAndFtxFormats) / sizeof(DdsDxgiAndFtxFormats[0]);
    for (int formatIdx = 0; formatIdx < formatCount; ++formatIdx)
    {
        const DdsDxgiAndFtxFormat& formatInfo = DdsDxgiAndFtxFormats[formatIdx];
        if (formatInfo.ftxFormat == ftxFormat)
        {
            return formatInfo.dxgiFormat;
        }
    }
    return DdsDxgi_Unknown;
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのヘッダを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] isDx10 DX10 形式なら true です。
//! @param[in] ftxFormat テクスチャフォーマットです。
//! @param[in] dxgiFormat dxgi フォーマットです。
//! @param[in] fcFormat フォーマット情報です。
//! @param[in] dimension テクスチャの次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] mipCount ミップマップのレベル数です。
//! @param[in] usesMask マスク情報を使用するなら true です。
//! @param[in] maskInfos RGBA 成分のマスク情報配列です。
//! @param[in] isBgr8 ピクセルデータが B8G8R8 形式なら true です。
//-----------------------------------------------------------------------------
void OutputDdsHeader(
    std::ostream& os,
    const bool isDx10,
    const FtxFormat ftxFormat,
    const DdsDxgi dxgiFormat,
    const DdsFourCc fcFormat,
    const FtxDimension dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount,
    const bool usesMask,
    const DdsMaskInfo* maskInfos,
    const bool isBgr8
)
{
    const bool is1d = (dimension == FtxDimension_1d || dimension == FtxDimension_1dArray);
    const bool is3d = (dimension == FtxDimension_3d);
    const bool isArray = RIsArrayDimension(dimension);
    const bool isCubeMap = RIsCubMapDimension(dimension);
    const bool isCubeArray = (dimension == FtxDimension_CubeMapArray);

    const bool usesLinearSize = RIsCompressedTextureFormat(ftxFormat);
    const size_t srcFaceBytes = RGetTextureDataSize(nullptr,
        ftxFormat, dimension, imageW, imageH, 1, 1);
    const size_t dstFaceBytes = (isBgr8) ? srcFaceBytes / 4 * 3 : srcFaceBytes;

    //-----------------------------------------------------------------------------
    // 通常のヘッダを出力します。
    const bool usesDepth = (is3d || (!isDx10 && isArray));
    const bool usesMipMap = (mipCount >= 2);
    DdsHeader header;
    header.dwWidth  = imageW;
    header.dwHeight = imageH;
    header.dwDepth  = (usesDepth) ? imageD : 1;
    header.dwMipMapCount = (usesMipMap) ? mipCount : 1;
    header.dwPitchOrLinearSize = static_cast<uint32_t>(
        (usesLinearSize) ? dstFaceBytes : dstFaceBytes / imageH);
    header.dwFlags = DDSD_CAPS | DDSD_HEIGHT | DDSD_WIDTH | DDSD_PIXELFORMAT |
        ((usesLinearSize) ? DDSD_LINEARSIZE  : DDSD_PITCH) |
        ((usesDepth     ) ? DDSD_DEPTH       :          0) |
        ((usesMipMap    ) ? DDSD_MIPMAPCOUNT :          0);
    header.dwCaps = DDSCAPS_TEXTURE |
        ((usesMipMap || usesDepth || isCubeMap) ? DDSCAPS_COMPLEX : 0) |
        ((usesMipMap                          ) ? DDSCAPS_MIPMAP  : 0);
    header.dwCaps2 =
        ((is3d     ) ? DDSCAPS2_VOLUME                       : 0) |
        ((isCubeMap) ? DDSCAPS2_CUBEMAP | DDSCAPS2_CUBEFACES : 0);
    if (isDx10 || !usesMask)
    {
        header.dwPfFlags = DDPF_FOURCC;
    }
    else
    {
        const bool hasAlpha = (maskInfos[3].m_Size != 0);
        const bool isLa8 = (ftxFormat == FtxFormat_Unorm_8_8 && hasAlpha);
        header.dwPfFlags = DDPF_RGB |
            ((hasAlpha                                   ) ? DDPF_ALPHAPIXELS : 0) |
            ((RGetComponentCount(ftxFormat) == 1 || isLa8) ? DDPF_LUMINANCE   : 0) |
            ((RIsSignedTextureFormat(ftxFormat)          ) ? DDPF_BUMPDUDV    : 0);
        header.dwRGBBitCount = maskInfos[0].m_Size +
                               maskInfos[1].m_Size +
                               maskInfos[2].m_Size +
                               maskInfos[3].m_Size;
        header.dwRBitMask        = maskInfos[0].m_Mask;
        header.dwGBitMask        = maskInfos[1].m_Mask;
        header.dwBBitMask        = maskInfos[2].m_Mask;
        header.dwRGBAlphaBitMask = maskInfos[3].m_Mask;
    }
    header.dwFourCC = (isDx10) ? DdsFourCc_Dx10 : fcFormat;
    os.write(reinterpret_cast<const char*>(&header), sizeof(header));

    //-----------------------------------------------------------------------------
    // DX10 ヘッダを出力します。
    if (isDx10)
    {
        DdsDx10Header dx10Header;
        dx10Header.dxgiFormat = dxgiFormat;
        dx10Header.resourceDimension = (is1d) ? DdsResDimension_Texture1d :
                                       (is3d) ? DdsResDimension_Texture3d :
                                       DdsResDimension_Texture2d;
        dx10Header.miscFlag = (isCubeMap) ? DdsResMisc_TextureCube : 0;
        dx10Header.arraySize = (isCubeArray) ? (imageD / RImage::CUBE_FACE_COUNT) :
                               (isArray    ) ? imageD                             :
                               1;
        os.write(reinterpret_cast<const char*>(&dx10Header), sizeof(dx10Header));
    }
}

//-----------------------------------------------------------------------------
//! @brief ピクセルデータを DDS ファイルの B5G6R5 形式に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//-----------------------------------------------------------------------------
void ConvertToDdsB5G6R5(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize
)
{
    uint8_t* pDst = pDstPixels;
    const uint8_t* pSrc = pSrcPixels;
    const size_t pixelCount = dstDataSize / sizeof(uint16_t);
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        const uint32_t s = pSrc[0] | (pSrc[1] << 8);
        const uint32_t r = (s      ) & 0x1f;
        const uint32_t g = (s >>  5) & 0x3f;
        const uint32_t b = (s >> 11) & 0x1f;
        const uint32_t d = b | (g << 5) | (r << 11);
        pDst[0] = static_cast<uint8_t>((d     ) & 0xff);
        pDst[1] = static_cast<uint8_t>((d >> 8) & 0xff);
        pDst += sizeof(uint16_t);
        pSrc += sizeof(uint16_t);
    }
}

//-----------------------------------------------------------------------------
//! @brief ピクセルデータを DDS ファイルの B5G5R5A1 形式に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//-----------------------------------------------------------------------------
void ConvertToDdsB5G5R5A1(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize
)
{
    uint8_t* pDst = pDstPixels;
    const uint8_t* pSrc = pSrcPixels;
    const size_t pixelCount = dstDataSize / sizeof(uint16_t);
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        const uint32_t s = pSrc[0] | (pSrc[1] << 8);
        const uint32_t r = (s      ) & 0x1f;
        const uint32_t g = (s >>  5) & 0x1f;
        const uint32_t b = (s >> 10) & 0x1f;
        const uint32_t a = (s >> 15) & 0x01;
        const uint32_t d = b | (g << 5) | (r << 10) | (a << 15);
        pDst[0] = static_cast<uint8_t>((d     ) & 0xff);
        pDst[1] = static_cast<uint8_t>((d >> 8) & 0xff);
        pDst += sizeof(uint16_t);
        pSrc += sizeof(uint16_t);
    }
}

//-----------------------------------------------------------------------------
//! @brief ピクセルデータを DDS ファイルの B4G4R4A4 形式に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//-----------------------------------------------------------------------------
void ConvertToDdsB4G4R4A4(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize
)
{
    uint8_t* pDst = pDstPixels;
    const uint8_t* pSrc = pSrcPixels;
    const size_t pixelCount = dstDataSize / sizeof(uint16_t);
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        const uint32_t s = pSrc[0] | (pSrc[1] << 8);
        const uint32_t r = (s      ) & 0x0f;
        const uint32_t g = (s >>  4) & 0x0f;
        const uint32_t b = (s >>  8) & 0x0f;
        const uint32_t a = (s >> 12) & 0x0f;
        const uint32_t d = b | (g << 4) | (r << 8) | (a << 12);
        pDst[0] = static_cast<uint8_t>((d     ) & 0xff);
        pDst[1] = static_cast<uint8_t>((d >> 8) & 0xff);
        pDst += sizeof(uint16_t);
        pSrc += sizeof(uint16_t);
    }
}

//-----------------------------------------------------------------------------
//! @brief ピクセルデータを DDS ファイルの B8G8R8 / B8G8R8A8 形式に変換します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] isBgr8 ピクセルデータが B8G8R8 形式なら true です。
//-----------------------------------------------------------------------------
void ConvertToDdsB8G8R8A8(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const bool isBgr8
)
{
    uint8_t* pDst = pDstPixels;
    const uint8_t* pSrc = pSrcPixels;
    const size_t dstPixelBytes = (isBgr8) ? R_RGB_BYTES : R_RGBA_BYTES;
    const size_t pixelCount = dstDataSize / dstPixelBytes;
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        const uint8_t r = pSrc[0];
        const uint8_t g = pSrc[1];
        const uint8_t b = pSrc[2];
        const uint8_t a = pSrc[3];
        pDst[2] = r;
        pDst[1] = g;
        pDst[0] = b;
        if (!isBgr8)
        {
            pDst[3] = a;
        }
        pDst += dstPixelBytes;
        pSrc += R_RGBA_BYTES;
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのピクセルデータを部分的に設定します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] srcLevelOfss 変換元の各ミップマップデータのイメージデータ先頭からのオフセット配列です。
//! @param[in] ftxFormat テクスチャフォーマットです。
//! @param[in] usesMask マスク情報を使用するなら true です。
//! @param[in] isBgr8 ピクセルデータが B8G8R8 形式なら true です。
//-----------------------------------------------------------------------------
void SetDdsSubPixels(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const FtxFormat ftxFormat,
    const bool usesMask,
    const bool isBgr8
)
{
    if (ftxFormat == FtxFormat_Unorm_5_6_5)
    {
        ConvertToDdsB5G6R5(pDstPixels, pSrcPixels, dstDataSize);
    }
    else if (ftxFormat == FtxFormat_Unorm_5_5_5_1)
    {
        ConvertToDdsB5G5R5A1(pDstPixels, pSrcPixels, dstDataSize);
    }
    else if (ftxFormat == FtxFormat_Unorm_4_4_4_4)
    {
        ConvertToDdsB4G4R4A4(pDstPixels, pSrcPixels, dstDataSize);
    }
    else if (usesMask && ftxFormat == FtxFormat_Unorm_8_8_8_8)
    {
        ConvertToDdsB8G8R8A8(pDstPixels, pSrcPixels, dstDataSize, isBgr8);
    }
    else
    {
        memcpy(pDstPixels, pSrcPixels, dstDataSize);
    }
}

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのピクセルデータを設定します。
//!
//! @param[out] pDstPixels 変換先ピクセルデータへのポインタです。
//! @param[in] pSrcPixels 変換元ピクセルデータへのポインタです。
//! @param[in] dstDataSize 変換先ピクセルデータのサイズです。
//! @param[in] srcDataSize 変換元ピクセルデータのサイズです。
//! @param[in] srcLevelOfss 変換元の各ミップマップデータのイメージデータ先頭からのオフセット配列です。
//! @param[in] ftxFormat テクスチャフォーマットです。
//! @param[in] dimension テクスチャの次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] usesMask マスク情報を使用するなら true です。
//! @param[in] isBgr8 ピクセルデータが B8G8R8 形式なら true です。
//! @param[in] swapsCubeZ キューブマップの場合に Z 方向のイメージデータをスワップして出力するなら true を指定します。
//-----------------------------------------------------------------------------
void SetDdsPixels(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const size_t dstDataSize,
    const size_t srcDataSize,
    const std::vector<size_t>& srcLevelOfss,
    const FtxFormat ftxFormat,
    const FtxDimension dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const bool usesMask,
    const bool isBgr8,
    const bool swapsCubeZ
)
{
    const int mipCount = static_cast<int>(srcLevelOfss.size());
    const bool isCubeMap = RIsCubMapDimension(dimension);
    if (dimension == FtxDimension_3d || imageD == 1 || (mipCount == 1 && !isCubeMap))
    {
        SetDdsSubPixels(pDstPixels, pSrcPixels, dstDataSize,
            ftxFormat, usesMask, isBgr8);
    }
    else
    {
        // キューブマップおよび配列テクスチャはミップマップの格納位置を変更します。
        // 必要ならキューブマップのフェースの格納順も変更します。
        uint8_t* pDst = pDstPixels;
        for (int iz = 0; iz < imageD; ++iz)
        {
            for (int level = 0; level < mipCount; ++level)
            {
                const int levelW = RMax(imageW >> level, 1);
                const int levelH = RMax(imageH >> level, 1);
                const size_t srcLevelOfs = srcLevelOfss[level];
                const size_t srcLevelBytes =
                    ((level < mipCount - 1) ? srcLevelOfss[level + 1] : srcDataSize) - srcLevelOfs;
                const size_t srcFaceBytes = srcLevelBytes / imageD;
                const size_t dstFaceBytes = (isBgr8) ? srcFaceBytes / 4 * 3 : srcFaceBytes;
                int srcFaceIdx = iz;
                if (isCubeMap && swapsCubeZ)
                {
                    int cubeFaceIdx = iz % RImage::CUBE_FACE_COUNT;
                    cubeFaceIdx = (cubeFaceIdx == 4) ? 5 :
                                  (cubeFaceIdx == 5) ? 4 :
                                  cubeFaceIdx;
                    srcFaceIdx = (iz / RImage::CUBE_FACE_COUNT) * RImage::CUBE_FACE_COUNT + cubeFaceIdx;
                }
                const uint8_t* pSrc = pSrcPixels + srcLevelOfs + srcFaceBytes * srcFaceIdx;
                SetDdsSubPixels(pDst, pSrc, dstFaceBytes,
                    ftxFormat, usesMask, isBgr8);
                pDst += dstFaceBytes;
            }
        }
    }
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief DDS ファイルのデータをデコードします。
//-----------------------------------------------------------------------------
RStatus RImage::DecodeDds( // DecodeDdsT
    REncoder* pEncoder,
    const uint8_t* fileBuf,
    const size_t fileSize,
    const int readFlags
)
{
    //-----------------------------------------------------------------------------
    // ヘッダを解析します。
    if (fileSize < 0x80)
    {
        return RStatus(RStatus::FAILURE, "DDS file is wrong: (invalid size): " + m_FilePath); // RShowError
    }
    if (std::string(reinterpret_cast<const char*>(fileBuf), 4) != "DDS ")
    {
        return RStatus(RStatus::FAILURE, "DDS file is wrong: (invalid signature): " + m_FilePath); // RShowError
    }
    const uint32_t headerSize = RGetMemLittleUInt(fileBuf + 0x04); // 124
    const uint32_t flags = RGetMemLittleInt(fileBuf + 0x08);
    m_ImageH = RMax(RGetMemLittleInt(fileBuf + 0x0c), 1);
    m_ImageW = RGetMemLittleInt(fileBuf + 0x10);
    m_ImageD = ((flags & DDSD_DEPTH) != 0) ? RMax(RGetMemLittleInt(fileBuf + 0x18), 1) : 1;
    m_MipCount = RMax(RGetMemLittleInt(fileBuf + 0x1c), 1);

    const uint32_t pfFlags = RGetMemLittleUInt(fileBuf + 0x50);
    const DdsFourCc fourCc = ((pfFlags & DDPF_FOURCC) != 0) ?
        static_cast<DdsFourCc>(RGetMemLittleUInt(fileBuf + 0x54)) : DdsFourCc_None;
    const uint32_t rgbBitCount = RGetMemLittleUInt(fileBuf + 0x58);
    const uint32_t maskR = RGetMemLittleUInt(fileBuf + 0x5c);
    const uint32_t maskG = RGetMemLittleUInt(fileBuf + 0x60);
    const uint32_t maskB = RGetMemLittleUInt(fileBuf + 0x64);
    const uint32_t maskA = RGetMemLittleUInt(fileBuf + 0x68);
    const uint32_t caps2 = RGetMemLittleUInt(fileBuf + 0x70);

    //-----------------------------------------------------------------------------
    // DX10 ヘッダを解析します。
    DdsDxgi dxgiFormat = DdsDxgi_Unknown;
    DdsResDimension resourceDimension = DdsResDimension_Texture2d;
    uint32_t miscFlag = 0;
    uint32_t arraySize = 1;
    if (fourCc == DdsFourCc_Dx10)
    {
        dxgiFormat        = static_cast<DdsDxgi>(RGetMemLittleUInt(fileBuf + 0x80));
        resourceDimension = static_cast<DdsResDimension>(RGetMemLittleUInt(fileBuf + 0x84));
        miscFlag          = RGetMemLittleUInt(fileBuf + 0x88);
        arraySize         = RMax(RGetMemLittleUInt(fileBuf + 0x8c), static_cast<uint32_t>(1));
    }

    //-----------------------------------------------------------------------------
    // フォーマット情報によっては dxgi フォーマットとして扱います。
    dxgiFormat = (fourCc == DdsFourCc_A8R8G8B8) ? DdsDxgi_B8G8R8A8_Unorm :
                 (fourCc == DdsFourCc_X8R8G8B8) ? DdsDxgi_B8G8R8X8_Unorm :
                 (fourCc == DdsFourCc_R5G6B5  ) ? DdsDxgi_B5G6R5_Unorm   :
                 (fourCc == DdsFourCc_A1R5G5B5) ? DdsDxgi_B5G5R5A1_Unorm :
                 (fourCc == DdsFourCc_A4R4G4B4) ? DdsDxgi_B4G4R4A4_Unorm :
                 dxgiFormat;

    //-----------------------------------------------------------------------------
    // マスク情報を取得します。
    DdsMaskInfo maskInfos[R_RGBA_COUNT];
    GetDdsMaskInfos(maskInfos, maskR, maskG, maskB, maskA, dxgiFormat);

    //-----------------------------------------------------------------------------
    // テクスチャフォーマットを取得します。
    const FtxFormat ftxFormat = GetFormatFromDds(
        dxgiFormat, pfFlags, fourCc, rgbBitCount, maskInfos);
    if (ftxFormat == FtxFormat_Invalid)
    {
        return RStatus(RStatus::FAILURE, "DDS file is wrong: (unsupported format): " + m_FilePath); // RShowError
    }
    m_Format = ftxFormat;
    m_IsFloat = RIsRealNumberTextureFormat(ftxFormat); // unorm_16 / snorm_16 系も実数扱いです。
    m_LinearFlag = (RIsSrgbFetchTextureFormat(ftxFormat)) ?
        FtxOpt::LINEAR_RGB : FtxOpt::LINEAR_NONE;

    //-----------------------------------------------------------------------------
    // テクスチャの次元を取得します。同時に奥行きを調整します。
    m_Dimension = GetDimensionFromDds(&m_ImageD,
        fourCc, caps2, resourceDimension, miscFlag, arraySize);

    //-----------------------------------------------------------------------------
    // アルファ成分の有無を取得します。
    m_HasAlpha = DdsHasAlpha(ftxFormat, dxgiFormat, fourCc, maskInfos);

    //-----------------------------------------------------------------------------
    // フォーマットによっては、デフォルトが成分選択が適切でないので、
    // 特別に成分選択を設定します。
    const bool isA8 = IsA8Format(rgbBitCount, maskInfos);
    const bool isRg = (RGetComponentCount(ftxFormat) == 2 && !m_HasAlpha);
    m_CompSel =
        (isA8) ? RGetCompSelFromString("0 0 0 r") : // unorm_8
        (isRg) ? RGetCompSelFromString("r g 0 1") : // unorm_8_8
        m_CompSel; // デフォルト

    //-----------------------------------------------------------------------------
    // ピクセルデータをデコードします。
    const uint8_t* pDdsPixels = fileBuf + 4 + headerSize +
        ((fourCc == DdsFourCc_Dx10) ? sizeof(DdsDx10Header) : 0);
    const bool getsOriginal = ((readFlags & ReadFlag_Original) != 0);
    if (getsOriginal)
    {
        const FtxFormat plainFormat =
            (m_IsFloat) ? FtxFormat_Float_32_32_32_32 : FtxFormat_Unorm_8_8_8_8;
        std::vector<size_t> originalLevelOfss;
        m_ImageDataSize = RGetTextureDataSize(&originalLevelOfss,
            plainFormat, static_cast<FtxDimension>(m_Dimension),
            m_ImageW, m_ImageH, m_ImageD, m_MipCount);
        m_pImageData = new uint8_t[m_ImageDataSize];
        m_MipDataSize = (m_MipCount >= 2) ? m_ImageDataSize - originalLevelOfss[1] : 0;
        GetDdsDecodedPixels(pEncoder,
            m_pImageData, pDdsPixels, m_ImageDataSize, originalLevelOfss,
            ftxFormat, static_cast<FtxDimension>(m_Dimension),
            m_ImageW, m_ImageH, m_ImageD, dxgiFormat, pfFlags, fourCc, rgbBitCount, maskInfos);
    }

    //-----------------------------------------------------------------------------
    // エンコードされたピクセルデータを取得します。
    // 必要に応じてテクスチャコンバータが扱う形式に変換します。
    if ((readFlags & ReadFlag_Encoded) != 0)
    {
        std::vector<size_t> encodedLevelOfss;
        m_EncodedDataSize = RGetTextureDataSize(&encodedLevelOfss,
            ftxFormat, static_cast<FtxDimension>(m_Dimension),
            m_ImageW, m_ImageH, m_ImageD, m_MipCount);
        m_pEncodedData = new uint8_t[m_EncodedDataSize];
        SetLevelOffsets(encodedLevelOfss);
        GetDdsEncodedPixels(m_pEncodedData, pDdsPixels, m_EncodedDataSize, encodedLevelOfss,
            ftxFormat, static_cast<FtxDimension>(m_Dimension),
            m_ImageW, m_ImageH, m_ImageD, dxgiFormat, fourCc, rgbBitCount, maskInfos);

        m_DccPreset = "";
        m_Hint = RGetDefaultHint(ftxFormat);
        m_MipGenFilter = FtxOpt::MIP_GEN_LINEAR;
        m_CompSel = (m_CompSel != FtxOpt::COMP_SEL_DEFAULT) ?
            m_CompSel : RGetDefaultCompSel(ftxFormat);
        m_WeightedCompress = false;
        m_InitialSwizzle = 0;

        if (!getsOriginal)
        {
            MoveEncodedDataToImageData();
        }
    }

    return RStatus::SUCCESS;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief DDS ファイルを出力します。
//-----------------------------------------------------------------------------
RStatus RImage::OutputDdsFile(
    const std::string& filePath,
    const bool outputsEncoded,
    const bool outputsMip,
    const bool swapsCubeZ,
    const bool forcesDx10
) const
{
    //-----------------------------------------------------------------------------
    // 出力フォーマットを決定します。
    const FtxFormat ftxFormat =
        (outputsEncoded) ? static_cast<FtxFormat>(m_Format) :
        (IsFloat()     ) ? FtxFormat_Float_32_32_32_32      :
        FtxFormat_Unorm_8_8_8_8;
    const DdsDxgi dxgiFormat = GetDdsDxgiFormat(ftxFormat);
    if (dxgiFormat == DdsDxgi_Unknown)
    {
        return RStatus(RStatus::FAILURE, "Unsupported format for DDS file: " + // RShowError;
            RGetTextureFormatString(ftxFormat));
    }

    //-----------------------------------------------------------------------------
    // DX10 形式かどうかを決定します。
    const FtxDimension dimension = static_cast<FtxDimension>(m_Dimension);
    const bool is1d = (dimension == FtxDimension_1d || dimension == FtxDimension_1dArray);
    const DdsFourCc fcFormat = GetDdsFourCcFormat(ftxFormat);
    const bool isDx10 = (forcesDx10 || is1d || IsArray() ||
        (!IsDdsMaskFormat(ftxFormat) && fcFormat == DdsFourCc_None));

    //-----------------------------------------------------------------------------
    // マスク情報を設定します。
    const bool usesMask = (!isDx10 && IsDdsMaskFormat(ftxFormat));
    DdsMaskInfo maskInfos[R_RGBA_COUNT];
    if (usesMask)
    {
        SetDdsMaskInfos(maskInfos, ftxFormat, m_HasAlpha);
    }
    const bool isBgr8 = (usesMask && ftxFormat == FtxFormat_Unorm_8_8_8_8 &&
        maskInfos[3].m_Size == 0);

    //-----------------------------------------------------------------------------
    // ファイルをオープンします。
    std::ofstream ofs(filePath.c_str(), std::ios::binary);
    if (!ofs)
    {
        return RStatus(RStatus::FAILURE, "Cannot open the file: " + filePath); // RShowError;
    }

    //-----------------------------------------------------------------------------
    // ヘッダを出力します。
    const int mipCount = (outputsMip) ? m_MipCount : 1;
    OutputDdsHeader(ofs, isDx10, ftxFormat, dxgiFormat, fcFormat,
        dimension, m_ImageW, m_ImageH, m_ImageD, mipCount,
        usesMask, maskInfos, isBgr8);

    //-----------------------------------------------------------------------------
    // ピクセルデータを出力します。
    std::vector<size_t> srcLevelOfss;
    const size_t srcDataSize = RGetTextureDataSize(&srcLevelOfss,
        ftxFormat, dimension, m_ImageW, m_ImageH, m_ImageD, mipCount);
    const size_t dstDataSize = (isBgr8) ? srcDataSize / 4 * 3 : srcDataSize;

    const uint8_t* pSrcPixels = (outputsEncoded && m_pEncodedData != nullptr) ?
        m_pEncodedData : m_pImageData;
    uint8_t* pDstPixels = new uint8_t[dstDataSize];
    SetDdsPixels(pDstPixels, pSrcPixels, dstDataSize, srcDataSize, srcLevelOfss,
        ftxFormat, dimension, m_ImageW, m_ImageH, m_ImageD,
        usesMask, isBgr8, swapsCubeZ);
    ofs.write(reinterpret_cast<const char*>(pDstPixels), dstDataSize);
    delete[] pDstPixels;

    return RStatus::SUCCESS;
}

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

