﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// OutputBntxData ConvertTiling ConvertUserData AddStrings AddMemoryBlocks
// SwapBntxEndian SwapUserDataEndian
// Info_SetData Info_SetUserDataArray SetUserDataObjectAndValue
// DisplayBntxInfo

// TexInfoBlock

//=============================================================================
// include
//=============================================================================
#pragma warning(push)
    #pragma warning(disable: 4996) // 'std::_Copy_impl': Function call with parameters that may be unsafe
    #ifndef NOMINMAX
    #define NOMINMAX
    #endif
    #include "BinaryFormat.h"
#pragma warning(pop)

#ifndef NN_GFX_CONFIG_INCLUDE_GL
#define NN_GFX_CONFIG_INCLUDE_GL 1
#endif

#define NN_GFX_CONFIG_INCLUDE_GX 1
#define NN_GFX_CONFIG_INCLUDE_NVN 1
#define NN_GFX_CONFIG_INCLUDE_VK 1

#include <nn/util/util_ResEndian.h>
#include <nn/utilTool/utilTool_BinarizerUtil.h>
#include <nn/gfx/gfx_MemoryPool.h>
#include <nn/gfx/gfx_Texture.h>
#include <nn/gfx/gfx_ResTexture.h>

// nn ライブラリのソースを include します。
// nn ライブラリを /MT でビルドするためです。
#include "../../../../../../../Alice/Sources/Libraries/util/util_BinaryFormat.cpp"
#include "../../../../../../../Alice/Sources/Libraries/util/util_IntrusiveDic.cpp"
#include "../../../../../../../Alice/Sources/Libraries/util/util_ResDic.cpp" // 辞書を使用するなら必要
#include "../../../../../../../Alice/Sources/Libraries/util/util_ResEndian.cpp"
#include "../../../../../../../Alice/Sources/Libraries/utilTool/utilTool_BinarizerContext.cpp"
#include "../../../../../../../Alice/Sources/Libraries/utilTool/detail/utilTool_BinarizerContextImpl.cpp"

using namespace std;
using namespace nn;

//=============================================================================
// texcvtr ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace texcvtr {

//! @brief バイナリファイルの対象プラットフォームを表す列挙型です。
enum BinPlatform
{
    BinPlatform_Generic = 0x206e6547, // "Gen "
    BinPlatform_NX      = 0x2020584e, // "NX  "
};

//=============================================================================
// gfx 構造体のサイズがバイナリファイル中の領域に収まるかチェックします。
//=============================================================================
NN_STATIC_ASSERT(sizeof(gfx::TMemoryPool<gfx::ApiTypeGl>::DataType) <= MemoryPoolBinaryRegionSize);
NN_STATIC_ASSERT(sizeof(gfx::TMemoryPool<gfx::ApiTypeGx>::DataType) <= MemoryPoolBinaryRegionSize);

NN_STATIC_ASSERT(sizeof(gfx::TTexture<gfx::ApiTypeGl>::DataType) <= TextureBinaryRegionSize);
NN_STATIC_ASSERT(sizeof(gfx::TTexture<gfx::ApiTypeGx>::DataType) <= TextureBinaryRegionSize);

NN_STATIC_ASSERT(sizeof(gfx::TTextureView<gfx::ApiTypeGl>::DataType) <= TextureViewBinaryRegionSize);
NN_STATIC_ASSERT(sizeof(gfx::TTextureView<gfx::ApiTypeGx>::DataType) <= TextureViewBinaryRegionSize);

NN_STATIC_ASSERT(sizeof(gfx::TMemoryPool<gfx::ApiTypeNvn>::DataType) <= MemoryPoolBinaryRegionSize);
NN_STATIC_ASSERT(sizeof(gfx::TTexture<gfx::ApiTypeNvn>::DataType) <= TextureBinaryRegionSize);
NN_STATIC_ASSERT(sizeof(gfx::TTextureView<gfx::ApiTypeNvn>::DataType) <= TextureViewBinaryRegionSize);

NN_STATIC_ASSERT(sizeof(gfx::TMemoryPool<gfx::ApiTypeVk>::DataType) <= MemoryPoolBinaryRegionSize);
NN_STATIC_ASSERT(sizeof(gfx::TTexture<gfx::ApiTypeVk>::DataType) <= TextureBinaryRegionSize);
NN_STATIC_ASSERT(sizeof(gfx::TTextureView<gfx::ApiTypeVk>::DataType) <= TextureViewBinaryRegionSize);

//=============================================================================
//! @brief テクスチャ情報ブロックのクラスです。
//=============================================================================
class TexInfoBlock : public ResTextureData
{
public:
    //! @brief 画像からテクスチャ情報ブロックのデータを設定します。
    //!
    //! @param[in,out] pContext バイナリコンバータコンテクストへのポインタです。
    //! @param[in] image 画像です。
    //! @param[in] pContainer テクスチャコンテナリソースへのポインタです。
    //! @param[in] pDataTop テクスチャデータブロック先頭へのポインタです。
    //! @param[in] userDataDicBlock ユーザーデータ辞書のメモリブロックです。
    //!
    void SetData(
        utilTool::BinarizerContext* pContext,
        const RImage& image,
        ResTextureContainerData* pContainer,
        uint8_t* pDataTop,
        util::MemorySplitter::MemoryBlock& userDataDicBlock
    );

    //! @brief エンディアンを反転します。
    //!
    //! @param[in,out] pCtx リソースのエンディアンへのポインタです。
    //! @param[in] pBase ポインタのベースアドレスです（リロケートされていれば nullptr）。
    //!
    void SwapEndian(util::ResEndian* pCtx, void* pBase);

protected:
    //! @brief 画像からテクスチャ情報ブロックのユーザーデータ群を設定します。
    //!
    //! @param[in,out] pContext バイナリコンバータコンテクストへのポインタです。
    //! @param[in,out] pUserDataTop ユーザーデータ領域先頭へのポインタです。
    //! @param[in] image 画像です。
    //! @param[in] userDataDicBlock ユーザーデータ辞書のメモリブロックです。
    //!
    void SetUserDataArray(
        utilTool::BinarizerContext* pContext,
        uint8_t* pUserDataTop,
        const RImage& image,
        util::MemorySplitter::MemoryBlock& userDataDicBlock
    );

    //! @brief ユーザーデータのエンディアンを反転します。
    //!
    //! @param[in,out] pCtx リソースのエンディアンへのポインタです。
    //! @param[in] pBase ポインタのベースアドレスです（リロケートされていれば nullptr）。
    //!
    void SwapUserDataEndian(util::ResEndian* pCtx, void* pBase);
};

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

const int UserDataValueSizeAlignment = 8; //!< ユーザーデータ値のサイズのアライメントです。

//-----------------------------------------------------------------------------
//! @brief std::string から文字列参照を取得します。
//!
//! @param[in] str 文字列です。
//!
//! @return 文字列参照を返します。
//-----------------------------------------------------------------------------
inline util::string_view GetStringView(const std::string& str)
{
    return util::string_view(str.c_str(), str.size());
}

//-----------------------------------------------------------------------------
//! @brief 対象プラットフォームがビッグエンディアンなら true を返します。
//!
//! @param[in] binPlatform プラットフォームです。
//!
//! @return 対象プラットフォームがビッグエンディアンなら true を返します。
//-----------------------------------------------------------------------------
bool IsBigEndian(const BinPlatform binPlatform)
{
    switch (binPlatform)
    {
    case BinPlatform_NX:
        return false;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャコンテナのエンディアンを反転します。
//!
//! @param[in,out] pCtx リソースのエンディアンへのポインタです。
//! @param[in,out] pContainer テクスチャコンテナへのポインタです。
//-----------------------------------------------------------------------------
void SwapEndian(util::ResEndian* pCtx, ResTextureContainerData* pContainer)
{
    util::SwapEndian(pCtx, &pContainer->textureCount);
}

//-----------------------------------------------------------------------------
//! @brief bntx ファイルのエンディアンを反転します。
//!
//! @param[in,out] pResTextureFileData テクスチャファイルデータリソースへのポインタです。
//-----------------------------------------------------------------------------
void SwapBntxEndian(ResTextureFileData* pResTextureFileData)
{
    util::BinaryFileHeader* pBinaryFileHeader = &pResTextureFileData->fileHeader;
    void* pBase = (pBinaryFileHeader->IsRelocated()) ? nullptr : pResTextureFileData;
    util::ResEndian resEndian(pBinaryFileHeader);

    ResTextureContainerData* pContainer = &pResTextureFileData->textureContainerData;
    SwapEndian(&resEndian, pContainer);

    util::BinaryBlockHeader* pBlockHeader = pBinaryFileHeader->GetFirstBlock();
    while (pBlockHeader != nullptr)
    {
        const int32_t packedSignature = pBlockHeader->signature.GetPacked();
        if (packedSignature == util::StringPool::PackedSignature)
        {
            util::StringPool* pStringPool = reinterpret_cast<util::StringPool*>(pBlockHeader);
            SwapEndian(&resEndian, pStringPool);
        }
        else if (packedSignature == ResTexture::Signature)
        {
            reinterpret_cast<TexInfoBlock*>(pBlockHeader)->SwapEndian(&resEndian, pBase);
        }
        pBlockHeader = pBlockHeader->GetNextBlock();
    }

    SwapEndian(&resEndian, pContainer->pTextureDic.ToPtr(pBase));
}

//-----------------------------------------------------------------------------
//! @brief gfx のイメージデータの次元を取得します。
//-----------------------------------------------------------------------------
gfx::ImageStorageDimension GetGfxStorageDimension(const FtxDimension dimension)
{
    switch (dimension)
    {
    case FtxDimension_1d:
    case FtxDimension_1dArray:
        return gfx::ImageStorageDimension_1d;
    case FtxDimension_2d:
    case FtxDimension_2dArray:
    case FtxDimension_2dMultisample:
    case FtxDimension_2dMultisampleArray:
    case FtxDimension_CubeMap:
    case FtxDimension_CubeMapArray:
        return gfx::ImageStorageDimension_2d;
    case FtxDimension_3d:
        return gfx::ImageStorageDimension_3d;
    default:
        return gfx::ImageStorageDimension_Undefined;
    }
}

//-----------------------------------------------------------------------------
//! @brief gfx のイメージの次元を取得します。
//-----------------------------------------------------------------------------
gfx::ImageDimension GetGfxDimension(const FtxDimension dimension)
{
    switch (dimension)
    {
    case FtxDimension_1d:                 return gfx::ImageDimension_1d;
    case FtxDimension_2d:                 return gfx::ImageDimension_2d;
    case FtxDimension_3d:                 return gfx::ImageDimension_3d;
    case FtxDimension_CubeMap:            return gfx::ImageDimension_CubeMap;
    case FtxDimension_1dArray:            return gfx::ImageDimension_1dArray;
    case FtxDimension_2dArray:            return gfx::ImageDimension_2dArray;
    case FtxDimension_2dMultisample:      return gfx::ImageDimension_2dMultisample;
    case FtxDimension_2dMultisampleArray: return gfx::ImageDimension_2dMultisampleArray;
    case FtxDimension_CubeMapArray:       return gfx::ImageDimension_CubeMapArray;
    default:                              return gfx::ImageDimension_2d;
    }
}

//-----------------------------------------------------------------------------
//! @brief gfx のイメージフォーマットを取得します。
//-----------------------------------------------------------------------------
gfx::ImageFormat GetGfxImageFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_8:                 return gfx::ImageFormat_R8_Unorm;
    case FtxFormat_Snorm_8:                 return gfx::ImageFormat_R8_Snorm;
    case FtxFormat_Uint_8:                  return gfx::ImageFormat_R8_Uint;
    case FtxFormat_Sint_8:                  return gfx::ImageFormat_R8_Sint;
    case FtxFormat_Unorm_16:                return gfx::ImageFormat_R16_Unorm;
    case FtxFormat_Snorm_16:                return gfx::ImageFormat_R16_Snorm;
    case FtxFormat_Uint_16:                 return gfx::ImageFormat_R16_Uint;
    case FtxFormat_Sint_16:                 return gfx::ImageFormat_R16_Sint;
    case FtxFormat_Uint_32:                 return gfx::ImageFormat_R32_Uint;
    case FtxFormat_Sint_32:                 return gfx::ImageFormat_R32_Sint;

    //case FtxFormat_Unorm_4_4:               return gfx::ImageFormat_R4_G4_Unorm; // 未定義
    case FtxFormat_Unorm_8_8:               return gfx::ImageFormat_R8_G8_Unorm;
    case FtxFormat_Snorm_8_8:               return gfx::ImageFormat_R8_G8_Snorm;
    case FtxFormat_Uint_8_8:                return gfx::ImageFormat_R8_G8_Uint;
    case FtxFormat_Sint_8_8:                return gfx::ImageFormat_R8_G8_Sint;
    case FtxFormat_Unorm_16_16:             return gfx::ImageFormat_R16_G16_Unorm;
    case FtxFormat_Snorm_16_16:             return gfx::ImageFormat_R16_G16_Snorm;
    case FtxFormat_Uint_16_16:              return gfx::ImageFormat_R16_G16_Uint;
    case FtxFormat_Sint_16_16:              return gfx::ImageFormat_R16_G16_Sint;
    case FtxFormat_Uint_32_32:              return gfx::ImageFormat_R32_G32_Uint;
    case FtxFormat_Sint_32_32:              return gfx::ImageFormat_R32_G32_Sint;

    case FtxFormat_Unorm_5_6_5:             return gfx::ImageFormat_R5_G6_B5_Unorm;
    case FtxFormat_Uint_32_32_32:           return gfx::ImageFormat_R32_G32_B32_Uint;
    case FtxFormat_Sint_32_32_32:           return gfx::ImageFormat_R32_G32_B32_Sint;

    case FtxFormat_Unorm_5_5_5_1:           return gfx::ImageFormat_R5_G5_B5_A1_Unorm;
    case FtxFormat_Unorm_4_4_4_4:           return gfx::ImageFormat_R4_G4_B4_A4_Unorm;

    case FtxFormat_Unorm_8_8_8_8:           return gfx::ImageFormat_R8_G8_B8_A8_Unorm;
    case FtxFormat_Snorm_8_8_8_8:           return gfx::ImageFormat_R8_G8_B8_A8_Snorm;
    case FtxFormat_Srgb_8_8_8_8:            return gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb;
    case FtxFormat_Uint_8_8_8_8:            return gfx::ImageFormat_R8_G8_B8_A8_Uint;
    case FtxFormat_Sint_8_8_8_8:            return gfx::ImageFormat_R8_G8_B8_A8_Sint;
    case FtxFormat_Unorm_10_10_10_2:        return gfx::ImageFormat_R10_G10_B10_A2_Unorm;
    case FtxFormat_Uint_10_10_10_2:         return gfx::ImageFormat_R10_G10_B10_A2_Uint;
    case FtxFormat_Unorm_16_16_16_16:       return gfx::ImageFormat_R16_G16_B16_A16_Unorm;
    case FtxFormat_Snorm_16_16_16_16:       return gfx::ImageFormat_R16_G16_B16_A16_Snorm;
    case FtxFormat_Uint_16_16_16_16:        return gfx::ImageFormat_R16_G16_B16_A16_Uint;
    case FtxFormat_Sint_16_16_16_16:        return gfx::ImageFormat_R16_G16_B16_A16_Sint;
    case FtxFormat_Uint_32_32_32_32:        return gfx::ImageFormat_R32_G32_B32_A32_Uint;
    case FtxFormat_Sint_32_32_32_32:        return gfx::ImageFormat_R32_G32_B32_A32_Sint;

    case FtxFormat_Float_16:                return gfx::ImageFormat_R16_Float;
    case FtxFormat_Float_32:                return gfx::ImageFormat_R32_Float;
    case FtxFormat_Float_16_16:             return gfx::ImageFormat_R16_G16_Float;
    case FtxFormat_Float_32_32:             return gfx::ImageFormat_R32_G32_Float;
    case FtxFormat_Float_11_11_10:          return gfx::ImageFormat_R11_G11_B10_Float;
    case FtxFormat_Float_32_32_32:          return gfx::ImageFormat_R32_G32_B32_Float;
    case FtxFormat_Float_16_16_16_16:       return gfx::ImageFormat_R16_G16_B16_A16_Float;
    case FtxFormat_Float_32_32_32_32:       return gfx::ImageFormat_R32_G32_B32_A32_Float;

    case FtxFormat_Unorm_Bc1:               return gfx::ImageFormat_Bc1_Unorm;
    case FtxFormat_Srgb_Bc1:                return gfx::ImageFormat_Bc1_UnormSrgb;
    case FtxFormat_Unorm_Bc2:               return gfx::ImageFormat_Bc2_Unorm;
    case FtxFormat_Srgb_Bc2:                return gfx::ImageFormat_Bc2_UnormSrgb;
    case FtxFormat_Unorm_Bc3:               return gfx::ImageFormat_Bc3_Unorm;
    case FtxFormat_Srgb_Bc3:                return gfx::ImageFormat_Bc3_UnormSrgb;
    case FtxFormat_Unorm_Bc4:               return gfx::ImageFormat_Bc4_Unorm;
    case FtxFormat_Snorm_Bc4:               return gfx::ImageFormat_Bc4_Snorm;
    case FtxFormat_Unorm_Bc5:               return gfx::ImageFormat_Bc5_Unorm;
    case FtxFormat_Snorm_Bc5:               return gfx::ImageFormat_Bc5_Snorm;

    case FtxFormat_Ufloat_Bc6:              return gfx::ImageFormat_Bc6_Ufloat;
    case FtxFormat_Float_Bc6:               return gfx::ImageFormat_Bc6_Float;
    case FtxFormat_Unorm_Bc7:               return gfx::ImageFormat_Bc7_Unorm;
    case FtxFormat_Srgb_Bc7:                return gfx::ImageFormat_Bc7_UnormSrgb;

    case FtxFormat_Unorm_Etc1:              return gfx::ImageFormat_Etc1_Unorm;
    case FtxFormat_Unorm_Etc2:              return gfx::ImageFormat_Etc2_Unorm;
    case FtxFormat_Srgb_Etc2:               return gfx::ImageFormat_Etc2_UnormSrgb;
    case FtxFormat_Unorm_Etc2_Mask:         return gfx::ImageFormat_Etc2_Mask_Unorm;
    case FtxFormat_Srgb_Etc2_Mask:          return gfx::ImageFormat_Etc2_Mask_UnormSrgb;
    case FtxFormat_Unorm_Etc2_Alpha:        return gfx::ImageFormat_Etc2_Alpha_Unorm;
    case FtxFormat_Srgb_Etc2_Alpha:         return gfx::ImageFormat_Etc2_Alpha_UnormSrgb;

    case FtxFormat_Unorm_Eac_11:            return gfx::ImageFormat_Eac_R11_Unorm;
    case FtxFormat_Snorm_Eac_11:            return gfx::ImageFormat_Eac_R11_Snorm;
    case FtxFormat_Unorm_Eac_11_11:         return gfx::ImageFormat_Eac_R11_G11_Unorm;
    case FtxFormat_Snorm_Eac_11_11:         return gfx::ImageFormat_Eac_R11_G11_Snorm;

    case FtxFormat_Unorm_Pvrtc1_2Bpp:       return gfx::ImageFormat_Pvrtc1_2Bpp_Unorm;
    case FtxFormat_Srgb_Pvrtc1_2Bpp:        return gfx::ImageFormat_Pvrtc1_2Bpp_UnormSrgb;
    case FtxFormat_Unorm_Pvrtc1_4Bpp:       return gfx::ImageFormat_Pvrtc1_4Bpp_Unorm;
    case FtxFormat_Srgb_Pvrtc1_4Bpp:        return gfx::ImageFormat_Pvrtc1_4Bpp_UnormSrgb;
    case FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp: return gfx::ImageFormat_Pvrtc1_Alpha_2Bpp_Unorm;
    case FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp:  return gfx::ImageFormat_Pvrtc1_Alpha_2Bpp_UnormSrgb;
    case FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp: return gfx::ImageFormat_Pvrtc1_Alpha_4Bpp_Unorm;
    case FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp:  return gfx::ImageFormat_Pvrtc1_Alpha_4Bpp_UnormSrgb;
    case FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp: return gfx::ImageFormat_Pvrtc2_Alpha_2Bpp_Unorm;
    case FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp:  return gfx::ImageFormat_Pvrtc2_Alpha_2Bpp_UnormSrgb;
    case FtxFormat_Unorm_Pvrtc2_Alpha_4Bpp: return gfx::ImageFormat_Pvrtc2_Alpha_4Bpp_Unorm;
    case FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp:  return gfx::ImageFormat_Pvrtc2_Alpha_4Bpp_UnormSrgb;

    case FtxFormat_Unorm_Astc_4x4:          return gfx::ImageFormat_Astc_4x4_Unorm;
    case FtxFormat_Srgb_Astc_4x4:           return gfx::ImageFormat_Astc_4x4_UnormSrgb;
    case FtxFormat_Unorm_Astc_5x4:          return gfx::ImageFormat_Astc_5x4_Unorm;
    case FtxFormat_Srgb_Astc_5x4:           return gfx::ImageFormat_Astc_5x4_UnormSrgb;
    case FtxFormat_Unorm_Astc_5x5:          return gfx::ImageFormat_Astc_5x5_Unorm;
    case FtxFormat_Srgb_Astc_5x5:           return gfx::ImageFormat_Astc_5x5_UnormSrgb;
    case FtxFormat_Unorm_Astc_6x5:          return gfx::ImageFormat_Astc_6x5_Unorm;
    case FtxFormat_Srgb_Astc_6x5:           return gfx::ImageFormat_Astc_6x5_UnormSrgb;
    case FtxFormat_Unorm_Astc_6x6:          return gfx::ImageFormat_Astc_6x6_Unorm;
    case FtxFormat_Srgb_Astc_6x6:           return gfx::ImageFormat_Astc_6x6_UnormSrgb;
    case FtxFormat_Unorm_Astc_8x5:          return gfx::ImageFormat_Astc_8x5_Unorm;
    case FtxFormat_Srgb_Astc_8x5:           return gfx::ImageFormat_Astc_8x5_UnormSrgb;
    case FtxFormat_Unorm_Astc_8x6:          return gfx::ImageFormat_Astc_8x6_Unorm;
    case FtxFormat_Srgb_Astc_8x6:           return gfx::ImageFormat_Astc_8x6_UnormSrgb;
    case FtxFormat_Unorm_Astc_8x8:          return gfx::ImageFormat_Astc_8x8_Unorm;
    case FtxFormat_Srgb_Astc_8x8:           return gfx::ImageFormat_Astc_8x8_UnormSrgb;
    case FtxFormat_Unorm_Astc_10x5:         return gfx::ImageFormat_Astc_10x5_Unorm;
    case FtxFormat_Srgb_Astc_10x5:          return gfx::ImageFormat_Astc_10x5_UnormSrgb;
    case FtxFormat_Unorm_Astc_10x6:         return gfx::ImageFormat_Astc_10x6_Unorm;
    case FtxFormat_Srgb_Astc_10x6:          return gfx::ImageFormat_Astc_10x6_UnormSrgb;
    case FtxFormat_Unorm_Astc_10x8:         return gfx::ImageFormat_Astc_10x8_Unorm;
    case FtxFormat_Srgb_Astc_10x8:          return gfx::ImageFormat_Astc_10x8_UnormSrgb;
    case FtxFormat_Unorm_Astc_10x10:        return gfx::ImageFormat_Astc_10x10_Unorm;
    case FtxFormat_Srgb_Astc_10x10:         return gfx::ImageFormat_Astc_10x10_UnormSrgb;
    case FtxFormat_Unorm_Astc_12x10:        return gfx::ImageFormat_Astc_12x10_Unorm;
    case FtxFormat_Srgb_Astc_12x10:         return gfx::ImageFormat_Astc_12x10_UnormSrgb;
    case FtxFormat_Unorm_Astc_12x12:        return gfx::ImageFormat_Astc_12x12_Unorm;
    case FtxFormat_Srgb_Astc_12x12:         return gfx::ImageFormat_Astc_12x12_UnormSrgb;

    default:                                return gfx::ImageFormat_Undefined;
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief gfx のチャンネルマッピングを設定します。
//-----------------------------------------------------------------------------
void SetGfxChannelMapping(Bit8* pChannelMapping, const FtxCompSel compSel)
{
    for (int rgbaIdx = 0; rgbaIdx < R_RGBA_COUNT; ++rgbaIdx)
    {
        const int sel = (compSel >> (3 - rgbaIdx) * 8) & 0xff;
        pChannelMapping[rgbaIdx] = static_cast<Bit8>(
            (sel == FtxComponent_Zero ) ? ChannelMapping_Zero  :
            (sel == FtxComponent_One  ) ? ChannelMapping_One   :
            (sel == FtxComponent_Red  ) ? ChannelMapping_Red   :
            (sel == FtxComponent_Green) ? ChannelMapping_Green :
            (sel == FtxComponent_Blue ) ? ChannelMapping_Blue  :
            (sel == FtxComponent_Alpha) ? ChannelMapping_Alpha :
            ChannelMapping_Zero);
    }
}

//-----------------------------------------------------------------------------
//! @brief 変換ユーティリティが初期化されていなければ初期化します。
//!
//! @param[in] binPlatform プラットフォームです。
//! @param[in] convUtil 変換ユーティリティです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus InitializeConverterUtility(
    const BinPlatform binPlatform,
    const ConverterUtility& convUtil
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // NX ユーティリティが初期化されていなければ初期化します。
    if (binPlatform == BinPlatform_NX)
    {
        NXUtility& nxUtility = *convUtil.GetNXUtility();
        if (!nxUtility.IsInitialized())
        {
            status = nxUtility.Initialize(nullptr);
            RCheckStatus(status);
        }
    }

    return status;
}

//-----------------------------------------------------------------------------
//! @brief プラットフォーム向けのタイリング変換をします。
//!
//! @param[in,out] pImage 画像へのポインタです。
//! @param[in] binPlatform プラットフォームです。
//! @param[in] convUtil 変換ユーティリティです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus ConvertTiling(
    RImage* pImage,
    const BinPlatform binPlatform,
    const ConverterUtility& convUtil
)
{
    //-----------------------------------------------------------------------------
    // フォーマットをチェックします。
    RImage& image = *pImage;
    const FtxFormat format = static_cast<FtxFormat>(image.GetFormat());
    const gfx::ImageFormat gfxFormat = GetGfxImageFormat(format);
    if (gfxFormat == gfx::ImageFormat_Undefined)
    {
        return RStatus(RStatus::FAILURE, "Unsupported format for texture binary file: " + // RShowError
             RGetTextureFormatString(format) + ": " + image.GetFilePath());
    }

    //-----------------------------------------------------------------------------
    // プラットフォーム向けのタイリング変換をします。
    RStatus status;
    if (binPlatform == BinPlatform_NX)
    {
        // NX 向けのタイリング変換をします。
        status = image.ConvertNXTiling(convUtil.GetNXUtility(), true);
    }
    return status;
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータを bntx ファイルの形式に変換します。
//!
//! @param[in,out] pImage 画像へのポインタです。
//-----------------------------------------------------------------------------
void ConvertUserData(RImage* pImage)
{
    RImage& image = *pImage;
    if (image.HasUserData())
    {
        RUserDataArray& userDataArray = image.GetUserDatas();
        for (size_t userDataIdx = 0; userDataIdx < userDataArray.size(); ++userDataIdx)
        {
            RUserData& userData = userDataArray[userDataIdx];
            userData.m_BinaryStringValues.clear();
            if (userData.IsString())
            {
                for (size_t strIdx = 0; strIdx < userData.m_StringValues.size(); ++strIdx)
                {
                    const std::string& str = userData.m_StringValues[strIdx];
                    const std::string binStr = RGetUtf8FromShiftJis(RReplaceCrLfToLf(str));
                    userData.m_BinaryStringValues.push_back(binStr);
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータ値のサイズを取得します。
//!
//! @param[in] userData ユーザーデータです。
//!
//! @return ユーザーデータ値のサイズを返します。
//-----------------------------------------------------------------------------
size_t GetUserDataValueSize(const RUserData& userData)
{
    switch (userData.m_Type)
    {
    case RUserData::INT:
        return sizeof(int  ) * userData.m_NumberValueCount;
    case RUserData::FLOAT:
        return sizeof(float) * userData.m_NumberValueCount;
    case RUserData::STRING:
    case RUserData::WSTRING:
        return sizeof(util::BinPtrToString) * userData.m_BinaryStringValues.size();
    case RUserData::STREAM:
        return userData.m_StreamValues.size();
    default:
        return 0;
    }
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータ値領域のサイズを取得します。
//!
//! @param[in] image 画像です。
//!
//! @return ユーザーデータ値領域のサイズを返します。
//-----------------------------------------------------------------------------
size_t GetUserDataValueRegionSize(const RImage& image)
{
    size_t regionSize = 0;
    const RUserDataArray& userDataArray = image.GetUserDatas();
    for (size_t userDataIdx = 0; userDataIdx < userDataArray.size(); ++userDataIdx)
    {
        const RUserData& userData = userDataArray[userDataIdx];
        regionSize += util::align_up(GetUserDataValueSize(userData), UserDataValueSizeAlignment);
    }
    return regionSize;
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータのオブジェクトと値を設定します。
//!
//! @param[in,out] pContext バイナリコンバータコンテクストへのポインタです。
//! @param[out] pResUserData ユーザーデータオブジェクトへのポインタです。
//! @param[in,out] ppValue ユーザーデータ値へのポインタのポインタです。
//! @param[in] userData ユーザーデータです。
//-----------------------------------------------------------------------------
void SetUserDataObjectAndValue(
    utilTool::BinarizerContext* pContext,
    gfx::ResUserDataData* pResUserData,
    uint8_t** ppValue,
    const RUserData& userData
)
{
    utilTool::BinarizerContext& context = *pContext;
    uint8_t*& pValue = *ppValue;
    context.LinkString(&pResUserData->pName, GetStringView(userData.m_Name));
    context.LinkPtr(&pResUserData->pData, pValue);
    const size_t valueSize = GetUserDataValueSize(userData);
    switch (userData.m_Type)
    {
    case RUserData::INT:
        {
            pResUserData->count = userData.m_NumberValueCount;
            pResUserData->type = gfx::ResUserData::Type_Int;
            std::istringstream iss(userData.m_NumberValuesText.c_str());
            int* pInt = reinterpret_cast<int*>(pValue);
            for (int valueIdx = 0; valueIdx < userData.m_NumberValueCount; ++valueIdx)
            {
                iss >> *pInt;
                ++pInt;
            }
        }
        break;
    case RUserData::FLOAT:
        {
            pResUserData->count = userData.m_NumberValueCount;
            pResUserData->type = gfx::ResUserData::Type_Float;
            std::istringstream iss(userData.m_NumberValuesText.c_str());
            float* pFloat = reinterpret_cast<float*>(pValue);
            for (int valueIdx = 0; valueIdx < userData.m_NumberValueCount; ++valueIdx)
            {
                iss >> *pFloat;
                ++pFloat;
            }
        }
        break;
    case RUserData::STRING:
    case RUserData::WSTRING:
        {
            pResUserData->count = static_cast<uint32_t>(userData.m_BinaryStringValues.size());
            pResUserData->type = gfx::ResUserData::Type_String;
            util::BinPtrToString* pPtr = reinterpret_cast<util::BinPtrToString*>(pValue);
            for (size_t strIdx = 0; strIdx < userData.m_BinaryStringValues.size(); ++strIdx)
            {
                const std::string& binStr = userData.m_BinaryStringValues[strIdx];
                context.LinkString(pPtr++, GetStringView(binStr));
            }
        }
        break;
    case RUserData::STREAM:
        {
            pResUserData->count = static_cast<uint32_t>(userData.m_StreamValues.size());
            pResUserData->type = gfx::ResUserData::Type_Stream;
            uint8_t* pU8 = pValue;
            for (size_t valueIdx = 0; valueIdx < userData.m_StreamValues.size(); ++valueIdx)
            {
                *pU8++ = userData.m_StreamValues[valueIdx];
            }
        }
        break;
    default:
        break;
    };
    pValue += util::align_up(valueSize, UserDataValueSizeAlignment);
}

//-----------------------------------------------------------------------------
//! @brief 画像をアライメントの大きい順にソートするための比較関数です。
//-----------------------------------------------------------------------------
bool IsAlignmentGreater(RImage*& r1, RImage*& r2)
{
    if (r1->GetAlignment() == r2->GetAlignment())
    {
        return (r1->GetWorkIndex() < r2->GetWorkIndex());
    }
    else
    {
        return (r1->GetAlignment() > r2->GetAlignment());
    }
}

//-----------------------------------------------------------------------------
//! @brief アライメントの大きい順に画像をソートします。
//!
//! @param[in,out] ppSortedImages ソートした画像へのポインタ配列を格納します。
//! @param[in] pImages 画像へのポインタ配列です。
//-----------------------------------------------------------------------------
void SortImageByAlignment(
    std::vector<RImage*>* ppSortedImages,
    const std::vector<RImage*>& pImages
)
{
    const int textureCount = static_cast<int>(pImages.size());
    std::vector<RImage*>& pSortedImages = *ppSortedImages;
    pSortedImages.clear();
    for (int texIdx = 0; texIdx < textureCount; ++texIdx)
    {
        pImages[texIdx]->SetWorkIndex(texIdx);
            // ↑アライメントが同じ場合に元の順番を保持するためのインデックスです。
        pSortedImages.push_back(pImages[texIdx]);
    }
    std::sort(pSortedImages.begin(), pSortedImages.end(), IsAlignmentGreater);
}

//-----------------------------------------------------------------------------
//! @brief 文字列プールに文字列を登録します。
//!
//! @param[in,out] pContext バイナリコンバータコンテクストへのポインタです。
//! @param[in] pImages 画像へのポインタ配列です。
//-----------------------------------------------------------------------------
void AddStrings(
    utilTool::BinarizerContext* pContext,
    const std::vector<RImage*>& pImages
)
{
    utilTool::BinarizerContext& context = *pContext;
    for (size_t texIdx = 0; texIdx < pImages.size(); ++texIdx)
    {
        const RImage& image = *pImages[texIdx];
        context.InsertString(GetStringView(image.GetName()));

        const RUserDataArray& userDataArray = image.GetUserDatas();
        for (size_t userDataIdx = 0; userDataIdx < userDataArray.size(); ++userDataIdx)
        {
            const RUserData& userData = userDataArray[userDataIdx];
            context.InsertString(GetStringView(userData.m_Name));
            if (userData.IsString())
            {
                for (size_t strIdx = 0; strIdx < userData.m_BinaryStringValues.size(); ++strIdx)
                {
                    const std::string& binStr = userData.m_BinaryStringValues[strIdx];
                    context.InsertString(GetStringView(binStr));
                }
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief メモリブロックを登録します。
//!
//! @param[in,out] pContext バイナリコンバータコンテクストへのポインタです。
//! @param[in,out] pHeaderMemoryBlock ファイルヘッダのメモリブロックへのポインタです。
//! @param[in,out] pTexDicBlock テクスチャ名辞書のメモリブロックへのポインタです。
//! @param[in,out] pInfoMemoryBlocks テクスチャ情報のメモリブロック配列へのポインタです。
//! @param[in,out] pUserDataDicBlocks ユーザーデータ辞書のメモリブロック配列へのポインタです。
//! @param[in,out] pDataMemoryBlock テクスチャデータのメモリブロックへのポインタです。
//! @param[in,out] pImages 画像へのポインタ配列です。
//! @param[in] binPlatform プラットフォームです。
//! @param[in] enablesMemoryPool メモリプールを有効にするなら true です。
//! @param[in] convUtil 変換ユーティリティです。
//-----------------------------------------------------------------------------
void AddMemoryBlocks(
    utilTool::BinarizerContext* pContext,
    util::MemorySplitter::MemoryBlock* pHeaderMemoryBlock,
    util::MemorySplitter::MemoryBlock* pTexDicBlock,
    std::vector<util::MemorySplitter::MemoryBlock>* pInfoMemoryBlocks,
    std::vector<util::MemorySplitter::MemoryBlock>* pUserDataDicBlocks,
    util::MemorySplitter::MemoryBlock* pDataMemoryBlock,
    const std::vector<RImage*>& pImages,
    const BinPlatform binPlatform,
    const bool enablesMemoryPool,
    const ConverterUtility& convUtil
)
{
    utilTool::BinarizerContext& context = *pContext;
    const int textureCount = static_cast<int>(pImages.size());

    //-----------------------------------------------------------------------------
    // アライメントの大きい順に画像をソートします。
    std::vector<RImage*> pSortedImages;
    SortImageByAlignment(&pSortedImages, pImages);

    //-----------------------------------------------------------------------------
    // ファイルヘッダのメモリブロックを登録します。
    pHeaderMemoryBlock->SetSize(
        sizeof(ResTextureFileData) +
        ((enablesMemoryPool) ? MemoryPoolBinaryRegionSize : 0) +
        sizeof(util::BinPtr) * textureCount);
    context.AddMemoryBlock(0, pHeaderMemoryBlock);

    //-----------------------------------------------------------------------------
    // 文字列プールを構築してメモリブロックを登録します。
    // リソース中で使用する文字列は、文字列プールの構築前に InsertString() で登録する必要があります。
    context.AddMemoryBlock(0, context.BuildStringPool());

    //-----------------------------------------------------------------------------
    // テクスチャ名辞書のメモリブロックを登録します。
    pTexDicBlock->SetSize(util::ResDic::CalculateSize(textureCount));
    context.AddMemoryBlock(0, pTexDicBlock);

    //-----------------------------------------------------------------------------
    // テクスチャ情報のメモリブロックを登録します。
    for (int texIdx = 0; texIdx < textureCount; ++texIdx)
    {
        const RImage& image = *pImages[texIdx];
        const int userDataCount = static_cast<int>(image.GetUserDatas().size());
        util::MemorySplitter::MemoryBlock& infoMemoryBlock = (*pInfoMemoryBlocks)[texIdx];
        infoMemoryBlock.SetSize(
            sizeof(TexInfoBlock) +
            TextureBinaryRegionSize +
            TextureViewBinaryRegionSize +
            sizeof(util::BinPtr) * image.GetMipCount() +
            sizeof(gfx::ResUserDataData) * userDataCount +
            GetUserDataValueRegionSize(image));
        context.AddMemoryBlock(0, &infoMemoryBlock);

        if (image.HasUserData())
        {
            util::MemorySplitter::MemoryBlock& userDataDicBlock = (*pUserDataDicBlocks)[texIdx];
            userDataDicBlock.SetSize(util::ResDic::CalculateSize(userDataCount));
            context.AddMemoryBlock(0, &userDataDicBlock);
        }
    }

    //-----------------------------------------------------------------------------
    // テクスチャデータのメモリブロックを登録します。
    int maxAlignment = 1;
    size_t dataSize = 0;
    for (int texIdx = 0; texIdx < textureCount; ++texIdx)
    {
        RImage& image = *pSortedImages[texIdx];
        //cerr << "Sorted: " << image.GetName() << ": " << image.GetAlignment() << endl;
        maxAlignment = RMax(image.GetAlignment(), maxAlignment);
        dataSize = RAlignValue(dataSize, image.GetAlignment());
        image.SetBinaryDataOffset(dataSize);
        dataSize += image.GetImageDataSize();
    }

    if (binPlatform == BinPlatform_NX && enablesMemoryPool)
    {
        int memoryPoolAddressAlignment = 1;
        int memoryPoolSizeAlignment = 1;
        convUtil.GetNXUtility()->GetMemoryPoolAlignment(
            &memoryPoolAddressAlignment, &memoryPoolSizeAlignment);
        //cerr << "memory pool alignment: " << memoryPoolAddressAlignment << ", " << memoryPoolSizeAlignment << endl;

        maxAlignment = RMax(maxAlignment, memoryPoolAddressAlignment);
        dataSize = RAlignValue(dataSize, memoryPoolSizeAlignment);
    }

    pDataMemoryBlock->SetSize(sizeof(util::BinaryBlockHeader) + dataSize);
    pDataMemoryBlock->SetAlignment(maxAlignment, sizeof(util::BinaryBlockHeader));
         // ↑テクスチャデータブロックのバイナリデータの先頭がアライメントされます。
    context.AddMemoryBlock(1, pDataMemoryBlock);
}

//-----------------------------------------------------------------------------
//! @brief bntx ファイルのユーザーデータ群を表示します（動作確認用）。
//!
//! @param[in] pContainer テクスチャコンテナリソースへのポインタです。
//-----------------------------------------------------------------------------
inline void DisplayBntxUserDataArray(const ResTextureContainerData* pContainer)
{
    const util::BinTPtr<gfx::ResTexture>* pResTextures = pContainer->pTexturePtrArray.Get();
    for (int texIdx = 0; texIdx < static_cast<int>(pContainer->textureCount); ++texIdx)
    {
        const gfx::ResTextureData& resTex = pResTextures[texIdx].Get()->ToData();
        const gfx::ResUserData* resUserDataArray = resTex.pUserDataArray.Get();
        if (resUserDataArray != nullptr)
        {
            const util::ResDic& userDataDic = *resTex.pUserDataDic.Get();
            cerr << "Tex" << texIdx << " : User Data Count: " << userDataDic.GetCount() << endl;
            for (int userDataIdx = 0; userDataIdx < userDataDic.GetCount(); ++userDataIdx)
            {
                const gfx::ResUserDataData& resUserData = resUserDataArray[userDataIdx].ToData();
                const void* pValue = resUserData.pData.Get();
                const util::string_view name = resUserData.pName.Get()->Get();
                cerr << "  " << name.data() << "[" << resUserData.count << "] = { ";
                switch (resUserData.type)
                {
                case gfx::ResUserData::Type_Int:
                    {
                        for (uint32_t valueIdx = 0; valueIdx < resUserData.count; ++valueIdx)
                        {
                            cerr << ((valueIdx == 0) ? "" : ", ")
                                 << reinterpret_cast<const int*>(pValue)[valueIdx];
                        }
                    }
                    break;
                case gfx::ResUserData::Type_Float:
                    {
                        for (uint32_t valueIdx = 0; valueIdx < resUserData.count; ++valueIdx)
                        {
                            cerr << ((valueIdx == 0) ? "" : ", ")
                                 << reinterpret_cast<const float*>(pValue)[valueIdx] << "f";
                        }
                    }
                    break;
                case gfx::ResUserData::Type_String:
                    {
                        const util::BinPtrToString* pStrs = reinterpret_cast<const util::BinPtrToString*>(pValue);
                        for (uint32_t strIdx = 0; strIdx < resUserData.count; ++strIdx)
                        {
                            const std::string str = RGetShiftJisFromUtf8(pStrs[strIdx].Get()->Get().data());
                            cerr << ((strIdx == 0) ? "" : ", ") << "\"" << str << "\"";
                        }
                    }
                    break;
                case gfx::ResUserData::Type_Stream:
                    {
                        const uint32_t dumpCount = std::min(resUserData.count, static_cast<uint32_t>(16));
                        for (uint32_t valueIdx = 0; valueIdx < dumpCount; ++valueIdx)
                        {
                            cerr << ((valueIdx == 0) ? "" : ", ")
                                 << RGetNumberString(reinterpret_cast<const uint8_t*>(pValue)[valueIdx], "0x%02x");
                        }
                        cerr << ((dumpCount == resUserData.count) ? "" : ", ...");
                    }
                    break;
                default:
                    break;
                }
                cerr << " }; // (" << userDataDic.FindIndex(name) << ")" << endl;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief bntx ファイルの情報を表示します（動作確認用）。
//!
//! @param[in,out] pContext バイナリコンバータコンテクストへのポインタです。
//-----------------------------------------------------------------------------
inline void DisplayBntxInfo(utilTool::BinarizerContext* pContext)
{
    //-----------------------------------------------------------------------------
    // メモリを確保して bntx ファイルのデータをコピーします。
    utilTool::BinarizerContext& context = *pContext;
    void* pDst = _aligned_malloc(context.GetBaseSize() + context.GetRelocationTableSize(), context.GetAlignment());
    memcpy(pDst, context.GetBasePtr(), context.GetBaseSize());
    memcpy(util::BytePtr(pDst, context.GetBaseSize()).Get(), context.GetRelocationTablePtr(), context.GetRelocationTableSize());

    //-----------------------------------------------------------------------------
    // 再配置します。
    ResTextureFileData* pResTextureFileData = reinterpret_cast<ResTextureFileData*>(pDst);
    util::BinaryFileHeader& fileHeader = pResTextureFileData->fileHeader;
    fileHeader.GetRelocationTable()->Relocate();

    //-----------------------------------------------------------------------------
    // bntx ファイルがビッグエンディアンならエンディアンを反転します。
    if (fileHeader.IsEndianReverse())
    {
        SwapBntxEndian(pResTextureFileData);
        fileHeader.SetByteOrderMark(util::ByteOrderMark_Normal);
    }

    //-----------------------------------------------------------------------------
    // 情報を表示します。
    const ResTextureContainerData* pContainer = &pResTextureFileData->textureContainerData;
    const util::ResDic& texDic = *pContainer->pTextureDic.Get();
    const util::BinTPtr<gfx::ResTexture>* pResTextures = pContainer->pTexturePtrArray.Get();
    const gfx::ResTextureData& resTex0 = pResTextures[0].Get()->ToData();
    cerr << "Name  : " << fileHeader.GetFileName().data() << endl;
    cerr << "Top   : " << pResTextureFileData << endl;
    cerr << "Infos : " << pContainer->pTexturePtrArray.Get() << endl;
    cerr << "Dic   : " << &texDic << endl;
    cerr << "Info0 : " << &resTex0 << endl;
    cerr << "Name0 : " << resTex0.pName.Get()->Get().data() << endl;
    cerr << "Fmt0  : " << hex << resTex0.textureInfoData.imageFormat << dec << endl;
    cerr << "Cntnr0: " << resTex0.pResTextureContainerData.Get() << endl;
    cerr << "Levs0 : " << resTex0.pMipPtrArray.Get() << endl;
    cerr << "Data  : " << pContainer->pTextureData.Get() << endl;
    cerr << "LevD0 : " << resTex0.pMipPtrArray.Get()[0].Get() << endl;
    cerr << "Count : " << pContainer->textureCount << endl;
    for (int texIdx = 0; texIdx < static_cast<int>(pContainer->textureCount); ++texIdx)
    {
        const nn::util::string_view key = texDic.GetKey(texIdx);
        cerr << "Tex" << texIdx << "  : " << key.data() << " (" << key.size() << ")" << endl;
    }
    cerr << "DicIdx: " << texDic.FindIndex(fileHeader.GetFileName()) << endl;
    cerr << "DicIdx: " << texDic.FindIndex(texDic.GetKey(pContainer->textureCount - 1)) << endl;
    DisplayBntxUserDataArray(pContainer);

    cerr << "ResTextureFileData     : " << sizeof(ResTextureFileData) << endl; // 88 (0x58)
    cerr << "ResTextureContainerData: " << sizeof(ResTextureContainerData) << endl; // 56 (0x38)
    cerr << "ResTextureData         : " << sizeof(ResTextureData) << endl; // 160 (0xa0)

    //-----------------------------------------------------------------------------
    // メモリを解放します。
    _aligned_free(pDst);
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 画像からテクスチャ情報ブロックのユーザーデータ群を設定します。
//-----------------------------------------------------------------------------
void TexInfoBlock::SetUserDataArray( // Info_SetUserDataArray
    utilTool::BinarizerContext* pContext,
    uint8_t* pUserDataTop,
    const RImage& image,
    util::MemorySplitter::MemoryBlock& userDataDicBlock
)
{
    //-----------------------------------------------------------------------------
    // 各ユーザーデータのオブジェクトと値を設定します。
    utilTool::BinarizerContext& context = *pContext;
    const RUserDataArray& userDataArray = image.GetUserDatas();
    gfx::ResUserDataData* resUserDataArray = reinterpret_cast<ResUserDataData*>(pUserDataTop);
    uint8_t* pValue = pUserDataTop + sizeof(gfx::ResUserDataData) * userDataArray.size();
    std::vector<const RUserData*> pSrcUserDataArray;
    for (size_t userDataIdx = 0; userDataIdx < userDataArray.size(); ++userDataIdx)
    {
        const RUserData& userData = userDataArray[userDataIdx];
        SetUserDataObjectAndValue(&context, &resUserDataArray[userDataIdx], &pValue, userData);
        pSrcUserDataArray.push_back(&userData);
    }

    //-----------------------------------------------------------------------------
    // ユーザーデータ辞書を構築します。
    utilTool::BinarizerUtil::BuildResDic(&context, &userDataDicBlock,
        pSrcUserDataArray.cbegin(), pSrcUserDataArray.cend(),
        [](const RUserData& userData)
        {
            return GetStringView(userData.m_Name);
        });

    //-----------------------------------------------------------------------------
    // ユーザーデータ関連のポインタを設定します。
    context.LinkPtr(&this->pUserDataArray, pUserDataTop);
    context.LinkPtr(&this->pUserDataDic, userDataDicBlock.Get<util::ResDic>(context.GetBasePtr()));
}

//-----------------------------------------------------------------------------
//! @brief 画像からテクスチャ情報ブロックのデータを設定します。
//-----------------------------------------------------------------------------
void TexInfoBlock::SetData( // Info_SetData
    utilTool::BinarizerContext* pContext,
    const RImage& image,
    ResTextureContainerData* pContainer,
    uint8_t* pDataTop,
    util::MemorySplitter::MemoryBlock& userDataDicBlock
)
{
    //-----------------------------------------------------------------------------
    // ブロックヘッダを設定します。
    utilTool::BinarizerContext& context = *pContext;
    this->blockHeader.signature.SetPacked(ResTexture::Signature);
    context.AddHeader(&this->blockHeader);

    //-----------------------------------------------------------------------------
    // テクスチャ初期化情報のメンバを設定します。
    gfx::TextureInfoData& infoData = this->textureInfoData;
    memset(&infoData, 0, sizeof(infoData));

    infoData.flags.Clear();
    if (image.UsesTextureLayout())
    {
        infoData.flags.SetBit(gfx::TextureInfoData::Flag_SpecifyTextureLayout, true);
    }
    if (image.IsSparseTiled())
    {
        infoData.flags.SetBit(gfx::TextureInfoData::Flag_SparseBinding, true);
        infoData.flags.SetBit(gfx::TextureInfoData::Flag_SparseResidency, true);
    }
    const FtxDimension dimension = static_cast<FtxDimension>(image.GetDimension());
    infoData.imageStorageDimension = static_cast<Bit8>(GetGfxStorageDimension(dimension));
    infoData.tileMode = static_cast<Bit16>(
        (image.GetTileMode() == FtxTileMode_Linear) ?
        gfx::TileMode_Linear : gfx::TileMode_Optimal);
    infoData.swizzle = static_cast<uint16_t>(image.GetInitialSwizzle());
    infoData.mipCount = static_cast<uint16_t>(image.GetMipCount());
    infoData.multisampleCount = static_cast<uint16_t>(1);
    infoData.imageFormat = static_cast<Bit32>(
        GetGfxImageFormat(static_cast<FtxFormat>(image.GetFormat())));
    infoData.gpuAccessFlags = static_cast<Bit32>(GpuAccess_Texture);
    infoData.width  = image.GetImageW();
    infoData.height = image.GetImageH();
    infoData.depth = (image.GetDimension() == FtxDimension_3d) ?
        image.GetImageD() : 1;
    infoData.arrayLength = (image.IsCubeMap() || image.IsArray()) ?
        image.GetImageD() : 1;
    const FtxTextureLayout textureLayout = image.GetTextureLayout();
    memcpy(infoData.textureLayout, &textureLayout, sizeof(infoData.textureLayout));

    //-----------------------------------------------------------------------------
    // テクスチャ情報のメンバを設定します。
    this->textureDataSize = static_cast<uint32_t>(image.GetImageDataSize());
    this->alignment = image.GetAlignment();
    SetGfxChannelMapping(this->channelMapping, image.GetCompSel());
    this->imageDimension = static_cast<Bit8>(GetGfxDimension(dimension));
    context.LinkString(&this->pName, GetStringView(image.GetName()));
    context.LinkPtr(&this->pResTextureContainerData, pContainer);

    uint8_t* pTextureObject = reinterpret_cast<uint8_t*>(this) + sizeof(*this);
    uint8_t* pTextureViewObject = pTextureObject + TextureBinaryRegionSize;
    util::BinPtr* pLevelDataArray = reinterpret_cast<util::BinPtr*>(
        pTextureViewObject + TextureViewBinaryRegionSize);
    for (int level = 0; level < image.GetMipCount(); ++level)
    {
        context.LinkPtr(&pLevelDataArray[level],
            pDataTop + image.GetBinaryDataOffset() + image.GetLevelOffset(level));
    }
    context.LinkPtr(&this->pMipPtrArray, pLevelDataArray);
    context.LinkPtr(&this->pTexture, pTextureObject);
    context.LinkPtr(&this->pTextureView, pTextureViewObject);

    //-----------------------------------------------------------------------------
    // ユーザーデータ群を設定します。
    if (image.HasUserData())
    {
        uint8_t* pUserDataTop = reinterpret_cast<uint8_t*>(
            pLevelDataArray + image.GetMipCount());
        SetUserDataArray(&context, pUserDataTop, image, userDataDicBlock);
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ情報ブロックのユーザーデータのエンディアンを反転します。
//-----------------------------------------------------------------------------
void TexInfoBlock::SwapUserDataEndian(util::ResEndian* pCtx, void* pBase)
{
    gfx::ResUserData* resUserDataArray = this->pUserDataArray.ToPtr(pBase);
    if (resUserDataArray != nullptr)
    {
        util::ResDic& userDataDic = *this->pUserDataDic.ToPtr(pBase);
        if (!pCtx->IsHostEndian())
        {
            util::SwapEndian(pCtx, &userDataDic);
        }

        for (int userDataIdx = 0; userDataIdx < userDataDic.GetCount(); ++userDataIdx)
        {
            gfx::ResUserDataData& resUserData = resUserDataArray[userDataIdx].ToData();
            if (!pCtx->IsHostEndian())
            {
                util::SwapEndian(pCtx, &resUserData.count);
            }

            if (resUserData.type == gfx::ResUserData::Type_Int)
            {
                int* pInt = reinterpret_cast<int*>(resUserData.pData.ToPtr(pBase));
                for (uint32_t valueIdx = 0; valueIdx < resUserData.count; ++valueIdx)
                {
                    util::SwapEndian(pCtx, pInt++);
                }
            }
            else if (resUserData.type == gfx::ResUserData::Type_Float)
            {
                float* pFloat = reinterpret_cast<float*>(resUserData.pData.ToPtr(pBase));
                for (uint32_t valueIdx = 0; valueIdx < resUserData.count; ++valueIdx)
                {
                    util::SwapEndian(pCtx, pFloat++);
                }
            }

            if (pCtx->IsHostEndian())
            {
                util::SwapEndian(pCtx, &resUserData.count);
            }
        }

        if (pCtx->IsHostEndian())
        {
            util::SwapEndian(pCtx, &userDataDic);
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ情報ブロックのエンディアンを反転します。
//-----------------------------------------------------------------------------
void TexInfoBlock::SwapEndian(util::ResEndian* pCtx, void* pBase)
{
    gfx::TextureInfoData& infoData = this->textureInfoData;
    util::SwapEndian(pCtx, &infoData.tileMode);
    util::SwapEndian(pCtx, &infoData.swizzle);
    util::SwapEndian(pCtx, &infoData.mipCount);
    util::SwapEndian(pCtx, &infoData.multisampleCount);
    util::SwapEndian(pCtx, &infoData.imageFormat);
    util::SwapEndian(pCtx, &infoData.gpuAccessFlags);
    util::SwapEndian(pCtx, &infoData.width);
    util::SwapEndian(pCtx, &infoData.height);
    util::SwapEndian(pCtx, &infoData.depth);
    util::SwapEndian(pCtx, &infoData.arrayLength);

    util::SwapEndian(pCtx, &this->textureDataSize);
    util::SwapEndian(pCtx, &this->alignment);

    SwapUserDataEndian(pCtx, pBase);
}

//-----------------------------------------------------------------------------
//! @brief bntx ファイルのデータを出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in,out] pImages 画像へのポインタ配列です。
//! @param[in] binName バイナリファイル名です。
//! @param[in] binTileMode bntx ファイルのタイルモードです。
//! @param[in] enablesMemoryPool メモリプールを有効にするなら true です。
//! @param[in] convUtil 変換ユーティリティです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
RStatus OutputBntxData(
    std::ostream& os,
    const std::vector<RImage*>& pImages,
    const std::string& binName,
    const std::string& binTileMode,
    const bool enablesMemoryPool,
    const ConverterUtility& convUtil
)
{
    RStatus status;

    //-----------------------------------------------------------------------------
    // プラットフォームをチェックします。
    BinPlatform binPlatform = BinPlatform_Generic;
    if (binTileMode == "nx")
    {
        binPlatform = BinPlatform_NX;
    }

    //-----------------------------------------------------------------------------
    // 変換ユーティリティが初期化されていなければ初期化します。
    status = InitializeConverterUtility(binPlatform, convUtil);
    RCheckStatus(status);

    //-----------------------------------------------------------------------------
    // プラットフォーム向けのタイリング変換、ユーザーデータ変換をします。
    const int textureCount = static_cast<int>(pImages.size());
    for (int texIdx = 0; texIdx < textureCount; ++texIdx)
    {
        RImage* pImage = pImages[texIdx];
        status = ConvertTiling(pImage, binPlatform, convUtil);
        RCheckStatus(status);

        ConvertUserData(pImage);
    }

    //-----------------------------------------------------------------------------
    // バイナリコンバータコンテクストを初期化します。
    utilTool::BinarizerContext context;
    context.Initialize(2);
    context.SetName(GetStringView(binName));

    //-----------------------------------------------------------------------------
    // 文字列プールに文字列を登録します。
    AddStrings(&context, pImages);

    //-----------------------------------------------------------------------------
    // メモリブロックを登録します。
    util::MemorySplitter::MemoryBlock headerMemoryBlock;
    util::MemorySplitter::MemoryBlock texDicBlock;
    std::vector<util::MemorySplitter::MemoryBlock> infoMemoryBlocks(textureCount);
    std::vector<util::MemorySplitter::MemoryBlock> userDataDicBlocks(textureCount);
    util::MemorySplitter::MemoryBlock dataMemoryBlock;
    AddMemoryBlocks(&context, &headerMemoryBlock, &texDicBlock,
        &infoMemoryBlocks, &userDataDicBlocks, &dataMemoryBlock,
        pImages, binPlatform, enablesMemoryPool, convUtil);

    //-----------------------------------------------------------------------------
    // 登録したメモリブロックからサイズを計算して全体のバッファを確保します。
    context.AllocateBase(); // 確保したバッファは 0 クリアされます。

    //-----------------------------------------------------------------------------
    // 文字列プールを変換します。
    context.ConvertStringPool();

    //-----------------------------------------------------------------------------
    // ファイルヘッダにデータを設定します。
    ResTextureFileData* pResTextureFileData = headerMemoryBlock.Get<ResTextureFileData>(context.GetBasePtr());
    util::BinaryBlockHeader* pDataBlock = dataMemoryBlock.Get<util::BinaryBlockHeader>(context.GetBasePtr());
    ResTextureContainerData* pContainer = &pResTextureFileData->textureContainerData;
    *reinterpret_cast<uint32_t*>(pContainer->targetPlatform) = binPlatform;
    pContainer->textureCount = textureCount;
    uint8_t* pTexturePool = reinterpret_cast<uint8_t*>(pResTextureFileData) + sizeof(ResTextureFileData);
    util::BinPtr* pTexturePtrArray = reinterpret_cast<util::BinPtr*>(
        pTexturePool + ((enablesMemoryPool) ? MemoryPoolBinaryRegionSize : 0));
    util::ResDic* pTexDic = texDicBlock.Get<util::ResDic>(context.GetBasePtr());
    context.LinkPtr(&pContainer->pTexturePtrArray, pTexturePtrArray);
    context.LinkPtr(&pContainer->pTextureData, pDataBlock);
    context.LinkPtr(&pContainer->pTextureDic, pTexDic);
    if (enablesMemoryPool)
    {
        context.LinkPtr(&pContainer->pTextureMemoryPool, pTexturePool);
    }

    //-----------------------------------------------------------------------------
    // テクスチャ名辞書を構築します。
    const bool buildDicResult = utilTool::BinarizerUtil::BuildResDic( // 内部でキー文字列の LinkString を行います。
        &context, &texDicBlock, pImages.cbegin(), pImages.cend(),
        [](const RImage& image)
        {
            return GetStringView(image.GetName());
        });
    R_UNUSED_VARIABLE(buildDicResult);
    //cerr << "Build Dic : " << buildDicResult << endl;

    //-----------------------------------------------------------------------------
    // テクスチャ情報とテクスチャデータにデータを設定します
    uint8_t* pDataTop = reinterpret_cast<uint8_t*>(pDataBlock) + sizeof(util::BinaryBlockHeader);
    for (int texIdx = 0; texIdx < textureCount; ++texIdx)
    {
        const RImage& image = *pImages[texIdx];
        const util::MemorySplitter::MemoryBlock& infoMemoryBlock = infoMemoryBlocks[texIdx];
        TexInfoBlock* pInfoBlock = infoMemoryBlock.Get<TexInfoBlock>(context.GetBasePtr());
        pInfoBlock->SetData(&context, image, pContainer, pDataTop, userDataDicBlocks[texIdx]);
        context.LinkPtr(&pTexturePtrArray[texIdx], pInfoBlock);
        memcpy(pDataTop + image.GetBinaryDataOffset(), image.GetImageData(), image.GetImageDataSize());
    }

    pDataBlock->signature.SetPacked(NN_UTIL_CREATE_SIGNATURE_4('B', 'R', 'T', 'D'));
    context.AddHeader(pDataBlock);

    //-----------------------------------------------------------------------------
    // リロケーションテーブルを構築します。
    context.AllocateRelocationTable();
    context.ConvertRelocationTable();

    //-----------------------------------------------------------------------------
    // 登録済みの各ヘッダを変換します。
    // アライメントやサイズ、次のヘッダへのオフセットを設定しています。
    util::BinaryFileHeader* pBinaryFileHeader = context.ConvertHeaders();

    //-----------------------------------------------------------------------------
    // ファイルヘッダのうち、自動で設定されない項目を設定します。
    pBinaryFileHeader->signature.SetPacked(ResTextureFile::Signature);
    pBinaryFileHeader->version.major = ResTextureFile::MajorVersion;
    pBinaryFileHeader->version.minor = ResTextureFile::MinorVersion;
    pBinaryFileHeader->version.micro = ResTextureFile::MicroVersion;
    pBinaryFileHeader->SetByteOrderMark(util::ByteOrderMark_Normal);
    pBinaryFileHeader->SetAddressSize(); // sizeof(BinPtr) = 64 で固定

    //-----------------------------------------------------------------------------
    // 対象プラットフォームがビッグエンディアンならエンディアンを反転します。
    if (IsBigEndian(binPlatform))
    {
        // 反転前の ByteOrderMark は Normal である必要があります。
        SwapBntxEndian(pResTextureFileData);
        // 反転後に ByteOrderMark を Reverse にします。
        pBinaryFileHeader->SetByteOrderMark(util::ByteOrderMark_Reverse);
    }

    //-----------------------------------------------------------------------------
    // バイナリデータを出力します。
    os.write(reinterpret_cast<const char*>(context.GetBasePtr()), context.GetBaseSize());
    os.write(reinterpret_cast<const char*>(context.GetRelocationTablePtr()), context.GetRelocationTableSize());

    //-----------------------------------------------------------------------------
    // bntx ファイルの情報を表示します。
    //DisplayBntxInfo(&context);

    return status;
} // NOLINT(impl/function_size)

//=============================================================================
// texcvtr ネームスペースを終了します。
//=============================================================================
} // texcvtr
} // tool
} // gfx
} // nn

