﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// ConvertByServer RunClient
// ServerMain RunServer ServerSideConvert

// ParamSet

//=============================================================================
// include
//=============================================================================
#include "Process.h"

using namespace std;

//-----------------------------------------------------------------------------
// 無名名前空間を開始します。
namespace
{

//=============================================================================
// constants
//=============================================================================
const DWORD DefaultServerTimeout = 20 * 1000; //!< サーバーがクライアントからの接続を待つ最大時間 [ms] のデフォルト値です。
const DWORD ServerMessageMaxSize = 512; //!< サーバーからクライアントに送信するメッセージの最大バイト数です。
const std::string SuccessMessage = "Success"; //!< サーバーからクライアントに送信する成功メッセージです。

const char PipeName[] = "\\\\.\\Pipe\\NintendoTextureConverterNvtt"; //!< プロセス間通信のためのパイプ名です。
const char MutexName[] = "NintendoTextureConverterNvtt"; //!< プロセス間通信のためのミューテックス名です。

//=============================================================================
//! @brief 変換パラメーターセットの構造体です。
//=============================================================================
struct ParamSet
{
    char dstFileMapName[128]; //!< 変換後のピクセルデータのファイルマッピング名です。
    char srcFileMapName[128]; //!< 変換前のピクセルデータのファイルマッピング名です。
    char dstFormat[32]; //!< 変換後のフォーマット文字列です。
    char srcFormat[32]; //!< 変換前のフォーマット文字列です。
    char quality[32]; //!< エンコード品質文字列です。
    int encodeFlag; //!< エンコードフラグです。
    int dimension; //!< 次元です。
    int imageW; //!< 画像の幅です。
    int imageH; //!< 画像の高さです。
    int imageD; //!< 画像の奥行きです。
    int mipCount; //!< ミップマップのレベル数です。
    int pid; //!< クライアントプロセスの PID です。
};

//-----------------------------------------------------------------------------
//! @brief サーバーがクライアントからの接続を待つ最大時間 [ms] を取得します。
//!
//! @return サーバーがクライアントからの接続を待つ最大時間を返します。
//-----------------------------------------------------------------------------
DWORD GetServerTimeout()
{
    const std::string timeStr = GetEnvVariable("NINTENDO_TEXTURE_CONVERTER_NVTT_PROCESS_TIMEOUT");
    if (!timeStr.empty())
    {
        if (_stricmp(timeStr.c_str(), "INFINITE") == 0)
        {
            return INFINITE;
        }
        else
        {
            return static_cast<DWORD>(strtoul(timeStr.c_str(), nullptr, 10));
        }
    }
    return DefaultServerTimeout;
}

//-----------------------------------------------------------------------------
//! @brief サーバープロセスで実行する関数の型です。
//!
//! @param[out] pErrMsg 処理失敗時にエラーメッセージを格納します。
//! @param[in] pData クライアントから受信したデータへのポインターです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
typedef bool (*ServerSideEntryType)(std::string* pErrMsg, void* pData);

//-----------------------------------------------------------------------------
//! @brief サーバー側の変換処理です。
//!
//! @param[out] pErrMsg 処理失敗時にエラーメッセージを格納します。
//! @param[in] pData クライアントから受信したデータへのポインターです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool ServerSideConvert(std::string* pErrMsg, void* pData)
{
    const ParamSet& params = *reinterpret_cast<ParamSet*>(pData);
    //NoteTrace("server pid: %d", GetCurrentProcessId());
    //NoteTrace("client pid: %d", params.pid);

    //-----------------------------------------------------------------------------
    // ファイルマッピングオブジェクトを開きます。
    HANDLE hDstFileMap = OpenFileMappingA(FILE_MAP_WRITE, FALSE, params.dstFileMapName);
    if (hDstFileMap == nullptr)
    {
        *pErrMsg = "Cannot open file mapping: " + std::string(params.dstFileMapName) + " (" + GetWindowsLastErrorMessage() + ")";
        return false;
    }
    HANDLE hSrcFileMap = OpenFileMappingA(FILE_MAP_READ, FALSE, params.srcFileMapName);
    if (hSrcFileMap == nullptr)
    {
        *pErrMsg = "Cannot open file mapping: " + std::string(params.srcFileMapName) + " (" + GetWindowsLastErrorMessage() + ")";
        CloseHandle(hDstFileMap);
        return false;
    }

    //-----------------------------------------------------------------------------
    // ファイルのビューをサーバープロセスのアドレス空間にマップして変換します。
    bool isSucceeded = false;
    int* pDstMem = reinterpret_cast<int*>(MapViewOfFile(hDstFileMap, FILE_MAP_WRITE, 0, 0, 0));
    if (pDstMem != nullptr)
    {
        const int* pSrcMem = reinterpret_cast<const int*>(MapViewOfFile(hSrcFileMap, FILE_MAP_READ, 0, 0, 0));
        if (pSrcMem != nullptr)
        {
            isSucceeded = ConvertCore(pDstMem, pSrcMem, params.dstFormat, params.srcFormat, params.quality,
                params.encodeFlag, params.dimension, params.imageW, params.imageH, params.imageD, params.mipCount);
            if (!isSucceeded)
            {
                *pErrMsg = "Cannot convert format";
            }
            UnmapViewOfFile(pSrcMem);
        }
        else
        {
            *pErrMsg = "Cannot map view of source file (" + GetWindowsLastErrorMessage() + ")";
        }
        UnmapViewOfFile(pDstMem);
    }
    else
    {
        *pErrMsg = "Cannot map view of destination file (" + GetWindowsLastErrorMessage() + ")";
    }
    CloseHandle(hDstFileMap);
    CloseHandle(hSrcFileMap);
    return isSucceeded;
}

//=============================================================================
//! @brief 監視スレッド情報の構造体です。
//=============================================================================
struct WatcherThreadInfo
{
    HANDLE hConnectEvent; //!< 接続イベントのハンドルです。
    HANDLE hConnectMutex; //!< プロセス間通信のためのミューテックスのハンドルです。
    bool isConnected; //!< クライアントからの接続があれば true です。
    bool isCanceled; //!< 一定時間クライアントからの接続がなくてキャンセルされたなら true です。
    const char* pipeName; //!< プロセス間通信のためのパイプ名です。
};

//-----------------------------------------------------------------------------
//! @brief 監視スレッドで実行する関数です。
//!
//! @param[in] pParam 引数へのポインターです。
//!
//! @return 終了コードを返します。
//-----------------------------------------------------------------------------
DWORD WINAPI WatcherThreadFunc(void* pParam)
{
    WatcherThreadInfo* pInfo = reinterpret_cast<WatcherThreadInfo*>(pParam);
    WaitForSingleObject(pInfo->hConnectEvent, GetServerTimeout()); // イベントがシグナル状態になるか一定時間経過するまで待機します。
    //NoteTrace("server time out: %d", pInfo->isConnected);
    if (!pInfo->isConnected)
    {
        // タイムアウト直後にクライアントから接続があった場合のために
        // ミューテックスが非シグナル状態（クライアントがロック中）なら
        // 接続イベントがシグナル状態になるまで待機します。
        HANDLE handles[2] = { pInfo->hConnectEvent, pInfo->hConnectMutex };
        WaitForMultipleObjects(2, handles, FALSE, INFINITE); // いずれかがシグナル状態になるまで待機（第 3 引数 fWaitAll が FALSE）
        //NoteTrace("server accessed: %d", pInfo->isConnected);
        if (!pInfo->isConnected)
        {
            pInfo->isCanceled = true;
            // RunServer の ConnectNamedPipe を終了させるためパイプのインスタンスに接続します。
            HANDLE hPipe = CreateFileA(pInfo->pipeName, GENERIC_READ | GENERIC_WRITE, 0, nullptr, OPEN_EXISTING, 0, nullptr);
            CloseHandle(hPipe);
        }
    }
    return 0;
}

//-----------------------------------------------------------------------------
//! @brief サーバーの処理を実行します。
//!        クライアントからの接続がある限りループします。
//!        クライアントからの接続が一定時間なければ正常終了します。
//!
//! @param[in] pipeName プロセス間通信のためのパイプ名です。
//! @param[in] mutexName プロセス間通信のためのミューテックス名です。
//! @param[in] entryFunc サーバープロセスで実行する関数です。
//! @param[in] inDataSize クライアントから受信するデータサイズです。
//!
//! @return 正常終了なら true を返します。
//-----------------------------------------------------------------------------
bool RunServer(
    const char* pipeName,
    const char* mutexName,
    ServerSideEntryType entryFunc,
    const size_t inDataSize
)
{
    bool serverResult = false;
    if (GetConsoleWindow() == nullptr)
    {
        //-----------------------------------------------------------------------------
        // ミューテックスを作成（取得）します。
        HANDLE hMutex;
        hMutex = CreateMutexA(nullptr, FALSE, mutexName);
        //NoteTrace("create mutex: %d: %p", GetLastError(), hMutex);
        if (hMutex == nullptr)
        {
            return serverResult;
        }

        //-----------------------------------------------------------------------------
        // サーバーのループ処理です。
        char* pipeInBuf = new char[inDataSize];
        while (true)
        {
            //-----------------------------------------------------------------------------
            // パイプを作成します。
            const int pipeMaxInstanceCount = 1;
            HANDLE hPipe = CreateNamedPipeA(pipeName,
                PIPE_ACCESS_DUPLEX | FILE_FLAG_FIRST_PIPE_INSTANCE,
                //PIPE_TYPE_BYTE | PIPE_READMODE_BYTE |
                PIPE_TYPE_MESSAGE | PIPE_READMODE_MESSAGE |
                PIPE_WAIT,
                pipeMaxInstanceCount, ServerMessageMaxSize,
                static_cast<DWORD>(inDataSize), 0, nullptr);
            if (hPipe == INVALID_HANDLE_VALUE)
            {
                break;
            }

            //-----------------------------------------------------------------------------
            // 監視スレッドを作成します。
            WatcherThreadInfo info;
            info.hConnectEvent = CreateEvent(nullptr, TRUE, FALSE, 0); // lpEventAttributes, bManualReset, bInitialState, lpName
            info.hConnectMutex = hMutex;
            info.isConnected = false;
            info.isCanceled = false;
            info.pipeName = pipeName;

            DWORD watcherThreadId;
            HANDLE hWatcherThread = CreateThread(nullptr, 0, WatcherThreadFunc, &info, 0, &watcherThreadId);

            //-----------------------------------------------------------------------------
            // クライアントまたは監視スレッドがパイプのインスタンスに接続するまで待機します。
            ConnectNamedPipe(hPipe, nullptr);
            if (info.isCanceled)
            {
                // 一定時間クライアントからの接続がなければサーバープロセスを終了します。
                CloseHandle(hPipe);
                WaitForSingleObject(hWatcherThread, INFINITE);
                CloseHandle(hWatcherThread);
                CloseHandle(info.hConnectEvent);
                serverResult = true;
                break;
            }
            info.isConnected = true;
            SetEvent(info.hConnectEvent); // イベントをシグナル状態にします。
            WaitForSingleObject(hWatcherThread, INFINITE); // 監視スレッドの終了を待ちます。
            CloseHandle(hWatcherThread);
            CloseHandle(info.hConnectEvent);

            //-----------------------------------------------------------------------------
            // クライアントからデータを受信して、処理を実行します。
            DWORD readBytes;
            if (!ReadFile(hPipe, pipeInBuf, static_cast<DWORD>(inDataSize), &readBytes, nullptr))
            {
                // パイプがメッセージモードで、送信されたメッセージ長が inDataSize より大きい場合、
                // ReadFile は FALSE を返し、エラーコードは ERROR_MORE_DATA (234) になります。
                if (GetLastError() != ERROR_MORE_DATA)
                {
                    CloseHandle(hPipe);
                    continue;
                }
            }
            std::string errMsg;
            const bool isSucceeded = entryFunc(&errMsg, pipeInBuf);

            //-----------------------------------------------------------------------------
            // 完了メッセージをクライアントに送信します。
            char pipeOutBuf[ServerMessageMaxSize];
            if (isSucceeded)
            {
                strcpy_s(pipeOutBuf, SuccessMessage.c_str());
            }
            else
            {
                strncpy_s(pipeOutBuf, (!errMsg.empty()) ? errMsg.c_str() : "Failed", _TRUNCATE);
                    // エラーメッセージの最大長は不定なので、バッファより大きければ切り捨てます。
                //NoteTrace("Error: %s", pipeOutBuf);
            }
            const int outStrLen = static_cast<int>(strlen(pipeOutBuf));
            DWORD writeBytes;
            WriteFile(hPipe, pipeOutBuf, outStrLen, &writeBytes, nullptr);
            //NoteTrace("write out str: %d / %d %d", writeBytes, outStrLen, GetLastError());
            FlushFileBuffers(hPipe);
            CloseHandle(hPipe);
        }
        delete[] pipeInBuf;
        CloseHandle(hMutex);
    }
    return serverResult;
}

//-----------------------------------------------------------------------------
//! @brief クライアントの処理を実行します。
//!
//! @param[out] pErrMsg 処理失敗時にエラーメッセージを格納します。
//! @param[in] pipeName プロセス間通信のためのパイプ名です。
//! @param[in] mutexName プロセス間通信のためのミューテックス名です。
//! @param[in] serverCmd サーバープロセスを起動するためのコマンドライン文字列です。
//! @param[in] pData サーバーに送信するデータへのポインターです。
//! @param[in] dataSize サーバーに送信するデータサイズです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool RunClient(
    std::string* pErrMsg,
    const char* pipeName,
    const char* mutexName,
    const char* serverCmd,
    const void* pData,
    const size_t dataSize
)
{
    //-----------------------------------------------------------------------------
    // ミューテックスを作成（取得）します。
    HANDLE hMutex;
    hMutex = CreateMutexA(nullptr, FALSE, mutexName);
    //cerr << "create mutex: " << GetLastError() << ": " << hMutex << endl;
    if (hMutex == nullptr)
    {
        *pErrMsg = "Cannot create mutex: " + std::string(mutexName) + " (" + GetWindowsLastErrorMessage() + ")";
        return false;
    }

    //-----------------------------------------------------------------------------
    // ミューテックスがシグナル状態になるまで待機します。
    WaitForSingleObject(hMutex, INFINITE);

    //-----------------------------------------------------------------------------
    // パイプを開きます。
    HANDLE hPipe = CreateFileA(pipeName, GENERIC_READ | GENERIC_WRITE, 0, nullptr, OPEN_EXISTING, 0, nullptr);
    if (hPipe == INVALID_HANDLE_VALUE)
    {
        // パイプが存在しない場合、サーバープロセスが起動していないので起動します。
        PROCESS_INFORMATION pi;
        STARTUPINFOA si;
        ZeroMemory(&pi, sizeof(pi));
        ZeroMemory(&si, sizeof(si));
        si.cb = sizeof(si);
        const size_t serverCmdLen = strlen(serverCmd) + 1;
        char* serverCmdCopy = new char[serverCmdLen];
        strcpy_s(serverCmdCopy, serverCmdLen, serverCmd);
        CreateProcessA(nullptr, serverCmdCopy, nullptr, nullptr, false, DETACHED_PROCESS, nullptr, nullptr, &si, &pi);
        delete[] serverCmdCopy;
        //cerr << "create server process: " << pi.dwProcessId << endl;
        while (hPipe == INVALID_HANDLE_VALUE)
        {
            hPipe = CreateFileA(pipeName, GENERIC_READ | GENERIC_WRITE, 0, nullptr, OPEN_EXISTING, 0, nullptr);
        }
    }

    //-----------------------------------------------------------------------------
    // パイプにデータをライトします。
    DWORD writeBytes;
    WriteFile(hPipe, pData, static_cast<DWORD>(dataSize), &writeBytes, nullptr);

    //-----------------------------------------------------------------------------
    // サーバーからの応答を待ちます。
    char pipeInBuf[ServerMessageMaxSize];
    bool isSucceeded = false;
    for (;;)
    {
        DWORD readBytes;
        if (!ReadFile(hPipe, pipeInBuf, sizeof(pipeInBuf), &readBytes, nullptr))
        {
            *pErrMsg = "Server does not respond (" + GetWindowsLastErrorMessage() + ")";
            break;
        }
        const std::string inStr(pipeInBuf, readBytes);
        //cerr << "read from server: " << readBytes << " [" << inStr << "]" << endl;

        if (inStr == SuccessMessage)
        {
            isSucceeded = true;
            break;
        }
        else
        {
            *pErrMsg = inStr;
            break;
        }
    }
    CloseHandle(hPipe);
    ReleaseMutex(hMutex);
    CloseHandle(hMutex);
    return isSucceeded;
}

//-----------------------------------------------------------------------------
// 無名名前空間を終了します。
} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief サーバープロセスのメイン関数です。
//!
//! @param[in] hwnd ウィンドウハンドルです。
//! @param[in] hinst インスタンスハンドルです。
//! @param[in] lpszCmdLine rundll32.exe で指定したコマンドライン引数です。
//! @param[in] nCmdShow ウィンドウの表示方法です。
//-----------------------------------------------------------------------------
extern "C" __declspec(dllexport) void WINAPI ServerMain(
    HWND hwnd,
    HINSTANCE hinst,
    LPSTR lpszCmdLine,
    int nCmdShow
)
{
    RunServer(PipeName, MutexName, ServerSideConvert, sizeof(ParamSet));
    NVTTDLL_UNUSED_VARIABLE(hwnd);
    NVTTDLL_UNUSED_VARIABLE(hinst);
    NVTTDLL_UNUSED_VARIABLE(lpszCmdLine);
    NVTTDLL_UNUSED_VARIABLE(nCmdShow);
}

//-----------------------------------------------------------------------------
//! @brief サーバープロセスで nvtt を使用して画像のフォーマットを変換します。
//-----------------------------------------------------------------------------
bool ConvertByServer(
    void* pDst,
    const void* pSrc,
    const size_t dstDataSize,
    const size_t srcDataSize,
    const std::string& dstFormatStr,
    const std::string& srcFormatStr,
    const std::string& qualityStr,
    const int encodeFlag,
    const int dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount
)
{
    //-----------------------------------------------------------------------------
    // 変換パラメーターセットを設定します。
    ParamSet params;
    params.pid = GetCurrentProcessId();
    //cerr << "param set size: " << sizeof(params) << endl;
    //cerr << "client pid: " << params.pid << endl;
    sprintf_s(params.dstFileMapName, "NintendoTextureConverterNvttDstPixels_%d", params.pid);
    sprintf_s(params.srcFileMapName, "NintendoTextureConverterNvttSrcPixels_%d", params.pid);
    strcpy_s(params.dstFormat, dstFormatStr.c_str());
    strcpy_s(params.srcFormat, srcFormatStr.c_str());
    strncpy_s(params.quality, qualityStr.c_str(), _TRUNCATE);
    params.encodeFlag = encodeFlag;
    params.dimension = dimension;
    params.imageW = imageW;
    params.imageH = imageH;
    params.imageD = imageD;
    params.mipCount = mipCount;

    //-----------------------------------------------------------------------------
    // エラー発生時に中断するための for 文です。
    bool isSucceeded = false;
    HANDLE hDstFileMap = nullptr;
    HANDLE hSrcFileMap = nullptr;
    for (;;)
    {
        //-----------------------------------------------------------------------------
        // ファイルマッピングオブジェクトを作成します。
        hDstFileMap = CreateFileMappingA(INVALID_HANDLE_VALUE, nullptr, PAGE_READWRITE,
            static_cast<DWORD>(dstDataSize >> 32), static_cast<DWORD>(dstDataSize & 0xffffffff),
            params.dstFileMapName);
        if (hDstFileMap == nullptr)
        {
            cerr << "Error: Cannot create file mapping: " << params.dstFileMapName
                 << " (" << GetWindowsLastErrorMessage() << ")" << endl;
            break;
        }

        hSrcFileMap = CreateFileMappingA(INVALID_HANDLE_VALUE, nullptr, PAGE_READWRITE,
            static_cast<DWORD>(srcDataSize >> 32), static_cast<DWORD>(srcDataSize & 0xffffffff),
            params.srcFileMapName);
        if (hSrcFileMap == nullptr)
        {
            cerr << "Error: Cannot create file mapping: " << params.srcFileMapName
                 << " (" << GetWindowsLastErrorMessage() << ")" << endl;
            break;
        }

        //-----------------------------------------------------------------------------
        // ファイルマッピングに変換前のピクセルデータをコピーします。
        void* pSrcMem = MapViewOfFile(hSrcFileMap, FILE_MAP_WRITE, 0, 0, 0);
        if (pSrcMem == nullptr)
        {
            cerr << "Error: Cannot map view of source file (client)"
                 << " (" << GetWindowsLastErrorMessage() << ")" << endl;
            break;
        }
        memcpy(pSrcMem, pSrc, srcDataSize);
        UnmapViewOfFile(pSrcMem);

        //-----------------------------------------------------------------------------
        // 変換を実行します。
        const std::string dllPath = GetCurrentDllPath();
        const std::string rundllPath = GetEnvVariable("SystemRoot") + "\\System32\\rundll32.exe";
        const std::string serverCmd = "\"" + rundllPath + "\" \"" + dllPath + "\",ServerMain";
        //cerr << "serverCmd: " << serverCmd << endl;
        std::string errMsg;
        if (!RunClient(&errMsg, PipeName, MutexName, serverCmd.c_str(), &params, sizeof(params)))
        {
            cerr << "Error: " << errMsg << endl;
            break;
        }

        //-----------------------------------------------------------------------------
        // ファイルマッピングから変換後のピクセルデータを取得します。
        void* pDstMem = MapViewOfFile(hDstFileMap, FILE_MAP_READ, 0, 0, 0);
        if (pDstMem == nullptr)
        {
            cerr << "Error: Cannot map view of destination file (client)"
                 << " (" << GetWindowsLastErrorMessage() << ")" << endl;
            break;
        }
        memcpy(pDst, pDstMem, dstDataSize);
        UnmapViewOfFile(pDstMem);
        isSucceeded = true;
        break;
    }

    //-----------------------------------------------------------------------------
    // ファイルマッピングオブジェクトを開放します。
    if (hDstFileMap != nullptr)
    {
        CloseHandle(hDstFileMap);
    }
    if (hSrcFileMap != nullptr)
    {
        CloseHandle(hSrcFileMap);
    }

    return isSucceeded;
}

