﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//=============================================================================
// include
//=============================================================================
#include <algorithm>
#include <cstdint>
#include <ctime>
#include <iostream>
#include <string>
#include <vector>

#include <windows.h>

//=============================================================================
// constants
//=============================================================================
const int NvttDllMaxPath = 512; //!< ファイルパスの最大長です。

//-----------------------------------------------------------------------------
// 未使用変数の警告を抑止するためのマクロです。
//-----------------------------------------------------------------------------
#define NVTTDLL_UNUSED_VARIABLE(Variable) (void)(&Variable);

//=============================================================================
//! @brief 時間計測のクラスです。
//=============================================================================
class NvttDllTimeMeasure
{
protected:
    std::clock_t m_StartClock; //!< 計測開始時のクロックです。

public:
    //! @brief コンストラクタです。
    NvttDllTimeMeasure()
    {
        Reset();
    }

    //! 計測開始時間を現在のクロックに設定します。
    void Reset()
    {
        m_StartClock = std::clock();
    }

    //! 計測開始時からの秒数を取得します。
    float GetSecond() const
    {
        return static_cast<float>(std::clock() - m_StartClock) / CLOCKS_PER_SEC;
    }

    //! 計測開始時からのミリ秒数を取得します。
    float GetMilliSecond() const
    {
        return GetSecond() * 1000.0f;
    }
};

//-----------------------------------------------------------------------------
//! @brief 開いているメモ帳のウィンドウにメッセージを出力します（デバッグ用）。
//-----------------------------------------------------------------------------
void NoteTrace(const char* format, ...);

//-----------------------------------------------------------------------------
//! @brief 環境変数の値を取得します。
//!
//! @param[in] name 環境変数の名前です。
//!
//! @return 環境変数の値を返します。環境変数が定義されていない場合は空文字を返します。
//-----------------------------------------------------------------------------
std::string GetEnvVariable(const char* name);

//-----------------------------------------------------------------------------
//! @brief Windows のシステムメッセージ文字列を取得します。
//!
//! @param[in] messageId メッセージ ID です。
//!
//! @return Windows のシステムメッセージ文字列を返します。
//-----------------------------------------------------------------------------
std::string GetWindowsSystemMessage(const int messageId);

//-----------------------------------------------------------------------------
//! @brief Windows の最後のエラーのメッセージ文字列を取得します。
//!
//! @return Windows の最後のエラーのメッセージ文字列を返します。
//-----------------------------------------------------------------------------
std::string GetWindowsLastErrorMessage();

//-----------------------------------------------------------------------------
//! @brief 現在の DLL ファイルのパスを取得します。
//!
//! @return 現在の DLL ファイルのパスを返します。
//-----------------------------------------------------------------------------
std::string GetCurrentDllPath();

//-----------------------------------------------------------------------------
//! @brief nvtt で画像のフォーマットを変換する処理のコア関数です。
//!
//! @param[out] pDst 変換後のデータを格納します。
//! @param[in] pSrc 変換前のデータです。
//! @param[in] dstFormatStr 変換後のフォーマット文字列です。
//! @param[in] srcFormatStr 変換前のフォーマット文字列です。
//! @param[in] qualityStr エンコード品質文字列です。
//! @param[in] encodeFlag エンコードフラグです。
//! @param[in] dimension 次元です。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageD 画像の奥行きです。
//! @param[in] mipCount ミップマップのレベル数です。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool ConvertCore(
    void* pDst,
    const void* pSrc,
    const std::string& dstFormatStr,
    const std::string& srcFormatStr,
    const std::string& qualityStr,
    const int encodeFlag,
    const int dimension,
    const int imageW,
    const int imageH,
    const int imageD,
    const int mipCount
);

