﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/util/util_BitUtil.h>

#include <nn/gfx/gfx_ResShaderData.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_ShaderConverterContext.h>

namespace nn {
namespace gfxTool {

nngfxToolResultCode ShaderConverterContext::Reset()
{
    try
    {
        m_BinarizerContext.Reset();
        m_CompilerManager.Reset();
        m_State = StaticCastAuto( State::Created );
    }
    catch( ... )
    {
        return ReportException();
    }

    return nngfxToolResultCode_Succeeded;
}

nngfxToolResultCode ShaderConverterContext::Convert(
    nngfxToolShaderConverterConvertOutput* pOutput,
    const nngfxToolShaderConverterConvertArg* pArg )
{
    try
    {
        if( !CheckState( State::Created ) || CheckState( State::Converted ) )
        {
            NN_GFXTOOL_THROW( nngfxToolResultCode_InvalidApiCall );
        }

        if( pArg->pCompileArg )
        {
            m_CompilerManager.Compile( pArg->pCompileArg );
            m_BinarizerContext.CalculateSize( pArg, m_CompilerManager.GetCompileOutput() );
        }
        else if( pArg->pMergeShaderFiles )
        {
            m_ShaderFileMerger.Merge( pArg );
            const_cast< nngfxToolShaderCompilerCompileArg*& >( pArg->pCompileArg ) =
                m_ShaderFileMerger.GetDummyCompileArg();
            m_BinarizerContext.CalculateSize(
                pArg, m_ShaderFileMerger.GetCompileOutput() );
            const_cast<nngfxToolShaderCompilerCompileArg*&>( pArg->pCompileArg ) = nullptr;
        }
        else
        {
            NN_GFXTOOL_THROW( nngfxToolResultCode_InvalidArgument );
        }

        if( pOutput )
        {
            pOutput->binarySize = StaticCastAuto( m_BinarizerContext.GetSize() );
            pOutput->binaryAlignment = StaticCastAuto( m_BinarizerContext.GetAlignment() );
            pOutput->pCompileOutput = m_CompilerManager.GetCompileOutput();
        }

        AddState( State::Converted );
    }
    catch( ... )
    {
        return ReportException();
    }

    return nngfxToolResultCode_Succeeded;
}

nngfxToolResultCode ShaderConverterContext::Serialize( void* pOutBinary, size_t binarySize )
{
    try
    {
        if( !CheckState( State::Converted ) || CheckState( State::Serialized ) )
        {
            NN_GFXTOOL_THROW( nngfxToolResultCode_InvalidApiCall );
        }

        memset( pOutBinary, 0, binarySize );
        m_BinarizerContext.Serialize( pOutBinary, binarySize );

        m_CompilerManager.DeleteCompileOutput();

        AddState( State::Serialized );
    }
    catch( ... )
    {
        return ReportException();
    }

    return nngfxToolResultCode_Succeeded;
}

ShaderConverterContext::HandleType ShaderConverterContext::Create()
{
    return reinterpret_cast< HandleType >( new ( std::nothrow ) ShaderConverterContext() );
}

nngfxToolResultCode ShaderConverterContext::Delete( ShaderConverterContext::HandleType handle )
{
    auto ptr = Get( handle );
    if( ptr == nullptr )
    {
        return nngfxToolResultCode_InvalidHandle;
    }
    delete ptr;
    return nngfxToolResultCode_Succeeded;
}

}
}
