﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <filesystem>

#include <nn/util/util_BitUtil.h>

#include <nn/gfx/gfx_ResShader.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_Common.h>
#include <gfxTool_ShaderBinarizerContext.h>
#include <gfxTool_ShaderCodeMerger.h>
#include <gfxTool_NvnShaderPoolBinarizer.h>

namespace nn {
namespace gfxTool {

void ShaderBinarizerContext::Reset()
{
    BaseBinarizerContext::Reset();

    m_ResShaderContainerBinarizer.Reset();
    m_Name.clear();
    m_pBinarizers.clear();
    m_Header.Initialize();
    m_PtrToNativeString.clear();
}

void ShaderBinarizerContext::CalculateSize( const nngfxToolShaderConverterConvertArg* pArg,
    const nngfxToolShaderCompilerCompileOutput* pOutput )
{
    Initialize( StaticCastAuto( Section::End ) );
    m_Name.assign( pArg->outputName.pValue, pArg->outputName.length );
    nn::util::string_view name( m_Name.data(), m_Name.length() );
    SetName( name );

    m_ResShaderContainerBinarizer.Initialize( pOutput, pArg );

    if( pArg->disableMergeProgram == 0 )
    {
        ShaderCodeMerger merger;
        merger.MergeCode( &m_ResShaderContainerBinarizer, pArg );
        NN_GFXTOOL_PRINT( "Merged shader code: %d -> %d\n",
            merger.GetAllCodeCount(), merger.GetAllCodeCount() - merger.GetAllMergedCodeCount() );
    }

    m_Header.SetSizeBy< nn::gfx::ResShaderFileData >();
    AddMemoryBlock( StaticCastAuto( Section::Common ), &m_Header );
    AddMemoryBlock( StaticCastAuto( Section::Common ), &m_ResShaderContainerBinarizer );
    m_ResShaderContainerBinarizer.RegisterChild( this );

    std::for_each( m_pBinarizers.begin(), m_pBinarizers.end(),
        []( BinarizerBase* pBinarizer ) { pBinarizer->CalculateSize(); } );

    if( pArg->disableMemoryPool == 0 &&
        pArg->pCompileArg->targetLowLevelApiType == nngfxToolShaderCompilerLowLevelApiType_Nvn )
    {
        if( pArg->pCompileArg->targetCodeType == nngfxToolShaderCompilerCodeType_Binary ||
            pArg->pCompileArg->targetCodeType == nngfxToolShaderCompilerCodeType_Binary_Source ||
            pArg->pCompileArg->targetCodeType == nngfxToolShaderCompilerCodeType_Binary_Ir )
        {
            size_t binarySectionSize = 0;
            for( auto block : m_ShaderBinaryBlocks )
            {
                binarySectionSize = nn::util::align_up( binarySectionSize, block->GetAlignment() );
                binarySectionSize += block->GetSize();
            }
            AddMemoryBlock( StaticCastAuto( Section::ShaderBinary ), &m_BinarySectionPaddingBlock );
            binarySectionSize += NvnShaderPoolBinarizer::GetCodeTailRoomSize();
            auto granularity = NvnShaderPoolBinarizer::GetMemoryPoolSizeGranularity();
            m_BinarySectionPaddingBlock.SetSize( NvnShaderPoolBinarizer::GetCodeTailRoomSize() +
                ( granularity - ( ( granularity - 1 ) & binarySectionSize ) ) );
        }
    }

    std::for_each( m_pBinarizers.begin(), m_pBinarizers.end(),
        [ this ]( BinarizerBase* pBinarizer ) { pBinarizer->Link( this ); } );

    auto pStringBlock = CalculateStringPool();
    AddMemoryBlock( StaticCastAuto( Section::StringPool ), pStringBlock );

    Calculate();
}

void ShaderBinarizerContext::Serialize( void* pOutMemory, size_t memorySize )
{
    if( GetSize() > memorySize )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_NotEnoughSize,
            "\n\tRequired: %d\nActual: %d", GetSize(), memorySize );
    }

    SetPtr( pOutMemory, GetSize() );

    std::for_each( m_pBinarizers.begin(), m_pBinarizers.end(),
        [ this ]( BinarizerBase* m_pBinarizer ) { m_pBinarizer->Convert( this ); } );

    auto pFileHeader = Convert();
    pFileHeader->signature.SetPacked( nn::gfx::ResShaderFile::Signature );
    pFileHeader->version.major = nn::gfx::ResShaderFile::MajorVersion;
    pFileHeader->version.minor = nn::gfx::ResShaderFile::MinorVersion;
    pFileHeader->version.micro = nn::gfx::ResShaderFile::MicroVersion;

    for( auto&& ptrToNativeString : m_PtrToNativeString )
    {
        auto pPtr = nn::util::BytePtr( ptrToNativeString.first->Get( GetPtr() ),
            ptrToNativeString.second ).Get< uint64_t >();
        *pPtr += offsetof( nn::util::BinString, _data );
    }
}

}
}
