﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <NVN/nvn.h>

#include <nn/gfx/gfx_Enum.h>
#include <nn/gfx/gfx_BufferData-api.nvn.8.h>
#include <nn/gfx/gfx_MemoryPoolData-api.nvn.8.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_NvnShaderPoolBinarizer.h>
#include <gfxTool_ShaderBinarizerContext.h>
#include <gfxTool_ResShaderContainerBinarizer.h>
#include <gfxTool_ResShaderVariationBinarizer.h>
#include <gfxTool_ResShaderProgramBinarizer.h>
#include <gfxTool_ShaderCodeBinarizer.h>

namespace nn {
namespace gfxTool {

void NvnShaderPoolBinarizer::Initialize( ResShaderContainerBinarizer* pParent )
{
    m_pParent = pParent;
    m_IsMemoryPoolEnable = pParent->GetArg()->disableMemoryPool == 0;
}

void NvnShaderPoolBinarizer::RegisterChild( ShaderBinarizerContext* pContext )
{
    if( m_IsMemoryPoolEnable )
    {
        pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), &m_MemoryPool );
    }
}

void NvnShaderPoolBinarizer::CalculateSize()
{
    SetSizeBy< ResTarget >();

    const int sizeOfMemoryPool = 320;
    NN_STATIC_ASSERT( sizeOfMemoryPool >=
        sizeof( nn::gfx::MemoryPoolImplData< nn::gfx::ApiVariationNvn8 > ) );
    m_MemoryPool.SetSize( sizeOfMemoryPool );
}

void NvnShaderPoolBinarizer::Link( ShaderBinarizerContext* pContext )
{
    const ResTarget* pTarget = nullptr;

    auto& shaderBinaryBlocks = pContext->GetShaderBinaryBlocks();
    if( shaderBinaryBlocks.size() > 0 )
    {
        pContext->LinkPtr( this, reinterpret_cast< const nn::util::BinPtr* >(
            &pTarget->memoryPoolInfo.pMemory ), shaderBinaryBlocks.front() );
    }
    if( m_IsMemoryPoolEnable )
    {
        pContext->LinkPtr( this, &pTarget->pMemoryPool, &m_MemoryPool );
    }
}

void NvnShaderPoolBinarizer::Convert( ShaderBinarizerContext* pContext )
{
    auto pTarget = Get< ResTarget >( pContext->GetPtr() );

    auto& shaderBinaryBlocks = pContext->GetShaderBinaryBlocks();

    pTarget->memoryPoolInfo.memoryPoolProperty = StaticCastAuto( nn::gfx::MemoryPoolProperty_CpuInvisible
        | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_ShaderCode );
    pTarget->memoryPoolInfo.memorySize = 0;
    // ShaderBinarizerContext 側でサイズ調整済み
    if( m_IsMemoryPoolEnable && shaderBinaryBlocks.size() > 0 )
    {
        pTarget->memoryPoolInfo.memorySize = NumericCastAuto( shaderBinaryBlocks.back()->GetPosition() +
            shaderBinaryBlocks.back()->GetSize() - shaderBinaryBlocks.front()->GetPosition() );
    }
}

size_t NvnShaderPoolBinarizer::GetMemoryPoolAlignment()
{
    return NVN_MEMORY_POOL_STORAGE_ALIGNMENT;
}

size_t NvnShaderPoolBinarizer::GetMemoryPoolSizeGranularity()
{
    return NVN_MEMORY_POOL_STORAGE_ALIGNMENT;
}

size_t NvnShaderPoolBinarizer::GetCodeTailRoomSize()
{
    return 1024;
}

}
}
