﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nvn/nvn.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_NvnShaderCodeBinarizer.h>
#include <gfxTool_ShaderBinarizerContext.h>
#include <gfxTool_ResShaderProgramBinarizer.h>
#include <gfxTool_ResShaderVariationBinarizer.h>
#include <gfxTool_NvnShaderPoolBinarizer.h>

namespace nn {
namespace gfxTool {

namespace {

static nn::util::BinTPtr< const void > nn::gfx::NvnDecomposedControlSection::* const s_pSubsections[
    static_cast< int  >( NvnDecomposedControlSectionBinarizer::ControlSubsectionType::End ) ] =
{
    &nn::gfx::NvnDecomposedControlSection::pMetaData,
    &nn::gfx::NvnDecomposedControlSection::pAssemblyData,
    &nn::gfx::NvnDecomposedControlSection::pSpecializationData,
    &nn::gfx::NvnDecomposedControlSection::pPragmaData,
    &nn::gfx::NvnDecomposedControlSection::pAssemblyLocalsData,
    &nn::gfx::NvnDecomposedControlSection::pUniform64InfoData
};

}

NvnShaderCodeBinarizer::NvnShaderCodeBinarizer()
    : ShaderCodeBinarizerBase()
{
}

NvnShaderCodeBinarizer::~NvnShaderCodeBinarizer() = default;

void NvnShaderCodeBinarizer::Initialize( const BinarizationTarget* pTarget,
    const ResShaderProgramBinarizer* pParent, ShaderStage stage )
{
    m_pTarget = pTarget;
    m_pParent = pParent;
    m_Stage = stage;

    m_DataBlock.merge = -1;
    m_ControlBlock.merge = -1;

    auto pArg = pParent->GetParent()->GetParent()->GetArg();
    const nngfxToolShaderCompilerVariationDefinition*
        nngfxToolShaderCompilerVariationDefinitionArg::*
        const pStageVariationDefinition[ static_cast< int >( ShaderStage::End ) ] =
    {
        &nngfxToolShaderCompilerVariationDefinitionArg::pVertexShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pHullShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pDomainShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pGeometryShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pPixelShaderVariationDefinition,
        &nngfxToolShaderCompilerVariationDefinitionArg::pComputeShaderVariationDefinition
    };
    if( pArg->decomposeBinary && pArg->pCompileArg->targetCodeType
        == nngfxToolShaderCompilerCodeType_Binary_Ir )
    {
        // 効果が期待できるのは定数バリエーションを使用しているときのみだが、
        // マージ経由の場合は容易にわからないこともあり、指定されたとおりにかける
        m_pDecomposedControlSectionBinarizer.reset( new NvnDecomposedControlSectionBinarizer() );
        m_pDecomposedControlSectionBinarizer->Initialize( pTarget->pControl.Get(), this );

        m_pFirstBinarizer = StaticCastAuto( m_pParent->GetParent()->GetParent()->GetVariationBinarizers(
            )->at( 0 ).GetBinaryProgramBinarizer()->GetShaderCodeBinarizer( m_Stage ) );
    }
}

void NvnShaderCodeBinarizer::RegisterChild( ShaderBinarizerContext* pContext )
{
    if( m_DataBlock.merge < 0 )
    {
        m_IsFirstBinary = pContext->GetShaderBinaryBlocks().size() < 1;
        pContext->AddMemoryBlock( StaticCastAuto( Section::ShaderBinary ), &m_DataBlock.block );
    }

    if( m_pDecomposedControlSectionBinarizer.get() )
    {
        pContext->AddMemoryBlock( StaticCastAuto( Section::Common ),
            m_pDecomposedControlSectionBinarizer.get() );
        m_pDecomposedControlSectionBinarizer->RegisterChild( pContext );

        if( this == m_pFirstBinarizer )
        {
            // 再構築用のスペース
            pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), &m_ControlBlock.block );
        }
    }
    else
    {
        if( m_ControlBlock.merge < 0 )
        {
            pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), &m_ControlBlock.block );
        }
    }

    if( m_pTarget->pNvnDebugDataHash.Get() )
    {
        pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), &m_DebugHashBlock );
    }
}

void NvnShaderCodeBinarizer::CalculateSize()
{
    SetSizeBy< ResTarget >();

    if( m_DataBlock.merge < 0 )
    {
        size_t alignment = GetShaderCodeAlignment();
        if( m_IsFirstBinary && GetParent()->GetParent()->GetParent()->GetArg()->disableMemoryPool == 0 )
        {
            alignment = NvnShaderPoolBinarizer::GetMemoryPoolAlignment();
        }
        m_DataBlock.block.SetAlignment( alignment );
        m_DataBlock.block.SetSize( StaticCastAuto( m_pTarget->dataSize ) );
    }

    if( m_pDecomposedControlSectionBinarizer.get() )
    {
        if( this == m_pFirstBinarizer )
        {
            auto pVariationBinarizers = m_pParent->GetParent()->GetParent()->GetVariationBinarizers();
            size_t maxControlSize = 0;
            for( auto&& variationBinarizer : *pVariationBinarizers )
            {
                auto pNvnShaderCodeBinarizer = static_cast< const NvnShaderCodeBinarizer* >(
                    variationBinarizer.GetBinaryProgramBinarizer()->GetShaderCodeBinarizer( m_Stage ) );
                maxControlSize = std::max NN_PREVENT_MACRO_FUNC(
                    maxControlSize, static_cast< size_t >( pNvnShaderCodeBinarizer->m_pTarget->controlSize ) );
            }
            m_ControlBlock.block.SetSize( maxControlSize );
        }
    }
    else
    {
        if( m_ControlBlock.merge < 0 )
        {
            // 制御メモリにアライメントは不要
            m_ControlBlock.block.SetSize( StaticCastAuto( m_pTarget->controlSize ) );
        }
    }

    if( m_pTarget->pNvnDebugDataHash.Get() )
    {
        m_DebugHashBlock.SetSizeBy< NVNdebugDataHash >();
    }
}

void NvnShaderCodeBinarizer::Link( ShaderBinarizerContext* pContext )
{
    ResTarget* pResTarget = nullptr;

    auto pVariationBinarizers = m_pParent->GetParent()->GetParent()->GetVariationBinarizers();

    if( m_DataBlock.merge < 0 )
    {
        pContext->LinkPtr( this, &pResTarget->pData, &m_DataBlock.block );
    }
    else
    {
        auto pMergeBinerizer = pVariationBinarizers->at( m_DataBlock.merge ).GetBinaryProgramBinarizer(
        )->GetShaderCodeBinarizer< NvnShaderCodeBinarizer >( m_Stage );
        pContext->LinkPtr( this, &pResTarget->pData, &pMergeBinerizer->m_DataBlock.block );
    }

    if( m_pDecomposedControlSectionBinarizer.get() )
    {
        pContext->LinkPtr( this, &pResTarget->pDecomposedControlSection,
            m_pDecomposedControlSectionBinarizer.get() );
        pContext->LinkPtr( this, &pResTarget->pControl, &m_pFirstBinarizer->m_ControlBlock.block );
    }
    else
    {
        if( m_ControlBlock.merge < 0 )
        {
            pContext->LinkPtr( this, &pResTarget->pControl, &m_ControlBlock.block );
        }
        else
        {
            auto pMergeBinarizer = pVariationBinarizers->at( m_ControlBlock.merge ).GetBinaryProgramBinarizer(
            )->GetShaderCodeBinarizer< NvnShaderCodeBinarizer >( m_Stage );
            pContext->LinkPtr( this, &pResTarget->pControl, &pMergeBinarizer->m_ControlBlock.block );
        }
    }

    if( m_pTarget->pNvnDebugDataHash.Get() )
    {
        pContext->LinkPtr( this, &pResTarget->pNvnDebugDataHash, &m_DebugHashBlock );
    }
}

void NvnShaderCodeBinarizer::Convert( ShaderBinarizerContext* pContext )
{
    auto pResTarget = Get< ResTarget >( pContext->GetPtr() );
    pResTarget->dataAddress = 0;
    pResTarget->dataSize = m_pTarget->dataSize;
    pResTarget->controlSize = m_pTarget->controlSize;
    pResTarget->scratchMemoryRecommended = m_pTarget->scratchMemoryRecommended;
    pResTarget->scratchMemoryPerWarp = m_pTarget->scratchMemoryPerWarp;

    if( m_DataBlock.merge < 0 )
    {
        memcpy( m_DataBlock.block.Get( pContext->GetPtr() ),
            m_pTarget->pData.Get(), m_pTarget->dataSize );
    }

    if( !m_pDecomposedControlSectionBinarizer.get() && m_ControlBlock.merge < 0 )
    {
        memcpy( m_ControlBlock.block.Get( pContext->GetPtr() ),
            m_pTarget->pControl.Get(), m_pTarget->controlSize );
    }

    if( m_pTarget->pNvnDebugDataHash.Get() )
    {
        memcpy( m_DebugHashBlock.Get( pContext->GetPtr() ),
            m_pTarget->pNvnDebugDataHash.Get(), sizeof( NVNdebugDataHash ) );
    }
}

int NvnShaderCodeBinarizer::GetShaderCodeAlignment()
{
    return 256;
}

void NvnDecomposedControlSectionBinarizer::Initialize(
    const void* pTarget, const NvnShaderCodeBinarizer* pParent )
{
    m_pTarget = pTarget;
    m_pParent = pParent;

    for( auto&& subsection : m_ControlSubsections )
    {
        subsection.mergeBlock.merge = -1;
    }

    // マージで参照するために先にサイズも計算する
    const int offsetDataOffset[ static_cast< int >( ControlSubsectionType::End ) ] =
    {
        0, 20, 1776, 1992, 32, 2024
    };
    const int offsetDataSize[ static_cast<int>( ControlSubsectionType::End ) ] =
    {
        0, 24, 1780, 1996, 28, 2028
    };
    for( int idxSubsection = 1; idxSubsection < static_cast<int>(
        ControlSubsectionType::End ); ++idxSubsection )
    {
        ptrdiff_t offset = *nn::util::ConstBytePtr( pTarget, offsetDataOffset[ idxSubsection ] ).Get< int32_t >();
        size_t size = *nn::util::ConstBytePtr( pTarget, offsetDataSize[ idxSubsection ] ).Get< int32_t >();
        m_ControlSubsections[ idxSubsection ].pData = nn::util::ConstBytePtr( pTarget, offset ).Get();
        m_ControlSubsections[ idxSubsection ].mergeBlock.block.SetSize( size );
    }
    m_ControlSubsections[ static_cast< int >( ControlSubsectionType::MetaData ) ].pData = pTarget;
    m_ControlSubsections[ static_cast< int >( ControlSubsectionType::MetaData ) ].mergeBlock.block.SetSize(
        nn::util::ConstBytePtr( pTarget ).Distance( m_ControlSubsections[
        static_cast< int >( ControlSubsectionType::AssemblyData ) ].pData ) );
}

void NvnDecomposedControlSectionBinarizer::RegisterChild( ShaderBinarizerContext* pContext )
{
    for( auto&& subsection : m_ControlSubsections )
    {
        if( subsection.mergeBlock.merge < 0 && subsection.mergeBlock.block.GetSize() > 0 )
        {
            pContext->AddMemoryBlock( StaticCastAuto( Section::Common ), &subsection.mergeBlock.block );
        }
    }
}

void NvnDecomposedControlSectionBinarizer::CalculateSize()
{
    SetSizeBy< ResTarget >();

    // サイズ計算済み
}

void NvnDecomposedControlSectionBinarizer::Link( ShaderBinarizerContext* pContext )
{
    const ResTarget* pResTarget = nullptr;

    auto pVariationBinarizers = m_pParent->GetParent()->GetParent()->GetParent()->GetVariationBinarizers();

    for( int idxSubsection = 0; idxSubsection < static_cast<int>(
        ControlSubsectionType::End ); ++idxSubsection )
    {
        auto& subsection = m_ControlSubsections[ idxSubsection ];
        if( subsection.mergeBlock.merge < 0 )
        {
            pContext->LinkPtr( this, &( pResTarget->*s_pSubsections[ idxSubsection ] ),
                &subsection.mergeBlock.block );
        }
        else
        {
            auto pMergeBinarizer = pVariationBinarizers->at(
                subsection.mergeBlock.merge ).GetBinaryProgramBinarizer(
                )->GetShaderCodeBinarizer< NvnShaderCodeBinarizer>(
                    m_pParent->GetStage() )->GetDecomposedControlSectionBinarizer();
            pContext->LinkPtr( this, &( pResTarget->*s_pSubsections[ idxSubsection ] ),
                &pMergeBinarizer->m_ControlSubsections[ idxSubsection ].mergeBlock.block );
        }
    }
}

void NvnDecomposedControlSectionBinarizer::Convert( ShaderBinarizerContext* pContext )
{
    for( int idxSubsection = 0; idxSubsection < static_cast<int>(
        ControlSubsectionType::End ); ++idxSubsection )
    {
        auto subsection = m_ControlSubsections[ idxSubsection ];
        if( subsection.mergeBlock.merge < 0 )
        {
            memcpy( subsection.mergeBlock.block.Get( pContext->GetPtr() ),
                subsection.pData, subsection.mergeBlock.block.GetSize() );
        }
    }
}

}
}
