﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <filesystem>

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>

#include <nn/gfx/gfx_ResShader.h>

#include <gfxTool_ConvertArg.h>
#include <gfxTool_CommandLineArg.h>
#include <gfxTool_CompileArg.h>
#include <gfxTool_CreateConvertArgOutput.h>
#include <gfxTool_ShaderConverterContext.h>

namespace {

void CheckRequiredOption( const nn::gfxTool::CommandLineOptionBase& option )
{
    if( !option.IsExisting() )
    {
        auto& definition = option.GetDefinition();
        if( definition.pLongName )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_LackArgument,
                "--%s is required.", definition.pLongName );
        }
        else
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_LackArgument,
                "-%c is required.", definition.shortName );
        }
    }
}

}

namespace nn {
namespace gfxTool {

ConvertArg::ConvertArg()
    : nngfxToolShaderConverterConvertArg()
{
}

ConvertArg::~ConvertArg() /* = default; */
{
}

nngfxToolResultCode ConvertArg::Initilaize(
    const nngfxToolShaderConverterCreateConvertArgArg* pArg )
{
    try
    {
        CommandLineParser parser;
        CommandLineArg input;
        InitializeOptions( &input, &parser );
        parser.Parse( pArg->optionCount, pArg->ppOptionArray );
        if( input.silent.IsExisting() )
        {
            Logger::SetLogOutputLevel( Logger::LogType::Default, Logger::LogLevel::Silent );
        }
        if( input.help.IsExisting() )
        {
            auto handle = GetModuleHandleA( nullptr );
            char name[ MAX_PATH ] = {};
            GetModuleFileNameA( handle, name, MAX_PATH );
            parser.PrintHelp( std::tr2::sys::path( name ).filename().string().data(),
                "This tool creates shader resources for gfx." );
        }
        if( input.version.IsExisting() )
        {
            auto handle = GetModuleHandleA( nullptr );
            char name[ MAX_PATH ] = {};
            GetModuleFileNameA( handle, name, MAX_PATH );
            NN_GFXTOOL_PRINT( "%s\n\tversion: %d.%d.%d\n", // TODO
                std::tr2::sys::path( name ).filename().string().data(),
                ShaderConverterContext::MajorVersion,
                ShaderConverterContext::MinorVersion,
                ShaderConverterContext::MicroVersion );
        }
        parser.CheckRequiredOption();

        if( input.outputName.IsExisting() )
        {
            m_Name = input.outputName.GetValue();
        }
        else if( input.outputPath.IsExisting() )
        {
            m_Name = std::tr2::sys::path( input.outputPath.GetValue() ).stem().string();
        }
        else
        {
            m_Name = "";
        }
        this->outputName.pValue = m_Name.c_str();
        this->outputName.length = NumericCastAuto( m_Name.length() );

        if( input.mergeShaderFile.IsExisting() )
        {
            this->mergeShaderFileCount = NumericCastAuto( input.mergeShaderFile.GetValue().size() );
            m_MergeShaderFiles.reserve( input.mergeShaderFile.GetValue().size() );
            for( int idxMergeShaderFile = 0; idxMergeShaderFile < this->mergeShaderFileCount; ++idxMergeShaderFile )
            {
                void* pFileData = nullptr;
                size_t fileSize;
                if( pArg->pReadInputFileCallback && pArg->pReadInputFileCallback( &pFileData, &fileSize,
                    input.mergeShaderFile.GetValue().at( idxMergeShaderFile ).c_str(), pArg->pReadInputFileCallbackParam ) )
                {
                    m_MergeShaderFiles.push_back( pFileData );
                }
                else
                {
                    NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_FailedToLoadFile,
                        "%s", input.mergeShaderFile.GetValue().at( idxMergeShaderFile ).c_str() );
                }
            }
            this->pMergeShaderFiles = &m_MergeShaderFiles[ 0 ];
            this->pCompileArg = nullptr;
        }
        else
        {
            CheckRequiredOption( input.apiType );
            CheckRequiredOption( input.codeType );
            CheckRequiredOption( input.sourceFormat );
            m_pCompileArg.reset( new CompileArg() );
            m_pCompileArg->Initialize( pArg, &input );
            this->pCompileArg = m_pCompileArg.get();
            this->mergeShaderFileCount = 0;
            this->pMergeShaderFiles = nullptr;
        }

        m_pCreateConvertArgOutput.reset( new CreateConvertArgOutput() );
        m_pCreateConvertArgOutput->Initialize( &input );
        this->pInformation = m_pCreateConvertArgOutput.get();

        this->disableMemoryPool = input.disableMemoryPool.IsExisting() ? 1 : 0;
        this->decomposeBinary = input.decomposeBinary.IsExisting() ? 1 : 0;
        this->enableFullReflection = input.reflectionFull.IsExisting() ? 1 : 0;
        this->disableMergeProgram = input.disableMergeProgram.IsExisting() ? 1 : 0;
    }
    catch( ... )
    {
        return ReportException();
    }

    return nngfxToolResultCode_Succeeded;
}

}
}
