﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <thread>

#include <gfxTool_CompileOption.h>
#include <gfxTool_CommandLineArg.h>
#include <gfxTool_CompileOption-glsl.h>
#include <gfxTool_CompileOption-gx.h>
#include <gfxTool_CompileOption-nvn.h>
#include <gfxTool_CompileOption-vk.h>

namespace nn {
namespace gfxTool {

void CompileOptionCommon::Initialize( const CommandLineArg* pInput )
{
    this->printProgress = pInput->printProgress.IsExisting() ? 0.1f : 1.1f;
    this->dump = NumericCastAuto( pInput->dumpDirectory.IsExisting() ?
        nngfxToolShaderCompilerDump_FullDump : nngfxToolShaderCompilerDump_NoDump );
    this->isShaderStatisticsEnabled = pInput->dumpDirectory.IsExisting() ? 1 : 0;
    this->isSeparationEnabled = pInput->separable.IsExisting() ? 1 : 0;
    this->isReflectionEnabled = pInput->reflection.IsExisting() || pInput->reflectionFull.IsExisting() ? 1 : 0;
    this->isCompressionEnabled = 1;
    this->isPreprocessEnabled = ( pInput->preprocess.IsExisting() ||
        pInput->invertGlPositionY.IsExisting() || pInput->remapGlPositionZ.IsExisting() ) ? 1 : 0;
    this->isInvertYEnabled = pInput->invertGlPositionY.IsExisting() ? 1 : 0;
    this->isRemapZEnabled = pInput->remapGlPositionZ.IsExisting() ? 1 : 0;
    this->logOutputLevel = StaticCastAuto( pInput->silent.IsExisting() ?
        nngfxToolShaderCompilerLogOutputLevel_Silent : nngfxToolShaderCompilerLogOutputLevel_Default );
    this->debugInfoLevel = nngfxToolShaderCompilerDebugInfoLevel_None;
    if( pInput->debugInfoDirectory.IsExisting() )
    {
        this->debugInfoLevel = nngfxToolShaderCompilerDebugInfoLevel_Debug;
    }
    if( pInput->debugInfoLevel.IsExisting() )
    {
        this->debugInfoLevel = StaticCastAuto( pInput->debugInfoLevel.GetValue() );
    }

    if( pInput->jobs.IsExisting() )
    {
        this->maxThreads = NumericCastAuto( pInput->jobs.GetValue() );
    }
    else if( pInput->maxThread.IsExisting() )
    {
        this->maxThreads = NumericCastAuto( pInput->maxThread.GetValue() );
    }
    else
    {
        this->maxThreads = NumericCastAuto( std::thread::hardware_concurrency() );
    }

    this->codePage = pInput->codePage.IsExisting() ?
        NumericCast< decltype( this->codePage ) >( pInput->codePage.GetValue() ) : 0;

    if( pInput->uniformRegisterBlockName.IsExisting() )
    {
        m_UniformRegisterBlockName = pInput->uniformRegisterBlockName.GetValue();
        TonngfxToolString( &this->uniformRegisterBlockName, m_UniformRegisterBlockName );
    }

    if( pInput->preprocessorDefinitions.IsExisting() )
    {
        auto& preprocessorDefinitions = pInput->preprocessorDefinitions.GetValue();
        m_PreprocessorDefinitionArray.resize( preprocessorDefinitions.size() );
        m_PreprocessorDefinitionArrayRaw.resize( preprocessorDefinitions.size() );
        for( int idxPreprocessorDefinition = 0, preprocessorDefinitionCountMax =
            NumericCastAuto( preprocessorDefinitions.size() );
            idxPreprocessorDefinition < preprocessorDefinitionCountMax; ++idxPreprocessorDefinition )
        {
            m_PreprocessorDefinitionArray[ idxPreprocessorDefinition ].assign(
                preprocessorDefinitions[ idxPreprocessorDefinition ].data(),
                preprocessorDefinitions[ idxPreprocessorDefinition ].length() );
            std::replace( m_PreprocessorDefinitionArray[ idxPreprocessorDefinition ].begin(),
                m_PreprocessorDefinitionArray[ idxPreprocessorDefinition ].end(), '=', ' ' );
            m_PreprocessorDefinitionArrayRaw[ idxPreprocessorDefinition ].pValue =
                m_PreprocessorDefinitionArray[ idxPreprocessorDefinition ].data();
            m_PreprocessorDefinitionArrayRaw[ idxPreprocessorDefinition ].length =
                NumericCastAuto( m_PreprocessorDefinitionArray[ idxPreprocessorDefinition ].length() );
        }

        this->preprocessorDefinitionCount = NumericCastAuto( preprocessorDefinitions.size() );
        this->pPreprocessorDefinitionArray = &m_PreprocessorDefinitionArrayRaw[ 0 ];
    }
}

CompileOption::CompileOption() /* = default; */
{
}

CompileOption::~CompileOption() /* = default; */
{
}

void CompileOption::Initialize( const CommandLineArg* pInput )
{
    m_pCompileOptionCommon.reset( new CompileOptionCommon() );
    m_pCompileOptionCommon->Initialize( pInput );
    AddCompileOption( nngfxToolShaderCompilerOptionType_Common, m_pCompileOptionCommon.get() );

    if( pInput->sourceFormat.GetValue() == SourceFormat::Glsl )
    {
        m_pCompileOptionGlsl.reset( new CompileOptionGlsl() );
        m_pCompileOptionGlsl->Initialize( pInput );
        AddCompileOption( nngfxToolShaderCompilerOptionType_Glsl, m_pCompileOptionGlsl.get() );
    }

    if( pInput->apiType.GetValue() == ApiType::Gx )
    {
        m_pCompileOptionGx.reset( new CompileOptionGx() );
        m_pCompileOptionGx->Initialize( pInput );
        AddCompileOption( StaticCastAuto( nngfxToolShaderCompilerOptionType_Gx ),
            m_pCompileOptionGx.get() );
    }

    if( pInput->apiType.GetValue() == ApiType::Nvn &&
        pInput->glslcOptionFlags.IsExisting() &&
        pInput->glslcOptionFlags.GetValue().size() > 0 )
    {
        m_pCompileOptionNvn.reset( new CompileOptionNvn() );
        m_pCompileOptionNvn->Initialize( pInput );
        AddCompileOption( StaticCastAuto( nngfxToolShaderCompilerOptionType_Nvn ),
            m_pCompileOptionNvn.get() );
    }

    if( pInput->apiType.GetValue() == ApiType::Vk )
    {
        m_pCompileOptionVk.reset( new CompileOptionVk() );
        m_pCompileOptionVk->Initialize( pInput );
        AddCompileOption( StaticCastAuto( nngfxToolShaderCompilerOptionType_Vk ),
            m_pCompileOptionVk.get() );
    }
}

void CompileOption::AddCompileOption( nngfxToolShaderCompilerOptionType optionType, void* pOption )
{
    m_CompileOptionArrayRaw.emplace_back();
    m_CompileOptionArrayRaw.back().optionType = optionType;
    m_CompileOptionArrayRaw.back().pOption = pOption;
}

}
}
