﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>

#include <nvnTool/nvnTool_GlslcInterface.h>

#include <nn/util/util_StringView.h>

#include <gfxTool_CompileOption-nvn.h>
#include <gfxTool_CommandLineArg.h>

namespace {

template< typename TOption, typename TCommandLineOption >
void AssignIfExisting( TOption* pOption, const TCommandLineOption& commandLineOption )
{
    if( commandLineOption.IsExisting() )
    {
        *pOption = nn::gfxTool::StaticCastAuto( commandLineOption.GetValue() );
    }
}
template< typename TOption, typename TCommandLineOption, int TableSize >
void AssignIfExisting( TOption* pOption, const TCommandLineOption& commandLineOption,
    const char* const ( &pEnumToStringTable )[ TableSize ] )
{
    if( commandLineOption.IsExisting() )
    {
        auto pValue = commandLineOption.GetValue().c_str();
        auto pEnd = pEnumToStringTable + TableSize;
        auto find = std::find_if( pEnumToStringTable, pEnd, [ pValue ]( const char* pString )
        {
            return strcmp( pString, pValue ) == 0;
        } );
        if( find == pEnd )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "%s is invalid.", pValue );
        }
        *pOption = nn::gfxTool::StaticCastAuto( std::distance( pEnumToStringTable, find ) );
    }
}

struct CommandLineArgGlslcOptionFlags
{
    nn::gfxTool::CommandLineOption< nn::gfxTool::Custom< std::string >::Type > spillControl;
    nn::gfxTool::CommandLineOption< int > tessellationAndPassthroughGS;
    nn::gfxTool::CommandLineOption< nn::gfxTool::Custom< std::string >::Type > enableFastMathMask;
    nn::gfxTool::CommandLineOption< nn::gfxTool::Custom< std::string >::Type > optLevel;
    nn::gfxTool::CommandLineOption< nn::gfxTool::Custom< std::string >::Type > unrollControl;
    nn::gfxTool::CommandLineOption< int > errorOnScratchMemUsage;
    nn::gfxTool::CommandLineOption< int > enableCBFOptimization;
    nn::gfxTool::CommandLineOption< int > enableWarpCulling;
    nn::gfxTool::CommandLineOption< nn::gfxTool::Custom< std::string >::Type > warnUninitControl;
};

void InitializeOptions( CommandLineArgGlslcOptionFlags* pInput, nn::gfxTool::CommandLineParser* pParser )
{
    pParser->SetOption( pInput->spillControl.Define( false, "spillControl", "" ) );
    pParser->SetOption( pInput->tessellationAndPassthroughGS.Define( false, "tessellationAndPassthroughGS", "" ) );
    pParser->SetOption( pInput->enableFastMathMask.Define( false, "enableFastMathMask", "" ) );
    pParser->SetOption( pInput->optLevel.Define( false, "optLevel", "" ) );
    pParser->SetOption( pInput->unrollControl.Define( false, "unrollControl", "" ) );
    pParser->SetOption( pInput->errorOnScratchMemUsage.Define( false, "errorOnScratchMemUsage", "" ) );
    pParser->SetOption( pInput->enableCBFOptimization.Define( false, "enableCBFOptimization", "" ) );
    pParser->SetOption( pInput->enableWarpCulling.Define( false, "enableWarpCulling", "" ) );
    pParser->SetOption( pInput->warnUninitControl.Define( false, "warnUninitControl", "" ) );
}

}

namespace nn {
namespace gfxTool {

CompileOptionNvn::CompileOptionNvn()
    : nngfxToolShaderCompilerCompileOptionNvn()
{
    this->spillControl = DEFAULT_SPILL;
    this->tessellationAndPassthroughGS = 0;
    this->enableFastMathMask = NVN_SHADER_STAGE_FRAGMENT_BIT;
    this->optLevel = GLSLC_OPTLEVEL_DEFAULT;
    this->unrollControl = GLSLC_LOOP_UNROLL_DEFAULT;
    this->errorOnScratchMemUsage = 0;
    this->enableCBFOptimization = 0;
    this->enableWarpCulling = 0;
}

void CompileOptionNvn::Initialize( const CommandLineArg* pInput )
{
    if( !pInput->glslcOptionFlags.IsExisting() )
    {
        return;
    }

    CommandLineParser parser;
    CommandLineArgGlslcOptionFlags argGlslc;
    InitializeOptions( &argGlslc, &parser );

    auto glslcOptionFlagsString = pInput->glslcOptionFlags.GetValue();
    CommandLineParser::CommandLineArg args(
        nn::util::string_view( glslcOptionFlagsString.c_str(), glslcOptionFlagsString.size() ) );
    parser.Parse( args.GetArgC(), args.GetArgv() );

    const char* const SpillControlStrings[] =
    {
        "DEFAULT_SPILL",
        "NO_SPILL"
    };

    const char* const OptLevelStrings[] =
    {
        "GLSLC_OPTLEVEL_DEFAULT",
        "GLSLC_OPTLEVEL_NONE"
    };

    const char* const UnrollControlStrings[] =
    {
        "GLSLC_LOOP_UNROLL_DEFAULT",
        "GLSLC_LOOP_UNROLL_NONE",
        "GLSLC_LOOP_UNROLL_ALL"
    };

    const char* const WarnUninitControlStrings[] =
    {
        "GLSLC_WARN_UNINIT_DEFAULT",
        "GLSLC_WARN_UNINIT_NONE",
        "GLSLC_WARN_UNINIT_ALL"
    };

    AssignIfExisting( &this->tessellationAndPassthroughGS, argGlslc.tessellationAndPassthroughGS );
    AssignIfExisting( &this->errorOnScratchMemUsage, argGlslc.errorOnScratchMemUsage );
    AssignIfExisting( &this->enableWarpCulling, argGlslc.enableWarpCulling );
    AssignIfExisting( &this->enableCBFOptimization, argGlslc.enableCBFOptimization );

    AssignIfExisting( &this->spillControl, argGlslc.spillControl, SpillControlStrings );
    AssignIfExisting( &this->optLevel, argGlslc.optLevel, OptLevelStrings );
    AssignIfExisting( &this->unrollControl, argGlslc.unrollControl, UnrollControlStrings );
    AssignIfExisting( &this->warnUninitControl, argGlslc.warnUninitControl, WarnUninitControlStrings );

    if( argGlslc.enableFastMathMask.IsExisting() )
    {
        // TODO: 列挙子と OR 演算への対応
        this->enableFastMathMask = LexicalCastAuto( argGlslc.enableFastMathMask.GetValue() );
    }
}

}
}
