﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/gfx/gfx_ResShaderData-api.nvn.h>

#include <gfxTool_ShaderCodeBinarizer.h>

namespace nn {
namespace gfxTool {

class NvnDecomposedControlSectionBinarizer;

class NvnShaderCodeBinarizer
    : public ShaderCodeBinarizerBase
{
public:
    typedef nn::gfx::NvnShaderCode ResTarget;
    typedef nn::gfx::NvnShaderCode BinarizationTarget;

    NvnShaderCodeBinarizer();

    ~NvnShaderCodeBinarizer();

    void Initialize( const BinarizationTarget* pTarget,
        const ResShaderProgramBinarizer* pParent, ShaderStage stage );

    virtual void RegisterChild( ShaderBinarizerContext* pContext ) override;
    virtual void CalculateSize() override;
    virtual void Convert( ShaderBinarizerContext* pContext ) override;
    virtual void Link( ShaderBinarizerContext* pContext ) override;

    const nn::util::MemorySplitter::MemoryBlock* GetShaderBinaryBlock() const override
    {
        return &m_DataBlock.block;
    }

    const nn::gfx::NvnShaderCode* GetShaderCode() const
    {
        return m_pTarget;
    }

    const ResShaderProgramBinarizer* GetParent() const
    {
        return m_pParent;
    }

    ShaderStage GetStage() const
    {
        return m_Stage;
    }

    NvnDecomposedControlSectionBinarizer* GetDecomposedControlSectionBinarizer()
    {
        return m_pDecomposedControlSectionBinarizer.get();
    }

    const NvnDecomposedControlSectionBinarizer* GetDecomposedControlSectionBinarizer() const
    {
        return m_pDecomposedControlSectionBinarizer.get();
    }

    void MergeData( int targetVariation )
    {
        m_DataBlock.merge = targetVariation;
    }

    void MergeControl( int targetVariation )
    {
        m_ControlBlock.merge = targetVariation;
    }

    static int GetShaderCodeAlignment();

private:
    const BinarizationTarget* m_pTarget;
    const ResShaderProgramBinarizer* m_pParent;
    ShaderStage m_Stage;

    MergeBlock m_DataBlock;
    MergeBlock m_ControlBlock;
    nn::util::MemorySplitter::MemoryBlock m_DebugHashBlock;

    bool m_IsFirstBinary;

    Custom< std::unique_ptr< NvnDecomposedControlSectionBinarizer > >::Type
        m_pDecomposedControlSectionBinarizer;

    const NvnShaderCodeBinarizer* m_pFirstBinarizer;
};

class NvnDecomposedControlSectionBinarizer
    : public BinarizerBase
{
public:
    typedef nn::gfx::NvnDecomposedControlSection ResTarget;

    enum class ControlSubsectionType
    {
        MetaData,
        AssemblyData,
        SpecializationData,
        PragmaData,
        AssemblyLocalsData,
        Uniform64InfoData,

        End
    };

    struct ControlSubsection
    {
        MergeBlock mergeBlock;
        const void* pData;
    };

    void Initialize( const void* pTarget, const NvnShaderCodeBinarizer* pParent );

    virtual void RegisterChild( ShaderBinarizerContext* pContext ) override;
    virtual void CalculateSize() override;
    virtual void Convert( ShaderBinarizerContext* pContext ) override;
    virtual void Link( ShaderBinarizerContext* pContext ) override;

    const ControlSubsection& GetControlSubsection( ControlSubsectionType type ) const
    {
        return m_ControlSubsections[ static_cast<int>( type ) ];
    }

    void MergeControlSubsection( ControlSubsectionType subsection, int targetVariation )
    {
        m_ControlSubsections[ static_cast< int >( subsection ) ].mergeBlock.merge = targetVariation;
    }

private:
    const void* m_pTarget;
    const NvnShaderCodeBinarizer* m_pParent;

    ControlSubsection m_ControlSubsections[ static_cast< int >( ControlSubsectionType::End ) ];
};

}
}
