﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <algorithm>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_VariationGrouper.h>

namespace nn {
namespace gfxTool {

void VariationGrouper::Initialize( const nngfxToolShaderCompilerCompileArg* pArg,
    LessVariation less, ShaderStage stage )
{
    if( GetStageSource( pArg, stage ).pValue == nullptr )
    {
        NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
    }

    auto pStageVariationDefinition = pArg->pVariationDefinitionArg ?
        GetStageVariationDefinition( pArg->pVariationDefinitionArg, stage ) : nullptr;

    m_VariationToGroup.resize( pArg->variationCount );
    m_GroupToHeadVariation.reserve( pArg->variationCount );
    Custom< std::vector< decltype( pArg->pVariationValueArray ) > >::Type pSortVariation;
    pSortVariation.reserve( pArg->variationCount );
    auto variationCount = NumericCast< int >( pArg->variationCount );
    for( int idxVariation = 0; idxVariation < variationCount; ++idxVariation )
    {
        pSortVariation.push_back( pArg->pVariationValueArray + idxVariation );
    }
    std::sort( pSortVariation.begin(), pSortVariation.end(), [&]( decltype( pArg->pVariationValueArray ) lhs,
        decltype( pArg->pVariationValueArray ) rhs ) { return less( pStageVariationDefinition,
        GetStageVariationValue( lhs, stage ), GetStageVariationValue( rhs, stage ) ); } );
    auto it = pSortVariation.cbegin();
    auto itBefore = it;
    m_VariationToGroup[ (*it) - pArg->pVariationValueArray ] = 0;
    m_GroupToHeadVariation.push_back( StaticCastAuto( (*it) - pArg->pVariationValueArray ) );
    ++it;
    for( auto itEnd = pSortVariation.cend(); it != itEnd; ++it, ++itBefore )
    {
        auto idxVariation = (*it) - pArg->pVariationValueArray;
        if( less( pStageVariationDefinition, GetStageVariationValue(
            *itBefore, stage ), GetStageVariationValue( *it, stage ) ) )
        {
            m_GroupToHeadVariation.push_back( StaticCastAuto( idxVariation ) );
        }
        m_VariationToGroup[ idxVariation ] = NumericCast< int >( m_GroupToHeadVariation.size() ) - 1;
    }
}

void VariationGrouper::Intersect( const Custom< std::vector< const VariationGrouper* > >::Type& grouperArray )
{
    if( grouperArray.size() < 2 )
    {
        if( grouperArray.size() < 1 )
        {
            NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
        }
        *this = *grouperArray[ 0 ];
        return;
    }
    auto& baseGrouper = *grouperArray.begin();
    for( auto& grouper : grouperArray )
    {
        if( baseGrouper->GetVariationToGroupTable().size() != grouper->GetVariationToGroupTable().size() )
        {
            NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
        }
    }

    auto variationCount = NumericCast< int >( baseGrouper->GetVariationToGroupTable().size() );
    m_VariationToGroup.resize( variationCount );
    m_GroupToHeadVariation.reserve( baseGrouper->GetGroupCount() );

    Custom< std::vector< decltype( &baseGrouper->GetVariationToGroupTable().at( 0 ) ) > >::Type baseSort;
    baseSort.reserve( baseGrouper->GetVariationToGroupTable().size() );
    for( int idxVariation = 0; idxVariation < variationCount; ++idxVariation )
    {
        baseSort.push_back( &baseGrouper->GetVariationToGroupTable().at( idxVariation ) );
    }
    std::sort( baseSort.begin(), baseSort.end(), []( decltype( *baseSort.begin() ) lhsIt,
        decltype( *baseSort.begin() ) rhsIt ) { return *lhsIt < *rhsIt; } );
    for( int idx = 0, idxLower = 0, count = NumericCastAuto( baseSort.size() ); idx < count; ++idx )
    {
        auto idxVariation = baseSort[ idx ] - &baseGrouper->GetVariationToGroupTable()[ 0 ];
        if( idx > 0 && *baseSort[ idx - 1 ] < *baseSort[ idx ] )
        {
            idxLower = idx;
        }
        auto group = NumericCast< int >( m_GroupToHeadVariation.size() );
        for( int idxTarget = idxLower; idxTarget < idx; ++idxTarget )
        {
            auto targetVariation = baseSort[ idxTarget ] - &baseGrouper->GetVariationToGroupTable()[ 0 ];
            bool isSame = true;
            for( auto& grouper : grouperArray )
            {
                if( grouper->GetVariationToGroupTable()[ idxVariation ] !=
                    grouper->GetVariationToGroupTable()[ targetVariation ] )
                {
                    isSame = false;
                    break;
                }
            }
            if( isSame )
            {
                group = m_VariationToGroup[ targetVariation ];
                break;
            }
        }
        m_VariationToGroup[ idxVariation ] = group;
        if( group == NumericCast< decltype( group ) >( m_GroupToHeadVariation.size() ) )
        {
            m_GroupToHeadVariation.push_back( StaticCastAuto( idxVariation ) );
        }
    }
}

bool VariationGrouper::LessConstantValue( const nngfxToolShaderCompilerVariationDefinition* definition,
    const nngfxToolShaderCompilerVariationValue* lhs, const nngfxToolShaderCompilerVariationValue* rhs )
{
    if( definition && lhs && rhs )
    {
        for( int idxDefinition = 0, definitionCount = NumericCastAuto(
            definition->variationConstantDefinitionCount ); idxDefinition < definitionCount; ++idxDefinition )
        {
            auto& idxConstant = definition->pVariationConstantDefinitionArray[
                idxDefinition ].indexInVariationConstantValueArray;
            if( NumericCast< uint32_t >( idxConstant ) >= lhs->variationConstantValueCount ||
                NumericCast< uint32_t >( idxConstant ) >= rhs->variationConstantValueCount )
            {
                auto& name = definition->pVariationConstantDefinitionArray[ idxDefinition ].name;
                Custom< std::string >::Type nameStr( name.pValue, name.length );
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidVariation,
                    "There is no etnry for %s in variation constant value array.", nameStr.c_str() );
            }
            auto& lhsConstant = lhs->pVariationConstantValueArray[ idxConstant ];
            auto& rhsConstant = rhs->pVariationConstantValueArray[ idxConstant ];
            if( lhsConstant.valueSizeIn32Bit != rhsConstant.valueSizeIn32Bit )
            {
                return lhsConstant.valueSizeIn32Bit < rhsConstant.valueSizeIn32Bit;
            }
            auto cmp = memcmp( lhsConstant.pValue, rhsConstant.pValue, lhsConstant.valueSizeIn32Bit * 4 );
            if( cmp != 0 )
            {
                return cmp < 0;
            }
        }
    }

    return false;
}

bool VariationGrouper::LessPreprocessorDefinition( const nngfxToolShaderCompilerVariationDefinition* definition,
    const nngfxToolShaderCompilerVariationValue* lhs, const nngfxToolShaderCompilerVariationValue* rhs )
{
    if( definition && lhs && rhs )
    {
        for( int idxDefinition = 0, definitionCount = NumericCastAuto(
            definition->preprocessorDefinitionDefinitionCount ); idxDefinition < definitionCount; ++idxDefinition )
        {
            auto& idxMacro = definition->pPreprocessorDefinitionDefinitionArray[
                idxDefinition ].indexInPreprocessorDefinitionValueArray;
            if( NumericCast< uint32_t >( idxMacro ) >= lhs->preprocessorDefinitionValueCount ||
                NumericCast< uint32_t >( idxMacro ) >= rhs->preprocessorDefinitionValueCount )
            {
                auto& name = definition->pPreprocessorDefinitionDefinitionArray[ idxDefinition ].name;
                Custom< std::string >::Type nameStr( name.pValue, name.length );
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidVariation,
                    "There is no entry for %s in preprocessor definition value array.", nameStr.c_str() );
            }
            auto& lhsMacro = lhs->pPreprocessorDefinitionValueArray[ idxMacro ];
            auto& rhsMacro = rhs->pPreprocessorDefinitionValueArray[ idxMacro ];
            if( lhsMacro.length != rhsMacro.length )
            {
                return lhsMacro.length < rhsMacro.length;
            }
            auto cmp = memcmp( lhsMacro.pValue, rhsMacro.pValue, lhsMacro.length );
            if( cmp != 0 )
            {
                return cmp < 0;
            }
        }
    }

    return false;
}

}
}
