﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <sstream>

#include <nn/util/util_BytePtr.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_GroupSource.h>

namespace {

inline const char* GetScalarName( nngfxToolShaderCompilerVariableType type )
{
    if( type >= nngfxToolShaderCompilerVariableType_End )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "VariableType" );
    }

    static const char* scalarNameTable[] =
    {
        "bool",
        "int",
        "uint",
        "float",
        "double"
    };

    return scalarNameTable[ type ];
}

inline const char* GetVectorName( nngfxToolShaderCompilerVariableType type, int vectorComponents )
{
    if( type >= nngfxToolShaderCompilerVariableType_End )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "VariableType" );
    }

    if( vectorComponents < 2 || vectorComponents > 4 )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "VectorComponents" );
    }

    static const char* vectorNameTable[][ 3 ] =
    {
        { "bool2", "bool3", "bool4" },
        { "int2", "int3", "int4" },
        { "uint2", "uint3", "uint4" },
        { "float2", "float3", "float4" },
        { "double2", "double3", "double4" }
    };

    return vectorNameTable[ type ][ vectorComponents - 2 ];
}

inline const char* GetMatrixName( nngfxToolShaderCompilerVariableType type, int column, int row )
{
    if( type >= nngfxToolShaderCompilerVariableType_End )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "VariableType" );;
    }
    if( column < 2 || column > 4 )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "MatrixColumn(%d)", column );;
    }
    if( row < 2 || row > 4 )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "MatrixRow(%d)", row );
    }

    static const char* matrixNameTable[][ 3 ][ 3 ] =
    {
        { { "bool2x2", "bool2x3", "bool2x4" }, { "bool3x2", "bool3x3", "bool3x4" }, { "bool4x2", "bool4x3", "bool4x4" } },
        { { "int2x2", "int2x3", "int2x4" }, { "int3x2", "int3x3", "int3x4" }, { "int4x2", "int4x3", "int4x4" } },
        { { "uint2x2", "uint2x3", "uint2x4" }, { "uint3x2", "uint3x3", "uint3x4" }, { "uint4x2", "uint4x3", "uint4x4" } },
        { { "float2x2", "float2x3", "float2x4" }, { "float3x2", "float3x3", "float3x4" }, { "float4x2", "float4x3", "float4x4" } },
        { { "double2x2", "double2x3", "double2x4" }, { "double3x2", "double3x3", "double3x4" }, { "double4x2", "double4x3", "double4x4" } }
    };

    int idxType;
    switch( type )
    {
    case nngfxToolShaderCompilerVariableType_Bool32:
        {
            idxType = 0;
        }
        break;
    case nngfxToolShaderCompilerVariableType_Int32:
        {
            idxType = 1;
        }
        break;
    case nngfxToolShaderCompilerVariableType_Uint32:
        {
            idxType = 2;
        }
        break;
    case nngfxToolShaderCompilerVariableType_Float32:
        {
            idxType = 3;
        }
        break;
    case nngfxToolShaderCompilerVariableType_Float64:
        {
            idxType = 4;
        }
        break;
    default:
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument,
            "%s matrix is not supported.", GetScalarName( type ) );
    }
    int idxColumn = column - 2;
    int idxRow = row - 2;

    return matrixNameTable[ idxType ][ idxColumn ][ idxRow ];
}

inline const char* GetTypeName( const nngfxToolShaderCompilerVariationConstantDefinition& definition )
{
    auto type = static_cast< nngfxToolShaderCompilerVariableType >( definition.type );
    if( definition.matrix.row > 1 )
    {
        return GetMatrixName( type, definition.matrix.column, definition.matrix.row );
    }
    else if( definition.vectorComponents > 1 )
    {
        return GetVectorName( type, definition.vectorComponents );
    }
    return GetScalarName( type );
}

inline const void* WriteScalar( nn::gfxTool::Custom< std::stringstream >::Type& ss,
    nngfxToolShaderCompilerVariableType type, const void* pValue )
{
    ptrdiff_t offset = 0;
    switch( type )
    {
    case nngfxToolShaderCompilerVariableType_Bool32:
        {
            ss << ( *static_cast< const int32_t* >( pValue ) ? "true" : "false" );
            offset = sizeof( int32_t );
        }
        break;
    case nngfxToolShaderCompilerVariableType_Int32:
        {
            ss << *static_cast< const int32_t* >( pValue );
            offset = sizeof( int32_t );
        }
        break;
    case nngfxToolShaderCompilerVariableType_Uint32:
        {
            ss << *static_cast< const uint32_t* >( pValue );
            offset = sizeof( uint32_t );
        }
        break;
    case nngfxToolShaderCompilerVariableType_Float32:
        {
            ss << *static_cast< const float* >( pValue );
            offset = sizeof( float );
        }
        break;
    case nngfxToolShaderCompilerVariableType_Float64:
        {
            ss << *static_cast< const double* >( pValue );
            offset = sizeof( double );
        }
        break;
    default:
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument,
            "Invalid variable type (%d).", nn::gfxTool::NumericCast< int >( type ) );
    }
    return nn::util::ConstBytePtr( pValue, offset ).Get();
}

inline const void* WriteVector( nn::gfxTool::Custom< std::stringstream >::Type& ss,
    nngfxToolShaderCompilerVariableType type, int vectorComponent, const void* pValue )
{
    ss.write( "{", 1 );
    for( int idxComponent = 0; idxComponent < vectorComponent; ++idxComponent )
    {
        if( idxComponent != 0 )
        {
            ss.write( ",", 1 );
        }
        pValue = WriteScalar( ss, type, pValue );
    }
    ss.write( "}", 1 );
    return pValue;
}

inline const void* WriteMatrix( nn::gfxTool::Custom< std::stringstream >::Type& ss,
    nngfxToolShaderCompilerVariableType type, int column, int row, const void* pValue )
{
    ss.write( "{", 1 );
    for( int idxRow = 0; idxRow < row; ++idxRow )
    {
        if( idxRow != 0 )
        {
            ss.write( ",", 1 );
        }
        pValue = WriteVector( ss, type, column, pValue );
    }
    ss.write( "}", 1 );
    return pValue;
}

inline const void* WriteValue( nn::gfxTool::Custom< std::stringstream >::Type& ss,
    const nngfxToolShaderCompilerVariationConstantDefinition& definition, const void* pValue )
{
    auto type = static_cast< nngfxToolShaderCompilerVariableType >( definition.type );

    for( int idxArray = 0, arrayCount = definition.arrayLength == 0 ? 1 :
        nn::gfxTool::NumericCast< int >( definition.arrayLength ); idxArray < arrayCount; ++idxArray )
    {
        if( idxArray != 0 )
        {
            ss.write( ",", 1 );
        }
        if( definition.matrix.row > 1 )
        {
            pValue = WriteMatrix( ss, type, definition.matrix.column, definition.matrix.row, pValue );
        }
        else if( definition.vectorComponents > 1 )
        {
            pValue = WriteVector( ss, type, definition.vectorComponents, pValue );
        }
        else
        {
            pValue = WriteScalar( ss, type, pValue );
        }
    }

    return pValue;
}

}

namespace nn {
namespace gfxTool {

template<>
void VariationConstantSource::Initialize< nngfxToolShaderCompilerShaderSourceFormat_Hlsl >(
    const nngfxToolShaderCompilerCompileArg* pArg,
    const Custom< std::vector< int > >::Type* pGroupToHeadVariationTable, ShaderStage stage )
{
    if( pGroupToHeadVariationTable == nullptr )
    {
        NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
    }

    m_Sources.resize( pGroupToHeadVariationTable->size() );

    if( pArg->pVariationDefinitionArg == nullptr )
    {
        return;
    }

    auto pVariationDefinition = GetStageVariationDefinition( pArg->pVariationDefinitionArg, stage );

    for( int idxGroup = 0, groupCount = NumericCastAuto(
        pGroupToHeadVariationTable->size() ); idxGroup < groupCount; ++idxGroup )
    {
        auto& dst = m_Sources[ idxGroup ];
        auto pStageVariationValue = GetStageVariationValue(
            pArg->pVariationValueArray + ( *pGroupToHeadVariationTable )[ idxGroup ], stage );
        if( pStageVariationValue == nullptr )
        {
            continue;
        }
        auto pValues = pStageVariationValue->pVariationConstantValueArray;
        if( pValues == nullptr )
        {
            continue;
        }
        nn::gfxTool::Custom< std::stringstream >::Type ss;
        for( int idxDefinition = 0, definitionCount = NumericCastAuto(
            pVariationDefinition->variationConstantDefinitionCount );
            idxDefinition < definitionCount; ++idxDefinition )
        {
            auto& constantDefinition = pVariationDefinition->pVariationConstantDefinitionArray[ idxDefinition ];
            auto& value = pValues[ constantDefinition.indexInVariationConstantValueArray ];
            auto typeName = GetTypeName( constantDefinition );
            ss.write( "static const ", 13 );
            ss << typeName;
            ss.write( " ", 1 );
            ss.write( constantDefinition.name.pValue, constantDefinition.name.length );
            if( constantDefinition.arrayLength > 1 )
            {
                ss.write( "[", 1 );
                ss << constantDefinition.arrayLength;
                ss.write( "]", 1 );
            }
            ss.write( "=", 1 );
            auto pValueEnd = WriteValue( ss, constantDefinition, value.pValue );
            auto read = NumericCast< int >( nn::util::ConstBytePtr( value.pValue ).Distance( pValueEnd ) );
            if( read > NumericCast< decltype( read ) >( value.valueSizeIn32Bit ) * 4 )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_NotEnoughSize,
                    "Variation Constant Definition(%d):%s, ValueSize:%d\n",
                    idxDefinition, typeName, value.valueSizeIn32Bit );
            }
            ss.write( ";\n", 2 );
        }
        dst = ss.str();
    }
}

}
}
