﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <filesystem>
#include <iomanip>
#include <future>

#if !defined( WIN32_LEAN_AND_MEAN )
    #define WIN32_LEAN_AND_MEAN
#endif
#if !defined( NOMINMAX )
    #define NOMINMAX
#endif
#include <nn/nn_Windows.h>

#include <nn/util/util_BytePtr.h>
#include <nn/util/util_BitArray.h>
#include <nn/crypto.h>

#include <nn/gfx/gfx_Enum.h>
#include <nn/gfx/gfx_ResShaderData-api.nvn.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_Compiler-nvn-binary.h>
#include <gfxTool_ShaderCompilerContext.h>
#include <gfxTool_CompileOptionManager.h>
#include <gfxTool_VariationManager.h>
#include <gfxTool_VariationGrouper.h>
#include <gfxTool_GroupSource.h>
#include <gfxTool_ShaderSourceManager.h>
#include <gfxTool_CompileOption-glsl.h>
#include <gfxTool_CompileOutput.h>
#include <gfxTool_OptionOutput.h>
#include <gfxTool_OptionOutputNvn.h>
#include <gfxTool_CompileOption-nvn.h>
#include <gfxTool_BoostPreprocessor.h>

namespace {

template< typename THeaderType >
struct SectionHeaderType;
template<>
struct SectionHeaderType< GLSLCgpuCodeHeader >
{
    static const GLSLCsectionTypeEnum GlslcSectionType = GLSLC_SECTION_TYPE_GPU_CODE;
};
template<>
struct SectionHeaderType< GLSLCasmDumpHeader >
{
    static const GLSLCsectionTypeEnum GlslcSectionType = GLSLC_SECTION_TYPE_ASM_DUMP;
};
template<>
struct SectionHeaderType< GLSLCperfStatsHeader >
{
    static const GLSLCsectionTypeEnum GlslcSectionType = GLSLC_SECTION_TYPE_PERF_STATS;
};
template<>
struct SectionHeaderType< GLSLCprogramReflectionHeader >
{
    static const GLSLCsectionTypeEnum GlslcSectionType = GLSLC_SECTION_TYPE_REFLECTION;
};
template<>
struct SectionHeaderType< GLSLCdebugInfoHeader >
{
    static const GLSLCsectionTypeEnum GlslcSectionType = GLSLC_SECTION_TYPE_DEBUG_INFO;
};

static const nngfxToolShaderCompilerReflectionVariableType s_NvnTypeToGfxTypeTable[] =
{
    nngfxToolShaderCompilerReflectionVariableType_Bool32,
    nngfxToolShaderCompilerReflectionVariableType_Bool32x2,
    nngfxToolShaderCompilerReflectionVariableType_Bool32x3,
    nngfxToolShaderCompilerReflectionVariableType_Bool32x4,
    nngfxToolShaderCompilerReflectionVariableType_Int32,
    nngfxToolShaderCompilerReflectionVariableType_Int32x2,
    nngfxToolShaderCompilerReflectionVariableType_Int32x3,
    nngfxToolShaderCompilerReflectionVariableType_Int32x4,
    nngfxToolShaderCompilerReflectionVariableType_Int8,
    nngfxToolShaderCompilerReflectionVariableType_Int8x2,
    nngfxToolShaderCompilerReflectionVariableType_Int8x3,
    nngfxToolShaderCompilerReflectionVariableType_Int8x4,
    nngfxToolShaderCompilerReflectionVariableType_Int16,
    nngfxToolShaderCompilerReflectionVariableType_Int16x2,
    nngfxToolShaderCompilerReflectionVariableType_Int16x3,
    nngfxToolShaderCompilerReflectionVariableType_Int16x4,
    nngfxToolShaderCompilerReflectionVariableType_Int64,
    nngfxToolShaderCompilerReflectionVariableType_Int64x2,
    nngfxToolShaderCompilerReflectionVariableType_Int64x3,
    nngfxToolShaderCompilerReflectionVariableType_Int64x4,
    nngfxToolShaderCompilerReflectionVariableType_Uint32,
    nngfxToolShaderCompilerReflectionVariableType_Uint32x2,
    nngfxToolShaderCompilerReflectionVariableType_Uint32x3,
    nngfxToolShaderCompilerReflectionVariableType_Uint32x4,
    nngfxToolShaderCompilerReflectionVariableType_Uint8,
    nngfxToolShaderCompilerReflectionVariableType_Uint8x2,
    nngfxToolShaderCompilerReflectionVariableType_Uint8x3,
    nngfxToolShaderCompilerReflectionVariableType_Uint8x4,
    nngfxToolShaderCompilerReflectionVariableType_Uint16,
    nngfxToolShaderCompilerReflectionVariableType_Uint16x2,
    nngfxToolShaderCompilerReflectionVariableType_Uint16x3,
    nngfxToolShaderCompilerReflectionVariableType_Uint16x4,
    nngfxToolShaderCompilerReflectionVariableType_Uint64,
    nngfxToolShaderCompilerReflectionVariableType_Uint64x2,
    nngfxToolShaderCompilerReflectionVariableType_Uint64x3,
    nngfxToolShaderCompilerReflectionVariableType_Uint64x4,
    nngfxToolShaderCompilerReflectionVariableType_Float32,
    nngfxToolShaderCompilerReflectionVariableType_Float32x2,
    nngfxToolShaderCompilerReflectionVariableType_Float32x3,
    nngfxToolShaderCompilerReflectionVariableType_Float32x4,
    nngfxToolShaderCompilerReflectionVariableType_Float16,
    nngfxToolShaderCompilerReflectionVariableType_Float16x2,
    nngfxToolShaderCompilerReflectionVariableType_Float16x3,
    nngfxToolShaderCompilerReflectionVariableType_Float16x4,
    nngfxToolShaderCompilerReflectionVariableType_Float64,
    nngfxToolShaderCompilerReflectionVariableType_Float64x2,
    nngfxToolShaderCompilerReflectionVariableType_Float64x3,
    nngfxToolShaderCompilerReflectionVariableType_Float64x4,

    nngfxToolShaderCompilerReflectionVariableType_Float32x2x2,
    nngfxToolShaderCompilerReflectionVariableType_Float32x3x3,
    nngfxToolShaderCompilerReflectionVariableType_Float32x4x4,
    nngfxToolShaderCompilerReflectionVariableType_Float32x2x3,
    nngfxToolShaderCompilerReflectionVariableType_Float32x2x4,
    nngfxToolShaderCompilerReflectionVariableType_Float32x3x2,
    nngfxToolShaderCompilerReflectionVariableType_Float32x3x4,
    nngfxToolShaderCompilerReflectionVariableType_Float32x4x2,
    nngfxToolShaderCompilerReflectionVariableType_Float32x4x3,
    nngfxToolShaderCompilerReflectionVariableType_Float32x2x2,
    nngfxToolShaderCompilerReflectionVariableType_Float32x3x3,
    nngfxToolShaderCompilerReflectionVariableType_Float32x4x4,
    nngfxToolShaderCompilerReflectionVariableType_Float32x2x3,
    nngfxToolShaderCompilerReflectionVariableType_Float32x2x4,
    nngfxToolShaderCompilerReflectionVariableType_Float32x3x2,
    nngfxToolShaderCompilerReflectionVariableType_Float32x3x4,
    nngfxToolShaderCompilerReflectionVariableType_Float32x4x2,
    nngfxToolShaderCompilerReflectionVariableType_Float32x4x3,

    nngfxToolShaderCompilerReflectionVariableType_Sampler1D,
    nngfxToolShaderCompilerReflectionVariableType_Sampler2D,
    nngfxToolShaderCompilerReflectionVariableType_Sampler3D,
    nngfxToolShaderCompilerReflectionVariableType_SamplerCube,
    nngfxToolShaderCompilerReflectionVariableType_Sampler1DShadow,
    nngfxToolShaderCompilerReflectionVariableType_Sampler2DShadow,
    nngfxToolShaderCompilerReflectionVariableType_Sampler1DArray,
    nngfxToolShaderCompilerReflectionVariableType_Sampler2DArray,
    nngfxToolShaderCompilerReflectionVariableType_Sampler1DArrayShadow,
    nngfxToolShaderCompilerReflectionVariableType_Sampler2DArrayShadow,
    nngfxToolShaderCompilerReflectionVariableType_Sampler2DMultisample,
    nngfxToolShaderCompilerReflectionVariableType_Sampler2DMultisampleArray,
    nngfxToolShaderCompilerReflectionVariableType_SamplerCubeShadow,
    nngfxToolShaderCompilerReflectionVariableType_SamplerBuffer,
    nngfxToolShaderCompilerReflectionVariableType_Sampler2DRect,
    nngfxToolShaderCompilerReflectionVariableType_Sampler2DRectShadow,

    nngfxToolShaderCompilerReflectionVariableType_IntSampler1D,
    nngfxToolShaderCompilerReflectionVariableType_IntSampler2D,
    nngfxToolShaderCompilerReflectionVariableType_IntSampler3D,
    nngfxToolShaderCompilerReflectionVariableType_IntSamplerCube,
    nngfxToolShaderCompilerReflectionVariableType_IntSampler1DArray,
    nngfxToolShaderCompilerReflectionVariableType_IntSampler2DArray,
    nngfxToolShaderCompilerReflectionVariableType_IntSampler2DMultisample,
    nngfxToolShaderCompilerReflectionVariableType_IntSampler2DMultisampleArray,
    nngfxToolShaderCompilerReflectionVariableType_IntSamplerBuffer,
    nngfxToolShaderCompilerReflectionVariableType_IntSampler2DRect,

    nngfxToolShaderCompilerReflectionVariableType_UintSampler1D,
    nngfxToolShaderCompilerReflectionVariableType_UintSampler2D,
    nngfxToolShaderCompilerReflectionVariableType_UintSampler3D,
    nngfxToolShaderCompilerReflectionVariableType_UintSamplerCube,
    nngfxToolShaderCompilerReflectionVariableType_UintSampler1DArray,
    nngfxToolShaderCompilerReflectionVariableType_UintSampler2DArray,
    nngfxToolShaderCompilerReflectionVariableType_UintSampler2DMultisample,
    nngfxToolShaderCompilerReflectionVariableType_UintSampler2DMultisampleArray,
    nngfxToolShaderCompilerReflectionVariableType_UintSamplerBuffer,
    nngfxToolShaderCompilerReflectionVariableType_UintSampler2DRect,

    nngfxToolShaderCompilerReflectionVariableType_Image1D,
    nngfxToolShaderCompilerReflectionVariableType_Image2D,
    nngfxToolShaderCompilerReflectionVariableType_Image3D,
    nngfxToolShaderCompilerReflectionVariableType_Image2DRect,
    nngfxToolShaderCompilerReflectionVariableType_ImageCube,
    nngfxToolShaderCompilerReflectionVariableType_ImageBuffer,
    nngfxToolShaderCompilerReflectionVariableType_Image1DArray,
    nngfxToolShaderCompilerReflectionVariableType_Image2DArray,
    nngfxToolShaderCompilerReflectionVariableType_ImageCubeMapArray,
    nngfxToolShaderCompilerReflectionVariableType_Image2DMultisample,
    nngfxToolShaderCompilerReflectionVariableType_Image2DMultisampleArray,
    nngfxToolShaderCompilerReflectionVariableType_IntImage1D,
    nngfxToolShaderCompilerReflectionVariableType_IntImage2D,
    nngfxToolShaderCompilerReflectionVariableType_IntImage3D,
    nngfxToolShaderCompilerReflectionVariableType_IntImage2DRect,
    nngfxToolShaderCompilerReflectionVariableType_IntImageCube,
    nngfxToolShaderCompilerReflectionVariableType_IntImageBuffer,
    nngfxToolShaderCompilerReflectionVariableType_IntImage1DArray,
    nngfxToolShaderCompilerReflectionVariableType_IntImage2DArray,
    nngfxToolShaderCompilerReflectionVariableType_IntImageCubeMapArray,
    nngfxToolShaderCompilerReflectionVariableType_IntImage2DMultisample,
    nngfxToolShaderCompilerReflectionVariableType_IntImage2DMultisampleArray,
    nngfxToolShaderCompilerReflectionVariableType_UintImage1D,
    nngfxToolShaderCompilerReflectionVariableType_UintImage2D,
    nngfxToolShaderCompilerReflectionVariableType_UintImage3D,
    nngfxToolShaderCompilerReflectionVariableType_UintImage2DRect,
    nngfxToolShaderCompilerReflectionVariableType_UintImageCube,
    nngfxToolShaderCompilerReflectionVariableType_UintImageBuffer,
    nngfxToolShaderCompilerReflectionVariableType_UintImage1DArray,
    nngfxToolShaderCompilerReflectionVariableType_UintImage2DArray,
    nngfxToolShaderCompilerReflectionVariableType_UintImageCubeMapArray,
    nngfxToolShaderCompilerReflectionVariableType_UintImage2DMultisample,
    nngfxToolShaderCompilerReflectionVariableType_UintImage2DMultisampleArray,

    nngfxToolShaderCompilerReflectionVariableType_SamplerCubeMapArray,
    nngfxToolShaderCompilerReflectionVariableType_IntSamplerCubeMapArray,
    nngfxToolShaderCompilerReflectionVariableType_UintSamplerCubeMapArray,
    nngfxToolShaderCompilerReflectionVariableType_SamplerCubeMapArrayShadow,

    nngfxToolShaderCompilerReflectionVariableType_Sampler,

    nngfxToolShaderCompilerReflectionVariableType_Texture1D,
    nngfxToolShaderCompilerReflectionVariableType_Texture2D,
    nngfxToolShaderCompilerReflectionVariableType_Texture3D,
    nngfxToolShaderCompilerReflectionVariableType_TextureCube,
    nngfxToolShaderCompilerReflectionVariableType_Texture1DShadow,
    nngfxToolShaderCompilerReflectionVariableType_Texture2DShadow,
    nngfxToolShaderCompilerReflectionVariableType_Texture1DArray,
    nngfxToolShaderCompilerReflectionVariableType_Texture2DArray,
    nngfxToolShaderCompilerReflectionVariableType_Texture1DArrayShadow,
    nngfxToolShaderCompilerReflectionVariableType_Texture2DArrayShadow,
    nngfxToolShaderCompilerReflectionVariableType_Texture2DMultisample,
    nngfxToolShaderCompilerReflectionVariableType_Texture2DMultisampleArray,
    nngfxToolShaderCompilerReflectionVariableType_TextureCubeShadow,
    nngfxToolShaderCompilerReflectionVariableType_TextureBuffer,
    nngfxToolShaderCompilerReflectionVariableType_Texture2DRect,
    nngfxToolShaderCompilerReflectionVariableType_Texture2DRectShadow,
    nngfxToolShaderCompilerReflectionVariableType_TextureCubeMapArray,
    nngfxToolShaderCompilerReflectionVariableType_TextureCubeMapArrayShadow,
    nngfxToolShaderCompilerReflectionVariableType_IntTexture1D,
    nngfxToolShaderCompilerReflectionVariableType_IntTexture2D,
    nngfxToolShaderCompilerReflectionVariableType_IntTexture3D,
    nngfxToolShaderCompilerReflectionVariableType_IntTextureCube,
    nngfxToolShaderCompilerReflectionVariableType_IntTexture1DArray,
    nngfxToolShaderCompilerReflectionVariableType_IntTexture2DArray,
    nngfxToolShaderCompilerReflectionVariableType_IntTexture2DMultisample,
    nngfxToolShaderCompilerReflectionVariableType_IntTexture2DMultisampleArray,
    nngfxToolShaderCompilerReflectionVariableType_IntTextureBuffer,
    nngfxToolShaderCompilerReflectionVariableType_IntTexture2DRect,
    nngfxToolShaderCompilerReflectionVariableType_IntTextureCubeMapArray,
    nngfxToolShaderCompilerReflectionVariableType_UintTexture1D,
    nngfxToolShaderCompilerReflectionVariableType_UintTexture2D,
    nngfxToolShaderCompilerReflectionVariableType_UintTexture3D,
    nngfxToolShaderCompilerReflectionVariableType_UintTextureCube,
    nngfxToolShaderCompilerReflectionVariableType_UintTexture1DArray,
    nngfxToolShaderCompilerReflectionVariableType_UintTexture2DArray,
    nngfxToolShaderCompilerReflectionVariableType_UintTexture2DMultisample,
    nngfxToolShaderCompilerReflectionVariableType_UintTexture2DMultisampleArray,
    nngfxToolShaderCompilerReflectionVariableType_UintTextureBuffer,
    nngfxToolShaderCompilerReflectionVariableType_UintTexture2DRect,
    nngfxToolShaderCompilerReflectionVariableType_UintTextureCubeMapArray
};

class Sha256GeneratorEx
    : public nn::crypto::Sha256Generator
{
public:
    template< typename T >
    void Update( const T& data )
    {
        Update( &data, sizeof( T ) );
    }

    void Update( const void* data, size_t dataSize )
    {
        nn::crypto::Sha256Generator::Update( data, dataSize );
    }
};

struct ShaderCache
{
    int32_t glslcOutputCount;
    int32_t offsetGlslcOutputs[ 1 ];

    GLSLCoutput* GetGlslcOutput( int idx )
    {
        return nn::util::BytePtr( this, offsetGlslcOutputs[ idx ] ).Get< GLSLCoutput >();
    }
};

const char* GetNvnGlslcDllName()
{
#if defined( _M_IX86 )
    return "NvnGlslc32.dll";
#else
    return "NvnGlslc.dll";
#endif
}

nn::gfxTool::Custom< std::string >::Type GetNvnGlslcPath()
{
    nn::gfxTool::Custom< std::string >::Type ret;
    auto path = std::tr2::sys::path( nn::gfxTool::GetModulePath(
        nn::gfxTool::GetShaderCompilerModuleHandle() ) ).parent_path().string();

#if defined( _M_IX86 )
    path += "\\..";
#endif

    // デバッグ実行
    ret = path + "\\..\\..\\..\\..\\..\\..\\..\\..\\..\\Tools\\Graphics\\NvnTools\\" + GetNvnGlslcDllName();
    if( !std::ifstream( ret.c_str() ).fail() )
    {
        return ret;
    }

    ret = path + "\\..\\NvnTools\\" + GetNvnGlslcDllName();
    if( !std::ifstream( ret.c_str() ).fail() )
    {
        return ret;
    }

    NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_DllNotFound, "%s is not found.", ret.c_str() );
}

nn::gfxTool::ShaderStage ToGfxStage( NVNshaderStage stage )
{
    static const nn::gfxTool::ShaderStage s_ShaderStages[] =
    {
        nn::gfxTool::ShaderStage::Vertex,
        nn::gfxTool::ShaderStage::Pixel,
        nn::gfxTool::ShaderStage::Geometry,
        nn::gfxTool::ShaderStage::Hull,
        nn::gfxTool::ShaderStage::Domain,
        nn::gfxTool::ShaderStage::Compute,
    };

    return s_ShaderStages[ static_cast< int >( stage ) ];
}

NVNshaderStage ToNvnStage( nn::gfxTool::ShaderStage stage )
{
    static const NVNshaderStage s_ShaderStages[] =
    {
        NVN_SHADER_STAGE_VERTEX,
        NVN_SHADER_STAGE_TESS_CONTROL,
        NVN_SHADER_STAGE_TESS_EVALUATION,
        NVN_SHADER_STAGE_GEOMETRY,
        NVN_SHADER_STAGE_FRAGMENT,
        NVN_SHADER_STAGE_COMPUTE
    };

    return s_ShaderStages[ static_cast< int >( stage ) ];
}

GLSLCdebugInfoLevelEnum ToGlslcDebugInfoLevel( nngfxToolShaderCompilerDebugInfoLevel debugInfoLevel )
{
    static const GLSLCdebugInfoLevelEnum s_DebugInfoLevels[] =
    {
        GLSLC_DEBUG_LEVEL_NONE,
        GLSLC_DEBUG_LEVEL_G0,
        GLSLC_DEBUG_LEVEL_G1,
        GLSLC_DEBUG_LEVEL_G2
    };

    return s_DebugInfoLevels[ debugInfoLevel ];
}

template< typename THeader >
const THeader* GetSectionHeader( const GLSLCoutput* pGlslcOutput )
{
    for( int idxSection = 0; idxSection < static_cast< int >( pGlslcOutput->numSections ); ++idxSection )
    {
        auto& genericHeader = pGlslcOutput->headers[ idxSection ].genericHeader;
        if( genericHeader.common.type == SectionHeaderType< THeader >::GlslcSectionType )
        {
            return reinterpret_cast< const THeader* >( &genericHeader );
        }
    }
    return nullptr;
}

void CalculateInputHash( void* pOutHash, size_t hashSize, const GLSLCversion& version,
    const GLSLCcompileObject& compileObject, const GLSLCspecializationBatch* pBatch )
{
    if( hashSize < nn::crypto::Sha256Generator::HashSize )
    {
        NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
    }

    Sha256GeneratorEx generator;
    generator.Initialize();

    // バージョン
    generator.Update( version.apiMajor );
    generator.Update( version.apiMinor );
    generator.Update( version.gpuCodeVersionMajor );
    generator.Update( version.gpuCodeVersionMinor );

    // オプション
    generator.Update( compileObject.options );

    // ソース
    generator.Update( compileObject.input.count );
    for( int idxShader = 0; idxShader < compileObject.input.count; ++idxShader )
    {
        generator.Update( compileObject.input.stages[ idxShader ] );
        auto source = compileObject.input.sources[ idxShader ];
        generator.Update( source, strlen( source ) );
        if( compileObject.input.spirvEntryPointNames )
        {
            auto entryPoint = compileObject.input.spirvEntryPointNames[ idxShader ];
            if( entryPoint )
            {
                generator.Update( entryPoint, strlen( entryPoint ) );
            }
        }
        if( compileObject.input.spirvModuleSizes )
        {
            generator.Update( compileObject.input.spirvModuleSizes[ idxShader ] );
        }
        if( compileObject.input.spirvSpecInfo )
        {
            if( auto pSpecInfo = compileObject.input.spirvSpecInfo[ idxShader ] )
            {
                generator.Update( pSpecInfo->numEntries );
                generator.Update( pSpecInfo->constantIDs,
                    sizeof( *pSpecInfo->constantIDs ) * pSpecInfo->numEntries );
                generator.Update( pSpecInfo->data, 4 * pSpecInfo->numEntries );
            }
        }
    }

    // バリエーション
    if( pBatch )
    {
        auto& batch = *pBatch;
        generator.Update( batch.numEntries );
        for( int idxEntry = 0; idxEntry < static_cast<int>( batch.numEntries ); ++idxEntry )
        {
            auto& set = batch.entries[ idxEntry ];
            generator.Update( set.numUniforms );
            for( int idxUniform = 0; idxUniform < static_cast<int>( set.numUniforms ); ++idxUniform )
            {
                auto& uniform = set.uniforms[ idxUniform ];
                generator.Update( uniform.numElements );
                generator.Update( uniform.elementSize );
                generator.Update( uniform.uniformName, strlen( uniform.uniformName ) );
                generator.Update( uniform.values, uniform.numElements * uniform.elementSize );
            }
        }
    }

    generator.GetHash( pOutHash, hashSize );
}

nn::gfxTool::Custom< std::string >::Type HashToString( void* pHash, size_t hashSize )
{
    nn::gfxTool::Custom< std::string >::Type ret = "";
    char buf[ 8 ] = {};
    for( int idx = 0; idx < static_cast< int >( hashSize ); ++idx )
    {
        snprintf( buf, sizeof( buf ), "%02x", static_cast<uint8_t*>( pHash )[ idx ] );
        ret.append( buf );
    }
    return ret;
}

decltype( auto ) ReadShaderCache( const nngfxToolShaderCompilerCompileArg* pArg,
    const GLSLCversion& version, const GLSLCcompileObject& compileObject,
    const GLSLCspecializationBatch* pSpecializationBatch )
{
    nn::gfxTool::Custom< std::vector< const GLSLCoutput* > >::Type ret;
    if( pArg->pReadShaderCacheCallback )
    {
        char hash[ nn::crypto::Sha256Generator::HashSize ];
        CalculateInputHash( hash, sizeof( hash ), version, compileObject, pSpecializationBatch );
        nn::gfxTool::Custom< std::string >::Type filename = HashToString( hash, sizeof( hash ) );
        void* pData = nullptr;
        size_t dataSize = 0;
        if( pArg->pReadShaderCacheCallback( &pData, &dataSize, filename.c_str(),
            pArg->pReadShaderCacheCallbackParam ) )
        {
            auto pCache = static_cast< ShaderCache* >( pData );
            ret.reserve( pCache->glslcOutputCount );
            for( int idx = 0; idx < pCache->glslcOutputCount; ++idx )
            {
                ret.push_back( pCache->GetGlslcOutput( idx ) );
            }
        }
    }

    return ret;
}

void WriteShaderCache( const nngfxToolShaderCompilerCompileArg* pArg,
    const GLSLCversion& version, const GLSLCcompileObject& compileObject,
    const GLSLCspecializationBatch* pSpecializationBatch, const GLSLCresults* const* ppResults )
{
    if( pArg->pWriteShaderCacheCallback )
    {
        int entryCount = pSpecializationBatch ? nn::gfxTool::NumericCastAuto(
            pSpecializationBatch->numEntries ) : 1;
        size_t fileSize = sizeof( ShaderCache ) + ( entryCount - 1 ) * sizeof( int32_t );
        for( int idxEntry = 0; idxEntry < entryCount; ++idxEntry )
        {
            fileSize += ppResults ? ppResults[ idxEntry ]->glslcOutput->size
                : compileObject.lastCompiledResults->glslcOutput->size;
        }
        std::unique_ptr< void, decltype( &free ) > data( malloc( fileSize ), free );
        auto pDst = static_cast< ShaderCache* >( data.get() );
        pDst->glslcOutputCount = entryCount;
        nn::util::BytePtr pDstGlslcOutput( pDst->offsetGlslcOutputs + entryCount );
        for( int idxEntry = 0; idxEntry < entryCount; ++idxEntry )
        {
            pDst->offsetGlslcOutputs[ idxEntry ] = nn::gfxTool::NumericCastAuto(
                nn::util::BytePtr( pDst ).Distance( pDstGlslcOutput.Get() ) );
            auto pGlslcOutput = ppResults ? ppResults[ idxEntry ]->glslcOutput
                : compileObject.lastCompiledResults->glslcOutput;
            memcpy( pDstGlslcOutput.Get(), pGlslcOutput, pGlslcOutput->size );
            pDstGlslcOutput.Advance( pGlslcOutput->size );
        }
        char hash[ nn::crypto::Sha256Generator::HashSize ];
        CalculateInputHash( hash, sizeof( hash ), version, compileObject, pSpecializationBatch );
        nn::gfxTool::Custom< std::string >::Type filename = HashToString( hash, sizeof( hash ) );
        pArg->pWriteShaderCacheCallback( pDst, fileSize,
            filename.c_str(), pArg->pWriteShaderCacheCallbackParam );
    }
}

const char* GetInitializationStatusString( GLSLCinitializationStatus initStatus )
{
    switch( initStatus )
    {
    case GLSLC_INIT_ERROR_UNINITIALIZED:
        return "GLSLC_INIT_ERROR_UNINITIALIZED";
    case GLSLC_INIT_SUCCESS:
        return "GLSLC_INIT_SUCCESS";
    case GLSLC_INIT_ERROR_ALLOC_FAILURE:
        return "GLSLC_INIT_ERROR_ALLOC_FAILURE";
    case GLSLC_INIT_ERROR_NO_ALLOC_CALLBACKS_SET:
        return "GLSLC_INIT_ERROR_NO_ALLOC_CALLBACKS_SET";
    default:
        return "Unknown";
    }
}

template< typename TEnum, int N >
const char* EnumToString( TEnum value, const char* const ( &pTable )[ N ] )
{
    return value < N ? pTable[ value ] : "?";
}

nn::gfxTool::Custom< std::string >::Type GlslcOptionFlagsToString( GLSLCoptionFlags flags )
{
    const char* const GlslcDebugInfoLevelStrings[] =
    {
        "GLSLC_DEBUG_LEVEL_NONE",
        "GLSLC_DEBUG_LEVEL_G0",
        "GLSLC_DEBUG_LEVEL_G1",
        "GLSLC_DEBUG_LEVEL_G2",
    };

    const char* const GlslcLanguageTypeStrings[] =
    {
        "GLSLC_LANGUAGE_GLSL",
        "GLSLC_LANGUAGE_GLES",
        "GLSLC_LANGUAGE_SPIRV"
    };

    const char* const GlslcSpillControlStrings[] =
    {
        "DEFAULT_SPILL",
        "NO_SPILL"
    };

    const char* const GlslcOptLevelStrings[] =
    {
        "GLSLC_OPTLEVEL_DEFAULT",
        "GLSLC_OPTLEVEL_NONE"
    };

    const char* const GlslcUnrollControlStrings[] =
    {
        "GLSLC_LOOP_UNROLL_DEFAULT",
        "GLSLC_LOOP_UNROLL_NONE",
        "GLSLC_LOOP_UNROLL_ALL"
    };

    auto WriteFlag = []( nn::gfxTool::Custom< std::string >::Type* pDst,
        const char* format, const char* flagName, auto value )
    {
        char buf[ 256 ] = {};
        snprintf( buf, sizeof( buf ), format, flagName, value );
        pDst->append( buf );
    };

    nn::gfxTool::Custom< std::string >::Type ret;
    const char* formatD = "%-41s: %d\n";
    const char* formatS = "%-41s: %s\n";
    WriteFlag( &ret, formatD, "glslSeparable", flags.glslSeparable );
    WriteFlag( &ret, formatD, "outputAssembly", flags.outputAssembly );
    WriteFlag( &ret, formatD, "outputGpuBinaries", flags.outputGpuBinaries );
    WriteFlag( &ret, formatD, "outputPerfStats", flags.outputPerfStats );
    WriteFlag( &ret, formatD, "ouptutShaderReflection", flags.outputShaderReflection );
    WriteFlag( &ret, formatS, "language", EnumToString( flags.language, GlslcLanguageTypeStrings ) );
    WriteFlag( &ret, formatS, "outputDebugInfo", EnumToString( flags.outputDebugInfo, GlslcDebugInfoLevelStrings ) );
    WriteFlag( &ret, formatS, "spillControl", EnumToString( flags.spillControl, GlslcSpillControlStrings ) );
    WriteFlag( &ret, formatD, "outputThinGpuBinaries", flags.outputThinGpuBinaries );
    WriteFlag( &ret, formatD, "tessellationAndPassthroughGS", flags.tessellationAndPassthroughGS );
    WriteFlag( &ret, formatD, "prioritizeConsecutiveTextureInstructions", flags.prioritizeConsecutiveTextureInstructions );
    WriteFlag( &ret, formatD, "enableFastMathMask", flags.enableFastMathMask );
    WriteFlag( &ret, formatS, "optLevel", EnumToString( flags.optLevel, GlslcOptLevelStrings ) );
    WriteFlag( &ret, formatS, "unrollControl", EnumToString( flags.unrollControl, GlslcUnrollControlStrings ) );
    WriteFlag( &ret, formatD, "errorOnScratchMemUsage", flags.errorOnScratchMemUsage );
    WriteFlag( &ret, formatD, "enableCBFOptimization", flags.enableCBFOptimization );
    WriteFlag( &ret, formatD, "enableWarpCulling", flags.enableWarpCulling );
    WriteFlag( &ret, formatD, "enableMultithreadCompilation", flags.enableMultithreadCompilation );

    return ret;
}

void SetupGlslcOptions( GLSLCoptions* pOutGlslcOptions,
    const nn::gfxTool::ShaderCompilerContext* pContext,
    const nngfxToolShaderCompilerCompileArg* pArg,
    const nn::gfxTool::NvnGlslcDll& nvnGlslcDll )
{
    if( pOutGlslcOptions == nullptr || pContext == nullptr )
    {
        NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
    }

    auto pCompileOptionManager = pContext->GetCompileOptionManager();
    auto pCommonOption = pCompileOptionManager->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >();
    auto pNvnOption = pCompileOptionManager->GetCompileOption< static_cast<
        nngfxToolShaderCompilerOptionType >( nngfxToolShaderCompilerOptionType_Nvn )>();

    GLSLCoptions& options = *pOutGlslcOptions;
    options = nvnGlslcDll.GlslcGetDefaultOptions();
    options.optionFlags.glslSeparable = pCommonOption->IsSeparationEnabled() ? 1 : 0;
    options.optionFlags.outputAssembly = pCommonOption->IsDumpEnabled() ? 1 : 0;
    options.optionFlags.outputGpuBinaries = 1;
    options.optionFlags.outputPerfStats = pCommonOption->IsShaderStatisticsEnabled() ? 1 : 0;
    options.optionFlags.outputShaderReflection = pCommonOption->IsReflectionEnabled() ? 1 : 0;
    options.optionFlags.language =
        pArg->shaderSourceFormat == nngfxToolShaderCompilerShaderSourceFormat_SpirV ?
        GLSLC_LANGUAGE_SPIRV : GLSLC_LANGUAGE_GLSL;
    options.optionFlags.outputDebugInfo =
        ToGlslcDebugInfoLevel( pCommonOption->GetDebugInfoLevel() );
    options.optionFlags.outputThinGpuBinaries =
        pArg->targetCodeType == nngfxToolShaderCompilerCodeType_Binary_Ir ? 0 : 1;
    if( pNvnOption->IsInitialized() )
    {
        auto& nvnOption = *pNvnOption->GetOption();
        options.optionFlags.spillControl = nn::gfxTool::StaticCastAuto( nvnOption.spillControl );
        options.optionFlags.tessellationAndPassthroughGS =
            nvnOption.tessellationAndPassthroughGS ? 1 : 0;
        options.optionFlags.prioritizeConsecutiveTextureInstructions =
            nvnOption.prioritizeConsecutiveTextureInstructions ? 1 : 0;
        options.optionFlags.enableFastMathMask = nvnOption.enableFastMathMask;
        options.optionFlags.optLevel = nn::gfxTool::StaticCastAuto( nvnOption.optLevel );
        options.optionFlags.unrollControl = nn::gfxTool::StaticCastAuto( nvnOption.unrollControl );
        options.optionFlags.errorOnScratchMemUsage = nvnOption.errorOnScratchMemUsage ? 1 : 0;
        options.optionFlags.enableCBFOptimization = nvnOption.enableCBFOptimization ? 1 : 0;
        options.optionFlags.enableWarpCulling = nvnOption.enableWarpCulling ? 1 : 0;
        options.optionFlags.warnUninitControl = nn::gfxTool::StaticCastAuto( nvnOption.warnUninitControl );
    }
    options.forceIncludeStdHeader = nullptr;
    options.includeInfo.numPaths = 0;
    options.includeInfo.paths = nullptr;
    options.xfbVaryingInfo.numVaryings = 0;
    options.xfbVaryingInfo.varyings = nullptr;

    options.optionFlags.enableMultithreadCompilation =
        options.optionFlags.outputAssembly ? 0 : 1; // NSBG-6407
}

nn::gfxTool::Custom< std::vector< int > >::Type CreateGroupVariations(
    const nn::gfxTool::ShaderCompilerContext* pContext,
    const nngfxToolShaderCompilerCompileArg* pArg, int idxGroup )
{
    nn::gfxTool::Custom< std::vector< int > >::Type variations;
    if( pArg->shaderSourceFormat == nngfxToolShaderCompilerShaderSourceFormat_Glsl )
    {
        auto combinedPreprocessorDefinitionGroup = idxGroup;
        auto pPreprocessorDefinitionGroup =
            pContext->GetVariationManager()->GetPreprocessorDefinitionGroup();
        auto variationCount = nn::gfxTool::NumericCast< int >(
            pPreprocessorDefinitionGroup->GetVariationToGroupTable().size() );
        variations.reserve( variationCount );
        for( int idxVariation = 0; idxVariation < variationCount; ++idxVariation )
        {
            if( pPreprocessorDefinitionGroup->GetVariationToGroupTable().at(
                idxVariation ) == combinedPreprocessorDefinitionGroup )
            {
                variations.push_back( idxVariation );
            }
        }
    }
    else
    {
        variations.push_back( idxGroup );
    }
    if( variations.size() < 1 )
    {
        NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
    }

    return variations;
}

void RetrieveCompileOutput( nn::gfxTool::ProgramOutput* pOutput,
    const GLSLCoutput* pGlslcOutput, const GLSLCcompileObject* pGlslcCompileObject )
{
    auto* pInfo = pOutput->GetInfo();
    pInfo->binaryFormat = 0;
    pInfo->codeType = nn::gfx::ShaderCodeType_Binary;
    pInfo->sourceFormat = 0;
    pInfo->flags.SetBit( nn::gfx::ShaderInfoData::Flag_SeparationEnable,
        pGlslcOutput->optionFlags.glslSeparable ? true : false );

    std::shared_ptr< GLSLCdebugDataHash > pDebugDataHash;
    for( int idxSection = 0; idxSection < static_cast<int>( pGlslcOutput->numSections ); ++idxSection )
    {
        auto& common = pGlslcOutput->headers[ idxSection ].genericHeader.common;
        if( common.type == GLSLC_SECTION_TYPE_DEBUG_INFO )
        {
            auto& debugInfoHeader = pGlslcOutput->headers[ idxSection ].debugInfoHeader;
            pDebugDataHash = std::shared_ptr< GLSLCdebugDataHash >(
                static_cast< GLSLCdebugDataHash* >( malloc( sizeof( GLSLCdebugDataHash ) ) ), free );
            *pDebugDataHash.get() = debugInfoHeader.debugDataHash;
            pOutput->SetAdditionalData( pDebugDataHash );
        }
    }

    for( int idxSection = 0; idxSection < static_cast< int >( pGlslcOutput->numSections ); ++idxSection )
    {
        auto& common = pGlslcOutput->headers[ idxSection ].genericHeader.common;
        if( common.type == GLSLC_SECTION_TYPE_GPU_CODE )
        {
            auto& gpuCodeHeader = pGlslcOutput->headers[ idxSection ].gpuCodeHeader;
            auto stage = ToGfxStage( gpuCodeHeader.stage );
            auto pData = nn::util::ConstBytePtr( pGlslcOutput, gpuCodeHeader.common.dataOffset ).Get();

            std::shared_ptr< void > pDataBinary( malloc( gpuCodeHeader.dataSize ), free );
            std::shared_ptr< void > pControlBinary( malloc( gpuCodeHeader.controlSize ), free );
            memcpy( pDataBinary.get(), nn::util::ConstBytePtr( pData,
                gpuCodeHeader.dataOffset ).Get(), gpuCodeHeader.dataSize );
            memcpy( pControlBinary.get(), nn::util::ConstBytePtr( pData,
                gpuCodeHeader.controlOffset ).Get(), gpuCodeHeader.controlSize );
            pOutput->SetAdditionalData( pDataBinary );
            pOutput->SetAdditionalData( pControlBinary );

            std::shared_ptr< nn::gfx::NvnShaderCode > pNvnShaderCode( new nn::gfx::NvnShaderCode() );
            pNvnShaderCode->pData.Set( pDataBinary.get() );
            pNvnShaderCode->pControl.Set( pControlBinary.get() );
            pNvnShaderCode->dataSize = gpuCodeHeader.dataSize;
            pNvnShaderCode->controlSize = gpuCodeHeader.controlSize;
            pNvnShaderCode->scratchMemoryRecommended =
                nn::gfxTool::NumericCastAuto( gpuCodeHeader.scratchMemBytesRecommended );
            pNvnShaderCode->scratchMemoryPerWarp =
                nn::gfxTool::NumericCastAuto( gpuCodeHeader.scratchMemBytesPerWarp );
            pNvnShaderCode->pNvnDebugDataHash.Set(
                reinterpret_cast< NVNdebugDataHash* >( pDebugDataHash.get() ) );
            pOutput->SetCode( stage, pNvnShaderCode );

            auto pOptionStageOutput = static_cast< nn::gfxTool::OptionOutputProgramCommon* >( pOutput->GetOptionOutput(
                nngfxToolShaderCompilerOptionOutputType_ProgramCommon ) )->GetOptionOutputStageCommon( stage );
            if( pGlslcOutput->optionFlags.outputAssembly )
            {
                auto found = std::find( pGlslcCompileObject->input.stages,
                    pGlslcCompileObject->input.stages + pGlslcCompileObject->input.count, gpuCodeHeader.stage );
                if( found != pGlslcCompileObject->input.stages + pGlslcCompileObject->input.count )
                {
                    std::shared_ptr< nn::gfxTool::Custom< std::string >::Type > pDump(
                        new nn::gfxTool::Custom< std::string >::Type() );
                    const char* const separator = "------------------------";
                    if( pGlslcCompileObject->options.optionFlags.language == GLSLC_LANGUAGE_GLSL ||
                        pGlslcCompileObject->options.optionFlags.language == GLSLC_LANGUAGE_GLES )
                    {
                        pDump->append( pGlslcCompileObject->input.sources[ std::distance(
                            pGlslcCompileObject->input.stages, found ) ] ).append( "\n" );
                    }
                    if( gpuCodeHeader.asmDumpSectionIdx )
                    {
                        auto& asmDumpSectionHeader = pGlslcOutput->headers[
                            gpuCodeHeader.asmDumpSectionIdx ].asmDumpHeader;
                        auto pAsmString = nn::util::ConstBytePtr( pGlslcOutput,
                            asmDumpSectionHeader.common.dataOffset ).Get< const char >();
                        pDump->append( separator ).append( "Assembly" ).append(
                            separator ).append( "\n" ).append( pAsmString ).append( "\n" );
                    }
                    pDump->append( separator ).append( "GLSLC Option Flags" ).append( separator ).append(
                        "\n" ).append( GlslcOptionFlagsToString( pGlslcOutput->optionFlags ) ).append( "\n" );
                    pOptionStageOutput->SetDump( pDump );
                }
            }
            if( gpuCodeHeader.perfStatsSectionNdx )
            {
                std::shared_ptr< GLSLCperfStatsData > pStats( new GLSLCperfStatsData() );
                auto& perfStatsHeader = pGlslcOutput->headers[
                    gpuCodeHeader.perfStatsSectionNdx ].perfStatsHeader;
                *pStats = *nn::util::ConstBytePtr( pGlslcOutput,
                    perfStatsHeader.common.dataOffset ).Get< GLSLCperfStatsData >();
                pOptionStageOutput->SetShaderStatistics( pStats );
            }

            auto pOptionStageOutputNvn = static_cast< nn::gfxTool::OptionOutputProgramNvn* >( pOutput->GetOptionOutput(
                nn::gfxTool::StaticCastAuto( nngfxToolShaderCompilerOptionOutputType_ProgramNvn ) ) )->GetOptionOutputStageNvn( stage );
            pOptionStageOutputNvn->GetOutput()->recommendedScratchMemorySizePerWarp =
                nn::gfxTool::NumericCastAuto( gpuCodeHeader.scratchMemBytesRecommended );
            pOptionStageOutputNvn->GetOutput()->requiredScratchMemorySizePerWarp =
                nn::gfxTool::NumericCastAuto( gpuCodeHeader.scratchMemBytesPerWarp );
        }
    }
}

void RetrieveReflection( nn::gfxTool::OptionOutputReflection* pOutput,
    const GLSLCoutput* pGlslcOutput )
{
    if( pOutput == nullptr || pGlslcOutput == nullptr )
    {
        NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
    }

    auto pGlslcReflectionHeader = GetSectionHeader< GLSLCprogramReflectionHeader >( pGlslcOutput );
    if( pGlslcReflectionHeader == nullptr )
    {
        return;
    }
    auto pReflectionData = nn::util::ConstBytePtr(
        pGlslcOutput, pGlslcReflectionHeader->common.dataOffset ).Get();
    auto pStringPool = nn::util::ConstBytePtr( pReflectionData,
        pGlslcReflectionHeader->stringPoolOffset ).Get();

    auto ToGfxType = []( GLSLCpiqTypeEnum nvnType )
    {
        return nvnType == GLSLC_PIQ_INVALID_TYPE
            ? nngfxToolShaderCompilerReflectionVariableType_Unknown : s_NvnTypeToGfxTypeTable[ nvnType ];
    };

    auto SetName = [ & ]( nngfxToolString* pDst, const GLSLCpiqName* pSrc )
    {
        pDst->pValue = nn::util::ConstBytePtr( pStringPool, pSrc->nameOffset ).Get< const char >();
        pDst->length = nn::gfxTool::NumericCastAuto( pSrc->nameLength - 1 );
    };

    auto SetStageBits = []( int32_t* pDst, NVNshaderStageBits src )
    {
        *pDst = 0;
        *pDst |= ( src & NVN_SHADER_STAGE_VERTEX_BIT )
            ? nngfxToolShaderCompilerReflectionStageReference_Vertex : 0;
        *pDst |= ( src & NVN_SHADER_STAGE_TESS_CONTROL_BIT )
            ? nngfxToolShaderCompilerReflectionStageReference_Hull : 0;
        *pDst |= ( src & NVN_SHADER_STAGE_TESS_EVALUATION_BIT )
            ? nngfxToolShaderCompilerReflectionStageReference_Domain : 0;
        *pDst |= ( src & NVN_SHADER_STAGE_GEOMETRY_BIT )
            ? nngfxToolShaderCompilerReflectionStageReference_Geometry : 0;
        *pDst |= ( src & NVN_SHADER_STAGE_FRAGMENT_BIT )
            ? nngfxToolShaderCompilerReflectionStageReference_Pixel : 0;
        *pDst |= ( src & NVN_SHADER_STAGE_COMPUTE_BIT )
            ? nngfxToolShaderCompilerReflectionStageReference_Compute : 0;
    };

    auto SetShaderSlots = []( nngfxToolShaderCompilerShaderSlot* pDst, const int32_t* pBindings )
    {
        pDst->vertexShaderSlot = pBindings[ NVN_SHADER_STAGE_VERTEX ];
        pDst->hullShaderSlot = pBindings[ NVN_SHADER_STAGE_TESS_CONTROL ];
        pDst->domainShaderSlot = pBindings[ NVN_SHADER_STAGE_TESS_EVALUATION ];
        pDst->geometryShaderSlot = pBindings[ NVN_SHADER_STAGE_GEOMETRY ];
        pDst->pixelShaderSlot = pBindings[ NVN_SHADER_STAGE_FRAGMENT ];
        pDst->computeShaderSlot = pBindings[ NVN_SHADER_STAGE_COMPUTE ];
    };

    auto pUniformBlocks = nn::util::ConstBytePtr( pReflectionData,
        pGlslcReflectionHeader->uniformBlockOffset ).Get< GLSLCuniformBlockInfo >();
    for( int idxUniformBlock = 0; idxUniformBlock < static_cast< int >(
        pGlslcReflectionHeader->numUniformBlocks ); ++idxUniformBlock )
    {
        auto& uniformBlock = pUniformBlocks[ idxUniformBlock ];
        nngfxToolShaderCompilerConstantBuffer constantBuffer = {};
        SetName( &constantBuffer.name, &uniformBlock.nameInfo );
        SetShaderSlots( &constantBuffer.shaderSlot, uniformBlock.bindings );
        constantBuffer.size = uniformBlock.size;
        constantBuffer.activeVariableCount = uniformBlock.numActiveVariables;
        SetStageBits( &constantBuffer.stages, uniformBlock.stagesReferencedIn );
        pOutput->AddReflection( constantBuffer );
    }

    auto pSsbos = nn::util::ConstBytePtr( pReflectionData,
        pGlslcReflectionHeader->ssboOffset ).Get< GLSLCssboInfo >();
    for( int idxSsbo = 0; idxSsbo < static_cast< int >( pGlslcReflectionHeader->numSsbo ); ++idxSsbo )
    {
        auto& ssbo = pSsbos[ idxSsbo ];
        nngfxToolShaderCompilerUnorderedAccessBuffer unorderedAccessBuffer = {};
        SetName( &unorderedAccessBuffer.name, &ssbo.nameInfo );
        SetShaderSlots( &unorderedAccessBuffer.shaderSlot, ssbo.bindings );
        unorderedAccessBuffer.size = ssbo.size;
        unorderedAccessBuffer.activeVariableCount = ssbo.numActiveVariables;
        SetStageBits( &unorderedAccessBuffer.stages, ssbo.stagesReferencedIn );
        pOutput->AddReflection( unorderedAccessBuffer );
    }

    auto pProgramInputs = nn::util::ConstBytePtr( pReflectionData,
        pGlslcReflectionHeader->programInputsOffset ).Get< GLSLCprogramInputInfo >();
    for( int idxProgramInput = 0; idxProgramInput < static_cast< int >(
        pGlslcReflectionHeader->numProgramInputs ); ++idxProgramInput )
    {
        auto& programInput = pProgramInputs[ idxProgramInput ];
        nngfxToolShaderCompilerShaderInput shaderInput = {};
        SetName( &shaderInput.name, &programInput.nameInfo );
        shaderInput.type = ToGfxType( programInput.type );
        shaderInput.shaderSlot = nn::gfxTool::NumericCastAuto( programInput.location );
        shaderInput.arrayCount = programInput.isArray;
        SetStageBits( &shaderInput.stages, programInput.stagesReferencedIn );
        pOutput->AddReflection( shaderInput );
    }

    auto pProgramOutputs = nn::util::ConstBytePtr( pReflectionData,
        pGlslcReflectionHeader->programOutputsOffset ).Get< GLSLCprogramOutputInfo >();
    for( int idxProgramOuptput = 0; idxProgramOuptput < static_cast< int >(
        pGlslcReflectionHeader->numProgramOutputs ); ++idxProgramOuptput )
    {
        auto& programOutput = pProgramOutputs[ idxProgramOuptput ];
        nngfxToolShaderCompilerShaderOutput shaderOutput = {};
        SetName( &shaderOutput.name, &programOutput.nameInfo );
        shaderOutput.type = ToGfxType( programOutput.type );
        shaderOutput.shaderSlot = nn::gfxTool::NumericCastAuto(
            programOutput.location < 0 ? programOutput.locationNdx : programOutput.location );
        shaderOutput.arrayCount = programOutput.sizeOfArray;
        SetStageBits( &shaderOutput.stages, programOutput.stagesReferencedIn );
        pOutput->AddReflection( shaderOutput );
    }

    auto pUniforms = nn::util::ConstBytePtr( pReflectionData,
        pGlslcReflectionHeader->uniformOffset ).Get< GLSLCuniformInfo >();
    for( int idxUniform = 0; idxUniform < static_cast< int >(
        pGlslcReflectionHeader->numUniforms ); ++idxUniform )
    {
        auto& uniform = pUniforms[ idxUniform ];
        auto type = ToGfxType( uniform.type );
        auto SetOpaqueUniformProperty = [ & ]( auto pReflection )
        {
            SetName( &pReflection->name, &uniform.nameInfo );
            pReflection->type = type;
            SetShaderSlots( &pReflection->shaderSlot, uniform.bindings );
            SetStageBits( &pReflection->stages, uniform.stagesReferencedIn );
            pReflection->arrayCount = uniform.sizeOfArray;
        };
        if( nn::gfxTool::IsSamplerType( type ) )
        {
            nngfxToolShaderCompilerSampler sampler = {};
            SetOpaqueUniformProperty( &sampler );
            pOutput->AddReflection( sampler );
        }
        else if( nn::gfxTool::IsImageType( type ) )
        {
            nngfxToolShaderCompilerImage image = {};
            SetOpaqueUniformProperty( &image );
            pOutput->AddReflection( image );
        }
        else if( nn::gfxTool::IsSeparateTextureType( type ) )
        {
            nngfxToolShaderCompilerSeparateTexture separateTexture = {};
            SetOpaqueUniformProperty( &separateTexture );
            pOutput->AddReflection( separateTexture );
        }
        else if( nn::gfxTool::IsSeparateSamplerType( type ) )
        {
            nngfxToolShaderCompilerSeparateSampler separateSampler = {};
            SetOpaqueUniformProperty( &separateSampler );
            pOutput->AddReflection( separateSampler );
        }
        else if( uniform.blockNdx >= 0 )
        {
            nngfxToolShaderCompilerConstantBufferVariable constantBufferVariable = {};
            SetName( &constantBufferVariable.name, &uniform.nameInfo );
            constantBufferVariable.type = type;
            constantBufferVariable.blockIndex = uniform.blockNdx;
            constantBufferVariable.offset = nn::gfxTool::StaticCastAuto( uniform.blockOffset );
            constantBufferVariable.arrayCount = uniform.sizeOfArray;
            SetStageBits( &constantBufferVariable.stages, uniform.stagesReferencedIn );
            pOutput->AddReflection( constantBufferVariable );
        }
    }

    auto pBufferVariable = nn::util::ConstBytePtr( pReflectionData,
        pGlslcReflectionHeader->bufferVariableOffset ).Get< GLSLCbufferVariableInfo >();
    for( int idxBufferVariable = 0; idxBufferVariable < static_cast< int >(
        pGlslcReflectionHeader->numBufferVariables ); ++idxBufferVariable )
    {
        auto& bufferVariable = pBufferVariable[ idxBufferVariable ];
        nngfxToolShaderCompilerUnorderedAccessBufferVariable unorderedAccessBufferVariable = {};
        SetName( &unorderedAccessBufferVariable.name, &bufferVariable.nameInfo );
        unorderedAccessBufferVariable.type = ToGfxType( bufferVariable.type );
        unorderedAccessBufferVariable.blockIndex = bufferVariable.blockNdx;
        unorderedAccessBufferVariable.offset = nn::gfxTool::StaticCastAuto( bufferVariable.blockOffset );
        unorderedAccessBufferVariable.arrayCount = bufferVariable.sizeOfArray;
        SetStageBits( &unorderedAccessBufferVariable.stages, bufferVariable.stagesReferencedIn );
        pOutput->AddReflection( unorderedAccessBufferVariable );
    }

    auto pShaderInfo = nn::util::ConstBytePtr( pReflectionData,
        pGlslcReflectionHeader->shaderInfoOffset ).Get< GLSLCperStageShaderInfo >();
    auto& shaderInfoCompute = pShaderInfo->shaderInfo[ NVN_SHADER_STAGE_COMPUTE ].compute;
    pOutput->GetOutput()->computeWorkGroupSizeX = shaderInfoCompute.workGroupSize[ 0 ];
    pOutput->GetOutput()->computeWorkGroupSizeY = shaderInfoCompute.workGroupSize[ 1 ];
    pOutput->GetOutput()->computeWorkGroupSizeZ = shaderInfoCompute.workGroupSize[ 2 ];
} // NOLINT

void RetrieveDebugInfo( const GLSLCoutput* pGlslcOutput,
    const nngfxToolShaderCompilerCompileArg* pArg )
{
    auto pDebugInfoHeader = GetSectionHeader< GLSLCdebugInfoHeader >( pGlslcOutput );
    if( pDebugInfoHeader )
    {
        auto& debugHash = pDebugInfoHeader->debugDataHash;
        nn::gfxTool::Custom< std::stringstream >::Type ss;
        nn::gfxTool::Custom< std::string >::Type debugInfoFilename;
        ss << std::hex << std::setfill( '0' ) << std::setw( 8 )
            << debugHash.debugHashHi32 << debugHash.debugHashLo32 << ".glslcoutput";
        ss >> debugInfoFilename;
        pArg->pWriteDebugInfoFileCallback( pGlslcOutput, pGlslcOutput->size,
            debugInfoFilename.c_str(), pArg->pWriteDebugInfoFileCallbackParam );
    }
}

}

namespace nn {
namespace gfxTool {

class SpirvSpecializationInfo
{
public:
    SpirvSpecializationInfo()
        : m_Data( nullptr, &free )
    {
    }

    void Initialize( const nngfxToolShaderCompilerCompileArg* pArg, int  variation, ShaderStage stage )
    {
        m_ConstantIds.clear();
        m_GlslcSpirvSpecializationInfo.numEntries = 0;
        m_GlslcSpirvSpecializationInfo.constantIDs = nullptr;
        m_GlslcSpirvSpecializationInfo.data = nullptr;

        if( auto pVariationDefinition = GetStageVariationDefinition( pArg->pVariationDefinitionArg, stage ) )
        {
            m_Data.reset( malloc( 4 * pVariationDefinition->variationConstantDefinitionCount ) );
            auto pVariationValue = GetStageVariationValue( pArg->pVariationValueArray + variation, stage );
            auto pVariationConstantValues = pVariationValue->pVariationConstantValueArray;
            for( int idxVariationConstant = 0, variationConstantCount = static_cast< int >(
                pVariationDefinition->variationConstantDefinitionCount );
                idxVariationConstant < variationConstantCount; ++idxVariationConstant )
            {
                auto& variationConstantDefinition =
                    pVariationDefinition->pVariationConstantDefinitionArray[ idxVariationConstant ];
                Custom< std::string >::Type name(
                    variationConstantDefinition.name.pValue, variationConstantDefinition.name.length );
                if( variationConstantDefinition.matrix.column > 0 ||
                    variationConstantDefinition.matrix.row > 0 ||
                    variationConstantDefinition.arrayLength > 0 )
                {
                    NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument,
                        "Specialization on non-scalar variable is not allowed in Spir-V (%s).", name.c_str() );
                }
                auto& variationConstantValue =
                    pVariationConstantValues[ variationConstantDefinition.indexInVariationConstantValueArray ];
                if( variationConstantValue.valueSizeIn32Bit != 1 )
                {
                    NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument,
                        "Value of specialization constant must be 4-byte in Spir-V (%s).", name.c_str() );
                }
                if( !std::all_of( name.cbegin(), name.cend(), std::isdigit ) )
                {
                    NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument,
                        "Name of spir-v specialization constant must be id (%s).", name.c_str() );
                }
                int id = LexicalCastAuto( name );
                m_ConstantIds.push_back( id );
                memcpy( nn::util::BytePtr( m_Data.get(), 4 * idxVariationConstant ).Get(),
                    variationConstantValue.pValue, 4 );
            }

            m_GlslcSpirvSpecializationInfo.numEntries = NumericCastAuto( m_ConstantIds.size() );
            if( m_GlslcSpirvSpecializationInfo.numEntries > 0 )
            {
                m_GlslcSpirvSpecializationInfo.constantIDs = &m_ConstantIds[ 0 ];
                m_GlslcSpirvSpecializationInfo.data = m_Data.get();
            }
        }
    }

    const GLSLCspirvSpecializationInfo* GetGlslcSpirvSpecilizationInfo() const
    {
        return &m_GlslcSpirvSpecializationInfo;
    }

private:
    GLSLCspirvSpecializationInfo m_GlslcSpirvSpecializationInfo;
    Custom< std::vector< uint32_t > >::Type m_ConstantIds;
    std::unique_ptr< void, decltype( &free ) > m_Data;
};

class SpecializationBatch
{
public:
    void Initialize( const nngfxToolShaderCompilerCompileArg* pArg,
        Custom< std::vector< int > >::Type& variations )
    {
        return Initialize( pArg, variations, 0, StaticCastAuto( variations.size() ) );
    }

    void Initialize( const nngfxToolShaderCompilerCompileArg* pArg,
        Custom< std::vector< int > >::Type& variations,
        int startVariation, int variationCount )
    {
        if( pArg == nullptr || startVariation < 0 ||
            startVariation + variationCount > static_cast< int >( variations.size() ) )
        {
            NN_GFXTOOL_THROW( nngfxToolResultCode_InternalError );
        }

        m_GlslcSpecializationBatch.numEntries = 0;
        m_GlslcSpecializationBatch.entries = nullptr;

        m_GlslcSpecializationBatch.numEntries = NumericCastAuto( variationCount );
        m_GlslcSpecializationSets.resize( variationCount );
        m_GlslcSpecializationBatch.entries = &m_GlslcSpecializationSets[ 0 ];
        int variationConstantDefinitionCount = 0;
        for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
        {
            if( auto pStageVariationDefinition = GetStageVariationDefinition(
                pArg->pVariationDefinitionArg, StaticCastAuto( idxStage ) ) )
            {
                variationConstantDefinitionCount += NumericCast< decltype(
                    variationConstantDefinitionCount ) >(
                    pStageVariationDefinition->variationConstantDefinitionCount );
            }
        }

        struct SpecializationUniformInfo
        {
            const char* uniformName;
            uint32_t numElements;
            int stage;
            int indexInVariationConstantValueArray;
        };
        struct ErrorCheckInfo
        {
            int stage;
            int indexInVariationConstantValueArray;
            int checkIndexInSpecializationUniformInfo;
        };

        Custom< std::vector< SpecializationUniformInfo > >::Type specializationUniformInfo;
        Custom< std::vector< ErrorCheckInfo > >::Type errorCheckInfo;
        specializationUniformInfo.reserve( variationConstantDefinitionCount );
        errorCheckInfo.reserve( variationConstantDefinitionCount );
        for( int idxStage = 0; idxStage < static_cast<int>( ShaderStage::End ); ++idxStage )
        {
            auto pStageVariationDefinition = GetStageVariationDefinition(
                pArg->pVariationDefinitionArg, StaticCastAuto( idxStage ) );
            if( pStageVariationDefinition == nullptr )
            {
                continue;
            }
            for( int idxVariationDefinition = 0; idxVariationDefinition < static_cast<int>(
                pStageVariationDefinition->variationConstantDefinitionCount ); ++idxVariationDefinition )
            {
                auto& definition = pStageVariationDefinition->pVariationConstantDefinitionArray[ idxVariationDefinition ];
                auto& name = definition.name;
                int idxUniform = 0;
                int uniformCount = static_cast<int>( specializationUniformInfo.size() );
                for( ; idxUniform < uniformCount; ++idxUniform )
                {
                    auto pCheckName = specializationUniformInfo[ idxUniform ].uniformName;
                    if( strcmp( name.pValue, pCheckName ) == 0 )
                    {
                        break;
                    }
                }
                if( idxUniform >= uniformCount )
                {
                    m_UniformNames.emplace_back( name.pValue, name.length );
                    specializationUniformInfo.emplace_back();
                    auto& info = specializationUniformInfo.back();
                    info.uniformName = m_UniformNames.back().c_str();
                    info.numElements = definition.arrayLength == 0 ? 1 : definition.arrayLength;
                    info.stage = idxStage;
                    info.indexInVariationConstantValueArray = definition.indexInVariationConstantValueArray;
                }
                else
                {
                    errorCheckInfo.emplace_back();
                    auto& info = errorCheckInfo.back();
                    info.stage = idxStage;
                    info.indexInVariationConstantValueArray = definition.indexInVariationConstantValueArray;
                    info.checkIndexInSpecializationUniformInfo = idxUniform;
                }
            }
        }
        int uniformCount = static_cast<int>( specializationUniformInfo.size() );
        m_GlslcSpecializationUniforms.reserve( uniformCount * variationCount );
        for( int idxSet = 0; idxSet < variationCount; ++idxSet )
        {
            auto idxVariation = variations[ idxSet + startVariation ];
            auto& set = m_GlslcSpecializationSets[ idxSet ];
            set.numUniforms = uniformCount;
            for( auto& info : specializationUniformInfo )
            {
                auto pStageVariationValue = GetStageVariationValue(
                    pArg->pVariationValueArray + idxVariation, StaticCastAuto( info.stage ) );
                auto& variationConstantValue = pStageVariationValue->pVariationConstantValueArray[
                    info.indexInVariationConstantValueArray ];
                m_GlslcSpecializationUniforms.emplace_back();
                auto& uniform = m_GlslcSpecializationUniforms.back();
                uniform.values = const_cast<void*>( variationConstantValue.pValue );
                uniform.uniformName = info.uniformName;
                uniform.numElements = info.numElements;
                uniform.elementSize = NumericCastAuto(
                    variationConstantValue.valueSizeIn32Bit * 4 / uniform.numElements );
            }
            for( auto& info : errorCheckInfo )
            {
                auto pStageVariationValue = GetStageVariationValue(
                    pArg->pVariationValueArray + idxVariation, StaticCastAuto( info.stage ) );
                auto& variationConstantValue = pStageVariationValue->pVariationConstantValueArray[
                    info.indexInVariationConstantValueArray ];
                auto& uniform = m_GlslcSpecializationUniforms[
                    idxSet * uniformCount + info.checkIndexInSpecializationUniformInfo ];
                if( memcmp( variationConstantValue.pValue, uniform.values, variationConstantValue.valueSizeIn32Bit * 4 ) )
                {
                    NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InconsistentSpecilizationUniform,
                        "Variation constant (%s) values are inconsistent (variation:%d).", uniform.uniformName, idxVariation );
                }
            }
            set.uniforms = set.numUniforms > 0 ? &m_GlslcSpecializationUniforms[ idxSet * uniformCount ] : nullptr;
        }
    } // NOLINT

    const GLSLCspecializationBatch* GetGlslcSpecializationBatch() const
    {
        return &m_GlslcSpecializationBatch;
    }

private:
    GLSLCspecializationBatch m_GlslcSpecializationBatch;
    Custom< std::vector< GLSLCspecializationSet > >::Type m_GlslcSpecializationSets;
    Custom< std::vector< GLSLCspecializationUniform > >::Type m_GlslcSpecializationUniforms;
    Custom< std::list< Custom< std::string >::Type > >::Type m_UniformNames;
};

class CompileObject
{
public:
    CompileObject()
        : m_CompileObject()
        , m_Sources()
        , m_pSources()
        , m_Stages()
        , m_ModuleSizes()
        , m_pSpirvSpecializationInfos()
        , m_SpirvSpecializationInfos()
    {
    }

    GLSLCcompileObject* GetGlslcCompileObject()
    {
        return &m_CompileObject;
    }

    void Initialize( const NvnGlslcDll& nvnGlslcDll )
    {
        if( !nvnGlslcDll.GlslcInitialize( &m_CompileObject )
            || m_CompileObject.initStatus != GLSLC_INIT_SUCCESS )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_Failed, "glslcInitialize failed: %s.",
                GetInitializationStatusString( m_CompileObject.initStatus ) );
        }
    }

    void SetOptions( const GLSLCoptions& options )
    {
        m_CompileObject.options = options;
    }

    void SetupInputs( const ShaderCompilerContext* pContext,
        const nngfxToolShaderCompilerCompileArg* pArg, int idxVariation )
    {
        auto pCompileOptionManager = pContext->GetCompileOptionManager();
        auto pCommonOption = pCompileOptionManager->GetCompileOption<
            nngfxToolShaderCompilerOptionType_Common >();
        auto pGlslOption = pCompileOptionManager->GetCompileOption<
            nngfxToolShaderCompilerOptionType_Glsl >();

        m_CompileObject.input.count = 0;
        m_CompileObject.input.sources = m_pSources;
        m_CompileObject.input.stages = m_Stages;
        m_CompileObject.input.spirvEntryPointNames = nullptr; // "main" 前提
        m_CompileObject.input.spirvSpecInfo = m_pSpirvSpecializationInfos;
        m_CompileObject.input.spirvModuleSizes = m_ModuleSizes;
        BoostPreprocessor preprocessors[ static_cast<int>( ShaderStage::End ) ];
        for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
        {
            auto stage = static_cast< ShaderStage >( idxStage );
            auto& stageSource = GetStageSource( pArg, stage );
            if( stageSource.pValue == nullptr )
            {
                continue;
            }

            if( m_CompileObject.options.optionFlags.language == GLSLC_LANGUAGE_SPIRV )
            {
                m_pSources[ m_CompileObject.input.count ] = stageSource.pValue;
                m_ModuleSizes[ m_CompileObject.input.count ] = stageSource.length;
                m_Stages[ m_CompileObject.input.count ] = ToNvnStage( stage );
                m_SpirvSpecializationInfos[ m_CompileObject.input.count ].Initialize( pArg, idxVariation, stage );
                m_pSpirvSpecializationInfos[ m_CompileObject.input.count ] =
                    m_SpirvSpecializationInfos[ m_CompileObject.input.count ].GetGlslcSpirvSpecilizationInfo();
                ++m_CompileObject.input.count;
            }
            else
            {
                auto preprocessorDefinitionGroup = pContext->GetVariationManager(
                )->GetPreprocessorDefinitionGroup( stage )->GetVariationToGroupTable().at( idxVariation );
                auto& preprocessorDefinitionSource = pContext->GetVariationManager(
                )->GetPreprocessorDefinitionSource( stage )->GetSources().at( preprocessorDefinitionGroup );

                auto pShaderSource = pContext->GetShaderSourceManager()->GetShaderSource( stage );

                auto& source = m_Sources[ m_CompileObject.input.count ];
                source.clear();
                if( pCommonOption->IsPreprocessEnabled() )
                {
                    Custom< std::string >::Type tmpSource;
                    tmpSource.append( pGlslOption->GetGlslHeader()->c_str() ).append(
                        pCommonOption->GetPreprocessorDefinitionSource()->c_str() ).append(
                        preprocessorDefinitionSource.c_str() ).append( stageSource.pValue, stageSource.length );

                    auto& preprocessor = preprocessors[ idxStage ];
                    preprocessor.SetReadFileCallback( pArg->pReadIncludeFileCallback,
                        pArg->pReadIncludeFileCallbackParam );
                    if( pArg->pVariationDefinitionArg )
                    {
                        preprocessor.SetVariationDefinition( GetStageVariationDefinition(
                            pArg->pVariationDefinitionArg, stage ) );
                    }
                    preprocessor.SetVariationConstantEmulationEnabled( false );
                    auto& uniformRegisterBlockName = pCommonOption->GetUniformRegisterBlockName();
                    preprocessor.SetUniformRegisterToBlockEnabled( !uniformRegisterBlockName.empty() );
                    preprocessor.Preprocess( tmpSource.c_str(), tmpSource.length() );
                }
                else
                {
                    source.append( pGlslOption->GetGlslHeader()->c_str() );
                    source.append( *pCommonOption->GetPreprocessorDefinitionSource().get() );
                    source.append( preprocessorDefinitionSource );
                    source.append( pShaderSource->includeExpandedSource.c_str() );
                    m_pSources[ m_CompileObject.input.count ] = source.c_str();
                    m_Stages[ m_CompileObject.input.count ] = ToNvnStage( stage );
                    ++m_CompileObject.input.count;
                }
            }
        }

        if( pCommonOption->IsPreprocessEnabled() &&
            m_CompileObject.options.optionFlags.language != GLSLC_LANGUAGE_SPIRV )
        {
            BoostPreprocessor::ResolveUniformRegisterBlock( static_cast<int>( ShaderStage::End ),
                preprocessors, pCommonOption->GetUniformRegisterBlockName().c_str() );
            for( int idxStage = 0; idxStage < static_cast<int>( ShaderStage::End ); ++idxStage )
            {
                auto stage = static_cast< ShaderStage >( idxStage );
                if( GetStageSource( pArg, stage ).pValue )
                {
                    m_Sources[ idxStage ] = preprocessors[ idxStage ].GetResult().str();
                    m_pSources[ m_CompileObject.input.count ] = m_Sources[ idxStage ].c_str();
                    m_Stages[ m_CompileObject.input.count ] = ToNvnStage( stage );
                    ++m_CompileObject.input.count;
                }
            }
        }
    }

    void Finalize( const NvnGlslcDll& nvnGlslcDll )
    {
        nvnGlslcDll.GlslcFinalize( &m_CompileObject );
    }

private:
    GLSLCcompileObject m_CompileObject;
    Custom< std::string >::Type m_Sources[ static_cast<int>( ShaderStage::End ) ];
    const char* m_pSources[ static_cast<int>( ShaderStage::End ) ];
    NVNshaderStage m_Stages[ static_cast<int>( ShaderStage::End ) ];
    uint32_t m_ModuleSizes[ static_cast<int>( ShaderStage::End ) ];
    const GLSLCspirvSpecializationInfo*
        m_pSpirvSpecializationInfos[ static_cast<int>( ShaderStage::End ) ];
    SpirvSpecializationInfo
        m_SpirvSpecializationInfos[ static_cast<int>( ShaderStage::End ) ];
};

typedef CompilerVariation< static_cast<
    nngfxToolShaderCompilerLowLevelApiType >( nngfxToolShaderCompilerLowLevelApiType_Nvn ),
    nngfxToolShaderCompilerCodeType_Binary >
    Target;

Compiler< Target >::Compiler() = default;

Compiler< Target >::~Compiler() = default;

int Compiler< Target >::GetGroupCount( const ShaderCompilerContext* pContext,
        const nngfxToolShaderCompilerCompileArg* pArg ) const
{
    return pArg->shaderSourceFormat == nngfxToolShaderCompilerShaderSourceFormat_Glsl ?
        pContext->GetVariationManager()->GetPreprocessorDefinitionGroup()->GetGroupCount() :
        pArg->variationCount;
}

void Compiler< Target >::PreCompile( CompileOutput* pOutput,
    const ShaderCompilerContext* pContext, const nngfxToolShaderCompilerCompileArg* pArg )
{
    if( !m_NvnGlslcDll.IsInitialized() )
    {
        if( !m_NvnGlslcDll.Initialize( GetNvnGlslcPath().c_str() ) )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_DllNotFound,
                "Failed to load %s.", GetNvnGlslcPath().c_str() );
        }
    }

    auto version = m_NvnGlslcDll.GlslcGetVersion();

    GLSLCoptions glslcOptions = {};
    SetupGlslcOptions( &glslcOptions, pContext, pArg, m_NvnGlslcDll );

    const int MaxThreadForGlslc = version.package >= 50 ? 64 : 16;
    int threadCount = 1;
    if( glslcOptions.optionFlags.enableMultithreadCompilation )
    {
        threadCount = std::min NN_PREVENT_MACRO_FUNC( GetThreadCount(), MaxThreadForGlslc );
    }
    SetThreadCount( threadCount );

    m_CompileObjects.reset( new CompileObject[ threadCount ] );
    for( int idxCompileObject = 0; idxCompileObject < threadCount; ++idxCompileObject )
    {
        auto& compileObject = m_CompileObjects[ idxCompileObject ];
        compileObject.Initialize( m_NvnGlslcDll );
        compileObject.SetOptions( glslcOptions );
    }

    NN_GFXTOOL_PRINT( "GLSLC version: %d.%d\n" // TODO
        "GPU Code version: %d.%d\n"
        "Package version: %d\n",
        version.apiMajor, version.apiMinor,
        version.gpuCodeVersionMajor, version.gpuCodeVersionMinor,
        version.package );
    uint16_t apiMajor = NumericCastAuto( version.apiMajor );
    uint16_t apiMinor = NumericCastAuto( version.apiMinor );
    uint16_t gpuCodeVersionMajor = NumericCastAuto( version.gpuCodeVersionMajor );
    uint16_t gpuCodeVersionMinor = NumericCastAuto( version.gpuCodeVersionMinor );
    pOutput->lowLevelCompilerVerison =
        ( static_cast< uint64_t >( apiMajor ) << 48 ) |
        ( static_cast< uint64_t >( apiMinor ) << 32 ) |
        ( static_cast< uint64_t >( gpuCodeVersionMajor ) << 16 ) |
        ( static_cast< uint64_t >( gpuCodeVersionMinor ) << 0 );
    if( version.apiMajor == 17 )
    {
        // 17.12 以降をサポート
        if( version.apiMinor < 12 )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidApiVersion,
                "Invalid glslc version.\n\tVersion: %d.%d\n\tExpected: >= 17.12\n",
                version.apiMajor, version.apiMinor );
        }
    }
    else if( GLSLC_API_VERSION_MAJOR != version.apiMajor ||
        GLSLC_API_VERSION_MINOR > version.apiMinor )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidApiVersion,
            "Invalid glslc version.\n\tVersion: %d.%d\n\tExpected: >= %d.%d\n",
            version.apiMajor, version.apiMinor,
            GLSLC_API_VERSION_MAJOR, GLSLC_API_VERSION_MINOR );
    }
}

void Compiler< Target >::CompileGroup( CompileOutput* pOutput, const ShaderCompilerContext* pContext,
    const nngfxToolShaderCompilerCompileArg* pArg, int idxGroup )
{
    auto idxCompileObject = GetThreadIndex( idxGroup );

    auto pCompileOptionManager= pContext->GetCompileOptionManager();
    auto pCommonOption = pCompileOptionManager->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >();

    auto variations = CreateGroupVariations( pContext, pArg, idxGroup );

    auto& compileObject = m_CompileObjects[ idxCompileObject ];
    compileObject.SetupInputs( pContext, pArg, variations.at( 0 ) );

    auto& glslcCompileObject = *compileObject.GetGlslcCompileObject();
    bool isBatchCompileMode =
        glslcCompileObject.options.optionFlags.language != GLSLC_LANGUAGE_SPIRV;

    Custom< std::string >::Type commonInfoLog = "";
    auto RetrieveVariationOutput = [ & ]( VariationOutput* pOutVariationOutput,
        const GLSLCoutput& glslcOutput, const nn::util::string_view& variationInfoLog )
    {
        auto pProgramOutput = pOutVariationOutput->GetBinaryOutput();
        std::shared_ptr< OptionOutputProgramCommon > pOptionOutputProgramCommon(
            new OptionOutputProgramCommon() );
        pOptionOutputProgramCommon->Initialize( pArg );
        pProgramOutput->AddOptionOutput(
            nngfxToolShaderCompilerOptionOutputType_ProgramCommon, pOptionOutputProgramCommon );
        std::shared_ptr< OptionOutputProgramNvn > pOptionOutputProgramNvn( new OptionOutputProgramNvn );
        pOptionOutputProgramNvn->Initialize( pArg );
        pProgramOutput->AddOptionOutput( StaticCastAuto(
            nngfxToolShaderCompilerOptionOutputType_ProgramNvn ), pOptionOutputProgramNvn );
        RetrieveCompileOutput( pProgramOutput, &glslcOutput, &glslcCompileObject );
        std::shared_ptr< Custom< std::string >::Type > pInfoLog( new Custom< std::string >::Type() );
        pInfoLog->append( commonInfoLog );
        if( variationInfoLog.data() )
        {
            pInfoLog->append( variationInfoLog.data(), variationInfoLog.size() );
        }
        if( glslcCompileObject.input.count > 0 && !pInfoLog->empty() )
        {
            pOptionOutputProgramCommon->GetOptionOutputStageCommon( StaticCastAuto(
                ToGfxStage( glslcCompileObject.input.stages[ 0 ] ) ) )->SetInfoLog( pInfoLog );
        }
        if( pCommonOption->IsReflectionEnabled() )
        {
            std::shared_ptr< OptionOutputReflection > pReflection( new OptionOutputReflection() );
            RetrieveReflection( pReflection.get(), &glslcOutput );
            pOptionOutputProgramCommon->SetReflection( pReflection );
        }
        if( pArg->pWriteDebugInfoFileCallback )
        {
            RetrieveDebugInfo( &glslcOutput, pArg );
        }
    };

    // 二階層目の並列化
    auto threadCount = std::min NN_PREVENT_MACRO_FUNC(
        pCommonOption->GetMaxThreads(), static_cast< int >( variations.size() ) );
    if( m_NvnGlslcDll.GlslcCompileSpecializedMT == nullptr ||
        glslcCompileObject.options.optionFlags.enableMultithreadCompilation == 0 )
    {
        threadCount = 1;
    }

    auto batchCount = threadCount; // threadCount ~ variations.size() の範囲で調整可

    Custom< std::vector< SpecializationBatch > >::Type specializationBatches;
    specializationBatches.resize( batchCount );

    auto memorySize = nn::util::BitArray::CalculateWorkMemorySize( batchCount );
    std::unique_ptr< void, decltype( &free ) > canSkipCompileMemory( malloc( memorySize ), free );
    nn::util::BitArray canSkipCompile( canSkipCompileMemory.get(), memorySize, batchCount );
    canSkipCompile.reset();

    auto version = m_NvnGlslcDll.GlslcGetVersion();
    int variationsPerBatch = static_cast< int >( variations.size() ) / batchCount;
    int restVariations = static_cast< int >( variations.size() ) - variationsPerBatch * batchCount;
    for( int idxBatch = 0; idxBatch < batchCount; ++idxBatch ) // TODO: ここも並列化可能だがボトルネックでない
    {
        int startVariation = idxBatch * variationsPerBatch +
            std::min NN_PREVENT_MACRO_FUNC( idxBatch, restVariations );
        int variationCount = variationsPerBatch + ( idxBatch < restVariations ? 1 : 0 );
        specializationBatches[ idxBatch ].Initialize( pArg, variations, startVariation, variationCount );
        canSkipCompile.set( idxBatch, false );
        auto cacheOutputs = ReadShaderCache( pArg, version, glslcCompileObject,
            isBatchCompileMode ? specializationBatches[ idxBatch ].GetGlslcSpecializationBatch() : nullptr );
        if( static_cast< int >( cacheOutputs.size() ) == variationCount )
        {
            for( int idxOutput = 0; idxOutput < variationCount; ++idxOutput )
            {
                RetrieveVariationOutput( pOutput->GetVariationOutput(
                    variations.at( startVariation + idxOutput ) ),
                    *cacheOutputs[ idxOutput ], nn::util::string_view( nullptr, 0 ) );
            }
            canSkipCompile.set( idxBatch, true );
        }
    }

    if( canSkipCompile.all() )
    {
        return;
    }

    // 以下コンパイル
    bool result = isBatchCompileMode ?
        m_NvnGlslcDll.GlslcCompilePreSpecialized( &glslcCompileObject ) :
        m_NvnGlslcDll.GlslcCompile( &glslcCompileObject ) != 0;
    auto pResults = glslcCompileObject.lastCompiledResults;
    if( pResults == nullptr || pResults->compilationStatus == nullptr )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "Compile error in NvnGlslc" );
    }

    auto pStatus = pResults->compilationStatus;
    if( pStatus->allocError )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_Failed, "Allocation error in NvnGlslc." );
    }

    if( !result || !pStatus->success )
    {
        // TODO
        static const char* s_StageNames[] =
        {
            "----Vertex Shader----",
            "----Pixel Shader----",
            "----Geometry Shader----",
            "----Hull Shader----",
            "----Domain Shader----",
            "----Compute Shader----"
        };
        Custom< std::string >::Type source;
        if( glslcCompileObject.options.optionFlags.language == GLSLC_LANGUAGE_GLSL )
        {
            for( int idxSource = 0; idxSource < static_cast<int>(
                glslcCompileObject.input.count ); ++idxSource )
            {
                source.append( "\n" ).append( s_StageNames[ glslcCompileObject.input.stages[
                    idxSource ] ] ).append( "\n" ).append( ConvertEncoding( AddLineNumber(
                    nn::util::string_view( glslcCompileObject.input.sources[ idxSource ] ) ),
                    pCommonOption->GetCodePage(), 0 ) );
            }
        }
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError,
            "\n[Variation: %d]\n%s\n----Error Log----\n%s",
            variations.at( 0 ), source.c_str(), pStatus->infoLog );
    }

    if( pStatus->infoLog )
    {
        commonInfoLog.append( pStatus->infoLog, pStatus->infoLogLength );
        NN_GFXTOOL_PRINT_WARNING( "Warning [Variation: %d]: %s\n",
            variations.at( 0 ), commonInfoLog.c_str() );
    }

    auto BatchCompile = [ & ]( int idxBatch )
    {
        if( threadCount > 1 )
        {
            _set_se_translator( []( unsigned int, struct _EXCEPTION_POINTERS* ep )
            {
                throw ep;
            } );
        }
        if( canSkipCompile.test( idxBatch ) )
        {
            return;
        }
        const GLSLCresults * const * ppResults = nullptr;
        if( m_NvnGlslcDll.GlslcCompileSpecializedMT )
        {
            ppResults = m_NvnGlslcDll.GlslcCompileSpecializedMT( &glslcCompileObject,
                specializationBatches[ idxBatch ].GetGlslcSpecializationBatch() );
        }
        else
        {
            ppResults = m_NvnGlslcDll.GlslcCompileSpecializedSt( &glslcCompileObject,
                specializationBatches[ idxBatch ].GetGlslcSpecializationBatch() );
        }
        if( ppResults == nullptr )
        {
            NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "Compile error in NvnGlslc" );
        }
        int startVariation = idxBatch * variationsPerBatch +
            std::min NN_PREVENT_MACRO_FUNC( idxBatch, restVariations );
        int variationCount = variationsPerBatch + ( idxBatch < restVariations ? 1 : 0 );
        for( int idxOutput = 0; idxOutput < variationCount; ++idxOutput )
        {
            auto pResult = ppResults[ idxOutput ];
            auto pStatus = pResult->compilationStatus;
            if( pStatus->allocError )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_Failed, "Allocation error in NvnGlslc." );
            }
            if( !pStatus->success )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_CompileError, "%s", pStatus->infoLog );
            }
            if( pResult->compilationStatus->infoLog )
            {
                Custom< std::string >::Type log( pResult->compilationStatus->infoLog,
                    pResult->compilationStatus->infoLogLength );
                NN_GFXTOOL_PRINT_WARNING( "Warning [Variation: %d]: %s\n",
                    startVariation + idxOutput, log.c_str() );
            }
            WriteShaderCache( pArg, version, glslcCompileObject, isBatchCompileMode ?
                specializationBatches[ idxBatch ].GetGlslcSpecializationBatch() : nullptr, ppResults );
            RetrieveVariationOutput( pOutput->GetVariationOutput(
                variations.at( startVariation + idxOutput ) ), *pResult->glslcOutput,
                nn::util::string_view( pResult->compilationStatus->infoLog,
                pResult->compilationStatus->infoLogLength ) );
        }
    };

    int batchesPerThread = batchCount / threadCount;
    int restBatches = batchCount - batchesPerThread * threadCount;
    auto BatchCompileThread = [ & ]( int idxThread )
    {
        int startBatch = idxThread * batchesPerThread +
            std::min NN_PREVENT_MACRO_FUNC( idxThread, restBatches );
        int batchCount = batchesPerThread + ( idxThread < restVariations ? 1 : 0 );
        for( int idxBatch = startBatch; idxBatch < startBatch + batchCount; ++idxBatch )
        {
            BatchCompile( idxBatch );
        }
    };

    if( isBatchCompileMode )
    {
        if( threadCount <= 1 )
        {
            BatchCompileThread( 0 );
        }
        else
        {
            Custom< std::vector< std::future< void > > >::Type futures;
            futures.reserve( threadCount );
            for( int idxThread = 0; idxThread < threadCount; ++idxThread )
            {
                futures.push_back( std::async( std::launch::async, BatchCompileThread, idxThread ) );
            }
            for( auto&& future : futures )
            {
                future.get();
            }
        }
    }
    else
    {
        WriteShaderCache( pArg, version, glslcCompileObject, nullptr, nullptr );
        RetrieveVariationOutput( pOutput->GetVariationOutput( variations.at( 0 ) ),
            *glslcCompileObject.lastCompiledResults->glslcOutput, nn::util::string_view() );
    }
} // NOLINT

void Compiler< Target >::PostCompile( CompileOutput*, const ShaderCompilerContext*,
    const nngfxToolShaderCompilerCompileArg* )
{
    for( int idxCompileObject = 0; idxCompileObject < GetThreadCount(); ++idxCompileObject )
    {
        m_CompileObjects[ idxCompileObject ].Finalize( m_NvnGlslcDll );
    }
}

}
}

NN_STATIC_ASSERT( sizeof( nngfxToolShaderCompilerCompileOptionNvn ) == 72 );
NN_STATIC_ASSERT( sizeof( nngfxToolShaderCompilerOptionOutputStageNvn ) == 40 );
NN_STATIC_ASSERT( sizeof( nngfxToolShaderCompilerOptionOutputProgramNvn ) == 80 );
