﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <atomic>

#include <gfxTool_Common.h>

namespace nn {
namespace gfxTool {

class ShaderCompilerContext;
class CompileOutput;

template< nngfxToolShaderCompilerLowLevelApiType ApiType,
    nngfxToolShaderCompilerCodeType CodeType >
struct CompilerVariation
{
    static const nngfxToolShaderCompilerLowLevelApiType ApiType = ApiType;
    static const nngfxToolShaderCompilerCodeType CodeType = CodeType;
};

template< typename T >
class Compiler;

class CompilerBase
{
public:
    virtual ~CompilerBase()
    {
    }

    virtual int GetGroupCount( const ShaderCompilerContext* pContext,
        const nngfxToolShaderCompilerCompileArg* pArg ) const = 0;

    virtual void PreCompile( CompileOutput* pOutput, const ShaderCompilerContext* pContext,
        const nngfxToolShaderCompilerCompileArg* pArg )
    {
        NN_UNUSED( pOutput );
        NN_UNUSED( pContext );
        NN_UNUSED( pArg );
    }

    virtual void CompileGroup( CompileOutput* pOutput, const ShaderCompilerContext* pContext,
        const nngfxToolShaderCompilerCompileArg* pArg, int idxGroup ) = 0;

    virtual void Compile( CompileOutput* pOutput, const ShaderCompilerContext* pContext,
        const nngfxToolShaderCompilerCompileArg* pArg );

    virtual void PostCompile( CompileOutput* pOutput, const ShaderCompilerContext* pContext,
        const nngfxToolShaderCompilerCompileArg* pArg )
    {
        NN_UNUSED( pOutput );
        NN_UNUSED( pContext );
        NN_UNUSED( pArg );
    }

    virtual void AdjustOutput( CompileOutput* pOutput ) const
    {
        NN_UNUSED( pOutput );
    }

    int GetThreadCount() const
    {
        return m_MaxThreads;
    }

    // PreCompile 内から設定可能。並列実行を許可しない場合は 1 に設定。
    void SetThreadCount( int value )
    {
        m_MaxThreads = value;
    }

    void SetGroupCount( int value )
    {
        m_GroupCount = value;
    }

    int GetThreadIndex( int idxGroup )
    {
        return idxGroup % m_MaxThreads;
    }

protected:
    virtual void IncrementCompiledCount();

    virtual void PrintProgress( int count );

    std::atomic< int > m_CompiledCount;
    int m_GroupCount;
    float m_PrintProgressRateInv;

    int m_MaxThreads;
};

}
}
