﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <conio.h>
#include <unordered_map>
#include <filesystem>
#include <fstream>
#include <Windows.h>
#include <d3dcompiler.h>	//!< TODO: fix d3d compiler path.

#include <nn/gfxTool/gfxTool_Error.h>
#include <gfxTool_HlslCrossCompilerDll.h>
#include <gfxTool_HlslCrossCompilerOptionParser.h>
#include <gfxTool_HlslCrossCompilerManager.h>
#include <gfxTool_HlslCrossCompilerReflection.h>
#include <gfxTool_HlslCrossCompilerDxCompiler.h>

const char* GetHlslShaderCompilerDllName()
{
    return
#if defined(_M_IX86)
    "x86/hlslcc.dll";
#else
    "hlslcc.dll";
#endif
}

namespace
{
const uint32_t HlslCrossCompilerVersion_Major = 1;
const uint32_t HlslCrossCompilerVersion_Minor = 20;
const uint32_t HlslCrossCompilerVersion_Micro = 0;

nn::gfxTool::HlslShaderCompilerOptionParser g_CommandLineOptionParser;
nn::gfxTool::HlslShaderCompilerManager g_HlslShaderCompilerManager;

void Execute( int argc, char** argv )
{
    nn::gfxTool::HlslCrossCompilerDll hlslCrossCompilerDll;
    auto path = std::tr2::sys::path( nn::gfxTool::GetModulePath( GetModuleHandle( nullptr ) ) ).parent_path().string();
    path.append( "/" ).append( GetHlslShaderCompilerDllName() );
    if(!hlslCrossCompilerDll.Initialize(path.c_str()))
    {
        NN_GFXTOOL_THROW_MSG(nngfxToolResultCode_DllNotFound, "Dll not found : %s\n", path.c_str());
    }

    // Parse options.
    g_CommandLineOptionParser.SetOptions( argc - 1, &argv[1] );
    g_CommandLineOptionParser.Parse();

    // Display help.
    if( g_CommandLineOptionParser.GetCommandLineOptionDefinitions().help.IsExisting() )
    {
        const char* name = "HlslCrossCmpiler";
        const char* pDescription = "HlslCrossCompiler.";
        g_CommandLineOptionParser.PrintHelp( name, pDescription );
    }

    if( g_CommandLineOptionParser.GetCommandLineOptionDefinitions().version.IsExisting() )
    {
        NN_GFXTOOL_PRINT( "HlslCrossCompiler.exe version: %d.%d.%d\n",
                            HlslCrossCompilerVersion_Major,
                            HlslCrossCompilerVersion_Minor,
                            HlslCrossCompilerVersion_Micro );
    }

    if( g_CommandLineOptionParser.GetCommandLineOptionDefinitions().silent.IsExisting() )
    {
        nn::gfxTool::Logger::SetLogOutputLevel( nn::gfxTool::Logger::LogType::Default, nn::gfxTool::Logger::LogLevel::Silent );
    }

    unsigned int hlslCompilerOptionFlags = D3DCOMPILE_ALL_RESOURCES_BOUND;
    std::unordered_map<std::string, std::string> shaderMacroArray;	// name, value
    std::vector<D3D_SHADER_MACRO> shaderMacroPtrArray;

    // DxCompiler : Setup compile option
    {
        auto options	= g_CommandLineOptionParser.GetCommandLineOptionDefinitions();
        if( options.hlslCompileOptionFlags.IsExisting() )
        {
            auto& flags = options.hlslCompileOptionFlags.GetValue();
            for( auto flag : flags )
            {
                hlslCompilerOptionFlags |= std::stoi( flag, nullptr, 0 );
            }
        }

        if( options.preprocessorDefinitions.IsExisting() )
        {
            auto& macros = options.preprocessorDefinitions.GetValue();
            for( auto& macro : macros )
            {
                nn::gfxTool::AddMacro( shaderMacroArray, macro );
            }
        }

        for( auto macro = shaderMacroArray.begin(); macro != shaderMacroArray.end(); ++macro )
        {
            D3D_SHADER_MACRO d3dShaderMacro{};
            d3dShaderMacro.Name = macro->first.c_str();
            d3dShaderMacro.Definition = macro->second.c_str();
            shaderMacroPtrArray.push_back( d3dShaderMacro );
        }
        D3D_SHADER_MACRO d3dShaderMacro = {};
        d3dShaderMacro.Name = nullptr;
        d3dShaderMacro.Definition = nullptr;
        shaderMacroPtrArray.push_back( d3dShaderMacro );

        if( options.includeDirectories.IsExisting() )
        {
            auto& includeDirectories = options.includeDirectories.GetValue();
            for( auto dir : includeDirectories )
            {
                nn::gfxTool::AddIncludeDirectory( dir );
            }
        }
    }

    // Compile to dxbc.
    std::vector<std::string> inputFileArray;
    std::vector<ID3DBlob*> blobPtrArray;
    {
        auto inputArgs	= g_CommandLineOptionParser.GetInputArgs();
        auto options	= g_CommandLineOptionParser.GetCommandLineOptionDefinitions();

        const char* pEntryPoint = "main";
        if( options.entryPoint.IsExisting() )
        {
            pEntryPoint = options.entryPoint.GetValue().c_str();
        }

        for( auto inputFile = inputArgs.begin(); inputFile != inputArgs.end(); ++inputFile )
        {
            char fullPath[_MAX_PATH];
            _fullpath(fullPath, inputFile->c_str(), _MAX_PATH);
            inputFileArray.push_back( std::string( fullPath ) );

            auto& file = inputFileArray.back();
            std::replace( file.begin(), file.end(), L'\\', L'/' );

            // adds include directory where a source file exists
            std::string filePath( file.begin(), file.begin() + file.find_last_of( '/' ) );
            nn::gfxTool::SetInitialSourceDirectory( filePath );
            ID3DBlob* pBlob = nullptr;
            nn::gfxTool::CompileShader( file.c_str(),
                                        pEntryPoint,
                                        options.shaderModel.GetValue().c_str(),
                                        &pBlob,
                                        hlslCompilerOptionFlags,
                                        &shaderMacroPtrArray[0] );
            blobPtrArray.push_back( pBlob );

        }
    }

    // HlslccManager : Setup
    HLSLccOptions hlslccOptions;
    unsigned int hlslccOptionFlags = HLSLCC_FLAG_TRANSLATE_MATRICES | HLSLCC_FLAG_UNIFORM_BUFFER_OBJECT;	// TODO: Cannot convert cbuffer properly w/o supporting ubo in glsl.
    {
        auto options	= g_CommandLineOptionParser.GetCommandLineOptionDefinitions();
        if( options.hlslccOptionFlags.IsExisting() )
        {
            auto& flags = options.hlslccOptionFlags.GetValue();
            for( auto flag : flags )
            {
                hlslccOptionFlags |= std::stoul( flag, nullptr, 0 );
            }
        }
        auto& sm = options.shaderModel.GetValue();
        if( sm.find( "gs" ) != std::string::npos )
        {
            hlslccOptionFlags |= HLSLCC_FLAG_GS_ENABLED;
        }
        if( ( sm.find( "hs" ) != std::string::npos ) || ( sm.find( "ds" ) != std::string::npos ) )
        {
            hlslccOptionFlags |= HLSLCC_FLAG_TESS_ENABLED;
        }

        // HLSLccOptions
        if(options.systemBufferBinding.IsExisting())
        {
            hlslccOptions.systemBufferBinding.Set(std::stoul( options.systemBufferBinding.GetValue(), nullptr, 0 ));
        }
        if(options.shiftPatchLocation.IsExisting())
        {
            hlslccOptions.shiftPatchLocation.Set(std::stoul( options.shiftPatchLocation.GetValue(), nullptr, 0 ));
        }
    }

    // HlslccManager : Convert
    std::vector<GLSLShader> convertedShaderArray;
    std::vector<nn::gfxTool::HlslShaderCompilerReflection> reflectionArray;
    convertedShaderArray.resize( inputFileArray.size() );
    reflectionArray.resize( inputFileArray.size() );
    {
        int shaderIdx = 0;
        for( auto& pBlob : blobPtrArray )
        {
            const char* pDxbc = reinterpret_cast<const char *>(pBlob->GetBufferPointer() );
            GLLang lanuage = LANG_DEFAULT; //LANG_440;
            GlExtensions extensions = {
                1, // ARB_explicit_attrib_location
                1, // ARB_explicit_uniform_location
                1, // ARB_shading_language_420pack
            };
            GLSLCrossDependencyData* pDependencies = nullptr;
            HLSLccSamplerPrecisionInfo samplerPrecisions;		//!< If needed, pass the precision of each sampler in hlsl.
            HLSLccReflection& reflectionCallback = reflectionArray[shaderIdx];
            GLSLShader& convertedShader = convertedShaderArray[shaderIdx];
            int result = 0;
            result = hlslCrossCompilerDll.TranslateHlslFromMemEx(pDxbc,
                                                                 hlslccOptionFlags,
                                                                 lanuage,
                                                                 &extensions,
                                                                 pDependencies,
                                                                 samplerPrecisions,
                                                                 reflectionCallback,
                                                                 &convertedShader,
                                                                 &hlslccOptions);
            if( result != 1 )
            {
                NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_Failed, "TranslateHLSL failed : %s\n", inputFileArray[shaderIdx].c_str() );
            }
            ++shaderIdx;
        }
    }

    // Output to file.
    {
        int shaderIdx = 0;
        std::string outputOptFile;
        auto options	= g_CommandLineOptionParser.GetCommandLineOptionDefinitions();
        if( options.output.IsExisting() )
        {
            char fullPath[_MAX_PATH];
            _fullpath( fullPath, options.output.GetValue().c_str(), _MAX_PATH );
            outputOptFile.assign( fullPath );
            std::replace( outputOptFile.begin(), outputOptFile.end(), L'\\', L'/' );
        }
        else
        {
            outputOptFile.assign( "" );
        }

        for( auto& convertedShader : convertedShaderArray )
        {
            std::string outputFile;
            auto& inputFile = inputFileArray[shaderIdx];
            auto offsetOfLastSlash = inputFile.find_last_of( "/" );
            NN_SDK_ASSERT( offsetOfLastSlash != std::string::npos, "Invalid input file name: %s", inputFile.c_str() );
            auto fileNameBegin = inputFile.begin() + offsetOfLastSlash + 1;
            if( outputOptFile == "" )	// when output file isn't specified output to same directory of input file.
            {
                outputFile.assign( inputFile.begin(), fileNameBegin );
                auto offsetOfLastPeriod = inputFile.rfind( ".", inputFile.length() );
                if( ( offsetOfLastPeriod == std::string::npos ) || ( offsetOfLastPeriod < offsetOfLastSlash ) )
                {
                    offsetOfLastPeriod = inputFile.length();
                }
                auto fileNameEnd = inputFile.begin() + offsetOfLastPeriod;
                outputFile.append( fileNameBegin, fileNameEnd );
                outputFile.append( ".glsl" );
            }
            else
            {
                outputFile = outputOptFile;
            }

            std::ofstream outFile( outputFile.c_str() );
            outFile << convertedShader.sourceCode;
            outFile.close();
            NN_GFXTOOL_PRINT( "Converted: %s\n", outputFile.c_str() );
#if 0
            if( options.reflection.IsExisting() )
            {
                std::string dumpReflection;
                reflectionArray[shaderIdx].DumpReflection( dumpReflection );

                std::string reflectionFile = outputOptFile;
                reflectionFile.append( fileNameBegin, fileNameEnd );
                reflectionFile.append( "_ref.txt" );
                outFile.open( reflectionFile.c_str() );
                outFile << dumpReflection;
                outFile.close();
            }
#endif

            ++shaderIdx;
        }
    }

    // Clean up.
    for( auto& pBlob : blobPtrArray )
    {
        if( pBlob != nullptr )
        {
            pBlob->Release();
            pBlob = nullptr;
        }
    }
} // NOLINT

}

int main( int argc, char** argv )
{
    nn::gfxTool::Logger::SetDefaultStream();
    try
    {
        Execute( argc, argv );
    }
    catch( ... )
    {
        return nn::gfxTool::ReportException();
    }

    return nngfxToolResultCode_Succeeded;
}
