﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <System.h>
#include <FrameBuffer.h>
#include <iostream>
#include <fstream>
#include <Viewer.h>


nw::ut::MemoryAllocator         gNwAllocator;
nw::eftdemo::System*            gSystem             = NULL;
nw::eftdemo::FrameBuffer        gFrameBuffer64;
nw::eftdemo::FrameBufferTexture gFrameBufferCopy;
nw::eftdemo::DepthBufferTexture gDepthBufferCopy;
nw::dev::PrimitiveRenderer*     gPrimitiveRenderer  = NULL;
nw::math::VEC3                  gCameraPosition;
nw::math::VEC3                  gCameraLookAt;
nw::math::Matrix34              gView;
nw::math::MTX44                 gProjectionMatrix;
nw::eftdemo::DrawParam          gDrawParam;
nw::eftdemo::Camera             gDemoCamera;


//------------------------------------------------------------------------------
//  初期化処理
//------------------------------------------------------------------------------
void Initialize()
{
    // ファイルシステムの初期化
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    nw::dev::FileDeviceManager::CreateArg fileDeviceArg;
    fileDeviceArg.allocator = &gNwAllocator;
    fileSystem->Initialize( fileDeviceArg );

    // デモシステムの作成
    nw::eftdemo::System::CreateArg    arg;

    arg.allocator   = &gNwAllocator;
    arg.width       = 640;
    arg.height      = 480;
    arg.drawMeter   = true;
    arg.fontPath    = "common/fonts/nintendo_NTLG-DB_002_Nw4f.bffnt";
    arg.clearColor  = 0x333333FF;
    arg.waitVBlank  = 0;

#if defined(NW_PLATFORM_CAFE)
    arg.width          = 1280;
    arg.height         = 720;
    arg.fontPath    = "common/fonts/nintendo_NTLG-DB_002_Nw4f.bffnt";
    arg.waitVBlank     = 1;
    arg.fontShaderPath = "common/shaders/font_BuildinShader.gsh";
#endif

    // コンフィグファイルをロードして再設定
    nw::eftdemo::System::LoadConfigFile(&gNwAllocator, &arg);

    gSystem = new nw::eftdemo::System( arg );

    // デモシステムの初期化
    gSystem->Initialize();

    // 入力インターフェースの初期化
    gSystem->InitializeInputInterface();

    // グラフィックシステムの初期化
    gSystem->InitializeGraphicsSystem();

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        // PrimitiveRenderer の初期化
        gPrimitiveRenderer = nw::dev::PrimitiveRenderer::GetInstance();

#if defined(NW_PLATFORM_WIN32)
        gPrimitiveRenderer->Initialize( &gNwAllocator );
#endif
#if defined(NW_PLATFORM_CAFE)
        // PrimitiveRenderer で用いるシェーダーバイナリへのパスを指定する。
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "common/shaders/dev_PrimitiveRenderer.gsh";
        loadArg.allocator = &gNwAllocator;
        u8* binary = fileSystem->Load( loadArg );
        gPrimitiveRenderer->InitializeFromBinary( &gNwAllocator, binary, loadArg.readSize );
        fileSystem->Unload( loadArg, binary );
#endif

    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    // カメラを初期化
    gCameraPosition.Set( 0.0f, 50.0f, 150.0f );
    gCameraLookAt.Set( 0.0f, 00.0f, 0.0f );
    gDemoCamera.SetPos( gCameraPosition );
    gDemoCamera.SetLookAtPos( gCameraLookAt );
    gDemoCamera.Preset();
}

//------------------------------------------------------------------------------
//  終了処理
//------------------------------------------------------------------------------
void Finalize()
{
    // PrimitiveRenderer終了処理
    gPrimitiveRenderer->Finalize( &gNwAllocator );

    // デモシステム終了処理
    gSystem->FinalizeGraphicsSystem();
    gSystem->FinalizeInputInterface();
    gSystem->Finalize();

    // ファイルシステム終了処理
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    fileSystem->Finalize();
}

//------------------------------------------------------------------------------
//  Main 関数
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
int PCSDKMain(int argc, char **argv)
#elif defined(NW_PLATFORM_CAFE)
int main(int argc, char **argv)
#endif
{
    NW_UNUSED_VARIABLE( argc );
    NW_UNUSED_VARIABLE( argv );

#if ( EFT_IS_WIN && _DEBUG )
    _CrtSetDbgFlag(_CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF);
#endif

    // アロケーターの初期化
    #define HEAP_SIZE ( 512 * 1024 * 1024 )
    void* addr = malloc( HEAP_SIZE );
    gNwAllocator.Initialize( addr, HEAP_SIZE );

    // システム初期化処理
    Initialize();

    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();

    // フレームバッファの初期化
    graphics->LockDrawContext();
    {
        gFrameBuffer64.Initialize( &gNwAllocator,
                                   gSystem->GetWidth(),
                                   gSystem->GetHeight(),
                                   nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_FLOAT16 );

        gFrameBufferCopy.Initialize( &gNwAllocator, &gFrameBuffer64, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_RGB16 );
        gDepthBufferCopy.Initialize( &gNwAllocator, &gFrameBuffer64 );
    }
    graphics->UnlockDrawContext();

    // ビューを初期化
    gCameraPosition.Set( 0.0f, 30.0f, 50.0f );
    gCameraLookAt.Set  ( 0.0f, 00.0f, 0.0f );
    nw::math::VEC3 camUp ( 0.0f, 1.0f, 0.0f );
    nw::math::VEC3 target( 0.0f, 0.0f, 0.0f);
    gView.Identity();
    gView.SetLookAt( gCameraPosition, camUp, target );

    // プロジェクションの初期化
    nw::math::MTX44Perspective( &gProjectionMatrix, nw::math::F_PI/3, gSystem->GetWidth()/(f32)gSystem->GetHeight(), 0.1f, 1000.f);

    // ビューアクラスの初期化
    Viewer viewer;
    graphics->LockDrawContext();
    {
        viewer.Initialize( &gNwAllocator, gSystem->GetWidth(), gSystem->GetHeight() );
        viewer.SetCamera( &gDemoCamera );
        viewer.SetProjectionMatrix( &gProjectionMatrix );
    }
    graphics->UnlockDrawContext();


    // メインループ
    while ( !gSystem->IsExiting() )
    {
        // システム デモパッドの更新
        gSystem->UpdateInputInterface();

        // カメラの更新
#if defined( NW_PLATFORM_CAFE )
        gDemoCamera.UpdateCamera( gSystem->GetPad(), NULL, true, true );
#endif
#if defined( NW_PLATFORM_WIN32 )
        gDemoCamera.UpdateCamera( gSystem->GetPad(), gSystem->GetMouse(), gSystem->GetMouse()->IsPointerOn(), true );
#endif
        gDemoCamera.GetPos( &gCameraPosition );
        gDemoCamera.GetLookAtPos( &gCameraLookAt );

#if defined( NW_PLATFORM_WIN32 )
        {
            f32 frameRate = 60.0f;
            viewer.GetFrameRate(&frameRate);
            gSystem->SetFps(frameRate);
        }
#elif defined( NW_PLATFORM_CAFE )
        {
            gSystem->SetWaitVBlankInterval(viewer.GetSwapInterval());
        }
#endif

        // リニア <-> ノンリニアの切り替え
        {
            static bool linearModeFlag = false;
            if ( viewer.GetLinearMode() != linearModeFlag )
            {
                gSystem->SetScanBuffer( viewer.GetLinearMode() );
                linearModeFlag = viewer.GetLinearMode();
            }
        }

        graphics->LockDrawContext();
        {
            // 書き込み先フレームバッファの選定
            gFrameBuffer64.Bind();
            gFrameBuffer64.SetClearColor( viewer.GetBackgroundColor() );
            gFrameBuffer64.Clear( true, true );

            // 描画パラメータ
            gDrawParam.mProjMtx           = gProjectionMatrix;
            gDrawParam.mViewMtx           = gDemoCamera.GetMatrix();
            gDrawParam.mCameraPos         = gCameraPosition;
            gDrawParam.mNearZ             = 0.1f;
            gDrawParam.mFarZ              = 10000.f;
            gDrawParam.mPrimitiveRenderer = gPrimitiveRenderer;
            gDrawParam.mShadowMap         = 0;
            gDrawParam.mGpuProfiler       = NULL; //nw::eftdemo::PerfAnalyzerCallBack;
            gDrawParam.mBaseFovy          = NW_MATH_DEG_TO_RAD( 60.0f ); // EffectMaker の標準の画角を EffectViewer の基準画角とする
            gDrawParam.mCurrentFovy       = viewer.GetFovy();

            // ビューアの計算処理,描画処理
            viewer.ProcCalc( gSystem->GetPad(), gDrawParam );
            viewer.ProcDraw( gDrawParam, &gFrameBuffer64, &gFrameBufferCopy, &gDepthBufferCopy );

            // デバッグ文字描画
            if ( viewer.DoDebugDraw() )
            {
#ifdef EFT_OGL
                glEnable(  GL_BLEND );
                glBlendFunc( GL_SRC_ALPHA , GL_ONE );
                glBlendEquationEXT(GL_FUNC_ADD_EXT);
                glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
#endif
                gSystem->GetTextWriter()->SetCursor( 0, 0 );
                viewer.DebugDraw( gSystem->GetTextWriter() );
                gSystem->GetTextWriter()->Flush();
            }
        }
        graphics->UnlockDrawContext();

        // バッファのスワップ
        gSystem->SwapBuffer( &gFrameBuffer64 );

        gSystem->EndFrame();

        //  VBlank 待ち
        gSystem->WaitForFlip();
    }


    graphics->LockDrawContext();
    {
        // ビューアの終了処理
        viewer.Finalize();

        // フレームワークの終了処理
        Finalize();
    }
    graphics->UnlockDrawContext();

    // アロケータ終了処理
    gNwAllocator.Finalize();
    free( addr );

    return 0;
}

