﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev.h>
#include <ModelUtility.h>
#include <G3dModelPreview.h>
#include <Viewer.h>
#include <Grid.h>
#include <System.h>
#include <PvrTextureTranslationGL.h>
#include <EffectBinaryPlayer.h>
#include <BackGroundTexture.h>
#include <ScreenDrawer.h>
#include <GlareEffect.h>

#include <nw/eft/eft2_System.h>
#include <nw/eft/eftvw2_ViewerSystem.h>
#include <nw/eft/eftvw2_FileSystem.h>
#if EFT_IS_WIN
#include <fstream>
#endif

#include <cafe/gx2/gx2Enum.h>
#include <nw/eft/eftut2_Heap.h>

nw::eftut2::NwHeap              g_Heap;                                 // エフェクトヒープ
nw::eftut2::NwHeap              g_ConnectionHeap;                       // EffectMakerとの通信用ヒープ
void*                           g_ConnectionHeapBuffer      = NULL;     // EffectMakerとの通信用ヒープ先メモリ
nw::eft2::System*               g_EftSystem                 = NULL;     // エフェクトシステム
nw::eftvw2::ViewerSystem*       g_EffectViewer              = NULL;     // エフェクトビューア
EffectBinaryPlayer*             g_EffectBinPlayer           = NULL;     // エフェクトバイナリプレイヤー
BackGroundTexture*              g_BackGroundTexture         = NULL;
nw::eftdemo::ScreenDrawer       g_ScreenDrawer;
nw::eftdemo::GlareEffect        g_Glare;


//------------------------------------------------------------------------------
//      ビューアデータの処理
//------------------------------------------------------------------------------
static void HandleViewerData( nw::eft2::Heap* eftHeap, void* pThis, void* pBin )
{
    NW_UNUSED_VARIABLE( eftHeap );

    Viewer* eftDemo = (Viewer*)pThis;

    nw::eft2::BinaryData* pNodeBinary = static_cast<nw::eft2::BinaryData*>( pBin );
    nw::eft2::BinaryData* pTextureBinary = pNodeBinary->GetNextData();

    if ( pNodeBinary->GetBinaryTag() == EFT_MAKE_TAG( 'V', 'I', 'W', 'D' ) )
    {
        nw::eft2::BinaryData* pNodeBinary = static_cast<nw::eft2::BinaryData*>( pBin );
        nw::eftvw2::ResCamera* camera = pNodeBinary->GetBinaryDataWithFlip<nw::eftvw2::ResCamera>();

        // 基本設定
        eftDemo->SetPlaySpeed( camera->frameRate );
        eftDemo->SetGridType( camera->gridType );
        eftDemo->SetLinearMode( camera->isLinearMode != 0 );

        // カメラ設定
        eftDemo->SetCameraSettings(
            camera->position.x, camera->position.y, camera->position.z,
            camera->target.x,   camera->target.y,   camera->target.z,
            camera->rotation,
            camera->projectionType,
            camera->aspectRatio,
            camera->fieldOfView,
            camera->imageHeight,
            camera->nearClip,
            camera->farClip );

        // 背景カラー
        eftDemo->SetBackgroundColor( camera->bgColor );

        // リアルタイム編集だったらここで終了
        if (camera->isRealtimeEditing == 1) return;

        //-----------------------------------------
        // 背景テクスチャ
        //-----------------------------------------
        nw::eft2::ResTexture* resTexture = pTextureBinary->GetBinaryDataWithFlip<nw::eft2::ResTexture>();

        void*   texBuffer          = ((char*)pTextureBinary->GetBinaryData()) + sizeof( nw::eft2::ResTexture );
        u32     texBufferSize      = pTextureBinary->GetBinarySize() - sizeof( nw::eft2::ResTexture );

        g_BackGroundTexture->Setup( 0, resTexture, texBuffer, texBufferSize );
        eftDemo->SetBackgroundImageRes( reinterpret_cast<const void*>( g_BackGroundTexture->GetTextureResource() ) );


        //-----------------------------------------
        // エフェクトバイナリ再生機能
        //-----------------------------------------
        nw::eft2::BinaryData* pSubBinary = pNodeBinary->GetSubData();
        if (pSubBinary != NULL)
        {
            // バイナリ転送対応
            if ( pSubBinary->GetBinaryTag() == EFT_MAKE_TAG( 'P', 'T', 'C', 'L' ) )
            {
                nw::eftvw2::ResPtclBinary* resPtcl = pSubBinary->GetBinaryDataWithFlip<nw::eftvw2::ResPtclBinary>();

                // 強制リロードのOn/Offを取得
                static int currentReloadCtrl = 0;
                bool forceReload = resPtcl->reloadCtrl != currentReloadCtrl;
                currentReloadCtrl = resPtcl->reloadCtrl;

                // エフェクトバイナリプレイヤークラスにパスを登録
                bool resSetup = g_EffectBinPlayer->SetupEffectBinary( reinterpret_cast<const char*>( resPtcl->filePath), forceReload );

                // 再生するエミッタセットのインデックス送り/戻し処理
                if ( !resSetup )
                {
                    static int currentIndexCtrl = 0;
                    int subIndex = resPtcl->esetIndexCtrl - currentIndexCtrl;
                    currentIndexCtrl = resPtcl->esetIndexCtrl;

                    if (subIndex > 0)
                    {
                        // インデックス送り
                        g_EffectBinPlayer->Send();
                    }
                    else if (subIndex < 0)
                    {
                        // インデックス戻し
                        g_EffectBinPlayer->Prev();
                    }
                }
            }
        }
    }
}

//------------------------------------------------------------------------------
//      ビューアメッセージの処理
//------------------------------------------------------------------------------
static void HandleViewerMessage( nw::eft2::Heap* eftHeap, void* pThis, void* pBin )
{
    NW_UNUSED_VARIABLE( eftHeap );

    Viewer* eftDemo = (Viewer*)pThis;

    struct Config
    {
        s32             frameRate;          //!< フレームレート
        s32             resolution;         //!< 解像度
        nw::math::VEC4  clearColor;         //!< 背景カラー
        f32             worldScaleRange;    //!< ワールドスケール範囲
        s32             worldScaleTime;     //!< ワールドスケール時間
        f32             gridScale;          //!< グリッドスケール
        s32             isLinearEditMode;   //!< リニア編集モード
    };

    Config* config = (Config *)pBin;

    eftDemo->SetFrameRate( (nw::eftdemo::INIT_SETTINGS_FRAME_RATE)config->frameRate );

    u32 resolution = 0;
    switch ( config->resolution )
    {
        case 0: resolution = 2; break;
        case 1: resolution = 1; break;
        case 2: resolution = 0; break;
    }

    // フレームレート / 解像度 / 背景色 をファイルとして出力
    nw::eftdemo::System::SaveConfigFile(
        (nw::eftdemo::INIT_SETTINGS_FRAME_RATE)config->frameRate,
        (nw::eftdemo::INIT_SETTINGS_RESOLUTION)resolution,
        0.0f, 0.0f, 0.0f, 1.0f );
}


//------------------------------------------------------------------------------
//  モデルプレビュー計算処理
//------------------------------------------------------------------------------
void CalcModelPreviewBlock( const nw::math::Matrix44*           projMatrix,
                           const nw::math::Matrix34*           viewMatrix,
                           const nw::math::Vector4*            lightPos,
                           nw::eftdemo::SimpleShader::ViewId   id )
{
    // モデルのマトリクスブロック更新処理
    {
        nw::eftvw2::Preview* preview = g_EffectViewer->GetPreviewHead();
        while( preview )
        {
            if (preview->GetPreviewType() == nw::eftvw2::Preview::EFT_VWR_PRV_MODEL )
            {
                G3dModelPreview* modelPreview = reinterpret_cast<G3dModelPreview*>(preview);
                modelPreview->CalcBlock(
                    reinterpret_cast<const nw::g3d::math::Mtx34*>(viewMatrix),
                    reinterpret_cast<const nw::g3d::math::Mtx44*>(projMatrix),
                    reinterpret_cast<const nw::g3d::math::Vec4*>(lightPos), id );
            }
            preview = preview->GetNextPreview();
        }

        // 以下の関数は この関数の外で呼び出す必要があります。
        nw::g3d::CPUCache::Sync(); // CPU キャッシュ操作の完了を待ちます。

        // GPU のリードキャッシュはここで一括して Invalidate します。
        nw::g3d::GPUCache::InvalidateAll();
    }
}

//------------------------------------------------------------------------------
//  モデルプレビュー描画処理
//------------------------------------------------------------------------------
void DrawModelPreview( nw::eftdemo::SimpleShader::ViewId id )
{
    nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_BLOCK);

    nw::eftvw2::Preview* preview = g_EffectViewer->GetPreviewHead();
    while( preview )
    {
        if (preview->GetPreviewType() == nw::eftvw2::Preview::EFT_VWR_PRV_MODEL && preview->IsVisible() )
        {
            G3dModelPreview* modelPreview =  reinterpret_cast<G3dModelPreview*>(preview);
            modelPreview->Draw( id );
        }
        preview = preview->GetNextPreview();
    }

    nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_REGISTER);
}

//------------------------------------------------------------------------------
//      EffectViewerDemo 変数設定
//------------------------------------------------------------------------------
void Viewer::InitDemoValue(){}

//------------------------------------------------------------------------------
//      EffectViewerDemo 初期化処理
//------------------------------------------------------------------------------
void Viewer::Initialize( nw::ut::MemoryAllocator* allocator, s32 screenWidth, s32 screenHeight )
{
    NW_NULL_ASSERT( allocator );
    mDisplayWindow    = true;

    // Heap
    mNwAllocator = allocator;
    g_Heap.SetNwAllocator( mNwAllocator );

    // System 初期化
    nw::eft2::Config  config;
    config.SetEffectHeap( &g_Heap );
    config.SetEffectDynamicHeap( &g_Heap );
    config.SetEmitterNum( 1024 );
    config.SetStripeNum( 128 );
    config.SetEmitterSetNum( 256 );
    config.SetEmitterNum( 1024 );
    config.SetDoubleBufferSize( 1024 * 32 );
    config.SetStripeNum( 128 );
    config.SetUniformBlockGpuChacheFlush( true );
    g_EftSystem = new nw::eft2::System( config );

#ifdef EFT_OGL
    SetupPvrTextureTrasnlationGL( g_EftSystem );
#endif

    // エフェクト インゲーム ビューア 初期化
    // 通信用ヒープとして32M切り出す。
    g_ConnectionHeapBuffer = g_Heap.Alloc( 1024 * 1024 * 32 );
    g_ConnectionHeap.Initialize( g_ConnectionHeapBuffer, 1024 * 1024 * 32 );

    // Viewer 初期化
    g_EffectViewer = new nw::eftvw2::ViewerSystem( &g_Heap, &g_ConnectionHeap, g_EftSystem );

    // ViewerData用のコールバックを登録します。
    g_EffectViewer->SetViewerDataCallBack( this, HandleViewerData );

    // ViewerMessage用のコールバックを登録します。
    g_EffectViewer->SetViewerMessageCallBack( this, HandleViewerMessage );

    // モデル初期化
    nw::eftdemo::InitializeG3dShaderResource( allocator );
    g_EffectViewer->SetCreateModelPreviewCallback(  G3dModelPreview::CreateModelPreview  );
    g_EffectViewer->SetDestroyModelPreviewCallback( G3dModelPreview::DestroyModelPreview );

    // 通信OK
    g_EffectViewer->SetAppIsInitialized();

    // エフェクトバイナリプレイヤー
    g_EffectBinPlayer = new EffectBinaryPlayer( g_EftSystem, &g_Heap, 0, 0 );

    // 背景テクスチャ
    g_BackGroundTexture = new BackGroundTexture( &g_Heap );

    // 2D描画クラスの初期化
    g_ScreenDrawer.Initialize( mNwAllocator );

    // グレア用のバッファ
    g_Glare.Initialize( mNwAllocator, screenWidth, screenHeight );
}

//------------------------------------------------------------------------------
//      EffectViewerDemo 終了処理
//------------------------------------------------------------------------------
void Viewer::Finalize()
{
    delete g_BackGroundTexture;
    delete g_EffectBinPlayer;
    delete g_EffectViewer;
    delete g_EftSystem;
    g_ScreenDrawer.Finalize();
    g_Glare.Finalize();
    g_ConnectionHeap.Finalize();
    g_Heap.Free( g_ConnectionHeapBuffer );
    g_Heap.Finalize();
}

//------------------------------------------------------------------------------
//      EffectViewerDemo 計算処理
//------------------------------------------------------------------------------
void Viewer::ProcCalc( nw::dev::Pad* pad, nw::eftdemo::DrawParam& drawParam  )
{
    // Lボタンでデバッグ描画の有効/無効を切り替える
    if ( pad->IsTrig( nw::dev::Pad::MASK_L ) )
    {
        mDoDebugDraw = !mDoDebugDraw;
    }

    // フレーム開始処理
    g_EftSystem->BeginFrame();

    // エフェクト計算処理
    g_EftSystem->Calc( 0, 1.0f, true );

    // ビューア計算処理を行い返り値をみて、ビューアグループのエフェクト計算処理を分岐
    if( !g_EffectViewer->ProcCalc( GetPlaySpeed(), drawParam.mViewMtx ) )
    {
        g_EftSystem->Calc( nw::eft2::EFT_VIEWER_GROUP, GetPlaySpeed(), true );  // 再生中
    }
    else
    {
        g_EftSystem->Calc( nw::eft2::EFT_VIEWER_GROUP, 0.0f, false );           // 一時停止
    }
}



//------------------------------------------------------------------------------
//      通常パスのエフェクト描画処理
//------------------------------------------------------------------------------
void Viewer::ProcDraw( nw::eftdemo::DrawParam&          drawParam,
                       nw::eftdemo::FrameBuffer*        currentFrameBuffer,
                       nw::eftdemo::FrameBufferTexture* frameBufferCopy,
                       nw::eftdemo::DepthBufferTexture* depthBufferCopy )
{
    // 背景画像描画処理
    if ( GetBackgroundTexture().IsInitialized() || GetBackgroundTextureRes() != NULL )
    {
        nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
        graphics->SetBlendEnable( false );
        graphics->SetDepthEnable( false, false );
        graphics->SetCullingMode( nw::gfnd::Graphics::CULLING_MODE_NONE );
        g_ScreenDrawer.DrawScreen( nw::eftdemo::ScreenDrawer::SHADER_TYPE_COPY,  reinterpret_cast<const nw::eft2::TextureResource*>( GetBackgroundTextureRes() ) , true );
        graphics->SetBlendEnable( true );
        graphics->SetDepthEnable( true, true );
    }

    // 背景描画処理
    if ( DoDebugDraw() )
    {
        nw::eftdemo::DefaultGrid          defaultGrid;
        nw::eftdemo::HighGrid             hgihGrid;
        nw::eftdemo::CubeGrid             cubeGrid;

        switch( GetGridType() )
        {
        case 0: defaultGrid.ProcDraw( drawParam ); break;
        case 1: hgihGrid.ProcDraw( drawParam ); break;
        case 2: cubeGrid.ProcDraw( drawParam ); break;
        }
    }

    // モデル描画
    {
#if defined(NW_PLATFORM_WIN32)
        static const GLenum renderTargetList[] = { GL_COLOR_ATTACHMENT0_EXT, GL_COLOR_ATTACHMENT1_EXT };
        glDrawBuffers( sizeof(renderTargetList) / sizeof(renderTargetList[0]), renderTargetList );
#endif

        // モデルプレビューの描画
        nw::math::VEC4 lightPos;
        lightPos.x = drawParam.mCameraPos.x;
        lightPos.y = drawParam.mCameraPos.y;
        lightPos.z = drawParam.mCameraPos.z;

        CalcModelPreviewBlock( &drawParam.mProjMtx, &drawParam.mViewMtx, &lightPos, nw::eftdemo::SimpleShader::VIEW_ID_CAMERA );
        DrawModelPreview( nw::eftdemo::SimpleShader::VIEW_ID_CAMERA );

#if defined(NW_PLATFORM_WIN32)
        glDrawBuffer( GL_COLOR_ATTACHMENT0_EXT );
#endif
    }

    // ここまでの描画結果をコピーし、エフェクトシステムに設定する
    currentFrameBuffer->CopyFrameBufferTexture( frameBufferCopy );
    currentFrameBuffer->CopyDepthBufferTexture( depthBufferCopy );
    currentFrameBuffer->Bind();

#if defined(NW_PLATFORM_WIN32)
    g_EftSystem->SetFrameBufferTexture( frameBufferCopy->GetGLTextureID() );
    g_EftSystem->SetDepthBufferTexture( depthBufferCopy->GetGLTextureID() );
#endif
#if defined(NW_PLATFORM_CAFE)
    g_EftSystem->SetFrameBufferTexture( frameBufferCopy->GetGX2Texture() );
    g_EftSystem->SetDepthBufferTexture( depthBufferCopy->GetGX2Texture() );
#endif


    // エフェクト描画処理
#if EFT_IS_CAFE
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
    GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
#endif

    nw::eft2::ViewParam view;
    view.Set( &drawParam.mViewMtx, &drawParam.mProjMtx, &drawParam.mCameraPos, drawParam.mNearZ, drawParam.mFarZ, drawParam.mBaseFovy, drawParam.mCurrentFovy );

    g_EftSystem->SwapBuffer();
    g_EftSystem->SetViewParam( &view );
//  g_EftSystem->SetEmitterDrawProfilerCallback( nw::eftdemo::PerfAnalyzerCallBack );

    // EffectMakerから指定された描画パス順での描画に変更予定
    g_EftSystem->AddSortBuffer( 0 );
    g_EftSystem->AddSortBuffer( nw::eft2::EFT_VIEWER_GROUP );
    g_EftSystem->DrawSortBuffer( true, NULL );

#if EFT_IS_CAFE
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_REGISTER );
    GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
#endif

    // グレア処理
    //g_Glare.Draw( &g_ScreenDrawer, currentFrameBuffer, 2 );
}

//---------------------------------------------------------------------------
//      エフェクトデモのデバッグ文字描画
//---------------------------------------------------------------------------
void Viewer::DebugDraw( nw::dev::DevTextWriter* writer )
{
    writer->Printf( "\n" );
    writer->Printf( " EmitterSet   (%4d/%4d)\n",       g_EftSystem->GetProcessingEmitterSetCount(), g_EftSystem->GetEmitterSetMaxNum() );
    writer->Printf( " Emitter      (%4d/%4d/%4d)\n",   g_EftSystem->GetProcessingEmitterCount(), g_EftSystem->GetEmitterMaxNum(), g_EftSystem->GetFreeEmitterActualNum() );
    writer->Printf( " EmitterAnm   (%4d)\n",           g_EftSystem->GetProcessingEmitterAnimNum() );
    writer->Printf( " CpuPtcl      (%4d)\n",           g_EftSystem->GetProcessingCpuParticleCount() );
    writer->Printf( " GpuPtcl      (%4d)\n",           g_EftSystem->GetProcessingGpuParticleCount() );
    writer->Printf( " GpuSoPtcl    (%4d)\n",           g_EftSystem->GetProcessingGpuSoParticleCount() );
    writer->Printf( " CpuEmitter   (%4d)\n",           g_EftSystem->GetProcessingEmitterCount( nw::eft2::EFT_EMITTER_CALC_TYPE_CPU ) );
    writer->Printf( " GpuEmitter   (%4d)\n",           g_EftSystem->GetProcessingEmitterCount( nw::eft2::EFT_EMITTER_CALC_TYPE_GPU ) );
    writer->Printf( " GpuSoEmitter (%4d)\n",           g_EftSystem->GetProcessingEmitterCount( nw::eft2::EFT_EMITTER_CALC_TYPE_GPU_SO ) );
    writer->Printf( " Stripe       (%4d)\n",           g_EftSystem->GetProcessingStripeCount() );
    writer->Printf( " SuperStripe  (%4d)\n",           g_EftSystem->GetProcessingSuperStripeCount() );
    writer->Printf( " D-HeapSize   (%8d)\n",           g_EftSystem->GetAllocedFromDynamicHeapSize() );
    writer->Printf( " TempBuffer   (%8d)\n",           g_EftSystem->GetUsedTemporaryBufferSize() );
    writer->Printf( " TempBufMax   (%8d)\n",           g_EftSystem->GetMaxUsedTemporaryBufferSize( nw::eft2::EFT_CPU_CORE_1 ) );
    if ( g_EffectViewer->IsLoop() )
    {
        writer->Printf( " Time      ( %f/%f )\n",   g_EffectViewer->GetTime(), g_EffectViewer->GetEndTime() );
    }
    else
    {
        writer->Printf( " Time      ( %f )\n",      g_EffectViewer->GetTime() );
    }
}

//---------------------------------------------------------------------------
//      カメラを設定します。
//---------------------------------------------------------------------------
void Viewer::SetCameraSettings(
    f32 posX, f32 posY, f32 posZ,
    f32 lookAtX, f32 lookAtY, f32 lookAtZ,
    f32 rotation,
    int projType,
    f32 aspect,
    f32 fov,
    f32 height,
    f32 znear, f32 zfar )
{
    if ( mVwrCamera == NULL || mProjectionMatrix == NULL ) return;

    nw::math::VEC3 zaxis( lookAtX - posX, lookAtY - posY, lookAtZ - posZ );
    if ( zaxis.LengthSquare() == 0.0f ) return;

    mFovy = fov;

    mVwrCamera->SetPos( posX, posY, posZ );
    mVwrCamera->SetLookAtPos( lookAtX, lookAtY, lookAtZ );

    // アップベクトルを計算
    {
        nw::math::VEC3 lookUp;

        if ( zaxis.x * zaxis.x + zaxis.z * zaxis.z != 0.0f)
        {
            lookUp = nw::math::VEC3( 0.0f, 1.0f, 0.0f );
        }
        else
        {
            lookUp = nw::math::VEC3( 0.0f, 0.0f, 1.0f );
        }

        nw::math::MTX34 rotMtx;
        rotMtx.SetRotate( zaxis, -rotation );

        nw::math::VEC3 rotLookUp;
        VEC3Transform( &rotLookUp, &rotMtx, &lookUp );

        mVwrCamera->GetMatrix();
        mVwrCamera->SetUpVec( rotLookUp );
        mVwrCamera->SetTwist( rotation );
    }

    if ( znear <= zfar * 0.000001f )
    {
        znear = zfar * 0.000001f;
    }

    if ( projType == 0 )
    {
        // パースペクティブ
        mProjectionMatrix->SetPerspective( fov, aspect, znear, zfar );
    }
    else
    {
        // オルソ
        f32 h = height / 2.0f;
        f32 w = h * aspect;

        mProjectionMatrix->SetOrtho( -w, w, -h, h, znear, zfar );
    }
}

//---------------------------------------------------------------------------
//      垂直同期する数を取得します。
//---------------------------------------------------------------------------
u32 Viewer::GetSwapInterval()
{
    if ( mFrameRate == 60.f )
    {
        return SWAP_INTERVAL_60FPS;
    }
    else if( mFrameRate == 30.f )
    {
        return SWAP_INTERVAL_30FPS;
    }
    else if( mFrameRate == 20.f )
    {
        return SWAP_INTERVAL_20FPS;
    }
    else if( mFrameRate == 15.f )
    {
        return SWAP_INTERVAL_15FPS;
    }

    return SWAP_INTERVAL_60FPS;
}
